-- $Id: sem-compunit-walkstatements-wf_assign.adb 16027 2010-02-10 18:22:16Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

separate (Sem.CompUnit.WalkStatements)
procedure wf_assign (Node  : in STree.SyntaxNode;
                     Scope : in Dictionary.Scopes)
is
   NameResult,
   ExpResult    : ExpRecord;
   ExpectedType,
   NameSym      : Dictionary.Symbol;
   MostEnclosingNameSym : Dictionary.Symbol;
   NameNode,                                  -- the name on the LHS
   ExpNode      : STree.SyntaxNode;           -- the expression on the RHS
   RefVar       : SeqAlgebra.Seq;
   OthersAggregate : Boolean;                 -- is this an unconstrained_array_assignment

   ---------------------------------------------------------------------

   procedure CheckWriteToStructuredVar (NameSym,
                                        ExpectedType : in Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in     RefVar;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpectedType,
   --#                                    NameSym,
   --#                                    RefVar,
   --#                                    TheHeap;
   is
   begin
      if ExpectedType /= Dictionary.GetType (NameSym) then
         --we must be writing to a component of a structured variable
         --so must add structure variable to list of referenced variables
         SeqAlgebra.AddMember (TheHeap,
                               RefVar,
                               Natural (Dictionary.SymbolRef (NameSym)));
      end if;
   end CheckWriteToStructuredVar;

   ---------------------------------------------------------------------

begin -- wf_assign
      -- ASSUME Node = assignment_statement
      -- This procedure checks the following:
      -- (0) if the child node is an unconstrained_array_assignment (if it is then step down
      --     a level in the tree before continuing with the other checks),
      -- (1) the assigned identifier is declared and visible, and
      -- (2) it is a variable, and
      -- (3) it is not an unconstrained array (unless this is an unconstrained_array_assignment)
      -- (4) for unconstrained array assignments the array must be one-dimensional
      -- (5) this variable is not a loop parameter, and
      -- (6) it may be a package own var declared in a non-enclosing scope but
      --     a warning is given.
      -- (7) it is not a formal parameter of mode in, and
      -- additions for streams
      -- (8) check that the assigned variable is not of mode in
      -- (9) check that assigning expression is not a mode out variable

   -- (0) Check if the child node is an unconstrained_array_assignment (if it is then step down
   --     a level in the tree before continuing with the other checks).
   OthersAggregate := SyntaxNodeType (Child_Node (Node)) = SPSymbols.unconstrained_array_assignment;

   if OthersAggregate then
      NameNode := Child_Node (Child_Node (Node));
   else
      NameNode := Child_Node (Node);
   end if;

   --# assert True;

   ExpNode  := Next_Sibling (NameNode);
   SeqAlgebra.CreateSeq (TheHeap, RefVar);

   -- Call WalkExpression to check the LHS of the assignment statement
   --# accept Flow, 10, AggregateStack.State, "Expected ineffective assignment";
   WalkExpression (  -- ineffective AS.S, ES.S, LH all OK
                     ExpNode               => NameNode,
                     Scope                 => Scope,
                     TypeContext           => Dictionary.GetUnknownTypeMark,
                     ContextRequiresStatic => False,
                     --to get
                     Result  => NameResult,
                     RefVar  => RefVar,
                     ComponentData => GlobalComponentData);
   --# end accept;

   NameSym := NameResult.OtherSymbol;
   MostEnclosingNameSym := Dictionary.GetMostEnclosingObject (NameSym);

   -- Check that LHS is something that can be assigned to
   if not NameResult.IsAVariable and then
      not Dictionary.IsUnknownTypeMark (NameResult.TypeSymbol)
   then
      ExpectedType := Dictionary.GetUnknownTypeMark;
      ErrorHandler.SemanticError (609,
                                  14,
                                  NodePosition (NameNode),
                                  LexTokenManager.Null_String);
   elsif NameSym = Dictionary.NullSymbol then
      ExpectedType := Dictionary.GetUnknownTypeMark;
      -- Check for attempts to assign to tagged type conversions:
      if NameResult.Sort = TypeResult and
        Dictionary.TypeIsTagged (NameResult.TypeSymbol)
      then
         if Dictionary.IsRecordSubcomponent (NameResult.VariableSymbol) then
            -- Assignment to view conversion is not implemented yet.
            ErrorHandler.SemanticError (129,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        LexTokenManager.Null_String);
         else
            -- View conversion to own type is not permitted in target of
            -- assignment.
            ErrorHandler.SemanticError (116,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        LexTokenManager.Null_String);
         end if;
      end if;
   else
      ExpectedType := NameResult.TypeSymbol;

      -- For an unconstrained_array_assignment the ExpectedType of the LHS will be the
      -- unconstrained array type, but the type of the RHS will be the type
      -- of the components of that array.
      if OthersAggregate then
         ExpectedType := Dictionary.GetArrayComponent (ExpectedType);
      end if;

      -- Seed syntax tree with expected type for run-time check.
      STree.AddNodeSymbol (Node, ExpectedType);

      -- (2) Check that LHS is a variable
      if Dictionary.IsVariableOrSubcomponent (NameSym) then
         -- If this is an unconstrained_array_assignment then it is an aggregate assignment so
         -- there is never any self-reference.
         if not OthersAggregate then
            -- Check for assignment to structured variables so that they generate
            -- a reference to the variable as well; A (I) = 3; is a reference of A
            -- as well as write to it.  Call moved here because if A is not a variable
            -- in the first place then the check is meaningless.
            CheckWriteToStructuredVar (NameSym,
                                       ExpectedType);
         end if;

         -- (3) Check that ExpectedType is not unconstrained array.
         -- (For unconstrained_array_assignments the LHS *must* be an unconstrained array
         -- but don't need to add guard here because if this is an unconstrained_array_assignment
         -- then ExpectedType will represent the component type, not the array type.)
         if Dictionary.IsUnconstrainedArrayType (ExpectedType) then
            ErrorHandler.SemanticError (39,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        LexTokenManager.Null_String);

         -- (4) If this is an unconstrained array assignment then the target type must be a
         --     one dimensional array. Although the grammar of unconstrained_array_assignment will
         --     not permit:
         --       X := (others => (others => 0));
         --     we still need to make sure that we trap the case where:
         --       X := (others => 0);
         --     when X is a multidimensional array.
         elsif OthersAggregate and then Dictionary.IsArrayTypeMark (NameResult.TypeSymbol, Scope)
                               and then Dictionary.GetNumberOfDimensions (NameResult.TypeSymbol) /= 1 then
            ErrorHandler.SemanticError (118,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        LexTokenManager.Null_String);

         -- (5) Check that LHS is not a loop parameter
         elsif Dictionary.IsLoopParameter (NameSym) then
            ErrorHandler.SemanticError (168,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        Dictionary.GetSimpleName (NameSym));

         -- (6) LHS may be a package own var declared in a non-enclosing scope but
         --     a warning is given.
         elsif not InPackageInitialization (Scope) and then
            Dictionary.IsOwnVariable (MostEnclosingNameSym) and then
            not IsEnclosingPackage (Dictionary.GetOwner (MostEnclosingNameSym),
                                    Scope)
         then
            ErrorHandler.SemanticWarningSym (169,
                                             NodePosition (NameNode),
                                             NameSym,
                                             Scope);

         -- If we are initializing a package own variable, check that the initialization
         -- was announced in the package specification.
         elsif InPackageInitialization (Scope) and then
            UnexpectedInitialization (MostEnclosingNameSym)
         then
            ErrorHandler.SemanticError (333,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        Dictionary.GetSimpleName (MostEnclosingNameSym));

         -- Protected state must be initialized at declaration
         elsif InPackageInitialization (Scope) and then
           Dictionary.PartitionElaborationPolicyIsConcurrent and then
           Dictionary.IsOwnVariable (NameSym) and then
           Dictionary.GetOwnVariableMode (NameSym) = Dictionary.DefaultMode and then
           (Dictionary.GetOwnVariableProtected (NameSym) or else
              Dictionary.IsVirtualElement (NameSym))
         then
            ErrorHandler.SemanticError (874,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        Dictionary.GetSimpleName (MostEnclosingNameSym));

         elsif OthersAggregate and (not Dictionary.IsSubprogramParameter (MostEnclosingNameSym)
                                 or not Dictionary.IsUnconstrainedArrayType (NameResult.TypeSymbol)) then
            -- If LHS is not a subprogram parameter then it can't be an aggregate assignment
            -- to an unconstrained array.
            -- If LHS is not unconstrained then this syntax is not permitted in SPARK.
            -- This error will be raised if there is an attempt to use the syntax for an
            -- unconstrained_array_assignment where the LHS is not an unconstrained array type at all. (Most
            -- likely the LHS is a normal array.) It should not be possible to get here if the
            -- LHS is an unconstrained array type that is not a parameter because SPARK does not
            -- permit objects of unconstrained array types to be declared.
            ErrorHandler.SemanticError (117,
                                        ErrorHandler.NoReference,
                                        NodePosition (ExpNode),
                                        LexTokenManager.Null_String);

         -- (7) Check LHS is not a formal parameter of mode in.
         elsif Dictionary.IsSubprogramParameter (MostEnclosingNameSym) then
            if Dictionary.GetSubprogramParameterMode (MostEnclosingNameSym) =
               Dictionary.InMode or
               Dictionary.GetSubprogramParameterMode (MostEnclosingNameSym) =
               Dictionary.DefaultMode
            then
               ErrorHandler.SemanticError (170,
                                           ErrorHandler.NoReference,
                                           NodePosition (NameNode),
                                           Dictionary.GetSimpleName (MostEnclosingNameSym));
            end if;

         -- Check for direct update of global by function.
         elsif Dictionary.IsFunction (Dictionary.GetEnclosingCompilationUnit (Scope)) and then
            Dictionary.IsGlobalVariable (Dictionary.GetAbstraction
                                         (Dictionary.GetEnclosingCompilationUnit (Scope),
                                          Scope),
                                         Dictionary.GetEnclosingCompilationUnit (Scope),
                                         MostEnclosingNameSym)
         then
            ErrorHandler.SemanticError (327,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        Dictionary.GetSimpleName (MostEnclosingNameSym));

         -- (8) Check LHS is not stream variable of mode in.
         elsif Dictionary.GetOwnVariableOrConstituentMode (MostEnclosingNameSym) =
            Dictionary.InMode then
            ErrorHandler.SemanticError (717,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        Dictionary.GetSimpleName (MostEnclosingNameSym));
         end if;
      else
         ExpectedType := Dictionary.GetUnknownTypeMark;
         ErrorHandler.SemanticError (6,
                                     ErrorHandler.NoReference,
                                     NodePosition (NameNode),
                                     Dictionary.GetSimpleName (NameSym));
      end if;
   end if;

   --# assert True;

   -- Call WalkExpression to check the RHS of the assignment statement.
   WalkExpression (ExpNode,
                   Scope,
                   ExpectedType, -- Should contextual type analysis be needed, the
                                 -- expression on the RHS
                                 -- is expected to have the type of the var. on the LHS,
                   False,
                     --to get
                   ExpResult,
                   RefVar,
                   GlobalComponentData);

   AssignmentCheck (NodePosition (ExpNode),
                    Scope,
                    ExpectedType,
                    ExpResult);

   -- (9) Check that we are not trying to read an out stream.
   if ExpResult.IsAVariable and then
      Dictionary.GetOwnVariableOrConstituentMode (ExpResult.VariableSymbol) =
      Dictionary.OutMode then
      ErrorHandler.SemanticErrorSym (718,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     ExpResult.VariableSymbol,
                                     Scope);
   end if;

   --# assert True;

   -- If expression represents an IN stream variable then put type of expression
   -- into syntax tree for the benefit of the RTC procedure ModelAssignmentStatement.
   if ExpResult.IsAVariable and then
      Dictionary.GetOwnVariableOrConstituentMode (ExpResult.VariableSymbol) =
      Dictionary.InMode then
      -- Mark the enclosing compilation unit as assigning an external variable
      -- This may be too coarse; may be we should just mark enclosing subprog?
      Dictionary.AddAssignsFromExternal (
         Dictionary.GetEnclosingCompilationUnit (Scope));

      STree.AddNodeSymbol (ExpNode, ExpResult.TypeSymbol);
      -- Check to see if the variable has been marked as always valid.
      -- Note that the OtherSymbol is checked,not the variableSymbol,
      -- since this will be the Subcomponent symbol if we are referring to
      -- a record component.
      if Dictionary.VariableOrSubcomponentIsMarkedValid (ExpResult.OtherSymbol) then
         --MCA & TJJ: do we also need to add a use of 'Always_Valid to the summary?
         --Debug.PrintSym ("Access is Always_Valid =", ExpResult.OtherSymbol);
         null;
      else
         -- and issue warning about possible validity problems.
         -- The warning is stronger when the external variable is a type that doesn't
         -- generate run-time checks.
         if Dictionary.TypeIsScalar (ExpResult.TypeSymbol) and then
           not Dictionary.TypeIsBoolean (ExpResult.TypeSymbol) then
            --weaker warning
            ErrorHandler.SemanticWarningSym (392,
                                             NodePosition (ExpNode),
                                             ExpResult.OtherSymbol,
                                             Scope);
         else
            --stronger warning
            ErrorHandler.SemanticWarningSym (393,
                                             NodePosition (ExpNode),
                                             ExpResult.OtherSymbol,
                                             Scope);
         end if;
      end if;
   end if;

   --# assert True;

   -- If the expression represents a use of unchecked conversion then plant the return
   -- type in the syntax tree for the benefit of the RTC procedure ModelAssignmentStatement
   -- TJJ: Note a more explicit way of designating and checking for this would
   -- be better so that it is easier to determine the extent of use of this idiom.
   if Dictionary.IsAnUncheckedConversion (ExpResult.OtherSymbol) then
      STree.AddNodeSymbol (ExpNode,
                                       Dictionary.GetType (ExpResult.OtherSymbol));
   end if;

   if Dictionary.TypeIsLimited (ExpResult.TypeSymbol, Scope) then
      ErrorHandler.SemanticError (308,
                                  ErrorHandler.NoReference,
                                  NodePosition (ExpNode),
                                  LexTokenManager.Null_String);
   end if;

   -- Patch flow relations to take into account stream volatility.
   AddStreamEffects (Table,
                     TheHeap,
                     Node,
                     NameSym,
                     RefVar);

   -- Add export and list of imports to RefList hash table.
   RefList.AddRelation (Table,
                        TheHeap,
                        Node,
                        NameSym,
                        RefVar);
end wf_assign;
