-- $Id: sem-compunit-walkexpression.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SPrint;
with ELStrings;

separate (Sem.CompUnit)
procedure WalkExpression (ExpNode               : in     STree.SyntaxNode;
                          Scope                 : in     Dictionary.Scopes;
                          TypeContext           : in     Dictionary.Symbol;
                          ContextRequiresStatic : in     Boolean;
                          Result                :    out ExpRecord;
                          RefVar                : in     SeqAlgebra.Seq;
                          ComponentData         : in out ComponentManager.ComponentData)
is

   NextNode,
   LastNode,
   LocalNode  : STree.SyntaxNode;
   NodeType   : SPSymbols.SPSymbol;
   LHeap      : Lists.List_Heap;
   Val        : Maths.Value;
   EStack     : ExpStack.ExpStackType;
   TStack     : TypeContextStack.TStackType;
   TempResult : ExpRecord;
   Sym        : Dictionary.Symbol;

   --!!!!!!!!!!!!!!!!!!!test!!!!!!!!!!!!!!!!!
   --X : SeqAlgebra.MemberOfSeq;
   --XS : EStrings.T;

   --------------------------------------------------------------
   -- Procedures for debugging Expression syntax and tree walking
   -- These are hidden with "derives ;" so as not to pollute the
   -- annotations
   --------------------------------------------------------------

   procedure Dump_Syntax_Tree
   --# derives ;
   is
      --# hide Dump_Syntax_Tree;
   begin
      if CommandLineData.Content.Debug.Expressions then
         SPrint.Dump_Syntax_Tree (ExpNode, 0);
      end if;
   end Dump_Syntax_Tree;

   procedure Dump_Down_Node
   --# derives ;
   is
      --# hide Dump_Down_Node;
   begin
      if CommandLineData.Content.Debug.Expressions then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "WalkExpression DOWN encounters node" &
                            ExaminerConstants.RefType'Image (STree.NodeToRef (LastNode)) &
                            ' ' &
                            SPSymbols.SPSymbol'Image (SyntaxNodeType (LastNode)),
                            0);
      end if;
   end Dump_Down_Node;

   procedure Dump_Up_Node
   --# derives ;
   is
      --# hide Dump_Up_Node;
   begin
      if CommandLineData.Content.Debug.Expressions then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "WalkExpression UP   encounters node" &
                            ExaminerConstants.RefType'Image (STree.NodeToRef (LastNode)) &
                            ' ' &
                            SPSymbols.SPSymbol'Image (SyntaxNodeType (LastNode)),
                            0);
      end if;
   end Dump_Up_Node;

   procedure Dump_Result
   --# derives ;
   is
      --# hide Dump_Result;
   begin
      if CommandLineData.Content.Debug.Expressions then
         if TempResult.IsConstant then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                 "WalkExpression constant result is ", 0);
            ELStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                E_Str => Maths.ValueToString (TempResult.Value));
         else
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                               "WalkExpression result is not constant", 0);
         end if;
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                           "WalkExpression Result is: ", 0);
         Put_ExpRecord (TempResult);
      end if;
   end Dump_Result;
begin --WalkExpression

   Dump_Syntax_Tree;

   ExpStack.Init (EStack);

   TypeContextStack.Init (TStack);
   TypeContextStack.Push (TypeContext, TStack);

   AggregateStack.Init;
   Lists.Init (LHeap);

   NextNode := ExpNode;
   loop -----------------------------------------------------down loop----
      LastNode := NextNode;
      NodeType := SyntaxNodeType (LastNode);

      Dump_Down_Node;

      case NodeType is
         when SPSymbols.character_literal =>
            GetCharacterLiteral (LastNode,
                                 --to get
                                 Val);
            ExpStack.Push (ExpRecord'(TypeSymbol            => Dictionary.GetPredefinedCharacterType,
                                      OtherSymbol           => Dictionary.NullSymbol,
                                      StreamSymbol          => Dictionary.NullSymbol,
                                      TaggedParameterSymbol => Dictionary.NullSymbol,
                                      VariableSymbol        => Dictionary.NullSymbol,
                                      ParamCount            => 0,
                                      ParamList             => Lists.Null_List,
                                      Sort                  => TypeResult,
                                      ArgListFound          => False,
                                      IsAVariable           => False,
                                      IsAnEntireVariable    => False,
                                      ErrorsInExpression    => False,
                                      HasOperators          => False,
                                      IsConstant            => True,
                                      IsStatic              => True,
                                      IsARange              => False,
                                      Value                 => Val,
                                      RangeRHS              => Maths.NoValue),
                           EStack);
            NextNode := STree.NullNode;

         when SPSymbols.string_literal =>
            GetStringLiteralLength (LastNode,
                                    --to get
                                    Val);

            ExpStack.Push
              (ExpRecord'
                 (TypeSymbol            => Dictionary.GetPredefinedStringType,
                  OtherSymbol           => Dictionary.NullSymbol,
                  StreamSymbol          => Dictionary.NullSymbol,
                  TaggedParameterSymbol => Dictionary.NullSymbol,
                  VariableSymbol        => Dictionary.NullSymbol,
                  ParamCount            => 0,
                  ParamList             => Lists.Null_List,
                  Sort                  => TypeResult,
                  ArgListFound          => False,
                  IsAVariable           => False,
                  IsAnEntireVariable    => False,
                  ErrorsInExpression    => False,
                  HasOperators          => False,
                  IsConstant            => True,
                  IsStatic              => (CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83),
                  IsARange              => False,
                  Value                 => Maths.NoValue,
                  RangeRHS              => Val),
               EStack);
            NextNode := STree.NullNode;

         when SPSymbols.numeric_literal =>
            LocalNode := Child_Node (Child_Node (LastNode));
            GetLiteralValue (LocalNode,
                              --to get
                             Val);

            if SyntaxNodeType (LocalNode) = SPSymbols.real_number then
               ExpStack.Push (ExpRecord'(TypeSymbol            => Dictionary.GetUniversalRealType,
                                         OtherSymbol           => Dictionary.NullSymbol,
                                         StreamSymbol          => Dictionary.NullSymbol,
                                         TaggedParameterSymbol => Dictionary.NullSymbol,
                                         VariableSymbol        => Dictionary.NullSymbol,
                                         ParamCount            => 0,
                                         ParamList             => Lists.Null_List,
                                         Sort                  => TypeResult,
                                         ArgListFound          => False,
                                         IsAVariable           => False,
                                         IsAnEntireVariable    => False,
                                         ErrorsInExpression    => False,
                                         HasOperators          => False,
                                         IsConstant            => True,
                                         IsStatic              => True,
                                         IsARange              => False,
                                         Value                 => Val,
                                         RangeRHS              => Maths.NoValue),
                              EStack);
            else
               ExpStack.Push (ExpRecord'(TypeSymbol            => Dictionary.GetUniversalIntegerType,
                                         OtherSymbol           => Dictionary.NullSymbol,
                                         StreamSymbol          => Dictionary.NullSymbol,
                                         TaggedParameterSymbol => Dictionary.NullSymbol,
                                         VariableSymbol        => Dictionary.NullSymbol,
                                         ParamCount            => 0,
                                         ParamList             => Lists.Null_List,
                                         Sort                  => TypeResult,
                                         ArgListFound          => False,
                                         IsAVariable           => False,
                                         IsAnEntireVariable    => False,
                                         ErrorsInExpression    => False,
                                         HasOperators          => False,
                                         IsConstant            => True,
                                         IsStatic              => True,
                                         IsARange              => False,
                                         Value                 => Val,
                                         RangeRHS              => Maths.NoValue),
                              EStack);
            end if;
            NextNode := STree.NullNode;

         when SPSymbols.selector =>
            NextNode := STree.NullNode;

         when SPSymbols.simple_name =>
            if SyntaxNodeType (ParentNode (LastNode)) =
               SPSymbols.named_argument_association
            then
               --do not look at identifier in this case
               NextNode := STree.NullNode;
            else
               NextNode := Child_Node (LastNode);
            end if;

         when SPSymbols.identifier =>
            wf_identifier (Node           => LastNode,
                           Scope          => Scope,
                           EStack         => EStack,
                           RefVar         => RefVar,
                           Context        => Code,
                           IsAnnotation   => False);
            NextNode := STree.NullNode;

         when SPSymbols.name_argument_list =>
            down_wf_name_argument_list (Node         => LastNode,
                                        Scope        => Scope,
                                        EStack       => EStack,
                                        HeapParam    => LHeap,
                                        NextNode     => NextNode,
                                        IsAnnotation => False);

         when SPSymbols.aggregate =>
            down_wf_aggregate (Node         => LastNode,
                               Scope        => Scope,
                               NextNode     => NextNode,
                               EStack       => EStack,
                               HeapParam    => LHeap,
                               IsAnnotation => False);

         when SPSymbols.aggregate_choice_rep =>
            down_wf_aggregate_choice_rep (Node         => LastNode,
                                          Scope        => Scope,
                                          EStack       => EStack,
                                          HeapParam    => LHeap,
                                          NextNode     => NextNode,
                                          IsAnnotation => False);

         when SPSymbols.record_component_selector_name =>
            wf_record_component_selector_name (Node         => LastNode,
                                               Scope        => Scope,
                                               EStack       => EStack,
                                               HeapParam    => LHeap,
                                               NextNode     => NextNode);

         when SPSymbols.aggregate_or_expression =>
            down_wf_aggregate_or_expression (Node         => LastNode,
                                             EStack       => EStack,
                                             IsAnnotation => False,
                                             NextNode     => NextNode);

         when SPSymbols.attribute_designator =>

            TypeContextStack.Push
              (AttributeDesignatorTypeFromContext (LastNode, False, EStack, TStack), TStack);

            NextNode := Child_Node (LastNode);

         when SPSymbols.range_constraint =>

            TypeContextStack.Push
              (RangeConstraintTypeFromContext (LastNode, EStack, TStack), TStack);

            NextNode := Child_Node (LastNode);

         when SPSymbols.simple_expression =>

            TypeContextStack.Push
              (SimpleExpressionTypeFromContext (LastNode, TStack), TStack);

            NextNode := Child_Node (LastNode);

         when SPSymbols.expression =>

            ExpressionTypeFromContext (Exp_Node         => LastNode,
                                       E_Stack          => EStack,
                                       T_Stack          => TStack,
                                       New_Context_Type => Sym);
            TypeContextStack.Push (Sym, TStack);

            NextNode := Child_Node (LastNode);

         when SPSymbols.primary =>

            TypeContextStack.Push
              (PrimaryTypeFromContext (LastNode, TStack), TStack);

            NextNode := Child_Node (LastNode);

         when others =>
            if NodeType in SPSymbols.SPNonTerminal then
               NextNode := Child_Node (LastNode);
            else
               NextNode := STree.NullNode;
            end if;
      end case;

      -------------------------------------------------up loop----------
      if NextNode = STree.NullNode then
         loop
            NextNode := Next_Sibling (LastNode);
            exit when NextNode /= STree.NullNode; --new branch to right

            NextNode := ParentNode (LastNode);
            LastNode := NextNode;
            NodeType := SyntaxNodeType (LastNode);

            Dump_Up_Node;

            case NodeType is
               when SPSymbols.expression =>
                  wf_expression (Node         => LastNode,
                                 Scope        => Scope,
                                 EStack       => EStack,
                                 TStack       => TStack,
                                 IsAnnotation => False);

                  TypeContextStack.Pop (TStack);

               when SPSymbols.expression_rep1 |
                    SPSymbols.expression_rep2 |
                    SPSymbols.expression_rep3 |
                    SPSymbols.expression_rep4 |
                    SPSymbols.expression_rep5 =>
                  wf_expression (Node         => LastNode,
                                 Scope        => Scope,
                                 EStack       => EStack,
                                 TStack       => TStack,
                                 IsAnnotation => False);

               when SPSymbols.simple_expression =>
                  wf_simple_expression (Node                  => LastNode,
                                        Scope                 => Scope,
                                        EStack                => EStack,
                                        TStack                => TStack,
                                        ContextRequiresStatic => ContextRequiresStatic,
                                        IsAnnotation          => False);

                  TypeContextStack.Pop (TStack);

               when SPSymbols.simple_expression_opt =>
                  wf_simple_expression_opt (Node         => LastNode,
                                            Scope        => Scope,
                                            EStack       => EStack,
                                            TStack       => TStack,
                                            IsAnnotation => False);

               when SPSymbols.term =>
                  wf_term (Node                  => LastNode,
                           Scope                 => Scope,
                           EStack                => EStack,
                           TStack                => TStack,
                           ContextRequiresStatic => ContextRequiresStatic,
                           IsAnnotation          => False);

               when SPSymbols.factor =>
                  wf_factor (Node         => LastNode,
                             Scope        => Scope,
                             EStack       => EStack,
                             TStack       => TStack,
                             IsAnnotation => False);

               when SPSymbols.relation =>
                  wf_relation (Node         => LastNode,
                               Scope        => Scope,
                               EStack       => EStack,
                               TStack       => TStack,
                               IsAnnotation => False);

               when SPSymbols.range_constraint =>

                  TypeContextStack.Pop (TStack);

               when SPSymbols.arange =>
                  wf_arange (Node         => LastNode,
                             Scope        => Scope,
                             EStack       => EStack,
                             IsAnnotation => False);

               when SPSymbols.selected_component =>
                  wf_selected_component (Node           => LastNode,
                                         Scope          => Scope,
                                         EStack         => EStack,
                                         RefVar         => RefVar,
                                         ComponentData  => ComponentData,
                                         Context        => Code,
                                         IsAnnotation   => False);

               when SPSymbols.attribute =>
                  wf_attribute (EStack => EStack);

               when SPSymbols.attribute_designator =>
                  wf_attribute_designator (Node => LastNode,
                                           Scope => Scope,
                                           EStack       => EStack,
                                           RefVar       => RefVar,
                                           IsAnnotation => False);

                  TypeContextStack.Pop (TStack);

               when SPSymbols.primary =>
                  wf_primary (Node          => LastNode,
                              Scope         => Scope,
                              EStack        => EStack,
                              RefVar        => RefVar,
                              IsAnnotation  => False,
                              ComponentData => ComponentData);

                  TypeContextStack.Pop (TStack);

               when SPSymbols.positional_argument_association =>
                  wf_positional_argument_association (Node   => LastNode, --may be changed
                                                      Scope        => Scope,
                                                      EStack       => EStack,
                                                      IsAnnotation => False,
                                                      RefVar        => RefVar,
                                                      ComponentData => ComponentData);

               when SPSymbols.named_argument_association =>
                  wf_named_argument_association (Node         => LastNode,
                                                 Scope        => Scope,
                                                 EStack       => EStack,
                                                 HeapParam    => LHeap,
                                                 IsAnnotation => False);

               when SPSymbols.name_argument_list =>
                  up_wf_name_argument_list (Node         => LastNode,
                                            EStack       => EStack,
                                            HeapParam    => LHeap,
                                            IsAnnotation => False);

                  --!!!!!!!!!!!!!!!!!!!!!!!!!test!!!!!!!!!!!!!!!!!!!!!!!!
                  --text_io.Put ("Referenced Variables: ");
                  --X := SeqAlgebra.FirstMember (TheHeap, RefVar);
                  --while not SeqAlgebra.IsNullMember (X) loop
                  --   LexTokenManager.LexStringToString
                  --     (Dictionary.GetSimpleName
                  --      (Dictionary.ConvertSymbolRef
                  --       (ExaminerConstants.RefType
                  --        (SeqAlgebra.ValueOfMember (TheHeap,
                  --                                 X)))),
                  --    XS);
                  --   text_io.put (XS.Content (1..XS.Length));
                  --   text_io.put (", ");
                  --   X := SeqALgebra.NextMember (TheHeap, X);
                  --end loop;
                  --text_io.put_line ("----------------------------");
                  --!!!!!!!!!!!!!!!!!!!!!!!!!test!!!!!!!!!!!!!!!!!!!!!!!!

               when SPSymbols.ancestor_part =>
                  wf_ancestor_part (Node         => LastNode, -- may be changed
                                    Scope        => Scope,
                                    EStack       => EStack,
                                    HeapParam    => LHeap,
                                    IsAnnotation => False);

               when SPSymbols.aggregate_choice =>
                  up_wf_aggregate_choice (Node         => LastNode,
                                          Scope        => Scope,
                                          EStack       => EStack);

               when SPSymbols.named_association_rep =>
                  up_wf_named_association_rep (Node         => LastNode,
                                               Scope        => Scope,
                                               EStack       => EStack,
                                               HeapParam    => LHeap,
                                               IsAnnotation => False);

               when SPSymbols.named_record_component_association =>
                  up_wf_named_record_component_association (Node         => LastNode,
                                                            Scope        => Scope,
                                                            EStack       => EStack,
                                                            HeapParam    => LHeap);

               when SPSymbols.positional_association |
                 SPSymbols.record_component_association =>
                  up_wf_positional_association (Node         => LastNode,
                                                EStack       => EStack);

               when SPSymbols.aggregate_or_expression =>
                  up_wf_aggregate_or_expression (Node => LastNode,  -- may be changed by call
                                                 Scope        => Scope,
                                                 EStack       => EStack,
                                                 IsAnnotation => False);

               when SPSymbols.positional_record_component_association =>
                  wf_positional_record_component_association (Node => LastNode,  -- may be changed by call
                                                              Scope        => Scope,
                                                              EStack       => EStack,
                                                              IsAnnotation => False);

               when SPSymbols.component_association =>
                  up_wf_component_association (Node         => LastNode,
                                               Scope        => Scope,
                                               EStack       => EStack,
                                               IsAnnotation => False);

               when SPSymbols.aggregate        |
                 SPSymbols.extension_aggregate =>
                  up_wf_aggregate (Node         => LastNode,
                                   Scope        => Scope,
                                   EStack       => EStack,
                                   IsAnnotation => False);

               when SPSymbols.qualified_expression =>
                  wf_qualified_expression (Node         => LastNode,
                                           Scope        => Scope,
                                           EStack       => EStack,
                                           IsAnnotation => False);

               when others =>

                  null;

            end case;

            exit when NextNode = ExpNode;  --got back to top
         end loop; --up
      end if;

      exit when NextNode = ExpNode; --met start point on way up

   end loop; --down

   if not ExpStack.HasOneEntry (EStack) then
      SystemErrors.FatalError (SystemErrors.ExpressionStackCorrupt, "WalkExpression");
   end if;

   if not (TypeContextStack.HasOneEntry (TStack) and then
           TypeContextStack.Top (TStack) = TypeContext) then
      SystemErrors.FatalError (SystemErrors.TypeContextStackCorrupt, "WalkExpression");
   end if;

   --# accept Flow, 10, EStack, "Expected ineffective assignment";
   ExpStack.Pop (TempResult, EStack); -- ineffective because final use of stack
   --# end accept;

   Dump_Result;

   Result := TempResult;

   --!!!!!!!!!!!!!!!!!!!!!!!!!test!!!!!!!!!!!!!!!!!!!!!!!!
   --  text_io.Put ("Reference Variables: ");
   --  X := SeqAlgebra.FirstMember (TheHeap, RefVar);
   --  while not SeqAlgebra.IsNullMember (X) loop
   --    LexTokenManager.LexStringToString (
   --                  Dictionary.GetSimpleName (
   --                     Dictionary.ConvertSymbolRef (
   --                       ExaminerConstants.RefType (
   --                          SeqAlgebra.ValueOfMember (TheHeap,
   --                                                   X)))),
   --                  XS);
   --    text_io.put (XS.Content (1..XS.Length));
   --    text_io.put (", ");
   --    X := SeqALgebra.NextMember (TheHeap, X);
   --  end loop;
   --  text_io.put_line ("----------------------------");
   --!!!!!!!!!!!!!!!!!!!!!!!!!test!!!!!!!!!!!!!!!!!!!!!!!!

end WalkExpression;
