-- $Id: sem-compunit-up_wf_package_body.adb 15946 2010-02-05 13:52:44Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SLI;

separate (Sem.CompUnit)

procedure up_wf_package_body (Node  : in     STree.SyntaxNode;
                              Scope : in out Dictionary.Scopes)
is
   IdentNode,
   EndDesigNode,
   NodeToReportErrors,
   NextNode,
   RefNode       : STree.SyntaxNode;
   PackSym       : Dictionary.Symbol;

   -------------------------------------------------------

   procedure CheckOwnVariablesDeclared (PackSym : in Dictionary.Symbol;
                                        Node    : in STree.SyntaxNode;
                                        Scope   : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      ConstituentList,
      OwnVarList       : Dictionary.Iterator;
      OwnVarSym,
      Constituent      : Dictionary.Symbol;

      procedure CheckStreamHasAddress (Sym : in Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Node;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        Sym;
      is
      begin -- CheckStreamHasAddress
         if Dictionary.GetOwnVariableOrConstituentMode (Sym) /=
           Dictionary.DefaultMode and then
           not Dictionary.TypeIsProtected (Dictionary.GetType (Sym)) and then
           not Dictionary.VariableHasAddressClause (Sym) and then
           not Dictionary.VariableHasPragmaImport (Sym)
         then
            ErrorHandler.SemanticWarning (395,
                                          NodePosition (Node),
                                          Dictionary.GetSimpleName (Sym));
         end if;
      end CheckStreamHasAddress;

      function OwnVarDeclaredRemotely (Constituent : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin -- OwnVarDeclaredRemotely
         return (Dictionary.IsOwnVariable (Constituent) and
                   Dictionary.GetContext
                   (Dictionary.GetRegion
                      (Dictionary.GetScope (Constituent))) =
                   Dictionary.ProgramContext);
      end OwnVarDeclaredRemotely;

   begin -- CheckOwnVariablesDeclared
      OwnVarList := Dictionary.FirstOwnVariable (PackSym);
      while not Dictionary.IsNullIterator (OwnVarList) loop
         OwnVarSym := Dictionary.CurrentSymbol (OwnVarList);
         if not Dictionary.GetOwnVariableIsInterruptStream (OwnVarSym) then
            if Dictionary.IsAbstractOwnVariable (OwnVarSym) then
               ConstituentList := Dictionary.FirstConstituent (OwnVarSym);
               while not Dictionary.IsNullIterator (ConstituentList) loop
                  Constituent := Dictionary.CurrentSymbol (ConstituentList);
                  if not (OwnVarDeclaredRemotely (Constituent) or else
                            Dictionary.IsDeclared (Constituent))
                  then
                     ErrorHandler.SemanticErrorSym (84,
                                                    20,
                                                    NodePosition (Node),
                                                    Constituent,
                                                    Scope);
                  elsif not OwnVarDeclaredRemotely (Constituent) then
                     -- it is declared, so check streams have addresses, etc.
                     CheckStreamHasAddress (Constituent);
                     -- CheckProtectedReallyIs (Constituent);
                  end if;
                  ConstituentList := Dictionary.NextSymbol (ConstituentList);
               end loop;
            else
               if not Dictionary.IsDeclared (OwnVarSym) then
                  ErrorHandler.SemanticError (28,
                                              20,
                                              NodePosition (Node),
                                              Dictionary.GetSimpleName (OwnVarSym));
               else -- it is declared, so check streams have addresses, etc.
                  CheckStreamHasAddress (OwnVarSym);
                  -- CheckProtectedReallyIs (OwnVarSym);
               end if;
            end if;
         end if;
         OwnVarList := Dictionary.NextSymbol (OwnVarList);
      end loop;
   end CheckOwnVariablesDeclared;

   -------------------------------------------------------

   -- Check that Integrity property of refinement constituents (if present)
   -- is OK with the Integrity of the abstract own variable(s), given
   -- the selected setting of CommandLineData.Content.InfoFlowPolicy
   procedure CheckIntegrityOfRefinements (PackSym : in Dictionary.Symbol;
                                          Node    : in STree.SyntaxNode;
                                          Scope   : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      ConstituentList,
      OwnVarList         : Dictionary.Iterator;
      OwnVarSym,
      Constituent        : Dictionary.Symbol;
      Abstract_Integrity    : LexTokenManager.Lex_String;
      Constituent_Integrity : LexTokenManager.Lex_String;

      procedure Check_Integrities
      --# global in     Abstract_Integrity;
      --#        in     CommandLineData.Content;
      --#        in     Constituent;
      --#        in     Constituent_Integrity;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Node;
      --#        in     OwnVarSym;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from Abstract_Integrity,
      --#                                        CommandLineData.Content,
      --#                                        Constituent,
      --#                                        Constituent_Integrity,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        OwnVarSym,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
      is
      begin
         if Abstract_Integrity = LexTokenManager.Null_String then
            if Constituent_Integrity = LexTokenManager.Null_String then
               -- Both null. No action required.
               -- Debug.PrintMsg ("Both integrities null, so no action", True);
               null;
            else
               -- Abstract integrity null, constituent not null - error!
               ErrorHandler.SemanticErrorSym2 (861,
                                               ErrorHandler.NoReference,
                                               NodePosition (Node),
                                               OwnVarSym,
                                               Constituent,
                                               Scope);
            end if;

         else
            if Constituent_Integrity = LexTokenManager.Null_String then
               -- Abstract var has an integrity, constituent doesn't.
               -- This is OK - the constituent inherits the Integrity of the abstract.
               -- Debug.PrintMsg ("Abstact integrity inherited by constituent", True);
               null;
            else
               -- Both have specific Integrity properties, so need to check
               -- that they are the same, else error 862
               if Maths.ValueRep (Abstract_Integrity) /= Maths.ValueRep (Constituent_Integrity) then
                  -- Debug.PrintMsg ("Different Values - NOT OK", True);
                  ErrorHandler.SemanticErrorSym2 (862,
                                                  ErrorHandler.NoReference,
                                                  NodePosition (Node),
                                                  OwnVarSym,
                                                  Constituent,
                                                  Scope);
               end if;

            end if;

         end if;

      end Check_Integrities;

   begin
      case CommandLineData.Content.InfoFlowPolicy is
         when CommandLineData.None =>
            -- Nothing to do if -policy is not set at all...
            null;
         when CommandLineData.Safety | CommandLineData.Security =>

            OwnVarList := Dictionary.FirstOwnVariable (PackSym);
            while not Dictionary.IsNullIterator (OwnVarList) loop
               OwnVarSym := Dictionary.CurrentSymbol (OwnVarList);
               if not Dictionary.GetOwnVariableIsInterruptStream (OwnVarSym) then
                  if Dictionary.IsAbstractOwnVariable (OwnVarSym) then

                     -- Debug.PrintSym ("Own var IS refined: ", OwnVarSym);

                     Abstract_Integrity := Dictionary.GetIntegrityProperty (OwnVarSym);
                     -- Debug.PrintLexStr (" and its integrity is: ", Abstract_Integrity);

                     ConstituentList := Dictionary.FirstConstituent (OwnVarSym);

                     while not Dictionary.IsNullIterator (ConstituentList) loop
                        Constituent := Dictionary.CurrentSymbol (ConstituentList);

                        Constituent_Integrity := Dictionary.GetIntegrityProperty (Constituent);

                        -- Debug.PrintSym ("  Considering constituent: ", Constituent);
                        -- Debug.PrintInt ("  With SymbolRef ", Integer (Dictionary.SymbolRef (Constituent)));
                        -- Debug.PrintLexStr ("  Constituent Integrity is: ", Constituent_Integrity);

                        Check_Integrities;

                        ConstituentList := Dictionary.NextSymbol (ConstituentList);
                     end loop;
                  -- else
                  --    Debug.PrintSym ("Own var is not refined: ", OwnVarSym);
                  end if;
               end if;
               OwnVarList := Dictionary.NextSymbol (OwnVarList);
            end loop;
      end case;

   end CheckIntegrityOfRefinements;

   -------------------------------------------------------






   procedure CheckExportedSubprogramsDeclared (PackSym : in Dictionary.Symbol;
                                               Node    : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is

      procedure CheckSubprograms (SubprogramIt : in Dictionary.Iterator;
                                  Node         : in STree.SyntaxNode)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        SubprogramIt;
      is
         SubProgSym : Dictionary.Symbol;
         It         : Dictionary.Iterator;
      begin
         It := SubprogramIt;
         while not Dictionary.IsNullIterator (It) loop
            SubProgSym := Dictionary.CurrentSymbol (It);
            if not (Dictionary.IsProofFunction (SubProgSym) or else
                      Dictionary.HasBody (SubProgSym)         or else
                      Dictionary.HasBodyStub (SubProgSym))
            then
               ErrorHandler.SemanticError (29,
                                           21,
                                           NodePosition (Node),
                                           Dictionary.GetSimpleName (SubProgSym));
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end CheckSubprograms;

   begin -- CheckExportedSubprogramsDeclared
      CheckSubprograms (Dictionary.FirstVisibleSubprogram (PackSym), Node);
      CheckSubprograms (Dictionary.FirstPrivateSubprogram (PackSym), Node);
   end CheckExportedSubprogramsDeclared;

   ----------------------------------------------------------------

   procedure CheckVisibleTaskTypesDeclared (PackSym : in Dictionary.Symbol;
                                            Node    : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      It         : Dictionary.Iterator;
      TaskSym : Dictionary.Symbol;

   begin
      It := Dictionary.FirstVisibleTaskType (PackSym);
      while not Dictionary.IsNullIterator (It) loop
         TaskSym := Dictionary.CurrentSymbol (It);
         if not (Dictionary.HasBody (TaskSym) or else
                   Dictionary.HasBodyStub (TaskSym))
         then
            ErrorHandler.SemanticError (896,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (TaskSym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckVisibleTaskTypesDeclared;

   ----------------------------------------------------------------

   procedure CheckPrivateTaskTypesDeclared (PackSym : in Dictionary.Symbol;
                                            Node    : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      It         : Dictionary.Iterator;
      TaskSym : Dictionary.Symbol;

   begin
      It := Dictionary.FirstPrivateTaskType (PackSym);
      while not Dictionary.IsNullIterator (It) loop
         TaskSym := Dictionary.CurrentSymbol (It);
         if not (Dictionary.HasBody (TaskSym) or else
                   Dictionary.HasBodyStub (TaskSym))
         then
            ErrorHandler.SemanticError (896,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (TaskSym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckPrivateTaskTypesDeclared;

   ----------------------------------------------------------------

   procedure CheckVisibleProtectedTypesDeclared (PackSym : in Dictionary.Symbol;
                                                 Node    : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      It         : Dictionary.Iterator;
      ProtectedSym : Dictionary.Symbol;

   begin
      It := Dictionary.FirstVisibleProtectedType (PackSym);
      while not Dictionary.IsNullIterator (It) loop
         ProtectedSym := Dictionary.CurrentSymbol (It);
         if not (Dictionary.HasBody (ProtectedSym) or else
                   Dictionary.HasBodyStub (ProtectedSym))
         then
            ErrorHandler.SemanticError (897,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (ProtectedSym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckVisibleProtectedTypesDeclared;

   ----------------------------------------------------------------

   procedure CheckPrivateProtectedTypesDeclared (PackSym : in Dictionary.Symbol;
                                                 Node    : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      It         : Dictionary.Iterator;
      ProtectedSym : Dictionary.Symbol;

   begin
      It := Dictionary.FirstPrivateProtectedType (PackSym);
      while not Dictionary.IsNullIterator (It) loop
         ProtectedSym := Dictionary.CurrentSymbol (It);
         if not (Dictionary.HasBody (ProtectedSym) or else
                   Dictionary.HasBodyStub (ProtectedSym))
         then
            ErrorHandler.SemanticError (897,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (ProtectedSym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckPrivateProtectedTypesDeclared;

   ----------------------------------------------------------------

   procedure CheckInitialization (PackSym       : in     Dictionary.Symbol;
                                  Node          : in     STree.SyntaxNode;
                                  Scope         : in     Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      ConstituentList : Dictionary.Iterator;
      InitOwnVarList  : Dictionary.Iterator;
      InitOwnVarSym   : Dictionary.Symbol;
      ConstituentSym  : Dictionary.Symbol;

   begin --CheckInitialization
      InitOwnVarList := Dictionary.FirstInitializedOwnVariable (PackSym);
      -- Note.  Moded own variables will not get picked up by this iterator since
      -- other wffs ensure that moded own vars can never appear in Init clauses
      -- Note.  Protected moded own variables will get picked up by this iterator
      -- as protected own variables are implicitly initialized.
      while not Dictionary.IsNullIterator (InitOwnVarList)
      loop
         InitOwnVarSym := Dictionary.CurrentSymbol (InitOwnVarList);
         if Dictionary.IsConcreteOwnVariable (InitOwnVarSym) then
            if not Dictionary.VariableIsInitialized (InitOwnVarSym) then
               ErrorHandler.UsageError (ErrorHandler.UndefinedVar,
                                        NodePosition (Node),
                                        InitOwnVarSym,
                                        Scope);

               ErrorHandler.UsageError (ErrorHandler.UndefinedExport,
                                        NodePosition (Node),
                                        InitOwnVarSym,
                                        Scope);
            end if;

         else --abstract own var
            ConstituentList := Dictionary.FirstConstituent (InitOwnVarSym);
            while not Dictionary.IsNullIterator (ConstituentList) loop
               ConstituentSym := Dictionary.CurrentSymbol (ConstituentList);
               if (not Dictionary.IsOwnVariable (ConstituentSym)) and then
                  (not Dictionary.VariableIsInitialized (ConstituentSym)) and then
                  (Dictionary.GetConstituentMode (ConstituentSym) =
                   Dictionary.DefaultMode)
               then
                  ErrorHandler.UsageError (ErrorHandler.UndefinedVar,
                                           NodePosition (Node),
                                           ConstituentSym,
                                           Scope);

                  ErrorHandler.UsageError (ErrorHandler.UndefinedExport,
                                           NodePosition (Node),
                                           ConstituentSym,
                                           Scope);

               end if;
               ConstituentList := Dictionary.NextSymbol (ConstituentList);
            end loop;
         end if;
         InitOwnVarList := Dictionary.NextSymbol (InitOwnVarList);
      end loop;
   end CheckInitialization;

   ----------------------------------------------------------------

   procedure CheckClosingIdentifier (EndNameNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentNode;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndNameNode,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdentNode,
   --#                                        LexTokenManager.State,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      StartIdentNode    : STree.SyntaxNode;
      EndIdentNode      : STree.SyntaxNode;
   begin
      StartIdentNode := IdentNode;
      EndIdentNode := LastChildOf (EndNameNode);
      loop
         -- check identifiers at current positions:
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => NodeLexString (StartIdentNode),
                                                                 Lex_Str2 => NodeLexString (EndIdentNode)) /= LexTokenManager.Str_Eq then
            ErrorHandler.SemanticError (58,
                                        ErrorHandler.NoReference,
                                        NodePosition (EndIdentNode),
                                        NodeLexString (StartIdentNode));
            exit;
         end if;

         -- move on to next identifiers:
         StartIdentNode := Next_Sibling (ParentNode (StartIdentNode));
         EndIdentNode   := Next_Sibling (ParentNode (EndIdentNode));

         -- finished when both exhausted:
         exit when SyntaxNodeType (StartIdentNode) /=
            SPSymbols.identifier and
            SyntaxNodeType (EndIdentNode) /=
            SPSymbols.identifier;

         -- check if only one exhausted (length mismatch):
         if SyntaxNodeType (StartIdentNode) /=
            SPSymbols.identifier or
            SyntaxNodeType (EndIdentNode) /=
            SPSymbols.identifier
         then
            ErrorHandler.SemanticError (615,
                                        ErrorHandler.NoReference,
                                        NodePosition (EndNameNode),
                                        LexTokenManager.Null_String);
            exit;
         end if;
      end loop;
   end CheckClosingIdentifier;

   ----------------------------------------------------------------

   procedure CheckOwnTasksDeclared (PackSym : in Dictionary.Symbol;
                                    Node    : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Node,
   --#                                        PackSym,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      It         : Dictionary.Iterator;
      OwnTaskSym : Dictionary.Symbol;
   begin
      It := Dictionary.FirstOwnTask (PackSym);
      while not Dictionary.IsNullIterator (It) loop
         OwnTaskSym := Dictionary.CurrentSymbol (It);
         if not Dictionary.IsDeclared (OwnTaskSym) then
            ErrorHandler.SemanticError (931,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (OwnTaskSym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckOwnTasksDeclared;

begin --up_wf_package_body
   PackSym := Dictionary.GetRegion (Scope);
   IdentNode := LastChildOf (Node);
   NextNode := Child_Node (LastSiblingOf (Child_Node (Node)));
   RefNode := Next_Sibling (Child_Node (Node));
   if SyntaxNodeType (RefNode) /= SPSymbols.refinement_definition then
      RefNode := STree.NullNode;
   end if;

   if SyntaxNodeType (NextNode) /= SPSymbols.hidden_part then
      NextNode := Next_Sibling (NextNode);
      if SyntaxNodeType (NextNode) = SPSymbols.declarative_part then
         NextNode := Next_Sibling (NextNode);
      end if;
      if SyntaxNodeType (NextNode) = SPSymbols.package_initialization then
         NextNode  := Child_Node (NextNode);
         if SyntaxNodeType (NextNode) = SPSymbols.hidden_part then
            EndDesigNode := STree.NullNode;
            NodeToReportErrors := NextNode;
         else -- should be sequence of statements
            EndDesigNode := Next_Sibling (NextNode);
            NodeToReportErrors := EndDesigNode;
         end if;
      else --no pack init part provided
         EndDesigNode := NextNode;
         NodeToReportErrors := EndDesigNode;
         CheckInitialization (PackSym,
                              Child_Node (EndDesigNode),
                              Scope);
      end if;

      if EndDesigNode /= STree.NullNode then
         CheckClosingIdentifier (EndDesigNode);
      end if;

      CheckAnnouncedTypesDeclared (PackSym,
                                   Scope,
                                   Child_Node (NodeToReportErrors));

      CheckOwnVariablesDeclared (PackSym,
                                 Child_Node (NodeToReportErrors),
                                 Scope);

      CheckIntegrityOfRefinements (PackSym,
                                   Child_Node (NodeToReportErrors),
                                   Scope);

      CheckExportedSubprogramsDeclared (PackSym,
                                        Child_Node (NodeToReportErrors));
      CheckVisibleTaskTypesDeclared (PackSym,
                                     Child_Node (NodeToReportErrors));
      CheckPrivateTaskTypesDeclared (PackSym,
                                     Child_Node (NodeToReportErrors));
      CheckVisibleProtectedTypesDeclared (PackSym,
                                          Child_Node (NodeToReportErrors));
      CheckPrivateProtectedTypesDeclared (PackSym,
                                          Child_Node (NodeToReportErrors));
      CheckEmbedBodies (PackSym,
                        Child_Node (NodeToReportErrors));

      CheckOwnTasksDeclared (PackSym,
                             Child_Node (NodeToReportErrors));

   end if;

   --  The cross-references for the refinement are generated after the
   --  full semantic analysis of the package body because we need to
   --  know if a refinement constituent is actually an abstract own
   --  variable or a body concrete own variable. If it is an abstract
   --  own variable, the refinement constituent is considered as a
   --  declaration, if it is a body concrete own variable, the
   --  refinement constituent is a usage of the body concrete variable
   --  that will be declared later in the package body.
   if ErrorHandler.Generate_SLI and then
     RefNode /= STree.NullNode then
      SLI.Generate_Xref_Refinement (Comp_Unit  => ContextManager.Ops.CurrentUnit,
                                    Parse_Tree => RefNode,
                                    Scope      => Dictionary.LocalScope (PackSym));
   end if;

   Scope := Dictionary.GetEnclosingScope (Scope);

end up_wf_package_body;
