-- $Id: sem-compunit-up_wf_aggregate_choice.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)

procedure up_wf_aggregate_choice (Node   : in     STree.SyntaxNode;
                                  Scope  : in     Dictionary.Scopes;
                                  EStack : in out ExpStack.ExpStackType)
is
   type CaseChoiceSorts is (SingleExpression, ExplicitRange, RangeConstraint);

   CaseChoiceSort : CaseChoiceSorts;
   FirstNode,
   SecondNode     : STree.SyntaxNode;
   NameExp,
   FirstResult,
   SecondResult   : ExpRecord;

   SemanticErrorsFound : Boolean := False;
   ChoiceLowerMathsValue : Maths.Value;
   ChoiceUpperMathsValue : Maths.Value
      := Maths.NoValue;  -- actually ineffectual, but stops spurious flowerrs
   IsARange : Boolean;
   IndexTypeSymbol     : Dictionary.Symbol;
   IndexTypeLowerBound : TypTypeBound;
   IndexTypeUpperBound : TypTypeBound;
   AggregateFlags      : TypAggFlags;
   EntryCounter        : Natural;
   CompleteRec         : CompleteCheck.T;
   ChoiceLowerBound,
   ChoiceUpperBound      : TypTypeBound
   := UnknownTypeBound;  -- actually ineffectual, but stops spurious flowerrs

   LowerBoundUnknown     : Boolean;
   UpperBoundUnknown     : Boolean
      := True;  -- actually ineffectual, but stops spurious flowerrs
   LowerBoundOutOfRange  : Boolean;
   UpperBoundOutOfRange  : Boolean
      := True;  -- actually ineffectual, but stops spurious flowerrs
   OutOfRangeSeen      : Boolean;
   OverlapSeen         : CompleteCheck.TypOverlapState;
   BothChoiceBoundsKnown : Boolean
      := False;  -- actually ineffectual, but stops spurious flowerrs
   RangeConstraintLowerBound     : TypTypeBound;
   RangeConstraintUpperBound     : TypTypeBound;

   ------------------------------------------------------------------------

   procedure ConvertChoiceBound (MathsValue      : in     Maths.Value;
                                 Bound           :    out TypTypeBound;
                                 UnknownBound    :    out Boolean;
                                 BoundOutOfRange :    out Boolean)
      --# derives Bound,
      --#         BoundOutOfRange,
      --#         UnknownBound    from MathsValue;
   is
      Int        : Integer;
      MathsError : Maths.ErrorCode;
   begin
      if Maths.HasNoValue (MathsValue) then
         Bound := TypTypeBound'(Value     => 0,
                                IsDefined => False);
         UnknownBound := True;
         BoundOutOfRange := False;
      else
         Maths.ValueToInteger (MathsValue, Int, MathsError);
         if MathsError = Maths.NoError then
            Bound := TypTypeBound'(Value     => Int,
                                   IsDefined => True);
            UnknownBound := False;
            BoundOutOfRange := False;
         else
            Bound := TypTypeBound'(Value     => 0,
                                   IsDefined => False);
            UnknownBound := False;
            BoundOutOfRange := True;
         end if;
      end if;
   end ConvertChoiceBound;

   ------------------------------------------------------------------------
   -- note: returns True if any of the bounds is undefined, unless the
   -- choice is not a range, in which case, ChoiceUpper is unused
   function IsChoiceInRange (ChoiceLower   : TypTypeBound;
                             ChoiceUpper   : TypTypeBound;
                             ChoiceIsRange : Boolean;
                             RangeLower    : TypTypeBound;
                             RangeUpper    : TypTypeBound) return Boolean
   is
      Result : Boolean;
   begin
      if (ChoiceLower.IsDefined and RangeLower.IsDefined
          and ChoiceLower.Value < RangeLower.Value)
         or
         (ChoiceLower.IsDefined and RangeUpper.IsDefined
          and ChoiceLower.Value > RangeUpper.Value)
         or
         (ChoiceIsRange and ChoiceUpper.IsDefined and RangeUpper.IsDefined
          and ChoiceUpper.Value > RangeUpper.Value)
      then
         Result := False;
      else
         Result := True;
      end if;

      return Result;
   end IsChoiceInRange;

   ------------------------------------------------------------------------

   function IsRangeEmpty (RangeLower : TypTypeBound;
                          RangeUpper : TypTypeBound) return Boolean
   is
      Result : Boolean;
   begin
      if RangeLower.IsDefined and RangeUpper.IsDefined and
         RangeLower.Value > RangeUpper.Value
      then
         Result := True;
      else
         Result := False;
      end if;

      return Result;
   end IsRangeEmpty;

   -----------------------------------------------------------------------
   procedure ConvertBooleanMathsValue (Value : in out Maths.Value)
      --# derives Value from *;
   is
   begin
      if Value = Maths.FalseValue then
         Value := Maths.ZeroInteger;
      elsif Value = Maths.TrueValue then
         Value := Maths.OneInteger;
      end if;
   end ConvertBooleanMathsValue;

   -----------------------------------------------------------------------


begin  -- up_wf_aggregate_choice
   -- ASSUME Node = aggregate_choice
   -- ASSUME aggregate is array aggregate with named association
   AggregateStack.Pop (IndexTypeSymbol,
                       IndexTypeLowerBound,
                       IndexTypeUpperBound,
                       AggregateFlags,
                       EntryCounter,
                       CompleteRec);

   FirstNode := Child_Node (Node);
   SecondNode := Next_Sibling (FirstNode);
   if SecondNode = STree.NullNode then
      CaseChoiceSort := SingleExpression;
   elsif SyntaxNodeType (SecondNode) =
      SPSymbols.simple_expression or
      SyntaxNodeType (SecondNode) =
      SPSymbols.annotation_simple_expression
   then
      CaseChoiceSort := ExplicitRange;
   else
      CaseChoiceSort := RangeConstraint;
   end if;

   --# assert True; -- for RTC generation

   case CaseChoiceSort is
      when SingleExpression =>
         ExpStack.Pop (FirstResult, EStack);
         ExpStack.Pop (NameExp, EStack);
         if Dictionary.IsUnknownTypeMark (FirstResult.TypeSymbol) then
            null;
         elsif Dictionary.CompatibleTypes (Scope,
                                           FirstResult.TypeSymbol,
                                           Dictionary.GetArrayIndex (NameExp.TypeSymbol,
                                                                     NameExp.ParamCount))
         then
            if not FirstResult.IsStatic then
               ErrorHandler.SemanticError (36,
                                           1,
                                           NodePosition (FirstNode),
                                           LexTokenManager.Null_String);
               SemanticErrorsFound := True;
            end if;
         else
            ErrorHandler.SemanticError (38,
                                        ErrorHandler.NoReference,
                                        NodePosition (FirstNode),
                                        LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         -- code to work out whether we have a single choice or a
         -- range and to collect the appropriate values
         -- note that these will be nonsense if semantic errors have been found
         ChoiceLowerMathsValue := FirstResult.Value;
         if FirstResult.IsARange then
            IsARange := True;
            ChoiceUpperMathsValue := FirstResult.RangeRHS;
         else
            IsARange := False;
         end if;
         NameExp.ErrorsInExpression := SemanticErrorsFound or
            NameExp.ErrorsInExpression or
            FirstResult.ErrorsInExpression;

      when ExplicitRange =>
         ExpStack.Pop (SecondResult, EStack);
         ExpStack.Pop (FirstResult, EStack);
         ExpStack.Pop (NameExp, EStack);
         if not Dictionary.CompatibleTypes (Scope,
                                            FirstResult.TypeSymbol,
                                            SecondResult.TypeSymbol)
         then
            ErrorHandler.SemanticError (42,
                                        ErrorHandler.NoReference,
                                        NodePosition (SecondNode),
                                        LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         elsif not Dictionary.CompatibleTypes (Scope,
                                               FirstResult.TypeSymbol,
                                               Dictionary.GetArrayIndex (NameExp.TypeSymbol,
                                                                         NameExp.ParamCount))
         then
            ErrorHandler.SemanticError (106,
                                        ErrorHandler.NoReference,
                                        NodePosition (FirstNode),
                                        LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         if not (FirstResult.IsStatic and SecondResult.IsStatic) then
            ErrorHandler.SemanticError (45,
                                        1,
                                        NodePosition (FirstNode),
                                        LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         -- code to collect the appropriate values for the extent of the range
         -- note that these will be nonsense if semantic errors have been found#
         ChoiceLowerMathsValue := FirstResult.Value;
         ChoiceUpperMathsValue := SecondResult.Value;
         IsARange := True;
         NameExp.ErrorsInExpression := SemanticErrorsFound or
            NameExp.ErrorsInExpression or
            FirstResult.ErrorsInExpression  or
            SecondResult.ErrorsInExpression;

      when RangeConstraint =>
         ExpStack.Pop (SecondResult, EStack);
         ExpStack.Pop (FirstResult, EStack);
         ExpStack.Pop (NameExp, EStack);
         if not Dictionary.CompatibleTypes (Scope,
                                            FirstResult.TypeSymbol,
                                            SecondResult.TypeSymbol)
         then
            ErrorHandler.SemanticError (106,
                                        ErrorHandler.NoReference,
                                        NodePosition (SecondNode),
                                        LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         elsif not Dictionary.CompatibleTypes (Scope,
                                               FirstResult.TypeSymbol,
                                               Dictionary.GetArrayIndex (NameExp.TypeSymbol,
                                                                         NameExp.ParamCount))
         then
            ErrorHandler.SemanticError (38,
                                        ErrorHandler.NoReference,
                                        NodePosition (FirstNode),
                                        LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         if not (FirstResult.IsConstant and FirstResult.IsARange) then
            ErrorHandler.SemanticError (95,
                                        ErrorHandler.NoReference,
                                        NodePosition (FirstNode),
                                        LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         if not SecondResult.IsStatic then
            ErrorHandler.SemanticError (45,
                                        1,
                                        NodePosition (SecondNode),
                                        LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         -- code to collect the appropriate values for the extent of the range
         -- note that these will be nonsense if semantic errors have been found
         ChoiceLowerMathsValue := SecondResult.Value;
         ChoiceUpperMathsValue := SecondResult.RangeRHS;
         IsARange := True;
         NameExp.ErrorsInExpression := SemanticErrorsFound or
            NameExp.ErrorsInExpression or
            FirstResult.ErrorsInExpression  or
            SecondResult.ErrorsInExpression;
         -- somewhere need to check that SecondResult range is within the type
         -- given by FirstResult
   end case;

   ExpStack.Push (NameExp, EStack);

   --# assert True;

   if not SemanticErrorsFound then
      ConvertBooleanMathsValue (ChoiceLowerMathsValue);
      ConvertChoiceBound (ChoiceLowerMathsValue,
                          ChoiceLowerBound,
                          LowerBoundUnknown,
                          LowerBoundOutOfRange);
      if IsARange then
         ConvertBooleanMathsValue (ChoiceUpperMathsValue);  -- CUMV always defined here
         ConvertChoiceBound (ChoiceUpperMathsValue,
                             ChoiceUpperBound,
                             UpperBoundUnknown,
                             UpperBoundOutOfRange);
      else
         ChoiceUpperBound := UnknownTypeBound;
      end if;

      --# assert True;

      if LowerBoundOutOfRange or (IsARange and then -- UBOOR always defined here
                                  UpperBoundOutOfRange) then
         BothChoiceBoundsKnown := False;
         ErrorHandler.SemanticWarning (305,
                                       NodePosition (FirstNode),
                                       LexTokenManager.Null_String);
      elsif LowerBoundUnknown or  -- UBOOR always defined here
         (IsARange and then UpperBoundUnknown) then
         BothChoiceBoundsKnown := False;
         CompleteRec.Undeterminable := True;
         ErrorHandler.SemanticWarning (200,
                                       NodePosition (FirstNode),
                                       LexTokenManager.Null_String);
      else
         BothChoiceBoundsKnown := True;
      end if;

      --# assert True;

      if BothChoiceBoundsKnown then
         -- check the case choice lies within controlling type
         if not IsChoiceInRange (ChoiceLowerBound,
                                 ChoiceUpperBound,
                                 IsARange,
                                 IndexTypeLowerBound,
                                 IndexTypeUpperBound)
         then
            if CaseChoiceSort = RangeConstraint then
               ErrorHandler.SemanticError (410,
                                           ErrorHandler.NoReference,
                                           NodePosition (SecondNode),
                                           LexTokenManager.Null_String);
            else
               ErrorHandler.SemanticError (410,
                                           ErrorHandler.NoReference,
                                           NodePosition (FirstNode),
                                           LexTokenManager.Null_String);
            end if;
            SemanticErrorsFound := True;
         elsif IsARange and IsRangeEmpty (ChoiceLowerBound,
                                          ChoiceUpperBound)
         then
            if CaseChoiceSort = RangeConstraint then
               ErrorHandler.SemanticError (409,
                                           ErrorHandler.NoReference,
                                           NodePosition (SecondNode),
                                           LexTokenManager.Null_String);
            else
               ErrorHandler.SemanticError (409,
                                           ErrorHandler.NoReference,
                                           NodePosition (FirstNode),
                                           LexTokenManager.Null_String);
            end if;
            SemanticErrorsFound := True;
         end if;

         -- check the case choice lies within RangeConstraint type
         if CaseChoiceSort = RangeConstraint then
            GetTypeBounds (FirstResult.TypeSymbol,
                           RangeConstraintLowerBound,
                           RangeConstraintUpperBound);

            if not IsChoiceInRange (ChoiceLowerBound,
                                    ChoiceUpperBound,
                                    IsARange,
                                    RangeConstraintLowerBound,
                                    RangeConstraintUpperBound)
            then
               ErrorHandler.SemanticError (413,
                                           ErrorHandler.NoReference,
                                           NodePosition (SecondNode),
                                           LexTokenManager.Null_String);
               SemanticErrorsFound := True;
            end if;
         end if;
      end if;
   end if;

   --# assert True;

   if (not SemanticErrorsFound) -- BCBK always defined if we reach it
      and then BothChoiceBoundsKnown
      and then (AggregateFlags.CheckCompleteness or AggregateFlags.CheckOverlap)
   then
      if IsARange then
         CompleteCheck.SeenRange (CompleteRec,
                                  ChoiceLowerBound.Value,
                                  ChoiceUpperBound.Value,
                                  OutOfRangeSeen,
                                  OverlapSeen);
      else
         CompleteCheck.SeenElement (CompleteRec, -- flow error expected
                                    ChoiceLowerBound.Value,
                                    OutOfRangeSeen,
                                    OverlapSeen);
      end if;
      if OutOfRangeSeen then
         AggregateFlags.OutOfRangeSeen := True;
      end if;
      if AggregateFlags.CheckOverlap and OverlapSeen = CompleteCheck.Overlap then
         ErrorHandler.SemanticError (407,
                                     ErrorHandler.NoReference,
                                     NodePosition (FirstNode),
                                     LexTokenManager.Null_String);
         SemanticErrorsFound := True;
      end if;
   end if;

   --# assert True;

   if SemanticErrorsFound then
      AggregateFlags.CheckCompleteness := False;
   end if;

   AggregateStack.Push (IndexTypeSymbol,
                        IndexTypeLowerBound,
                        IndexTypeUpperBound,
                        AggregateFlags,
                        EntryCounter,
                        CompleteRec);

end up_wf_aggregate_choice;
