-- $Id: sem-compunit-down_wf_aggregate.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)

procedure down_wf_aggregate (Node         : in     STree.SyntaxNode;
                             Scope        : in     Dictionary.Scopes;
                             NextNode     :    out STree.SyntaxNode;
                             EStack       : in out ExpStack.ExpStackType;
                             HeapParam    : in out Lists.List_Heap;
                             IsAnnotation : in     Boolean)
is
   QUAL_LOOKUP : constant Annotation_Symbol_Table :=
      Annotation_Symbol_Table'(False => SPSymbols.qualified_expression,
                               True  => SPSymbols.annotation_qualified_expression);

   HasOthersPart               : Boolean;
   AssociationType             : TypAggAssociationType;
   NameExp                     : ExpRecord;
   OthersNode                  : STree.SyntaxNode;
   Ptr                         : Lists.List;
   UnknownOrIndiscreteFound    : Boolean;
   IndexTypeMark               : Dictionary.Symbol;
   ErrorFound                  : Boolean := False;

   --------------------------------------------------------------------

   procedure CreateAggregateStackEntry (IndexTypeSymbol : in Dictionary.Symbol;
                                        AssociationType : in TypAggAssociationType;
                                        HasOthersPart   : in Boolean;
                                        Scope           : in Dictionary.Scopes)
   -- this procedure discriminates between the cases listed in S.P0468.53.11
   -- and sets up the stack entry accordingly
   --
   -- preconditions to entry to this procedure:
   --     aggregate is an array aggregate
   --
   -- NB aggregate may be a lone others clause
   --# global in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out AggregateStack.State;
   --# derives AggregateStack.State from *,
   --#                                   AssociationType,
   --#                                   Dictionary.Dict,
   --#                                   HasOthersPart,
   --#                                   IndexTypeSymbol,
   --#                                   LexTokenManager.State,
   --#                                   Scope;
   is
      CompleteCheckRangeFrom : Integer;
      CompleteCheckRangeTo : Integer;
      CompleteCheckRangeState : CompleteCheck.TypRangeState;
      TypeLowerBound : TypTypeBound;
      TypeUpperBound : TypTypeBound;
      CompleteRec : CompleteCheck.T;
      CheckCompleteness : Boolean;
      WarnNoOthers : Boolean;
      CheckOverlap : Boolean;
      SignalOutOfRange : Boolean;
   begin
      -- if index type unknown or not discrete then cannot do much at all
      if Dictionary.IsUnknownTypeMark (IndexTypeSymbol)
         or not Dictionary.IsDiscreteTypeMark (IndexTypeSymbol, Scope)
      then
         TypeLowerBound := UnknownTypeBound;
         TypeUpperBound := UnknownTypeBound;
         CompleteCheckRangeFrom := -(ExaminerConstants.CompleteCheckSize / 2);
         CompleteCheckRangeTo := (CompleteCheckRangeFrom + ExaminerConstants.CompleteCheckSize) - 1;
         --NB we 'know' that CompleteCheckRangeState will return RangeDoesFit,
         --   so the value is ignored, giving a flow error
         --# accept Flow, 10, CompleteCheckRangeState, "Expected ineffective assignment to CompleteCheckRangeState";
         CompleteCheck.Init (CompleteRec,               -- expect flow error
                             CompleteCheckRangeFrom,
                             CompleteCheckRangeTo,
                             CompleteCheckRangeState);
         --# end accept;
         CheckCompleteness := False;
         WarnNoOthers := False;
         SignalOutOfRange := False;
         if     Dictionary.IsUnknownTypeMark (IndexTypeSymbol)
            and AssociationType = AggregateIsNamed
         then
            CheckOverlap := True;
         else
            CheckOverlap := False;
         end if;
      else
         -- get bounds from dictionary
         GetTypeBounds (IndexTypeSymbol, TypeLowerBound, TypeUpperBound);

         if not (TypeLowerBound.IsDefined and TypeUpperBound.IsDefined) then
            -- one or other bound is unknown to the dictionary
            -- set flags accordingly
            CheckCompleteness := False;
            WarnNoOthers := True;
            if AssociationType = AggregateIsPositional then
               CheckOverlap := False;
               SignalOutOfRange :=  False;
            else
               CheckOverlap := True;
               SignalOutOfRange :=  True;
            end if;

            -- set up range for completeness checker
            -- if both bounds unknown use symmetric range
            if (not TypeLowerBound.IsDefined) and (not TypeUpperBound.IsDefined) then
               CompleteCheckRangeFrom := -(ExaminerConstants.CompleteCheckSize / 2);
               CompleteCheckRangeTo   := (CompleteCheckRangeFrom
                  + ExaminerConstants.CompleteCheckSize) - 1;
               -- otherwise use range extending from known bound
            elsif TypeLowerBound.IsDefined then
               CompleteCheckRangeFrom := TypeLowerBound.Value;
               CompleteCheckRangeTo   := (CompleteCheckRangeFrom
                  + ExaminerConstants.CompleteCheckSize) - 1;
            else  -- TypeUpperBound.IsDefined
               CompleteCheckRangeTo   := TypeUpperBound.Value;
               CompleteCheckRangeFrom := (CompleteCheckRangeTo
                  - ExaminerConstants.CompleteCheckSize) + 1;
            end if;
            --NB we 'know' that CompleteCheckRangeState will return RangeDoesFit,
            --   so the value is ignored, giving a flow error
            --# accept Flow, 10, CompleteCheckRangeState, "Expected ineffective assignment to CompleteCheckRangeState";
            CompleteCheck.Init (CompleteRec,               -- expect flow error
                                CompleteCheckRangeFrom,
                                CompleteCheckRangeTo,
                                CompleteCheckRangeState);
            --# end accept;

         else
            -- both bounds known to dictionary
            -- set up completeness checker
            CompleteCheck.Init (CompleteRec,
                                TypeLowerBound.Value,
                                TypeUpperBound.Value,
                                CompleteCheckRangeState);

            -- for positional association, the question of whether the
            -- type is too big for the completeness checker is irrelevant
            if AssociationType = AggregateIsPositional then
               CheckCompleteness := True;
               WarnNoOthers := False;
               CheckOverlap := False;
               SignalOutOfRange := False;
            else
               -- set flags according to whether range fits in completeness checker
               if CompleteCheckRangeState = CompleteCheck.RangeDoesFit then
                  CheckCompleteness := True;
                  WarnNoOthers := False;
                  CheckOverlap := True;
                  SignalOutOfRange := False;
               else
                  CheckCompleteness := False;
                  WarnNoOthers := True;
                  CheckOverlap := True;
                  SignalOutOfRange := True;
               end if;
            end if;
         end if;
      end if;

      AggregateStack.Push (IndexTypeSymbol,
                           TypeLowerBound,
                           TypeUpperBound,
                           TypAggFlags'(CheckCompleteness => CheckCompleteness,
                                        WarnNoOthers => WarnNoOthers,
                                        CheckOverlap => CheckOverlap,
                                        SignalOutOfRange => SignalOutOfRange,
                                        OutOfRangeSeen => False,
                                        MoreEntriesThanNatural => False,
                                        HasOthersPart => HasOthersPart,
                                        AssociationType => AssociationType),
                           0,
                           CompleteRec);

   end CreateAggregateStackEntry;

   --------------------------------------------------------------------
begin
   -- code to determine association type enhanced to detect the
   -- occurrence of a 'lone' others clause, and moved here so that
   -- the information is available to anonymous aggregates
   -- code determining presence of others part moved here for same reason
   case SyntaxNodeType (Child_Node (Child_Node (Node))) is
      when SPSymbols.positional_association |
         SPSymbols.annotation_positional_association =>
         AssociationType := AggregateIsPositional;
      when SPSymbols.named_association |
         SPSymbols.annotation_named_association =>
         AssociationType := AggregateIsNamed;
      when others =>
         AssociationType := AggregateIsLoneOthers;
   end case;

   --# assert True; -- for RTC generation

   OthersNode := Child_Node (Child_Node (Child_Node (Node)));

   -- PNA observation 9/3/02: I can't see how this first if part can ever be true given the current
   -- grammar.  The else if part may be.  Expression should be aggregate_or_expression.
   if SyntaxNodeType (OthersNode) =
      SPSymbols.expression or else
      SyntaxNodeType (OthersNode) =
      SPSymbols.annotation_expression
   then
      HasOthersPart := True;
   else

      OthersNode := Next_Sibling (OthersNode);

      if SyntaxNodeType (OthersNode) =
         SPSymbols.aggregate_or_expression or
         SyntaxNodeType (OthersNode) =
         SPSymbols.annotation_aggregate_or_expression
      then
         HasOthersPart := True;
      else
         HasOthersPart := False;
      end if;
   end if;

   --# assert True; -- for RTC generation

   HasOthersPart := HasOthersPart or
      (AssociationType = AggregateIsLoneOthers);

   if SyntaxNodeType (ParentNode (Node)) =
      QUAL_LOOKUP (IsAnnotation)
   then --this is a top level, not embedded, aggregate
      ExpStack.Pop (NameExp, EStack);

      case NameExp.Sort is
         when IsTypeMark =>
            NameExp.IsConstant := True;
            if Dictionary.IsArrayTypeMark (NameExp.TypeSymbol, Scope) then
               if Dictionary.IsUnconstrainedArrayType (NameExp.TypeSymbol) and
                  not IsAnnotation then
                  -- Qualified aggregates of unconstrained array types only permitted in
                  -- annotation context.
                  -- So you are allowed to say, for example, "post X = T'(others => 0)" for a
                  -- subprogram that initializes an unconstrained array or "post X /= Y" where
                  -- both X and Y are unconstrained array types.
                  ErrorFound := True;
                  ErrorHandler.SemanticError (39,
                                              ErrorHandler.NoReference,
                                              NodePosition (ParentNode (Node)),
                                              LexTokenManager.Null_String);
               end if;

               --# assert True; -- for RTC generation

               NameExp.ParamCount := 1;   --used to record depth of dimension reached
               CreateAggregateStackEntry (Dictionary.GetArrayIndex (NameExp.TypeSymbol,
                                                                    1),
                                          AssociationType,
                                          HasOthersPart,
                                          Scope);
               -- check types of all array dimensions, and warn if checking
               -- may be incomplete because any of the index types is unknown
               -- or indiscrete
               UnknownOrIndiscreteFound := False;
               for I in Positive range
                  1 .. Dictionary.GetNumberOfDimensions (NameExp.TypeSymbol)
               loop

                  --# assert True; -- for RTC generation

                  IndexTypeMark := Dictionary.GetArrayIndex (NameExp.TypeSymbol, I);
                  if Dictionary.IsUnknownTypeMark (IndexTypeMark)
                     or else
                     (not Dictionary.IsDiscreteTypeMark (IndexTypeMark, Scope))
                  then
                     UnknownOrIndiscreteFound := True;
                  end if;
               end loop;

               --# assert True; -- for RTC generation

               if UnknownOrIndiscreteFound then
                  ErrorFound := True;
                  ErrorHandler.SemanticWarning (307,
                                                NodePosition (ParentNode (Node)),
                                                LexTokenManager.Null_String);
               end if;
               NameExp.ErrorsInExpression := NameExp.ErrorsInExpression or ErrorFound;
               ExpStack.Push (NameExp, EStack);
               NextNode := Child_Node (Node);

            elsif Dictionary.IsRecordTypeMark (NameExp.TypeSymbol, Scope) then
               if HasOthersPart then
                  ExpStack.Push (UnknownTypeRecord, EStack);
                  NextNode := STree.NullNode;
                  ErrorHandler.SemanticError (53,
                                              ErrorHandler.NoReference,
                                              NodePosition (OthersNode),
                                              LexTokenManager.Null_String);

               elsif Dictionary.TypeIsExtendedTagged (NameExp.TypeSymbol) and then
                 Dictionary.ExtendedTaggedHasPrivateAncestors (NameExp.TypeSymbol, Scope) then
                  ExpStack.Push (UnknownTypeRecord, EStack);
                  NextNode := STree.NullNode;
                  ErrorHandler.SemanticErrorSym (833,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (ParentNode (Node)),
                                                 NameExp.TypeSymbol,
                                                 Scope);

               else -- OK, not illegal tagged record and has no others clause

                  -- NameExp.TypeSymbol here might denote a record subtype.  For subsequent
                  -- checking of the aggregate, we need the root record type, so...
                  NameExp.TypeSymbol := Dictionary.GetRootType (NameExp.TypeSymbol);

                  if AssociationType = AggregateIsNamed then
                     CreateNameList (Ptr, HeapParam);
                     NameExp.ParamList := Ptr;
                     ExpStack.Push (NameExp, EStack);
                     NextNode := Child_Node (Node);
                  else --positional association
                     NameExp.ParamCount := 0;
                     ExpStack.Push (NameExp, EStack);
                     NextNode := Child_Node (Node);
                  end if;
               end if;
            else --not a record or array
               ExpStack.Push (UnknownTypeRecord, EStack);
               ErrorHandler.SemanticError (33,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           LexTokenManager.Null_String);
               NextNode := STree.NullNode;
            end if;

         when IsUnknown =>
            --illegal name prefix but we can continue walk to check internal
            --validity of any expressions that follow.
            ExpStack.Push (UnknownTypeRecord, EStack);
            NextNode := Child_Node (Node);

         when others =>
            ExpStack.Push (UnknownTypeRecord, EStack);
            ErrorHandler.SemanticError (95,
                                        ErrorHandler.NoReference,
                                        NodePosition (ParentNode (Node)),
                                        LexTokenManager.Null_String);
            NextNode := Child_Node (Node);
      end case;

   else --it is an embedded aggregate of a multi-dim array
      ExpStack.Pop (NameExp, EStack);
      --increase depth of dimension count
      NameExp.ParamCount := NameExp.ParamCount + 1;
      CreateAggregateStackEntry (Dictionary.GetArrayIndex (NameExp.TypeSymbol,
                                                           NameExp.ParamCount),
                                 AssociationType,
                                 HasOthersPart,
                                 Scope);
      ExpStack.Push (NameExp, EStack);
      NextNode := Child_Node (Node);
   end if;
end down_wf_aggregate;
