-- $Id: requiredunits.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with
  SPARK_IO,
  CommandLineData; -- used for trace/debug statements only, not inherited

with SPSymbols;
with SystemErrors;

use type SPSymbols.SPSymbol;

package body RequiredUnits
is


   -- Following hidden trace routine enabled by -debug=u -----------------------------------
   procedure Trace (Msg : String)
   --# derives null from Msg;
   is
      --# hide Trace;
   begin
      if CommandLineData.Content.Debug.Units then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            Msg,
                            0);
      end if;
   end Trace;
   ------------------------------------------------------------------------------------------

   procedure Init (TopNode    : in     STree.SyntaxNode;
                   InheritPtr :    out STree.SyntaxNode;
                   UnitType   :    out ContextManager.UnitTypes;
                   UnitName   :    out LexTokenLists.Lists)
   is
      InheritPtrLocal,
      NamePtr         : STree.SyntaxNode;
      UnitTypeLocal   : ContextManager.UnitTypes;
      UnitNameLocal   : LexTokenLists.Lists;

   begin -- Init
         -- TopNode is  a compilation unit

      --Grammar:
      -- compilation_unit :
      --       context_clause library_unit
      --     | context_clause secondary_unit
      --     | library_unit
      --     | secondary_unit
      --     | apragma
      --     | ;
      Trace ("Entering RequiredUnits.Init");
      UnitTypeLocal := ContextManager.InvalidUnit;
      NamePtr := STree.NullNode;
      InheritPtrLocal := STree.Child_Node (TopNode);
      if STree.SyntaxNodeType (InheritPtrLocal) =
         SPSymbols.context_clause
      then
         InheritPtrLocal := STree.Next_Sibling (InheritPtrLocal);
      end if;
      -- any WITH clause [and/or use type clause] now skipped, we have library_unit or
      -- secondary unit node

      -- Handle library  units -----------------------------------
      if STree.SyntaxNodeType (InheritPtrLocal) =
         SPSymbols.library_unit
      then
         -- library_unit :
         --       package_declaration | private_package_declaration
         --     | main_program_declaration
         --     | private_generic_package_declaration
         --     | generic_declaration
         --     | generic_package_instantiation;

         Trace ("   library unit found");
         InheritPtrLocal := STree.Child_Node (InheritPtrLocal);
         if STree.SyntaxNodeType (InheritPtrLocal) =
            SPSymbols.package_declaration
            or else STree.SyntaxNodeType (InheritPtrLocal) =
            SPSymbols.private_package_declaration
         then -- handle package specification declaration
            Trace ("   package declaration found");
            UnitTypeLocal := ContextManager.PackageSpecification;
            InheritPtrLocal := STree.Child_Node (InheritPtrLocal);

            if STree.SyntaxNodeType (InheritPtrLocal) =
               SPSymbols.inherit_clause
            then
               NamePtr := STree.Next_Sibling (InheritPtrLocal);
            else
               NamePtr := InheritPtrLocal;
            end if;
            -- NamePtr is now at package_specification
            NamePtr := STree.Child_Node (NamePtr);

         elsif STree.SyntaxNodeType (InheritPtrLocal) =
            SPSymbols.main_program_declaration
         then -- handle main program
            Trace ("   main program found");
            UnitTypeLocal := ContextManager.MainProgram;
            InheritPtrLocal := STree.Child_Node (InheritPtrLocal);
            if STree.SyntaxNodeType (InheritPtrLocal) =
               SPSymbols.inherit_clause
            then
               NamePtr := STree.Next_Sibling (InheritPtrLocal);
            else
               NamePtr := InheritPtrLocal;
            end if;

            NamePtr := STree.Next_Sibling (NamePtr);
            NamePtr := STree.Child_Node (NamePtr);
            NamePtr := STree.Child_Node (NamePtr);
            -- need to go down again if func subprog
            if STree.SyntaxNodeType (NamePtr) = SPSymbols.designator then
               NamePtr := STree.Child_Node (NamePtr);
            end if;

         elsif STree.SyntaxNodeType (InheritPtrLocal) =
           SPSymbols.generic_declaration or
           STree.SyntaxNodeType (InheritPtrLocal) =
           SPSymbols.private_generic_package_declaration
         then -- handle generic declaration

            -- generic_declaration :
            --      generic_subprogram_declaration
            --    | generic_package_declaration ;
            --
            -- generic_subprogram_declaration :
            --      generic_formal_part subprogram_declaration ;
            --
            -- private_generic_package_declaration :
            --     inherit_clause RWprivate generic_formal_part package_specification semicolon
            --    |               RWprivate generic_formal_part package_specification semicolon ;
            --
            -- generic_package_declaration :
            --     inherit_clause generic_formal_part package_specification semicolon
            --    |               generic_formal_part package_specification semicolon ;

            Trace ("   generic declaration  found");
            UnitTypeLocal := ContextManager.GenericDeclaration;
            if STree.SyntaxNodeType (InheritPtrLocal) =
              SPSymbols.generic_declaration then
               InheritPtrLocal := STree.Child_Node (InheritPtrLocal);
               -- skipping over generic_subprogram_declaration or generic_package_declaration
            end if;
            InheritPtrLocal := STree.Child_Node (InheritPtrLocal);
            NamePtr := InheritPtrLocal;
            if STree.SyntaxNodeType (InheritPtrLocal) =
               SPSymbols.inherit_clause
            then
               NamePtr := STree.Next_Sibling (InheritPtrLocal);
            end if;
            NamePtr := STree.Next_Sibling (NamePtr);
            -- at this point, NamePtr points to either package_specification or subprogram_declaration
            -- InheritPtrLocal points at either an inherit clause or a generic formal part (in the latter
            -- case it will get set to null by a tidy up if statement below

            -- Now we need to find the unit's name
            if STree.SyntaxNodeType (NamePtr) = SPSymbols.package_specification then
               NamePtr := STree.Child_Node (NamePtr);
            elsif STree.SyntaxNodeType (NamePtr) = SPSymbols.subprogram_declaration then
               NamePtr := STree.LastChildOf (NamePtr);
               -- above skips procedure spec, function spec etc. and gets identifier node
            else
               SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree,
                                        "Unknown generic unit kind in RequiredUnits.Init");
            end if;

         elsif STree.SyntaxNodeType (InheritPtrLocal) =
           SPSymbols.generic_package_instantiation
         then -- handle generic package instantiation

            -- generic_package_instantiation :
            --      dotted_simple_name package_annotation (is new) identifier generic_actual_part semicolon
            --    | dotted_simple_name package_annotation (is new) identifier                     semicolon ;

            Trace ("   generic declaration found");
            UnitTypeLocal := ContextManager.GenericPackageInstantiation;
            NamePtr := STree.Child_Node (InheritPtrLocal);
            InheritPtrLocal := STree.NullNode;

            -- add elsif here for generic subprog bods

         else
            InheritPtrLocal := STree.NullNode;
         end if;

         -- Tidy up InheritPtrLocal; if it is not pointing at an inherit clause then set it to null
         if STree.SyntaxNodeType (InheritPtrLocal) =
            SPSymbols.inherit_clause
         then
            InheritPtrLocal := STree.Child_Node
              (STree.Child_Node (InheritPtrLocal));
         else
            InheritPtrLocal := STree.NullNode;
         end if;


         -- Handle secondary units -----------------------------------
      elsif STree.SyntaxNodeType (InheritPtrLocal) =
         SPSymbols.secondary_unit
      then
         -- secondary_unit :
         --       library_unit_body  | subunit ;

         Trace ("   secondary unit found");
         InheritPtrLocal := STree.Child_Node (InheritPtrLocal);

         if STree.SyntaxNodeType (InheritPtrLocal) =
            SPSymbols.library_unit_body
         then
            -- library_unit_body :
            --      package_body
            --      generic_subprogram_body ;

            InheritPtrLocal := STree.Child_Node (InheritPtrLocal);

            if STree.SyntaxNodeType (InheritPtrLocal) =
              SPSymbols.package_body
            then
               -- package_body :
               --       dotted_simple_name                       package_implementation semicolon
               --     | dotted_simple_name refinement_definition package_implementation semicolon ;
               Trace ("   package body found");
               UnitTypeLocal := ContextManager.PackageBody;
               NamePtr := STree.Child_Node (InheritPtrLocal);

            elsif STree.SyntaxNodeType (InheritPtrLocal) =
              SPSymbols.generic_subprogram_body
            then
               -- generic_subprogram_body :
               --      subprogram_body ;
               -- subprogram_body :
               --      procedure_specification procedure_annotation subprogram_implementation
               --    | function_specification  function_annotation  subprogram_implementation ;
               -- procedure_specification :
               --      identifier
               --    | identifier formal_part ;
               --
               -- function_specification :
               --      designator formal_part type_mark
               --    | designator             type_mark ;
               --
               -- designator :
               --      identifier ;
               Trace ("   generic subprogram body found");
               UnitTypeLocal := ContextManager.GenericSubprogramBody;
               NamePtr := STree.LastChildOf (InheritPtrLocal);
               InheritPtrLocal := STree.NullNode;

            else
               SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree,
                                        "Unknown library unit bdoy kind in RequiredUnits.Init");
            end if;

         elsif STree.SyntaxNodeType (InheritPtrLocal) =
            SPSymbols.subunit
         then
            -- subunit :
            --    parent_unit_name proper_body ;
            --
            -- proper_body :
            --       subprogram_body  | package_body | task_body | protected_body ;

            Trace ("   subunit found");
            UnitTypeLocal := ContextManager.SubUnit;
            NamePtr := STree.Child_Node
              (STree.Next_Sibling
                 (STree.Child_Node (InheritPtrLocal)));
            -- NamePtr is one of subprogram_body  | package_body | task_body | protected_body ;

            InheritPtrLocal := STree.Child_Node
              (STree.Child_Node (InheritPtrLocal));


            if STree.SyntaxNodeType (NamePtr) =
               SPSymbols.subprogram_body
            then
               -- subprogram_body :
               --       procedure_specification procedure_annotation subprogram_implementation
               --     | function_specification function_annotation subprogram_implementation ;
               Trace ("   subprogram body found");
               NamePtr := STree.Child_Node (STree.Child_Node (NamePtr));
               if STree.SyntaxNodeType (NamePtr) =
                  SPSymbols.designator -- function
               then
                  NamePtr := STree.Child_Node (NamePtr);
               end if;
               -- NamePtr points at identifier of subunit

            elsif STree.SyntaxNodeType (NamePtr) =
              SPSymbols.protected_body
            then
               -- protected_body :
               --       identifier protected_operation_item identifier semicolon ;
               Trace ("   protected body found");
               NamePtr := STree.Child_Node (NamePtr);

            elsif STree.SyntaxNodeType (NamePtr) =
              SPSymbols.task_body
            then
               -- task_body :
               --      identifier subprogram_implementation ;
               Trace ("   task body found");
               NamePtr := STree.Child_Node (NamePtr);

            else -- must be separate package body
               Trace ("   separate package body assumed");
               NamePtr := STree.Child_Node
                 (STree.Child_Node (NamePtr));
            end if;
         else
            InheritPtrLocal := STree.NullNode;
         end if;

      elsif STree.SyntaxNodeType (InheritPtrLocal) =
         SPSymbols.apragma
      then
         Trace ("   pragma found");
         InheritPtrLocal := STree.NullNode;
         UnitTypeLocal := ContextManager.InterUnitPragma;

      else
         InheritPtrLocal := STree.NullNode;
      end if;

      if NamePtr = STree.NullNode then
         UnitName := LexTokenLists.Null_List;
      elsif STree.SyntaxNodeType (NamePtr) =
         SPSymbols.dotted_simple_name then
         UnitNameLocal := LexTokenLists.Null_List;
         NamePtr := STree.LastChildOf (NamePtr);
         loop
            LexTokenLists.Append (UnitNameLocal,
                                  STree.NodeLexString (NamePtr));
            NamePtr := STree.Next_Sibling
              (STree.ParentNode (NamePtr));
            exit when STree.SyntaxNodeType (NamePtr) /=
               SPSymbols.identifier;
         end loop;
         UnitName := UnitNameLocal;
      else
         UnitNameLocal := LexTokenLists.Null_List;
         LexTokenLists.Append (UnitNameLocal,
                               STree.NodeLexString (NamePtr));
         UnitName := UnitNameLocal;
      end if;
      UnitType := UnitTypeLocal;
      InheritPtr := InheritPtrLocal;
   end Init;

   procedure CopyPUNtoList (PUNNode : in     STree.SyntaxNode;
                            List    :    out LexTokenLists.Lists)
   --# global in STree.Table;
   --# derives List from PUNNode,
   --#                   STree.Table;
   is
      NextNode : STree.SyntaxNode;
      LList    : LexTokenLists.Lists;
   begin
      NextNode := PUNNode;
      while STree.SyntaxNodeType (NextNode) /=
         SPSymbols.simple_name
      loop
         NextNode := STree.Child_Node (NextNode);
      end loop;
      LList := LexTokenLists.Null_List;
      loop
         LexTokenLists.Append (LList, STree.NodeLexString (
                                  STree.Child_Node (NextNode)));

         NextNode := STree.Next_Sibling (STree.ParentNode (NextNode));
         exit when STree.SyntaxNodeType (NextNode) /=
            SPSymbols.simple_name;
      end loop;
      List := LList;
   end CopyPUNtoList;

   procedure Next (InheritPtr   : in out STree.SyntaxNode;
                   RequiredUnit :    out LexTokenLists.Lists;
                   Found        :    out Boolean)
   is

      procedure CopyInheritedPackageToList (DottedNameNode : STree.SyntaxNode)
      --# global in     STree.Table;
      --#        in out RequiredUnit;
      --# derives RequiredUnit from *,
      --#                           DottedNameNode,
      --#                           STree.Table;
      is
         CurrentNode : STree.SyntaxNode;
      begin
         Trace ("Entering RequiredUnits.CopyInheritedPackageToList");
         CurrentNode := STree.LastChildOf (DottedNameNode);
         loop
            exit when CurrentNode = STree.NullNode;
            LexTokenLists.Append (List => RequiredUnit,
                                  Item => STree.NodeLexString (CurrentNode));
            CurrentNode := STree.Next_Sibling (STree.ParentNode (CurrentNode));
         end loop;
      end CopyInheritedPackageToList;

      -------------------------------

   begin -- Next
      Trace ("Entering RequiredUnits.Next");
      RequiredUnit := LexTokenLists.Null_List;
      if InheritPtr = STree.NullNode then
         Found := False;
      else
         Found := True;
         if STree.SyntaxNodeType (InheritPtr) =
           SPSymbols.simple_name
         then
            LexTokenLists.Append (List => RequiredUnit,
                                  Item => STree.NodeLexString (STree.Child_Node (InheritPtr)));
         elsif STree.SyntaxNodeType (InheritPtr) =
           SPSymbols.dotted_simple_name
         then
            CopyInheritedPackageToList (InheritPtr);
            InheritPtr := STree.NullNode;
         elsif STree.SyntaxNodeType (InheritPtr) =
           SPSymbols.inherit_clause_rep
         then
            CopyInheritedPackageToList (STree.Next_Sibling (InheritPtr));
            InheritPtr := STree.Child_Node (InheritPtr);
         elsif STree.SyntaxNodeType (InheritPtr) =
           SPSymbols.parent_unit_name
         then
            CopyPUNtoList (InheritPtr, RequiredUnit);
            InheritPtr := STree.NullNode;
         end if;
      end if;
   end Next;

end RequiredUnits;
