-- $Id: maths-parsestring.adb 14620 2009-10-28 13:40:30Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Maths)
-- This procedure breaks a SPARK literal string down into its constituent parts for
-- further processing

procedure ParseString (S                  : in     EStrings.T;
                       DecimalPointFound,
                       ExponentFound,
                       BaseFound          :    out Boolean;
                       Base               :    out Natural;
                       CoreString,
                       ExpString          :    out EStrings.T;
                       ExpSign            :    out Character;
                       PlacesAfterPoint   :    out EStrings.Lengths;
                       LegalSyntax        :    out Boolean)

   -- # derives DecimalPointFound from S &
   -- #         ExponentFound     from S &
   -- #         BaseFound         from S &
   -- #         Base              from S &
   -- #         CoreString        from S &
   -- #         ExpString         from S &
   -- #         ExpSign           from S &
   -- #         PlacesAfterPoint  from S &
   -- #         LegalSyntax       from S;

   -- NOTES
   --   BaseString will be set to "10" if BaseFound = FALSE
   --   ExpString is "0" if ExpFound = FALSE
   --   ExpSign is plus if ExpFound = FALSE
   --   PlacesAferPoint is 0 if DecimalPointFound = FALSE
   --   LegalSyntax only implies that string looks like an Ada literal

is

   EndIndicator : constant Character := Character'Val (0);

   type CharSet is array (Character) of Boolean;
   type ParserState is (Initial,
                        LeadingZero,
                        LeadingUnderline,
                        LaterDigits,
                        BaseStart,
                        BasedPart,
                        EndBase,
                        DecimalStart,
                        DecimalPart,
                        ExpStart,
                        ExpPart,
                        Finished);


   SyntaxOk     : Boolean;
   State        : ParserState;
   Acceptable,
   LegalDigit   : CharSet;
   DigitsRead,
   PlacesCount,
   InPtr        : EStrings.Lengths;
   Ch           : Character;
   Buffer       : EStrings.T;

   -----------------------------------------------

   procedure Caps (Ch : in out Character)
      --converts Characters a through f to upper case
      --# derives Ch from *;

   is
   begin
      if (Ch >= 'a') and (Ch <= 'f') then
         Ch := Character'Val ((Character'Pos (Ch)  -
                               Character'Pos ('a')) +
                               Character'Pos ('A'));
      end if;
   end Caps;

   -----------------------------------------------

   procedure Store (Ch : in Character)
   --# global in out Buffer;
   --# derives Buffer from *,
   --#                     Ch;
   is
      Success : Boolean;
   begin
      --# accept F, 10, Success, "Ineffective assignment here OK";
      EStrings.Append_Char (E_Str   => Buffer,
                            Ch      => Ch,
                            Success => Success);
      --# end accept;
      --# accept F, 33, Success, "Expect Success unused";
   end Store;

   -----------------------------------------------

   function LegalUnderline (Ok : CharSet) return Boolean
      --# global in InPtr;
      --#        in S;
   is
      Ch      : Character;
      OkSoFar : Boolean;
   begin
      OkSoFar := InPtr < EStrings.Get_Length (E_Str => S);
      if OkSoFar then
         Ch := EStrings.Get_Element (E_Str => S,
                                     Pos   => InPtr + 1);
         Caps (Ch);
         OkSoFar := Ok (Ch);
      end if;
      return OkSoFar;
   end LegalUnderline;

   -----------------------------------------------

   procedure CalcBase
      --# global in     Buffer;
      --#        in out SyntaxOk;
      --#           out Base;
      --#           out LegalDigit;
      --# derives Base,
      --#         LegalDigit from Buffer &
      --#         SyntaxOk   from *,
      --#                         Buffer;
   is
      LocalBase,
      i          : Natural;
   begin
      if EStrings.Get_Length (E_Str => Buffer) = 2 then
         LocalBase := 10 *
           Natural (CharToDigit (EStrings.Get_Element (E_Str => Buffer,
                                                       Pos   => 1))) +
           Natural (CharToDigit (EStrings.Get_Element (E_Str => Buffer,
                                                       Pos   => 2)));
      else -- must be 1
         LocalBase := Natural (CharToDigit (EStrings.Get_Element (E_Str => Buffer,
                                                                  Pos   => 1)));
      end if;

      Base := LocalBase;

      LegalDigit := CharSet'(Character => False);
      i := 0;
      while i < LocalBase loop
         if i <= 9 then
            LegalDigit (Character'Val (i + Character'Pos ('0'))) := True;
         else
            LegalDigit (Character'Val ((i + Character'Pos ('A')) - 10)) := True;
         end if;
         i := i + 1;
      end loop;

      if (LocalBase < 2) or (LocalBase > 16) then
         SyntaxOk := False;
      end if;
   end CalcBase;

   -----------------------------------------------

   procedure DoInitial
      --# global in     Ch;
      --#        in out Buffer;
      --#        in out DigitsRead;
      --#           out Acceptable;
      --#           out State;
      --# derives Acceptable,
      --#         State      from Ch &
      --#         Buffer,
      --#         DigitsRead from *,
      --#                         Ch;
   is
   begin
      if Ch = '0' then
         Acceptable := CharSet'(EndIndicator => True,
                                '0' .. '9'   => True,
                                '_'          => True,
                                '.'          => True,
                                others       => False);
         State := LeadingZero;
      else                        --must be '1'..'9'
         Store (Ch);
         DigitsRead := DigitsRead + 1;
         Acceptable := CharSet'(EndIndicator => True,
                                '0' .. '9'   => True,
                                '_'          => True,
                                '#'          => True,
                                '.'          => True,
                                'E'          => True,
                                others       => False);
         State := LaterDigits;
      end if;
   end DoInitial;

   -----------------------------------------------

   procedure DoLeadingZero
   --# global in     Ch;
   --#        in out Acceptable;
   --#        in out Buffer;
   --#        in out CoreString;
   --#        in out DigitsRead;
   --#        in out State;
   --# derives Acceptable,
   --#         Buffer,
   --#         CoreString,
   --#         DigitsRead,
   --#         State      from *,
   --#                         Ch;
   is
      Success : Boolean;
   begin
      if Ch = '_' then
         Acceptable := CharSet'('0' .. '9' => True,
                                others     => False);
         State := LeadingUnderline;
      elsif Ch = '.' then
         Acceptable := CharSet'('0' .. '9' => True,
                                others     => False);
         State := DecimalStart;
      elsif (Ch >= '1') and (Ch <= '9') then
         Store (Ch);
         DigitsRead := DigitsRead + 1;
         Acceptable := CharSet'(EndIndicator => True,
                                '0' .. '9'   => True,
                                '_'          => True,
                                '#'          => True,
                                '.'          => True,
                                'E'          => True,
                                others       => False);
         State := LaterDigits;
      elsif Ch = EndIndicator then
         CoreString := EStrings.Empty_String;
         --# accept F, 10, Success, "Ineffective assignment here OK";
         EStrings.Append_Char (E_Str   => CoreString,
                               Ch      => '0',
                               Success => Success);  -- exit during leading zero must give zero result
         --# end accept;
         State := Finished;

         --else its another leading zero and state remains unchanged
      end if;
      --# accept F, 33, Success, "Expect Success unused";
   end DoLeadingZero;

   -----------------------------------------------

   procedure DoLeadingUnderline
      --# global in     Ch;
      --#        in out Buffer;
      --#        in out DigitsRead;
      --#           out Acceptable;
      --#           out State;
      --# derives Acceptable,
      --#         State      from Ch &
      --#         Buffer,
      --#         DigitsRead from *,
      --#                         Ch;
   is
   begin
      if (Ch >= '1') and (Ch <= '9') then
         Store (Ch);
         DigitsRead := DigitsRead + 1;
         Acceptable := CharSet'(EndIndicator => True,
                                '0' .. '9'   => True,
                                '_'          => True,
                                '#'          => True,
                                '.'          => True,
                                'E'          => True,
                                others       => False);
         State := LaterDigits;
      else -- must be '0'
         Acceptable := CharSet'(EndIndicator => True,
                                '0' .. '9'   => True,
                                '_'          => True,
                                '.'          => True,
                                others       => False);
         State := LeadingZero;
      end if;
   end DoLeadingUnderline;

   -----------------------------------------------

   procedure DoLaterDigits
      --# global in     Ch;
      --#        in     InPtr;
      --#        in     S;
      --#        in out Acceptable;
      --#        in out Base;
      --#        in out BaseFound;
      --#        in out Buffer;
      --#        in out CoreString;
      --#        in out DigitsRead;
      --#        in out LegalDigit;
      --#        in out State;
      --#        in out SyntaxOk;
      --# derives Acceptable,
      --#         Buffer     from *,
      --#                         Buffer,
      --#                         Ch,
      --#                         DigitsRead &
      --#         Base,
      --#         CoreString,
      --#         LegalDigit from *,
      --#                         Buffer,
      --#                         Ch &
      --#         BaseFound,
      --#         DigitsRead,
      --#         State      from *,
      --#                         Ch &
      --#         SyntaxOk   from *,
      --#                         Buffer,
      --#                         Ch,
      --#                         InPtr,
      --#                         S;
   is
   begin
      case Ch is
         when EndIndicator =>
            CoreString := Buffer;
            State := Finished;

         when '0' .. '9' => ---------------------------------------------------
            if DigitsRead < 3 then
               DigitsRead := DigitsRead + 1;
               Store (Ch);
            else
               Acceptable ('#') := False;
               Store (Ch);
            end if;

         when '.'      => ---------------------------------------------------
            Acceptable := CharSet'('0' .. '9' => True,
                                   others   => False);
            State := DecimalStart;

         when 'E'      => ---------------------------------------------------
            CoreString := Buffer;
            Buffer := EStrings.Empty_String;
            Acceptable := CharSet'('+'        => True,
                                   '0' .. '9' => True,
                                   others     => False);
            State := ExpStart;

         when '_'      => ---------------------------------------------------
            SyntaxOk := LegalUnderline (CharSet'('0' .. '9' => True,
                                                 others     => False));

         when '#'      => ---------------------------------------------------
            BaseFound     := True;
            CalcBase;                 -- also calcs LegalDigit set
            Buffer := EStrings.Empty_String;
            Acceptable    := LegalDigit;
            State         := BaseStart;

         when others   => ---------------------------------------------------
            null; -- can't occur
      end case;
   end DoLaterDigits;

   -----------------------------------------------

   procedure DoBaseStart
      --# global in     Ch;
      --#        in out Acceptable;
      --#        in out Buffer;
      --#           out State;
      --# derives Acceptable from * &
      --#         Buffer     from *,
      --#                         Ch &
      --#         State      from ;
   is
   begin
      Store (Ch);                  --which must be an acceptable digit
      Acceptable ('#') := True;
      Acceptable ('_') := True;
      State := BasedPart;
   end DoBaseStart;

   -----------------------------------------------

   procedure DoBasedPart
      --# global in     Ch;
      --#        in     InPtr;
      --#        in     LegalDigit;
      --#        in     S;
      --#        in out Acceptable;
      --#        in out Buffer;
      --#        in out CoreString;
      --#        in out State;
      --#        in out SyntaxOk;
      --# derives Acceptable,
      --#         Buffer,
      --#         State      from *,
      --#                         Ch &
      --#         CoreString from *,
      --#                         Buffer,
      --#                         Ch &
      --#         SyntaxOk   from *,
      --#                         Ch,
      --#                         InPtr,
      --#                         LegalDigit,
      --#                         S;
   is
   begin
      case Ch is
         when '#'    =>
            CoreString := Buffer;
            Buffer := EStrings.Empty_String;
            Acceptable := CharSet'(EndIndicator => True,
                                   'E' => True,
                                   others => False);
            State := EndBase;

         when '_'    =>
            SyntaxOk := LegalUnderline (LegalDigit);

         when others =>                  --must be a legal digit
            Store (Ch);

      end case;
   end DoBasedPart;

   -----------------------------------------------

   procedure DoEndBase
      --# global in     Ch;
      --#        in out Acceptable;
      --#           out State;
      --# derives Acceptable from *,
      --#                         Ch &
      --#         State      from Ch;
   is
   begin
      if Ch = EndIndicator then
         State := Finished;
      else                                               -- must be 'E'
         Acceptable := CharSet'('+'        => True,
                                '0' .. '9' => True,
                                others     => False);
         State := ExpStart;
      end if;
   end DoEndBase;

   -----------------------------------------------

   procedure DoDecimalStart
      --# global in     Ch;
      --#        in out Buffer;
      --#        in out PlacesCount;
      --#           out Acceptable;
      --#           out DecimalPointFound;
      --#           out State;
      --# derives Acceptable,
      --#         DecimalPointFound,
      --#         State             from  &
      --#         Buffer            from *,
      --#                                Ch &
      --#         PlacesCount       from *;
   is
   begin
      DecimalPointFound := True;
      Store (Ch);
      PlacesCount := PlacesCount + 1;
      Acceptable := CharSet'(EndIndicator => True,
                             '0' .. '9'   => True,
                             '_'          => True,
                             'E'          => True,
                             others       => False);
      State := DecimalPart;
   end DoDecimalStart;

   -----------------------------------------------

   procedure DoDecimalPart
      --# global in     Ch;
      --#        in     InPtr;
      --#        in     S;
      --#        in out Acceptable;
      --#        in out Buffer;
      --#        in out CoreString;
      --#        in out PlacesCount;
      --#        in out State;
      --#        in out SyntaxOk;
      --# derives Acceptable,
      --#         Buffer,
      --#         PlacesCount,
      --#         State       from *,
      --#                          Ch &
      --#         CoreString  from *,
      --#                          Buffer,
      --#                          Ch &
      --#         SyntaxOk    from *,
      --#                          Ch,
      --#                          InPtr,
      --#                          S;
   is
   begin
      case Ch is
         when EndIndicator =>
            CoreString := Buffer;
            State := Finished;
         when 'E'     =>
            CoreString := Buffer;
            Buffer := EStrings.Empty_String;
            Acceptable := CharSet'('+'        => True,
                                   '-'        => True,
                                   '0' .. '9' => True,
                                   others     => False);
            State := ExpStart;
         when '_'     =>
            SyntaxOk := LegalUnderline (CharSet'('0' .. '9' => True,
                                                 others     => False));
         when others  =>
            Store (Ch);
            PlacesCount := PlacesCount + 1;
      end case;
   end DoDecimalPart;

   -----------------------------------------------

   procedure DoExpStart
      --# global in     Ch;
      --#        in out Buffer;
      --#           out Acceptable;
      --#           out ExponentFound;
      --#           out ExpSign;
      --#           out State;
      --# derives Acceptable,
      --#         ExpSign,
      --#         State         from Ch &
      --#         Buffer        from *,
      --#                            Ch &
      --#         ExponentFound from ;
   is
   begin
      ExponentFound := True;
      case Ch is
         when '-' |
              '+'    => ExpSign    := Ch;
                        Acceptable := CharSet'('0' .. '9' => True,
                                               others     => False);
                        State := ExpPart;
         when others => ExpSign := '+';
                        Store (Ch);
                        Acceptable := CharSet'(EndIndicator => True,
                                               '0' .. '9'   => True,
                                               '_'          => True,
                                               others       => False);
                        State := ExpPart;
      end case;
   end DoExpStart;

   -----------------------------------------------

   procedure DoExpPart
      --# global in     Ch;
      --#        in     InPtr;
      --#        in     S;
      --#        in out Acceptable;
      --#        in out Buffer;
      --#        in out expString;
      --#        in out State;
      --#        in out SyntaxOk;
      --# derives Acceptable,
      --#         Buffer,
      --#         State      from *,
      --#                         Ch &
      --#         expString  from *,
      --#                         Buffer,
      --#                         Ch &
      --#         SyntaxOk   from *,
      --#                         Ch,
      --#                         InPtr,
      --#                         S;
   is
   begin
      case Ch is
         when EndIndicator =>
            ExpString := Buffer;
            State := Finished;
         when '_' =>
            SyntaxOk := LegalUnderline (CharSet'('0' .. '9' => True,
                                                 others     => False));
         when others =>               -- '0'..'9'
            Store (Ch);
            Acceptable := CharSet'(EndIndicator => True,
                                   '0' .. '9'   => True,
                                   '_'          => True,
                                   others       => False);
      end case;
   end DoExpPart;

   -----------------------------------------------

begin -- ParseString

      Acceptable := CharSet'('0' .. '9' => True,
                             others => False);
      State             := Initial;
      SyntaxOk          := True;
      DecimalPointFound := False;
      ExponentFound     := False;
      ExpSign           := '+';
      BaseFound         := False;
      DigitsRead        := 0;
      PlacesCount       := 0;
      Base              := 10;

      CoreString := EStrings.Empty_String;
      ExpString := EStrings.Copy_String (Str => "0");
      Buffer     := EStrings.Empty_String;
      LegalDigit := CharSet'(Character => False);

      InPtr := 1;

      loop
         exit when not SyntaxOk;                    -- don't look beyond first error

         if InPtr > EStrings.Get_Length (E_Str => S) then -- end of string
            Ch := EndIndicator;
         else
            Ch := EStrings.Get_Element (E_Str => S,
                                        Pos   => InPtr);                -- get Character
            Caps (Ch);
         end if;

         -- check legality of Character against acceptable set
         if not Acceptable (Ch) then
            SyntaxOk := False;
            exit;
         end if;

         -- if we get here we have legal Character to deal with;
         case State is
            when Initial           => DoInitial;
            when LeadingZero       => DoLeadingZero;
            when LeadingUnderline  => DoLeadingUnderline;
            when LaterDigits       => DoLaterDigits;
            when BaseStart         => DoBaseStart;
            when BasedPart         => DoBasedPart;
            when EndBase           => DoEndBase;
            when DecimalStart      => DoDecimalStart;
            when DecimalPart       => DoDecimalPart;
            when ExpStart          => DoExpStart;
            when ExpPart           => DoExpPart;
            when Finished          => null; -- can't be reached because of exit below
         end case;

         exit when State = Finished;

         InPtr := InPtr + 1;
      end loop;

      PlacesAfterPoint := PlacesCount;
      LegalSyntax      := SyntaxOk;
end ParseString;
