-- $Id: mainloop.adb 15860 2010-02-01 17:06:34Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Casing;
with CommandLineData;
with ConfigFile;
with ContextManager;
with ContextManager.Ops;
with Dictionary;
with Error_Types;
with ErrorHandler;
with EStrings;
with ELStrings;
with File_Utils;
with FileSystem;
with IndexManager;
with LexTokenLists;
with LexTokenManager;
with MetaFile;
with RequiredUnits;
with ScreenEcho;
with Sem;
with SLI;
with SPARK_IO;
with SparkHTML;
with SparkLex;
with SPParser;
with Statistics;
with STree;
with VCG;
with XMLReport;

use type CommandLineData.RuleGenerationPolicies;
use type ContextManager.FileDescriptors;
use type ContextManager.FileStatus;
use type ContextManager.UnitDescriptors;
use type ContextManager.UnitStatus;
use type ContextManager.UnitTypes;
use type Dictionary.Symbol;
use type ErrorHandler.ErrorLevel;
use type LexTokenManager.Str_Comp_Result;
use type SPARK_IO.File_Status;
use type SPARK_IO.File_Type;

package body MainLoop
is

   procedure PrintFileName (File  : in SPARK_IO.File_Type;
                            Name  : in EStrings.T;
                            Plain : in Boolean)
   --  all-from-all derives for data flow analysis only
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Name,
   --#                                Plain;
   is
      Stat : FileSystem.TypFileSpecStatus;
      FullName : EStrings.T;
   begin
      --# accept F, 10, Stat, "Stat not used" &
      --#        F, 33, Stat, "Stat not used";
      if Plain then
         FullName := EStrings.Lower_Case (E_Str => FileSystem.JustFile (Name, True));
      else
         FileSystem.FindFullFileName (Name, Stat, FullName);
      end if;

      if File = SPARK_IO.Standard_Output then
         ScreenEcho.Put_ExaminerString (FullName);
      else
         EStrings.Put_String (File  => File,
                              E_Str => FullName);
      end if;
   end PrintFileName; -- Stat deliberately unused

   procedure InitialiseProcessing
   --# global in     CommandLineData.Content;
   --#        in out LexTokenManager.State;
   --#        in out SparkHTML.GenerateHTML;
   --#        in out SPARK_IO.File_Sys;
   --#           out Dictionary.Dict;
   --#           out SparkHTML.HTMLWorkDir;
   --#           out SparkHTML.SparkWorkDir;
   --#           out SparkLex.CurrLine;
   --#           out XMLReport.State;
   --# derives Dictionary.Dict,
   --#         SparkHTML.HTMLWorkDir,
   --#         SparkHTML.SparkWorkDir,
   --#         SPARK_IO.File_Sys      from CommandLineData.Content,
   --#                                     LexTokenManager.State,
   --#                                     SPARK_IO.File_Sys &
   --#         LexTokenManager.State  from *,
   --#                                     CommandLineData.Content &
   --#         SparkHTML.GenerateHTML from *,
   --#                                     CommandLineData.Content,
   --#                                     LexTokenManager.State,
   --#                                     SPARK_IO.File_Sys &
   --#         SparkLex.CurrLine,
   --#         XMLReport.State        from ;
   is
   begin
      SparkLex.ClearLineContext;
      LexTokenManager.Initialise_String_Table;
      Dictionary.Initialize (CommandLineData.Content.WriteDict);
      --# accept F, 602, SparkHTML.HTMLWorkDir, SparkHTML.HTMLWorkDir, "Only used in HTML selected" &
      --#        F, 602, SparkHTML.SparkWorkDir, SparkHTML.SparkWorkDir, "Only used in HTML selected";
      if CommandLineData.Content.HTML then
         SparkHTML.InitSparkHTML;
      end if;
      XMLReport.Init;
   end InitialiseProcessing;

   procedure OpenFile (FileDescriptor : in ContextManager.FileDescriptors)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ContextManager.Ops.FileHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ContextManager.Ops.FileHeap,
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          ContextManager.Ops.FileHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          FileDescriptor,
   --#                                          LexTokenManager.State,
   --#                                          SPARK_IO.File_Sys &
   --#         SparkLex.CurrLine           from *,
   --#                                          ContextManager.Ops.FileHeap,
   --#                                          FileDescriptor,
   --#                                          LexTokenManager.State,
   --#                                          SPARK_IO.File_Sys;
   is
      Status         : SPARK_IO.File_Status;
      FileContext    : SparkLex.LineContext;
      SourceFile     : SPARK_IO.File_Type := SPARK_IO.Null_File;
      SourceFileName : EStrings.T;
      ErrorContext   : ErrorHandler.ErrorContexts;
   begin
      SourceFileName := LexTokenManager.Lex_String_To_String (Lex_Str => ContextManager.Ops.GetSourceFileName (FileDescriptor));
      FileSystem.OpenSourceFile (SourceFile, SourceFileName, Status);

      if Status = SPARK_IO.Ok then
         ContextManager.Ops.SetSourceFile (FileDescriptor, SourceFile);
         SparkLex.ClearLineContext;
         SparkLex.StoreLineContext (FileContext);
         ContextManager.Ops.SetLineContext (FileDescriptor, FileContext);
         ErrorHandler.ErrorInit (SourceFileName, CommandLineData.Content.Echo);
         ErrorHandler.GetErrorContext (ErrorContext);
         ContextManager.Ops.SetErrorContext (FileDescriptor, ErrorContext);
         ContextManager.Ops.SetFileStatus (FileDescriptor, ContextManager.FileOpen);
      else
         ScreenEcho.Put_String ("Cannot open file ");
         PrintFileName (SPARK_IO.Standard_Output, SourceFileName, CommandLineData.Content.PlainOutput);
         ScreenEcho.New_Line (1);
         ContextManager.Ops.SetFileStatus (FileDescriptor, ContextManager.UnableToOpen);
         ErrorHandler.SetFileOpenError;
      end if;
   end OpenFile;

   procedure CreateFileContext (SourceFileName : in     EStrings.T;
                                FileDescriptor :    out ContextManager.FileDescriptors)
   --# global in out ContextManager.Ops.FileHeap;
   --#        in out LexTokenManager.State;
   --# derives ContextManager.Ops.FileHeap,
   --#         LexTokenManager.State       from *,
   --#                                          LexTokenManager.State,
   --#                                          SourceFileName &
   --#         FileDescriptor              from ContextManager.Ops.FileHeap;
   is
      FD2                        : ContextManager.FileDescriptors;
      Lex_String_Source_Filename : LexTokenManager.Lex_String;
   begin
      ContextManager.Ops.CreateFileDescriptor (FD2);
      LexTokenManager.Insert_Examiner_String (Str     => SourceFileName,
                                              Lex_Str => Lex_String_Source_Filename);
      ContextManager.Ops.SetSourceFileName (FD2,
                                            Lex_String_Source_Filename);
      ContextManager.Ops.SetFileStatus (FD2,
                                        ContextManager.FileCreated);
      FileDescriptor := FD2;
   end CreateFileContext;

   procedure PrepareNextArgumentFile (TheFileName    : in     EStrings.T;
                                      DoListing      : in     Boolean;
                                      TheListingName : in     EStrings.T;
                                      FileDescriptor :    out ContextManager.FileDescriptors)
   --# global in out ContextManager.Ops.FileHeap;
   --#        in out LexTokenManager.State;
   --# derives ContextManager.Ops.FileHeap from *,
   --#                                          DoListing,
   --#                                          LexTokenManager.State,
   --#                                          TheFileName,
   --#                                          TheListingName &
   --#         LexTokenManager.State       from *,
   --#                                          TheFileName &
   --#         FileDescriptor              from ContextManager.Ops.FileHeap;
   is
      FD2 : ContextManager.FileDescriptors;
   begin
      CreateFileContext (TheFileName,  FD2);
      if DoListing then
         ContextManager.Ops.SetListingReq (FD2, True);
         ContextManager.Ops.SetListingFileName (FD2, TheListingName);
      end if;
      FileDescriptor := FD2;
   end PrepareNextArgumentFile;

   procedure ShowEndOfFile (ToFile : in SPARK_IO.File_Type)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                ToFile;
   is
   begin

      SPARK_IO.New_Line (ToFile, 2);
      SPARK_IO.Put_Line (ToFile,
                         "--End of file--------------------------------------------------",
                         0);
   end ShowEndOfFile;

   procedure EchoWarningOrNote (AboutFile : in EStrings.T;
                                Msg       : in String)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                AboutFile,
   --#                                CommandLineData.Content,
   --#                                Msg;
   is
   begin
      if CommandLineData.Content.Echo then
         if CommandLineData.Content.Brief then
            -- Echo the first filename from the command-line.  If it starts with an '@'
            -- then chop it off, so the file is recognized by the IDE.
            if EStrings.Get_Length (E_Str => AboutFile) >= 2 and then
              EStrings.Get_Element (E_Str => AboutFile,
                                    Pos   => 1) = '@' then

               ScreenEcho.Put_ExaminerString (EStrings.Section (AboutFile, 2, EStrings.Get_Length (E_Str => AboutFile) - 1));
            else
               ScreenEcho.Put_ExaminerString (AboutFile);
            end if;

            ScreenEcho.Put_String (":1:1: ");
            ScreenEcho.Put_Line (Msg);
         else
            ScreenEcho.New_Line (1);
            ScreenEcho.Put_Line (Msg);
         end if;
      end if;
   end EchoWarningOrNote;


   procedure OutputDictionaryFile
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict   from *,
   --#                                CommandLineData.Content,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State;
   is
      Status        : SPARK_IO.File_Status;
      Dict_Message  : constant String := "Generating dictionary file";
   begin
      if CommandLineData.Content.WriteDict then
         if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
            ScreenEcho.Echo (EStrings.Copy_String (Str => Dict_Message));
         end if;
         Dictionary.Write (CommandLineData.Content.DictFileName,
                           Status);
         if Status /= SPARK_IO.Ok then
            ScreenEcho.Put_Line ("Error writing dictionary file: ");
            PrintFileName (SPARK_IO.Standard_Output,
                           CommandLineData.Content.DictFileName,
                           False);
            ScreenEcho.New_Line (1);
         end if;
      end if;
   end OutputDictionaryFile;


   procedure PrintLexTokenList (File : in SPARK_IO.File_Type;
                                List : in LexTokenLists.Lists)
   --# global in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                LexTokenManager.State,
   --#                                List;
   is
   begin
      if LexTokenLists.Get_Length (List => List) = 0 then
         SPARK_IO.Put_String (File, "Unexpected Empty Lextoken List", 0);
      else
         LexTokenLists.Print_List (File => File,
                                   List => List);
      end if;
   end PrintLexTokenList;


   -- Following hidden trace routines enabled by -debug=u -----------------------------------
   procedure Trace (Msg : String)
   --# derives null from Msg;
   is
      --# hide Trace;
   begin
      if CommandLineData.Content.Debug.Units then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            Msg,
                            0);
      end if;
   end Trace;

   procedure TraceUnit (List  : in LexTokenLists.Lists;
                        Types : in ContextManager.UnitTypeSets)
   --# derives null from List,
   --#                   Types;
   is
      --# hide TraceUnit;
      FirstType : Boolean := True;
   begin
      if CommandLineData.Content.Debug.Units then
         SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                              "   unit name: ",
                              0);
         PrintLexTokenList (SPARK_IO.Standard_Output,
                            List);
         SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                              " which is {",
                              0);
         for I in ContextManager.UnitTypes loop
            if Types (I) then
               if not FirstType then
                  SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                       ", ",
                                       0);
               end if;
               SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                    ContextManager.UnitTypes'Image (I),
                                    0);
               FirstType := False;
            end if;
         end loop;
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "}",
                            0);
      end if;
   end TraceUnit;
   -----------------------------------------------------------------------------------

   function GetUnitName (List : in LexTokenLists.Lists) return EStrings.T
   --# global in LexTokenManager.State;
   is
      ReturnString : EStrings.T;
   begin
      if LexTokenLists.Get_Length (List => List) = 0 then
         ReturnString := EStrings.Copy_String (Str => "Unexpected Empty LexToken List");
      else
         ReturnString := LexTokenLists.Token_List_To_String (Token_List => List);
      end if;
      return ReturnString;
   end GetUnitName;

   procedure PrintUnitType (File     : in SPARK_IO.File_Type;
                            UnitType : in ContextManager.UnitTypes)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                UnitType;
   is
   begin
      case UnitType is
         when ContextManager.MainProgram =>
            SPARK_IO.Put_String (File, "main program", 0);
         when ContextManager.PackageOrGenericDeclaration =>
            SPARK_IO.Put_String (File, "package or generic specification", 0);
         when ContextManager.PackageSpecification =>
            SPARK_IO.Put_String (File, "package specification", 0);
         when ContextManager.PackageBody =>
            SPARK_IO.Put_String (File, "package body", 0);
         when ContextManager.SubUnit =>
            SPARK_IO.Put_String (File, "subunit", 0);
         when ContextManager.GenericDeclaration =>
            SPARK_IO.Put_String (File, "generic declaration", 0);
         when ContextManager.GenericPackageInstantiation =>
            SPARK_IO.Put_String (File, "generic package instantiation", 0);
         when ContextManager.GenericSubprogramBody =>
            SPARK_IO.Put_String (File, "generic subprogram body", 0);
         when ContextManager.InterUnitPragma =>
            null;
         when ContextManager.InvalidUnit =>
            null;
      end case;
   end PrintUnitType;


   function GetUnitType (UnitType : in ContextManager.UnitTypes) return EStrings.T
   is
      ReturnStr : EStrings.T;
   begin
      case UnitType is
         when ContextManager.MainProgram =>
            ReturnStr := EStrings.Copy_String (Str => "main program");
         when ContextManager.PackageOrGenericDeclaration =>
            ReturnStr := EStrings.Copy_String (Str => "package or generic specification");
         when ContextManager.PackageSpecification =>
            ReturnStr := EStrings.Copy_String (Str => "package specification");
         when ContextManager.PackageBody =>
            ReturnStr := EStrings.Copy_String (Str => "package body");
         when ContextManager.SubUnit =>
            ReturnStr := EStrings.Copy_String (Str => "subunit");
         when ContextManager.GenericDeclaration =>
            ReturnStr := EStrings.Copy_String (Str => "generic declaration");
         when ContextManager.GenericPackageInstantiation =>
            ReturnStr := EStrings.Copy_String (Str => "generic package instantiation");
         when ContextManager.GenericSubprogramBody =>
            ReturnStr := EStrings.Copy_String (Str => "generic subprogram body");
         when ContextManager.InterUnitPragma =>
            ReturnStr := EStrings.Copy_String (Str => "InterUnitPragma");
         when ContextManager.InvalidUnit =>
            ReturnStr := EStrings.Copy_String (Str => "InvalidUnit");
      end case;
      return ReturnStr;
   end GetUnitType;

   procedure PrintReason (File       : in SPARK_IO.File_Type;
                          UnitStatus : in ContextManager.UnitStatus)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                UnitStatus;
   is
   begin
      case UnitStatus is
         when ContextManager.NoIndexFile =>
            SPARK_IO.Put_String (File, "No index file specified.", 0);
         when ContextManager.NotInIndexFile =>
            SPARK_IO.Put_String (File, "Omitted from index file.", 0);
         when ContextManager.CannotOpenFile =>
            SPARK_IO.Put_String (File, "Cannot open source file specified in index file.", 0);
         when ContextManager.UnableToLocate =>
            SPARK_IO.Put_String (File, "Declaration not found in source code.", 0);
         when others =>
            null;  -- should not happen because only called when unit is not found
      end case;
   end PrintReason;

   -- This function is not currently used but will be required when
   -- XMLReport is updated to contain the reason
   function GetReason (UnitStatus : in ContextManager.UnitStatus) return EStrings.T
   is
      ReturnStr : EStrings.T;
   begin
      case UnitStatus is
         when ContextManager.NoIndexFile =>
            ReturnStr := EStrings.Copy_String (Str => "No index file specified.");
         when ContextManager.NotInIndexFile =>
            ReturnStr := EStrings.Copy_String (Str => "Omitted from index file.");
         when ContextManager.CannotOpenFile =>
            ReturnStr := EStrings.Copy_String (Str => "Cannot open source file specified in index file.");
         when ContextManager.UnableToLocate =>
            ReturnStr := EStrings.Copy_String (Str => "Declaration not found in source code.");
         when others =>
            ReturnStr := EStrings.Empty_String;  -- only called when unit is not found
      end case;
      return ReturnStr;
   end GetReason;


   procedure AppendToReportFile (ReportFile     : in SPARK_IO.File_Type;
                                 Purpose        : in Error_Types.ConversionRequestSource;
                                 FileDescriptor : in ContextManager.FileDescriptors)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ContextManager.Ops.FileHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives ContextManager.Ops.FileHeap,
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys,
   --#         XMLReport.State             from CommandLineData.Content,
   --#                                          ContextManager.Ops.FileHeap,
   --#                                          ContextManager.Ops.UnitHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          FileDescriptor,
   --#                                          LexTokenManager.State,
   --#                                          Purpose,
   --#                                          ReportFile,
   --#                                          SPARK_IO.File_Sys,
   --#                                          XMLReport.State;
   is
      ErrorContext : ErrorHandler.ErrorContexts;

      function GetSourceFileName (FileDescriptor : in ContextManager.FileDescriptors)
                                 return EStrings.T
      --# global in ContextManager.Ops.FileHeap;
      --#        in LexTokenManager.State;
      is
      begin
         return LexTokenManager.Lex_String_To_String (Lex_Str => ContextManager.Ops.GetSourceFileName (FileDescriptor));
      end GetSourceFileName;


      procedure PrintSourceFileName (ReportFile     : in SPARK_IO.File_Type;
                                     FileDescriptor : in ContextManager.FileDescriptors)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.FileHeap;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.Ops.FileHeap,
      --#                                FileDescriptor,
      --#                                LexTokenManager.State,
      --#                                ReportFile;
      is
      begin
         SPARK_IO.Put_String (ReportFile, "Source Filename:   ", 0);
         PrintFileName (ReportFile,
                        LexTokenManager.Lex_String_To_String (Lex_Str => ContextManager.Ops.GetSourceFileName (FileDescriptor)),
                        CommandLineData.Content.PlainOutput);
         SPARK_IO.New_Line (ReportFile, 1);
      end PrintSourceFileName;

      procedure PrintListingFileName (ReportFile     : in SPARK_IO.File_Type;
                                      FileDescriptor : in ContextManager.FileDescriptors)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.FileHeap;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.Ops.FileHeap,
      --#                                FileDescriptor,
      --#                                ReportFile;
      is
         FileName : EStrings.T;
      begin
         if ContextManager.Ops.ListingReqt (FileDescriptor) then
            SPARK_IO.Put_String (ReportFile, "Listing Filename:  ", 0);
            ContextManager.Ops.GetListingFileName (FileDescriptor, FileName);
            CommandLineData.Normalize_FileName_To_Output_Directory (FileName);
            PrintFileName (ReportFile, FileName, CommandLineData.Content.PlainOutput);
            SPARK_IO.New_Line (ReportFile, 1);
         else
            SPARK_IO.Put_Line (ReportFile, "No Listing File", 0);
         end if;
      end PrintListingFileName;

      procedure PrintUnitStatus (File           : in SPARK_IO.File_Type;
                                 UnitDescriptor : in ContextManager.UnitDescriptors)
      --# global in     ContextManager.Ops.UnitHeap;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                ContextManager.Ops.UnitHeap,
      --#                                File,
      --#                                UnitDescriptor;
      is
      begin
         case ContextManager.Ops.GetUnitStatus (UnitDescriptor) is
            when ContextManager.NoUnitEntry =>
               null;
            when ContextManager.UnitCreated =>
               null;
            when ContextManager.UnitParsed
               | ContextManager.UnitDeferred =>
               SPARK_IO.Put_String (File, "Unit has been parsed only, " &
                                    "any errors are listed below.", 0);
            when ContextManager.UnitAnalysed =>
               SPARK_IO.Put_String (File, "Unit has been analysed, " &
                                    "any errors are listed below.", 0);
            when ContextManager.NoIndexFile  =>
               SPARK_IO.Put_String (File, "Unit required but not found in source file and " &
                                    "no index file has been specified.", 0);
            when ContextManager.NotInIndexFile =>
               SPARK_IO.Put_String (File, "Unit required but not found in source file or " &
                                    "in index file.", 0);
            when ContextManager.CannotOpenFile =>
               SPARK_IO.Put_String (File, "Source file for required Unit given " &
                                    "in index file cannot be opened.", 0);
            when ContextManager.UnableToLocate =>
               SPARK_IO.Put_String (File, "Unit required but not found.", 0);
         end case;
      end PrintUnitStatus;

      function GetUnitStatus (UnitDescriptor : in ContextManager.UnitDescriptors) return EStrings.T
      --# global in ContextManager.Ops.UnitHeap;
      is
         ReturnString : EStrings.T;
      begin
         ReturnString := EStrings.Empty_String;
         case ContextManager.Ops.GetUnitStatus (UnitDescriptor) is
            when ContextManager.NoUnitEntry =>
               null;
            when ContextManager.UnitCreated =>
               null;
            when ContextManager.UnitParsed
               | ContextManager.UnitDeferred =>
               ReturnString := EStrings.Copy_String (Str => "parsed");
            when ContextManager.UnitAnalysed =>
               ReturnString := EStrings.Copy_String (Str => "analysed");
            when ContextManager.NoIndexFile  =>
               ReturnString := EStrings.Copy_String (Str => "not found and no index file specified");
            when ContextManager.NotInIndexFile =>
               ReturnString := EStrings.Copy_String (Str => "not found in source file or " &
                                                       "in index file.");
            when ContextManager.CannotOpenFile =>
               ReturnString := EStrings.Copy_String (Str => "unit source file name given " &
                                                       "in index file cannot be opened.");
            when ContextManager.UnableToLocate =>
               ReturnString := EStrings.Copy_String (Str => "not found");
         end case;
         return ReturnString;
      end GetUnitStatus;

      procedure OutputUnitList (ReportFile     : in SPARK_IO.File_Type;
                                FileDescriptor : in ContextManager.FileDescriptors)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitHeap;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.Ops.UnitHeap,
      --#                                FileDescriptor,
      --#                                LexTokenManager.State,
      --#                                ReportFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.Ops.UnitHeap,
      --#                                FileDescriptor,
      --#                                LexTokenManager.State;
      is
         UnitDescriptor       : ContextManager.UnitDescriptors;
         UnitName             : LexTokenLists.Lists;
         UnitType             : ContextManager.UnitTypes;
         StoredFileDescriptor : ContextManager.FileDescriptors;
         UnitOutput           : Boolean;
      begin
         UnitOutput := False;
         UnitDescriptor := ContextManager.Ops.FirstUnitDescriptor;
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SUnitsInFile,
                                    ReportFile);
         end if;
         loop
            exit when UnitDescriptor = ContextManager.NullUnit;
            ContextManager.Ops.GetFileDescriptor (UnitDescriptor,
                                                  StoredFileDescriptor);
            if (ContextManager.Ops.GetUnitStatus (UnitDescriptor) /=
                ContextManager.NoUnitEntry) and
               FileDescriptor = StoredFileDescriptor
            then
               ContextManager.Ops.GetUnitName (UnitDescriptor, UnitName, UnitType);
               if LexTokenLists.Get_Length (List => UnitName) /= 0 then
                  --# accept F, 41, "Stable expression expected";
                  if CommandLineData.Content.XML then
                     XMLReport.AdaUnit (GetUnitName (UnitName),
                                        GetUnitType (UnitType),
                                        GetUnitStatus (UnitDescriptor),
                                        ReportFile);
                     UnitOutput := True;
                  else
                     UnitOutput := True;
                     SPARK_IO.Put_String (ReportFile, "   Unit name:  ", 0);
                     PrintLexTokenList (ReportFile, UnitName);
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_String (ReportFile, "   Unit type:  ", 0);
                     PrintUnitType (ReportFile, UnitType);
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_String (ReportFile, "   ", 0);
                     PrintUnitStatus (ReportFile, UnitDescriptor);
                     SPARK_IO.New_Line (ReportFile, 2);
                  end if;
                  --# end accept;
               end if;
            end if;
            UnitDescriptor := ContextManager.Ops.NextUnitDescriptor (UnitDescriptor);
         end loop;
         if CommandLineData.Content.XML then
            XMLReport.EndSection (XMLReport.SUnitsInFile,
                                  ReportFile);
         end if;
         if not UnitOutput then
            SPARK_IO.Put_Line (ReportFile, "***     No units in file", 0);
            SPARK_IO.New_Line (ReportFile, 2);
         end if;
      end OutputUnitList;

   begin -- AppendToReportFile
      SPARK_IO.New_Line (ReportFile, 1);
      if not CommandLineData.Content.XML then
         PrintSourceFileName (ReportFile, FileDescriptor);
      end if;
      if ContextManager.Ops.GetFileStatus (FileDescriptor) = ContextManager.UnableToOpen then
         if CommandLineData.Content.XML then
            XMLReport.Start_File (Plain_Output => CommandLineData.Content.PlainOutput,
                                  F_Name       => GetSourceFileName (FileDescriptor),
                                  Report       => ReportFile);
            OutputUnitList (ReportFile, FileDescriptor);
            XMLReport.EndFile (ReportFile);
         else
            SPARK_IO.New_Line (ReportFile, 1);
            OutputUnitList (ReportFile, FileDescriptor);
            SPARK_IO.Put_Line (ReportFile, "***     Unable to open source file", 0);
         end if;
      else
         if CommandLineData.Content.XML then
            XMLReport.Start_File (Plain_Output => CommandLineData.Content.PlainOutput,
                                  F_Name       => GetSourceFileName (FileDescriptor),
                                  Report       => ReportFile);
            OutputUnitList (ReportFile, FileDescriptor);
            ContextManager.Ops.GetErrorContext (FileDescriptor, ErrorContext);
            ErrorHandler.SetErrorContext (ErrorContext);
            ErrorHandler.AppendErrors (ReportFile,
                                       Purpose);
            ErrorHandler.GetErrorContext (ErrorContext);
            ContextManager.Ops.SetErrorContext (FileDescriptor, ErrorContext);
            XMLReport.EndFile (ReportFile);
         else
            PrintListingFileName (ReportFile, FileDescriptor);
            SPARK_IO.New_Line (ReportFile, 1);
            OutputUnitList (ReportFile, FileDescriptor);
            ContextManager.Ops.GetErrorContext (FileDescriptor, ErrorContext);
            ErrorHandler.SetErrorContext (ErrorContext);
            ErrorHandler.AppendErrors (ReportFile,
                                       Purpose);
            ErrorHandler.GetErrorContext (ErrorContext);
            ContextManager.Ops.SetErrorContext (FileDescriptor, ErrorContext);
         end if;
      end if;
   end AppendToReportFile;



   procedure OutputReportFile (ReportFileErrorList : in out SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     ConfigFile.State;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     Dictionary.Dict;
   --#        in     IndexManager.State;
   --#        in     SparkHTML.HTMLWorkDir;
   --#        in     SparkHTML.SparkWorkDir;
   --#        in     Statistics.TableUsage;
   --#        in out ContextManager.Ops.FileHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SparkHTML.GenerateHTML;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives ContextManager.Ops.FileHeap,
   --#         ErrorHandler.ErrorContext,
   --#         XMLReport.State             from CommandLineData.Content,
   --#                                          ConfigFile.State,
   --#                                          ContextManager.Ops.FileHeap,
   --#                                          ContextManager.Ops.UnitHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IndexManager.State,
   --#                                          LexTokenManager.State,
   --#                                          SPARK_IO.File_Sys,
   --#                                          XMLReport.State &
   --#         LexTokenManager.State       from *,
   --#                                          CommandLineData.Content,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IndexManager.State,
   --#                                          SPARK_IO.File_Sys,
   --#                                          XMLReport.State &
   --#         ReportFileErrorList         from *,
   --#                                          CommandLineData.Content,
   --#                                          SPARK_IO.File_Sys &
   --#         SparkHTML.GenerateHTML      from *,
   --#                                          CommandLineData.Content,
   --#                                          ConfigFile.State,
   --#                                          ContextManager.Ops.FileHeap,
   --#                                          ContextManager.Ops.UnitHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IndexManager.State,
   --#                                          LexTokenManager.State,
   --#                                          ReportFileErrorList,
   --#                                          SPARK_IO.File_Sys,
   --#                                          Statistics.TableUsage,
   --#                                          XMLReport.State &
   --#         SPARK_IO.File_Sys           from *,
   --#                                          CommandLineData.Content,
   --#                                          ConfigFile.State,
   --#                                          ContextManager.Ops.FileHeap,
   --#                                          ContextManager.Ops.UnitHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IndexManager.State,
   --#                                          LexTokenManager.State,
   --#                                          ReportFileErrorList,
   --#                                          SparkHTML.GenerateHTML,
   --#                                          SparkHTML.HTMLWorkDir,
   --#                                          SparkHTML.SparkWorkDir,
   --#                                          Statistics.TableUsage,
   --#                                          XMLReport.State;
   is
      ReportFile     : SPARK_IO.File_Type;
      OK             : Boolean;

      Report_Message : constant String := "Generating report file";

      procedure CreateReportFile (ReportFile : out SPARK_IO.File_Type;
                                  OK         : out Boolean)
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives OK,
      --#         ReportFile        from CommandLineData.Content,
      --#                                SPARK_IO.File_Sys &
      --#         SPARK_IO.File_Sys,
      --#         XMLReport.State   from CommandLineData.Content,
      --#                                SPARK_IO.File_Sys,
      --#                                XMLReport.State;
      is
         LocalFile : SPARK_IO.File_Type;
         Status    : SPARK_IO.File_Status;
         FileName  : EStrings.T;
      begin
         OK := True;
         LocalFile := SPARK_IO.Null_File;

         FileName := CommandLineData.Content.ReportFileName;
         CommandLineData.Normalize_FileName_To_Output_Directory (FileName);
         FileName := FileSystem.CaseOfFilesForCreate (FileName);

         EStrings.Create (File         => LocalFile,
                          Name_Of_File => FileName,
                          Form_Of_File => "",
                          Status       => Status);
         if Status = SPARK_IO.Ok then
            if CommandLineData.Content.XML then
               XMLReport.StartSection (XMLReport.SReport,
                                       LocalFile);
               XMLReport.StartSection (XMLReport.SPrologue,
                                       LocalFile);
            else
               File_Utils.PrintAHeader (LocalFile,
                                        "Report of SPARK Examination",
                                        File_Utils.OtherFile);
            end if;
         else
            OK := False;
            ScreenEcho.Put_Line ("Unable to open report file: ");
            PrintFileName (SPARK_IO.Standard_Output,
                           CommandLineData.Content.ReportFileName,
                           False);
            ScreenEcho.New_Line (1);
         end if;
         ReportFile := LocalFile;
      end CreateReportFile;

      procedure CloseReportFile (ReportFile : in out SPARK_IO.File_Type)
      --# global in out SPARK_IO.File_Sys;
      --# derives ReportFile,
      --#         SPARK_IO.File_Sys from *,
      --#                                ReportFile;
      is
         Status     : SPARK_IO.File_Status;
      begin
         --# accept F, 10, Status, "Status not used here" &
         --#        F, 33, Status, "Status not used here";
         SPARK_IO.Close (ReportFile, Status);
      end CloseReportFile;

      procedure OutputFileList (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.FileHeap;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.Ops.FileHeap,
      --#                                LexTokenManager.State,
      --#                                ReportFile;
      is
         FileDescriptor : ContextManager.FileDescriptors;
      begin
         if not CommandLineData.Content.XML then
            FileDescriptor := ContextManager.Ops.FirstFileDescriptor;
            SPARK_IO.New_Line (ReportFile, 2);
            if FileDescriptor = ContextManager.NullFile then
               SPARK_IO.Put_Line (ReportFile, "No source files used", 0);
            else
               if not CommandLineData.Content.XML then
                  SPARK_IO.Put_Line (ReportFile, "Source Filename(s) used were:", 0);
               end if;
               loop
                  exit when FileDescriptor = ContextManager.NullFile;
                  SPARK_IO.Put_String (ReportFile, "   ", 0);
                  PrintFileName (ReportFile,
                                 LexTokenManager.Lex_String_To_String (Lex_Str => ContextManager.Ops.GetSourceFileName (FileDescriptor)),
                                 CommandLineData.Content.PlainOutput);
                  SPARK_IO.New_Line (ReportFile, 1);
                  FileDescriptor := ContextManager.Ops.NextFileDescriptor (FileDescriptor);
               end loop;
            end if;
         end if;
      end OutputFileList;

      procedure OutputErrorList (ReportFile          : in     SPARK_IO.File_Type;
                                 ReportFileErrorList : in out SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitHeap;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ContextManager.Ops.FileHeap;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives ContextManager.Ops.FileHeap,
      --#         ErrorHandler.ErrorContext,
      --#         XMLReport.State             from CommandLineData.Content,
      --#                                          ContextManager.Ops.FileHeap,
      --#                                          ContextManager.Ops.UnitHeap,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.State,
      --#                                          ReportFile,
      --#                                          SPARK_IO.File_Sys,
      --#                                          XMLReport.State &
      --#         ReportFileErrorList         from * &
      --#         SPARK_IO.File_Sys           from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.Ops.FileHeap,
      --#                                          ContextManager.Ops.UnitHeap,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.State,
      --#                                          ReportFile,
      --#                                          ReportFileErrorList,
      --#                                          XMLReport.State;
      is
         subtype LineIndex is Positive range 1 .. 512;
         subtype Lines is String (LineIndex);
         CurrentLine : Lines;
         Stop        : Natural;
         Unused1,
         Unused2     : SPARK_IO.File_Status;
         FileDescriptor : ContextManager.FileDescriptors;
      begin
         --# accept F, 10, Unused1, "Not required" &
         --#        F, 33, Unused1, "Not required" &
         --#        F, 10, Unused2, "Not required" &
         --#        F, 33, Unused2, "Not required";
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SResults,
                                    ReportFile);
         end if;

         --first output errors associated with files picked up from index mechanism
         FileDescriptor := ContextManager.Ops.FirstFileDescriptor;
         loop
            exit when FileDescriptor = ContextManager.NullFile;
            if not ContextManager.Ops.ErrorsReported (FileDescriptor) then
               AppendToReportFile (ReportFile,
                                   Error_Types.ForReportIndexedFiles,
                                   FileDescriptor);
            end if;
            FileDescriptor := ContextManager.Ops.NextFileDescriptor (FileDescriptor);
         end loop;

         --then append errors from files on command line/meta files
         SPARK_IO.Reset (ReportFileErrorList, SPARK_IO.In_File, Unused1);
         while not SPARK_IO.End_Of_File (ReportFileErrorList)
         loop
            SPARK_IO.Get_Line (ReportFileErrorList,
                                 --to get
                               CurrentLine,
                               Stop);
            if Stop = 0 then
               SPARK_IO.New_Line (ReportFile, 1);
            else
               SPARK_IO.Put_Line (ReportFile,
                                  CurrentLine,
                                  Stop);
            end if;
         end loop;
         SPARK_IO.Close (ReportFileErrorList, Unused2);
      end OutputErrorList;

      procedure OutputUnitsRequiredButNotFound (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitHeap;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.Ops.UnitHeap,
      --#                                LexTokenManager.State,
      --#                                ReportFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.Ops.UnitHeap,
      --#                                LexTokenManager.State;
      is
         StatusCol : constant Positive := 50;
         ReasonCol : constant Positive := 72;

         UnitDescriptor : ContextManager.UnitDescriptors;
         UnitName       : LexTokenLists.Lists;
         UnitType       : ContextManager.UnitTypes;

         FirstTime      : Boolean;
      begin
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SUnitsNotFound,
                                    ReportFile);

            UnitDescriptor := ContextManager.Ops.FirstUnitDescriptor;

            loop
               exit when UnitDescriptor = ContextManager.NullUnit;
               if ContextManager.Ops.GetUnitStatus (UnitDescriptor) in
                 ContextManager.UnitNotFound
               then
                  ContextManager.Ops.GetUnitName (UnitDescriptor, UnitName, UnitType);

                  -- XMLReport needs to be updated to include the reason
                  XMLReport.AdaUnit (LexTokenLists.Token_List_To_String (Token_List => UnitName),
                                     GetUnitType (UnitType),
                                     GetReason (ContextManager.Ops.GetUnitStatus (UnitDescriptor)),
                                     ReportFile);

               end if;
               UnitDescriptor := ContextManager.Ops.NextUnitDescriptor (UnitDescriptor);
            end loop;

            XMLReport.EndSection (XMLReport.SUnitsNotFound,
                                  ReportFile);

         else
            FirstTime := True;
            UnitDescriptor := ContextManager.Ops.FirstUnitDescriptor;
            loop
               exit when UnitDescriptor = ContextManager.NullUnit;
               if ContextManager.Ops.GetUnitStatus (UnitDescriptor) in
                 ContextManager.UnitNotFound
               then
                  SPARK_IO.New_Line (ReportFile, 1);
                  if FirstTime then
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_Line (ReportFile,
                                        "The following were required during " &
                                          "analysis but could not be located:", 0);
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_String (ReportFile, "   Unit name", 0);
                     SPARK_IO.Set_Col (ReportFile, StatusCol);
                     SPARK_IO.Put_String (ReportFile, "Unit type", 0);
                     SPARK_IO.Set_Col (ReportFile, ReasonCol);
                     SPARK_IO.Put_Line (ReportFile, "Reason", 0);
                     SPARK_IO.Put_String (ReportFile, "   ---------", 0);
                     SPARK_IO.Set_Col (ReportFile, StatusCol);
                     SPARK_IO.Put_String (ReportFile, "---------", 0);
                     SPARK_IO.Set_Col (ReportFile, ReasonCol);
                     SPARK_IO.Put_Line (ReportFile, "------", 0);
                     FirstTime := False;
                  end if;
                  ContextManager.Ops.GetUnitName (UnitDescriptor, UnitName, UnitType);
                  SPARK_IO.Put_String (ReportFile, "   ", 0);
                  PrintLexTokenList (ReportFile, UnitName);
                  SPARK_IO.Set_Col (ReportFile, StatusCol);
                  PrintUnitType (ReportFile, UnitType);
                  SPARK_IO.Set_Col (ReportFile, ReasonCol);
                  PrintReason (ReportFile, ContextManager.Ops.GetUnitStatus (UnitDescriptor));
               end if;
               UnitDescriptor := ContextManager.Ops.NextUnitDescriptor (UnitDescriptor);
            end loop;
            SPARK_IO.New_Line (ReportFile, 2);
         end if;
      end OutputUnitsRequiredButNotFound;


      procedure OutputUnitsWithCyclicRequirements (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitHeap;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.Ops.UnitHeap,
      --#                                LexTokenManager.State,
      --#                                ReportFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                ContextManager.Ops.UnitHeap,
      --#                                LexTokenManager.State;
      is
         StatusCol : constant Positive := 50;

         UnitDescriptor : ContextManager.UnitDescriptors;
         UnitName       : LexTokenLists.Lists;
         UnitType       : ContextManager.UnitTypes;

         FirstTime      : Boolean;
      begin
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SCyclicRequirements,
                                    ReportFile);
            UnitDescriptor := ContextManager.Ops.FirstUnitDescriptor;
            loop
               exit when UnitDescriptor = ContextManager.NullUnit;
               if ContextManager.Ops.UnitInCycle (UnitDescriptor) then
                  ContextManager.Ops.GetUnitName (UnitDescriptor, UnitName, UnitType);
                  XMLReport.Unit (LexTokenLists.Token_List_To_String (Token_List => UnitName),
                                  GetUnitType (UnitType),
                                  ReportFile);
               end if;
               UnitDescriptor := ContextManager.Ops.NextUnitDescriptor (UnitDescriptor);
            end loop;
            XMLReport.EndSection (XMLReport.SCyclicRequirements,
                                  ReportFile);
         else
            FirstTime := True;
            UnitDescriptor := ContextManager.Ops.FirstUnitDescriptor;
            loop
               exit when UnitDescriptor = ContextManager.NullUnit;
               if ContextManager.Ops.UnitInCycle (UnitDescriptor) then
                  SPARK_IO.New_Line (ReportFile, 1);
                  if FirstTime then
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_Line (ReportFile,
                                        "A cycle was detected when determining the required units " &
                                          "of the following:", 0);
                     SPARK_IO.New_Line (ReportFile, 1);
                     SPARK_IO.Put_String (ReportFile, "   Unit name", 0);
                     SPARK_IO.Set_Col (ReportFile, StatusCol);
                     SPARK_IO.Put_Line (ReportFile, "Unit type", 0);
                     SPARK_IO.Put_String (ReportFile, "   ---------", 0);
                     SPARK_IO.Set_Col (ReportFile, StatusCol);
                     SPARK_IO.Put_Line (ReportFile, "---------", 0);
                     FirstTime := False;
                  end if;
                  ContextManager.Ops.GetUnitName (UnitDescriptor, UnitName, UnitType);
                  SPARK_IO.Put_String (ReportFile, "   ", 0);
                  PrintLexTokenList (ReportFile, UnitName);
                  SPARK_IO.Set_Col (ReportFile, StatusCol);
                  PrintUnitType (ReportFile, UnitType);
               end if;
               UnitDescriptor := ContextManager.Ops.NextUnitDescriptor (UnitDescriptor);
            end loop;
            if not FirstTime then
               SPARK_IO.New_Line (ReportFile, 2);
            end if;

         end if;
      end OutputUnitsWithCyclicRequirements;



      procedure OutputCommandLine (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                ReportFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content;
      is
         TmpString  : EStrings.T;
         Option_Str : ELStrings.T;

         function PlainOutput (Est : EStrings.T)
                              return EStrings.T
         --# global in CommandLineData.Content;
         is
            Result : EStrings.T;
         begin
            if CommandLineData.Content.PlainOutput and then
              not CommandLineData.Content.XML then
               Result := EStrings.Lower_Case (E_Str => Est);
            else
               Result := Est;
            end if;
            return Result;
         end PlainOutput;

      begin

         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SCommandline,
                                    ReportFile);
         else
            SPARK_IO.Put_Line (ReportFile, "Options:", 0);
         end if;

         CommandLineData.Output_Command_Line (Prefix     => "    ",
                                              XML        => CommandLineData.Content.XML,
                                              Option_Str => Option_Str);
         ELStrings.Put_String (File  => ReportFile,
                               E_Str => Option_Str);

         --# assert True;

         if not CommandLineData.Content.XML then
            SPARK_IO.New_Line (ReportFile, 1);
            SPARK_IO.Put_Line (ReportFile, "Selected files:", 0);
         end if;

         --# assert True;

         for I in CommandLineData.SourceFileCounts
           range 1 .. CommandLineData.Content.NumberSource
         loop
            TmpString := PlainOutput (CommandLineData.Content.SourceFileList (I).SourceFileName);

            --# accept F, 41, "Stable expression OK here";
            if CommandLineData.Content.XML then
               XMLReport.Filename (Plain_Output => CommandLineData.Content.PlainOutput,
                                   File         => TmpString);
               EStrings.Put_String (File  => ReportFile,
                                    E_Str => TmpString);
            else
               SPARK_IO.Put_String (ReportFile, "   ", 0);
               EStrings.Put_Line (File  => ReportFile,
                                  E_Str => TmpString);
            end if;
            --# end accept;
         end loop;

         --# assert True;

         if CommandLineData.Content.XML then
            XMLReport.EndSection (XMLReport.SCommandline,
                                  ReportFile);
         else
            SPARK_IO.New_Line (ReportFile, 2);
         end if;
      end OutputCommandLine;

      procedure OutputIndexList (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     IndexManager.State;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                IndexManager.State,
      --#                                LexTokenManager.State,
      --#                                ReportFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                IndexManager.State;
      is
      begin
         IndexManager.ListIndexFile (ReportFile => ReportFile);
      end OutputIndexList;

      procedure OutputMetaFileList (ReportFile : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys,
      --#         XMLReport.State   from CommandLineData.Content,
      --#                                ReportFile,
      --#                                SPARK_IO.File_Sys,
      --#                                XMLReport.State;
      is
         MetaFileUsed : Boolean := False;
         FileName : EStrings.T;
      begin
         if CommandLineData.Content.XML then
            XMLReport.StartSection (XMLReport.SMetaFiles,
                                    ReportFile);
         else
            SPARK_IO.New_Line (ReportFile, 1);
         end if;

         for CurrentSource in CommandLineData.SourceFilePositions
            range 1 .. CommandLineData.Content.NumberSource
         loop
            FileName := CommandLineData.Content.SourceFileList (CurrentSource).SourceFileName;
            if EStrings.Get_Element (E_Str => FileName,
                                     Pos   => 1) = '@' then
               MetaFile.ReportFileContent (ReportFile, FileName, MetaFileUsed);
            end if;
         end loop;
         if not MetaFileUsed and not CommandLineData.Content.XML then
            SPARK_IO.New_Line (ReportFile, 1);
            SPARK_IO.Put_Line (ReportFile, "No Meta Files used", 0);
         end if;

         if CommandLineData.Content.XML then
            XMLReport.EndSection (XMLReport.SMetaFiles,
                                    ReportFile);
         end if;

      end OutputMetaFileList;

   begin  --OutputReportFile

      if CommandLineData.Content.Report then
         CreateReportFile (ReportFile, OK);
         if OK then
            if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
               ScreenEcho.Echo (EStrings.Copy_String (Str => Report_Message));
            end if;
            OutputCommandLine (ReportFile);
            OutputIndexList (ReportFile);
            OutputMetaFileList (ReportFile);
            ErrorHandler.OutputWarningList (ReportFile);
            Dictionary.OutputTargetDataFile (ReportFile);
            ConfigFile.OutputConfigFile (ReportFile);
            OutputFileList (ReportFile);
            OutputUnitsRequiredButNotFound (ReportFile);
            OutputUnitsWithCyclicRequirements (ReportFile);
            if CommandLineData.Content.XML then
               XMLReport.EndSection (XMLReport.SPrologue,
                                     ReportFile);
            end if;
            OutputErrorList (ReportFile,
                             ReportFileErrorList);
            if CommandLineData.Content.SyntaxOnly then
               SPARK_IO.New_Line (ReportFile, 1);
               SPARK_IO.Put_Line (ReportFile,
                                  "Warning: No semantic checks carried out, text " &
                                  "may not be legal SPARK", 0);
            end if;
            if not CommandLineData.Content.DoInformationFlow then
               SPARK_IO.New_Line (ReportFile, 1);
               SPARK_IO.Put_Line (ReportFile,
                                  "Note: Information flow analysis not carried out", 0);
            end if;
            if CommandLineData.Content.WriteStatistics then
               Statistics.WriteOutput (ReportFile);
            end if;
            if not CommandLineData.Content.PlainOutput then
               ErrorHandler.OutputReferenceList (ReportFile);
            end if;
            if CommandLineData.Content.XML then
               XMLReport.EndSection (XMLReport.SReport,
                                     ReportFile);
            else
               ShowEndOfFile (ReportFile);
            end if;
            --# accept F, 10, ReportFile, "Final assignment after close";
            CloseReportFile (ReportFile);
            --# end accept;
            if CommandLineData.Content.HTML then
               SparkHTML.GenReportHTML;
            end if;
         end if;
      end if;
   end OutputReportFile;

   procedure EchoListingFile (FileName : in EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                FileName;
   is
      Message : EStrings.T;
   begin
      if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
         Message := EStrings.Copy_String (Str => "Generating listing file ");
         if CommandLineData.Content.PlainOutput then
            EStrings.Append_Examiner_String
              (E_Str1 => Message,
               E_Str2 => EStrings.Lower_Case (E_Str => FileName));
         else
            EStrings.Append_Examiner_String (E_Str1 => Message,
                                             E_Str2 => FileName);
         end if;
         ScreenEcho.Echo (Message);
      end if;
   end EchoListingFile;

   procedure OutputListingFile (FileDescriptor : in ContextManager.FileDescriptors)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ContextManager.Ops.FileHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ContextManager.Ops.FileHeap,
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          ContextManager.Ops.FileHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          FileDescriptor,
   --#                                          LexTokenManager.State,
   --#                                          SPARK_IO.File_Sys;
   is
      ListingFile          : SPARK_IO.File_Type := SPARK_IO.Null_File;
      PlainListingFileName : EStrings.T;
      ListingFileName      : EStrings.T;
      Status               : SPARK_IO.File_Status;
      ErrorContext         : ErrorHandler.ErrorContexts;
   begin
      ContextManager.Ops.GetListingFileName (FileDescriptor, ListingFileName);

      PlainListingFileName := ListingFileName;
      CommandLineData.Normalize_FileName_To_Output_Directory (ListingFileName);

      ListingFileName := FileSystem.CaseOfFilesForCreate (ListingFileName);
      EStrings.Create (File         => ListingFile,
                       Name_Of_File => ListingFileName,
                       Form_Of_File => "",
                       Status       => Status);
      if Status /= SPARK_IO.Ok then
         ScreenEcho.Put_String ("Listing file, ");
         PrintFileName (SPARK_IO.Standard_Output, ListingFileName, False);
         ScreenEcho.Put_Line (", could not be created");
      else
         EchoListingFile (PlainListingFileName);
         File_Utils.PrintAHeader (ListingFile,
                                  "Listing of SPARK Text",
                                  File_Utils.OtherFile);
         ContextManager.Ops.GetErrorContext (FileDescriptor, ErrorContext);
         ErrorHandler.SetErrorContext (ErrorContext);
         ErrorHandler.PrintErrors (ListingFile, Error_Types.ForListing);
         if CommandLineData.Content.SyntaxOnly then

            -- HTML directives:
            --! <NameFormat> <Name>
            --! <ErrorFormat> <Error>

            -- HTML output
            --! <Name> warning-no-semantic-checks
            --! <Error> Warning : No semantic checks carried out, text may not be legal SPARK.
            --! Issued when the Examiner is used solely to check the syntax of a
            --! SPARK text: this does not check the semantics of a program
            --! (e.g. the correctness of the annotations) and
            --! therefore does not guarantee that a program is legal SPARK.
            SPARK_IO.New_Line (ListingFile, 1);
            SPARK_IO.Put_Line (ListingFile,
                               "Warning: No semantic checks carried out, text " &
                               "may not be legal SPARK", 0);
         end if;
         if not CommandLineData.Content.DoInformationFlow then
            --! <Name> note-information-flow
            --! <Error> Note: Information flow analysis not carried out
            --! This is issued as a reminder that information flow analysis has
            --! not been carried out in this run of the Examiner: information
            --! flow errors may be present undetected in the text analysed
            SPARK_IO.New_Line (ListingFile, 1);
            SPARK_IO.Put_Line (ListingFile,
                               "Note: Information flow analysis not carried out", 0);
         end if;

         case CommandLineData.Content.LanguageProfile is
            when CommandLineData.SPARK83 =>
               --! <Name> note-ada-83
               --! <Error> Note: Ada 83 language rules selected
               --! Issued when the Examiner is used in SPARK 83 mode
               SPARK_IO.New_Line (ListingFile, 1);
               SPARK_IO.Put_Line (ListingFile,
                                  "Note: Ada83 language rules selected", 0);

            when CommandLineData.SPARK95 |
              CommandLineData.SPARK2005 =>
               null;
         end case;

         ErrorHandler.GetErrorContext (ErrorContext);
         ContextManager.Ops.SetErrorContext (FileDescriptor, ErrorContext);
         ShowEndOfFile (ListingFile);
         --# accept F, 10, ListingFile, "Final assignment after close" &
         --#        F, 10, Status,      "Final assignment after close";
         SPARK_IO.Close (ListingFile, Status);
         --# end accept;
      end if;
   end OutputListingFile;


   procedure CreateUnitContext (FileDescriptor : in     ContextManager.FileDescriptors;
                                UnitDescriptor :    out ContextManager.UnitDescriptors)
   --# global in out ContextManager.Ops.FileHeap;
   --#        in out ContextManager.Ops.UnitHeap;
   --# derives ContextManager.Ops.FileHeap,
   --#         ContextManager.Ops.UnitHeap from *,
   --#                                          ContextManager.Ops.FileHeap,
   --#                                          FileDescriptor &
   --#         UnitDescriptor              from ContextManager.Ops.UnitHeap;
   is
      UD2       : ContextManager.UnitDescriptors;
      UnitCount : Natural;
   begin
      ContextManager.Ops.CreateUnitDescriptor (UD2);
      ContextManager.Ops.SetUnitStatus (UD2, ContextManager.UnitCreated);
      ContextManager.Ops.SetFileDescriptor (UD2, FileDescriptor);
      if FileDescriptor /= ContextManager.NullFile then
         ContextManager.Ops.IncUnitCount (FileDescriptor);
         ContextManager.Ops.GetUnitCount (FileDescriptor, UnitCount);
         ContextManager.Ops.SetUnitNumber (UD2, UnitCount);
      else
         ContextManager.Ops.SetUnitNumber (UD2, 0);
      end if;
      ContextManager.Ops.SetVCG (UD2, False);
      UnitDescriptor := UD2;
   end CreateUnitContext;

   procedure ParseCurrentUnit
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitStack;
   --#        in     Dictionary.Dict;
   --#        in out ContextManager.Ops.FileHeap;
   --#        in out ContextManager.Ops.UnitHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --#           out SparkLex.CurrLine;
   --# derives ContextManager.Ops.FileHeap,
   --#         ContextManager.Ops.UnitHeap,
   --#         ErrorHandler.ErrorContext,
   --#         LexTokenManager.State,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys,
   --#         STree.Table                 from CommandLineData.Content,
   --#                                          ContextManager.Ops.FileHeap,
   --#                                          ContextManager.Ops.UnitHeap,
   --#                                          ContextManager.Ops.UnitStack,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.State,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table;
   is
      FileDescriptor : ContextManager.FileDescriptors;
      SourceFile     : SPARK_IO.File_Type;
      Success        : SPARK_IO.File_Status;
      MaxStackSize   : Natural;
      FileEnd        : Boolean;
      Severity       : ErrorHandler.ErrorLevel;
      ParseTree      : STree.SyntaxNode;
      FileContext    : SparkLex.LineContext;
      ErrorContext   : ErrorHandler.ErrorContexts;
      AlreadyClosed  : Boolean;
   begin -- ParseCurrentUnit
      --# accept F, 10, MaxStackSize, "Not required here" &
      --#        F, 33, MaxStackSize, "Not required here" &
      --#        F, 10, Success,      "Not required here" &
      --#        F, 33, Success,      "Not required here" &
      --#        F, 10, SourceFile,   "Final assignment in Close";
      ContextManager.Ops.GetFileDescriptor (ContextManager.Ops.CurrentUnit, FileDescriptor);
      ContextManager.Ops.GetSourceFile (FileDescriptor, SourceFile);
      ContextManager.Ops.GetLineContext (FileDescriptor, FileContext);
      SparkLex.RestoreLineContext (FileContext);
      ContextManager.Ops.GetErrorContext (FileDescriptor, ErrorContext);
      ErrorHandler.SetErrorContext (ErrorContext);
      SPParser.SPParse (SourceFile, MaxStackSize, FileEnd); -- MaxStackSize unused
      STree.RetrieveCurrentRoot (ParseTree);
      ContextManager.Ops.SetParseTree (ContextManager.Ops.CurrentUnit, ParseTree);
      ContextManager.Ops.SetUnitStatus (ContextManager.Ops.CurrentUnit, ContextManager.UnitParsed);
      if FileEnd then
         if ContextManager.Ops.GetFileStatus (FileDescriptor) = ContextManager.FileOpen then
            ContextManager.Ops.SetFileStatus (FileDescriptor, ContextManager.FileEnd);
            SPARK_IO.Close (SourceFile, Success); -- flow error as Success unused
         end if;
         AlreadyClosed := True;
      else
         AlreadyClosed := False;
      end if;
      ErrorHandler.GetErrorSeverity (Severity);
      if Severity = ErrorHandler.Fatal then
         ContextManager.Ops.SetUnitStatus (ContextManager.Ops.CurrentUnit,
                                           ContextManager.UnitAnalysed);
         ContextManager.Ops.SetFileStatus (FileDescriptor, ContextManager.FileEnd);
         if not AlreadyClosed then
            SPARK_IO.Close (SourceFile, Success); -- flow error as Success unused
         end if;
      end if;
      SparkLex.StoreLineContext (FileContext);
      ContextManager.Ops.SetLineContext (FileDescriptor, FileContext);
      ErrorHandler.GetErrorContext (ErrorContext);
      ContextManager.Ops.SetErrorContext (FileDescriptor, ErrorContext);
   end ParseCurrentUnit;

   procedure EchoCompilationUnit (UnitName : in LexTokenLists.Lists;
                                  UnitType : in ContextManager.UnitTypes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                LexTokenManager.State,
   --#                                UnitName,
   --#                                UnitType;
   is
      Message : EStrings.T;
   begin

      if CommandLineData.Content.Echo and then
        not CommandLineData.Content.Brief and then
        UnitType /= ContextManager.InterUnitPragma then

         Message := EStrings.Copy_String (Str => "Examining ");
         case UnitType is
            when ContextManager.MainProgram          =>
               EStrings.Append_String (E_Str => Message,
                                       Str   => "main program ");
            when ContextManager.PackageSpecification =>
               EStrings.Append_String (E_Str => Message,
                                       Str   => "the specification of package ");
            when ContextManager.PackageBody          =>
               EStrings.Append_String (E_Str => Message,
                                       Str   => "the body of package ");
            when ContextManager.SubUnit              =>
               EStrings.Append_String (E_Str => Message,
                                       Str   => "subunit ");
            when ContextManager.GenericDeclaration =>
               EStrings.Append_String (E_Str => Message,
                                       Str   => "generic declaration ");
            when ContextManager.GenericPackageInstantiation =>
               EStrings.Append_String (E_Str => Message,
                                       Str   => "generic package instantiation ");
            when ContextManager.GenericSubprogramBody =>
               EStrings.Append_String (E_Str => Message,
                                       Str   => "generic subprogram body ");

            -- following don't get examined so don't need suitable message
            when ContextManager.InterUnitPragma |
              ContextManager.InvalidUnit |
              ContextManager.PackageOrGenericDeclaration =>
               null;
         end case;

         EStrings.Append_Examiner_String
           (E_Str1 => Message,
            E_Str2 => LexTokenManager.Lex_String_To_String
              (Lex_Str => LexTokenLists.Get_Element (List => UnitName,
                                                     Pos  => LexTokenLists.Get_Length (List => UnitName))));
         ScreenEcho.Echo (Message);
      end if;
   end EchoCompilationUnit;

   function GetLikelyUnitType (PossibleUnitTypes : ContextManager.UnitTypeSets)
      return ContextManager.UnitTypes
   is
      ActualUnitType : ContextManager.UnitTypes;
   begin
      -- since we didn't find the unit we need to guess what kind of thing we were
      -- looking for so that we can make a sensible report about missing units in the
      -- report file.  We use the "sets" of possible units to decide.
      if PossibleUnitTypes (ContextManager.PackageSpecification) then
         if PossibleUnitTypes (ContextManager.GenericDeclaration) then
            -- both package spec and generic dec are member
            ActualUnitType := ContextManager.PackageOrGenericDeclaration;
         else
            -- just package spec is member
            ActualUnitType := ContextManager.PackageSpecification;
         end if;
      elsif PossibleUnitTypes (ContextManager.GenericDeclaration) then
         -- etc
         ActualUnitType := ContextManager.GenericDeclaration;
      elsif PossibleUnitTypes (ContextManager.SubUnit) then
         ActualUnitType := ContextManager.SubUnit;
      else
         ActualUnitType := ContextManager.PackageBody;
      end if;
      return ActualUnitType;
   end GetLikelyUnitType;

   procedure PrintLookupFailMessage (CurrFile     : in EStrings.T;
                                     RequiredUnit : in EStrings.T;
                                     ExpectedSort : in EStrings.T;
                                     Message      : in String)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                CurrFile,
   --#                                ExpectedSort,
   --#                                Message,
   --#                                RequiredUnit;
   is
      CurrStr : EStrings.T;
   begin
      if EStrings.Get_Length (E_Str => CurrFile) > 0 then
         if CommandLineData.Content.PlainOutput then
            CurrStr := EStrings.Lower_Case (E_Str => FileSystem.JustFile
                                              (CurrFile, True));
         elsif CommandLineData.Content.Brief then
            CurrStr := FileSystem.JustFile (CurrFile, True);
         else
            CurrStr := CurrFile;
         end if;
      else
         CurrStr := EStrings.Empty_String;
      end if;

      if CommandLineData.Content.Brief then
         ScreenEcho.Put_ExaminerString (CurrStr);
         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_Integer (1, 0, 10);
         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_Integer (1, 0, 10);
         ScreenEcho.Put_String (": ");
      else
         ScreenEcho.New_Line (1);
         ScreenEcho.Put_String ("In ");
         ScreenEcho.Put_ExaminerString (CurrStr);
         ScreenEcho.Put_String (": ");
      end if;

      ScreenEcho.Put_ExaminerString (ExpectedSort);
      ScreenEcho.Put_Char (' ');
      ScreenEcho.Put_ExaminerString (RequiredUnit);
      ScreenEcho.Put_String (" is not declared in this file ");
      ScreenEcho.Put_String ("and ");
      ScreenEcho.Put_ExaminerString (ExpectedSort);
      ScreenEcho.Put_Char (' ');
      ScreenEcho.Put_ExaminerString (RequiredUnit);
      ScreenEcho.Put_Char (' ');
      ScreenEcho.Put_Line (Message);
   end PrintLookupFailMessage;

   procedure AddRequiredUnit (RequiredUnit      : in LexTokenLists.Lists;
                              PossibleUnitTypes : in ContextManager.UnitTypeSets)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ContextManager.Ops.FileHeap;
   --#        in out ContextManager.Ops.UnitHeap;
   --#        in out ContextManager.Ops.UnitStack;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out IndexManager.LastChar;
   --#        in out IndexManager.State;
   --#        in out LexTokenManager.State;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ContextManager.Ops.FileHeap,
   --#         ContextManager.Ops.UnitHeap,
   --#         ErrorHandler.ErrorContext    from CommandLineData.Content,
   --#                                           ContextManager.Ops.FileHeap,
   --#                                           ContextManager.Ops.UnitHeap,
   --#                                           Dictionary.Dict,
   --#                                           ErrorHandler.ErrorContext,
   --#                                           IndexManager.State,
   --#                                           LexTokenManager.State,
   --#                                           PossibleUnitTypes,
   --#                                           RequiredUnit,
   --#                                           SPARK_IO.File_Sys &
   --#         ContextManager.Ops.UnitStack,
   --#         SPARK_IO.File_Sys            from CommandLineData.Content,
   --#                                           ContextManager.Ops.FileHeap,
   --#                                           ContextManager.Ops.UnitHeap,
   --#                                           ContextManager.Ops.UnitStack,
   --#                                           Dictionary.Dict,
   --#                                           ErrorHandler.ErrorContext,
   --#                                           IndexManager.State,
   --#                                           LexTokenManager.State,
   --#                                           PossibleUnitTypes,
   --#                                           RequiredUnit,
   --#                                           SPARK_IO.File_Sys &
   --#         IndexManager.LastChar,
   --#         IndexManager.State           from *,
   --#                                           CommandLineData.Content,
   --#                                           ContextManager.Ops.UnitHeap,
   --#                                           ErrorHandler.ErrorContext,
   --#                                           IndexManager.State,
   --#                                           LexTokenManager.State,
   --#                                           PossibleUnitTypes,
   --#                                           RequiredUnit,
   --#                                           SPARK_IO.File_Sys &
   --#         LexTokenManager.State,
   --#         SparkLex.CurrLine            from *,
   --#                                           CommandLineData.Content,
   --#                                           ContextManager.Ops.FileHeap,
   --#                                           ContextManager.Ops.UnitHeap,
   --#                                           ErrorHandler.ErrorContext,
   --#                                           IndexManager.State,
   --#                                           LexTokenManager.State,
   --#                                           PossibleUnitTypes,
   --#                                           RequiredUnit,
   --#                                           SPARK_IO.File_Sys;

   is
      UnitDescriptor : ContextManager.UnitDescriptors;
      SourceFileName : LexTokenManager.Lex_String;
      ActualUnitType : ContextManager.UnitTypes;
      FileDescriptor : ContextManager.FileDescriptors;
      FileStr        : EStrings.T;
      Found          : Boolean;
   begin
      Trace ("In MainLoop.AddRequiredUnit");
      TraceUnit (RequiredUnit, PossibleUnitTypes);
      ContextManager.Ops.GetUnitByName (RequiredUnit, PossibleUnitTypes, UnitDescriptor);
      if UnitDescriptor = ContextManager.NullUnit then
         ContextManager.Ops.GetFileDescriptor (ContextManager.Ops.CurrentUnit, FileDescriptor);

         -- A unit which was previously referenced, but could not be located
         -- can have a NullFile entry, and therefore won't have an entry
         -- in the SourceFileName table either.  We need to report something, so...
         if FileDescriptor = ContextManager.NullFile then
            FileStr := EStrings.Copy_String (Str => "UNKNOWN");
         else
            FileStr := LexTokenManager.Lex_String_To_String (Lex_Str => ContextManager.Ops.GetSourceFileName (FileDescriptor));
         end if;

         if CommandLineData.Content.Index then
            Trace ("   unit not known, calling IndexManager.LookUp");
            IndexManager.LookUp (RequiredUnit,
                                 PossibleUnitTypes,
                                 SourceFileName,
                                 ActualUnitType,
                                 Found);
            if Found then
               Trace ("      Found");
               FileDescriptor := ContextManager.Ops.GetFileByName (SourceFileName);
               if FileDescriptor = ContextManager.NullFile then
                  CreateFileContext (LexTokenManager.Lex_String_To_String (Lex_Str => SourceFileName), FileDescriptor);
                  OpenFile (FileDescriptor);
               end if;
               CreateUnitContext (FileDescriptor, UnitDescriptor);
               ContextManager.Ops.SetUnitName (UnitDescriptor, RequiredUnit, ActualUnitType);
               if ContextManager.Ops.GetFileStatus (FileDescriptor) =
                  ContextManager.UnableToOpen
               then
                  Trace ("      unit name found in index file but cannot open specified file");
                  ActualUnitType := GetLikelyUnitType (PossibleUnitTypes);
                  PrintLookupFailMessage (FileStr,
                                          GetUnitName (RequiredUnit),
                                          GetUnitType (ActualUnitType),
                                          "source file entry in index file cannot be opened.");
                  ContextManager.Ops.SetUnitStatus (UnitDescriptor,
                                                    ContextManager.CannotOpenFile);
               else
                  ContextManager.Ops.PushUnit (UnitDescriptor);
               end if;
            else
               Trace ("      not found in Index File");
               ActualUnitType := GetLikelyUnitType (PossibleUnitTypes);
               PrintLookupFailMessage (FileStr,
                                       GetUnitName (RequiredUnit),
                                       GetUnitType (ActualUnitType),
                                       "is not referenced in an index file.");
               CreateUnitContext (ContextManager.NullFile, UnitDescriptor);
               ContextManager.Ops.SetUnitName (UnitDescriptor, RequiredUnit, ActualUnitType);
               ContextManager.Ops.SetUnitStatus (UnitDescriptor,
                                                 ContextManager.NotInIndexFile);
            end if;
         else
            Trace ("   unit not known, and no index file specified");
               ActualUnitType := GetLikelyUnitType (PossibleUnitTypes);
               PrintLookupFailMessage (FileStr,
                                       GetUnitName (RequiredUnit),
                                       GetUnitType (ActualUnitType),
                                       "cannot be located as no index file has been specified.");
            CreateUnitContext (ContextManager.NullFile, UnitDescriptor);
            ContextManager.Ops.SetUnitName (UnitDescriptor, RequiredUnit, ActualUnitType);
            ContextManager.Ops.SetUnitStatus (UnitDescriptor,
                                              ContextManager.NoIndexFile);
         end if;
      else
         Trace ("   unit already seen");
         if ContextManager.Ops.GetUnitStatus (UnitDescriptor) =
            ContextManager.UnitParsed
         then
            ContextManager.Ops.MarkUnitInCycle (UnitDescriptor);
         end if;
         ContextManager.Ops.PushUnit (UnitDescriptor);
      end if;
   end AddRequiredUnit;

   procedure SetUpRequiredUnits
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ContextManager.Ops.FileHeap;
   --#        in out ContextManager.Ops.UnitHeap;
   --#        in out ContextManager.Ops.UnitStack;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out IndexManager.LastChar;
   --#        in out IndexManager.State;
   --#        in out LexTokenManager.State;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ContextManager.Ops.FileHeap,
   --#         ContextManager.Ops.UnitHeap,
   --#         ContextManager.Ops.UnitStack,
   --#         ErrorHandler.ErrorContext,
   --#         IndexManager.LastChar,
   --#         IndexManager.State,
   --#         LexTokenManager.State,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys            from *,
   --#                                           CommandLineData.Content,
   --#                                           ContextManager.Ops.FileHeap,
   --#                                           ContextManager.Ops.UnitHeap,
   --#                                           ContextManager.Ops.UnitStack,
   --#                                           Dictionary.Dict,
   --#                                           ErrorHandler.ErrorContext,
   --#                                           IndexManager.State,
   --#                                           LexTokenManager.State,
   --#                                           SPARK_IO.File_Sys,
   --#                                           STree.Table;
   is
      InheritPtr        : STree.SyntaxNode;
      UnitType          : ContextManager.UnitTypes;
      UnitName          : LexTokenLists.Lists;
      CurrentUnitName   : LexTokenLists.Lists;
      CurrentUnitType   : ContextManager.UnitTypes;

      function FoundTopUnit (WantedName : LexTokenLists.Lists;
                             WantedType : ContextManager.UnitTypes;
                             FoundName  : LexTokenLists.Lists;
                             FoundType  : ContextManager.UnitTypes) return Boolean
      --# global in LexTokenManager.State;
      is
      begin
         Trace ("In MainLoop.FoundTopUnit");
         return LexTokenLists.Get_Length (List => WantedName) = 0 or else
           (LexTokenLists.Eq_Unit (First_Item => WantedName,
                                   Second     => FoundName) and
             WantedType = FoundType);
      end FoundTopUnit;

      procedure ReplaceTopUnit (CurrentUnitName : in LexTokenLists.Lists;
                                CurrentUnitType : in ContextManager.UnitTypes)
      --# global in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ContextManager.Ops.FileHeap;
      --#        in out ContextManager.Ops.UnitHeap;
      --#        in out ContextManager.Ops.UnitStack;
      --# derives ContextManager.Ops.FileHeap,
      --#         ContextManager.Ops.UnitStack from ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           LexTokenManager.State &
      --#         ContextManager.Ops.UnitHeap  from *,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           LexTokenManager.State,
      --#                                           STree.Table;
      is
         NewUnitDescriptor,
         OldUnitDescriptor : ContextManager.UnitDescriptors;
         FileDescriptor    : ContextManager.FileDescriptors;
         FoundUnit         : LexTokenLists.Lists;
         FoundUnitType     : ContextManager.UnitTypes;
         FoundUnitTypeSet  : ContextManager.UnitTypeSets;
         FoundUnitDescriptor : ContextManager.UnitDescriptors;
         FoundFileDescriptor : ContextManager.FileDescriptors;
         ParseTree         : STree.SyntaxNode;
      begin
         --# accept F, 10, OldUnitDescriptor, "Not required here" &
         --#        F, 33, OldUnitDescriptor, "Not required here";
         Trace ("In MainLoop.ReplaceTopUnit");
         if LexTokenLists.Get_Length (List => CurrentUnitName) = 0 then
            ContextManager.Ops.PopUnit (OldUnitDescriptor);
         else
            ContextManager.Ops.GetFileDescriptor (ContextManager.Ops.CurrentUnit,
                                                  FileDescriptor);
            FoundFileDescriptor := FileDescriptor;
            ContextManager.Ops.GetUnitName (ContextManager.Ops.CurrentUnit,
                                            FoundUnit, FoundUnitType);
            FoundUnitTypeSet :=
               ContextManager.UnitTypeSets'(others => False);
            FoundUnitTypeSet (FoundUnitType) := True;
            ContextManager.Ops.GetUnitByName (FoundUnit, FoundUnitTypeSet,
                                              FoundUnitDescriptor);
            if ContextManager.Ops.GetUnitStatus (FoundUnitDescriptor) =
               ContextManager.UnitCreated then
               ContextManager.Ops.GetFileDescriptor (FoundUnitDescriptor,
                                                     FoundFileDescriptor);
            end if;
            if FoundUnitDescriptor /= ContextManager.Ops.CurrentUnit and then
               FoundFileDescriptor = FileDescriptor
            then -- special case of earlier pending unit found
               ContextManager.Ops.SetUnitStatus (FoundUnitDescriptor,
                                                 ContextManager.UnitParsed);
               ContextManager.Ops.GetParseTree (ContextManager.Ops.CurrentUnit,
                                                ParseTree);
               ContextManager.Ops.SetParseTree (FoundUnitDescriptor,
                                                ParseTree);
               ContextManager.Ops.SetParseTree (ContextManager.Ops.CurrentUnit,
                                                STree.NullNode);
               ContextManager.Ops.SetUnitName (ContextManager.Ops.CurrentUnit,
                                               CurrentUnitName,
                                               CurrentUnitType);
               case ContextManager.Ops.GetFileStatus (FileDescriptor) is
                  when ContextManager.FileEnd =>
                     ContextManager.Ops.SetUnitStatus (ContextManager.Ops.CurrentUnit,
                                                       ContextManager.UnableToLocate);
                     ContextManager.Ops.PopUnit (OldUnitDescriptor);
                  when ContextManager.UnableToOpen =>
                     ContextManager.Ops.SetUnitStatus (ContextManager.Ops.CurrentUnit,
                                                       ContextManager.CannotOpenFile);
                     ContextManager.Ops.PopUnit (OldUnitDescriptor);
                  when others =>
                     ContextManager.Ops.SetUnitStatus (ContextManager.Ops.CurrentUnit,
                                                       ContextManager.UnitCreated);
               end case;
            else -- 'normal' case
               CreateUnitContext (FileDescriptor, NewUnitDescriptor);
               ContextManager.Ops.SetUnitStatus (ContextManager.Ops.CurrentUnit,
                                                 ContextManager.UnitDeferred);
               ContextManager.Ops.PopUnit (OldUnitDescriptor);
               ContextManager.Ops.SetUnitName (NewUnitDescriptor,
                                               CurrentUnitName,
                                               CurrentUnitType);
               case ContextManager.Ops.GetFileStatus (FileDescriptor) is
                  when ContextManager.FileEnd =>
                     ContextManager.Ops.SetUnitStatus (NewUnitDescriptor,
                                                       ContextManager.UnableToLocate);
                  when ContextManager.UnableToOpen =>
                     ContextManager.Ops.SetUnitStatus (NewUnitDescriptor,
                                                       ContextManager.CannotOpenFile);
                  when others =>
                     ContextManager.Ops.PushUnit (NewUnitDescriptor);
               end case;
            end if;
         end if;
      end ReplaceTopUnit;


      procedure AddInheritedPackages
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     STree.Table;
      --#        in out ContextManager.Ops.FileHeap;
      --#        in out ContextManager.Ops.UnitHeap;
      --#        in out ContextManager.Ops.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.LastChar;
      --#        in out IndexManager.State;
      --#        in out InheritPtr;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.Ops.FileHeap,
      --#         ContextManager.Ops.UnitHeap,
      --#         ContextManager.Ops.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         IndexManager.LastChar,
      --#         IndexManager.State,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys            from *,
      --#                                           CommandLineData.Content,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           Dictionary.Dict,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           InheritPtr,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           STree.Table &
      --#         InheritPtr                   from *,
      --#                                           STree.Table;
      is
         RequiredUnit      : LexTokenLists.Lists;
         Found             : Boolean;

         -- If a package is inherited and therefore required we do not want to add it
         -- to the required unit list if it is predefined.  Currently only
         -- Ada.Characters.Latin_1 is predefined. If Ravenscar is selected, package
         -- Ada.Real_Time is also predefined.
         function IsPredefinedPackage return Boolean
         --# global in CommandLineData.Content;
         --#        in LexTokenManager.State;
         --#        in RequiredUnit;
         is
            Result : Boolean;
         begin
            case LexTokenLists.Get_Length (List => RequiredUnit) is
               when 1 =>
                  -- Ada
                  Result :=  LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                            Pos  => 1),
                     Lex_Str2 => LexTokenManager.Ada_Token) = LexTokenManager.Str_Eq;
               when 2 =>
                  -- Ada.Real_Time, Ada.Characters or Ada.Interrupts
                  Result :=
                    LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                            Pos  => 1),
                     Lex_Str2 => LexTokenManager.Ada_Token) = LexTokenManager.Str_Eq and then
                    (LexTokenManager.Lex_String_Case_Insensitive_Compare
                       (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                               Pos  => 2),
                        Lex_Str2 => LexTokenManager.Characters_Token) = LexTokenManager.Str_Eq or else
                       (CommandLineData.RavenscarSelected and then
                          (LexTokenManager.Lex_String_Case_Insensitive_Compare
                             (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                                     Pos  => 2),
                              Lex_Str2 => LexTokenManager.Real_Time_Token) = LexTokenManager.Str_Eq or else
                             LexTokenManager.Lex_String_Case_Insensitive_Compare
                             (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                                     Pos  => 2),
                              Lex_Str2 => LexTokenManager.Interrupts_Token) = LexTokenManager.Str_Eq or else
                             LexTokenManager.Lex_String_Case_Insensitive_Compare
                             (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                                     Pos  => 2),
                              Lex_Str2 => LexTokenManager.Synchronous_Task_Control_Token) = LexTokenManager.Str_Eq)));
               when 3 =>
                  -- Ada.Characters.Latin_1
                  Result := LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                            Pos  => 1),
                     Lex_Str2 => LexTokenManager.Ada_Token) = LexTokenManager.Str_Eq and then
                    LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                            Pos  => 2),
                     Lex_Str2 => LexTokenManager.Characters_Token) = LexTokenManager.Str_Eq and then
                    LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                            Pos  => 3),
                     Lex_Str2 => LexTokenManager.Latin_1_Token) = LexTokenManager.Str_Eq;
               when others =>
                  Result := False;
            end case;
            return Result;
         end IsPredefinedPackage;

         ------------------------

         function SystemNeededAndProvidedByConfigFile return Boolean
         --# global in Dictionary.Dict;
         --#        in LexTokenManager.State;
         --#        in RequiredUnit;
         is
         begin
            return LexTokenLists.Get_Length (List => RequiredUnit) = 1 and then
              LexTokenManager.Lex_String_Case_Insensitive_Compare
              (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                      Pos  => 1),
               Lex_Str2 => LexTokenManager.System_Token) = LexTokenManager.Str_Eq and then
              Dictionary.IsDefined (LexTokenManager.System_Token,
                                    Dictionary.GlobalScope,
                                    Dictionary.ProgramContext);
         end SystemNeededAndProvidedByConfigFile;

         ------------------------

         function IsPredefinedGeneric return Boolean
         --# global in LexTokenManager.State;
         --#        in RequiredUnit;
         is
            Result : Boolean;
         begin
            case LexTokenLists.Get_Length (List => RequiredUnit) is
               when 1 =>
                  -- Unchecked_Conversion
                  Result :=  LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                            Pos  => 1),
                     Lex_Str2 => LexTokenManager.Unchecked_Conversion_Token) = LexTokenManager.Str_Eq;
               when 2 =>
                  -- Ada.Unchecked_Conversion
                  Result :=
                    LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                            Pos  => 1),
                     Lex_Str2 => LexTokenManager.Ada_Token) = LexTokenManager.Str_Eq and then
                    LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => RequiredUnit,
                                                            Pos  => 2),
                     Lex_Str2 => LexTokenManager.Unchecked_Conversion_Token) = LexTokenManager.Str_Eq;
               when others =>
                  Result := False;
            end case;
            return Result;
         end IsPredefinedGeneric;

         ------------------------

      begin -- AddInheritedPackages
         Trace ("In MainLoop.AddInheritedPackages");
         loop
            RequiredUnits.Next (InheritPtr, RequiredUnit, Found);
            exit when not Found;

            -- Ignore dotted package names in 83 mode and ignore predefined packages
            -- in 95 mode.  Ignore System if provided by config file. Ignore
            -- predefined generic units.
            --# accept F, 41, "Expect stable expression here";
            case CommandLineData.Content.LanguageProfile is
               when CommandLineData.SPARK83 =>


                  if LexTokenLists.Get_Length (List => RequiredUnit) <= 1 and then
                    not SystemNeededAndProvidedByConfigFile and then
                    not IsPredefinedGeneric
                  then
                     AddRequiredUnit (RequiredUnit, ContextManager.InheritableItem);
                  end if;

               when CommandLineData.SPARK95 |
                 CommandLineData.SPARK2005 =>

                  if not IsPredefinedPackage and then
                    not SystemNeededAndProvidedByConfigFile and then
                    not IsPredefinedGeneric
                  then
                     AddRequiredUnit (RequiredUnit, ContextManager.InheritableItem);
                  end if;
            end case;
            --# end accept;
         end loop;
      end AddInheritedPackages;

      procedure HandleMainProgram
      --# global in     CommandLineData.Content;
      --#        in     CurrentUnitName;
      --#        in     CurrentUnitType;
      --#        in     Dictionary.Dict;
      --#        in     STree.Table;
      --#        in     UnitName;
      --#        in     UnitType;
      --#        in out ContextManager.Ops.FileHeap;
      --#        in out ContextManager.Ops.UnitHeap;
      --#        in out ContextManager.Ops.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.LastChar;
      --#        in out IndexManager.State;
      --#        in out InheritPtr;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.Ops.FileHeap,
      --#         ContextManager.Ops.UnitHeap,
      --#         ContextManager.Ops.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         IndexManager.LastChar,
      --#         IndexManager.State,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys            from *,
      --#                                           CommandLineData.Content,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           Dictionary.Dict,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           InheritPtr,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           STree.Table,
      --#                                           UnitName,
      --#                                           UnitType &
      --#         InheritPtr                   from *,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           LexTokenManager.State,
      --#                                           STree.Table,
      --#                                           UnitName,
      --#                                           UnitType;
      is
      begin
         Trace ("In MainLoop.HandleMainProgram");
         ContextManager.Ops.SetUnitName (ContextManager.Ops.CurrentUnit, UnitName, UnitType);
         if FoundTopUnit (CurrentUnitName, CurrentUnitType, UnitName, UnitType) then
            AddInheritedPackages;
         else
            ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
         end if;
      end HandleMainProgram;


      procedure HandleInterUnitPragma
      --# global in     CurrentUnitName;
      --#        in     CurrentUnitType;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in     UnitName;
      --#        in     UnitType;
      --#        in out ContextManager.Ops.FileHeap;
      --#        in out ContextManager.Ops.UnitHeap;
      --#        in out ContextManager.Ops.UnitStack;
      --# derives ContextManager.Ops.FileHeap,
      --#         ContextManager.Ops.UnitStack from ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           LexTokenManager.State,
      --#                                           UnitName,
      --#                                           UnitType &
      --#         ContextManager.Ops.UnitHeap  from *,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           LexTokenManager.State,
      --#                                           STree.Table,
      --#                                           UnitName,
      --#                                           UnitType;
      is
         NewUnitName       : LexTokenLists.Lists;
      begin
         Trace ("In MainLoop.HandleInterUnitPragma");
         NewUnitName := LexTokenLists.Null_List;
         ContextManager.Ops.SetUnitName
            (ContextManager.Ops.CurrentUnit, NewUnitName, UnitType);
         if not FoundTopUnit (CurrentUnitName, CurrentUnitType, UnitName, UnitType) then
            ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
         end if;
      end HandleInterUnitPragma;

      procedure HandlePackageSpecification
      --# global in     CommandLineData.Content;
      --#        in     CurrentUnitName;
      --#        in     CurrentUnitType;
      --#        in     Dictionary.Dict;
      --#        in     STree.Table;
      --#        in     UnitName;
      --#        in     UnitType;
      --#        in out ContextManager.Ops.FileHeap;
      --#        in out ContextManager.Ops.UnitHeap;
      --#        in out ContextManager.Ops.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.LastChar;
      --#        in out IndexManager.State;
      --#        in out InheritPtr;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.Ops.FileHeap,
      --#         ContextManager.Ops.UnitHeap,
      --#         ContextManager.Ops.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         IndexManager.LastChar,
      --#         IndexManager.State,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys            from *,
      --#                                           CommandLineData.Content,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           Dictionary.Dict,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           InheritPtr,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           STree.Table,
      --#                                           UnitName,
      --#                                           UnitType &
      --#         InheritPtr                   from *,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           LexTokenManager.State,
      --#                                           STree.Table,
      --#                                           UnitName,
      --#                                           UnitType;
      is
         procedure AddParent (UnitName : in LexTokenLists.Lists)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in out ContextManager.Ops.FileHeap;
         --#        in out ContextManager.Ops.UnitHeap;
         --#        in out ContextManager.Ops.UnitStack;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out IndexManager.LastChar;
         --#        in out IndexManager.State;
         --#        in out LexTokenManager.State;
         --#        in out SparkLex.CurrLine;
         --#        in out SPARK_IO.File_Sys;
         --# derives ContextManager.Ops.FileHeap,
         --#         ContextManager.Ops.UnitHeap,
         --#         ErrorHandler.ErrorContext    from CommandLineData.Content,
         --#                                           ContextManager.Ops.FileHeap,
         --#                                           ContextManager.Ops.UnitHeap,
         --#                                           Dictionary.Dict,
         --#                                           ErrorHandler.ErrorContext,
         --#                                           IndexManager.State,
         --#                                           LexTokenManager.State,
         --#                                           SPARK_IO.File_Sys,
         --#                                           UnitName &
         --#         ContextManager.Ops.UnitStack,
         --#         SPARK_IO.File_Sys            from CommandLineData.Content,
         --#                                           ContextManager.Ops.FileHeap,
         --#                                           ContextManager.Ops.UnitHeap,
         --#                                           ContextManager.Ops.UnitStack,
         --#                                           Dictionary.Dict,
         --#                                           ErrorHandler.ErrorContext,
         --#                                           IndexManager.State,
         --#                                           LexTokenManager.State,
         --#                                           SPARK_IO.File_Sys,
         --#                                           UnitName &
         --#         IndexManager.LastChar,
         --#         IndexManager.State           from *,
         --#                                           CommandLineData.Content,
         --#                                           ContextManager.Ops.UnitHeap,
         --#                                           ErrorHandler.ErrorContext,
         --#                                           IndexManager.State,
         --#                                           LexTokenManager.State,
         --#                                           SPARK_IO.File_Sys,
         --#                                           UnitName &
         --#         LexTokenManager.State,
         --#         SparkLex.CurrLine            from *,
         --#                                           CommandLineData.Content,
         --#                                           ContextManager.Ops.FileHeap,
         --#                                           ContextManager.Ops.UnitHeap,
         --#                                           ErrorHandler.ErrorContext,
         --#                                           IndexManager.State,
         --#                                           LexTokenManager.State,
         --#                                           SPARK_IO.File_Sys,
         --#                                           UnitName;
         is
            RequiredUnit : LexTokenLists.Lists;
            Dummy_Item   : LexTokenManager.Lex_String;
         begin
            Trace ("In MainLoop.HandlePackageSpecification.AddParent");
            RequiredUnit := UnitName;
            --# accept F, 10, Dummy_Item, "Ineffective assignment here OK";
            LexTokenLists.Pop (List => RequiredUnit,
                               Item => Dummy_Item);
            --# end accept;
            AddRequiredUnit (RequiredUnit, ContextManager.PackageSpecificationSet);
            --# accept F, 33, Dummy_Item, "Expect Dummy_Item unused";
         end AddParent;

      begin -- HandlePackageSpecification
         Trace ("In MainLoop.HandlePackageSpecification");
         ContextManager.Ops.SetUnitName (ContextManager.Ops.CurrentUnit, UnitName, UnitType);
         if FoundTopUnit (CurrentUnitName, CurrentUnitType, UnitName, UnitType) then
            AddInheritedPackages;
            case CommandLineData.Content.LanguageProfile is
               when CommandLineData.SPARK83 =>
                  null;
               when CommandLineData.SPARK95 |
                 CommandLineData.SPARK2005 =>
                  if LexTokenLists.Get_Length (List => UnitName) > 1 and then
                    LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => UnitName,
                                                            Pos  => 1),
                     Lex_Str2 => LexTokenManager.Ada_Token) /= LexTokenManager.Str_Eq and then
                    LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => UnitName,
                                                            Pos  => 1),
                     Lex_Str2 => LexTokenManager.System_Token) /= LexTokenManager.Str_Eq then
                     AddParent (UnitName);
                  end if;
            end case;
         else
            ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
         end if;
      end HandlePackageSpecification;

      procedure HandlePackageBody
      --# global in     CommandLineData.Content;
      --#        in     CurrentUnitName;
      --#        in     CurrentUnitType;
      --#        in     Dictionary.Dict;
      --#        in     STree.Table;
      --#        in     UnitName;
      --#        in out ContextManager.Ops.FileHeap;
      --#        in out ContextManager.Ops.UnitHeap;
      --#        in out ContextManager.Ops.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.LastChar;
      --#        in out IndexManager.State;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.Ops.FileHeap,
      --#         ContextManager.Ops.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         IndexManager.LastChar,
      --#         IndexManager.State,
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine,
      --#         SPARK_IO.File_Sys            from *,
      --#                                           CommandLineData.Content,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           Dictionary.Dict,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           UnitName &
      --#         ContextManager.Ops.UnitHeap  from *,
      --#                                           CommandLineData.Content,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           Dictionary.Dict,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           STree.Table,
      --#                                           UnitName;
      is
         Components : IndexManager.ComponentLists;
         Index      : IndexManager.ComponentIndex;
      begin
         Trace ("In MainLoop.HandlePackageBody");
         ContextManager.Ops.SetUnitName (ContextManager.Ops.CurrentUnit,
                                         UnitName,
                                         ContextManager.PackageBody);
         if FoundTopUnit (CurrentUnitName, CurrentUnitType,
                          UnitName, ContextManager.PackageBody)
         then
            case CommandLineData.Content.LanguageProfile is
               when CommandLineData.SPARK83 =>
                  if LexTokenLists.Get_Length (List => UnitName) <= 1 then
                     AddRequiredUnit (UnitName, ContextManager.PackageSpecificationSet);
                  end if;

               when CommandLineData.SPARK95 |
                 CommandLineData.SPARK2005 =>
                  -- look for components in index file
                  IndexManager.LookUpComponents (UnitName, Components);
                  Index := IndexManager.ComponentIndex'First;
                  loop
                     exit when LexTokenLists.Eq_Unit (First_Item => Components (Index),
                                                      Second     => LexTokenLists.Null_List);
                     AddRequiredUnit (Components (Index), ContextManager.PackageSpecificationSet);
                     Index := IndexManager.ComponentIndex'Succ (Index);
                  end loop;

                  if LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => LexTokenLists.Get_Element (List => UnitName,
                                                            Pos  => 1),
                     Lex_Str2 => LexTokenManager.Ada_Token) /= LexTokenManager.Str_Eq then
                     AddRequiredUnit (UnitName, ContextManager.PackageSpecificationSet);
                  end if;

            end case;

         else
            ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
         end if;
      end HandlePackageBody;

      -- PNA annotation below mangled because I have based it on wf_package_body and it is likely
      -- to end up quite close to that.  For the purposes of CFR 1340 I have commented out the
      -- bits of the annotation that aren't needed.
      procedure HandleGenericDeclaration
      --# global in     ContextManager.Ops.UnitStack;
      --#        in     UnitName;
      --#        in out ContextManager.Ops.UnitHeap;
      --# derives ContextManager.Ops.UnitHeap from *,
      --#                                          ContextManager.Ops.UnitStack,
      --#                                          UnitName;
      --#                                          --CurrentUnitName,
      --#                                          --CurrentUnitType; -- &
      --#         --SPARK_IO.FILE_SYS           from *,
      --#         --                                 --Dictionary.Dict,
      --#         --                                 --LexTokenManager.State,
      --#         --                                 --CommandLineData.Content,
      --#         --                                 --ErrorHandler.ErrorContext,
      --#         --                                 --ContextManager.Ops.FileHeap,
      --#         --                                 ContextManager.Ops.UnitHeap,
      --#         --                                 ContextManager.Ops.UnitStack,
      --#         --                                 UnitName,
      --#         --                                 CurrentUnitName,
      --#         --                                 CurrentUnitType &
      --#         --SparkLex.CurrLine           from *,
      --#         --                                 --SPARK_IO.FILE_SYS,
      --#         --                                 --LexTokenManager.State,
      --#         --                                 CommandLineData.Content,
      --#         --                                 --ContextManager.Ops.FileHeap,
      --#         --                                 ContextManager.Ops.UnitHeap,
      --#         --                                 ContextManager.Ops.UnitStack,
      --#         --                                 UnitName,
      --#         --                                 CurrentUnitName,
      --#         --                                 CurrentUnitType;
      is
      begin -- HandleGenericDeclaration
         Trace ("In MainLoop.HandleGenericDeclaration");
         ContextManager.Ops.SetUnitName (ContextManager.Ops.CurrentUnit,
                                         UnitName,
                                         ContextManager.GenericDeclaration);
         -- incomplete, doesn't yet handle required units of a generic unit PNA 21/11/03 GenericTBD
      end HandleGenericDeclaration;

      -- PNA annotation below mangled because I have based it on wf_package_body and it is likely
      -- to end up quite close to that.  For the purposes of CFR 1340 I have commented out the
      -- bits of the annotation that aren't needed.
      procedure HandleGenericPackageInstantiation
      --# global in     ContextManager.Ops.UnitStack;
      --#        in     UnitName;
      --#        in out ContextManager.Ops.UnitHeap;
      --# derives ContextManager.Ops.UnitHeap from *,
      --#                                          ContextManager.Ops.UnitStack,
      --#                                          UnitName;
      --#                                          --CurrentUnitName,
      --#                                          --CurrentUnitType; -- &
      --#         --SPARK_IO.FILE_SYS           from *,
      --#         --                                 --Dictionary.Dict,
      --#         --                                 --LexTokenManager.State,
      --#         --                                 --CommandLineData.Content,
      --#         --                                 --ErrorHandler.ErrorContext,
      --#         --                                 --ContextManager.Ops.FileHeap,
      --#         --                                 ContextManager.Ops.UnitHeap,
      --#         --                                 ContextManager.Ops.UnitStack,
      --#         --                                 UnitName,
      --#         --                                 CurrentUnitName,
      --#         --                                 CurrentUnitType &
      --#         --SparkLex.CurrLine           from *,
      --#         --                                 --SPARK_IO.FILE_SYS,
      --#         --                                 --LexTokenManager.State,
      --#         --                                 CommandLineData.Content,
      --#         --                                 --ContextManager.Ops.FileHeap,
      --#         --                                 ContextManager.Ops.UnitHeap,
      --#         --                                 ContextManager.Ops.UnitStack,
      --#         --                                 UnitName,
      --#         --                                 CurrentUnitName,
      --#         --                                 CurrentUnitType;
      is
      begin -- HandleGenericPackageInstantiation
         Trace ("In MainLoop.HandleGenericPackageInstantiation");
         ContextManager.Ops.SetUnitName (ContextManager.Ops.CurrentUnit,
                                         UnitName,
                                         ContextManager.GenericPackageInstantiation);
         -- incomplete, doesn't yet handle required units of a generic unit PNA 21/11/03 GenericTBD
      end HandleGenericPackageInstantiation;

      procedure HandleGenericSubprogramBody
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     UnitName;
      --#        in out ContextManager.Ops.FileHeap;
      --#        in out ContextManager.Ops.UnitHeap;
      --#        in out ContextManager.Ops.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.LastChar;
      --#        in out IndexManager.State;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.Ops.FileHeap,
      --#         ContextManager.Ops.UnitHeap,
      --#         ContextManager.Ops.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys            from CommandLineData.Content,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           Dictionary.Dict,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           UnitName &
      --#         IndexManager.LastChar,
      --#         IndexManager.State           from *,
      --#                                           CommandLineData.Content,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           UnitName &
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine            from *,
      --#                                           CommandLineData.Content,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           UnitName;
      is
      begin -- HandleGenericSubprogramBody
         Trace ("In MainLoop.HandleGenericSubprogramBody");
         ContextManager.Ops.SetUnitName (ContextManager.Ops.CurrentUnit,
                                         UnitName,
                                         ContextManager.GenericSubprogramBody);
         -- generic body found so we need to see its spec
         AddRequiredUnit (UnitName, ContextManager.GenericDeclarationSet);
      end HandleGenericSubprogramBody;

      procedure HandleSubUnit
      --# global in     CommandLineData.Content;
      --#        in     CurrentUnitName;
      --#        in     CurrentUnitType;
      --#        in     Dictionary.Dict;
      --#        in     STree.Table;
      --#        in     UnitName;
      --#        in out ContextManager.Ops.FileHeap;
      --#        in out ContextManager.Ops.UnitHeap;
      --#        in out ContextManager.Ops.UnitStack;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out IndexManager.LastChar;
      --#        in out IndexManager.State;
      --#        in out InheritPtr;
      --#        in out LexTokenManager.State;
      --#        in out SparkLex.CurrLine;
      --#        in out SPARK_IO.File_Sys;
      --# derives ContextManager.Ops.FileHeap,
      --#         ContextManager.Ops.UnitHeap,
      --#         ContextManager.Ops.UnitStack,
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys            from CommandLineData.Content,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           Dictionary.Dict,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           InheritPtr,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           STree.Table,
      --#                                           UnitName &
      --#         IndexManager.LastChar,
      --#         IndexManager.State           from *,
      --#                                           CommandLineData.Content,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           InheritPtr,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           STree.Table,
      --#                                           UnitName &
      --#         LexTokenManager.State,
      --#         SparkLex.CurrLine            from *,
      --#                                           CommandLineData.Content,
      --#                                           ContextManager.Ops.FileHeap,
      --#                                           ContextManager.Ops.UnitHeap,
      --#                                           ContextManager.Ops.UnitStack,
      --#                                           CurrentUnitName,
      --#                                           CurrentUnitType,
      --#                                           ErrorHandler.ErrorContext,
      --#                                           IndexManager.State,
      --#                                           InheritPtr,
      --#                                           LexTokenManager.State,
      --#                                           SPARK_IO.File_Sys,
      --#                                           STree.Table,
      --#                                           UnitName &
      --#         InheritPtr                   from *,
      --#                                           STree.Table;
      is
         RequiredUnit      : LexTokenLists.Lists;
         NewUnitName       : LexTokenLists.Lists;
         Found             : Boolean;
      begin
         Trace ("In MainLoop.HandleSubUnit");
         RequiredUnits.Next (InheritPtr, RequiredUnit, Found);
         if Found then
            NewUnitName := RequiredUnit;
            LexTokenLists.Append (List => NewUnitName,
                                  Item => LexTokenLists.Get_Element (List => UnitName,
                                                                     Pos  => 1));
            ContextManager.Ops.SetUnitName (ContextManager.Ops.CurrentUnit,
                                            NewUnitName,
                                            ContextManager.SubUnit);
            if FoundTopUnit (CurrentUnitName, CurrentUnitType,
                             NewUnitName, ContextManager.SubUnit)
            then
               if LexTokenLists.Get_Length (List => RequiredUnit) = 1 then
                  AddRequiredUnit (RequiredUnit, ContextManager.BodySet);
               else
                  AddRequiredUnit (RequiredUnit, ContextManager.SubUnitSet);
               end if;
            else
               ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
            end if;
         end if;
      end HandleSubUnit;

      procedure LookAtParseTree
      --# global in     ContextManager.Ops.UnitHeap;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in     STree.Table;
      --#           out InheritPtr;
      --#           out UnitName;
      --#           out UnitType;
      --# derives InheritPtr,
      --#         UnitName,
      --#         UnitType   from ContextManager.Ops.UnitHeap,
      --#                         ContextManager.Ops.UnitStack,
      --#                         STree.Table;
      is
         ParseTree         : STree.SyntaxNode;
      begin
         ContextManager.Ops.GetParseTree (ContextManager.Ops.CurrentUnit, ParseTree);
         RequiredUnits.Init (ParseTree, InheritPtr, UnitType, UnitName);
      end LookAtParseTree;

   begin -- SetUpRequiredUnits
      Trace ("In MainLoop.SetUpRequiredUnits");
      LookAtParseTree;
      ContextManager.Ops.GetUnitName (ContextManager.Ops.CurrentUnit, CurrentUnitName, CurrentUnitType);

      if LexTokenLists.Eq_Unit (First_Item => UnitName,
                                Second     => LexTokenLists.Null_List)
        and then UnitType /= ContextManager.InterUnitPragma
      then
         ContextManager.Ops.SetUnitStatus (ContextManager.Ops.CurrentUnit, ContextManager.NoUnitEntry);
         ReplaceTopUnit (CurrentUnitName, CurrentUnitType);
      else
         --# accept F, 10, InheritPtr, "Not required here";
         case UnitType is
            when ContextManager.MainProgram =>
               HandleMainProgram;
            when ContextManager.PackageSpecification =>
               HandlePackageSpecification;
            when ContextManager.PackageBody =>
               HandlePackageBody;
            when ContextManager.SubUnit =>
               HandleSubUnit;
            when ContextManager.GenericDeclaration =>
               HandleGenericDeclaration;
            when ContextManager.GenericPackageInstantiation =>
               HandleGenericPackageInstantiation;
            when ContextManager.GenericSubprogramBody =>
               HandleGenericSubprogramBody;
            when ContextManager.InterUnitPragma =>
               HandleInterUnitPragma;
            when ContextManager.InvalidUnit |
              ContextManager.PackageOrGenericDeclaration =>
               null;
         end case;
         --# end accept;
      end if;
   end SetUpRequiredUnits;

   procedure ProcessUnits (FileDescriptor : in ContextManager.FileDescriptors)
   --# global in     CommandLineData.Content;
   --#        in out ContextManager.Ops.FileHeap;
   --#        in out ContextManager.Ops.UnitHeap;
   --#        in out ContextManager.Ops.UnitStack;
   --#        in out Declarations.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Graph.Table;
   --#        in out IndexManager.LastChar;
   --#        in out IndexManager.State;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out STree.Table;
   --#        in out VCG.Invoked;
   --#        in out XMLReport.State;
   --# derives ContextManager.Ops.FileHeap,
   --#         ContextManager.Ops.UnitHeap,
   --#         ContextManager.Ops.UnitStack,
   --#         Declarations.State,
   --#         Dictionary.Dict,
   --#         ErrorHandler.ErrorContext,
   --#         Graph.Table,
   --#         IndexManager.LastChar,
   --#         IndexManager.State,
   --#         LexTokenManager.State,
   --#         SparkLex.CurrLine,
   --#         Statistics.TableUsage,
   --#         StmtStack.S,
   --#         STree.Table,
   --#         VCG.Invoked,
   --#         XMLReport.State              from *,
   --#                                           CommandLineData.Content,
   --#                                           ContextManager.Ops.FileHeap,
   --#                                           ContextManager.Ops.UnitHeap,
   --#                                           ContextManager.Ops.UnitStack,
   --#                                           Dictionary.Dict,
   --#                                           ErrorHandler.ErrorContext,
   --#                                           FileDescriptor,
   --#                                           IndexManager.State,
   --#                                           LexTokenManager.State,
   --#                                           SLI.State,
   --#                                           SPARK_IO.File_Sys,
   --#                                           STree.Table &
   --#         SLI.State,
   --#         SPARK_IO.File_Sys            from CommandLineData.Content,
   --#                                           ContextManager.Ops.FileHeap,
   --#                                           ContextManager.Ops.UnitHeap,
   --#                                           ContextManager.Ops.UnitStack,
   --#                                           Dictionary.Dict,
   --#                                           ErrorHandler.ErrorContext,
   --#                                           FileDescriptor,
   --#                                           IndexManager.State,
   --#                                           LexTokenManager.State,
   --#                                           SLI.State,
   --#                                           SPARK_IO.File_Sys,
   --#                                           STree.Table,
   --#                                           XMLReport.State;
   is
      UnitDescriptor      : ContextManager.UnitDescriptors;
      SLI_Unit_Descriptor : ContextManager.UnitDescriptors := ContextManager.NullUnit;
      Severity            : ErrorHandler.ErrorLevel;
      Unit_Name           : LexTokenLists.Lists;
      Unit_Type           : ContextManager.UnitTypes;
      Scope               : Dictionary.Scopes;
      Prefix_Sym          : Dictionary.Symbol;
      Source_Filename     : EStrings.T;
      L_Source_Filename   : EStrings.T;
      File_Status         : FileSystem.TypFileSpecStatus;

      procedure AnalyseCurrentUnit
      --# global in     CommandLineData.Content;
      --#        in     ContextManager.Ops.UnitStack;
      --#        in out ContextManager.Ops.FileHeap;
      --#        in out ContextManager.Ops.UnitHeap;
      --#        in out Declarations.State;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Graph.Table;
      --#        in out LexTokenManager.State;
      --#        in out SLI.State;
      --#        in out SLI_Unit_Descriptor;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out StmtStack.S;
      --#        in out STree.Table;
      --#        in out VCG.Invoked;
      --#           out SparkLex.CurrLine;
      --# derives ContextManager.Ops.FileHeap,
      --#         Declarations.State,
      --#         Dictionary.Dict,
      --#         ErrorHandler.ErrorContext,
      --#         Graph.Table,
      --#         LexTokenManager.State,
      --#         SLI.State,
      --#         SPARK_IO.File_Sys,
      --#         Statistics.TableUsage,
      --#         StmtStack.S,
      --#         STree.Table,
      --#         VCG.Invoked                 from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.Ops.FileHeap,
      --#                                          ContextManager.Ops.UnitHeap,
      --#                                          ContextManager.Ops.UnitStack,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.State,
      --#                                          SLI.State,
      --#                                          SPARK_IO.File_Sys,
      --#                                          STree.Table &
      --#         SLI_Unit_Descriptor         from *,
      --#                                          CommandLineData.Content,
      --#                                          ContextManager.Ops.FileHeap,
      --#                                          ContextManager.Ops.UnitHeap,
      --#                                          ContextManager.Ops.UnitStack,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          SPARK_IO.File_Sys &
      --#         ContextManager.Ops.UnitHeap from *,
      --#                                          ContextManager.Ops.UnitStack &
      --#         SparkLex.CurrLine           from ContextManager.Ops.FileHeap,
      --#                                          ContextManager.Ops.UnitHeap,
      --#                                          ContextManager.Ops.UnitStack;
      is
         FileDescriptor : ContextManager.FileDescriptors;
         SourceFile     : SPARK_IO.File_Type;
         UnitName       : LexTokenLists.Lists;
         UnitType       : ContextManager.UnitTypes;
         Success        : SPARK_IO.File_Status;
         Severity       : ErrorHandler.ErrorLevel;
         FileContext    : SparkLex.LineContext;
         ParseTree      : STree.SyntaxNode;
         ErrorContext   : ErrorHandler.ErrorContexts;
         Do_VCG         : Boolean;
      begin
         ContextManager.Ops.GetParseTree (ContextManager.Ops.CurrentUnit, ParseTree);
         ContextManager.Ops.GetFileDescriptor (ContextManager.Ops.CurrentUnit, FileDescriptor);
         ContextManager.Ops.GetSourceFile (FileDescriptor, SourceFile);
         ContextManager.Ops.GetLineContext (FileDescriptor, FileContext);
         ContextManager.Ops.GetVCG (ContextManager.Ops.CurrentUnit, Do_VCG);
         ContextManager.Ops.GetUnitName (ContextManager.Ops.CurrentUnit, UnitName, UnitType);
         SparkLex.RestoreLineContext (FileContext);
         ContextManager.Ops.GetErrorContext (FileDescriptor, ErrorContext);
         ErrorHandler.SetErrorContext (ErrorContext);
         EchoCompilationUnit (UnitName, UnitType);
         Dictionary.SetCurrentFileName (LexTokenManager.Lex_String_To_String (Lex_Str => ContextManager.Ops.GetSourceFileName (FileDescriptor)));
         if ErrorHandler.Generate_SLI then
            ContextManager.Ops.Get_Unit (Descriptor      => FileDescriptor,
                                         Unit_Descriptor => SLI_Unit_Descriptor);
         end if;
         Sem.CompUnit (TopNode => ParseTree,
                       Do_VCG  => Do_VCG);
         Casing.Check_Node_Casing (Top_Node => ParseTree);
         ErrorHandler.GetErrorSeverity (Severity);

         --# accept F, 10, SourceFile, "Final assignment in Close" &
         --#        F, 10, Success,    "Final assignment in Close" &
         --#        F, 33, Success,    "Final assignment in Close";
         if Severity = ErrorHandler.Fatal then
            ContextManager.Ops.SetFileStatus (FileDescriptor, ContextManager.FileEnd);
            SPARK_IO.Close (SourceFile, Success);
         end if;
         SparkLex.StoreLineContext (FileContext);
         ContextManager.Ops.SetLineContext (FileDescriptor, FileContext);
         ErrorHandler.GetErrorContext (ErrorContext);
         ContextManager.Ops.SetErrorContext (FileDescriptor, ErrorContext);

         -- return the ParseTree to the syntax tree free list, except sub-trees
         -- that are later needed by the VCG.  If the user has selected /rules=none,
         -- then we _can_ return constant declarations as well.
         STree.DeleteSyntaxTree
           (Root          => ParseTree,
            KeepConstants => CommandLineData.Content.ConstantRules /= CommandLineData.NoRules);

         ContextManager.Ops.SetUnitStatus (ContextManager.Ops.CurrentUnit,
                                           ContextManager.UnitAnalysed);
      end AnalyseCurrentUnit;

   begin -- ProcessUnits
      loop
         exit when ContextManager.Ops.GetFileStatus (FileDescriptor) =
            ContextManager.FileEnd;
         CreateUnitContext (FileDescriptor, UnitDescriptor);
         ContextManager.Ops.SetVCG (UnitDescriptor,
                                    CommandLineData.Content.VCG or CommandLineData.Content.DPC);
         ContextManager.Ops.PushUnit (UnitDescriptor);
         loop
            exit when ContextManager.Ops.CurrentUnit = ContextManager.NullUnit;
            case ContextManager.Ops.GetUnitStatus (ContextManager.Ops.CurrentUnit) is
               when ContextManager.NoUnitEntry => null;
               when ContextManager.UnitCreated =>
                  ParseCurrentUnit;
                  ErrorHandler.GetErrorSeverity (Severity);
                  if Severity /= ErrorHandler.Fatal then
                     SetUpRequiredUnits;
                  end if;
               when ContextManager.UnitDeferred =>
                  ContextManager.Ops.SetUnitStatus (ContextManager.Ops.CurrentUnit,
                                                    ContextManager.UnitParsed);
                  SetUpRequiredUnits;
               when ContextManager.UnitParsed =>
                  --# accept F, 41, "Stable expression OK here";
                  if CommandLineData.Content.SyntaxOnly then
                     ContextManager.Ops.PopUnit (UnitDescriptor);
                  else
                     AnalyseCurrentUnit;
                  end if;
                  --# end accept;
               when ContextManager.UnitAnalysed |
                 ContextManager.NoIndexFile .. ContextManager.UnableToLocate =>
                  ContextManager.Ops.PopUnit (UnitDescriptor);
            end case;
         end loop;
         if not ErrorHandler.Syntax_Or_Semantic_Error then
            L_Source_Filename := LexTokenManager.Lex_String_To_String (Lex_Str => ContextManager.Ops.GetSourceFileName (Descriptor => FileDescriptor));
            ContextManager.Ops.GetUnitName
              (Descriptor => UnitDescriptor,
               UnitName   => Unit_Name,
               UnitType   => Unit_Type);
            --# accept F, 41, "Stable expression expected here";
            if CommandLineData.Content.Debug.FileNames then
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => "MAINLOOP ",
                                    Stop => 0);
            end if;
            --# end accept;
            --# accept F, 10, File_Status, "Expect File_Status Unused";
            FileSystem.FindFullFileName
              (FileSpec     => L_Source_Filename,
               FileStatus   => File_Status,
               FullFileName => Source_Filename);
            --# end accept;
            if Unit_Type /= ContextManager.InterUnitPragma then
               IndexManager.Add_Unit (Unit            => Unit_Name,
                                      Unit_Types      => Unit_Type,
                                      Source_Filename => Source_Filename);
            end if;
         end if;
      end loop;
      if ErrorHandler.Generate_SLI and then
        SLI_Unit_Descriptor /= ContextManager.NullUnit then
         ContextManager.Ops.GetUnitName
           (Descriptor => SLI_Unit_Descriptor,
            UnitName   => Unit_Name,
            UnitType   => Unit_Type);
         Scope      := Dictionary.GlobalScope;
         Prefix_Sym := Dictionary.NullSymbol;
         for I in LexTokenLists.Positions range LexTokenLists.Positions'First .. LexTokenLists.Get_Length (List => Unit_Name) loop
            SLI.Look_Up (Prefix      => Prefix_Sym,
                         Scope       => Scope,
                         Subprog_Sym => Dictionary.NullSymbol,
                         Lex_Str     => LexTokenLists.Get_Element (List => Unit_Name,
                                                                   Pos  => I),
                         Pos         => LexTokenManager.Null_Token_Position);
            exit when Prefix_Sym = Dictionary.NullSymbol;
         end loop;
         if Unit_Type = ContextManager.MainProgram or else
           Unit_Type = ContextManager.PackageBody or else
           Unit_Type = ContextManager.SubUnit or else
           (Unit_Type = ContextManager.PackageSpecification and then
              Prefix_Sym /= Dictionary.NullSymbol and then
              not Dictionary.PackageRequiresBody (ThePackage => Prefix_Sym)) then
            ContextManager.Ops.Dependency_Closure (Descriptor => FileDescriptor);
            SLI.Create_File (File_Descriptor => FileDescriptor);
            SLI.Header (File_Descriptor => FileDescriptor);
            SLI.Dump_Xref (File_Descriptor => FileDescriptor);
            SLI.Close_File;
         end if;
      end if;
      --# accept F, 33, File_Status, "Expect File_Status unused";
   end ProcessUnits;

   procedure CloseFile (FileDescriptor      : in ContextManager.FileDescriptors;
                        ReportFileErrorList : in SPARK_IO.File_Type;
                        ReportNeeded        : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.UnitHeap;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     SparkHTML.GenerateHTML;
   --#        in out ContextManager.Ops.FileHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives ContextManager.Ops.FileHeap,
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys,
   --#         XMLReport.State             from CommandLineData.Content,
   --#                                          ContextManager.Ops.FileHeap,
   --#                                          ContextManager.Ops.UnitHeap,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          FileDescriptor,
   --#                                          LexTokenManager.State,
   --#                                          ReportFileErrorList,
   --#                                          ReportNeeded,
   --#                                          SparkHTML.GenerateHTML,
   --#                                          SPARK_IO.File_Sys,
   --#                                          XMLReport.State;
   is
   begin
      if ContextManager.Ops.ListingReqt (FileDescriptor) and not CommandLineData.Content.NoListings then
         OutputListingFile (FileDescriptor);
         if CommandLineData.Content.HTML then
            SparkHTML.GenListingHTML (FileDescriptor);
         end if;
      end if;
      if ReportNeeded then
         AppendToReportFile (ReportFileErrorList,
                             Error_Types.ForReportSelectedFiles,
                             FileDescriptor);
         ContextManager.Ops.SetErrorsReported (FileDescriptor);
      end if;
   end CloseFile;

   procedure ProcessFiles
   is
      FileDescriptor : ContextManager.FileDescriptors;
      FileFound,
      ConfigSuccess,
      ConfigRead,
      DoListing      : Boolean;
      FileName,
      TheFileName,
      TheListingName : EStrings.T;
      TheMetaFile    : MetaFile.MetaFiles;
      ReportFileErrorList : SPARK_IO.File_Type := SPARK_IO.Null_File;

      procedure OpenErrorList
      --# global in out ReportFileErrorList;
      --#        in out SPARK_IO.File_Sys;
      --# derives ReportFileErrorList,
      --#         SPARK_IO.File_Sys   from ReportFileErrorList,
      --#                                  SPARK_IO.File_Sys;
      is
         Unused : SPARK_IO.File_Status;
      begin
         --# accept F, 10, Unused, "Not required here" &
         --#        F, 33, Unused, "Not required here";
         SPARK_IO.Create (ReportFileErrorList, 0, "", "",
                           --to get
                          Unused);
      end OpenErrorList;
      --assignment to Unused is ineffective, consider testing and calling system errors

      procedure TraceMetaFileNames
      --# global in CommandLineData.Content;
      --#        in DoListing;
      --#        in FileFound;
      --#        in TheFileName;
      --#        in TheListingName;
      --# derives null from CommandLineData.Content,
      --#                   DoListing,
      --#                   FileFound,
      --#                   TheFileName,
      --#                   TheListingName;
      is
         --# hide TraceMetaFileNames;
      begin
         if FileFound and CommandLineData.Content.Debug.FileNames then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                 "MetaFile.NextName returns:", 0);

            PrintFileName (SPARK_IO.Standard_Output,
                           TheFileName,
                           CommandLineData.Content.PlainOutput);
            if DoListing then
               SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                    ", ", 0);

               PrintFileName (SPARK_IO.Standard_Output,
                              TheListingName,
                              CommandLineData.Content.PlainOutput);
            else
               SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                                  ", No listing file", 0);
            end if;
            ScreenEcho.New_Line (1);
         end if;
      end TraceMetaFileNames;

   begin --ProcessFiles
      InitialiseProcessing;
      OpenErrorList;
      ErrorHandler.ReadWarningFile;
      Dictionary.ReadTargetDataFile;
      --# accept F, 10, ConfigRead, "Not required here" &
      --#        F, 33, ConfigRead, "Not required here";
      ConfigFile.ReadConfigFile (ConfigRead, ConfigSuccess);
      if ConfigSuccess then
         for CurrentSource in CommandLineData.SourceFilePositions
           range 1 .. CommandLineData.Content.NumberSource
         loop
            FileName := CommandLineData.Content.SourceFileList (CurrentSource).SourceFileName;
            if EStrings.Get_Element (E_Str => FileName,
                                     Pos   => 1) = '@' then --process a meta file
               MetaFile.Create (FileName,
                                TheMetaFile);
               loop
                  MetaFile.NextName (TheMetaFile,
                                       -- to get
                                     TheFileName,
                                     DoListing,
                                     TheListingName,
                                     FileFound);
                  exit when not FileFound;

                  TraceMetaFileNames;

                  PrepareNextArgumentFile (TheFileName,
                                           DoListing,
                                           TheListingName,
                                             -- to get
                                           FileDescriptor);
                  OpenFile (FileDescriptor);
                  if ContextManager.Ops.GetFileStatus (FileDescriptor) /=
                    ContextManager.UnableToOpen
                  then
                     ProcessUnits (FileDescriptor);
                     CloseFile (FileDescriptor,
                                ReportFileErrorList,
                                CommandLineData.Content.Report);
                  end if;
               end loop;

            else --its a single source file
               TheFileName := FileName;
               TheListingName := CommandLineData.Content.SourceFileList (CurrentSource).ListingFileName;
               DoListing := CommandLineData.Content.SourceFileList (CurrentSource).Listing;
               PrepareNextArgumentFile (TheFileName,
                                        DoListing,
                                        TheListingName,
                                          -- to get
                                        FileDescriptor);
               OpenFile (FileDescriptor);
               if ContextManager.Ops.GetFileStatus (FileDescriptor) /=
                 ContextManager.UnableToOpen
               then
                  ProcessUnits (FileDescriptor);
                  CloseFile (FileDescriptor,
                             ReportFileErrorList,
                             CommandLineData.Content.Report);
               end if;
            end if;
         end loop;  --each file on command line
      else
         EchoWarningOrNote (CommandLineData.Content.TargetConfigFile,
                            "Warning - analysis aborted due to errors in target configuration file");
      end if;
      LexTokenManager.Report_Usage;
      Dictionary.ReportUsage;
      STree.ReportUsage;
      SLI.Cleanup;
      --# accept F, 10, ReportFileErrorList, "Final assignment on Close";
      OutputReportFile (ReportFileErrorList);
      --# end accept;
      OutputDictionaryFile;
      if CommandLineData.Content.SyntaxOnly then
         EchoWarningOrNote (CommandLineData.Content.SourceFileList (1).SourceFileName,
                            "Warning - No semantic checks carried out, text " &
                            "may not be legal SPARK");
      end if;
      if CommandLineData.Content.VCG and then -- Any sort of VC generation requested
        not VCG.Invoked and then              -- and VCG not actually invoked on any body
        ErrorHandler.GetErrorsType < 3 then   -- and no syntax or semantic errors

         EchoWarningOrNote (CommandLineData.Content.SourceFileList (1).SourceFileName,
                            "Warning - VC generation requested but no bodies presented. No VCs generated.");
      end if;


      if not CommandLineData.Content.DoInformationFlow then
         EchoWarningOrNote (CommandLineData.Content.SourceFileList (1).SourceFileName,
                            "Note - Information flow analysis not carried out");
      end if;

      case CommandLineData.Content.LanguageProfile is
         when CommandLineData.SPARK83 =>
            EchoWarningOrNote (CommandLineData.Content.SourceFileList (1).SourceFileName,
                               "Note - Ada83 language rules selected");

         when CommandLineData.SPARK95 |
           CommandLineData.SPARK2005 =>
            null;
      end case;

      ErrorHandler.EchoTotalErrorCount;

      if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
         ScreenEcho.New_Line (2);
         ScreenEcho.Put_Line ("-----------End of SPARK Examination--------------------------------");
      end if;
   end ProcessFiles;
end MainLoop;
