-- $Id: indexmanager.adb 15731 2010-01-22 14:29:13Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


-- Options for handling generic units are marked with -- GenOption --
-- With the code commented out, generic declarations are indexed as "specification"
-- With the code included, they are indicated by "generic_declaration"
-- The former is probably better because if a package body is found, there is no way
-- of knowing whether a generic spec or normal spec is required.  We know this for a
-- generic subprgoram body because any subprogram body that is not a main program must
-- be generic.  It seems uneccesarily complicated to change the language of index files
-- so as to say something about generic suprgorams that we can't say about generic
-- packages.  Therefore overloading "specification" seems the best option.



with Ada.Characters.Latin_1;
with CommandLineData;
with FileSystem;
with IndexManager.Cache;
with IndexManager.Index_Table_P;
with SystemErrors;
package body IndexManager
--# own State is IndexManager.Cache.The_Unit_Hash,
--#              IndexManager.Index_Table_P.Index_Table,
--#              IndexManager.Index_Table_P.Fatal_Error;
is

   LastChar   : Character;

   procedure LookInFile (Lookup_Components : in     Boolean;
                         RequiredUnit      : in     LexTokenLists.Lists;
                         PossibleUnitTypes : in     ContextManager.UnitTypeSets;
                         Top_Filename      : in     LexTokenManager.Lex_String;
                         ReturnedFileName  :    out LexTokenManager.Lex_String;
                         ActualUnitType    :    out ContextManager.UnitTypes;
                         Components        :    out ComponentLists;
                         Found             :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out Cache.The_Unit_Hash;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Index_Table_P.Fatal_Error;
   --#        in out Index_Table_P.Index_Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#           out LastChar;
   --# derives ActualUnitType,
   --#         Cache.The_Unit_Hash,
   --#         Components,
   --#         Found,
   --#         Index_Table_P.Index_Table,
   --#         LastChar,
   --#         LexTokenManager.State,
   --#         ReturnedFileName,
   --#         SPARK_IO.File_Sys         from Cache.The_Unit_Hash,
   --#                                        CommandLineData.Content,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Index_Table_P.Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        Lookup_Components,
   --#                                        PossibleUnitTypes,
   --#                                        RequiredUnit,
   --#                                        SPARK_IO.File_Sys,
   --#                                        Top_Filename &
   --#         ErrorHandler.ErrorContext,
   --#         Index_Table_P.Fatal_Error from *,
   --#                                        Cache.The_Unit_Hash,
   --#                                        CommandLineData.Content,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Index_Table_P.Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        Lookup_Components,
   --#                                        PossibleUnitTypes,
   --#                                        RequiredUnit,
   --#                                        SPARK_IO.File_Sys,
   --#                                        Top_Filename;
   is

      LReturnedFileName    : EStrings.T;
      LActualUnitType      : ContextManager.UnitTypes;
      LComponents          : ComponentLists;
      LFound               : Boolean;
      EntryUnit            : LexTokenLists.Lists;
      EntryType            : EntryTypes;
      EntryFileName        : EStrings.T;
      EntryComponents      : ComponentLists;
      ValidEntry           : Boolean;
      FirstInFile          : Boolean;
      Position             : FilePosition;
      LastFileName         : LexTokenManager.Lex_String;
      PossErrorType        : LibraryManagerErrors;
      InAuxIndex           : Boolean;
      Done                 : Boolean;
      FileSpecStatus       : FileSystem.TypFileSpecStatus;
      IndexFile            : SPARK_IO.File_Type;
      FileStatus           : SPARK_IO.File_Status;
      AuxIndexUnit         : LexTokenLists.Lists;
      SuperIndexFound      : Boolean;
      IndexFileName        : EStrings.T;
      CurrentIndexFileName : LexTokenManager.Lex_String;
      Source_Position      : FilePosition;

      procedure UngetChar (Ch : in Character)
      --# global out LastChar;
      --# derives LastChar from Ch;
      is
      begin
         LastChar := Ch;
      end UngetChar;

      function IsEndOfBufferedLine (File : SPARK_IO.File_Type) return Boolean
      --# global in LastChar;
      --#        in SPARK_IO.File_Sys;
      is
         Answer : Boolean;
      begin
         Answer := False;
         if (SPARK_IO.End_Of_Line (File)) then
            if (LastChar = Ada.Characters.Latin_1.NUL) then
               Answer := True;
            end if;
         end if;
         return Answer;
      end IsEndOfBufferedLine;

      function IsEndOfBufferedFile (File : SPARK_IO.File_Type) return Boolean
      --# global in LastChar;
      --#        in SPARK_IO.File_Sys;
      is
         Answer : Boolean;
      begin
         Answer := False;
         if (SPARK_IO.End_Of_File (File)) then
            if (LastChar = Ada.Characters.Latin_1.NUL) then
               Answer := True;
            end if;
         end if;
         return Answer;
      end IsEndOfBufferedFile;

      procedure Read_Buffered_Char (File           : in     SPARK_IO.File_Type;
                                    Index_Filename : in     LexTokenManager.Lex_String;
                                    Ch             :    out Character)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Index_Table_P.Fatal_Error;
      --#        in out LastChar;
      --#        in out SPARK_IO.File_Sys;
      --# derives Ch,
      --#         LastChar                  from File,
      --#                                        LastChar,
      --#                                        SPARK_IO.File_Sys &
      --#         ErrorHandler.ErrorContext,
      --#         Index_Table_P.Fatal_Error from *,
      --#                                        File,
      --#                                        LastChar,
      --#                                        SPARK_IO.File_Sys &
      --#         SPARK_IO.File_Sys         from *,
      --#                                        CommandLineData.Content,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        File,
      --#                                        Index_Filename,
      --#                                        LastChar,
      --#                                        LexTokenManager.State;
      is
         ReadChar      : Character;
         TokenPosition : FilePosition;
      begin
         if LastChar = Ada.Characters.Latin_1.NUL then
            -- No buffered character
            SPARK_IO.Get_Char (File, ReadChar);
            if (ReadChar = Ada.Characters.Latin_1.HT) or (ReadChar = Ada.Characters.Latin_1.CR) then
               ReadChar := ' ';
            end if;
         else
            -- Read the buffered character
            ReadChar := LastChar;
            LastChar := Ada.Characters.Latin_1.NUL;
         end if;

         -- Check for comments; skip line if found
         if ReadChar = '-' then
            if not IsEndOfBufferedFile (File) then
               SPARK_IO.Get_Char (File, ReadChar);
               if ReadChar = '-' then
                  --it is a comment so skip to end of line
                  SPARK_IO.Skip_Line (File, 1);
                  ReadChar := ' ';
               else
                  --only one '-' so its not a comment
                  UngetChar (ReadChar);
                  ReadChar := '-';
               end if;
            else
               -- A single dash then EOF - nonsense
               --error reading; assume no file can end with '-'

               -- In priciple calls to Line and Col may cause a run-time exception
               -- if Line or Col > Count'Last defined in the TEXT_IO package
               -- but in practice this is exception will not be raised because, for
               -- instance, in gnat Count'Last = Natural'Last
               TokenPosition := FilePosition'(Line => SPARK_IO.Line (File),
                                              Col  => SPARK_IO.Col (File));
               Index_Table_P.Output_Error (E              => ESComment,
                                           Source_File    => Index_Filename,
                                           Token_Position => TokenPosition,
                                           Token_String   => EStrings.Empty_String);
            end if;
         end if;

         Ch := ReadChar;
      end Read_Buffered_Char;

      -- Get all the contiguous whitespace (including newlines)
      -- up to end of file or a non-whitespace char.  Unget
      -- non-whitespace.
      procedure Skip_White_Space (File           : in SPARK_IO.File_Type;
                                  Index_Filename : in LexTokenManager.Lex_String)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Index_Table_P.Fatal_Error;
      --#        in out LastChar;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.ErrorContext,
      --#         Index_Table_P.Fatal_Error,
      --#         LastChar,
      --#         SPARK_IO.File_Sys         from *,
      --#                                        CommandLineData.Content,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        File,
      --#                                        Index_Filename,
      --#                                        LastChar,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.File_Sys;
      is
         MyChar : Character;
      begin  -- Skip_White_Space
         while IsEndOfBufferedLine (File) and (not IsEndOfBufferedFile (File))
         loop
            SPARK_IO.Skip_Line (File, 1);
         end loop;
         if not IsEndOfBufferedFile (File) then
            Read_Buffered_Char (File           => File,
                                Index_Filename => Index_Filename,
                                Ch             => MyChar);
            if (MyChar /= ' ') then
               UngetChar (MyChar);
            end if;
            while (MyChar = ' ') and (not IsEndOfBufferedFile (File))
            loop
               if IsEndOfBufferedLine (File) then
                  SPARK_IO.Skip_Line (File, 1);
               else
                  -- Read the next char, as long as it's whitespace
                  Read_Buffered_Char (File           => File,
                                      Index_Filename => Index_Filename,
                                      Ch             => MyChar);
                  if (MyChar /= ' ') then
                     UngetChar (MyChar);
                  end if;
               end if;
            end loop;
         end if;
      end Skip_White_Space;

      procedure Get_Token_Position (IndexFile      : in     SPARK_IO.File_Type;
                                    Index_Filename : in     LexTokenManager.Lex_String;
                                    Position       :    out FilePosition)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Index_Table_P.Fatal_Error;
      --#        in out LastChar;
      --#        in out SPARK_IO.File_Sys;
      --# derives Index_Table_P.Fatal_Error from *,
      --#                                        CommandLineData.Content,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IndexFile,
      --#                                        Index_Filename,
      --#                                        LastChar,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.File_Sys &
      --#         ErrorHandler.ErrorContext,
      --#         LastChar,
      --#         Position,
      --#         SPARK_IO.File_Sys         from CommandLineData.Content,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IndexFile,
      --#                                        Index_Filename,
      --#                                        LastChar,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.File_Sys;
      is
      begin
         Skip_White_Space (File           => IndexFile,
                           Index_Filename => Index_Filename);
         Position := FilePosition'(Line => SPARK_IO.Line (IndexFile),
                                   Col  => SPARK_IO.Col (IndexFile));
      end Get_Token_Position;

      procedure Read_Entry (IndexFile       : in     SPARK_IO.File_Type;
                            Index_Filename  : in     LexTokenManager.Lex_String;
                            EntryUnitExport :    out LexTokenLists.Lists;
                            EntryType       :    out EntryTypes;
                            EntryFileName   :    out EStrings.T;
                            EntryComponents :    out ComponentLists;
                            ValidEntry      :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Index_Table_P.Fatal_Error;
      --#        in out LastChar;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives EntryComponents,
      --#         EntryFileName,
      --#         EntryType,
      --#         EntryUnitExport,
      --#         ErrorHandler.ErrorContext,
      --#         LastChar,
      --#         LexTokenManager.State,
      --#         SPARK_IO.File_Sys,
      --#         ValidEntry                from CommandLineData.Content,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IndexFile,
      --#                                        Index_Filename,
      --#                                        LastChar,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.File_Sys &
      --#         Index_Table_P.Fatal_Error from *,
      --#                                        CommandLineData.Content,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IndexFile,
      --#                                        Index_Filename,
      --#                                        LastChar,
      --#                                        LexTokenManager.State,
      --#                                        SPARK_IO.File_Sys;
      is
         EntryNameValid,
         IsInValid,
         ComponentsValid,
         FileLocationValid : Boolean;
         EntryUnit : LexTokenLists.Lists;
         EntryTypeLoc : EntryTypes;
         CurrentPosition : FilePosition;

         -- Get the next single character; unget any whitespace
         procedure Get_Single_Char (File           : in     SPARK_IO.File_Type;
                                    Index_Filename : in     LexTokenManager.Lex_String;
                                    InQuotedString : in     Boolean;
                                    Ch             :    out Character;
                                    OK             :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     LexTokenManager.State;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out Index_Table_P.Fatal_Error;
         --#        in out LastChar;
         --#        in out SPARK_IO.File_Sys;
         --# derives Ch,
         --#         OK                        from File,
         --#                                        LastChar,
         --#                                        SPARK_IO.File_Sys &
         --#         LastChar                  from *,
         --#                                        File,
         --#                                        InQuotedString,
         --#                                        SPARK_IO.File_Sys &
         --#         ErrorHandler.ErrorContext,
         --#         Index_Table_P.Fatal_Error from *,
         --#                                        File,
         --#                                        LastChar,
         --#                                        SPARK_IO.File_Sys &
         --#         SPARK_IO.File_Sys         from *,
         --#                                        CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        File,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State;
         is
            MyChar : Character;
         begin  -- Get_Single_Char
            if IsEndOfBufferedFile (File) then
               Ch := ' ';
               OK := False;
            elsif IsEndOfBufferedLine (File) then
               Ch := ' ';
               OK := True;
            else
               Read_Buffered_Char (File           => File,
                                   Index_Filename => Index_Filename,
                                   Ch             => MyChar);

               -- Allow for quoted strings containing spaces
               if (MyChar = ' ') and not InQuotedString then
                  UngetChar (MyChar);
               end if;

               OK := True;
               Ch := MyChar;
            end if;
         end Get_Single_Char;

         procedure Read_An_Identifier (IndexFile      : in     SPARK_IO.File_Type;
                                       Index_Filename : in     LexTokenManager.Lex_String;
                                       Identifier     : in out EStrings.T;
                                       Valid          :    out Boolean;
                                       Follower       :    out Character)
         --# global in     CommandLineData.Content;
         --#        in     LexTokenManager.State;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out Index_Table_P.Fatal_Error;
         --#        in out LastChar;
         --#        in out SPARK_IO.File_Sys;
         --# derives ErrorHandler.ErrorContext,
         --#         Follower,
         --#         LastChar,
         --#         SPARK_IO.File_Sys,
         --#         Valid                     from CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys &
         --#         Identifier,
         --#         Index_Table_P.Fatal_Error from *,
         --#                                        CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys;
         is
            Ch       : Character;
            ReadOK   : Boolean;
            AppendOK : Boolean;
         begin    --Read_An_Identifier
            Skip_White_Space (File           => IndexFile,
                              Index_Filename => Index_Filename);
            Get_Single_Char (File           => IndexFile,
                             Index_Filename => Index_Filename,
                             InQuotedString => False,
                             Ch             => Ch,
                             OK             => ReadOK);
            if ReadOK and (Ch in 'a' .. 'z' or Ch in 'A' .. 'Z') then
               Identifier := EStrings.Empty_String;
               loop
                  EStrings.Append_Char (E_Str   => Identifier,
                                        Ch      => Ch,
                                        Success => AppendOK);
                  if not AppendOK then
                     SystemErrors.FatalError (SystemErrors.UnitNameInIndexTooLong, "in Read_An_Identifier");
                  end if;
                  Get_Single_Char (File           => IndexFile,
                                   Index_Filename => Index_Filename,
                                   InQuotedString => False,
                                   Ch             => Ch,
                                   OK             => ReadOK);
                  exit when not ReadOK or else
                    (Ch not in 'A' .. 'Z' and
                       Ch not in 'a' .. 'z' and
                       Ch not in '0' .. '9' and
                       Ch /= '_');
               end loop;
               Valid := IsEndOfBufferedFile (IndexFile) or ReadOK;
               Follower := Ch;
            else
               Valid := False;
               Follower := ' ';
            end if;
         end Read_An_Identifier;

         procedure Read_A_Unit_Name (IndexFile      : in     SPARK_IO.File_Type;
                                     Index_Filename : in     LexTokenManager.Lex_String;
                                     UnitName       :    out LexTokenLists.Lists;
                                     Valid          :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out Index_Table_P.Fatal_Error;
         --#        in out LastChar;
         --#        in out LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives Index_Table_P.Fatal_Error,
         --#         LexTokenManager.State     from *,
         --#                                        CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys &
         --#         ErrorHandler.ErrorContext,
         --#         LastChar,
         --#         SPARK_IO.File_Sys,
         --#         UnitName,
         --#         Valid                     from CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys;
         is
            Id              : EStrings.T;
            ValidIdentifier : Boolean;
            NextCh          : Character;
            Token           : LexTokenManager.Lex_String;
            NameList        : LexTokenLists.Lists;
         begin -- Read_A_Unit_Name
            Id := EStrings.Empty_String;
            NameList := LexTokenLists.Null_List;
            loop
               Read_An_Identifier (IndexFile      => IndexFile,
                                   Index_Filename => Index_Filename,
                                   Identifier     => Id,
                                   Valid          => ValidIdentifier,
                                   Follower       => NextCh);
               exit when not ValidIdentifier;
               LexTokenManager.Insert_Examiner_String (Str     => Id,
                                                       Lex_Str => Token);
               LexTokenLists.Append (NameList, Token);
               exit when NextCh /= '.';
            end loop;
            Valid := ValidIdentifier and NextCh = ' ';
            UnitName := NameList;
         end Read_A_Unit_Name;

         procedure Read_An_Entry_Name (IndexFile       : in     SPARK_IO.File_Type;
                                       Index_Filename  : in     LexTokenManager.Lex_String;
                                       EntryUnitExport :    out LexTokenLists.Lists;
                                       EntryType       :    out EntryTypes;
                                       EntryNameValid  :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out Index_Table_P.Fatal_Error;
         --#        in out LastChar;
         --#        in out LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives EntryNameValid,
         --#         EntryType,
         --#         EntryUnitExport,
         --#         ErrorHandler.ErrorContext,
         --#         LastChar,
         --#         LexTokenManager.State,
         --#         SPARK_IO.File_Sys         from CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys &
         --#         Index_Table_P.Fatal_Error from *,
         --#                                        CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys;
         is
            UnitValid           : Boolean;
            AString             : EStrings.T;
            StringValid         : Boolean;
            NextCh              : Character;
            EntryUnit           : LexTokenLists.Lists;
            Position            : FilePosition;
         begin  -- Read_An_Entry_Name
            EntryType := InvalidEntryType;
            AString := EStrings.Empty_String;
            -- Get position of start of entry
            Get_Token_Position (IndexFile      => IndexFile,
                                Index_Filename => Index_Filename,
                                Position       => Position);
            Read_A_Unit_Name (IndexFile      => IndexFile,
                              Index_Filename => Index_Filename,
                              UnitName       => EntryUnit,
                              Valid          => UnitValid);
            if UnitValid then
               if LexTokenLists.Get_Length (List => EntryUnit) = 1 and then
                 LexTokenManager.Lex_String_Case_Insensitive_Compare
                 (Lex_Str1 => LexTokenLists.Get_Element (List => EntryUnit,
                                                         Pos  => 1),
                  Lex_Str2 => LexTokenManager.Super_Index_Token) = LexTokenManager.Str_Eq then
                  EntryType      := SuperIndex;
                  EntryNameValid := True;
               else
                  -- Get the position of the start of the unit type
                  Get_Token_Position (IndexFile      => IndexFile,
                                      Index_Filename => Index_Filename,
                                      Position       => Position);
                  Read_An_Identifier (IndexFile      => IndexFile,
                                      Index_Filename => Index_Filename,
                                      Identifier     => AString,
                                      Valid          => StringValid,
                                      Follower       => NextCh);
                  if StringValid and NextCh = ' ' then
                     if EStrings.Eq1_String (E_Str => AString,
                                             Str   => "auxindex") then
                        EntryType := AuxIndex;
                        EntryNameValid := True;
                     elsif EStrings.Eq1_String (E_Str => AString,
                                                Str   => "main_program") then
                        EntryType := mainp;
                        EntryNameValid := True;
                     elsif EStrings.Eq1_String (E_Str => AString,
                                                Str   => "specification") or else
                       EStrings.Eq1_String (E_Str => AString,
                                            Str   => "spec") then
                        EntryType := pspec;
                        EntryNameValid := True;
                        -- GenOption -- elsif EStrings.Eq1String (AString, "generic_declaration") then
                        -- GenOption --    EntryType := GenericDec;
                        -- GenOption --    EntryNameValid := True;
                     elsif EStrings.Eq1_String (E_Str => AString,
                                                Str   => "body") then
                        EntryType := pbodi;
                        EntryNameValid := True;
                     elsif EStrings.Eq1_String (E_Str => AString,
                                                Str   => "subunit") then
                        EntryType := subunit;
                        EntryNameValid := True;
                     elsif EStrings.Eq1_String (E_Str => AString,
                                                Str   => "components") then
                        EntryType := ComponentList;
                        EntryNameValid := True;
                     else
                        EntryNameValid := False;
                        Index_Table_P.Output_Error (E              => ESUnitEntry,
                                                    Source_File    => Index_Filename,
                                                    Token_Position => Position,
                                                    Token_String   => AString);
                     end if;
                  else
                     EntryNameValid := False;
                     Index_Table_P.Output_Error (E              => ESUnitEntry,
                                                 Source_File    => Index_Filename,
                                                 Token_Position => Position,
                                                 Token_String   => EStrings.Empty_String);
                  end if;
               end if;
            else
               EntryNameValid := False;
               Index_Table_P.Output_Error (E              => EWIllegalUnitName,
                                           Source_File    => Index_Filename,
                                           Token_Position => Position,
                                           Token_String   => EStrings.Empty_String);
            end if;
            EntryUnitExport := EntryUnit;
         end Read_An_Entry_Name;

         procedure Skip_Is_In (IndexFile      : in     SPARK_IO.File_Type;
                               Index_Filename : in     LexTokenManager.Lex_String;
                               IsInValid      :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     LexTokenManager.State;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out Index_Table_P.Fatal_Error;
         --#        in out LastChar;
         --#        in out SPARK_IO.File_Sys;
         --# derives Index_Table_P.Fatal_Error from *,
         --#                                        CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys &
         --#         ErrorHandler.ErrorContext,
         --#         IsInValid,
         --#         LastChar,
         --#         SPARK_IO.File_Sys         from CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys;
         is
            Id      : EStrings.T;
            IdValid : Boolean;
            IsValid : Boolean;
            InValid : Boolean;
            NextCh  : Character;
         begin
            Id := EStrings.Empty_String;
            Read_An_Identifier (IndexFile      => IndexFile,
                                Index_Filename => Index_Filename,
                                Identifier     => Id,
                                Valid          => IdValid,
                                Follower       => NextCh);
            if IdValid and NextCh = ' ' then
               IsValid := EStrings.Eq1_String (E_Str => Id,
                                               Str   => "is");
            else
               IsValid := False;
            end if;

            if IsValid then
               Read_An_Identifier (IndexFile      => IndexFile,
                                   Index_Filename => Index_Filename,
                                   Identifier     => Id,
                                   Valid          => IdValid,
                                   Follower       => NextCh);

               if IdValid and NextCh = ' ' then
                  InValid := EStrings.Eq1_String (E_Str => Id,
                                                  Str   => "in");
               else
                  InValid := False;
               end if;

               if InValid then
                  IsInValid := True;
               else
                  IsInValid := False;
               end if;
            else
               IsInValid := False;
            end if;
         end Skip_Is_In;

         procedure Read_A_File_Location (IndexFile         : in     SPARK_IO.File_Type;
                                         Index_Filename    : in     LexTokenManager.Lex_String;
                                         EntryFileName     :    out EStrings.T;
                                         FileLocationValid :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     LexTokenManager.State;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out Index_Table_P.Fatal_Error;
         --#        in out LastChar;
         --#        in out SPARK_IO.File_Sys;
         --# derives EntryFileName,
         --#         FileLocationValid         from CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys &
         --#         ErrorHandler.ErrorContext,
         --#         Index_Table_P.Fatal_Error,
         --#         LastChar,
         --#         SPARK_IO.File_Sys         from *,
         --#                                        CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys;
         is
            Ch             : Character;
            ReadOK         : Boolean;
            AppendOK       : Boolean;
            EStr           : EStrings.T;
            InQuotedString : Boolean;
         begin
            InQuotedString := False;

            EStr := EStrings.Empty_String;

            Skip_White_Space (File           => IndexFile,
                              Index_Filename => Index_Filename);
            Get_Single_Char (File           => IndexFile,
                             Index_Filename => Index_Filename,
                             InQuotedString => False,
                             Ch             => Ch,
                             OK             => ReadOK);
            loop
               -- Allow for quoted strings containing spaces
               exit when not ReadOK or
                 (Ch = ' ' and not InQuotedString);

               if Ch = Ada.Characters.Latin_1.Quotation then
                  InQuotedString := not InQuotedString;
               else
                  EStrings.Append_Char (E_Str   => EStr,
                                        Ch      => Ch,
                                        Success => AppendOK);
                  if not AppendOK then
                     SystemErrors.FatalError (SystemErrors.FileNameInIndexTooLong, "in Read_A_File_Location");
                  end if;
               end if;

               Get_Single_Char (File           => IndexFile,
                                Index_Filename => Index_Filename,
                                InQuotedString => InQuotedString,
                                Ch             => Ch,
                                OK             => ReadOK);
            end loop;
            Skip_White_Space (File           => IndexFile,
                              Index_Filename => Index_Filename);
            EntryFileName := EStr;
            FileLocationValid := EStrings.Get_Length (E_Str => EStr) > 0;

         end Read_A_File_Location;

         procedure Read_Components (IndexFile      : in     SPARK_IO.File_Type;
                                    Index_Filename : in     LexTokenManager.Lex_String;
                                    TheComponents  :    out ComponentLists;
                                    EntryValid     :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out Index_Table_P.Fatal_Error;
         --#        in out LastChar;
         --#        in out LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives EntryValid,
         --#         ErrorHandler.ErrorContext,
         --#         LastChar,
         --#         SPARK_IO.File_Sys,
         --#         TheComponents             from CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys &
         --#         Index_Table_P.Fatal_Error,
         --#         LexTokenManager.State     from *,
         --#                                        CommandLineData.Content,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        IndexFile,
         --#                                        Index_Filename,
         --#                                        LastChar,
         --#                                        LexTokenManager.State,
         --#                                        SPARK_IO.File_Sys;
         is
            Id        : EStrings.T;
            Token     : LexTokenManager.Lex_String;
            Index     : ComponentIndex;
            Component : LexTokenLists.Lists;
            Valid     : Boolean;
            NextCh    : Character;
            CurrentPosition : FilePosition;
         begin
            TheComponents := ComponentLists'(others => LexTokenLists.Null_List);
            Index := ComponentIndex'First;
            Id    := EStrings.Empty_String;

            -- Locate the position of the "are"
            Get_Token_Position (IndexFile      => IndexFile,
                                Index_Filename => Index_Filename,
                                Position       => CurrentPosition);
            Read_An_Identifier (IndexFile      => IndexFile,
                                Index_Filename => Index_Filename,
                                Identifier     => Id,
                                Valid          => Valid,
                                Follower       => NextCh);

            Valid := Valid and then NextCh = ' ' and then
              EStrings.Eq1_String (E_Str => Id,
                                   Str   => "are");

            if Valid then
               loop --over components
                  Id := EStrings.Empty_String;
                  Component := LexTokenLists.Null_List;
                  loop --over component identifiers

                     -- Locate the position of the next identifier
                     Get_Token_Position (IndexFile      => IndexFile,
                                         Index_Filename => Index_Filename,
                                         Position       => CurrentPosition);

                     Read_An_Identifier (IndexFile      => IndexFile,
                                         Index_Filename => Index_Filename,
                                         Identifier     => Id,
                                         Valid          => Valid,
                                         Follower       => NextCh);
                     exit when not Valid;
                     LexTokenManager.Insert_Examiner_String (Str     => Id,
                                                             Lex_Str => Token);
                     LexTokenLists.Append (Component, Token);
                     exit when NextCh /= '.';
                  end loop;
                  exit when not Valid;
                  TheComponents (Index) := Component;
                  if Index < ComponentIndex'Last then
                     Index := ComponentIndex'Succ (Index);
                  else
                     SystemErrors.FatalError (SystemErrors.IndexComponentListFull, "");
                  end if;
                  exit when NextCh /= ',';
               end loop;
               Valid := Valid and NextCh = ' ';
               if not Valid then
                  Index_Table_P.Output_Error (E              => ESComponents,
                                              Source_File    => Index_Filename,
                                              Token_Position => CurrentPosition,
                                              Token_String   => EStrings.Empty_String);
               end if;
            else
               Index_Table_P.Output_Error (E              => ESAre,
                                           Source_File    => Index_Filename,
                                           Token_Position => CurrentPosition,
                                           Token_String   => EStrings.Empty_String);
            end if;
            TheComponents (Index) := LexTokenLists.Null_List; -- list terminator
            EntryValid := Valid;
         end Read_Components;

      begin  -- Read_Entry
         EntryComponents := ComponentLists'(others => LexTokenLists.Null_List);

         Read_An_Entry_Name (IndexFile       => IndexFile,
                             Index_Filename  => Index_Filename,
                             EntryUnitExport => EntryUnit,
                             EntryType       => EntryTypeLoc,
                             EntryNameValid  => EntryNameValid);
         EntryType := EntryTypeLoc;

         if EntryNameValid then
            if EntryTypeLoc = ComponentList then
               EntryFileName := EStrings.Empty_String;
               Read_Components (IndexFile      => IndexFile,
                                Index_Filename => Index_Filename,
                                TheComponents  => EntryComponents,
                                EntryValid     => ComponentsValid);
               if ComponentsValid then
                  ValidEntry := True;
               else
                  ValidEntry := False;
                  -- Errors are reported by Read_Components
                  SPARK_IO.Skip_Line (IndexFile, 1);
               end if;
            else
               -- Locate the start of "is in"
               Get_Token_Position (IndexFile      => IndexFile,
                                   Index_Filename => Index_Filename,
                                   Position       => CurrentPosition);

               Skip_Is_In (IndexFile      => IndexFile,
                           Index_Filename => Index_Filename,
                           IsInValid      => IsInValid);
               if IsInValid then
                  -- Locate the start of file location
                  Get_Token_Position (IndexFile      => IndexFile,
                                      Index_Filename => Index_Filename,
                                      Position       => CurrentPosition);

                  Read_A_File_Location (IndexFile         => IndexFile,
                                        Index_Filename    => Index_Filename,
                                        EntryFileName     => EntryFileName,
                                        FileLocationValid => FileLocationValid);
                  if FileLocationValid then
                     ValidEntry := True;
                  else
                     ValidEntry := False;
                     EntryFileName := EStrings.Empty_String;
                     Index_Table_P.Output_Error (E              => ESFileLocation,
                                                 Source_File    => Index_Filename,
                                                 Token_Position => CurrentPosition,
                                                 Token_String   => EStrings.Empty_String);
                  end if;
               else
                  ValidEntry := False;
                  EntryFileName := EStrings.Empty_String;
                  Index_Table_P.Output_Error (E              => ESIsIn,
                                              Source_File    => Index_Filename,
                                              Token_Position => CurrentPosition,
                                              Token_String   => EStrings.Empty_String);
                  SPARK_IO.Skip_Line (IndexFile, 1);
               end if;
            end if;
         else
            ValidEntry := False;
            EntryFileName := EStrings.Empty_String;
            -- Error reported in call to ReadAnEntry
            SPARK_IO.Skip_Line (IndexFile, 1);
         end if;
         EntryUnitExport := EntryUnit;
      end Read_Entry;

      procedure HandleMainP (EntryUnit         : in     LexTokenLists.Lists;
                             RequiredUnit      : in     LexTokenLists.Lists;
                             EntryFileName     : in     EStrings.T;
                             PossibleUnitTypes : in     ContextManager.UnitTypeSets;
                             Found             : in out Boolean;
                             ReturnedFileName  :    out EStrings.T;
                             ActualUnitType    :    out ContextManager.UnitTypes)
      --# global in LexTokenManager.State;
      --# derives ActualUnitType   from EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit &
      --#         Found            from *,
      --#                               EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit &
      --#         ReturnedFileName from EntryFileName,
      --#                               EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit;
      is
      begin
         if PossibleUnitTypes (ContextManager.MainProgram) and then
           LexTokenLists.Eq_Unit (First_Item => EntryUnit,
                                  Second     => RequiredUnit)
         then
            Found            := True;
            ReturnedFileName := EntryFileName;
            ActualUnitType   := ContextManager.MainProgram;
         else
            ReturnedFileName := EStrings.Empty_String;
            ActualUnitType   := ContextManager.InvalidUnit;
         end if;
      end HandleMainP;

      procedure HandlePSpec (EntryUnit         : in     LexTokenLists.Lists;
                             RequiredUnit      : in     LexTokenLists.Lists;
                             EntryFileName     : in     EStrings.T;
                             PossibleUnitTypes : in     ContextManager.UnitTypeSets;
                             Found             : in out Boolean;
                             ReturnedFileName  :    out EStrings.T;
                             ActualUnitType    :    out ContextManager.UnitTypes)
      --# global in LexTokenManager.State;
      --# derives ActualUnitType   from EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit &
      --#         Found            from *,
      --#                               EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit &
      --#         ReturnedFileName from EntryFileName,
      --#                               EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit;
      is
      begin
         if PossibleUnitTypes (ContextManager.PackageSpecification) and then
           LexTokenLists.Eq_Unit (First_Item => EntryUnit,
                                  Second     => RequiredUnit)
         then
            Found            := True;
            ReturnedFileName := EntryFileName;
            ActualUnitType   := ContextManager.PackageSpecification;
         elsif PossibleUnitTypes (ContextManager.GenericDeclaration) and then -- GenOption --
           LexTokenLists.Eq_Unit (First_Item => EntryUnit,
                                  Second     => RequiredUnit)                 -- GenOption --
         then                                                                 -- GenOption --
            Found            := True;                                         -- GenOption --
            ReturnedFileName := EntryFileName;                                -- GenOption --
            ActualUnitType   := ContextManager.GenericDeclaration;            -- GenOption --
         else
            ReturnedFileName := EStrings.Empty_String;
            ActualUnitType   := ContextManager.InvalidUnit;
         end if;
      end HandlePSpec;

      -- GenOption -- procedure HandleGenericDec
      -- GenOption -- --# global in     EntryFileName;
      -- GenOption -- --#        in     RequiredUnit;
      -- GenOption -- --#        in     PossibleUnitTypes;
      -- GenOption -- --#        in     EntryUnit;
      -- GenOption -- --#        in out Status;
      -- GenOption -- --#           out ReturnedFileName;
      -- GenOption -- --#           out ActualUnitType;
      -- GenOption -- --# derives ReturnedFileName from EntryFileName,
      -- GenOption -- --#                               RequiredUnit,
      -- GenOption -- --#                               PossibleUnitTypes,
      -- GenOption -- --#                               EntryUnit &
      -- GenOption -- --#         ActualUnitType   from RequiredUnit,
      -- GenOption -- --#                               PossibleUnitTypes,
      -- GenOption -- --#                               EntryUnit &
      -- GenOption -- --#         Status           from *,
      -- GenOption -- --#                               RequiredUnit,
      -- GenOption -- --#                               PossibleUnitTypes,
      -- GenOption -- --#                               EntryUnit;
      -- GenOption -- is
      -- GenOption -- begin
      -- GenOption --    if PossibleUnitTypes (ContextManager.GenericDeclaration) and then
      -- GenOption --      LexTokenLists.EqUnit (EntryUnit, RequiredUnit)
      -- GenOption --    then
      -- GenOption --       Status := FoundSource;
      -- GenOption --       ReturnedFileName := EntryFileName;
      -- GenOption --       ActualUnitType := ContextManager.GenericDeclaration;
      -- GenOption --    else
      -- GenOption --       ReturnedFileName := EStrings.EmptyString;
      -- GenOption --       ActualUnitType := ContextManager.InvalidUnit;
      -- GenOption --    end if;
      -- GenOption -- end HandleGenericDec;

      procedure HandlePBodi (EntryUnit         : in     LexTokenLists.Lists;
                             RequiredUnit      : in     LexTokenLists.Lists;
                             EntryFileName     : in     EStrings.T;
                             PossibleUnitTypes : in     ContextManager.UnitTypeSets;
                             Found             : in out Boolean;
                             ReturnedFileName  :    out EStrings.T;
                             ActualUnitType    :    out ContextManager.UnitTypes)
      --# global in LexTokenManager.State;
      --# derives ActualUnitType   from EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit &
      --#         Found            from *,
      --#                               EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit &
      --#         ReturnedFileName from EntryFileName,
      --#                               EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit;
      is
      begin
         if PossibleUnitTypes (ContextManager.PackageBody) and then
           LexTokenLists.Eq_Unit (First_Item => EntryUnit,
                                  Second     => RequiredUnit)
         then
            Found            := True;
            ReturnedFileName := EntryFileName;
            ActualUnitType   := ContextManager.PackageBody;
         else
            ReturnedFileName := EStrings.Empty_String;
            ActualUnitType   := ContextManager.InvalidUnit;
         end if;
      end HandlePBodi;

      procedure HandleSubUnit (EntryUnit            : in     LexTokenLists.Lists;
                               RequiredUnit         : in     LexTokenLists.Lists;
                               EntryFileName        : in     EStrings.T;
                               PossibleUnitTypes    : in     ContextManager.UnitTypeSets;
                               Found                : in out Boolean;
                               ReturnedFileName     :    out EStrings.T;
                               ActualUnitType       :    out ContextManager.UnitTypes)
      --# global in LexTokenManager.State;
      --# derives ActualUnitType   from EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit &
      --#         Found            from *,
      --#                               EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit &
      --#         ReturnedFileName from EntryFileName,
      --#                               EntryUnit,
      --#                               LexTokenManager.State,
      --#                               PossibleUnitTypes,
      --#                               RequiredUnit;
      is
      begin
         if PossibleUnitTypes (ContextManager.SubUnit) and then
           LexTokenLists.Eq_Unit (First_Item => EntryUnit,
                                  Second     => RequiredUnit)
         then
            Found            := True;
            ReturnedFileName := EntryFileName;
            ActualUnitType   := ContextManager.SubUnit;
         else
            ReturnedFileName := EStrings.Empty_String;
            ActualUnitType   := ContextManager.InvalidUnit;
         end if;
      end HandleSubUnit;

      procedure HandleComponents (EntryUnit       : in     LexTokenLists.Lists;
                                  RequiredUnit    : in     LexTokenLists.Lists;
                                  EntryComponents : in     ComponentLists;
                                  Components      : in out ComponentLists;
                                  Found           : in out Boolean)
      --# global in LexTokenManager.State;
      --# derives Components from *,
      --#                         EntryComponents,
      --#                         EntryUnit,
      --#                         LexTokenManager.State,
      --#                         RequiredUnit &
      --#         Found      from *,
      --#                         EntryUnit,
      --#                         LexTokenManager.State,
      --#                         RequiredUnit;
      is
      begin
         if LexTokenLists.Eq_Unit (First_Item => EntryUnit,
                                   Second     => RequiredUnit) then
            Found      := True;
            Components := EntryComponents;
         end if;
      end HandleComponents;

   begin --LookInFile
      ReturnedFileName := LexTokenManager.Null_String;
      ActualUnitType   := ContextManager.InvalidUnit;
      Components       := ComponentLists'(others => LexTokenLists.Null_List);
      Found            := False;
      LastChar         := Ada.Characters.Latin_1.NUL;
      if CommandLineData.Content.Index then
         --  We never assume that the list of index file has been
         --  initialized.

         --# accept F, 10, FileSpecStatus, "Expect FileSpecStatus Unused";
         FileSystem.FindFullFileName
           (FileSpec     => CommandLineData.Content.IndexFileName,
            FileStatus   => FileSpecStatus,
            FullFileName => IndexFileName);
         --# end accept;
         FileSystem.CheckExtension
           (Fn  => IndexFileName,
            Ext => EStrings.Copy_String (Str => CommandLineData.Default_Index_Extension));
         Index_Table_P.Add_Index_File (Filename => FileSystem.InterpretRelative
                                         (FileName            => IndexFileName,
                                          RelativeToDirectory => IndexFileName));
         Index_Table_P.Get_Next_Index_File (Unit           => RequiredUnit,
                                            Top_Filename   => Top_Filename,
                                            Filename       => CurrentIndexFileName,
                                            File_Type      => EntryType,
                                            Aux_Index_Unit => AuxIndexUnit,
                                            Position       => Source_Position);
         if CommandLineData.Content.Debug.FileNames then
            --  Debug
            SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                 Item => "INDEX MANAGER GET INDEX FILE = ",
                                 Stop => 0);
            Index_Table_P.Debug_Put_EStr (EStr     => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName),
                                          New_Line => True);
         end if;
         InAuxIndex := EntryType = AuxIndex;
         case EntryType is
            when SuperIndex => PossErrorType := EWSuper;
            when others     => PossErrorType := EWIndex;
         end case;
         if LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => CurrentIndexFileName,
            Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq then
            LastFileName := LexTokenManager.Null_String;
            Done         := False;
            loop
               IndexFile := SPARK_IO.Null_File;
               EStrings.Open (File         => IndexFile,
                              Mode_Of_File => SPARK_IO.In_File,
                              Name_Of_File => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName),
                              Form_Of_File => "",
                              Status       => FileStatus);

               --# accept F, 41, "Stable expression expected here";
               if CommandLineData.Content.Debug.FileNames then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => "INDEX MANAGER OPEN INDEX FILE = ",
                                       Stop => 0);
                  Index_Table_P.Debug_Put_EStr (EStr     => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName),
                                                New_Line => True);
               end if;
               --# end accept;
               if FileStatus = SPARK_IO.Ok then
                  LReturnedFileName := EStrings.Empty_String;
                  LActualUnitType   := ContextManager.InvalidUnit;
                  LComponents       := ComponentLists'(others => LexTokenLists.Null_List);
                  LFound            := False;

                  SuperIndexFound   := False;
                  FirstInFile       := True;
                  LastChar          := Ada.Characters.Latin_1.NUL;
                  loop
                     exit when SPARK_IO.End_Of_File (IndexFile);
                     -- Get Start of Entry
                     Get_Token_Position (IndexFile      => IndexFile,
                                         Index_Filename => CurrentIndexFileName,
                                         Position       => Position);
                     Read_Entry (IndexFile       => IndexFile,
                                 Index_Filename  => CurrentIndexFileName,
                                 EntryUnitExport => EntryUnit,
                                 EntryType       => EntryType,
                                 EntryFileName   => EntryFileName,
                                 EntryComponents => EntryComponents,
                                 ValidEntry      => ValidEntry);

                     if ValidEntry then

                        if InAuxIndex and then
                          not LexTokenLists.Prefix_Unit (Poss_Prefix => AuxIndexUnit,
                                                         Prefixed    => EntryUnit)
                        then
                           Index_Table_P.Output_Error (E              => EWAux,
                                                       Source_File    => CurrentIndexFileName,
                                                       Token_Position => Position,
                                                       Token_String   => LexTokenLists.Token_List_To_String (Token_List => AuxIndexUnit));
                        end if;

                        case EntryType is
                           when SuperIndex       =>
                              if InAuxIndex or SuperIndexFound or not FirstInFile then
                                 Index_Table_P.Output_Error (E              => EWUnexpectedSuper,
                                                             Source_File    => CurrentIndexFileName,
                                                             Token_Position => Position,
                                                             Token_String   => EStrings.Empty_String);
                              else
                                 SuperIndexFound := True;
                                 FileSystem.CheckExtension
                                   (Fn  => EntryFileName,
                                    Ext => EStrings.Copy_String (Str => CommandLineData.Default_Index_Extension));
                                 Index_Table_P.Add_Super_Index_File (Filename    => FileSystem.InterpretRelative
                                                                       (FileName            => EntryFileName,
                                                                        RelativeToDirectory => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName)),
                                                                     Position    => Position,
                                                                     Source_File => CurrentIndexFileName);
                                 --# accept F, 41, "Stable expression expected here";
                                 if CommandLineData.Content.Debug.FileNames then
                                    --  Debug
                                    SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                                         Item => "INDEX MANAGER ADD SUPER INDEX FILE = ",
                                                         Stop => 0);
                                    Index_Table_P.Debug_Put_EStr (EStr     => FileSystem.InterpretRelative
                                                                    (FileName            => EntryFileName,
                                                                     RelativeToDirectory => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName)),
                                                                  New_Line => True);
                                 end if;
                                 --# end accept;
                              end if;
                           when AuxIndex         =>
                              FileSystem.CheckExtension
                                (Fn  => EntryFileName,
                                 Ext => EStrings.Copy_String (Str => CommandLineData.Default_Index_Extension));
                              Index_Table_P.Add_Aux_Index_File (Filename    => FileSystem.InterpretRelative
                                                                  (FileName            => EntryFileName,
                                                                   RelativeToDirectory => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName)),
                                                                Unit        => EntryUnit,
                                                                Position    => Position,
                                                                Source_File => CurrentIndexFileName);
                              --# accept F, 41, "Stable expression expected here";
                              if CommandLineData.Content.Debug.FileNames then
                                 --  Debug
                                 SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                                      Item => "INDEX MANAGER ADD AUX INDEX FILE = ",
                                                      Stop => 0);
                                 Index_Table_P.Debug_Put_EStr (EStr     => FileSystem.InterpretRelative
                                                                 (FileName            => EntryFileName,
                                                                  RelativeToDirectory => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName)),
                                                               New_Line => True);
                              end if;
                              --# end accept;
                           when mainp            =>
                              --# accept F, 41, "Stable expression expected here";
                              if not Lookup_Components then
                                 HandleMainP (EntryUnit         => EntryUnit,
                                              RequiredUnit      => RequiredUnit,
                                              EntryFileName     => EntryFileName,
                                              PossibleUnitTypes => PossibleUnitTypes,
                                              Found             => LFound,
                                              ReturnedFileName  => LReturnedFileName,
                                              ActualUnitType    => LActualUnitType);
                              end if;
                              --# end accept;
                              FileSystem.CheckExtension
                                (Fn  => EntryFileName,
                                 Ext => CommandLineData.Content.SourceExtension);
                              Cache.Add_Unit (Unit            => EntryUnit,
                                              Unit_Types      => ContextManager.MainProgram,
                                              Source_Filename => FileSystem.InterpretRelative
                                                (FileName            => EntryFileName,
                                                 RelativeToDirectory => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName)),
                                              Index_Filename  => CurrentIndexFileName,
                                              Index_Position  => Position);
                           when pspec            =>
                              --# accept F, 41, "Stable expression expected here";
                              if not Lookup_Components then
                                 HandlePSpec (EntryUnit         => EntryUnit,
                                              RequiredUnit      => RequiredUnit,
                                              EntryFileName     => EntryFileName,
                                              PossibleUnitTypes => PossibleUnitTypes,
                                              Found             => LFound,
                                              ReturnedFileName  => LReturnedFileName,
                                              ActualUnitType    => LActualUnitType);
                              end if;
                              --# end accept;
                              FileSystem.CheckExtension
                                (Fn  => EntryFileName,
                                 Ext => CommandLineData.Content.SourceExtension);
                              Cache.Add_Unit (Unit            => EntryUnit,
                                              Unit_Types      => ContextManager.PackageSpecification,
                                              Source_Filename => FileSystem.InterpretRelative
                                                (FileName            => EntryFileName,
                                                 RelativeToDirectory => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName)),
                                              Index_Filename  => CurrentIndexFileName,
                                              Index_Position  => Position);
                              -- GenOption -- when GenericDec => HandleGenericDec;
                           when pbodi            =>
                              --# accept F, 41, "Stable expression expected here";
                              if not Lookup_Components then
                                 HandlePBodi (EntryUnit         => EntryUnit,
                                              RequiredUnit      => RequiredUnit,
                                              EntryFileName     => EntryFileName,
                                              PossibleUnitTypes => PossibleUnitTypes,
                                              Found             => LFound,
                                              ReturnedFileName  => LReturnedFileName,
                                              ActualUnitType    => LActualUnitType);
                              end if;
                              --# end accept;
                              FileSystem.CheckExtension
                                (Fn  => EntryFileName,
                                 Ext => CommandLineData.Content.SourceExtension);
                              Cache.Add_Unit (Unit            => EntryUnit,
                                              Unit_Types      => ContextManager.PackageBody,
                                              Source_Filename => FileSystem.InterpretRelative
                                                (FileName            => EntryFileName,
                                                 RelativeToDirectory => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName)),
                                              Index_Filename  => CurrentIndexFileName,
                                              Index_Position  => Position);
                           when subunit          =>
                              --# accept F, 41, "Stable expression expected here";
                              if not Lookup_Components then
                                 HandleSubUnit (EntryUnit         => EntryUnit,
                                                RequiredUnit      => RequiredUnit,
                                                EntryFileName     => EntryFileName,
                                                PossibleUnitTypes => PossibleUnitTypes,
                                                Found             => LFound,
                                                ReturnedFileName  => LReturnedFileName,
                                                ActualUnitType    => LActualUnitType);
                              end if;
                              --# end accept;
                              FileSystem.CheckExtension
                                (Fn  => EntryFileName,
                                 Ext => CommandLineData.Content.SourceExtension);
                              Cache.Add_Unit (Unit            => EntryUnit,
                                              Unit_Types      => ContextManager.SubUnit,
                                              Source_Filename => FileSystem.InterpretRelative
                                                (FileName            => EntryFileName,
                                                 RelativeToDirectory => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName)),
                                              Index_Filename  => CurrentIndexFileName,
                                              Index_Position  => Position);
                           when ComponentList    =>
                              --# accept F, 41, "Stable expression expected here";
                              if Lookup_Components then
                                 HandleComponents (EntryUnit       => EntryUnit,
                                                   RequiredUnit    => RequiredUnit,
                                                   EntryComponents => EntryComponents,
                                                   Components      => LComponents,
                                                   Found           => LFound);
                              end if;
                              --# end accept;
                              Cache.Add_Components (Unit           => EntryUnit,
                                                    Components     => EntryComponents,
                                                    Index_Filename => CurrentIndexFileName,
                                                    Index_Position => Position);
                           when InvalidEntryType => null;
                        end case;
                     end if;
                     if LFound then
                        ActualUnitType := LActualUnitType;
                        Components     := LComponents;
                        Found          := True;
                        --# accept F, 41, "Stable expression expected here";
                        if not Lookup_Components then
                           FileSystem.CheckExtension
                             (Fn  => LReturnedFileName,
                              Ext => CommandLineData.Content.SourceExtension);
                           LexTokenManager.Insert_Examiner_String
                             (Str     => FileSystem.InterpretRelative
                                (FileName            => LReturnedFileName,
                                 RelativeToDirectory => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName)),
                              Lex_Str => ReturnedFileName);
                        end if;
                        --# end accept;
                        --# accept F, 41, "Stable expression expected here";
                        if CommandLineData.Content.Debug.FileNames then
                           --  Debug
                           if Lookup_Components then
                              SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                                   Item => "INDEXMANAGER FOUND ",
                                                   Stop => 0);
                              for I in ComponentIndex loop
                                 if Components (I) /= LexTokenLists.Null_List then
                                    LexTokenLists.Print_List (File => SPARK_IO.Standard_Output,
                                                              List => Components (I));
                                 end if;
                              end loop;
                           else
                              SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                                   Item => "INDEXMANAGER FOUND IN ",
                                                   Stop => 0);
                              Index_Table_P.Debug_Put_EStr (EStr     => LexTokenManager.Lex_String_To_String (Lex_Str => ReturnedFileName),
                                                            New_Line => False);
                              SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                                   Item => " FOR TYPE ",
                                                   Stop => 0);
                              Cache.ContextManager_UnitTypes_Image (Unit_Type => ActualUnitType);
                           end if;
                           SPARK_IO.New_Line (File    => SPARK_IO.Standard_Output,
                                              Spacing => 1);
                        end if;
                        --# end accept;
                     end if;
                     LFound      := False;
                     FirstInFile := False;
                  end loop;

                  LastFileName := CurrentIndexFileName;
                  Index_Table_P.Index_File_Done (Filename => CurrentIndexFileName);
                  --# accept F, 10, IndexFile, "Expect ineffective assignment" &
                  --#        F, 10, FileStatus, "Expect ineffective assignment";
                  SPARK_IO.Close (IndexFile, FileStatus);
                  --# end accept;
                  --# accept F, 41, "Stable expression expected here";
                  if CommandLineData.Content.Debug.FileNames then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "INDEX MANAGER CLOSE INDEX FILE = ",
                                          Stop => 0);
                     Index_Table_P.Debug_Put_EStr (EStr     => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName),
                                                   New_Line => True);
                  end if;
                  --# end accept;
                  Index_Table_P.Stop_SPARK;
                  if Found then
                     Done  := True;
                  else
                     Index_Table_P.Get_Next_Index_File (Unit           => RequiredUnit,
                                                        Top_Filename   => Top_Filename,
                                                        Filename       => CurrentIndexFileName,
                                                        File_Type      => EntryType,
                                                        Aux_Index_Unit => AuxIndexUnit,
                                                        Position       => Source_Position);
                     --# accept F, 41, "Stable expression expected here";
                     if CommandLineData.Content.Debug.FileNames then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "INDEX MANAGER GET INDEX FILE = ",
                                             Stop => 0);
                        Index_Table_P.Debug_Put_EStr (EStr     => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName),
                                                      New_Line => True);
                     end if;
                     --# end accept;
                     InAuxIndex := EntryType = AuxIndex;
                     case EntryType is
                        when SuperIndex => PossErrorType := EWSuper;
                        when others     => PossErrorType := EWIndex;
                     end case;
                     if LexTokenManager.Lex_String_Case_Insensitive_Compare
                       (Lex_Str1 => CurrentIndexFileName,
                        Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
                        Found := False;
                        Done  := True;
                     end if;
                  end if;
               else
                  Found := False;
                  Done  := True;
                  Index_Table_P.Index_File_Done (Filename => CurrentIndexFileName);
                  Index_Table_P.Output_Error (E              => PossErrorType,
                                              Source_File    => LastFileName,
                                              Token_Position => Source_Position,
                                              Token_String   => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentIndexFileName));
               end if;
               exit when Done;
            end loop;
         end if;
      end if;
      --# accept F, 33, FileSpecStatus, "Expect FileSpecStatus unused";
   end LookInFile;

   procedure LookUp (RequiredUnit      : in     LexTokenLists.Lists;
                     PossibleUnitTypes : in     ContextManager.UnitTypeSets;
                     SourceFileName    :    out LexTokenManager.Lex_String;
                     ActualUnitType    :    out ContextManager.UnitTypes;
                     Found             :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out Cache.The_Unit_Hash;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Index_Table_P.Fatal_Error;
   --#        in out Index_Table_P.Index_Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#           out LastChar;
   --# derives ActualUnitType,
   --#         Cache.The_Unit_Hash,
   --#         Found,
   --#         Index_Table_P.Index_Table,
   --#         LastChar,
   --#         LexTokenManager.State,
   --#         SourceFileName,
   --#         SPARK_IO.File_Sys         from Cache.The_Unit_Hash,
   --#                                        CommandLineData.Content,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Index_Table_P.Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        PossibleUnitTypes,
   --#                                        RequiredUnit,
   --#                                        SPARK_IO.File_Sys &
   --#         ErrorHandler.ErrorContext,
   --#         Index_Table_P.Fatal_Error from *,
   --#                                        Cache.The_Unit_Hash,
   --#                                        CommandLineData.Content,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Index_Table_P.Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        PossibleUnitTypes,
   --#                                        RequiredUnit,
   --#                                        SPARK_IO.File_Sys;
   is
      Components                : ComponentLists;

      SourceFileName_From_Cache : LexTokenManager.Lex_String;
      SourceFileName_From_File  : LexTokenManager.Lex_String;
      ActualUnitType_From_Cache : ContextManager.UnitTypes;
      ActualUnitType_From_File  : ContextManager.UnitTypes;
      Found_From_Cache          : Boolean;
      Found_From_File           : Boolean;
      Need_To_Look_In_File      : Boolean;

      Index_Filename_From_Cache : LexTokenManager.Lex_String;
      Index_Filename            : LexTokenManager.Lex_String;
      File_Type                 : EntryTypes;
      Dummy_Aux_Index_Unit      : LexTokenLists.Lists;
      Dummy_Source_Position     : FilePosition;

      procedure Trace (S : in LexTokenManager.Lex_String;
                       M : in String)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                LexTokenManager.State,
      --#                                M,
      --#                                S;
      is
      begin
         if CommandLineData.Content.Debug.FileNames then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, M, 0);

            if CommandLineData.Content.PlainOutput then
               --# accept F, 22, "Stable expression here OK";
               if FileSystem.UseWindowsCommandLine then
               --# end accept;
                  EStrings.Put_Line
                    (File  => SPARK_IO.Standard_Output,
                     E_Str => EStrings.Lower_Case
                       (E_Str => FileSystem.JustFile
                          (LexTokenManager.Lex_String_To_String (Lex_Str => S), True)));
               else
                  EStrings.Put_Line
                    (File  => SPARK_IO.Standard_Output,
                     E_Str => FileSystem.JustFile
                       (LexTokenManager.Lex_String_To_String (Lex_Str => S), True));
               end if;
            else
               EStrings.Put_Line
                 (File  => SPARK_IO.Standard_Output,
                  E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => S));
            end if;
         end if;
      end Trace;

   begin --LookUp
      if CommandLineData.Content.Debug.FileNames then
         --  Debug
         SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                              Item => "INDEX MANAGER LOOKING FOR ",
                              Stop => 0);
         LexTokenLists.Print_List (File => SPARK_IO.Standard_Output,
                                   List => RequiredUnit);
         SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                              Item => " IN",
                              Stop => 0);
      end if;
      --  Try to find the required unit in the cache.
      SourceFileName_From_Cache := LexTokenManager.Null_String;
      ActualUnitType_From_Cache := ContextManager.InvalidUnit;
      Index_Filename_From_Cache := LexTokenManager.Null_String;
      Found_From_Cache          := False;
      Found_From_File           := False;
      for I in ContextManager.UnitTypes loop
         if PossibleUnitTypes (I) then
            ActualUnitType_From_Cache := I;
            Cache.Get_Unit (Required_Unit   => RequiredUnit,
                            Unit_Types      => ActualUnitType_From_Cache,
                            Source_Filename => SourceFileName_From_Cache,
                            Index_Filename  => Index_Filename_From_Cache,
                            Found           => Found_From_Cache);
            --# accept F, 41, "Stable expression expected here";
            if CommandLineData.Content.Debug.FileNames then
               --  Debug
               SPARK_IO.Put_Char (File => SPARK_IO.Standard_Output,
                                  Item => ' ');
               Cache.ContextManager_UnitTypes_Image (Unit_Type => ActualUnitType_From_Cache);
            end if;
            --# end accept;
         end if;
         exit when Found_From_Cache;
      end loop;
      if CommandLineData.Content.Debug.FileNames then
         --  Debug
         SPARK_IO.New_Line (File    => SPARK_IO.Standard_Output,
                            Spacing => 1);
      end if;
      if Found_From_Cache then
         --  The required unit has been found in the cache => check if
         --  a more local definition of the required unit may exist in
         --  a not yet parsed auxiliary index file.
         --
         --  The Filename and Aux_Index_Unit values returned from the
         --  call to Index_Table_P.Get_Next_Index_File, rather the
         --  Index_Filename_From_Cache is retained as the most global
         --  index file for LookInFile.

         --# accept F, 10, Dummy_Aux_Index_Unit, "Ineffective assignment here OK" &
         --#        F, 10, Dummy_Source_Position, "Ineffective assignment here OK";
         Index_Table_P.Get_Next_Index_File (Unit           => RequiredUnit,
                                            Top_Filename   => Index_Filename_From_Cache,
                                            Filename       => Index_Filename,
                                            File_Type      => File_Type,
                                            Aux_Index_Unit => Dummy_Aux_Index_Unit,
                                            Position       => Dummy_Source_Position);
         --# end accept;
         Need_To_Look_In_File := File_Type = AuxIndex;
         if CommandLineData.Content.Debug.FileNames then
            --  Debug

            --  Index_Table_P.Get_Next_Index may return an yet
            --  unparsed auxiliary index or super index.
            SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                 Item => "INDEX FILENAME NOT YET PARSED : ",
                                 Stop => 0);
            Index_Table_P.Debug_Put_EStr (EStr     => LexTokenManager.Lex_String_To_String (Lex_Str => Index_Filename),
                                          New_Line => True);
         end if;
      else
         --  The required unit is not in the cache => need to lookup
         --  in the index files.
         Index_Filename_From_Cache := LexTokenManager.Null_String;
         Need_To_Look_In_File      := True;
      end if;
      if Need_To_Look_In_File then
         --  The required unit has not been found in the cache => need
         --  to lookup in the index files
         --
         --  or
         --
         --  a more local definition of the required unit may exist in
         --  a not yet parsed index file => need to lookup in the more
         --  local index files only.

         --# accept F, 10, Components, "Ineffective assignment here OK";
         LookInFile (Lookup_Components => False,
                     RequiredUnit      => RequiredUnit,
                     PossibleUnitTypes => PossibleUnitTypes,
                     Top_Filename      => Index_Filename_From_Cache,
                     ReturnedFileName  => SourceFileName_From_File,
                     ActualUnitType    => ActualUnitType_From_File,
                     Components        => Components,
                     Found             => Found_From_File);
         --# end accept;
         if Found_From_File then
            --  The required unit has been found in the index files or
            --  a more local definition of the required unit has been
            --  found in a previously unparsed auxiliary index file =>
            --  the answer is the value coming from the index file.
            SourceFileName := SourceFileName_From_File;
            ActualUnitType := ActualUnitType_From_File;
         elsif Found_From_Cache then
            --  A more local definition of the required unit has not
            --  been found in the index files => the answer is the
            --  value stored in the cache.
            SourceFileName := SourceFileName_From_Cache;
            ActualUnitType := ActualUnitType_From_Cache;
         else
            --  The required unit has not been found either in the
            --  cache or in the index files => the required unit can
            --  not be located.
            SourceFileName := LexTokenManager.Null_String;
            ActualUnitType := ContextManager.InvalidUnit;
         end if;
      else
         --  The required unit has been found in the cache and no more
         --  local definition of the required unit can exist => the
         --  answer is the value stored in the cache.
         SourceFileName := SourceFileName_From_Cache;
         ActualUnitType := ActualUnitType_From_Cache;
      end if;
      Found := Found_From_Cache or Found_From_File;
      if Found then
         Trace (SourceFileName,
                "IndexManager.Lookup SourceFileName is:");
      end if;
      --# accept F, 33, Components, "Expect Components unused" &
      --#        F, 33, Dummy_Aux_Index_Unit, "Expect Dummy_Aux_Index_Unit unused" &
      --#        F, 33, Dummy_Source_Position, "Expect Dummy_Source_Position unused" &
      --#        F, 602, LastChar, LastChar, "Expect array df error";
   end LookUp;

   procedure LookUpComponents (RequiredUnit : in     LexTokenLists.Lists;
                               Components   :    out ComponentLists)
   --# global in     CommandLineData.Content;
   --#        in out Cache.The_Unit_Hash;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Index_Table_P.Fatal_Error;
   --#        in out Index_Table_P.Index_Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#           out LastChar;
   --# derives Cache.The_Unit_Hash,
   --#         Components,
   --#         Index_Table_P.Index_Table,
   --#         LastChar,
   --#         LexTokenManager.State,
   --#         SPARK_IO.File_Sys         from Cache.The_Unit_Hash,
   --#                                        CommandLineData.Content,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Index_Table_P.Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        RequiredUnit,
   --#                                        SPARK_IO.File_Sys &
   --#         ErrorHandler.ErrorContext,
   --#         Index_Table_P.Fatal_Error from *,
   --#                                        Cache.The_Unit_Hash,
   --#                                        CommandLineData.Content,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Index_Table_P.Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        RequiredUnit,
   --#                                        SPARK_IO.File_Sys;
   is
      PossibleUnitTypes         : constant ContextManager.UnitTypeSets :=
        ContextManager.UnitTypeSets'(ContextManager.InvalidUnit => True,
                                     others                     => False);
      SourceFileName            : LexTokenManager.Lex_String;
      ActualUnitType            : ContextManager.UnitTypes;

      Components_From_Cache     : ComponentLists;
      Components_From_File      : ComponentLists;
      Found_From_Cache          : Boolean;
      Found_From_File           : Boolean;
      Need_To_Look_In_File      : Boolean;

      Index_Filename_From_Cache : LexTokenManager.Lex_String;
      Index_Filename            : LexTokenManager.Lex_String;
      File_Type                 : EntryTypes;
      Dummy_Aux_Index_Unit      : LexTokenLists.Lists;
      Dummy_Source_Position     : FilePosition;
   begin --LookUpComponents
      if CommandLineData.Content.Debug.FileNames then
         --  Debug
         SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                              Item => "INDEX MANAGER LOOKING UP COMPONENTS FOR ",
                              Stop => 0);
         LexTokenLists.Print_List (File => SPARK_IO.Standard_Output,
                                   List => RequiredUnit);
         SPARK_IO.New_Line (File    => SPARK_IO.Standard_Output,
                            Spacing => 1);
      end if;
      --  Try to find the required unit in the cache.
      Cache.Get_Components (Required_Unit  => RequiredUnit,
                            Components     => Components_From_Cache,
                            Index_Filename => Index_Filename_From_Cache,
                            Found          => Found_From_Cache);
      if Found_From_Cache then
         --  The required unit has been found in the cache => check if
         --  a more local definition of the required unit may exist in
         --  a not yet parsed auxiliary index file.

         --# accept F, 10, Dummy_Aux_Index_Unit, "Ineffective assignment here OK" &
         --#        F, 10, Dummy_Source_Position, "Ineffective assignment here OK";
         Index_Table_P.Get_Next_Index_File (Unit           => RequiredUnit,
                                            Top_Filename   => Index_Filename_From_Cache,
                                            Filename       => Index_Filename,
                                            File_Type      => File_Type,
                                            Aux_Index_Unit => Dummy_Aux_Index_Unit,
                                            Position       => Dummy_Source_Position);
         --# end accept;
         Need_To_Look_In_File := File_Type = AuxIndex;
         if CommandLineData.Content.Debug.FileNames then
            --  Debug

            --  Index_Table_P.Get_Next_Index may return an yet
            --  unparsed auxiliary index or super index.
            SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                 Item => "INDEX FILENAME NOT YET PARSED : ",
                                 Stop => 0);
            Index_Table_P.Debug_Put_EStr (EStr     => LexTokenManager.Lex_String_To_String (Lex_Str => Index_Filename),
                                          New_Line => True);
         end if;
      else
         --  The required unit is not in the cache => need to lookup
         --  in the index files.
         Index_Filename_From_Cache := LexTokenManager.Null_String;
         Need_To_Look_In_File      := True;
      end if;
      if Need_To_Look_In_File then
         --  The required unit has not been found in the cache => need
         --  to lookup in the index files
         --
         --  or
         --
         --  a more local definition of the required unit may exist in
         --  a not yet parsed index file => need to lookup in the more
         --  local index files only.

         --# accept F, 10, SourceFileName, "Ineffective assignment here OK" &
         --#        F, 10, ActualUnitType, "Ineffective assignment here OK";
         LookInFile (Lookup_Components => True,
                     RequiredUnit      => RequiredUnit,
                     PossibleUnitTypes => PossibleUnitTypes,
                     Top_Filename      => Index_Filename_From_Cache,
                     ReturnedFileName  => SourceFileName,
                     ActualUnitType    => ActualUnitType,
                     Components        => Components_From_File,
                     Found             => Found_From_File);
         --# end accept;
         if Found_From_File then
            --  The required unit has been found in the index files or
            --  a more local definition of the required unit has been
            --  found in a previously unparsed auxiliary index file =>
            --  the answer is the value coming from the index file.
            Components := Components_From_File;
         elsif Found_From_Cache then
            --  A more local definition of the required unit has not
            --  been found in the index files => the answer is the
            --  value stored in the cache.
            Components := Components_From_Cache;
         else
            --  The required unit has not been found either in the
            --  cache or in the index files => the required unit can
            --  not be located.
            Components := ComponentLists'(others => LexTokenLists.Null_List);
         end if;
      else
         --  The required unit has been found in the cache and no more
         --  local definition of the required unit can exist => the
         --  answer is the value stored in the cache.
         Components := Components_From_Cache;
      end if;
      --# accept F, 33, SourceFileName, "Expect ActualUnitType unused" &
      --#        F, 33, ActualUnitType, "Expect ActualUnitType unused" &
      --#        F, 33, Dummy_Aux_Index_Unit, "Expect Dummy_Aux_Index_Unit unused" &
      --#        F, 33, Dummy_Source_Position, "Expect Dummy_Source_Position unused" &
      --#        F, 602, LastChar, LastChar, "Expect array df error";
   end LookUpComponents;

   procedure Add_Unit (Unit            : in LexTokenLists.Lists;
                       Unit_Types      : in ContextManager.UnitTypes;
                       Source_Filename : in EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in     Index_Table_P.Index_Table;
   --#        in out Cache.The_Unit_Hash;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Index_Table_P.Fatal_Error;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Cache.The_Unit_Hash,
   --#         ErrorHandler.ErrorContext,
   --#         Index_Table_P.Fatal_Error from *,
   --#                                        Cache.The_Unit_Hash,
   --#                                        Index_Table_P.Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        Source_Filename,
   --#                                        Unit,
   --#                                        Unit_Types &
   --#         LexTokenManager.State     from *,
   --#                                        Source_Filename &
   --#         SPARK_IO.File_Sys         from *,
   --#                                        Cache.The_Unit_Hash,
   --#                                        CommandLineData.Content,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Index_Table_P.Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        Source_Filename,
   --#                                        Unit,
   --#                                        Unit_Types;
   is
   begin
      Cache.Add_Unit (Unit            => Unit,
                      Unit_Types      => Unit_Types,
                      Source_Filename => Source_Filename,
                      Index_Filename  => LexTokenManager.Null_String,
                      Index_Position  => FilePosition'(Line => 1,
                                                       Col  => 1));
   end Add_Unit;

   procedure ListIndexFile (ReportFile : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     Index_Table_P.Index_Table;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Index_Table_P.Index_Table,
   --#                                LexTokenManager.State,
   --#                                ReportFile,
   --#                                XMLReport.State &
   --#         XMLReport.State   from *,
   --#                                CommandLineData.Content,
   --#                                Index_Table_P.Index_Table;
   is
   begin
      Index_Table_P.List_Index_File (Report_File => ReportFile);
   end ListIndexFile;

begin
   LastChar := Character'First;
end IndexManager;
