-- $Id: indexmanager-index_table_p.ads 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with EStrings;
with LexTokenLists;

--# inherit ErrorHandler,
--#         EStrings,
--#         ExaminerConstants,
--#         CommandLineData,
--#         FileSystem,
--#         IndexManager,
--#         LexTokenLists,
--#         LexTokenManager,
--#         SPARK_IO,
--#         ScreenEcho,
--#         SystemErrors,
--#         XMLReport;

private package IndexManager.Index_Table_P
--# own Index_Table : Index_Tables;
--#     Fatal_Error : Boolean;
--# initializes Index_Table,
--#             Fatal_Error;
is

   procedure Stop_SPARK;
   --# global in Fatal_Error;
   --# derives null from Fatal_Error;

   -- Write a string to standard output, followed by a newline.
   -- If plain output specified then force to lower case to avoid
   -- platform-specific diffs (eg Windows drive letters).
   procedure Debug_Put_EStr (EStr     : in EStrings.T;
                             New_Line : in Boolean);
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Estr,
   --#                                New_Line;

   procedure Output_Error (E              : in IndexManager.LibraryManagerErrors;
                           Source_File    : in LexTokenManager.Lex_String;
                           Token_Position : in IndexManager.FilePosition;
                           Token_String   : in EStrings.T);
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Fatal_Error;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys         from *,
   --#                                        CommandLineData.Content,
   --#                                        E,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        Source_File,
   --#                                        Token_Position,
   --#                                        Token_String &
   --#         ErrorHandler.ErrorContext,
   --#         Fatal_Error               from *,
   --#                                        E;

   --  Add an index file that is not an auxiliary index file or a
   --  super index file in the list of index files.
   procedure Add_Index_File (Filename : in EStrings.T);
   --# global in out Index_Table;
   --#        in out LexTokenManager.State;
   --# derives Index_Table,
   --#         LexTokenManager.State from *,
   --#                                    Filename,
   --#                                    LexTokenManager.State;

   --  Add a super index file in the list of index files.
   procedure Add_Super_Index_File (Filename    : in EStrings.T;
                                   Position    : in IndexManager.FilePosition;
                                   Source_File : in LexTokenManager.Lex_String);
   --# global in     CommandLineData.Content;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Fatal_Error;
   --#        in out Index_Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys         from *,
   --#                                        CommandLineData.Content,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Filename,
   --#                                        Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        Position,
   --#                                        Source_File &
   --#         ErrorHandler.ErrorContext,
   --#         Fatal_Error               from *,
   --#                                        Filename,
   --#                                        Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        Source_File &
   --#         Index_Table               from *,
   --#                                        Filename,
   --#                                        LexTokenManager.State,
   --#                                        Position,
   --#                                        Source_File &
   --#         LexTokenManager.State     from *,
   --#                                        Filename;

   --  Add an auxiliary index file in the list of index files.
   procedure Add_Aux_Index_File (Filename    : in EStrings.T;
                                 Unit        : in LexTokenLists.Lists;
                                 Position    : in IndexManager.FilePosition;
                                 Source_File : in LexTokenManager.Lex_String);
   --# global in     CommandLineData.Content;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Fatal_Error;
   --#        in out Index_Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys         from *,
   --#                                        CommandLineData.Content,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Filename,
   --#                                        Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        Position,
   --#                                        Source_File &
   --#         ErrorHandler.ErrorContext,
   --#         Fatal_Error               from *,
   --#                                        Filename,
   --#                                        Index_Table,
   --#                                        LexTokenManager.State,
   --#                                        Source_File &
   --#         Index_Table               from *,
   --#                                        Filename,
   --#                                        LexTokenManager.State,
   --#                                        Position,
   --#                                        Source_File,
   --#                                        Unit &
   --#         LexTokenManager.State     from *,
   --#                                        Filename;

   --  Mark an index file in the list of index files as done.
   procedure Index_File_Done (Filename : in LexTokenManager.Lex_String);
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Index_Table;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Filename,
   --#                                Index_Table,
   --#                                LexTokenManager.State &
   --#         Index_Table       from *,
   --#                                Filename,
   --#                                LexTokenManager.State;

   --  Check if the parent index file is actually an ancestor of the
   --  index file. A is an ancestor of B if A is more local than B. We
   --  assume that A is an ancestor of A. The empty index filename is
   --  the ancestor of all the index files.
   function Is_File_Ancestor (Parent_Filename : in LexTokenManager.Lex_String;
                              Filename        : in LexTokenManager.Lex_String) return Boolean;
   --# global in Index_Table;
   --#        in LexTokenManager.State;

   --  Get the more relevant index file if one exists:
   --  * an auxiliary file which :
   --    + has the prefix of the unit
   --    + has not been already parsed
   --    + is more local than the top index file
   --  * if no more local auxiliary index files are found, the index
   --    file is a normal index file or a super index file which
   --    + has not been already parsed
   --    + is more local than the top index file
   --  * if no index file or super index file are found, the returned
   --    filename is empty
   --
   --  An empty top index filename represent the more global index
   --  file.
   --
   --  The Aux_Index_Unit is only defined for auxiliary index files.
   procedure Get_Next_Index_File (Unit           : in     LexTokenLists.Lists;
                                  Top_Filename   : in     LexTokenManager.Lex_String;
                                  Filename       :    out LexTokenManager.Lex_String;
                                  File_Type      :    out IndexManager.EntryTypes;
                                  Aux_Index_Unit :    out LexTokenLists.Lists;
                                  Position       :    out IndexManager.FilePosition);
   --# global in     CommandLineData.Content;
   --#        in     Index_Table;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Index_Table,
   --#                                LexTokenManager.State,
   --#                                Top_Filename,
   --#                                Unit &
   --#         Aux_Index_Unit,
   --#         Filename,
   --#         File_Type,
   --#         Position          from Index_Table,
   --#                                LexTokenManager.State,
   --#                                Top_Filename,
   --#                                Unit;

   --  Check if the parent index file is actually an auxiliary
   --  ancestor of the index file. A is an auxiliary ancestor of B if
   --  A is in the same subtree of auxiliary index files as B and A is
   --  an ancestor of B in this subtree of auxiliary index files. We
   --  assume that A is an auxiliary ancestor of A. The empty index
   --  filename can not be in a auxiliary subtree.
   function Is_Aux_File_Ancestor (Parent_Index_Filename : in LexTokenManager.Lex_String;
                                  Index_Filename        : in LexTokenManager.Lex_String) return Boolean;
   --# global in Index_Table;
   --#        in LexTokenManager.State;

   procedure List_Index_File (Report_File : in SPARK_IO.File_Type);
   --# global in     CommandLineData.Content;
   --#        in     Index_Table;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Index_Table,
   --#                                LexTokenManager.State,
   --#                                Report_File,
   --#                                XMLReport.State &
   --#         XMLReport.State   from *,
   --#                                CommandLineData.Content,
   --#                                Index_Table;

end IndexManager.Index_Table_P;
