-- $Id: heap.adb 12447 2009-02-13 15:47:31Z Trevor Jennings $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Statistics,
     SystemErrors;

package body Heap
is

   procedure Initialize (TheHeap : out HeapRecord)
   is
   begin
      TheHeap.HighMark     := NullAtom;
      TheHeap.NextFreeAtom := NullAtom;

      -- Ensure de-referencing of null atom's attributes is safe
      --# accept F, 23,           TheHeap.ListOfAtoms, "Partial initialization OK" &
      --#        F, 602, TheHeap, TheHeap.ListOfAtoms, "Partial initialization OK";
      TheHeap.ListOfAtoms (0) := AtomDescriptor'(PointerA => NullAtom,
                                                 PointerB => NullAtom,
                                                 ValueA   => 0,
                                                 ValueB   => 0);

   end Initialize;

   --------------------------------------------------------------------

   procedure CreateAtom (TheHeap : in out HeapRecord;
                         NewAtom :    out Atom)
   is
      A : Atom;
   begin
      if TheHeap.NextFreeAtom /= NullAtom then
         --there are atoms in the returned free list
         A := TheHeap.NextFreeAtom;
         TheHeap.NextFreeAtom := TheHeap.ListOfAtoms (TheHeap.NextFreeAtom).PointerA;
         TheHeap.ListOfAtoms (A).PointerA := NullAtom;
         TheHeap.ListOfAtoms (A).PointerB := NullAtom;
         NewAtom := A;
      elsif TheHeap.HighMark < Atom (ExaminerConstants.HeapListLength) then
         --returned free list empty but haven't used up array yet
         TheHeap.HighMark := TheHeap.HighMark + 1;
         A := TheHeap.HighMark;
         TheHeap.ListOfAtoms (A).PointerA := NullAtom;
         TheHeap.ListOfAtoms (A).PointerB := NullAtom;
         NewAtom := A;
      else
         -- Array and returned atoms in free list both used up
         -- and set usage to 100% before exiting
         Statistics.SetTableUsage (Statistics.RelationTable,
                                   ExaminerConstants.HeapListLength);
         SystemErrors.FatalError (SystemErrors.EmptyHeap, "in Heap.CreateAtom");
         NewAtom := NullAtom;
      end if;
   end CreateAtom;

   --------------------------------------------------------------------

   procedure DisposeOfAtom (TheHeap : in out HeapRecord;
                            OldAtom : in     Atom)
   is
   begin
      TheHeap.ListOfAtoms (OldAtom).PointerA := TheHeap.NextFreeAtom;
      TheHeap.NextFreeAtom := OldAtom;
   end DisposeOfAtom;

   --------------------------------------------------------------------

   function APointer (TheHeap : HeapRecord;
                      A       : Atom) return Atom
   is
   begin
      return TheHeap.ListOfAtoms (A).PointerA;
   end APointer;

   --------------------------------------------------------------------

   function BPointer (TheHeap : HeapRecord;
                      A       : Atom) return Atom
   is
   begin
      return TheHeap.ListOfAtoms (A).PointerB;
   end BPointer;

   --------------------------------------------------------------------

   function AValue (TheHeap : HeapRecord;
                    A       : Atom) return Natural
   is
   begin
      return TheHeap.ListOfAtoms (A).ValueA;
   end AValue;

   --------------------------------------------------------------------

   function BValue (TheHeap : HeapRecord;
                    A       : Atom) return Natural
   is
   begin
      return TheHeap.ListOfAtoms (A).ValueB;
   end BValue;

   --------------------------------------------------------------------

   procedure UpdateAPointer (TheHeap : in out HeapRecord;
                             A,
                             Pointer : in     Atom)
   is
   begin
      TheHeap.ListOfAtoms (A).PointerA := Pointer;
   end UpdateAPointer;

   --------------------------------------------------------------------

   procedure UpdateBPointer (TheHeap : in out HeapRecord;
                             A,
                             Pointer : in     Atom)
   is
   begin
      TheHeap.ListOfAtoms (A).PointerB := Pointer;
   end UpdateBPointer;

   --------------------------------------------------------------------

   procedure UpdateAValue (TheHeap : in out HeapRecord;
                           A       : in     Atom;
                           Value   : in     Natural)
   is
   begin
      TheHeap.ListOfAtoms (A).ValueA := Value;
   end UpdateAValue;

   --------------------------------------------------------------------

   procedure UpdateBValue (TheHeap : in out HeapRecord;
                           A       : in     Atom;
                           Value   : in     Natural)
   is
   begin
      TheHeap.ListOfAtoms (A).ValueB := Value;
   end UpdateBValue;

   --------------------------------------------------------------------

   function IsNullPointer (A : Atom) return Boolean
   is
   begin
      return A = NullAtom;
   end IsNullPointer;

   procedure ReportUsage (TheHeap : in HeapRecord)
   is
   begin
      -- As the heap now uses the free list before increasing HighMark,
      -- the max usage is HighMark
      Statistics.SetTableUsage (Statistics.RelationTable,
                                Integer (TheHeap.HighMark));
   end ReportUsage;

   --------------------------------------------------------------------------
   function HeapSize return Natural
   is
      SU : constant := 8; -- bits in a byte
   begin
      return ((HeapRecord'Size + SU) - 1) / SU;
   end HeapSize;


end Heap;
