-- $Id: graph.ads 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Cells,
     EStrings,
     Labels,
     SPARK_IO,
     ExaminerConstants,
     Dictionary;

use type SPARK_IO.File_Status;

--# inherit Cells,
--#         Clists,
--#         CommandLineData,
--#         DAG_IO,
--#         Debug,
--#         Declarations,
--#         Dictionary,
--#         EStrings,
--#         ExaminerConstants,
--#         Labels,
--#         LexTokenManager,
--#         Pairs,
--#         SPARK_IO,
--#         Statistics,
--#         Structures,
--#         SystemErrors;

package Graph
--# own Table : TableType;
--# initializes Table;
is
   ----------------------------------------------------------------------------
   -- The Graph.Table state forms the core data structure of the VC Generator.
   --
   -- It represents the Basic Path Graph (BPG) of the subprogram for which
   -- we are generating VCs.
   --
   -- The BPG is initially built by DAG.BuildGraph, based on data coming
   -- from the Syntax Tree, Dictionary and Flow Analyser.  This BPG has many
   -- "Unspecified" nodes representing every statement of the subprogram under
   -- analysis.  The "arcs" of the graph (between nodes) reflect the simple
   -- control-flow structure of the source program.  Each of these is
   -- generated with a predicate/action Pair structure, representing the
   -- path-traversal condition and (possible) assignment performed by that arc.
   --
   -- After DAG.BuildGraph, the calling environment then invokes Graph.GenVCs.
   --
   -- After GenVCs
   -- ------------
   -- All "Unspecified" nodes are removed, leaving only nodes representing
   -- the pre-codition, post-condition, assertions, checks and so on.
   -- At this point the Pair attached to each arc represents the Verification
   -- Condition for that basic path, after application of Hoare's assignment
   -- axiom to generate weakest pre-conditions and so on.
   --
   ----------------------------------------------------------------------------

   type Dump_Kind is (DPCs, VCs, PFs);
   subtype Valid_Dump_Kind is Dump_Kind range DPCs .. VCs;
   subtype DOT_Dump_Kind is Dump_Kind range VCs .. PFs;

   --# type TableType is abstract; -- for proof functions below

   -- Each node (or "Statement") in the basic-path graph (BPG) is numbered thus...
   subtype MatrixIndex is Integer range 1 .. ExaminerConstants.VCGMatrixOrder;

   -- ...and has a type:
   type ProofContextType is (Assertion,         -- Explicit assert annotation
                             DefaultAssertion,  -- Examiner-generated assertion (loop invariant)
                             CheckStatement,    -- Explicit check annotation
                             RunTimeCheck,      -- Examiner-generated run-time check
                             PreconCheck,       -- Check of a call to a subprogram with a precondition
                             Postcondition,     -- Postcodition
                             Precondition,      -- Precondition
                             Unspecified);      -- Other nodes

   ----------------------------------------------------------------------
   -- The BPG has various attributes which are manipulated by the
   -- following operations
   ----------------------------------------------------------------------


   -----------------------
   -- Number of Statements
   -----------------------

   function GetNmbrOfStmts return MatrixIndex;
   --# global in Table;

   procedure IncNmbrOfStmts;
   --# global in out Table;
   --# derives Table from *;
   --# pre GetNmbrOfStmts (Table) < ExaminerConstants.VCGMatrixOrder;
   --# post GetNmbrOfStmts (Table) = GetNmbrOfStmts (Table~) + 1;

   procedure SetNmbrOfStmts (N : in MatrixIndex);
   --# global in out Table;
   --# derives Table from *,
   --#                    N;
   --# post GetNmbrOfStmts (Table) = N;



   -----------------
   -- Proof Contexts
   -----------------

   -- Returns the ProofContextType for statement N
   --# function GetProofContext (T : in TableType;
   --#                           N : in MatrixIndex) return ProofContextType;

   -- Sets the Proof Context of the largest numbered statement
   procedure SetProofContext (X : in ProofContextType);
   --# global in out Table;
   --# derives Table from *,
   --#                    X;
   --# post GetProofContext (Table, GetNmbrOfStmts (Table)) = X;

   -- Sets the Proof Context for the precondition, which
   -- is always statement 1.
   procedure SetFirstProofContext (X : in ProofContextType);
   --# global in out Table;
   --# derives Table from *,
   --#                    X;
   --# post GetProofContext (Table, 1) = X;


   ---------------------
   -- Assertion Location
   ---------------------

   -- Each statement in the BPG may have a (possibly null)
   -- Assertion associated with it.  This is an FDL predicate
   -- represented as s DAG.

   -- Returns the Assertion for the node denoted by GetNmbrOfStms
   function GetAssertionLocn return Cells.Cell;
   --# global in Table;

   -- Returns the Assertion for the node denoted by GetNmbrOfStms - 1
   function GetPrecedingAssertionLocn return Cells.Cell;
   --# global in Table;
   --# pre GetNmbrOfStmts (Table) > 1;

   -- Sets the Assertion for the node denoted by GetNmbrOfStms
   procedure SetAssertionLocn (X : in Cells.Cell);
   --# global in out Table;
   --# derives Table from *,
   --#                    X;
   --# post GetAssertionLocn (Table) = X;

   -- Sets the Assertion for the precondition, which is always statement 1
   procedure SetFirstAssertionLocn (X : in Cells.Cell);
   --# global in out Table;
   --# derives Table from *,
   --#                    X;


   ------------------------------------------------------------------
   -- TextLineNmbr - associates a statement with a source line number
   ------------------------------------------------------------------

   -- Set the line number associated with the statement denoted by GetNmbrOfStmts
   procedure SetTextLineNmbr (X : in Integer);
   --# global in out Table;
   --# derives Table from *,
   --#                    X;

   -- Set the line number associated with statement denoted by Index
   procedure InsertTextLineNmbr (Index : in MatrixIndex;
                                 X     : in Integer);
   --# global in out Table;
   --# derives Table from *,
   --#                    Index,
   --#                    X;


   ------------------------------------------------------------------
   -- Refinements VCs - In addition to those generated directly
   -- from the BPG, two additional VCs are generated for consistency
   -- of state refinement and two more for sub-class consistency (aka
   -- the VCs for "LSP" or "Covariance".)  These are stored in four
   -- further attributes of this package, thus:
   ------------------------------------------------------------------

   procedure SetRefinementPreCheck (X : in Cells.Cell);
   --# global in out Table;
   --# derives Table from *,
   --#                    X;

   procedure SetRefinementPostCheck (X : in Cells.Cell);
   --# global in out Table;
   --# derives Table from *,
   --#                    X;

   procedure SetSubclassPreCheck (X : in Cells.Cell);
   --# global in out Table;
   --# derives Table from *,
   --#                    X;

   procedure SetSubclassPostCheck (X : in Cells.Cell);
   --# global in out Table;
   --# derives Table from *,
   --#                    X;


   --------------------------------------
   -- (Re-)Initialization of this package
   --------------------------------------


   -- Reset the BPG ready to start over with a new subprogram.
   -- The Table state is returned to the same state it was
   -- in following the elaboration of this package.
   procedure ReInitializeGraph;
   --# global out Table;
   --# derives Table from ;


   ------------------------------------------------------------------
   -- Creation of the BPG
   --
   -- The BPG is stored as a matrix of "Coefficients", each
   -- of which represents an arc from statement I to
   -- statement J with Label K.
   --
   -- The following procedure provides a Constructor for
   -- adding an arc to the BPG
   ------------------------------------------------------------------
   procedure CreateCoeff (Heap : in out Cells.Heap_Record;
                          I, J : in     MatrixIndex;
                          K    : in     Labels.Label);
   --# global in out Statistics.TableUsage;
   --#        in out Table;
   --# derives Heap                  from *,
   --#                                    I,
   --#                                    J,
   --#                                    K,
   --#                                    Table &
   --#         Statistics.TableUsage from *,
   --#                                    Heap &
   --#         Table                 from *,
   --#                                    Heap,
   --#                                    I,
   --#                                    J;


   -- ...and lookup of a Label given statements I and J
   function Coefficient (Heap : Cells.Heap_Record;
                         I, J : MatrixIndex) return Labels.Label;
   --# global in Table;


   ------------------------------------------------------------------
   -- Generation and Printing of Verification Conditions
   ------------------------------------------------------------------

   -- GenVCs essentially reduces the BPG by removing "Unspecified"
   -- statements, using Hoare's assignment axiom.  The traveral-
   -- condition and action of such statements are back-substituted,
   -- and this process is repeated until only explicit pre-condition,
   -- post-condition, assertion, and check statements are left.
   -- The arcs between those remaining statements are left decorated
   -- with their appropriate VC.
   procedure GenVCs (Heap           : in out Cells.Heap_Record;
                     OutputFile     : in     SPARK_IO.File_Type;
                     OutputFileName : in     EStrings.T;
                     Scope          : in     Dictionary.Scopes;
                     GenVCFailure   :    out Boolean);
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out Table;
   --# derives GenVCFailure          from CommandLineData.Content,
   --#                                    Heap,
   --#                                    Table &
   --#         Heap,
   --#         Statistics.TableUsage,
   --#         Table                 from *,
   --#                                    CommandLineData.Content,
   --#                                    Heap,
   --#                                    Table &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    Heap,
   --#                                    OutputFile,
   --#                                    Table &
   --#         null                  from OutputFileName,
   --#                                    Scope;

   procedure Print_VCs_Or_DPCs (Heap       : in out Cells.Heap_Record;
                                OutputFile : in     SPARK_IO.File_Type;
                                Scope      : in     Dictionary.Scopes;
                                Kind       : in     Valid_Dump_Kind);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     Table;
   --#        in out Declarations.State;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives Declarations.State,
   --#         Heap,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage from *,
   --#                                    Declarations.State,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    Kind,
   --#                                    LexTokenManager.State,
   --#                                    Table &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    Declarations.State,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    Kind,
   --#                                    LexTokenManager.State,
   --#                                    OutputFile,
   --#                                    Scope,
   --#                                    Table;

   -------------------------------------------
   -- Debug and Tracing
   -------------------------------------------

   -- Produces a human-readable output of Graph.Table on standard output
   procedure Dump_Graph_Table (Heap           : in out Cells.Heap_Record;
                               Scope          : in     Dictionary.Scopes;
                               Print_Edges_As : in     DOT_Dump_Kind);
   --# global in Table;
   --# derives Heap from * &
   --#         null from Print_Edges_As,
   --#                   Scope,
   --#                   Table;

   -- Produces BPG in DOT format.
   -- The DOT format is that expected by the tools from www.graphviz.org
   --
   -- OutputFileName is the name of the VCG file being generated.  This is
   -- transformed by removing the ".vcg" extension, adding the numeric
   -- suffix given, then adding ".dot" as a new extension.  This allows
   -- for a numerically indexed sequence of graphs to be produced for a single
   -- subprogram.
   procedure Dump_Graph_Dot (Heap                 : in out Cells.Heap_Record;
                             OutputFileName       : in     EStrings.T;
                             OutputFileNameSuffix : in     Natural;
                             Scope                : in     Dictionary.Scopes;
                             Print_Edges_As       : in     DOT_Dump_Kind);
   --# global in Table;
   --# derives Heap from * &
   --#         null from OutputFileName,
   --#                   OutputFileNameSuffix,
   --#                   Print_Edges_As,
   --#                   Scope,
   --#                   Table;

end Graph;
