-- $Id: graph.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Clists,
     CommandLineData,
     Debug,
     Declarations,
     EStrings.Not_Spark,
     DAG_IO,
     LexTokenManager,
     Pairs,
     Structures,
     SystemErrors;

package body Graph
--# own Table is Row,
--#              Column,
--#              InDegree,
--#              OutDegree,
--#              NmbrOfStmts,
--#              AssertionLocn,
--#              ProofContext,
--#              TextLineNmbr,
--#              RefinementPreCheck,
--#              RefinementPostCheck,
--#              SubclassPreCheck,
--#              SubclassPostCheck;
-- If more refinement constituents are added here, then
-- the initialization code in the package body elaboration part
-- AND in procedure ReInitializeGraph will need to be updated.
is
   type VectorOfCells     is array (MatrixIndex) of Cells.Cell;
   type ProofContextArray is array (MatrixIndex) of ProofContextType;
   type VectorOfIntegers  is array (MatrixIndex) of Integer;
   type VectorOfDegrees   is array (MatrixIndex) of Natural;

   Row                 : VectorOfCells;
   Column              : VectorOfCells;
   InDegree            : VectorOfDegrees;
   OutDegree           : VectorOfDegrees;
   NmbrOfStmts         : MatrixIndex;
   AssertionLocn       : VectorOfCells;
   ProofContext        : ProofContextArray;
   TextLineNmbr        : VectorOfIntegers;
   RefinementPreCheck  : Cells.Cell;
   RefinementPostCheck : Cells.Cell;
   SubclassPreCheck    : Cells.Cell;
   SubclassPostCheck   : Cells.Cell;


   --------------------------------------------------------------------------
   procedure IncNmbrOfStmts
   --# global in out NmbrOfStmts;
   --# derives NmbrOfStmts from *;
   is
   begin
      if NmbrOfStmts = ExaminerConstants.VCGMatrixOrder then
         SystemErrors.FatalError (SystemErrors.VCGGraphSizeExceeded, "");
      end if;
      NmbrOfStmts := NmbrOfStmts + 1;
   end IncNmbrOfStmts;

   --------------------------------------------------------------------------

   procedure SetNmbrOfStmts (N : in MatrixIndex)
   --# global out NmbrOfStmts;
   --# derives NmbrOfStmts from N;
   is
   begin
      NmbrOfStmts := N;
   end SetNmbrOfStmts;

   --------------------------------------------------------------------------

   function GetNmbrOfStmts return MatrixIndex
   --# global in NmbrOfStmts;
   is
   begin
      return NmbrOfStmts;
   end GetNmbrOfStmts;

   --------------------------------------------------------------------------

   procedure SetProofContext (X : in ProofContextType)
   --# global in     NmbrOfStmts;
   --#        in out ProofContext;
   --# derives ProofContext from *,
   --#                           NmbrOfStmts,
   --#                           X;
   is
   begin
      ProofContext (NmbrOfStmts) := X;
   end SetProofContext;

   --------------------------------------------------------------------------

   procedure SetFirstProofContext (X : in ProofContextType)
   --# global in out ProofContext;
   --# derives ProofContext from *,
   --#                           X;
   is
   begin
      ProofContext (1) := X;
   end SetFirstProofContext;

   --------------------------------------------------------------------------

   procedure SetAssertionLocn (X : in Cells.Cell)
   --# global in     NmbrOfStmts;
   --#        in out AssertionLocn;
   --# derives AssertionLocn from *,
   --#                            NmbrOfStmts,
   --#                            X;
   is
   begin
      AssertionLocn (NmbrOfStmts) := X;
   end SetAssertionLocn;

   --------------------------------------------------------------------------

   procedure SetFirstAssertionLocn (X : in Cells.Cell)
   --# global in out AssertionLocn;
   --# derives AssertionLocn from *,
   --#                            X;
   is
   begin
      AssertionLocn (1) := X;
   end SetFirstAssertionLocn;

   --------------------------------------------------------------------------

   function GetAssertionLocn return Cells.Cell
   --# global in AssertionLocn;
   --#        in NmbrOfStmts;
   is
   begin
      return AssertionLocn (NmbrOfStmts);
   end GetAssertionLocn;

   --------------------------------------------------------------------------

   function GetPrecedingAssertionLocn return Cells.Cell
   --# global in AssertionLocn;
   --#        in NmbrOfStmts;
   --  pre     NmbrOfStmts > 1;
   is
   begin
      return AssertionLocn (NmbrOfStmts - 1);
   end GetPrecedingAssertionLocn;

   --------------------------------------------------------------------------

   procedure SetTextLineNmbr (X : in Integer)
   --# global in     NmbrOfStmts;
   --#        in out TextLineNmbr;
   --# derives TextLineNmbr from *,
   --#                           NmbrOfStmts,
   --#                           X;
   is
   begin
      TextLineNmbr (NmbrOfStmts) := X;
   end SetTextLineNmbr;

   -----------------------------------------------------------------------

   procedure InsertTextLineNmbr (Index : in MatrixIndex;
                                 X     : in Integer)
   --# global in out TextLineNmbr;
   --# derives TextLineNmbr from *,
   --#                           Index,
   --#                           X;
   is
   begin
      TextLineNmbr (Index) := X;
   end InsertTextLineNmbr;

   -----------------------------------------------------------------------

   procedure SetRefinementPreCheck (X : in Cells.Cell)
   --# global out RefinementPreCheck;
   --# derives RefinementPreCheck from X;
   is
   begin
      RefinementPreCheck := X;
   end SetRefinementPreCheck;

   --------------------------------------------------------------------------

   procedure SetRefinementPostCheck (X : in Cells.Cell)
   --# global out RefinementPostCheck;
   --# derives RefinementPostCheck from X;
   is
   begin
      RefinementPostCheck := X;
   end SetRefinementPostCheck;

   --------------------------------------------------------------------------

   procedure SetSubclassPreCheck (X : in Cells.Cell)
   --# global out SubclassPreCheck;
   --# derives SubclassPreCheck from X;
   is
   begin
      SubclassPreCheck := X;
   end SetSubclassPreCheck;

   --------------------------------------------------------------------------

   procedure SetSubclassPostCheck (X : in Cells.Cell)
   --# global out SubclassPostCheck;
   --# derives SubclassPostCheck from X;
   is
   begin
      SubclassPostCheck := X;
   end SetSubclassPostCheck;

   --------------------------------------------------------------------------

   procedure ReInitializeGraph
   --# global out AssertionLocn;
   --#        out Column;
   --#        out InDegree;
   --#        out NmbrOfStmts;
   --#        out OutDegree;
   --#        out ProofContext;
   --#        out RefinementPostCheck;
   --#        out RefinementPreCheck;
   --#        out Row;
   --#        out SubclassPostCheck;
   --#        out SubclassPreCheck;
   --#        out TextLineNmbr;
   --# derives AssertionLocn,
   --#         Column,
   --#         InDegree,
   --#         NmbrOfStmts,
   --#         OutDegree,
   --#         ProofContext,
   --#         RefinementPostCheck,
   --#         RefinementPreCheck,
   --#         Row,
   --#         SubclassPostCheck,
   --#         SubclassPreCheck,
   --#         TextLineNmbr        from ;
   is
   begin
      -- If this procedure changes, then the package
      -- elaboration code at the end of this compilation
      -- unit will also need to be updated.

      --# accept F, 23, Row,           "Initialization is total" &
      --#        F, 23, Column,        "Initialization is total" &
      --#        F, 23, InDegree,      "Initialization is total" &
      --#        F, 23, OutDegree,     "Initialization is total" &
      --#        F, 23, ProofContext,  "Initialization is total" &
      --#        F, 23, TextLineNmbr,  "Initialization is total" &
      --#        F, 23, AssertionLocn, "Initialization is total";
      for I in MatrixIndex loop
         Row (I)           := Cells.Null_Cell;
         Column (I)        := Cells.Null_Cell;
         InDegree (I)      := 0;
         OutDegree (I)     := 0;
         ProofContext (I)  := Unspecified;
         TextLineNmbr (I)  := 0;
         AssertionLocn (I) := Cells.Null_Cell;
      end loop;
      --# end accept;
      NmbrOfStmts := 1;
      RefinementPreCheck  := Cells.Null_Cell;
      RefinementPostCheck := Cells.Null_Cell;
      SubclassPreCheck := Cells.Null_Cell;
      SubclassPostCheck := Cells.Null_Cell;
      --# accept F, 602, Row, Row,                     "Initialization is total" &
      --#        F, 602, Column, Column,               "Initialization is total" &
      --#        F, 602, InDegree, InDegree,           "Initialization is total" &
      --#        F, 602, OutDegree, OutDegree,         "Initialization is total" &
      --#        F, 602, ProofContext, ProofContext,   "Initialization is total" &
      --#        F, 602, TextLineNmbr, TextLineNmbr,   "Initialization is total" &
      --#        F, 602, AssertionLocn, AssertionLocn, "Initialization is total";
   end ReInitializeGraph;

   --------------------------------------------------------------------------

   procedure CreateCoeff (Heap : in out Cells.Heap_Record;
                          I, J : in     MatrixIndex;
                          K    : in     Labels.Label)
   --# global in out Column;
   --#        in out InDegree;
   --#        in out OutDegree;
   --#        in out Row;
   --#        in out Statistics.TableUsage;
   --# derives Column                from *,
   --#                                    Heap,
   --#                                    J &
   --#         Heap                  from *,
   --#                                    Column,
   --#                                    I,
   --#                                    J,
   --#                                    K,
   --#                                    Row &
   --#         InDegree              from *,
   --#                                    J &
   --#         OutDegree             from *,
   --#                                    I &
   --#         Row                   from *,
   --#                                    Heap,
   --#                                    I &
   --#         Statistics.TableUsage from *,
   --#                                    Heap;
   -- creates coefficient A(I, J), with value K;
   is
      -- 2 NewElement Cells used to reduce heap coupling in flow relations
      NewRowElement,
      NewColElement : Cells.Cell;
   begin
      -- Create both new cells first
      Cells.Create_Cell (Heap, NewRowElement);
      Cells.Create_Cell (Heap, NewColElement);

      -- Set row pointer;
      Cells.Set_Natural_Value (Heap, NewRowElement, J);
      Cells.Set_B_Ptr (Heap, NewRowElement, Labels.LabelHead (K));
      Cells.Set_A_Ptr (Heap, NewRowElement, Row (I));
      Row (I) := NewRowElement;
      OutDegree (I) := OutDegree (I) + 1;

      -- Set column pointer;
      Cells.Set_Natural_Value (Heap, NewColElement, I);
      Cells.Set_B_Ptr (Heap, NewColElement, Labels.LabelHead (K));
      Cells.Set_A_Ptr (Heap, NewColElement, Column (J));
      Column (J) := NewColElement;
      InDegree (J) := InDegree (J) + 1;
   end CreateCoeff;

   --------------------------------------------------------------------------

   function Coefficient (Heap : Cells.Heap_Record;
                         I, J : MatrixIndex) return Labels.Label
   --# global in Row;
   is
      Elem,
      CoeffCell : Cells.Cell;
   begin
      CoeffCell := Cells.Null_Cell;

      -- Pick out the head of the coefficient list for Row I
      Elem := Row (I);
      loop
         exit when Cells.Is_Null_Cell (Elem);

         -- Search the coeff list until a coefficient for column J
         -- is found
         if Cells.Get_Natural_Value (Heap, Elem) = J then
            -- Got it!  Return the Label associated with this coefficient
            CoeffCell := Cells.Get_B_Ptr (Heap, Elem);
            exit;
         end if;
         Elem := Cells.Get_A_Ptr (Heap, Elem);
      end loop;
      return Labels.CellToLabel (CoeffCell);
   end Coefficient;

   --------------------------------------------------------------------------

   procedure Dump_Graph_Dot (Heap                 : in out Cells.Heap_Record;
                             OutputFileName       : in     EStrings.T;
                             OutputFileNameSuffix : in     Natural;
                             Scope                : in     Dictionary.Scopes;
                             Print_Edges_As       : in     DOT_Dump_Kind)
   --# global in AssertionLocn;
   --#        in Column;
   --#        in InDegree;
   --#        in NmbrOfStmts;
   --#        in OutDegree;
   --#        in ProofContext;
   --#        in Row;
   --# derives Heap from * &
   --#         null from AssertionLocn,
   --#                   Column,
   --#                   InDegree,
   --#                   NmbrOfStmts,
   --#                   OutDegree,
   --#                   OutputFileName,
   --#                   OutputFileNameSuffix,
   --#                   Print_Edges_As,
   --#                   ProofContext,
   --#                   Row,
   --#                   Scope;
   is
      --# hide Dump_Graph_Dot;
      Arc         : Cells.Cell;
      ArcFound    : Boolean;
      ArcLabel    : Labels.Label;
      CurrentPair : Pairs.Pair;
      OutputFile  : SPARK_IO.File_Type;
      OK          : SPARK_IO.File_Status;

      procedure Form_And_Open_Output_File
      is
         -- Chop of the .vcg extension
         FN : constant String := EStrings.Not_Spark.Get_String
           (E_Str => EStrings.Section (E_Str     => OutputFileName,
                                       Start_Pos => 1,
                                       Length    => EStrings.Get_Length (E_Str => OutputFileName) - 4));
         -- Form the suffix string and chop off the leading space
         Suffix : constant String := Natural'Image (OutputFileNameSuffix);
         Chopped_Suffix : constant String := String (Suffix (2 .. Suffix'Last));

         DOT_Name : constant String := FN & "_" & Chopped_Suffix & ".dot";
      begin
         SPARK_IO.Create (OutputFile,
                          DOT_Name'Length,
                          DOT_Name,
                          "",
                          OK);
      end Form_And_Open_Output_File;

      procedure PutS (S : in String)
      is
      begin
         SPARK_IO.Put_String (OutputFile, S, 0);
      end PutS;

      procedure PutL (S : in String)
      is
      begin
         SPARK_IO.Put_Line (OutputFile, S, 0);
      end PutL;

      procedure PrintLogicalExpn_DOT (Root : in Cells.Cell)
      is
         SubExpnList : Cells.Cell;
         ListMember  : Cells.Cell;
      begin
         Clists.CreateList (Heap, SubExpnList);
         DAG_IO.Partition (Root, SubExpnList, Heap);

         ListMember := Clists.FirstCell (Heap, SubExpnList);

         DAG_IO.PrintDag (Heap,
                          OutputFile,
                          Cells.Get_B_Ptr (Heap, ListMember),
                          Scope,
                          DAG_IO.No_Wrap);
         ListMember := Clists.NextCell (Heap, ListMember);
         loop
            exit when Cells.Is_Null_Cell (ListMember);
            PutS (" and\l");
            DAG_IO.PrintDag (Heap,
                             OutputFile,
                             Cells.Get_B_Ptr (Heap, ListMember),
                             Scope,
                             DAG_IO.No_Wrap);
            ListMember := Clists.NextCell (Heap, ListMember);
         end loop;
         PutS ("\l");

         Clists.DisposeOfList (Heap, SubExpnList);
      end PrintLogicalExpn_DOT;


      procedure PrintPTC
      is
         Predicate : Cells.Cell;
      begin
         PutS ("taillabel=""");
         if Pairs.IsTrue (Heap, CurrentPair) then
            PutS ("true");
         else
            Predicate := Cells.Get_B_Ptr (Heap, Pairs.PairHead (CurrentPair));
            PrintLogicalExpn_DOT (Predicate);
         end if;
         PutS ("""");
      end PrintPTC;

      procedure PrintAction
      is
         SuppressWrap : Boolean := False;

         Action    : Cells.Cell;
         ModCell   : Cells.Cell;
      begin
         PutS ("headlabel=""");

         if Pairs.IsUnitAction (Heap, CurrentPair) then
            PutS ("null");
         else
            Action := Cells.Get_C_Ptr (Heap, Pairs.PairHead (CurrentPair));
            ModCell := Clists.FirstCell (Heap, Action);

            DAG_IO.PrintCellContents (Heap,
                                      OutputFile,
                                      ModCell,
                                      SuppressWrap,
                                      Scope,
                                      DAG_IO.No_Wrap,
                                      False);
            PutS (" := ");
            DAG_IO.PrintDag (Heap,
                             OutputFile,
                             Cells.Get_B_Ptr (Heap, ModCell),
                             Scope,
                             DAG_IO.No_Wrap);
            ModCell := Clists.NextCell (Heap, ModCell);

            loop
               exit when Cells.Is_Null_Cell (ModCell);

               PutS (" &\n");
               DAG_IO.PrintCellContents (Heap,
                                         OutputFile,
                                         ModCell,
                                         SuppressWrap,
                                         Scope,
                                         DAG_IO.No_Wrap,
                                         False);
               PutS (" := ");
               DAG_IO.PrintDag (Heap,
                                OutputFile,
                                Cells.Get_B_Ptr (Heap, ModCell),
                                Scope,
                                DAG_IO.No_Wrap);
               ModCell := Clists.NextCell (Heap, ModCell);
            end loop;

         end if;
         PutS ("""");
      end PrintAction;

      procedure PrintVC
      is
         Hypotheses  : Cells.Cell;
         Conclusions : Cells.Cell;
      begin
         PutS ("label=""");
         if Pairs.IsTrue (Heap, CurrentPair) then
            PutS ("true");
         else
            Hypotheses  := Cells.Get_B_Ptr (Heap, Pairs.PairHead (CurrentPair));
            PrintLogicalExpn_DOT (Hypotheses);

            PutS ("\l->\l");

            Conclusions := Cells.Get_C_Ptr (Heap, Pairs.PairHead (CurrentPair));
            PrintLogicalExpn_DOT (Conclusions);
         end if;

         PutS ("""");
      end PrintVC;

      procedure PrintNodeDetail (I : in MatrixIndex)
      is
      begin
         SPARK_IO.Put_Integer (OutputFile, I, 0, 10);

         case ProofContext (I) is
            when Precondition |
              Assertion |
              DefaultAssertion |
              Postcondition =>

               -- Make cut-point nodes filled in 50% gray, so they are easy to see
               PutS (" [style=filled,color=gray50,");
            when others =>
               PutS (" [");
         end case;
         PutS ("label=""");

         -- Write label as node number and node type...
         SPARK_IO.Put_String (OutputFile,
                              MatrixIndex'Image (I) & ' ' & ProofContextType'Image (ProofContext (I)),
                              0);

         -- ...and source line if present
         if TextLineNmbr (I) /= 0 then
            PutS (" line ");
            SPARK_IO.Put_Integer (OutputFile, Integer (TextLineNmbr (I)), 0, 10);
         end if;

         PutS ("\n");
         PrintLogicalExpn_DOT (AssertionLocn (I));
         PutS ("""];");
      end PrintNodeDetail;

   begin
      Form_And_Open_Output_File;

      if OK = SPARK_IO.Ok then
         PutS ("digraph ");
         EStrings.Put_String
           (File  => OutputFile,
            E_Str => EStrings.Lower_Case
              (E_Str => LexTokenManager.Lex_String_To_String
                 (Lex_Str => Dictionary.GetSimpleName (Dictionary.GetRegion (Scope)))));
         PutL (" {");
         PutL ("ranksep=""1.0 equally"";");
         PutL ("nodesep=1.0;");
         PutL ("node [shape=box,fontname=helvetica];");
         PutL ("edge [labelfontname=helvetica,labelfontsize=10];");

         -- Nodes
         for I in MatrixIndex range 1 .. NmbrOfStmts loop
            if InDegree (I) = 0 and OutDegree (I) = 0 then
               null; -- node not connected, so skip
            else
               if I = 1 then -- Precondition
                  PutS ("{ rank = source; ");
                  PrintNodeDetail (I);
                  PutS (" }");
               elsif I = NmbrOfStmts then -- Postcondition
                  PutS ("{ rank = sink; ");
                  PrintNodeDetail (I);
                  PutS (" }");
               else
                  PrintNodeDetail (I);
               end if;
               SPARK_IO.New_Line (OutputFile, 1);
            end if;
         end loop;


         -- Edges
         -- For all statements except the precondition
         for Node in MatrixIndex range 2 .. NmbrOfStmts loop
            -- If that node has predecessors
            if InDegree (Node) > 0 then

               -- Then search the coefficients in the Matrix for all
               -- Predecessors whose Successor is Node.
               for Predec in MatrixIndex range 1 .. NmbrOfStmts - 1 loop
                  ArcFound := False;
                  Arc := Column (Node);
                  while (not ArcFound) and (not Cells.Is_Null_Cell (Arc)) loop
                     if Cells.Get_Natural_Value (Heap, Arc) = Predec then
                        ArcFound := True;
                     else
                        Arc := Cells.Get_A_Ptr (Heap, Arc);
                     end if;
                  end loop;

                  if ArcFound then
                     -- Found an arc from Statement Predec to Statement Node

                     ArcLabel := Labels.CellToLabel (Cells.Get_B_Ptr (Heap, Arc));

                     CurrentPair := Labels.FirstPair (Heap, ArcLabel);
                     loop
                        exit when Pairs.IsNullPair (CurrentPair);

                        SPARK_IO.Put_Integer (OutputFile, Predec, 0, 10);
                        PutS (" -> ");
                        SPARK_IO.Put_Integer (OutputFile, Node, 0, 10);

                        PutS ("[style=");

                        case ProofContext (Node) is
                           when CheckStatement |
                             RunTimeCheck |
                             PreconCheck =>

                              PutS ("dashed");

                           when Assertion |
                             DefaultAssertion |
                             Postcondition =>
                              PutS ("bold,headport=n,tailport=s");

                              -- Increase weight for forward edges terminating
                              -- at an assertion of postcondition.
                              if Node > Predec then
                                 PutS (",weight=8.0");
                              end if;


                           when others =>
                              PutS ("solid");
                        end case;

                        case Print_Edges_As is
                           when PFs =>
                              PutS (",");
                              PrintPTC;
                              PutS (",");
                              PrintAction;
                           when VCs =>
                              PutS (",");
                              PrintVC;
                        end case;

                        PutS ("];");

                        SPARK_IO.New_Line (OutputFile, 1);

                        CurrentPair := Labels.NextPair (Heap, CurrentPair);
                     end loop;


                  end if;
               end loop;
            end if;
         end loop;

         PutL ("}");
      end if;

   end Dump_Graph_Dot;

   ----------------------------------------------------------------------

   procedure GenVCs (Heap           : in out Cells.Heap_Record;
                     OutputFile     : in     SPARK_IO.File_Type;
                     OutputFileName : in     EStrings.T;
                     Scope          : in     Dictionary.Scopes;
                     GenVCFailure   :    out Boolean)
   --# global in     AssertionLocn;
   --#        in     CommandLineData.Content;
   --#        in     NmbrOfStmts;
   --#        in     ProofContext;
   --#        in out Column;
   --#        in out InDegree;
   --#        in out OutDegree;
   --#        in out Row;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives Column,
   --#         GenVCFailure,
   --#         InDegree,
   --#         OutDegree,
   --#         Row                   from Column,
   --#                                    CommandLineData.Content,
   --#                                    Heap,
   --#                                    InDegree,
   --#                                    NmbrOfStmts,
   --#                                    OutDegree,
   --#                                    ProofContext,
   --#                                    Row &
   --#         Heap,
   --#         Statistics.TableUsage from *,
   --#                                    AssertionLocn,
   --#                                    Column,
   --#                                    CommandLineData.Content,
   --#                                    Heap,
   --#                                    InDegree,
   --#                                    NmbrOfStmts,
   --#                                    OutDegree,
   --#                                    ProofContext,
   --#                                    Row &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Column,
   --#                                    CommandLineData.Content,
   --#                                    Heap,
   --#                                    InDegree,
   --#                                    NmbrOfStmts,
   --#                                    OutDegree,
   --#                                    OutputFile,
   --#                                    ProofContext,
   --#                                    Row &
   --#         null                  from OutputFileName,
   --#                                    Scope;
   is
      ArcLabel         : Labels.Label;
      AssertionCopy,
      Conjunction,
      CurrentPredicate,
      MatrixElement,
      WP               : Cells.Cell;
      CurrentPair      : Pairs.Pair;
      Failure          : Boolean;
      InitialNode      : MatrixIndex;
      Graph_Suffix     : Natural;

      function IsCheckStatement (X : ProofContextType) return Boolean
      is
      begin
         return X = CheckStatement or else
                X = RunTimeCheck or else
                X = PreconCheck;
      end IsCheckStatement;

      function IsAssertStatement (X : ProofContextType) return Boolean
      is
      begin
         return X = Assertion or else
                X = DefaultAssertion;
      end IsAssertStatement;


      -- deletes pointers to A(I, J);
      procedure DeleteCoeff (Heap : in out Cells.Heap_Record;
                             I, J : in     MatrixIndex)
      --# global in out Column;
      --#        in out InDegree;
      --#        in out OutDegree;
      --#        in out Row;
      --# derives Column,
      --#         Row       from *,
      --#                        Heap,
      --#                        I,
      --#                        J &
      --#         Heap      from *,
      --#                        Column,
      --#                        I,
      --#                        J,
      --#                        Row &
      --#         InDegree  from *,
      --#                        J &
      --#         OutDegree from *,
      --#                        I;
      is
         L1, M1, L2, M2 : Cells.Cell;
      begin
         -- Delete row pointer;
         L1 := Row (I);
         if Cells.Get_Natural_Value (Heap, L1) = J then
            Row (I) := Cells.Get_A_Ptr (Heap, L1);
         else
            loop
               M1 := L1;
               L1 := Cells.Get_A_Ptr (Heap, L1);
               exit when Cells.Get_Natural_Value (Heap, L1) = J;
            end loop;
            Cells.Set_A_Ptr (Heap, M1, Cells.Get_A_Ptr (Heap, L1));
         end if;
         OutDegree (I) := OutDegree (I) - 1;

         -- delete column pointer;
         L2 := Column (J);
         if Cells.Get_Natural_Value (Heap, L2) = I then
            Column (J) := Cells.Get_A_Ptr (Heap, L2);
         else
            loop
               M2 := L2;
               L2 := Cells.Get_A_Ptr (Heap, L2);
               exit when Cells.Get_Natural_Value (Heap, L2) = I;
            end loop;
            Cells.Set_A_Ptr (Heap, M2, Cells.Get_A_Ptr (Heap, L2));
         end if;
         Cells.Dispose_Of_Cell (Heap, L1);
         Cells.Dispose_Of_Cell (Heap, L2);
         InDegree (J) := InDegree (J) - 1;

         --# accept F, 601, Column, Row, "False coupling OK";
      end DeleteCoeff;


      -----------------------------------------------------------------
      -- Partially eliminate statement K of program,
      -- where K denotes a Check statement
      --
      -- For each sequence of paths I -> K -> J, replace
      -- this with I -> J with a Label formed from the
      -- Product of LabelIK and LabelKJ BUT
      -- leave paths I -> K remaining in place.
      --
      -- Repeat until all successors of K
      -- have been considered, at which point K will have
      -- no remaining successors.
      -----------------------------------------------------------------
      procedure Partial_Eliminate (K : in MatrixIndex)
      --# global in     NmbrOfStmts;
      --#        in     ProofContext;
      --#        in out Column;
      --#        in out Heap;
      --#        in out InDegree;
      --#        in out OutDegree;
      --#        in out Row;
      --#        in out Statistics.TableUsage;
      --# derives Column,
      --#         Heap,
      --#         InDegree,
      --#         OutDegree,
      --#         Row,
      --#         Statistics.TableUsage from *,
      --#                                    Column,
      --#                                    Heap,
      --#                                    K,
      --#                                    NmbrOfStmts,
      --#                                    Row &
      --#         null                  from ProofContext;
      is
         P1,
         P2,
         Product : Labels.Label;
      begin
         SystemErrors.RTAssert (IsCheckStatement (ProofContext (K)),
                                SystemErrors.PreconditionFailure,
                                "Trying to Partial_Eliminate a node which isn't a Check");

         -- For all statements J except the Precondition...
         for J in MatrixIndex range 2 .. NmbrOfStmts loop

            -- If J is a successor or K
            if not Labels.IsNull (Coefficient (Heap, K, J)) then

               -- For all statements I except the Postcondition
               for I in MatrixIndex range 1 .. NmbrOfStmts - 1 loop

                  -- If I is a predecessor of K
                  if not Labels.IsNull (Coefficient (Heap, I, K)) then

                     -- I is a predecessor of K;

                     -- form product, taking deep copies of the Labels
                     -- on the paths from I to K and K to J first:
                     Labels.CopyLabel (Heap, Coefficient (Heap, I, K), P1);
                     Labels.CopyLabel (Heap, Coefficient (Heap, K, J), P2);
                     Labels.MultiplyLabels (Heap, P1, P2, Product);

                     -- Check to see of an existing path from
                     -- I to J already exists.
                     if Labels.IsNull (Coefficient (Heap, I, J)) then
                        -- no existing path, so just add a new path from
                        -- I to J with Label Product
                        CreateCoeff (Heap, I, J, Product);
                     else
                        -- Existing path from I to J, so add Product to its
                        -- Label
                        Labels.AddLabels (Heap, Coefficient (Heap, I, J), Product);
                     end if;

                  end if;
               end loop;

               -- Once we've dealt with all the predecessors I that form
               -- paths from I to J via K for a specific Label from K to J,
               -- we can delete the Label from K to J, before going
               -- on to consider the next successor J.
               DeleteCoeff (Heap, K, J);

               -- NOTE that we DON'T delete the original Labels I -> K here,
               -- since these forms the VCs for "all paths reaching a check"
               -- that we need. This is why this is _partial_ eliminate of
               -- statement K
            end if;
         end loop;

         SystemErrors.RTAssert (OutDegree (K) = 0,
                                SystemErrors.PostConditionFailure,
                                "OutDegree of node is not zero after Partial_Eliminate");
      end Partial_Eliminate;



      --------------------------------------------------------------------------
      -- Eliminate statement K of program.
      --
      -- For each sequence of paths I -> K -> J, replace
      -- this with I -> J with a Label formed from the
      -- Product of LabelIK and LabelKJ.
      --
      -- Repeat until all predecessors and successors of K
      -- have been considered, at which point K will have
      -- no reamaining predecessors and successors, effectively
      -- removing it from the BPG.
      --------------------------------------------------------------------------
      procedure Eliminate (Heap : in out Cells.Heap_Record;
                           K    : in     MatrixIndex)
      --# global in     NmbrOfStmts;
      --#        in     ProofContext;
      --#        in out Column;
      --#        in out InDegree;
      --#        in out OutDegree;
      --#        in out Row;
      --#        in out Statistics.TableUsage;
      --# derives Column,
      --#         Heap,
      --#         InDegree,
      --#         OutDegree,
      --#         Row,
      --#         Statistics.TableUsage from *,
      --#                                    Column,
      --#                                    Heap,
      --#                                    InDegree,
      --#                                    K,
      --#                                    NmbrOfStmts,
      --#                                    OutDegree,
      --#                                    Row &
      --#         null                  from ProofContext;
      is
         P1,
         P2,
         Product : Labels.Label;
      begin
         SystemErrors.RTAssert (ProofContext (K) = Unspecified,
                                SystemErrors.PreconditionFailure,
                                "Trying to eliminate a node which isn't UNSPECIFIED");

         -- For each statement
         for I in MatrixIndex range 1 .. NmbrOfStmts - 1 loop

            if not Labels.IsNull (Coefficient (Heap, I, K)) then
               -- I is a predecessor of K, since the Label connecting I to K is not null

               for J in MatrixIndex range 2 .. NmbrOfStmts loop

                  if not Labels.IsNull (Coefficient (Heap, K, J)) then
                     -- J is a successor of K, since the Label connecting K to J is not null

                     -- We've found two nodes I and J such that I is a predecessor
                     -- of K and J is a successor of K in the BPG. Graphically, we've
                     -- found a sequence such as:
                     --    I -> K -> J
                     -- in the BPG.

                     if OutDegree (K) = 1 then
                        -- J is last successor, so take a shallow
                        -- copy of its content into P1
                        P1 := Coefficient (Heap, I, K);
                     else
                        -- Not the last successor, so take a deep
                        -- copy into P1
                        Labels.CopyLabel (Heap, Coefficient (Heap, I, K), P1);
                     end if;

                     if InDegree (K) = 1 then
                        -- I is the last predecessor, so take a shallow
                        -- copy of its content into P2
                        P2 := Coefficient (Heap, K, J);

                        -- If I is the final predecessor, then we won't
                        -- be needed the information regarding successor J again,
                        -- so that Coeff can now be deleted from the BPG
                        DeleteCoeff (Heap, K, J);
                     else
                        -- Not the last predecessor, so take a deep copy into P2
                        Labels.CopyLabel (Heap, Coefficient (Heap, K, J), P2);
                     end if;

                     -- Form the Product of P1 and P2
                     Labels.MultiplyLabels (Heap, P1, P2, Product);

                     -- Check to see of an existing path from
                     -- I to J already exists.
                     if Labels.IsNull (Coefficient (Heap, I, J)) then
                        -- no existing path, so just add a new path from
                        -- I to J with Label Product
                        CreateCoeff (Heap, I, J, Product);
                     else
                        -- Existing path from I to J, so add Product to its
                        -- Label
                        Labels.AddLabels (Heap, Coefficient (Heap, I, J), Product);
                     end if;
                  end if;
               end loop;

               -- Finally, having created or augmented a Label for the path from I to J,
               -- the path from I to K can be deleted.
               DeleteCoeff (Heap, I, K);
            end if;
         end loop;

         -- After elimination, statement K should have both
         -- InDegree and OutDegree set to 0 - i.e. no predecessors
         -- and no successors
         SystemErrors.RTAssert (InDegree (K) = 0,
                                SystemErrors.PostConditionFailure,
                                "InDegree of node is not zero after Eliminate");
         SystemErrors.RTAssert (OutDegree (K) = 0,
                                SystemErrors.PostConditionFailure,
                                "OutDegree of node is not zero after Eliminate");
      end Eliminate;

   begin -- GenVCs
      Failure := False;
      Graph_Suffix := 1;

      for K in MatrixIndex range 2 .. NmbrOfStmts - 1 loop

         -- HTML Directives
         --! <NameFormat> <Name>
         --! <ErrorFormat> <"!!! "><Error>

         --! <Name> program-has-a-cyclic-path-without-an-assertion
         --! <Error> Program has a cyclic path without an assertion.
         --! SPARK generates VCs for paths between cutpoints in the code; these must
         --! be chosen by the developer in such a way that every loop traverses at
         --! least one cutpoint. If the SPARK
         --! Examiner detects a loop which is not broken by a cutpoint,
         --! it cannot generate verification
         --! conditions for the subprogram in which the loop is located,
         --! and instead, issues this
         --! warning. This can only be corrected by formulating a suitable
         --! loop-invariant assertion for
         --! the loop and including it as an assertion in the SPARK text
         --! at the appropriate point.

         if not Labels.IsNull (Coefficient (Heap, K, K)) then
            SPARK_IO.New_Line (OutputFile, 1);
            SPARK_IO.Put_Line (OutputFile,
                               "!!! Program has a cyclic path without an assertion.", 0);
            Failure := True;
            exit;
         end if;

         if IsCheckStatement (ProofContext (K)) then
            -- Explicit Check, Runtime Check, or Precondition Check
            Partial_Eliminate (K);
         elsif not IsAssertStatement (ProofContext (K)) then
            -- Not a Check nor an Assert of any kind.
            -- Can't be Precondition or PostCondition since K cannot
            -- denote these given range of the enclosing loop, so must be
            -- Unspecified
            Eliminate (Heap, K);
         end if;

         --# accept F, 41, "Stable expression expected here";
         if CommandLineData.Content.Debug.VCG_All then
            Dump_Graph_Dot (Heap, OutputFileName, Graph_Suffix, Scope, PFs);
         end if;
         --# end accept;
         Graph_Suffix := Graph_Suffix + 1;

      end loop;

      -- We now have a BPG with all UNSPECIFIED nodes removed - leaving only
      -- explicit assertions, the pre-condition, the post-condition and checks.
      -- Each arc is labelled with its path-traveral condition and action like
      -- a path-function.

      if not Failure then
         -- To generate verification conditions, we do one final application
         -- of the assignment axiom to generate the VC, which is essentially
         --   (Precondition and PTC) -> Postcondition (Action)
         for K in MatrixIndex range 2 .. NmbrOfStmts loop

            if InDegree (K) > 0 then
               MatrixElement := Column (K);
               while not Cells.Is_Null_Cell (MatrixElement) loop
                  InitialNode := Cells.Get_Natural_Value (Heap, MatrixElement);
                  ArcLabel := Labels.CellToLabel (Cells.Get_B_Ptr (Heap, MatrixElement));
                  CurrentPair := Labels.FirstPair (Heap, ArcLabel);
                  while not Pairs.IsNullPair (CurrentPair) loop
                     -- replace path traversal condition p of a pair (p, R) by the
                     -- predicate a /\ p, where a is the assertion at the beginning
                     -- of the path represented by (p, R);
                     Structures.CopyStructure (Heap,
                                               AssertionLocn (InitialNode),
                                               AssertionCopy);
                     if Pairs.IsTrue (Heap, CurrentPair) then
                        Cells.Set_B_Ptr (Heap, Pairs.PairHead (CurrentPair), AssertionCopy);
                     else
                        CurrentPredicate := Cells.Get_B_Ptr (Heap, Pairs.PairHead (CurrentPair));
                        Pairs.FormConjunction (Heap, AssertionCopy, CurrentPredicate, Conjunction);
                        Cells.Set_B_Ptr (Heap, Pairs.PairHead (CurrentPair), Conjunction);
                     end if;

                     -- replace action part R of a pair (p, R) by weakest pre-
                     -- condition WP = q!R, where q is the assertion at the end of the
                     -- path represented by (p, R);
                     Structures.CopyStructure (Heap, AssertionLocn (K), AssertionCopy);

                     if Pairs.IsUnitAction (Heap, CurrentPair) then
                        Cells.Set_C_Ptr (Heap, Pairs.PairHead (CurrentPair), AssertionCopy);
                     else

                        Pairs.CombinePredicateWithAction
                          (Heap        => Heap,
                           Action_R    => Cells.Get_C_Ptr (Heap, Pairs.PairHead (CurrentPair)),
                           Predicate_q => AssertionCopy,
                           Result      => WP);

                        Cells.Set_C_Ptr (Heap, Pairs.PairHead (CurrentPair), WP);
                     end if;
                     CurrentPair := Labels.NextPair (Heap, CurrentPair);
                  end loop;
                  MatrixElement := Cells.Get_A_Ptr (Heap, MatrixElement);
               end loop;
            end if;
         end loop;

         -- Finally, if requested, print out the BPG with VCs on each arc.
         if CommandLineData.Content.Debug.VCG_All then
            Dump_Graph_Dot (Heap, OutputFileName, Graph_Suffix, Scope, VCs);
         end if;

      end if;
      GenVCFailure := Failure;
   end GenVCs;

   ------------------------------------------------------------------------

   procedure Print_VCs_Or_DPCs (Heap       : in out Cells.Heap_Record;
                                OutputFile : in     SPARK_IO.File_Type;
                                Scope      : in     Dictionary.Scopes;
                                Kind       : in     Valid_Dump_Kind)
   --# global in     Column;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     InDegree;
   --#        in     NmbrOfStmts;
   --#        in     ProofContext;
   --#        in     RefinementPostCheck;
   --#        in     RefinementPreCheck;
   --#        in     SubclassPostCheck;
   --#        in     SubclassPreCheck;
   --#        in     TextLineNmbr;
   --#        in out Declarations.State;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives Declarations.State,
   --#         Heap,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage from *,
   --#                                    Column,
   --#                                    Declarations.State,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    InDegree,
   --#                                    Kind,
   --#                                    LexTokenManager.State,
   --#                                    NmbrOfStmts,
   --#                                    ProofContext,
   --#                                    RefinementPostCheck,
   --#                                    RefinementPreCheck,
   --#                                    SubclassPostCheck,
   --#                                    SubclassPreCheck &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Column,
   --#                                    CommandLineData.Content,
   --#                                    Declarations.State,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    InDegree,
   --#                                    Kind,
   --#                                    LexTokenManager.State,
   --#                                    NmbrOfStmts,
   --#                                    OutputFile,
   --#                                    ProofContext,
   --#                                    RefinementPostCheck,
   --#                                    RefinementPreCheck,
   --#                                    Scope,
   --#                                    SubclassPostCheck,
   --#                                    SubclassPreCheck,
   --#                                    TextLineNmbr;
   is
      Arc           : Cells.Cell;
      ArcLabel      : Labels.Label;
      CurrentPair   : Pairs.Pair;
      VCCounter     : Natural;
      ArcFound      : Boolean;
      LexString     : LexTokenManager.Lex_String;
      SubProgString : EStrings.T;

      --------------------------------------------------------------
      procedure PrintSubprogPrefix
      --# global in     Dictionary.Dict;
      --#        in     OutputFile;
      --#        in     Scope;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dictionary.Dict,
      --#                                OutputFile,
      --#                                Scope;
      is
      begin
         if Dictionary.IsFunction (Dictionary.GetRegion (Scope)) then
            SPARK_IO.Put_String (OutputFile, "function_", 0);
         elsif Dictionary.IsProcedure (Dictionary.GetRegion (Scope)) then
            SPARK_IO.Put_String (OutputFile, "procedure_", 0);
         elsif Dictionary.IsTaskType (Dictionary.GetRegion (Scope)) then
            SPARK_IO.Put_String (OutputFile, "task_type_", 0);
         end if;
      end PrintSubprogPrefix;

      -------------------------------------------------------------

      procedure PrintRefinementChecks (Heap    : in out Cells.Heap_Record;
                                       Counter : in     Natural)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     OutputFile;
      --#        in     RefinementPostCheck;
      --#        in     RefinementPreCheck;
      --#        in     Scope;
      --#        in     SubclassPostCheck;
      --#        in     SubclassPreCheck;
      --#        in     SubProgString;
      --#        in out Declarations.State;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --# derives Declarations.State,
      --#         Heap,
      --#         Statistics.TableUsage from *,
      --#                                    Counter,
      --#                                    Declarations.State,
      --#                                    Dictionary.Dict,
      --#                                    Heap,
      --#                                    LexTokenManager.State,
      --#                                    RefinementPostCheck,
      --#                                    RefinementPreCheck,
      --#                                    SubclassPostCheck,
      --#                                    SubclassPreCheck &
      --#         LexTokenManager.State from *,
      --#                                    Counter,
      --#                                    RefinementPostCheck,
      --#                                    RefinementPreCheck,
      --#                                    SubclassPostCheck,
      --#                                    SubclassPreCheck &
      --#         SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    Counter,
      --#                                    Declarations.State,
      --#                                    Dictionary.Dict,
      --#                                    Heap,
      --#                                    LexTokenManager.State,
      --#                                    OutputFile,
      --#                                    RefinementPostCheck,
      --#                                    RefinementPreCheck,
      --#                                    Scope,
      --#                                    SubclassPostCheck,
      --#                                    SubclassPreCheck,
      --#                                    SubProgString;
      is
         Counter_Local : Natural;
         LexString     : LexTokenManager.Lex_String;
      begin
         Counter_Local := Counter;
         if not (Cells.Is_Null_Cell (RefinementPreCheck) and then
                   Cells.Is_Null_Cell (RefinementPostCheck))
         then --refinement VCs are needed
            SPARK_IO.Put_Line (OutputFile, "For checks of refinement integrity: ", 0);
            SPARK_IO.New_Line (OutputFile, 1);

            -- mark VC with unique hash code
--            IO_Routines.HashVCFormula (Heap,
--                                       OutputFile,
--                                       Pairs.CellToPair (RefinementPreCheck),
--                                       Scope);

            PrintSubprogPrefix;
            EStrings.Put_String (File  => OutputFile,
                                 E_Str => SubProgString);
            SPARK_IO.Put_Char (OutputFile, '_');
            LexTokenManager.Insert_Nat (N       => Counter_Local,
                                        Lex_Str => LexString);
            EStrings.Put_String
              (File  => OutputFile,
               E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => LexString));
            SPARK_IO.Put_Line (OutputFile, ".", 0);

            DAG_IO.PrintVCFormula
               (Heap, OutputFile, Pairs.CellToPair (RefinementPreCheck), Scope, DAG_IO.Default_Wrap_Limit);
            Declarations.FindVCFormulaDeclarations
               (Heap, Pairs.CellToPair (RefinementPreCheck), True);

            SPARK_IO.New_Line (OutputFile, 1);

            if not (Cells.Is_Null_Cell (RefinementPostCheck)) then
               Counter_Local := Counter_Local + 1;

               -- mark VC with unique hash code
--               IO_Routines.HashVCFormula (Heap,
--                                          OutputFile,
--                                          Pairs.CellToPair (RefinementPostCheck),
--                                          Scope);

               PrintSubprogPrefix;
               EStrings.Put_String (File  => OutputFile,
                                    E_Str => SubProgString);
               SPARK_IO.Put_Char (OutputFile, '_');
               LexTokenManager.Insert_Nat (N       => Counter_Local,
                                           Lex_Str => LexString);
               EStrings.Put_String
                 (File  => OutputFile,
                  E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => LexString));
               SPARK_IO.Put_Line (OutputFile, ".", 0);

               DAG_IO.PrintVCFormula
                  (Heap, OutputFile, Pairs.CellToPair (RefinementPostCheck), Scope, DAG_IO.Default_Wrap_Limit);
               Declarations.FindVCFormulaDeclarations
                  (Heap, Pairs.CellToPair (RefinementPostCheck), True);

               SPARK_IO.New_Line (OutputFile, 1);
               Counter_Local := Counter_Local + 1; -- for the benefit of subclass check that follows
            end if;
         end if;

         -- do subclass refinements checksre if needed
         if not (Cells.Is_Null_Cell (SubclassPreCheck) and then
                   Cells.Is_Null_Cell (SubclassPostCheck)) then
            SPARK_IO.Put_Line (OutputFile, "For checks of subclass inheritance integrity: ", 0);
            SPARK_IO.New_Line (OutputFile, 1);

            -- mark VC with unique hash code
--            IO_Routines.HashVCFormula (Heap,
--                                       OutputFile,
--                                       Pairs.CellToPair (SubclassPreCheck),
--                                       Scope);

            PrintSubprogPrefix;
            EStrings.Put_String (File  => OutputFile,
                                 E_Str => SubProgString);
            SPARK_IO.Put_Char (OutputFile, '_');
            LexTokenManager.Insert_Nat (N       => Counter_Local,
                                        Lex_Str => LexString);
            EStrings.Put_String
              (File  => OutputFile,
               E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => LexString));
            SPARK_IO.Put_Line (OutputFile, ".", 0);

            DAG_IO.PrintVCFormula
              (Heap, OutputFile, Pairs.CellToPair (SubclassPreCheck), Scope, DAG_IO.Default_Wrap_Limit);
            Declarations.FindVCFormulaDeclarations
              (Heap, Pairs.CellToPair (SubclassPreCheck), True);


            SPARK_IO.New_Line (OutputFile, 1);

            if not Cells.Is_Null_Cell (SubclassPostCheck) then
               -- mark VC with unique hash code
--               IO_Routines.HashVCFormula (Heap,
--                                          OutputFile,
--                                          Pairs.CellToPair (SubclassPostCheck),
--                                          Scope);

               Counter_Local := Counter_Local + 1;
               PrintSubprogPrefix;
               EStrings.Put_String (File  => OutputFile,
                                    E_Str => SubProgString);
               SPARK_IO.Put_Char (OutputFile, '_');
               LexTokenManager.Insert_Nat (N       => Counter_Local,
                                           Lex_Str => LexString);
               EStrings.Put_String
                 (File  => OutputFile,
                  E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => LexString));
               SPARK_IO.Put_Line (OutputFile, ".", 0);

               DAG_IO.PrintVCFormula
                 (Heap, OutputFile, Pairs.CellToPair (SubclassPostCheck), Scope, DAG_IO.Default_Wrap_Limit);
               Declarations.FindVCFormulaDeclarations
                 (Heap, Pairs.CellToPair (SubclassPostCheck), True);

               SPARK_IO.New_Line (OutputFile, 1);
            end if;
         end if;
      end PrintRefinementChecks;

      -------------------------------------------------------------

   begin -- Print_VCs_Or_DPCs
      SubProgString := EStrings.Lower_Case
        (E_Str => LexTokenManager.Lex_String_To_String
           (Lex_Str => Dictionary.GetSimpleName (Dictionary.GetRegion (Scope))));
      SPARK_IO.New_Line (OutputFile, 2);
      VCCounter := 0;
      for Node in MatrixIndex range 2 .. NmbrOfStmts loop

         if InDegree (Node) > 0 then

            for Predec in MatrixIndex range 1 .. NmbrOfStmts - 1 loop
               ArcFound := False;
               Arc := Column (Node);
               while (not ArcFound) and (not Cells.Is_Null_Cell (Arc)) loop
                  if Cells.Get_Natural_Value (Heap, Arc) = Predec then
                     ArcFound := True;
                  else
                     Arc := Cells.Get_A_Ptr (Heap, Arc);
                  end if;
               end loop;
               if ArcFound then
                  SPARK_IO.Put_String (OutputFile, "For path(s) from ", 0);
                  if Predec = 1 then
                     SPARK_IO.Put_String (OutputFile, "start", 0);
                  else
                     if ProofContext (Predec) = Assertion then
                        SPARK_IO.Put_String (OutputFile, "assertion of line ", 0);
                     elsif ProofContext (Predec) = DefaultAssertion then
                        SPARK_IO.Put_String (OutputFile, "default assertion of line ", 0);
                     else -- error case, above two cover all legal cases
                        SPARK_IO.Put_String (OutputFile, "!!!unknown assertion of line ", 0);
                     end if;
                     SPARK_IO.Put_Integer (OutputFile, TextLineNmbr (Predec), 1, 10);
                  end if;
                  SPARK_IO.Put_String (OutputFile, " to ", 0);

                  --# accept F, 41, "Stable expression here OK";
                  case ProofContext (Node) is
                     when Assertion =>
                        SPARK_IO.Put_String (OutputFile, "assertion of line ", 0);
                        SPARK_IO.Put_Integer (OutputFile, TextLineNmbr (Node), 1, 10);

                     when DefaultAssertion =>
                        SPARK_IO.Put_String (OutputFile, "default assertion of line ", 0);
                        SPARK_IO.Put_Integer (OutputFile, TextLineNmbr (Node), 1, 10);

                     when CheckStatement =>
                        SPARK_IO.Put_String (OutputFile, "check associated with statement of line ", 0);
                        SPARK_IO.Put_Integer (OutputFile, TextLineNmbr (Node), 1, 10);

                     when RunTimeCheck =>
                        SPARK_IO.Put_String (OutputFile, "run-time check associated with statement of line ", 0);
                        SPARK_IO.Put_Integer (OutputFile, TextLineNmbr (Node), 1, 10);

                     when PreconCheck =>
                        SPARK_IO.Put_String (OutputFile, "precondition check associated with statement of line ", 0);
                        SPARK_IO.Put_Integer (OutputFile, TextLineNmbr (Node), 1, 10);

                     when Postcondition =>
                        SPARK_IO.Put_String (OutputFile, "finish", 0);

                     when Precondition |
                          Unspecified    => null;
                  end case;
                  --# end accept;

                  SPARK_IO.Put_Line (OutputFile, ":", 0);
                  SPARK_IO.New_Line (OutputFile, 1);
                  ArcLabel := Labels.CellToLabel (Cells.Get_B_Ptr (Heap, Arc));
                  CurrentPair := Labels.FirstPair (Heap, ArcLabel);

                  while not Pairs.IsNullPair (CurrentPair) loop
--                     IO_Routines.HashVCFormula (Heap,
--                                                OutputFile,
--                                                CurrentPair,
--                                                Scope);
                     PrintSubprogPrefix;
                     EStrings.Put_String (File  => OutputFile,
                                          E_Str => SubProgString);
                     SPARK_IO.Put_Char (OutputFile, '_');
                     VCCounter := VCCounter + 1;
                     LexTokenManager.Insert_Nat (N       => VCCounter,
                                                 Lex_Str => LexString);
                     EStrings.Put_String
                       (File  => OutputFile,
                        E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => LexString));
                     SPARK_IO.Put_Line (OutputFile, ".", 0);

                     --# accept F, 41, "Stable expression here expected and OK";
                     case Kind is
                        when VCs =>
                           DAG_IO.PrintVCFormula
                             (Heap, OutputFile, CurrentPair, Scope, DAG_IO.Default_Wrap_Limit);
                        when DPCs =>
                           case ProofContext (Node) is

                              when Assertion |
                                DefaultAssertion |
                                Postcondition =>

                                 DAG_IO.PrintDPC
                                   (Heap, OutputFile, CurrentPair, Scope, DAG_IO.Default_Wrap_Limit);
                              when others =>
                                 SPARK_IO.Put_Line
                                   (OutputFile,
                                    "*** true .          /* DPC not required for intermediate check */", 0);
                                 SPARK_IO.New_Line (OutputFile, 1);
                           end case;
                     end case;
                     --# end accept;

                     -- Find the FDL declarations needed. If we're printing DPCs, then DON'T
                     -- ignore trivially True VCs, since we _will_ produce the hypotheses
                     -- list for these, so we do need FDL declarations for any entities therein.
                     Declarations.FindVCFormulaDeclarations (Heap                   => Heap,
                                                             PredicatePair          => CurrentPair,
                                                             IgnoreTriviallyTrueVCs => (Kind = VCs));

                     CurrentPair := Labels.NextPair (Heap, CurrentPair);
                     SPARK_IO.New_Line (OutputFile, 1);
                  end loop;
               end if;
            end loop;
         end if;
      end loop;

      case Kind is
         when VCs =>
            PrintRefinementChecks (Heap, VCCounter + 1);
         when DPCs =>
            null;
      end case;


   end Print_VCs_Or_DPCs;

   procedure Dump_Graph_Table (Heap           : in out Cells.Heap_Record;
                               Scope          : in     Dictionary.Scopes;
                               Print_Edges_As : in     DOT_Dump_Kind)
   --# global in AssertionLocn;
   --#        in Column;
   --#        in InDegree;
   --#        in NmbrOfStmts;
   --#        in OutDegree;
   --#        in ProofContext;
   --#        in RefinementPostCheck;
   --#        in RefinementPreCheck;
   --#        in Row;
   --#        in SubclassPostCheck;
   --#        in SubclassPreCheck;
   --#        in TextLineNmbr;
   --# derives Heap from * &
   --#         null from AssertionLocn,
   --#                   Column,
   --#                   InDegree,
   --#                   NmbrOfStmts,
   --#                   OutDegree,
   --#                   Print_Edges_As,
   --#                   ProofContext,
   --#                   RefinementPostCheck,
   --#                   RefinementPreCheck,
   --#                   Row,
   --#                   Scope,
   --#                   SubclassPostCheck,
   --#                   SubclassPreCheck,
   --#                   TextLineNmbr;
   is
      --# hide Dump_Graph_Table;
      Arc           : Cells.Cell;
      ArcLabel      : Labels.Label;
      CurrentPair   : Pairs.Pair;
      VCCounter     : Natural;
      ArcFound      : Boolean;
   begin
      Debug.PrintInt ("Number of Statements is: ", Integer (NmbrOfStmts));
      for I in MatrixIndex range 1 .. NmbrOfStmts loop
         if InDegree (I) = 0 and OutDegree (I) = 0 then
            Debug.PrintMsg ("Statement" & Integer'Image (I) & " not connected", True);
         else
            Debug.PrintInt ("Statement", I);
            Debug.PrintMsg ("   Proof Context    = " & ProofContextType'Image (ProofContext (I)), True);
            Debug.PrintInt ("   Text Line Number =", Integer (TextLineNmbr (I)));
            Debug.PrintInt ("   In Degree        =", Integer (InDegree (I)));
            Debug.PrintInt ("   Out Degree       =", Integer (OutDegree (I)));
            Debug.PrintDAG ("   Assertion Locn   = ", AssertionLocn (I), Heap, Scope);
         end if;
      end loop;

      -- Now Dump each arc represented by each coefficient in the Matrix itself
      -- Basically the same algorithm as PrintVCs above.

      VCCounter := 0;
      -- For all statements except the precondition
      for Node in MatrixIndex range 2 .. NmbrOfStmts loop
         -- If that node has predecessors
         if InDegree (Node) > 0 then

            -- Then search the coefficients in the Matrix for all
            -- Predecessors whose Successor is Node.
            for Predec in MatrixIndex range 1 .. NmbrOfStmts - 1 loop
               ArcFound := False;
               Arc := Column (Node);
               while (not ArcFound) and (not Cells.Is_Null_Cell (Arc)) loop
                  if Cells.Get_Natural_Value (Heap, Arc) = Predec then
                     ArcFound := True;
                  else
                     Arc := Cells.Get_A_Ptr (Heap, Arc);
                  end if;
               end loop;

               if ArcFound then
                  -- Found an arc from Statement Predec to Statement Node
                  Debug.PrintMsg
                    ("Found an arc from Stm" & Integer'Image (Predec) &
                       " to Stm" & Integer'Image (Node), True);

                  -- Fetch the Label associated with that arc
                  ArcLabel := Labels.CellToLabel (Cells.Get_B_Ptr (Heap, Arc));

                  case Print_Edges_As is
                     when PFs =>
                        DAG_IO.PrintLabel (Heap, SPARK_IO.Standard_Output, ArcLabel, Scope, DAG_IO.Default_Wrap_Limit);
                     when VCs =>
                        -- Fetch the first Pair attached to that Label
                        CurrentPair := Labels.FirstPair (Heap, ArcLabel);
                        while not Pairs.IsNullPair (CurrentPair) loop
                           VCCounter := VCCounter + 1;
                           Debug.PrintInt ("Pair", VCCounter);
                           DAG_IO.PrintVCFormula (Heap, SPARK_IO.Standard_Output, CurrentPair, Scope, DAG_IO.Default_Wrap_Limit);
                           CurrentPair := Labels.NextPair (Heap, CurrentPair);
                        end loop;
                  end case;
               end if;
            end loop;
         end if;
      end loop;

   end Dump_Graph_Table;


begin

   -- This code matches that in ReInitializeGraph

   --# accept F, 23, Row,           "Initialization is total" &
   --#        F, 23, Column,        "Initialization is total" &
   --#        F, 23, InDegree,      "Initialization is total" &
   --#        F, 23, OutDegree,     "Initialization is total" &
   --#        F, 23, ProofContext,  "Initialization is total" &
   --#        F, 23, TextLineNmbr,  "Initialization is total" &
   --#        F, 23, AssertionLocn, "Initialization is total";
   for I in MatrixIndex loop
      Row (I)           := Cells.Null_Cell;
      Column (I)        := Cells.Null_Cell;
      InDegree (I)      := 0;
      OutDegree (I)     := 0;
      ProofContext (I)  := Unspecified;
      TextLineNmbr (I)  := 0;
      AssertionLocn (I) := Cells.Null_Cell;
   end loop;
   NmbrOfStmts := 1;
   RefinementPreCheck  := Cells.Null_Cell;
   RefinementPostCheck := Cells.Null_Cell;
   SubclassPreCheck := Cells.Null_Cell;
   SubclassPostCheck := Cells.Null_Cell;

   --# accept F, 602, Row, Row,                     "Initialization is total" &
   --#        F, 602, Column, Column,               "Initialization is total" &
   --#        F, 602, InDegree, InDegree,           "Initialization is total" &
   --#        F, 602, OutDegree, OutDegree,         "Initialization is total" &
   --#        F, 602, ProofContext, ProofContext,   "Initialization is total" &
   --#        F, 602, TextLineNmbr, TextLineNmbr,   "Initialization is total" &
   --#        F, 602, AssertionLocn, AssertionLocn, "Initialization is total";
end Graph;
