-- $Id: estrings.ads 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--  EStrings
--
--  Purpose:
--   Provides an ADT for a fixed-length String type.
--
--  In addition to the basic constructors and operations, simple I/O
--  procedure are prodived to Put and Read such strings to/from a SPARK_IO file.
--
--  Clients:
--   Used throughout the Examiner and other tools.
--
--  Use:
--   Declare an EString.T
--
--   Initialization via the constant EmptyString or via
--   constructor CopyString
--
--  Extension:
--   None planned.  See ELStrings for a similar (but longer) String ADT.
--------------------------------------------------------------------------------

with SPARK_IO;

--# inherit Ada.Characters.Handling,
--#         Ada.Characters.Latin_1,
--#         SPARK_IO;

package EStrings
is
   -----------------------------------------------------------------------------
   -- Types and subtypes.  Note these are visible, so a client _can_ directly
   -- access the internal structure of EStrings.T as a String subtype
   -----------------------------------------------------------------------------

   subtype Lengths is Natural range 0 .. 255;
   Max_String_Length : constant Lengths := Lengths'Last;

   subtype Positions is Lengths range 1 .. Max_String_Length;

   type T is private;

   type Order_Types is (First_One_First, Second_One_First, Neither_First);

   -----------------------------------------------------------------------------
   -- Constants
   -----------------------------------------------------------------------------

   Empty_String : constant T;

   -----------------------------------------------------------------------------
   -- Operations
   -----------------------------------------------------------------------------

   -- Equality operators
   function Eq_String (E_Str1, E_Str2 : T) return Boolean;

   function Eq1_String (E_Str : T;
                        Str   : String) return Boolean;

   function Eq_CS_String (E_Str1, E_Str2 : T) return Boolean;

   function Is_Empty (E_Str : T) return Boolean;

   function Get_Length (E_Str : T) return Lengths;

   function Get_Element (E_Str : T;
                         Pos   : Positions) return Character;

   function Copy_String (Str : String) return T;

   -- Append Str to EStr, truncating on overflow
   procedure Append_String (E_Str : in out T;
                            Str   : in     String);
   --# derives E_Str from *,
   --#                    Str;

   -- Append EStr2 to EStr1, truncating on overflow
   procedure Append_Examiner_String (E_Str1 : in out T;
                                     E_Str2 : in     T);
   --# derives E_Str1 from *,
   --#                     E_Str2;

   function Lower_Case (E_Str : T) return T;

   function Upper_Case (E_Str : T) return T;

   function Lower_Case_Char (E_Str : T;
                             Pos   : Positions) return T;

   function Upper_Case_Char (E_Str : T;
                             Pos   : Positions) return T;

   function Translate (E_Str     : T;
                       From_Char : Character;
                       To_Char   : Character) return T;

   procedure Append_Char (E_Str   : in out T;
                          Ch      : in     Character;
                          Success :    out Boolean);
   --# derives E_Str   from *,
   --#                      Ch &
   --#         Success from E_Str;

   -- Find_Sub_String_After: for use in summary tool
   -- find the specified SearchString, starting at the specified position in
   -- the given T
   procedure Find_Sub_String_After (E_Str         : in     T;
                                    Search_Start  : in     Positions;
                                    Search_String : in     String;
                                    String_Found  :    out Boolean;
                                    String_Start  :    out Positions);
   --# derives String_Found,
   --#         String_Start from E_Str,
   --#                           Search_Start,
   --#                           Search_String;

   -- Find_Sub_String: for use in summary tool
   -- find specified SearchString in the given T
   procedure Find_Sub_String (E_Str         : in     T;
                              Search_String : in     String;
                              String_Found  :    out Boolean;
                              String_Start  :    out Positions);
   --# derives String_Found,
   --#         String_Start from E_Str,
   --#                           Search_String;

   procedure Find_Examiner_Sub_String (E_Str         : in     T;
                                       Search_String : in     T;
                                       String_Found  :    out Boolean;
                                       String_Start  :    out Positions);
   --# derives String_Found,
   --#         String_Start from E_Str,
   --#                           Search_String;

   -- Pop_Char takes as input a T.  It removes the first character
   -- from the string and returns it in the Char output parameter.  If the empty
   -- string is passed in then the outputs are:
   --    EStr = EmptyString
   --    Char = ' '
   procedure Pop_Char (E_Str : in out T;
                       Char  :    out Character);
   --# derives Char,
   --#         E_Str from E_Str;

   -- Find_Char_After: find specified character in EStr, starting at specified
   -- position
   procedure Find_Char_After (E_Str        : in     T;
                              Search_Start : in     Positions;
                              Search_Char  : in     Character;
                              Char_Found   :    out Boolean;
                              Char_Pos     :    out Positions);
   --# derives Char_Found,
   --#         Char_Pos   from E_Str,
   --#                         Search_Char,
   --#                         Search_Start;

   -- Find_Char: find first occurrence of specified character in EStr
   procedure Find_Char (E_Str       : in     T;
                        Search_Char : in     Character;
                        Char_Found  :    out Boolean;
                        Char_Pos    :    out Positions);
   --# derives Char_Found,
   --#         Char_Pos   from E_Str,
   --#                         Search_Char;

   function Lex_Order (First_Name, Second_Name : T) return Order_Types;

   -- Section added for use in summary tool
   -- returns the specified subsection of the string
   -- if the subsection lies outside the string, empty string returned
   function Section (E_Str     : T;
                     Start_Pos : Positions;
                     Length    : Lengths) return T;

   -- Trim added for use in summary tool
   -- removes ' ', Latin_1.HT, Latin_1.LF, Latin_1.CR from either end of the string
   function Trim (E_Str : T) return T;

   procedure Get_Int_From_String (Source   : in     T;
                                  Item     :    out Integer;
                                  Start_Pt : in     Positive;
                                  Stop     :    out Natural);
   --# derives Item,
   --#         Stop from Source,
   --#                   Start_Pt;

   -----------------------------------------------------------------------------
   -- I/O procedures
   -----------------------------------------------------------------------------

   procedure Create (File         : in out SPARK_IO.File_Type;
                     Name_Of_File : in     T;
                     Form_Of_File : in     String;
                     Status       :    out SPARK_IO.File_Status);
   --# global in out SPARK_IO.File_Sys;
   --# derives File,
   --#         SPARK_IO.File_Sys,
   --#         Status            from File,
   --#                                Form_Of_File,
   --#                                Name_Of_File,
   --#                                SPARK_IO.File_Sys;

   procedure Open (File         : in out SPARK_IO.File_Type;
                   Mode_Of_File : in     SPARK_IO.File_Mode;
                   Name_Of_File : in     T;
                   Form_Of_File : in     String;
                   Status       :    out SPARK_IO.File_Status);
   --# global in out SPARK_IO.File_Sys;
   --# derives File,
   --#         SPARK_IO.File_Sys,
   --#         Status            from File,
   --#                                Form_Of_File,
   --#                                Mode_Of_File,
   --#                                Name_Of_File,
   --#                                SPARK_IO.File_Sys;

   procedure Put_String (File  : in SPARK_IO.File_Type;
                         E_Str : in T);
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                E_Str,
   --#                                File;

   procedure Put_Line (File  : in SPARK_IO.File_Type;
                       E_Str : in T);
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                E_Str,
   --#                                File;

   procedure Get_Line (File  : in     SPARK_IO.File_Type;
                       E_Str :    out T);
   --# global in out SPARK_IO.File_Sys;
   --# derives E_Str,
   --#         SPARK_IO.File_Sys from File,
   --#                                SPARK_IO.File_Sys;

private

   subtype Contents is String (Positions);

   type T is record
      Length  : Lengths;
      Content : Contents;
   end record;

   Empty_String : constant T :=
     T'(Length => 0, Content => Contents'(Positions => ' '));

end EStrings;
