-- $Id: errorhandler.ads 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

--------------------------------------------------------------------------------
--  ErrorHandler
--
--  Purpose:
--    The package ErrorHandler is used to report almost all warnings and errors
--    detected by the Examiner during analysis and produce summaries of the
--    number of warnings and errors encountered.
--
--    ErrorHandler is also responsible for the definition and suppression
--    of warnings specified in the warning control file (see Examiner User
--    Manual section 4.6)
--
--    The processing of justifications using the accept annotation
--   (see Examiner User Manual section 9.1) is also managed by ErrorHandler.
--
--    The Examiner will terminate with a status code dependent on whether an
--    error has occurred and if applicable the class of error.  The ErrorHandler
--    package provides a function for determining the status code.  The exit
--    codes are declared below.
--
--    Errors are accumulated as they are encountered in ErrorContexts.
--    The ErrorHandler only manages a single ErrorContext (represented
--    by the own variable ErrorContext) but an ErrorContext for each source
--    file read is maintained by the ContextManager.
--
--    The main ErrorHandler interface provides subprograms to manage
--    ErrorContexts, add various classes of errors and warnings to the
--    accumulated errors and others to initialize, reset and list or print the
--    accumulated warnings and errors or summaries thereof.
--
--    Another section of the ErrorHandler interface is concerned with the
--    Warning Control file and listing the warning messages that have been
--    suppressed.
--
--    A further part of the ErrorHandler interface manages Justifications.
--
--  Clients:
--    The ErrorHandler module is used for reporting almost all errors except
--    errors detected within the command line (See sparkmain.adb) and fatal
--    errors which immediately terminate the Examiner by calling
--    SystemErrors.FatalError.
--    Consequently ErrorHandler is used by many parts of the Examiner and has
--    many clients, for instance, the FlowAnalyser and Sem.
--    ErrorHandler is also used by SPARKmake.
--
--  Use:
--    Reporting Errors and ErrorContexts:
--
--    (1) When a source file (or configuration file) is opened initialize the
--        ErrorContext by calling ErrorHandler.ErrorInit
--       (or errorHandler.SparkMakeInit from SPARKmake).
--        This ErrorContext is retrieved by calling GetErrorContext and is
--        associated with the source file and saved by calling
--        ContextManager.SetContext (See mainloop.adb).
--
--    (2) For each compilation unit (or a configuration file) to be analysed
--        retrieve the ErrorContext associated with the file containing the
--        compilation unit by calling ContextManager.GetErrorContext and set
--        the current ErrorContext maintained by the ErrorHandler to the
--        retrieved ErrorContext by calling ErrorHandler.SetErrorContext
--        (See mainloop.adb)
--
--     (3) Prior to analysis a body or package initialization call
--         ErrorHandler.ErrorReset to reset the severity of the ErrorContext as
--         this property only applies to a single body or package
--         initialization (See sem-compunit.adb).
--
--     (4) To add a warning or error to the accumulated messages of the
--         ErrorContext call an appropriate interface subprogram from those
--         declared below.
--
--     (5) On completing analysis of each subprogram body
--        (subprogram_implementation) call ErrorHandler.FlushEchoMessages to
--         complete any messages relating to the subprogram that are sent to
--         the screen (standard output) (See sem-compunit.adb).
--
--     (6) After the analysis of each unit ErrorHandler.GetErrorSeverity may
--         be called to determine the worst severity of any errors encountered.
--         The severity levels are declared below (See sem-compunit.adb).
--
--     (7) Call ErrorHander.PrintErrors to generate a listing file annotated
--         with the accumulated error messages for the corresponding source
--         or configuration file (See mainloop.adb).
--
--         Call ErrorHandler.AppendErrors to add errors related to a source
--         or configuration file to the report file (see mainloop.adb).
--
--     (8) Call ErrorHandler.EchoTotalErrorCount after all files have been
--         analysed to display the total error count on the screen
--         (standard_output) (See mainloop.adb).
--
--     (9) Call ErrorHandler.GetErrorsType to obtain the exit code which
--         defines (the worst) type of error encountered during analysis.
--         See below for definition of values.
--
--    (10) If references to Ada LRM 83 or 95, SPARK 83 or 95,
--         Examiner User Manual, VCG or RTC Manuals or
--         Barnes "High Integrity Software" are associated with any warning or
--         error messages issued may be listed to a file by calling
--         ErrorHandler.OutputReferenceList. It is normally called to add
--         these to the report file unless the -brief switch is specified
--         (See mainloop.adb).
--
--    (11) If it is not possible to open any input file (configuration file,
--         meta file or source file), ErrorHandler.SetFileOpenError should
--         be called to register this failure.  The failure is not recorded as
--         part of an ErrorContext and related to a single file but as a
--         pervasive event (See mainloop.adb).
--
--     Warning File:
--
--     (1) Read in the warning control file by calling
--         ErrorHandler.ReadWarningFile (See mainloop.adb).
--
--         Warnings which are correctly designated in the warning control file
--         (see Examiner User Manual 9.2.1) will now be suppressed.
--
--     (2) A list of suppressed may be written to a file by calling
--         ErrorHandler.OutputWarningList.  It is called in mainloop.adb
--         to write the list to the report file.
--
--     Justifications using Accept Annotations:
--
--     (1) Call ErrorHandler.StartUnit when a new unit is to be analysed
--         (see sem-compunit.adb).
--
--     (2) Call ErrorHandler.StartJustification when an accept annoation is
--         encountered (See sem-compunit-wf_justification_statement.adb).
--
--         The errors justified by the accept annotation will no longer be
--         reported except in the justification summary.
--
--     (3) Call ErrorHandler.EndJustification when an end accept annotation is
--         encountered (See sem-compunit-wf_justification_statement.adb).
--
--     (4) Call ErrorHandler.EndUnit when the analysis of a unit is complete
--         (See sem-compunit.adb).
--
--  Extension:
--    Warnings:
--
--    To add a new warning
--
--    (1) Choose a unique warning number within the range defined by
--        Error_Types.ErrNumRange - distinct from any number listed
--        in the Examiner User Manual section 6.3 "Warning Messages".
--
--    (2) Consider whether the warning is to be suppressible and if suppressible
--        whether it is a severe warning.
--
--    (3) If it is to be suppressible a new keyword must be introduced.
--
--    If the warning is to be suppressible...
--
--    (4) Add a suitable enumeration element to the type WarningElements putting
--        it into the correct part of the list depending on whether it is a
--        severe (starred) warning or not.
--        Keep UnexpectedAddressClauses as the last  non-severe element of
--        the enumeration or it corrupts the subtype definition of
--        "SevereWarnings".
--
--    (5) In errorhandler-warningstatus.adb, procedure GetDescription,
--        add an extra case alternative to describe the new warning.
--
--    (6) In errorhandler-warningstatus-readwarningfile.adb extend procedure
--        ProcessOption to recognise new keyword.
--
--    (7) In the body of ErrorHandler, in procedures SemanticWarningWithPosition
--        or SemanticWarningWithoutPosition, add if/elsif clauses to increment
--        counters, rather than display warning, when suppression has been
--        selected.

--    For all warnings independent of suppression
--
--    (8) If the chosen error number is less than 400 add an extra case
--        alternative to procedure WarningWithPosition define the new warning
--        text and its explanation in file
--        errorhandler-conversions-tostring-warningwithposition.adb.
--
--        If the chosen error number is greater or equal to 400 add an extra
--        case alternative to procedure WarningWithoutPosition define the new
--        warning text and its explanation in file
--        errorhandler-conversions-tostring-warningwithoutposition.adb.
--
--        In either of the above two cases the format of the warning text to be
--        added is a catenation of strings to an EString.T for the standard
--        warning text followed by a number of comment lines starting with --!.
--        The text in the comments starting with --! constitute the error
--        explanation which is given when the Examiner -error_explanation
--        command line switch is set to first or every_occurrence.
--
--        Two new files are automatically generated from the above two files
--        during the build process to provide SPARK procedures which incorporate
--        the error explanations see
--        errorhandler-conversions-tostring-warningwithposition-warningwithpositionexpl.adb and
--        errorhandler-conversions-tostring-warningwithoutposition-warningwithoutpositionexpl.adb.
--
--    To add a new note
--
--    (1) Choose a unique note number within the range defined by
--        Error_Types.ErrNumRange - distinct from any number listed
--        in the Examiner User Manual section 6.4 "Notes".
--
--    (2) Add an extra case alternative to procedure Note in
--        errorhandler-conversions-tostring-note.adb defining the text and
--        explanation of the note as described for warnings (8).
--
--    To add a new semantic error
--    (excluding those related to inconsistencies between abstract and refined
--     dependency relations)
--
--    (1) Choose a unique semantic error number within the range defined by
--        Error_Types.ErrNumRange - distinct from any number listed
--        in the Examiner User Manual section 6.2 "Semantic Errors".
--
--    (2) Add an extra case alternative to procedure SemanticErr in
--        errorhandler-conversions-tostring-semanticerr.adb defining the text
--        and explanation of the semantic error as described for warnings (8).
--
--    (3) If the error message contains a reference to a document which is
--        already in the list of document references maintained in
--        errorhandler-conversions-tostring-appendreference.adb, then when a
--        semantic error reporting subprogram is called the document reference
--        number may be given.
--
--    (4) If the message contains a reference to a document which is not in
--        the list of documents in
--        errorhandler-conversions-tostring-appendreference.adb then in the file
--        increment the value of the constant  MaxReferences and add a new
--        document reference as the last element of the constant array
--        ReferenceTable.
--
--    To add a new error related to inconsistencies between abstract and
--    refined dependency relations
--
--    (1) Choose a unique semantic error number within the range defined by
--        Error_Types.ErrNumRange - distinct from any number listed
--        in the Examiner User Manual section 8.3.7
--        "Inconsistencies between abstract and refined dependency relations".
--
--    (2) Add an extra case alternative to procedure DepSemanticErr in
--        errorhandler-conversions-tostring-depsemanticerr.adb defining the text
--        and explanation of the semantic error as described for warnings (8).
--
--    To add a new control-flow error
--
--    (1) Choose a unique control-flow error number within the range defined by
--        Error_Types.ErrNumRange - distinct from any number listed
--        in the Examiner User Manual section 7.2 "Control-flow Errors".
--
--    (2) Add a new enumeration literal to the type ControlFlowErrType declared
--        below to represent the new error.
--
--    (3) Add an extra case alternative to procedure ControlFlowError in
--        errorhandler-conversions-tostring-controlflowerror.adb defining the
--        text and explanation of the control-flow error as described for
--        warnings (8). Additionally associate the chosen error number
--        with the text using a comment with the syntax --! <name> ErrorNumber.
--
--    To add a new data-flow error
--
--    (1) Choose a unique data-flow error number within the range defined by
--        Error_Types.ErrNumRange - distinct from any number listed
--        in the Examiner User Manual section 8.3
--        "Data and information-flow Errors".
--
--    (2) Add a new enumeration literal to the type DataFlowErrType declared
--        below to represent the new error.
--
--    (3) In errorhandler.adb, procedure DataFlowError, add the new data-flow
--        enumeration to the appropriate alternative of the case statement
--        dependent on whether the new error is a conditional or unconditional
--        flow-error.
--
--    (4) Add an extra case alternative to either procedure CondlFlowErr or
--        procedure UncondFlowErr in
--        errorhandler-conversions-tostring-condlflowerr.adb or
--        errorhandler-conversions-tostring-uncondflowerr.adb respectively
--        defining the text and explanation of the data-flow error as described
--        for warnings (8). Additionally associate the chosen error number
--        with the text using a comment with the syntax --! <name> ErrorNumber.
--
--    To add a new information-flow error
--
--    (1) Choose a unique information-flow error number within the range defined
--        by Error_Types.ErrNumRange - distinct from any number listed
--        in the Examiner User Manual section 8.3
--        "Data and information-flow Errors".
--
--    (2) Add a new enumeration literal to one of the types FullDependErrType,
--        StabilityErrType, IndexType or UsageErrType dependent on the category
--        of the error to be reported.  If a new litereral is to be added to
--        FullDependencyErrType it must appear immediately before NotUsedNew so
--        that it is in the subtype DependencyErrType.
--
--    (3) If the error is of the subtype DependencyErrType determine whether
--        the error is a conditional or unconditional flow error and
--        Add the new enumeration literal to the appropriate case alternative
--        of the procedure DependencyError in errorhandler.adb.
--
--    (4) Add an extra case alternative to the appropriate procedure
--        CondlDependency, IneffectiveStatement, StabilityError,
--        UncondDependency or UsageError in
--        errorhandler-conversions-tostring-condldependency.adb
--        errorhandler-conversions-tostring-ineffectivestatement.adb
--        errorhandler-conversions-tostring-stabilityerror.adb
--        errorhandler-conversions-tostring-unconddependency.adb
--        errorhandler-conversions-tostring-usageerror.adb
--        respectively defining the text and explanation of the information-flow
--        error as described for warnings (8).
--        Additionally associate the chosen error number with the text using a
--        comment with the syntax --! <name> ErrorNumber.
--
--    Lexical and syntax errors are not extended in this package but are
--    extended within the SPARKLex and SPParser subsystems.
--
--    It is not expected that the Justification interface will be extended.
--
--------------------------------------------------------------------------------

with Dictionary;
with Error_IO;
with Error_Types;
with EStrings;
with ExaminerConstants;
with LexTokenManager;
with SPARK_IO;
with SPExpectedSymbols;
with SPSymbols;

use type Dictionary.Scopes;
use type Dictionary.Symbol;
use type Error_Types.Error_Class;
use type Error_Types.Names;
use type Error_Types.NumericError;
use type Error_Types.StringError;
use type LexTokenManager.Line_Numbers;
use type LexTokenManager.Str_Comp_Result;
use type LexTokenManager.Token_Position;
use type SPARK_IO.File_Status;
use type SPARK_IO.File_Type;
use type SPExpectedSymbols.SPEssSymRange;
use type SPSymbols.SPSymbol;

pragma Elaborate_All (SPARK_IO);

--# inherit Ada.Characters.Handling,
--#         Ada.Characters.Latin_1,
--#         CommandLineData,
--#         CommandLineHandler,
--#         Dictionary,
--#         ELStrings,
--#         Error_IO,
--#         Error_Types,
--#         EStrings,
--#         ExaminerConstants,
--#         FileSystem,
--#         LexTokenManager,
--#         ScreenEcho,
--#         SPARK_IO,
--#         SPExpectedSymbols,
--#         SPRelations,
--#         SPSymbols,
--#         SystemErrors,
--#         XMLReport;

package ErrorHandler
--# own ErrorContext;
--# initializes ErrorContext;
is

   -- Used to define the most serious sort of error encountered in an analysis
   -- Fatal is most serious and NoError the least.
   type ErrorLevel         is (Fatal, SemanticErrs, FlowErrs, FlowWarning, NoError);

   -- Enumeration literals representing the different sorts of control-flow
   -- error that may be reported.
   type ControlFlowErrType is (MisplacedExit,
                               MisplacedReturn, MissingReturn, ReturnInProc);

   -- Enumeration literals representing the different sorts of data-flow
   -- error that may be reported.
   type DataFlowErrType    is (ExpnUndefined, ExpnMayBeUndefined, InvariantExp,
                               StmtUndefined, StmtMayBeUndefined);

   -- Enumeration literals representing the different sorts of information-flow
   -- errors related to dependencies that may be reported.
   -- Only the enumeration literals contained within the subtype
   -- DependencyErrType may be used externally.
   type FullDependErrType  is (NotUsed,
                               MayBeUsed,
                               Uninitialised,
                               IneffInit,  --this is own var of embedded pack
                               IneffLocalInit,
                               IntegrityViolation,
                               MayBeIntegrityViolation,
                               NotUsedNew,
                               MayBeUsedNew,
                               NotUsedContinue,
                               MayBeUsedContinue);
   subtype DependencyErrType is
     FullDependErrType range NotUsed .. MayBeIntegrityViolation;


   -- Enumeration literals representing the different sorts of information-flow
   -- errors related to stable conditional expressions that may be reported.
   type StabilityErrType   is (StableExitCond, StableForkCond);

   -- Enumeration literals representing the different sorts of information-flow
   -- errors related to stable expressions in loops that may be reported.
   type IndexType          is (IndexZero, IndexOne, LargerIndex);

   -- Enumeration literals representing the different sorts of information-flow
   -- errors related miscellaneous errors that may be reported.
   type UsageErrType       is (UnusedImport, UndefinedExport, UndefinedVar,
                               UnreferencedVar, RedefinedImport, IneffectiveImport,
                               ReferencedButNotInPartition, UpdatedButNotInPartition,
                               UninitializedProtectedElement);

   type ErrSymRange        is range 1 .. 5;
   --# assert ErrSymRange'Base is Short_Short_Integer; -- For GNAT

   -- An array of parser symbols used in reporting and recovering from
   -- syntax errors (see spparser.adb).
   type ErrSymList         is array (ErrSymRange) of SPSymbols.SPSymbol;

   -- A list of enumeration literals representing each of the suppressible
   -- warnings (and a general literal for notes).
   -- The literals up to and including UnexpectedAddressClauses are considered
   -- to be severe warnings in that they detect uses that may affect the
   -- semantics of the program.
   -- When severe warnings are suppressed in the list of suppressed warnings
   -- issued by the Examiner they are post-fixed with a '*' character.
   -- Non-sever warnings do not have a post-fix character.
   type WarningElements is (Pragmas,
                            HiddenParts,
                            HandlerParts,
                            RepresentationClauses,
                            InterruptHandlers,
                            StaticExpressions,
                            Style_Check_Casing,
                            ExternalVariableAssignment,
                            UncheckedConversion,
                            ImportedObjects,
                            RealRTCs,
                            DefaultLoopAssertions,
                            ExpressionReordering,
                            UnexpectedAddressClauses,
                              -- those below are not "severe"
                            DirectUpdates,
                            WithClauses,
                            UnusedVariables,
                            UnuseablePrivateTypes,
                            ConstantVariables,
                            TypeConversions,
                            SLI_Generation,
                            Index_Manager_Duplicates,
                            DeclareAnnotations,
                            OthersClauses,
                            Ada2005ReservedWords,
                            ObsolescentFeatures,
                            Notes);

   -- "Severe" warnings are for language elements that potentially change the
   -- semantics of a program.  These are indicated with an additional note
   -- in the report file, even if they are suppressed.
   subtype SevereWarnings is
     WarningElements range Pragmas .. UnexpectedAddressClauses;

   type Count is range 0 .. ExaminerConstants.MaxErrorsPerFile;
   --# assert Count'Base is Integer; -- for the Large Examiner

   -- An array of counters, one for eaach type of suppressible warning
   -- This forms part of the error context and is not expexted to be
   -- accessed externally.
   type Counters is array (WarningElements) of Count;

   ------- Types used for justification of expected errors and warnings --------

   -- A justification may be applied to a warning or a flow-error.
   type JustificationKinds is (FlowMessage, WarningMessage);

   -- When a message is marked as expected it is unknown whether it will
   -- eventually have to match it using a LexString or Symbol (because the error
   -- may be added using, say, SemanticWarning or SemanticWarningSym).
   -- Since both forms available in wf_justification_statement
   -- both are stored in the justification table and the right one is picked
   -- when comparisons are needed.
   -- Note that we only have both values for simple identifier like X, if the
   -- item being added is of the form P.Q.X then we can only have, and only
   -- need, the symbol because warnings for objects of this form are always
   -- added with a symbol.
   type JustificationIdentifier is record
      StringForm : LexTokenManager.Lex_String;
      SymbolForm : Dictionary.Symbol;
   end record;


   NullJustificationIdentifier : constant JustificationIdentifier :=
     JustificationIdentifier'(StringForm => LexTokenManager.Null_String,
                              SymbolForm => Dictionary.NullSymbol);

   -- A message that we may want to suppress may have more than one idenfiier
   -- that needs to match (e.g. information flow dependency errors).
   -- There are not any errors that refer to more than two but, in any case
   -- the behaviour is made generic by using an array of identifiers.
   MaxJustificationIdentifierLength : constant := 2;
   subtype JustificationIdentifierIndex is
     Integer range 1 .. MaxJustificationIdentifierLength;
   type JustificationIdentifiers is
     array (JustificationIdentifierIndex) of JustificationIdentifier;
   NullJustificationIdentifiers : constant JustificationIdentifiers :=
     JustificationIdentifiers'(others => NullJustificationIdentifier);

   ---------------- End of Justification type declarations ---------------------

   -- An ErrorContext maintains all the warning and error information related
   -- to a single source or configuration file.
   type ErrorContexts       is private;
   NullErrorContext : constant ErrorContexts;

   -- This constant is used to indicate that the warning or error message
   -- does not refer to any document.
   -- At the moment references are represented as simple Natural numbers
   -- and the list of references is held as a array, ReferenceTable declared in
   -- errorhandler-conversions-tostring-appendreference.adb.
   -- Currently there are 22 reference entries.  It would be better if this
   -- constant and the reference table were made more controlled and abstract.
   NoReference : constant Natural := 0;

   -- TotalErrorCounts used to create summary counts of errors for
   -- screen echo at end of Examination.
   -- Note the special handling of the count of suppressed warnings
   -- which does not form part of the grand total.  This has been done for
   -- consistency with the way error counts in report files are generated.
   -- A total error count is maintained by ErrorHandler and it is not expected
   -- that this declaration will be used externally.
   type CountedErrorKinds is (SyntaxOrSemantic, Flow, Warning);
   type ExplicitErrorCounts is array (CountedErrorKinds) of Count;
   type TotalErrorCounts is record
      GrandTotal         : Count;
      ExplicitErrorCount : ExplicitErrorCounts;
      SuppressedWarnings : Count;
      JustifiedWarnings  : Count;
   end record;
   NullTotalErrorCounts : constant TotalErrorCounts :=
     TotalErrorCounts'(GrandTotal         => Count'First,
                       ExplicitErrorCount => ExplicitErrorCounts'(CountedErrorKinds => Count'First),
                       SuppressedWarnings => Count'First,
                       JustifiedWarnings => Count'First);

   -- Exit codes for the Examiner
   -- 0   = Success with no unjustified errors or warnings
   -- 1   = Unjustified Warnings
   -- 2   = Unjustified Flow Errors
   -- 3   = Syntax/Semantic Errors
   -- 4-7 = Reserved
   -- 8   = Invocation Error e.g. contradictory command-line switches
   -- 9   = Internal Error e.g. table overflow or internal exception
   subtype Exit_Code is Integer range 0 .. 9;

   -- Flushes any pending message text in the EchoAccumulator, which is part of
   -- the ErrorContext maintained by the ErrorHandler, to the standard output.
   -- It should be called on the completion of analysis of each subprogram
   -- implementation.
   procedure FlushEchoMessages;
   --# global in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from *,
   --#                                ErrorContext;

   -- This subporgram initializes the current ErrorContext and is called when
   -- a new source file or configuration file is opened for reading.
   procedure ErrorInit (SourceFileName : in EStrings.T;
                        Echo           : in Boolean);
   --# global in     Dictionary.Dict;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext      from *,
   --#                                Dictionary.Dict,
   --#                                Echo,
   --#                                SourceFileName,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                SourceFileName;

   -- This subprogram is similar to ErrorInit but is for use by SPARKmake
   -- rather than the Examiner.
   procedure SparkMakeInit;
   --# global in     Dictionary.Dict;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext      from *,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dictionary.Dict,
   --#                                ErrorContext;

   -- Obtains the current ErrorContext from the ErrorHandler.
   procedure GetErrorContext (Context : out ErrorContexts);
   --# global in     Dictionary.Dict;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives Context,
   --#         ErrorContext      from ErrorContext,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dictionary.Dict,
   --#                                ErrorContext;

   -- Sets the current ErrorContext of the ErrorHandler to the one supplied
   -- by the parameter Context.
   procedure SetErrorContext (Context : in ErrorContexts);
   --# global in     Dictionary.Dict;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from Context,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                SPARK_IO.File_Sys;

   -- Called prior to analysis of a body or package initialization
   -- to reset the severity property of the ErrorContext as
   -- this property only applies to a single body or package initialization.
   procedure ErrorReset;
   --# global in out ErrorContext;
   --# derives ErrorContext from *;

   -- Returns the severity property from the current ErrorContext.
   procedure GetErrorSeverity (Severity : out ErrorLevel);
   --# global in ErrorContext;
   --# derives Severity from ErrorContext;

   --  This function returns TRUE if a syntax error or a semantic
   --  error has already been met.
   function Syntax_Or_Semantic_Error return Boolean;
   --# global in ErrorContext;

   --  The function returns TRUE if we are still generating SLI.
   function Generate_SLI return Boolean;
   --# global in CommandLineData.Content;
   --#        in ErrorContext;

   -- Called to report successful analysis of a unit.  A successful analysis
   -- may be determined by calling the procedure GetErrorSeverity.
   procedure ReportSuccess (Position   : in LexTokenManager.Token_Position;
                            SubprogStr : in LexTokenManager.Lex_String);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                SPARK_IO.File_Sys,
   --#                                SubprogStr;

   -- Reports a warning that a hidden body has been encountered.
   procedure HiddenText (Position : in LexTokenManager.Token_Position;
                         UnitStr  : in LexTokenManager.Lex_String;
                         UnitTyp  : in SPSymbols.SPSymbol);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                SPARK_IO.File_Sys,
   --#                                UnitStr,
   --#                                UnitTyp;

   -- Reports a warning that a hidden exception handler has been encountered.
   procedure HiddenHandler (Position : in LexTokenManager.Token_Position;
                            UnitStr  : in LexTokenManager.Lex_String;
                            UnitTyp  : in SPSymbols.SPSymbol);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                SPARK_IO.File_Sys,
   --#                                UnitStr,
   --#                                UnitTyp;

   -- Reports a warning that an unrecognised representation clause has been
   -- encountered.
   procedure RepresentationClause (Position : in LexTokenManager.Token_Position);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                SPARK_IO.File_Sys;

   -- Reports a warning that an unrecognised pragma has been encountered.
   procedure APragma (PragmaName : in LexTokenManager.Lex_String;
                      Position   : in LexTokenManager.Token_Position);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                PragmaName,
   --#                                SPARK_IO.File_Sys;

   -- Warns that a cut point has been introduced into a loop by inserting a
   -- default assertion.
   procedure AddCutPoint (AtLine : LexTokenManager.Line_Numbers);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from AtLine,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                SPARK_IO.File_Sys;

   -- This subprogram is called to report a semantic warning involving a single
   -- named entity represented by a LexString.
   -- See the package header documentation for a description of warnings and
   -- their definition.
   procedure SemanticWarning (ErrNum   : in Natural;
                              Position : in LexTokenManager.Token_Position;
                              IdStr    : in LexTokenManager.Lex_String);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                IdStr,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                SPARK_IO.File_Sys;

   --  This subprogram is called to report that we stop the SLI
   --  generation.
   procedure SLI_Generation_Warning (Position : in LexTokenManager.Token_Position;
                                     Id_Str   : in LexTokenManager.Lex_String);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                Id_Str,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                SPARK_IO.File_Sys;

   -- This subprogram is called to report a semantic warning involving a single
   -- named entity represented by a Dictionary.Symbol.
   -- See the package header documentation for a description of warnings and
   -- their definition.
   procedure SemanticWarningSym (ErrNum   : in Natural;
                                 Position : in LexTokenManager.Token_Position;
                                 Sym      : in Dictionary.Symbol;
                                 Scope    : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                Sym;

   -- This subprogram is called to report a note involving a single
   -- Reports a lexical error detected by SPARKlex.
   procedure LexError (ErrorMessage,
                       RecoveryMessage : in     String;
                       ErrorItem       : in     LexTokenManager.Lex_Value);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys     from CommandLineData.Content,
   --#                                    Dictionary.Dict,
   --#                                    ErrorContext,
   --#                                    ErrorItem,
   --#                                    ErrorMessage,
   --#                                    LexTokenManager.State,
   --#                                    RecoveryMessage,
   --#                                    SPARK_IO.File_Sys &
   --#         LexTokenManager.State from *,
   --#                                    ErrorItem,
   --#                                    ErrorMessage,
   --#                                    RecoveryMessage;

   -- Reports an error detected by SPParser.
   procedure SyntaxError (ErrorItem                       : in LexTokenManager.Lex_Value;
                          CurrentSym, EntrySymbol         : in SPSymbols.SPSymbol;
                          NoOfTerminals, NoOfNonTerminals : in SPExpectedSymbols.SPEssSymRange;
                          TerminalList, NonTerminalList   : in SPExpectedSymbols.SPExpSymList);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys     from CommandLineData.Content,
   --#                                    CurrentSym,
   --#                                    Dictionary.Dict,
   --#                                    EntrySymbol,
   --#                                    ErrorContext,
   --#                                    ErrorItem,
   --#                                    LexTokenManager.State,
   --#                                    NonTerminalList,
   --#                                    NoOfNonTerminals,
   --#                                    NoOfTerminals,
   --#                                    SPARK_IO.File_Sys,
   --#                                    TerminalList &
   --#         LexTokenManager.State from *,
   --#                                    CurrentSym,
   --#                                    EntrySymbol,
   --#                                    ErrorItem,
   --#                                    NonTerminalList,
   --#                                    NoOfNonTerminals,
   --#                                    NoOfTerminals,
   --#                                    TerminalList;

   -- Reports on a syntax error recovery action taken by SPParser.
   procedure SyntaxRecovery (RecoveryPosn     : in LexTokenManager.Lex_Value;
                             ReplacementSym   : in SPSymbols.SPSymbol;
                             NextSym          : in SPSymbols.SPTerminal;
                             NoOfSyms         : in Natural;
                             SymList          : in ErrSymList);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys     from CommandLineData.Content,
   --#                                    Dictionary.Dict,
   --#                                    ErrorContext,
   --#                                    LexTokenManager.State,
   --#                                    NextSym,
   --#                                    NoOfSyms,
   --#                                    RecoveryPosn,
   --#                                    ReplacementSym,
   --#                                    SPARK_IO.File_Sys,
   --#                                    SymList &
   --#         LexTokenManager.State from *,
   --#                                    ErrorContext,
   --#                                    NextSym,
   --#                                    NoOfSyms,
   --#                                    RecoveryPosn,
   --#                                    ReplacementSym,
   --#                                    SymList;

   -- named entity represented by a LexString.
   -- See the package header documentation for a description of notes and
   -- their definition.
   procedure SemanticNote (ErrNum   : in Natural;
                           Position : in LexTokenManager.Token_Position;
                           IdStr    : in LexTokenManager.Lex_String);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                IdStr,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                SPARK_IO.File_Sys;

   -- This subprogram is called to report inconsistencies between abstract and
   -- refined dependency relations involving two named entities represented by
   -- LexStrings.
   -- See the package header documentation for a description of these errors and
   -- their definition.
   procedure DepSemanticError (ErrNum   : in Natural;
                               Position : in LexTokenManager.Token_Position;
                               IdStr1   : in LexTokenManager.Lex_String;
                               IdStr2   : in LexTokenManager.Lex_String);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                IdStr1,
   --#                                IdStr2,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                SPARK_IO.File_Sys;

   -- This subprogram is called to report inconsistencies between abstract and
   -- refined dependency relations involving two named entities represented by
   -- Dictionary.Symbols.
   -- See the package header documentation for a description of these errors and
   -- their definition.
   procedure DepSemanticErrorSym (ErrNum   : in Natural;
                                  Position : in LexTokenManager.Token_Position;
                                  Sym1     : in Dictionary.Symbol;
                                  Sym2     : in Dictionary.Symbol;
                                  Scope    : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                Sym1,
   --#                                Sym2;

   -- This subprogram is called to report a semantic error involving a single
   -- named entity represented by a LexString.
   -- The parameter "Reference" facilitates referencing a document within
   -- the warning.  A list of document references is maintained in
   -- errorhandler-conversions-tostring-appendreference.adb.  If a document is
   -- not referenced then the constant ErrorHandler.NoReference should be used
   -- to signify this.
   -- See the package header documentation for a description of semantic errors
   -- and their definition.
   procedure SemanticError (ErrNum    : in Natural;
                            Reference : in Natural;
                            Position  : in LexTokenManager.Token_Position;
                            IdStr     : in LexTokenManager.Lex_String);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                IdStr,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Reference,
   --#                                SPARK_IO.File_Sys;

   -- This subprogram is called to report a semantic error involving two
   -- named entitities represented by LexStrings.
   -- The parameter "Reference" facilitates referencing a document within
   -- the warning.  A list of document references is maintained in
   -- errorhandler-conversions-tostring-appendreference.adb.  If a document is
   -- not referenced then the constant ErrorHandler.NoReference should be used
   -- to signify this.
   -- See the package header documentation for a description of semantic errors
   -- and their definition.
   procedure SemanticError2 (ErrNum    : in Natural;
                             Reference : in Natural;
                             Position  : in LexTokenManager.Token_Position;
                             IdStr1    : in LexTokenManager.Lex_String;
                             IdStr2    : in LexTokenManager.Lex_String);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                IdStr1,
   --#                                IdStr2,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Reference,
   --#                                SPARK_IO.File_Sys;

   -- This subprogram is called to report a semantic error involving a single
   -- named entity represented by a Dictionary.Symbol.
   -- The parameter "Reference" facilitates referencing a document within
   -- the warning.  A list of document references is maintained in
   -- errorhandler-conversions-tostring-appendreference.adb.  If a document is
   -- not referenced then the constant ErrorHandler.NoReference should be used
   -- to signify this.
   -- See the package header documentation for a description of semantic errors
   -- and their definition.
   procedure SemanticErrorSym (ErrNum    : in Natural;
                               Reference : in Natural;
                               Position  : in LexTokenManager.Token_Position;
                               Sym       : in Dictionary.Symbol;
                               Scope     : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Reference,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                Sym;

   -- This subprogram is called to report a semantic error involving two
   -- named entitities represented by Dictionary.Symbols.
   -- The parameter "Reference" facilitates referencing a document within
   -- the warning.  A list of document references is maintained in
   -- errorhandler-conversions-tostring-appendreference.adb.  If a document is
   -- not referenced then the constant ErrorHandler.NoReference should be used
   -- to signify this.
   -- See the package header documentation for a description of semantic errors
   -- and their definition.
   procedure SemanticErrorSym2 (ErrNum    : in Natural;
                                Reference : in Natural;
                                Position  : in LexTokenManager.Token_Position;
                                Sym       : in Dictionary.Symbol;
                                Sym2      : in Dictionary.Symbol;
                                Scope     : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Reference,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                Sym,
   --#                                Sym2;

   -- This subprogram is called to report a semantic error involving two
   -- named entitities represented one by a LexString and one by a
   -- Dictionary.Symbol.
   -- The parameter "Reference" facilitates referencing a document within
   -- the warning.  A list of document references is maintained in
   -- errorhandler-conversions-tostring-appendreference.adb.  If a document is
   -- not referenced then the constant ErrorHandler.NoReference should be used
   -- to signify this.
   -- See the package header documentation for a description of semantic errors
   -- and their definition.
   procedure SemanticErrorLex1Sym1 (ErrNum    : in Natural;
                                    Reference : in Natural;
                                    Position  : in LexTokenManager.Token_Position;
                                    IdStr     : in LexTokenManager.Lex_String;
                                    Sym       : in Dictionary.Symbol;
                                    Scope     : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                IdStr,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Reference,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                Sym;

   -- This subprogram is called to report a semantic error involving three
   -- named entitities represented one by a LexString and two by
   -- Dictionary.Symbols.
   -- The parameter "Reference" facilitates referencing a document within
   -- the warning.  A list of document references is maintained in
   -- errorhandler-conversions-tostring-appendreference.adb.  If a document is
   -- not referenced then the constant ErrorHandler.NoReference should be used
   -- to signify this.
   -- See the package header documentation for a description of semantic errors
   -- and their definition.
   procedure SemanticErrorLex1Sym2 (ErrNum    : in Natural;
                                    Reference : in Natural;
                                    Position  : in LexTokenManager.Token_Position;
                                    IdStr     : in LexTokenManager.Lex_String;
                                    Sym       : in Dictionary.Symbol;
                                    Sym2      : in Dictionary.Symbol;
                                    Scope     : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                IdStr,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Reference,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                Sym,
   --#                                Sym2;

   -- This subprogram is called to report a semantic error involving three
   -- named entitities represented by Dictionary.Symbols.
   -- The parameter "Reference" facilitates referencing a document within
   -- the warning.  A list of document references is maintained in
   -- errorhandler-conversions-tostring-appendreference.adb.  If a document is
   -- not referenced then the constant ErrorHandler.NoReference should be used
   -- to signify this.
   -- See the package header documentation for a description of semantic errors
   -- and their definition.
   procedure SemanticErrorSym3 (ErrNum    : in Natural;
                                Reference : in Natural;
                                Position  : in LexTokenManager.Token_Position;
                                Sym       : in Dictionary.Symbol;
                                Sym2      : in Dictionary.Symbol;
                                Sym3      : in Dictionary.Symbol;
                                Scope     : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrNum,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Reference,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                Sym,
   --#                                Sym2,
   --#                                Sym3;

   -- This subprogram is called to report a control-flow error.
   -- See the package header documentation for a description of control-flow
   -- errors and their definition.
   procedure ControlFlowError (ErrType  : in ControlFlowErrType;
                               Position : in LexTokenManager.Token_Position);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                ErrType,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                SPARK_IO.File_Sys;

   -- This subprogram is called to report a data-flow error involving a single
   -- named entity represented by a Dictionary.Symbol.
   -- See the package header documentation for a description of data-flow errors
   -- and their definition.
   procedure DataFlowError (ErrType  : in DataFlowErrType;
                            Position : in LexTokenManager.Token_Position;
                            VarSym   : in Dictionary.Symbol;
                            Scope    : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                ErrType,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                VarSym;

   -- This subprogram is called to report an ineffective statement involving a
   -- single named entity represented by a Dictionary.Symbol.
   procedure IneffectiveStmt (Position : in LexTokenManager.Token_Position;
                              VarSym   : in Dictionary.Symbol;
                              Scope    : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                VarSym;

   -- This subprogram is called to report a stable conditional or a stable
   -- expression in a loop.
   -- See the package header documentation for a description of information-flow
   -- errors and their definition.
   procedure StabilityError (ErrType        : in StabilityErrType;
                             Position       : in LexTokenManager.Token_Position;
                             StabilityIndex : in IndexType);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                ErrType,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                SPARK_IO.File_Sys,
   --#                                StabilityIndex;

   -- This subprogram is called to report a dependency error involving two
   -- named entitities, the imported and exported variables, represented by
   -- Dictionary.Symbols.
   -- See the package header documentation for a description of information-flow
   -- errors and their definition.
   procedure DependencyError (ErrType      : in DependencyErrType;
                              Position     : in LexTokenManager.Token_Position;
                              ImportVarSym : in Dictionary.Symbol;
                              ExportVarSym : in Dictionary.Symbol;
                              Scope        : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                ErrType,
   --#                                ExportVarSym,
   --#                                ImportVarSym,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys;

   -- This subprogram is called to report a usage error involving a single
   -- named entitity represented by a Dictionary.Symbol.
   -- See the package header documentation for a description of information-flow
   -- errors and their definition.
   procedure UsageError (Errtype  : in UsageErrType;
                         Position : in LexTokenManager.Token_Position;
                         VarSym   : in Dictionary.Symbol;
                         Scope    : in Dictionary.Scopes);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                ErrType,
   --#                                LexTokenManager.State,
   --#                                Position,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                VarSym;

   -- ReadWarningFile reads in and parses the warning control file specified on
   -- the command line.  It sets up an internal store (part of the state
   -- represented by the own variable ErrorContext) which records which warnings
   -- have been suppressed by the inclusion of their keyword in the warning file.
   -- See the package header documentation for a description of the use of the
   -- warning file and the definition of suppressible warnings.
   procedure ReadWarningFile;
   --# global in     CommandLineData.Content;
   --#        in out ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         LexTokenManager.State,
   --#         SPARK_IO.File_Sys     from CommandLineData.Content,
   --#                                    ErrorContext,
   --#                                    LexTokenManager.State,
   --#                                    SPARK_IO.File_Sys;

   -- OutputWarningList writes out a list of the warnings to the file parameter
   -- "ToFile" that are currently suppressed based on the internal store of
   -- warnings suppressed by reading the warning control file (part of the state
   -- represented by the own variable ErrorContext).
   -- See the package header documentation for a description of the use of the
   -- warning file and the definition of suppressible warnings.
   procedure OutputWarningList (ToFile : in SPARK_IO.File_Type);
   --# global in     CommandLineData.Content;
   --#        in     ErrorContext;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                ToFile,
   --#                                XMLReport.State &
   --#         XMLReport.State   from *,
   --#                                CommandLineData.Content,
   --#                                ErrorContext;

   -- OutputReferenceList writes out a list of the document references that are
   -- associated with reported semantic erros to the file parameter "ToFile"
   -- See the package header documentation for a description of semantic errors
   -- and document references.
   procedure OutputReferenceList (ToFile : in SPARK_IO.File_Type);
   --# global in     CommandLineData.Content;
   --#        in     ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                ErrorContext,
   --#                                ToFile;

   -- This subprogram writes a source listing to the file parameter "Listing"
   -- in which the accumulated reported warnings and errors are interspersed
   -- with program text to indicate the location of the warning or error.
   -- A table showing the successful justifications encountered in the source
   -- file and a summary of the suppressed warnings are appended to the end of
   -- the listing file.
   -- The parameter "Purpose" identifies the type of listing that is being
   -- generated and is used in conjunction with the -error_explanations command
   -- line switch to control the generation of extended messages.
   -- For this subprogram, given its use, it is likely that Purpose
   -- will either be ForListing - indicating this is a sorce listing or
   -- ForReportIndexedFiles used when listing the configuration file.
   procedure PrintErrors (Listing : in SPARK_IO.File_Type;
                          Purpose : in Error_Types.ConversionRequestSource);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Listing,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys;

   -- AppendErrors extends the report file passed as the parameter "Report"
   -- adding the accumulated errors from the current ErrorContext.
   -- A table showing the successful justifications encountered within the
   -- current ErrorContext and a summary of the suppressed warnings are
   -- added after the reported warnings and errors.
   -- The parameter "Purpose" identifies the type of listing that is being
   -- generated and is used in conjunction with the -error_explanations command
   -- line switch to control the generation of extended messages.
   -- For this subprogram, given its use, it is likely that Purpose
   -- will either be ForReportSelectedFiles - indicating this is a file that
   -- is explicitly given on the command line or in a meta file, or
   -- ForReportIndexedFiles - indicating that this is a file which was accessed
   -- indirectly via an index file..
   procedure AppendErrors (Report  : in SPARK_IO.File_Type;
                           Purpose : in Error_Types.ConversionRequestSource);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys,
   --#         XMLReport.State   from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                Purpose,
   --#                                Report,
   --#                                SPARK_IO.File_Sys,
   --#                                XMLReport.State;

   --  This procedure write at the standard output an error in an
   --  index file.
   procedure Index_Manager_Error (S                 : in String;
                                  Source_File       : in LexTokenManager.Lex_String;
                                  Line_No           : in Positive;
                                  Col_No            : in Positive;
                                  Token_String      : in EStrings.T;
                                  Is_Token_Filename : in Boolean;
                                  Is_Fatal          : in Boolean);
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Col_No,
   --#                                CommandLineData.Content,
   --#                                ErrorContext,
   --#                                Is_Fatal,
   --#                                Is_Token_Filename,
   --#                                LexTokenManager.State,
   --#                                Line_No,
   --#                                S,
   --#                                Source_File,
   --#                                Token_String &
   --#         ErrorContext      from *,
   --#                                Is_Fatal;

   -- When called writes the total warnings and errors encountered during the
   -- Examiner run to the standard output.
   procedure EchoTotalErrorCount;
   --# global in     CommandLineData.Content;
   --#        in     ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                ErrorContext;

   -- Call this subprogram if it is not possible to open a required input file
   -- during analysis.  It will record the fact that this has happened and
   -- it will be reflected in the exit code returned by a call to GetErrorsType.
   procedure SetFileOpenError;
   --# global in out ErrorContext;
   --# derives ErrorContext from *;

   -- At the end of an Examiner run call this subprogram to obtain an exit code
   -- reflecting the most severe type of error encountered during the run.
   -- The exit code so obtained can then be used in a call to Set_Exit_Status.
   function GetErrorsType return Exit_Code;
   --# global in ErrorContext;


   ------ Exported calls concerned with the error justification mechanism ------

   -- Called at the start of analysis of each unit to initialize the
   -- justification state for the unit.
   -- The justification state is part of the state represented by the own
   -- variable ErrorContext.
   procedure StartUnit;
   --# global in out ErrorContext;
   --# derives ErrorContext from *;

   -- Called at the end of analysis of each unit to close the
   -- justification state for the unit.
   -- The justification state is part of the state represented by the own
   -- variable ErrorContext.
   procedure EndUnit;
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContext,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorContext,
   --#                                LexTokenManager.State,
   --#                                SPARK_IO.File_Sys;

   -- Called when an accept annotation is encountered in the source text to
   -- register the justification.
   -- If the maximum allowed justifications within a unit has been reached when
   -- registering the justification the parameter MaximumJustificationsReached
   -- is True but justification is registered.
   -- The MaximumJustificationsReached is only ever set True once, when the
   -- table first fills up.  If the table is already full then False is returned.
      -- at the point of call where the table first fills, not at every call thereafter.
   procedure StartJustification (Position                     : in     LexTokenManager.Token_Position;
                                 Line                         : in     LexTokenManager.Line_Numbers;
                                 Kind                         : in     JustificationKinds;
                                 ErrNum                       : in     Natural;
                                 Identifiers                  : in     JustificationIdentifiers;
                                 Explanation                  : in     LexTokenManager.Lex_String;
                                 MaximumJustificationsReached :    out Boolean);
   --# global in out ErrorContext;
   --# derives ErrorContext                 from *,
   --#                                           ErrNum,
   --#                                           Explanation,
   --#                                           Identifiers,
   --#                                           Kind,
   --#                                           Line,
   --#                                           Position &
   --#         MaximumJustificationsReached from ErrorContext;

   -- Called when an end accept annotation is encountered in the source text to
   -- end the scope of a justification.
   -- The parameter UnmatchedEnd returns True if there is no accept annotation
   -- matching the end accept annotation provided the Justification table
   -- is not full in which case it will always return False to avoid false
   -- alarms.
   procedure EndJustification (Line         : in     LexTokenManager.Line_Numbers;
                               UnmatchedEnd :    out Boolean);
   --# global in out ErrorContext;
   --# derives ErrorContext,
   --#         UnmatchedEnd from ErrorContext,
   --#                           Line;

private --------------------------------------------------------------

   -- The following declarations define a Data Table which is a core component
   -- of the Justification structure.
   -- A DataTableTableEntry defines a single element of the table and contains
   -- all the information associated with a single justification.
   -- DataTables defines the array of DataTableEntries which holds all the
   -- justifications for a unit.
   MaxTableEntries : constant := ExaminerConstants.MaxJustificationsPerFile;
   type DataTablePtr is range 0 .. MaxTableEntries;
   type DataTableEntry is record
      Kind             : JustificationKinds;
      ErrNum           : Natural;
      Identifiers      : JustificationIdentifiers;
      Explanation      : LexTokenManager.Lex_String;
      -- location of the justification clause for error reporting purposes
      Position         : LexTokenManager.Token_Position;
      -- location of start justify (for multiple clauses this is the line of the justify statement as a whole)
      StartLine        : LexTokenManager.Line_Numbers;
      -- location of end justify or end of unit (const EndLineOfUnitMarker) if there is no matching end justify
      EndLine          : LexTokenManager.Line_Numbers;
      EndFound         : Boolean;   -- explicit end justify found or not
      MatchCount       : Natural;   -- number of times this justification matched
      MatchLine        : LexTokenManager.Line_Numbers; -- last line where match occurred
                                                      -- linkage
      Previous         : DataTablePtr;
   end record;
   -- Initializing constants for Data Tables
   EndOfList : constant DataTablePtr := 0;
   EmptyDataTableEntry : constant DataTableEntry := DataTableEntry'
     (Kind => FlowMessage,
      ErrNum => 0,
      Identifiers => NullJustificationIdentifiers,
      Explanation => LexTokenManager.Null_String,
      Position => LexTokenManager.Token_Position'(Start_Line_No => 0,
                                                  Start_Pos     => 0),
      StartLine => 0,
      EndLine  => 0,
      EndFound  => False,
      MatchCount => 0,
      MatchLine => 0,
      Previous => EndOfList);

   -- DataTables defines the array of DataTableEntries representing all the
   -- justifications for a unit.
   subtype DataTableIndex is DataTablePtr range 1 .. MaxTableEntries;
   type DataTables is array (DataTableIndex) of DataTableEntry;
   -- Initializing constant
   EmptyDataTable : constant DataTables :=
     DataTables'(DataTableIndex => EmptyDataTableEntry);

   -- The Stack is the second major component of the justification structure
   -- A StackRecord is a single element of the stack, UnitStackArray is the
   -- stack storage and UnitStacks is the complete stack including a stack
   -- pointer.
   MaxStackSize : constant := 10; -- Perhaps move this to ExaminerConstants later
   type StackPtrs is range 0 .. MaxStackSize;
   subtype StackIndex is StackPtrs range 1 .. MaxStackSize;
   type StackRecord is record
      ListItems : DataTablePtr;
      SemanticErrorInUnit : Boolean;
   end record;
   type UnitStackArray is array (StackIndex) of StackRecord;
   type UnitStacks is record
      Vector : UnitStackArray;
      Ptr    : StackPtrs;
   end record;
   -- initializing constant
   EmptyStack : constant UnitStacks := UnitStacks'
     (Ptr => 0,
      Vector => UnitStackArray'
        (StackIndex => StackRecord'(ListItems => EndOfList,
                                    SemanticErrorInUnit => False)));

   -- The justification structure including a DataTable, a UnitStack, a pointer
   -- for the current entry in the DataTable and a Boolean indicating whether
   -- the justification table is accepting more entries (is not full).
   type JustificationsDataTables is record
      DataTable : DataTables;
      CurrentSlot : DataTablePtr;
      AcceptingMoreEntries : Boolean;
      UnitStack   : UnitStacks;
   end record;
   -- initializing constant
   EmptyJustificationsDataTable : constant JustificationsDataTables :=
     JustificationsDataTables'(DataTable => EmptyDataTable,
                               CurrentSlot => EndOfList,
                               AcceptingMoreEntries => True,
                               UnitStack => EmptyStack);
   --------- End of justification table data structure definitions -------------

   -- We can now declare the actual announced private type
   -- The private type ErrorContexts contains a JustificationTable which is
   -- itself a complex data structure used to track the justification of
   -- expected flow errors and warnings.
   -- It is part of ErrorContexts because we need a different copy of the data
   -- structure for each file we are examining.
   -- The data structure types can be declared here because they are only used
   -- in the body of ErrorHandler and in the embedded package Justifications.
   --
   -- More details of the rationale for the Justifications Table data structure
   -- can be found in ErrorHandler.Justifications.adb
   --
   type ErrorContexts is
      record
         RecoveryMessages : Boolean;
         Echo             : Boolean;
         Severity         : ErrorLevel;
         NumErrs          : Count;
         NumMessage       : Count;
         LineNo           : LexTokenManager.Line_Numbers;
         CurrentLine      : EStrings.T;
         Errs             : Error_IO.File_Type;
         Source           : SPARK_IO.File_Type;
         SourceFileName   : EStrings.T;
         Counter          : Counters;
         JustificationsDataTable : JustificationsDataTables;
      end record;

   NullErrorContext : constant ErrorContexts :=
     ErrorContexts'(RecoveryMessages => False,
                    Echo             => False,
                    Severity         => ErrorLevel'First,
                    NumErrs          => Count'First,
                    NumMessage       => Count'First,
                    LineNo           => LexTokenManager.Line_Numbers'First,
                    CurrentLine      => EStrings.Empty_String,
                    Errs             => Error_IO.Null_File,
                    Source           => SPARK_IO.Null_File,
                    SourceFileName   => EStrings.Empty_String,
                    Counter          => Counters'(others => Count'First),
                    JustificationsDataTable => EmptyJustificationsDataTable);
end ErrorHandler;
