-- $Id: errorhandler-warningstatus.adb 15674 2010-01-20 16:17:20Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

separate (ErrorHandler)
package body WarningStatus
--# own SuppressionList is SuppressedElement,
--#                        PragmaList,
--#                        SuppressAllPragmas,
--#                        SomethingSuppressed;
is

   subtype PragmaCounts is Integer range
      0 .. ExaminerConstants.MaxPragmasInWarningFile;
   subtype PragmaIndex  is Integer range
      1 .. ExaminerConstants.MaxPragmasInWarningFile;

   type    PragmaArrays is array (PragmaIndex) of LexTokenManager.Lex_String;
   type    PragmaLists  is record
      PragmaArray : PragmaArrays;
      PragmaCount : PragmaCounts;
   end record;

   type SuppressedElementArray is array (ErrorHandler.WarningElements) of Boolean;


   SuppressedElement     : SuppressedElementArray;
   PragmaList            : PragmaLists;
   SomethingSuppressed   : Boolean;
   SuppressAllPragmas    : Boolean;

   ---------------------------------------------------------------------------

   function PragmaFound (Pragma_Name : LexTokenManager.Lex_String)
                        return Boolean
   --# global in LexTokenManager.State;
   --#        in PragmaList;
   is
      Look_At,
      Left,
      Right     : Integer;
      Found     : Boolean;
      Match_Res : LexTokenManager.Str_Comp_Result;

      function Match_Check (Pos : Integer) return LexTokenManager.Str_Comp_Result
      --# global in LexTokenManager.State;
      --#        in PragmaList;
      --#        in Pragma_Name;
      is
      begin
         return LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => PragmaList.PragmaArray (Pos),
            Lex_Str2 => Pragma_Name);
      end Match_Check;

   begin
      Left := 0;
      Right := PragmaList.PragmaCount + 1;
      Found := False;

      loop
         exit when (Left + 1) = Right;
         Look_At := (Left + Right) / 2;

         Match_Res := Match_Check (Look_At);

         if Match_Res = LexTokenManager.Str_Eq then
            Found := True;
            exit;
         end if;

         if Match_Res = LexTokenManager.Str_First then
            Left := Look_At;
         else
            Right := Look_At;
         end if;
      end loop;

      return Found;
   end PragmaFound;

   ---------------------------------------------------------------------------

   procedure ReadWarningFile
   --# global in     CommandLineData.Content;
   --#        in out ErrorHandler.FileOpenError;
   --#        in out LexTokenManager.State;
   --#        in out PragmaList;
   --#        in out SomethingSuppressed;
   --#        in out SPARK_IO.File_Sys;
   --#        in out SuppressAllPragmas;
   --#        in out SuppressedElement;
   --# derives ErrorHandler.FileOpenError from *,
   --#                                         CommandLineData.Content,
   --#                                         SPARK_IO.File_Sys &
   --#         LexTokenManager.State,
   --#         SPARK_IO.File_Sys,
   --#         SuppressedElement          from *,
   --#                                         CommandLineData.Content,
   --#                                         LexTokenManager.State,
   --#                                         PragmaList,
   --#                                         SPARK_IO.File_Sys &
   --#         PragmaList,
   --#         SuppressAllPragmas         from CommandLineData.Content,
   --#                                         LexTokenManager.State,
   --#                                         PragmaList,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SuppressAllPragmas &
   --#         SomethingSuppressed        from *,
   --#                                         CommandLineData.Content,
   --#                                         LexTokenManager.State,
   --#                                         PragmaList,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SuppressedElement;
      is separate;

   ---------------------------------------------------------------------------

   function IsSuppressed (TheElement : ErrorHandler.WarningElements) return Boolean
   --# global in SuppressedElement;
   is
   begin
      return SuppressedElement (TheElement);
   end IsSuppressed;

   ---------------------------------------------------------------------------

   function PragmaIsSuppressed (PragmaName : LexTokenManager.Lex_String)
                               return Boolean
   --# global in LexTokenManager.State;
   --#        in PragmaList;
   --#        in SuppressAllPragmas;
   is
      Result : Boolean;
   begin
      Result := SuppressAllPragmas;
      if not Result then
         Result := PragmaFound (PragmaName);
      end if;
      return Result;
   end PragmaIsSuppressed;

   ---------------------------------------------------------------------------

   function GetDescription (Item : in ErrorHandler.WarningElements;
                            P1   : in String;
                            P2   : in String) return EStrings.T
   is
      TmpString,
      Result     : EStrings.T;
      Posn       : EStrings.Lengths;
      AlwaysTrue : Boolean;
   begin
      -- The messages are parmeterised for different message formats.
      -- The substring %%1 is substituted with the string P1 and
      -- the substring %%2 is substituted with the string P2.
      -- Some message formats require a simple "s character to represent a
      -- plural whereas others require the string "(s)".  In one case in one
      -- format has a plural whereas another does not.
      -- This complex scheme is to maintain compatibility with an earlier
      -- message scheme.
      case Item is
         when ErrorHandler.Pragmas               =>
            TmpString := EStrings.Copy_String (Str => "Pragma%%1");
         when ErrorHandler.HiddenParts           =>
            TmpString := EStrings.Copy_String (Str => "Hidden part%%1");
         when ErrorHandler.HandlerParts           =>
            TmpString := EStrings.Copy_String (Str => "Hidden exception handler part%%1");
         when ErrorHandler.RepresentationClauses =>
            TmpString := EStrings.Copy_String (Str => "Representation clause%%1");
         when ErrorHandler.DirectUpdates         =>
            TmpString := EStrings.Copy_String (Str => "Direct update%%1 of own variable(s) of non-enclosing package%%1");
         when ErrorHandler.WithClauses           =>
            TmpString := EStrings.Copy_String (Str => "With clause%%1 lacking a supporting inherit");
         when ErrorHandler.StaticExpressions     =>
            TmpString := EStrings.Copy_String (Str => "Static expression%%1 too complex for Examiner");
         when ErrorHandler.Style_Check_Casing    =>
            TmpString := EStrings.Copy_String (Str => "Style check casing");
         when ErrorHandler.UnusedVariables       =>
            TmpString := EStrings.Copy_String (Str => "Variable%%1 declared but not used");
         when ErrorHandler.ConstantVariables     =>
            TmpString := EStrings.Copy_String (Str => "Variable%%1 used as constants");
         when ErrorHandler.TypeConversions       =>
            TmpString := EStrings.Copy_String (Str => "Unnecessary type conversion%%1");
         when ErrorHandler.SLI_Generation        =>
            TmpString := EStrings.Copy_String (Str => "Stop SLI generation");
         when ErrorHandler.Index_Manager_Duplicates =>
            TmpString := EStrings.Copy_String (Str => "Duplicate entry in index files");
         when ErrorHandler.OthersClauses         =>
            TmpString := EStrings.Copy_String (Str => "Unnecessary others clause%%1");
         when ErrorHandler.ImportedObjects       =>
            TmpString := EStrings.Copy_String (Str => "Use%%1 of pragma Import on objects");
         when ErrorHandler.UnexpectedAddressClauses     =>
            TmpString := EStrings.Copy_String (Str => "Unexpected address clause%%1");
         when ErrorHandler.ExpressionReordering     =>
            TmpString := EStrings.Copy_String (Str => "Reordering of expressions");
         when ErrorHandler.Notes     =>
            TmpString := EStrings.Copy_String (Str => "Note%%1");
         when ErrorHandler.UnuseablePrivateTypes     =>
            TmpString := EStrings.Copy_String (Str => "Private type%%1 lacking method of initialization");
         when ErrorHandler.ExternalVariableAssignment     =>
            TmpString := EStrings.Copy_String (Str => "Assignment%%2 or return%%2 of external variables");
         when ErrorHandler.DeclareAnnotations     =>
            TmpString := EStrings.Copy_String (Str => "Declare annotations in non Ravenscar programs");
         when ErrorHandler.InterruptHandlers     =>
            TmpString := EStrings.Copy_String (Str => "Protected objects that include interrupt handlers");
         when ErrorHandler.UncheckedConversion     =>
            TmpString := EStrings.Copy_String (Str => "Use%%1 of instantiations of Unchecked_Conversion");
         when ErrorHandler.Ada2005ReservedWords =>
            TmpString := EStrings.Copy_String (Str => "Use%%1 of Ada2005 reserved words");
         when ErrorHandler.ObsolescentFeatures =>
            TmpString := EStrings.Copy_String (Str => "Use%%1 of obsolete feature from Ada83 in SPARK 95 mode");
         when ErrorHandler.DefaultLoopAssertions =>
            TmpString := EStrings.Copy_String (Str => "Generation of default loop assertions");
         when ErrorHandler.RealRTCs =>
            TmpString := EStrings.Copy_String (Str => "Generation of RTCs on real numbers");
      end case;

      Posn := 0;
      Result := EStrings.Empty_String;
      while Posn < EStrings.Get_Length (E_Str => TmpString) loop
         Posn := Posn + 1;
         if Posn + 2 <= EStrings.Get_Length (E_Str => TmpString) and then
           EStrings.Get_Element (E_Str => TmpString,
                                 Pos   => Posn) = '%' and then
           EStrings.Get_Element (E_Str => TmpString,
                                 Pos   => Posn + 1) = '%' and then
           (EStrings.Get_Element (E_Str => TmpString,
                                  Pos   => Posn + 2) = '1' or else
              EStrings.Get_Element (E_Str => TmpString,
                                    Pos   => Posn + 2) = '2') then
            if EStrings.Get_Element (E_Str => TmpString,
                                     Pos   => Posn + 2) = '1' then
               EStrings.Append_String (E_Str => Result,
                                       Str   => P1);
            else
               EStrings.Append_String (E_Str => Result,
                                       Str   => P2);
            end if;
            Posn := Posn + 2;
         else
            --# accept F, 10, AlwaysTrue,
            --#        "The loop conditions ensure that the cheracter will always be appended";
            EStrings.Append_Char (E_Str   => Result,
                                  Ch      => EStrings.Get_Element (E_Str => TmpString,
                                                                   Pos   => Posn),
                                  Success => AlwaysTrue);
            --# end accept;
         end if;
      end loop;

      --# accept F, 33, AlwaysTrue,
      --#        "The loop constraints ensure result of EStrings.AppendChar is always True.";
      return Result;
   end GetDescription;


   ---------------------------------------------------------------------------

   procedure OutputWarningList (ToFile : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     PragmaList;
   --#        in     SomethingSuppressed;
   --#        in     SuppressAllPragmas;
   --#        in     SuppressedElement;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                LexTokenManager.State,
   --#                                PragmaList,
   --#                                SomethingSuppressed,
   --#                                SuppressAllPragmas,
   --#                                SuppressedElement,
   --#                                ToFile,
   --#                                XMLReport.State &
   --#         XMLReport.State   from *,
   --#                                CommandLineData.Content,
   --#                                PragmaList,
   --#                                SomethingSuppressed,
   --#                                SuppressAllPragmas,
   --#                                SuppressedElement;
   is
      Description : EStrings.T;

      procedure PutPragmas
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.State;
      --#        in     PragmaList;
      --#        in     SuppressAllPragmas;
      --#        in     SuppressedElement;
      --#        in     ToFile;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                LexTokenManager.State,
      --#                                PragmaList,
      --#                                SuppressAllPragmas,
      --#                                SuppressedElement,
      --#                                ToFile,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                CommandLineData.Content,
      --#                                PragmaList,
      --#                                SuppressAllPragmas,
      --#                                SuppressedElement;
      is
         Wrap       : constant Integer := 72;
         Margin     : constant Integer := 14;
         Column     : Integer;
         Punct      : Character;
         Pragma_Str : EStrings.T;

         procedure PutPragmaName (Str : LexTokenManager.Lex_String)
         --# global in     LexTokenManager.State;
         --#        in     ToFile;
         --#        in out Column;
         --#        in out Punct;
         --#        in out SPARK_IO.File_Sys;
         --# derives Column            from *,
         --#                                LexTokenManager.State,
         --#                                Str &
         --#         Punct             from  &
         --#         SPARK_IO.File_Sys from *,
         --#                                Column,
         --#                                LexTokenManager.State,
         --#                                Punct,
         --#                                Str,
         --#                                ToFile;
         is
            Result : EStrings.T;
         begin
            Result := LexTokenManager.Lex_String_To_String (Lex_Str => Str);
            SPARK_IO.Put_Char (ToFile, Punct);
            SPARK_IO.Put_Char (ToFile, ' ');
            Punct := ',';
            Column := Column + 2;
            if Column + EStrings.Get_Length (E_Str => Result) > Wrap then
               SPARK_IO.New_Line (ToFile, 1);
               SPARK_IO.Put_String (ToFile, "            ", 0);
               Column := Margin;
            end if;
            EStrings.Put_String (File  => ToFile,
                                 E_Str => Result);
            Column := Column + EStrings.Get_Length (E_Str => Result);
         end PutPragmaName;

         function GetPragmaName (Str : LexTokenManager.Lex_String) return EStrings.T
         --# global in LexTokenManager.State;
         is
         begin
            return LexTokenManager.Lex_String_To_String (Lex_Str => Str);
         end GetPragmaName;


      begin --PutPragmas
         if CommandLineData.Content.XML then
            if SuppressAllPragmas then
               Pragma_Str := EStrings.Copy_String (Str => "all");
               XMLReport.SuppressedPragma (Item => Pragma_Str);
               EStrings.Put_String (File  => ToFile,
                                    E_Str => Pragma_Str);
            elsif SuppressedElement (ErrorHandler.Pragmas) then
               for I in Integer range 1 .. PragmaList.PragmaCount loop
                  Pragma_Str := GetPragmaName (PragmaList.PragmaArray (I));
                  XMLReport.SuppressedPragma (Item => Pragma_Str);
                  EStrings.Put_String (File  => ToFile,
                                       E_Str => Pragma_Str);
               end loop;
            end if;
         else
            if SuppressAllPragmas then
               SPARK_IO.Put_Line (ToFile, "   All pragmas", 0);
            elsif SuppressedElement (ErrorHandler.Pragmas) then
               Column := Margin;
               Punct := ':';
               SPARK_IO.Put_String (File => ToFile,
                                    Item => "   ",
                                    Stop => 0);
               EStrings.Put_String (File  => ToFile,
                                    E_Str => GetDescription (Item => ErrorHandler.Pragmas,
                                                             P1   => "s",
                                                             P2   => ""));
               for I in Integer range 1 .. PragmaList.PragmaCount loop
                  PutPragmaName (PragmaList.PragmaArray (I));
               end loop;
            end if;
            SPARK_IO.New_Line (ToFile, 1);
         end if;
      end PutPragmas;

   begin  --OutputWarningList
      if CommandLineData.Content.XML then
         XMLReport.StartSection (XMLReport.SWarningsConfig,
                                 ToFile);
         if SomethingSuppressed then
            for I in ErrorHandler.WarningElements range
              ErrorHandler.HiddenParts .. ErrorHandler.WarningElements'Last
            loop
               if SuppressedElement (I) then
                  Description := GetDescription (Item => I,
                                                 P1   => "s",
                                                 P2   => "");
                  XMLReport.Suppressed (Item => Description);
                  EStrings.Put_String (File  => ToFile,
                                       E_Str => Description);
               end if;
            end loop;
            PutPragmas;
         end if;
         XMLReport.EndSection (XMLReport.SWarningsConfig,
                                 ToFile);
      else
         SPARK_IO.New_Line (ToFile, 2);
         if SomethingSuppressed then
            SPARK_IO.Put_Line (ToFile, "Summary warning reporting selected for:", 0);
            for I in ErrorHandler.WarningElements range
              ErrorHandler.HiddenParts .. ErrorHandler.WarningElements'Last
            loop
               if SuppressedElement (I) then
                  SPARK_IO.Put_String (File => ToFile,
                                       Item => "   ",
                                       Stop => 0);
                  EStrings.Put_Line (File  => ToFile,
                                     E_Str => GetDescription (Item => I,
                                                              P1   => "s",
                                                              P2   => ""));
               end if;
            end loop;
            PutPragmas;
         else
            SPARK_IO.Put_Line (ToFile, "Full warning reporting selected", 0);
         end if;

      end if;

   end OutputWarningList;

   ---------------------------------------------------------------------------

   procedure ReportSuppressedWarnings (ToFile  : in SPARK_IO.File_Type;
                                       Counter : ErrorHandler.Counters)
   --# global in     SomethingSuppressed;
   --#        in     SuppressedElement;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Counter,
   --#                                SomethingSuppressed,
   --#                                SuppressedElement,
   --#                                ToFile;
   is
      indent : constant Integer := 6;
      TotalWarnings : Integer;
      SevereWarning : Boolean := False;
      TmpString : EStrings.T;

      procedure PutCount (Count : in Integer;
                          Width : in Integer)
      --# global in     ToFile;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Count,
      --#                                ToFile,
      --#                                Width;
      is
      begin
         SPARK_IO.Put_Integer (ToFile, Count, Width, 10);
      end PutCount;

   begin  --ReportSuppressedWarnings
      if SomethingSuppressed then

         TotalWarnings := 0;
         for I in ErrorHandler.WarningElements loop
            TotalWarnings := TotalWarnings + Integer (Counter (I));
         end loop;

         if TotalWarnings = 0 then
            SPARK_IO.Put_Line (ToFile, "No summarized warnings", 0);
            SPARK_IO.New_Line (ToFile, 1);
         else
            PutCount (TotalWarnings, 0);
            SPARK_IO.Put_Line (ToFile, " summarized warning(s), comprising:", 0);
            for I in ErrorHandler.WarningElements loop
               if SuppressedElement (I) and then Counter (I) > 0 then
                  PutCount (Integer (Counter (I)), indent);
                  SPARK_IO.Put_Char (File => ToFile, Item => ' ');
                  TmpString := GetDescription (Item => I,
                                               P1   => "(s)",
                                               P2   => "(s)");
                  if EStrings.Get_Length (E_Str => TmpString) > 0 then
                     TmpString := EStrings.Lower_Case_Char (E_Str => TmpString,
                                                            Pos   => 1);
                  end if;

                  EStrings.Put_String (File  => ToFile,
                                       E_Str => TmpString);

                  if I in ErrorHandler.SevereWarnings then
                     SPARK_IO.Put_Char (ToFile, '*');
                     SevereWarning := True;
                  end if;
                  SPARK_IO.New_Line (ToFile, 1);
               end if;
            end loop;
            if SevereWarning then
               SPARK_IO.Put_Line
                  (ToFile,
                   "(*Note: the above warnings may affect the validity of the analysis.)",
                   0);
            end if;
            SPARK_IO.New_Line (ToFile, 1);
         end if;
      end if;
   end ReportSuppressedWarnings;

   ---------------------------------------------------------------------------

begin
   SuppressedElement      := SuppressedElementArray'(others => False);
   PragmaList.PragmaCount := 0; --will cause flow error
   SuppressAllPragmas     := False;
   SomethingSuppressed    := False;
   --# accept Flow, 32, PragmaList.PragmaArray, "Init. is partial but effective." &
   --#        Flow, 31, PragmaList.PragmaArray, "Init. is partial but effective." &
   --#        Flow, 602, PragmaList, PragmaList.PragmaArray, "Init. is partial but effective.";
end WarningStatus;
