-- $Id: errorhandler-warningstatus-readwarningfile.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Ada.Characters.Latin_1;
with CommandLineHandler;

separate (ErrorHandler.WarningStatus)
procedure ReadWarningFile
is
   Option      : EStrings.T;
   FileOk      : Boolean;
   WarningFile : SPARK_IO.File_Type;

   procedure OpenFile
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --#           out FileOk;
   --#           out WarningFile;
   --# derives FileOk,
   --#         SPARK_IO.File_Sys,
   --#         WarningFile       from CommandLineData.Content,
   --#                                SPARK_IO.File_Sys;
   is
      FileName       : EStrings.T;
      FileSpecStatus : FileSystem.TypFileSpecStatus;
      FileStatus     : SPARK_IO.File_Status;

   begin
      --# accept Flow, 10, FileSpecStatus, "Expected ineffective assignment to FileSpecStatus";
      FileSystem.FindFullFileName (CommandLineData.Content.WarningFileName,
                                   FileSpecStatus,
                                   FileName);
      --# end accept;

      WarningFile := SPARK_IO.Null_File; -- to avoid error on opening

      EStrings.Open (File         => WarningFile,
                     Mode_Of_File => SPARK_IO.In_File,
                     Name_Of_File => FileName,
                     Form_Of_File => "",
                     Status       => FileStatus);

      if FileStatus = SPARK_IO.Ok then
         FileOk := True;
      else
         FileOk := False;
         ScreenEcho.Put_String ("Cannot open file ");
         if CommandLineData.Content.PlainOutput then
            ScreenEcho.Put_ExaminerLine
               (EStrings.Lower_Case
                (E_Str => FileSystem.JustFile (FileName, True)));
         else
            ScreenEcho.Put_ExaminerLine (FileName);
         end if;
      end if;
      --# accept Flow, 33, FileSpecStatus, "Expected FileSpecStatus to be neither referenced nor exported";
   end OpenFile;

   ------------------------------------------------

   procedure CloseFile
   --# global in     WarningFile;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                WarningFile;
   is
      FileStatus     : SPARK_IO.File_Status;
   begin
      --# accept Flow, 10, FileStatus, "Expected ineffective assignment to FileStatus" &
      --#        Flow, 10, WarningFile, "Not assigned to. Due to Text_IO mode in out";
      SPARK_IO.Close (WarningFile,
                      FileStatus);
      --# end accept;
      --# accept Flow, 33, FileStatus, "Expected FileStatus to be neither referenced nor exported" &
      --#        Flow, 34, WarningFile, "Not assigned to. Due to Text_IO mode in out";
   end CloseFile;

   ------------------------------------------------

   procedure GetString (File : in     SPARK_IO.File_Type;
                        Str  :    out EStrings.T)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys,
   --#         Str               from File,
   --#                                SPARK_IO.File_Sys;
   is
      CharOk : Boolean;
      Ch     : Character;
      AppendOK : Boolean;
      EStr   : EStrings.T;

      procedure GetChar (File : in     SPARK_IO.File_Type;
                         Ch   :    out Character;
                         OK   :    out Boolean)
      --# global in out SPARK_IO.File_Sys;
      --# derives Ch,
      --#         OK,
      --#         SPARK_IO.File_Sys from File,
      --#                                SPARK_IO.File_Sys;
      is
         ChLocal : Character;
      begin
         if SPARK_IO.End_Of_File (File) then
            OK := False;
            Ch := ' ';
         elsif SPARK_IO.End_Of_Line (File) then
            SPARK_IO.Skip_Line (File, 1);
            OK := True;
            Ch := ' ';
         else
            SPARK_IO.Get_Char (File, ChLocal);
            if (ChLocal = Ada.Characters.Latin_1.HT) or (ChLocal = Ada.Characters.Latin_1.CR) then
               ChLocal := ' ';
            end if;
            if ChLocal = '-' then --must be comment start
               SPARK_IO.Skip_Line (File, 1);
               OK := True;
               Ch := ' ';
            else --valid character to return
               OK := True;
               Ch := ChLocal;
            end if;
         end if;
      end GetChar;

   begin --GetString
      EStr := EStrings.Empty_String;

      --skip leading white space
      loop
         GetChar (File, Ch, CharOk);
         exit when Ch /= ' ';
         exit when not CharOk;
      end loop;

      if CharOk then
         loop
            EStrings.Append_Char (E_Str   => EStr,
                                  Ch      => Ch,
                                  Success => AppendOK);
            if not AppendOK then
               SystemErrors.FatalError (SystemErrors.WarningNameTooLong, "");
            end if;
            GetChar (File, Ch, CharOk);
            exit when Ch = ' ';
            exit when not CharOk;
         end loop;
      end if;
      Str := EStr;
   end GetString;

   -----------------------------------

   procedure InvalidOption (Opt : EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in     WarningFile;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Opt,
   --#                                WarningFile;
   is
   begin
      if CommandLineData.Content.Brief then
         if CommandLineData.Content.PlainOutput then
            ScreenEcho.Put_ExaminerString
              (EStrings.Lower_Case (E_Str => CommandLineData.Content.WarningFileName));
         else
            ScreenEcho.Put_ExaminerString
              (CommandLineData.Content.WarningFileName);
         end if;

         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_Integer (SPARK_IO.Line (WarningFile), 0, 10);
         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_Integer (1, 0, 10);
         ScreenEcho.Put_Char (':');
         ScreenEcho.Put_String ("Invalid warning option: ");
         ScreenEcho.Put_ExaminerLine (Opt);
      else
         ScreenEcho.Put_String ("Invalid warning option: ");
         ScreenEcho.Put_ExaminerLine (Opt);
      end if;
   end InvalidOption;

   -------------------------------

   procedure ProcessOption (Opt : in EStrings.T)
   --# global in     CommandLineData.Content;
   --#        in     WarningFile;
   --#        in out LexTokenManager.State;
   --#        in out PragmaList;
   --#        in out SPARK_IO.File_Sys;
   --#        in out SuppressAllPragmas;
   --#        in out SuppressedElement;
   --# derives LexTokenManager.State,
   --#         PragmaList            from LexTokenManager.State,
   --#                                    Opt,
   --#                                    PragmaList,
   --#                                    SPARK_IO.File_Sys,
   --#                                    WarningFile &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    Opt,
   --#                                    PragmaList,
   --#                                    WarningFile &
   --#         SuppressAllPragmas    from *,
   --#                                    Opt,
   --#                                    SPARK_IO.File_Sys,
   --#                                    WarningFile &
   --#         SuppressedElement     from *,
   --#                                    Opt;
   is
      OptionMatch : Boolean;

      procedure ProcessPragma
      --# global in     WarningFile;
      --#        in out LexTokenManager.State;
      --#        in out PragmaList;
      --#        in out SPARK_IO.File_Sys;
      --#        in out SuppressAllPragmas;
      --# derives LexTokenManager.State,
      --#         SPARK_IO.File_Sys     from *,
      --#                                    PragmaList,
      --#                                    SPARK_IO.File_Sys,
      --#                                    WarningFile &
      --#         PragmaList            from *,
      --#                                    LexTokenManager.State,
      --#                                    SPARK_IO.File_Sys,
      --#                                    WarningFile &
      --#         SuppressAllPragmas    from *,
      --#                                    SPARK_IO.File_Sys,
      --#                                    WarningFile;
      is
         PragmaName : EStrings.T;
         Match      : Boolean;

         procedure AddPragmaName (Prag : in EStrings.T)
         --# global in out LexTokenManager.State;
         --#        in out PragmaList;
         --#        in out SPARK_IO.File_Sys;
         --# derives LexTokenManager.State,
         --#         SPARK_IO.File_Sys     from *,
         --#                                    Prag,
         --#                                    PragmaList &
         --#         PragmaList            from *,
         --#                                    LexTokenManager.State,
         --#                                    Prag;
         is
            LexName : LexTokenManager.Lex_String;
         begin
            if PragmaList.PragmaCount <
               (ExaminerConstants.MaxPragmasInWarningFile - 1) then
               LexTokenManager.Insert_Examiner_String (Str     => Prag,
                                                       Lex_Str => LexName);
               PragmaList.PragmaCount := PragmaList.PragmaCount + 1;
               PragmaList.PragmaArray (PragmaList.PragmaCount) := LexName;
            else -- too many
               ScreenEcho.Put_String ("Too many pragmas, ignoring: ");
               ScreenEcho.Put_ExaminerLine (Prag);
            end if;
         end AddPragmaName;

      begin --ProcessPragma
         GetString (WarningFile,
                     --to get
                    PragmaName);
         if EStrings.Get_Length (E_Str => PragmaName) /= 0 then
            CommandLineHandler.Check_Option_Name (Opt_Name => PragmaName,
                                                  Str      => "all",
                                                  OK       => Match);
            if Match then
               SuppressAllPragmas := True;
            else
               AddPragmaName (PragmaName);
            end if;
         else
            ScreenEcho.Put_Line ("Pragma name missing");
         end if;
      end ProcessPragma;

      --------------------------

   begin --ProcessOption
      OptionMatch := False;
      case EStrings.Get_Element (E_Str => Opt,
                                 Pos   => 1) is
         when 'p' | 'P' =>
            case EStrings.Get_Element (E_Str => Opt,
                                       Pos   => 3) is
               when 'a' | 'A' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "pragma",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.Pragmas) := True;
                     ProcessPragma;
                  end if;
               when 'i' | 'I' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "private_types",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.UnuseablePrivateTypes) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'h' | 'H' =>
            case EStrings.Get_Element (E_Str => Opt,
                                       Pos    => 2) is
               when 'a' | 'A' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "handler_parts",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.HandlerParts) := True;
                  end if;
               when 'i' | 'I' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "hidden_parts",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.HiddenParts) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'r' | 'R' =>
            case EStrings.Get_Element (E_Str => Opt,
                                       Pos   => 3) is
               when 'a' | 'A' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "real_rtcs",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.RealRTCs) := True;
                  end if;
               when 'p' | 'P' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "representation_clauses",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.RepresentationClauses) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'd' | 'D' =>
            case EStrings.Get_Element (E_Str => Opt,
                                       Pos   => 2) is
               when 'i' | 'I' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "direct_updates",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.DirectUpdates) := True;
                  end if;
               when 'e' | 'E' =>
                  case EStrings.Get_Element (E_Str => Opt,
                                             Pos   => 3) is
                     when 'c' | 'C' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                              Str      => "declare_annotations",
                                                              OK       => OptionMatch);
                        if OptionMatch then
                           SuppressedElement (ErrorHandler.DeclareAnnotations) := True;
                        end if;
                     when 'f' | 'F' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                              Str      => "default_loop_assertions",
                                                              OK       => OptionMatch);
                        if OptionMatch then
                           SuppressedElement (ErrorHandler.DefaultLoopAssertions) := True;
                        end if;
                     when others =>
                        null; -- falls through with OptionMatch false and generates error
                  end case;
               when others =>
                  null;
            end case;
         when 'w' | 'W' =>
            CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                  Str      => "with_clauses",
                                                  OK       => OptionMatch);
            if OptionMatch then
               SuppressedElement (ErrorHandler.WithClauses) := True;
            end if;
         when 's' | 'S' =>
            case EStrings.Get_Element (E_Str => Opt,
                                       Pos   => 3) is
               when 'a' | 'A' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "static_expressions",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.StaticExpressions) := True;
                  end if;
               when 'y' | 'Y' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "style_check_casing",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.Style_Check_Casing) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'u' | 'U' =>
            case EStrings.Get_Element (E_Str => Opt,
                                       Pos   => 3) is
               when 'u' | 'U' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "unused_variables",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.UnusedVariables) := True;
                  end if;
               when 'c' | 'C' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "unchecked_conversion",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.UncheckedConversion) := True;
                  end if;
               when others =>
                  null; -- falls through with OptionMatch false and generates error
            end case;
         when 'c' | 'C' =>
            CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                  Str      => "constant_variables",
                                                  OK       => OptionMatch);
            if OptionMatch then
               SuppressedElement (ErrorHandler.ConstantVariables) := True;
            end if;
         when 't' | 'T' =>
            CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                  Str      => "type_conversions",
                                                  OK       => OptionMatch);
            if OptionMatch then
               SuppressedElement (ErrorHandler.TypeConversions) := True;
            end if;
         when 'n' | 'N' =>
            CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                  Str      => "notes",
                                                  OK       => OptionMatch);
            if OptionMatch then
               SuppressedElement (ErrorHandler.Notes) := True;
            end if;
         when 'o' | 'O' =>
            case EStrings.Get_Element (E_Str => Opt,
                                       Pos   => 2) is
               when 'b' | 'B' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "obsolescent_features",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.ObsolescentFeatures) := True;
                  end if;
               when 't' | 'T' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "others_clauses",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.OthersClauses) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'a' | 'A' =>
            case EStrings.Get_Element (E_Str => Opt,
                                       Pos   => 3) is
               when 'a' | 'A' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "ada2005_reserved_words",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.Ada2005ReservedWords) := True;
                  end if;
               when 'd' | 'D' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "address_clauses",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.UnexpectedAddressClauses) := True;
                  end if;
               when others =>
                  null;
            end case;
         when 'e' | 'E' =>
            case EStrings.Get_Element (E_Str => Opt,
                                       Pos   => 3) is
               when 'p' | 'P' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "expression_reordering",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.ExpressionReordering) := True;
                  end if;
               when 't' | 'T' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "external_assignment",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.ExternalVariableAssignment) := True;
                  end if;
               when others =>
                  null;
            end case;

         when 'i' | 'I' =>
            case EStrings.Get_Element (E_Str => Opt,
                                       Pos   => 2) is
               when 'n' | 'N' =>
                  case EStrings.Get_Element (E_Str => Opt,
                                             Pos   => 3) is
                     when 'd' | 'D' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                              Str      => "index_manager_duplicates",
                                                              OK       => OptionMatch);
                        if OptionMatch then
                           SuppressedElement (ErrorHandler.Index_Manager_Duplicates) := True;
                        end if;
                     when 't' | 'T' =>
                        CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                              Str      => "interrupt_handlers",
                                                              OK       => OptionMatch);
                        if OptionMatch then
                           SuppressedElement (ErrorHandler.InterruptHandlers) := True;
                        end if;
                     when others =>
                        null;
                  end case;
               when 'm' | 'M' =>
                  CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                        Str      => "imported_objects",
                                                        OK       => OptionMatch);
                  if OptionMatch then
                     SuppressedElement (ErrorHandler.ImportedObjects) := True;
                  end if;
               when others =>
                  null;
            end case;
         when others =>
            null;
      end case;
      if not OptionMatch then
         InvalidOption (Opt);
      end if;
   end ProcessOption;

   -------------------------------

   procedure SortPragmas
   --# global in     LexTokenManager.State;
   --#        in     SuppressAllPragmas;
   --#        in out PragmaList;
   --# derives PragmaList from *,
   --#                         LexTokenManager.State,
   --#                         SuppressAllPragmas;
   is
      J   : Integer;
      Val : LexTokenManager.Lex_String;
   begin
      if not SuppressAllPragmas and then
        PragmaList.PragmaCount > 1 then
         for I in reverse Integer range 1 .. PragmaList.PragmaCount - 1 loop
            J   := I;
            Val := PragmaList.PragmaArray (J);
            while J < PragmaList.PragmaCount and then
              LexTokenManager.Lex_String_Case_Insensitive_Compare
              (Lex_Str1 => PragmaList.PragmaArray (J + 1),
               Lex_Str2 => Val) = LexTokenManager.Str_First loop
               PragmaList.PragmaArray (J) := PragmaList.PragmaArray (J + 1);
               J := J + 1;
            end loop;
            PragmaList.PragmaArray (J) := Val;
         end loop;
      end if;
   end SortPragmas;

   -------------------------------

begin --ReadWarningFile
   if CommandLineData.Content.Warning then
      OpenFile;
      if FileOk then
         if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
            ScreenEcho.New_Line (1);
            ScreenEcho.Put_Line ("           Reading warning control file ...");
         end if;

         loop
            GetString (WarningFile,
                        --to get
                       Option);
            exit when EStrings.Get_Length (E_Str => Option) = 0;
            ProcessOption (Option);
         end loop;

         CloseFile;

         SortPragmas;

         for i in ErrorHandler.WarningElements loop
            SomethingSuppressed := SomethingSuppressed or
               SuppressedElement (i);
         end loop;
      else
         ErrorHandler.FileOpenError := True;
      end if;
   end if;
end ReadWarningFile;
