-- $Id: errorhandler-justifications.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with LexTokenManager;
with SystemErrors;
with CommandLineData;
with EStrings;
with XMLReport;

separate (ErrorHandler)
package body Justifications
is

   -- The Data Table, stored in ErrorContextRec, has to serve two separate purposes:
   -- (1) a list of justifications in lexical order for listing at end of rep and lst files
   -- (2) a list structured by unit within the file that identifies which justifications are
   --     "in scope" at any particular time and which deal with nested units.

   -- For the first role, we just use the array of records in order.  Index 1 is the first justification
   -- and CurrentSlot the last one.  We can find all entries bt looping over 1 .. CurrentSlot.

   -- For the second role we use a stack and a linkages in the table.  Each table entry has a previous pointer
   -- that points to the next lexically earlier entry of that unit.  The constant EndOfList means there
   -- are no more entries.  For any particular unit, the starting point for the search will be the TOS of the
   -- unit stack.

   -- When we start processing a unit we push EndOfList to start with but always keep TOS pointing at the most recent
   -- table entry that has been added.

   -- When we finish processing the errors for a unit, the stack is popped thus removing all
   -- justifications for that unit from consideration but leaving them unchanged in the table for the first
   -- purpose described earlier.

   -- LastLine-------------------------------------------------------------------
   -- The default scope of a justification is from the start justify to the end of the unit
   -- in which it appears.  An end justify can restrict this scope.  We don't actually need
   -- to know the last line number as we can use the following constant to represent it.
   -- In effect EndOfLineUnit has the value "infinity".
   EndLineOfUnitMarker : constant LexTokenManager.Line_Numbers := 0;

   -- constant of private type declared in package spec
   NullUnmatchedJustificationIterator : constant UnmatchedJustificationIterator :=
     UnmatchedJustificationIterator'(CurrentTableEntry => ErrorHandler.EndOfList,
                                     CurrentPosition => LexTokenManager.Token_Position'(Start_Line_No => 0,
                                                                                        Start_Pos     => 0));

   NoMatchMessage : constant String := "!!! No match found.";

   -- local stack operations -------------------------------
   procedure StackNewUnit (UnitStack : in out ErrorHandler.UnitStacks)
   --# derives UnitStack from *;
   is
   begin
      SystemErrors.RTAssert (UnitStack.Ptr < ErrorHandler.MaxStackSize,
                             SystemErrors.PreconditionFailure,
                             "Stack overflow in error justification handler");
      UnitStack.Ptr := UnitStack.Ptr + 1;
      UnitStack.Vector (UnitStack.Ptr) := ErrorHandler.StackRecord'(ListItems => ErrorHandler.EndOfList,
                                                                    SemanticErrorInUnit => False);
   end StackNewUnit;

   function CurrentUnitListHead (UnitStack : ErrorHandler.UnitStacks) return ErrorHandler.DataTablePtr
   is
   begin
      SystemErrors.RTAssert (UnitStack.Ptr > 0,
                             SystemErrors.PreconditionFailure,
                             "Stack underflow in error justification handler");
      return UnitStack.Vector (UnitStack.Ptr).ListItems;
   end CurrentUnitListHead;

   procedure UpdateCurrentUnitListHead (WhichTable   : in out ErrorHandler.JustificationsDataTables)
   --# derives WhichTable from *;
   is
   begin
      SystemErrors.RTAssert (WhichTable.UnitStack.Ptr > 0,
                             SystemErrors.PreconditionFailure,
                             "Stack underflow in error justification handler");
      -- Set top of the stack that is associated with WhichTable to most recently added table entry index
      WhichTable.UnitStack.Vector (WhichTable.UnitStack.Ptr).ListItems := WhichTable.CurrentSlot;
   end UpdateCurrentUnitListHead;

   function CurrentUnitHasSemanticErrors (UnitStack : ErrorHandler.UnitStacks) return Boolean
   is
   begin
      SystemErrors.RTAssert (UnitStack.Ptr > 0,
                             SystemErrors.PreconditionFailure,
                             "Stack underflow in error justification handler");
      return UnitStack.Vector (UnitStack.Ptr).SemanticErrorInUnit;
   end CurrentUnitHasSemanticErrors;

   procedure StackPopUnit (UnitStack : in out ErrorHandler.UnitStacks)
   --# derives UnitStack from *;
   is
   begin
      SystemErrors.RTAssert (UnitStack.Ptr > 0,
                             SystemErrors.PreconditionFailure,
                             "Stack underflow in error justification handler");
      UnitStack.Ptr := UnitStack.Ptr - 1;
   end StackPopUnit;

   function StackIsEmpty (UnitStack : ErrorHandler.UnitStacks) return Boolean
   is
   begin
      return UnitStack.Ptr = 0;
   end StackIsEmpty;

   -- exported operations -------------------------------------------------------

   procedure StartUnit (WhichTable : in out ErrorHandler.JustificationsDataTables)
   is
   begin
      StackNewUnit (WhichTable.UnitStack);
   end StartUnit;

   procedure SetCurrentUnitHasSemanticErrors (WhichTable : in out ErrorHandler.JustificationsDataTables)
   is
   begin
      -- If a semantic error occurs before we get into the declarative part, or statements
      -- of a unit then the stack will be empty; however, there are no justifiable warnings
      -- for these regions so we can simply ignore the call
      if not StackIsEmpty (WhichTable.UnitStack) then
         WhichTable.UnitStack.Vector (WhichTable.UnitStack.Ptr).SemanticErrorInUnit := True;
      end if;
   end SetCurrentUnitHasSemanticErrors;

   -- Operations concerned with reaching the end of a subprogram or other unit.  We provide an
   -- iterator for finding all the unmatched justifications so that Errorhandler.EndUnit can report
   -- them and also a stack Pop operation to clear the now completed unit from scope.

   -- local functions shared by FirstUnmatchedJustification and NextUnmatchedJustification
   -- Don't report unmatched flow  messages if a semantic error has occurred
   function IgnoreFlowWhenSemanticErrors (WhichTable  : ErrorHandler.JustificationsDataTables;
                                          CurrentItem : ErrorHandler.DataTablePtr) return Boolean
   is
   begin
      return CurrentUnitHasSemanticErrors (WhichTable.UnitStack)
        and then WhichTable.DataTable (CurrentItem).Kind = ErrorHandler.FlowMessage;
   end IgnoreFlowWhenSemanticErrors;

   -- Don't report unmatched IFA messages if IFA turned off
   function IgnoreInformationFlowErrors (WhichTable  : ErrorHandler.JustificationsDataTables;
                                         CurrentItem : ErrorHandler.DataTablePtr) return Boolean
   --# global in CommandLineData.Content;
   is
      function IsIFA (Num : Natural) return Boolean
      is
      begin
         return Num = 50 or else Num = 601 or else Num = 602;
      end IsIFA;
   begin -- IgnoreInformationFlowErrors
      return (not CommandLineData.Content.DoInformationFlow) and then
        WhichTable.DataTable (CurrentItem).Kind = ErrorHandler.FlowMessage and then
        IsIFA (WhichTable.DataTable (CurrentItem).ErrNum);
   end IgnoreInformationFlowErrors;


   procedure FirstUnmatchedJustification (It         :    out UnmatchedJustificationIterator;
                                          WhichTable : in     ErrorHandler.JustificationsDataTables)
   is
      CurrentItem : ErrorHandler.DataTablePtr;

   begin
      -- establish default answer
      It := NullUnmatchedJustificationIterator;
      -- seek unmatched items
      CurrentItem := CurrentUnitListHead (WhichTable.UnitStack);
      while CurrentItem /= ErrorHandler.EndOfList loop
         if WhichTable.DataTable (CurrentItem).MatchCount = 0 and then
            -- Unmatched item found,
            -- but we ignore it if it is a flow error justification and the flow analyser hasn't run
           (not IgnoreFlowWhenSemanticErrors (WhichTable, CurrentItem)) and then
           (not IgnoreInformationFlowErrors (WhichTable, CurrentItem)) then
            It := UnmatchedJustificationIterator'(CurrentTableEntry => WhichTable.DataTable (CurrentItem).Previous,
                                                  CurrentPosition => WhichTable.DataTable (CurrentItem).Position);

            exit;
         end if;
         CurrentItem := WhichTable.DataTable (CurrentItem).Previous;
      end loop;
   end FirstUnmatchedJustification;

   procedure NextUnmatchedJustification (It         : in out UnmatchedJustificationIterator;
                                         WhichTable : in     ErrorHandler.JustificationsDataTables)
   is
      CurrentItem : ErrorHandler.DataTablePtr;
   begin
      CurrentItem := It.CurrentTableEntry;
      -- establish default answer
      It := NullUnmatchedJustificationIterator;
      -- seek unmatched items
      while CurrentItem /= ErrorHandler.EndOfList loop
         if WhichTable.DataTable (CurrentItem).MatchCount = 0 and then
            -- Unmatched item found,
            -- but we ignore it if it is a flow error justification and the flow analyser hasn't run
           (not IgnoreFlowWhenSemanticErrors (WhichTable, CurrentItem)) and then
           (not IgnoreInformationFlowErrors (WhichTable, CurrentItem)) then
            It := UnmatchedJustificationIterator'(CurrentTableEntry => WhichTable.DataTable (CurrentItem).Previous,
                                                  CurrentPosition => WhichTable.DataTable (CurrentItem).Position);

            exit;
         end if;
         CurrentItem := WhichTable.DataTable (CurrentItem).Previous;
      end loop;
   end NextUnmatchedJustification;

   function ErrorPosition (It : UnmatchedJustificationIterator) return LexTokenManager.Token_Position
   is
   begin
      return It.CurrentPosition;
   end ErrorPosition;

   function IsNullIterator (It : UnmatchedJustificationIterator) return Boolean
   is
   begin
      return It = NullUnmatchedJustificationIterator;
   end IsNullIterator;

   procedure EndUnit (WhichTable : in out ErrorHandler.JustificationsDataTables)
   is
   begin
      -- Discard all justifications belonging to this now finished unit
      StackPopUnit (WhichTable.UnitStack);
   end EndUnit;

   -- end of operations associated with reaching the end of a unit

   procedure StartJustification (WhichTable                   : in out ErrorHandler.JustificationsDataTables;
                                 Position                     : in     LexTokenManager.Token_Position;
                                 Line                         : in     LexTokenManager.Line_Numbers;
                                 Kind                         : in     ErrorHandler.JustificationKinds;
                                 ErrNum                       : in     Natural;
                                 Identifiers                  : in     ErrorHandler.JustificationIdentifiers;
                                 Explanation                  : in     LexTokenManager.Lex_String;
                                 MaximumJustificationsReached :    out Boolean)
   is
      NewEntry : ErrorHandler.DataTableEntry;
   begin
      -- The return parameter below is only ever set True once, when the table first fills up.  If the
      -- table is already full then we return False because we only want to generate one warning
      -- at the point of call where the table first fills, not at every call thereafter.
      MaximumJustificationsReached := False;

      if WhichTable.AcceptingMoreEntries then
         WhichTable.CurrentSlot := WhichTable.CurrentSlot + 1;
         if WhichTable.CurrentSlot = ErrorHandler.MaxTableEntries then
            MaximumJustificationsReached := True;       -- signal to caller that table has just become full
            WhichTable.AcceptingMoreEntries := False;   -- remember that table is full for future calls
         end if;

         NewEntry := ErrorHandler.DataTableEntry'
           (Kind        => Kind,
            ErrNum      => ErrNum,
            Identifiers => Identifiers,
            Explanation => Explanation,
            Position    => Position,
            StartLine   => Line,
            EndLine     => EndLineOfUnitMarker,
            EndFound    => False,
            MatchCount  => 0,
            MatchLine   => WhichTable.DataTable (WhichTable.CurrentSlot).MatchLine,
            Previous    => CurrentUnitListHead (WhichTable.UnitStack));

         WhichTable.DataTable (WhichTable.CurrentSlot) := NewEntry;
         UpdateCurrentUnitListHead (WhichTable);
      end if;
   end StartJustification;

   procedure EndJustification (WhichTable   : in out ErrorHandler.JustificationsDataTables;
                               Line         : in     LexTokenManager.Line_Numbers;
                               UnmatchedEnd :    out Boolean)
   is
      EntryToCheck     : ErrorHandler.DataTablePtr;
      MatchFound       : Boolean := False;
      StartingingLine  : LexTokenManager.Line_Numbers;
   begin -- EndJustification
      SystemErrors.RTAssert (not StackIsEmpty  (WhichTable.UnitStack),
                             SystemErrors.PreconditionFailure,
                             "Stack underflow in EndJustification");

      EntryToCheck := CurrentUnitListHead (WhichTable.UnitStack);
      while EntryToCheck /= ErrorHandler.EndOfList loop
         if not WhichTable.DataTable (EntryToCheck).EndFound then
            -- a start justify with no matching end has been found
            MatchFound := True;
            WhichTable.DataTable (EntryToCheck).EndFound := True;
            WhichTable.DataTable (EntryToCheck).EndLine := Line; -- end justify restricts line range over which it is valid

            -- At this point we have matched one begin with one end; however, there is one further check to do to
            -- deal with the form of justify statement that has several clauses separated by '&'.  In this case we will
            -- have several entries all with the same start line.  We want to set all of these to be closed by the
            -- end we have just found.
            StartingingLine := WhichTable.DataTable (EntryToCheck).StartLine;
            EntryToCheck := WhichTable.DataTable (EntryToCheck).Previous;
            while EntryToCheck /= ErrorHandler.EndOfList
            --# assert MatchFound;
            loop
               -- we process further linked table entries until we find one that has a different start line number
               -- and therefore is not part of the same multiple entry clause
               exit when WhichTable.DataTable (EntryToCheck).StartLine /= StartingingLine;

               -- if we get to here, the line number is the same and it is part of the same clause
               WhichTable.DataTable (EntryToCheck).EndFound := True;
               WhichTable.DataTable (EntryToCheck).EndLine := Line; -- restricts line range over which justify valid

               EntryToCheck := WhichTable.DataTable (EntryToCheck).Previous;
            end loop;
         end if;
         exit when MatchFound; -- each end justify should only match one start
         EntryToCheck := WhichTable.DataTable (EntryToCheck).Previous;
      end loop;
      -- Tell caller that end didn't match a start so that warning can be raised; however, don't return True
      -- if the table has filled up otherwise we will get lots of unmatched end warnings for the justifications
      -- that never got added because the table was full
      UnmatchedEnd := not MatchFound and WhichTable.AcceptingMoreEntries;
   end EndJustification;

   procedure CheckWhetherJustified (WhichTable  : in out ErrorHandler.JustificationsDataTables;
                                    Line        : in     LexTokenManager.Token_Position;
                                    Kind        : in     ErrorHandler.JustificationKinds;
                                    ErrNum      : in     Natural;
                                    Identifiers : in     ErrorHandler.JustificationIdentifiers;
                                    MatchFound  :    out Boolean)
   is
      EntryToCheck     : ErrorHandler.DataTablePtr;

      function MatchingEntryFound (TheTableEntry : ErrorHandler.DataTableEntry;
                                   Line          : LexTokenManager.Line_Numbers;
                                   Kind          : ErrorHandler.JustificationKinds;
                                   ErrNum        : Natural;
                                   Identifiers   : ErrorHandler.JustificationIdentifiers) return Boolean
      --# global in LexTokenManager.State;
      is
         function BelowEndLine (Line, EndLine : LexTokenManager.Line_Numbers) return Boolean
         is
            Result : Boolean;
         begin
            if EndLine = EndLineOfUnitMarker then
               Result := True;
            else
               Result := Line <= EndLine;
            end if;
            return Result;
         end BelowEndLine;

         function IdentifiersMatch (TheTableEntry : ErrorHandler.DataTableEntry;
                                    Identifiers   : ErrorHandler.JustificationIdentifiers) return Boolean
         --# global in LexTokenManager.State;
         is
            Result : Boolean := True;

            function IdentifierMatches (FromTheTable,
                                        FromTheCall : ErrorHandler.JustificationIdentifier) return Boolean
            --# global in LexTokenManager.State;
            is
               Result : Boolean;
            begin
               -- Tricky comparison.  FromTheCall will contain: a null string and a valid symbol;
               -- or a valid string and a null symbol; or both will be null.
               -- FromTheTable will contain either both null or both valid.
               -- We need to match as follows:
               if FromTheCall = ErrorHandler.NullJustificationIdentifier then
                  -- both null, so we require FromTheTable to be exactly the same
                  Result := FromTheTable = ErrorHandler.NullJustificationIdentifier;

               elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => FromTheCall.StringForm,
                                                                          Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
                  Result := FromTheCall.SymbolForm = FromTheTable.SymbolForm;

               else -- Strings aren't null so compare them
                  Result := LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => FromTheCall.StringForm,
                                                                                 Lex_Str2 => FromTheTable.StringForm) = LexTokenManager.Str_Eq;
               end if;

               return Result;
            end IdentifierMatches;

         begin -- IdentifiersMatch
            for I in Integer range 1 .. ErrorHandler.MaxJustificationIdentifierLength loop
               if not IdentifierMatches (TheTableEntry.Identifiers (I),
                                         Identifiers (I)) then
                  Result := False;
                  exit;
               end if;
            end loop;
            return Result;
         end IdentifiersMatch;

      begin -- MatchingEntryFound
         return
           Line >= TheTableEntry.StartLine and then
           BelowEndLine (Line, TheTableEntry.EndLine) and then
           ErrNum = TheTableEntry.ErrNum and then
           Kind = TheTableEntry.Kind and then
           IdentifiersMatch (TheTableEntry,
                             Identifiers); -- last because it is much the most expensive test
      end MatchingEntryFound;

   begin -- CheckWhetherJustified
      MatchFound := False;
      if CommandLineData.Content.JustificationOption = CommandLineData.Ignore then
         null;
      else
         if not StackIsEmpty  (WhichTable.UnitStack) then -- can't have a match if nothing is even in stack yet
            EntryToCheck := CurrentUnitListHead (WhichTable.UnitStack);
            while EntryToCheck /= ErrorHandler.EndOfList loop
               if MatchingEntryFound (WhichTable.DataTable (EntryToCheck),
                                      Line.Start_Line_No,
                                      Kind,
                                      ErrNum,
                                      Identifiers) then

                  -- note how many times we got a match
                  WhichTable.DataTable (EntryToCheck).MatchCount := WhichTable.DataTable (EntryToCheck).MatchCount + 1;
                  -- and retain the most recent line number where it happened
                  WhichTable.DataTable (EntryToCheck).MatchLine := Line.Start_Line_No;

                  -- finally, return result to caller
                  MatchFound := True;
                  exit;
               end if;
               EntryToCheck := WhichTable.DataTable (EntryToCheck).Previous;
            end loop;
         end if;
      end if;
   end CheckWhetherJustified;

   function TableContainsEntries (WhichTable : in     ErrorHandler.JustificationsDataTables)
                                  return Boolean
   is
   begin
      return WhichTable.CurrentSlot > 0;
   end TableContainsEntries;

   procedure PrintJustifications (WhichTable : in     ErrorHandler.JustificationsDataTables;
                                  File       : in     SPARK_IO.File_Type)
   is
      procedure PrintCommonHeader
      --# global in     File;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                File;
      is
      begin
         -- This string is common to both "Full" and "Brief" justifications modes,
         -- so that the justifications summary table (in either mode) can be
         -- recognized by the HTML report file generator.  If this string changes,
         -- then ProcessReportLine in sparkhtml.adb will also need to be updated.
         SPARK_IO.Put_Line (File,
                            "Expected messages marked with the accept annotation", 0);
      end PrintCommonHeader;

      procedure PrintFullListing
      --# global in     CommandLineData.Content;
      --#        in     File;
      --#        in     LexTokenManager.State;
      --#        in     WhichTable;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                File,
      --#                                LexTokenManager.State,
      --#                                WhichTable;
      is
         procedure PrintHeaders
         --# global in     File;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                File;
         is
         begin
            PrintCommonHeader;
            SPARK_IO.Put_Line (File,
                               "Type Msg    Lines              Reason                    Match", 0);
            SPARK_IO.Put_Line (File,
                               "     No.  From    To                                    No.  Line", 0);
         end PrintHeaders;

         procedure PrintKind (TheType : in ErrorHandler.JustificationKinds)
         --# global in     File;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                File,
         --#                                TheType;
         is
         begin
            case TheType is
               when ErrorHandler.FlowMessage =>
                  SPARK_IO.Put_String (File, "Flow ", 0);
               when ErrorHandler.WarningMessage =>
                  SPARK_IO.Put_String (File, "Warn ", 0);
            end case;
         end PrintKind;

         procedure PrintLineNo (TheLine : in LexTokenManager.Line_Numbers)
         --# global in     CommandLineData.Content;
         --#        in     File;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                CommandLineData.Content,
         --#                                File,
         --#                                TheLine;
         is
         begin
            if TheLine = 0 then
               SPARK_IO.Put_String (File, "   end", 0);
            elsif CommandLineData.Content.PlainOutput then
               SPARK_IO.Put_String (File, "      ", 0);
            else
               SPARK_IO.Put_Integer (File,
                                     Integer (TheLine),
                                     6,
                                     10);
            end if;
         end PrintLineNo;

         procedure PrintExplanation (LexExplanation : in LexTokenManager.Lex_String)
         --# global in     File;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                File,
         --#                                LexExplanation,
         --#                                LexTokenManager.State;
         is
            ExExplanation : EStrings.T;
            ChIdx,
            PrintedChars : Natural;
            Ch : Character;
            FieldWidth : constant := 32;
         begin
            ExExplanation := LexTokenManager.Lex_String_To_String (Lex_Str => LexExplanation);

            SPARK_IO.Put_String (File, "  ", 0);
            ChIdx := 1;
            PrintedChars := 0;
            loop
               exit when ChIdx > EStrings.Get_Length (E_Str => ExExplanation);
               exit when PrintedChars >= FieldWidth;

               Ch := EStrings.Get_Element (E_Str => ExExplanation,
                                           Pos   => ChIdx);
               if Ch /= '"' then -- strip quotes
                  SPARK_IO.Put_Char (File, EStrings.Get_Element (E_Str => ExExplanation,
                                                                 Pos   => ChIdx));
                  PrintedChars := PrintedChars + 1;
               end if;
               ChIdx := ChIdx + 1;
            end loop;
            -- if we haven't reached FieldWidth then pad out with spaces
            for I in Natural range PrintedChars .. FieldWidth loop
               SPARK_IO.Put_Char (File, ' ');
            end loop;
         end PrintExplanation;

      begin -- PrintFullListing
         PrintHeaders;
         for I in ErrorHandler.DataTableIndex range 1 .. WhichTable.CurrentSlot loop
            PrintKind (WhichTable.DataTable (I).Kind);

            SPARK_IO.Put_Integer (File,
                                  WhichTable.DataTable (I).ErrNum,
                                  3,
                                  10);

            PrintLineNo (WhichTable.DataTable (I).StartLine);
            PrintLineNo (WhichTable.DataTable (I).EndLine);

            PrintExplanation (WhichTable.DataTable (I).Explanation);
            SPARK_IO.Put_Integer (File,
                                  WhichTable.DataTable (I).MatchCount,
                                  4,
                                  10);

            if WhichTable.DataTable (I).MatchCount = 0 then
               SPARK_IO.Put_String (File,
                                    "  " & NoMatchMessage, 0);
            else
               PrintLineNo (WhichTable.DataTable (I).MatchLine);
            end if;
            SPARK_IO.New_Line (File, 1);
         end loop;
         SPARK_IO.New_Line (File, 2);
      end PrintFullListing;


      procedure PrintBriefListing
      --# global in     File;
      --#        in     WhichTable;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                File,
      --#                                WhichTable;
      is
         FailedMatches     : Natural := 0;
      begin
         PrintCommonHeader;
         for I in ErrorHandler.DataTableIndex range 1 .. WhichTable.CurrentSlot loop
            if WhichTable.DataTable (I).MatchCount = 0 then
               FailedMatches := FailedMatches + 1;
            end if;
         end loop;
         SPARK_IO.Put_Integer (File,
                               Integer (WhichTable.CurrentSlot),
                               0,
                               10);
         SPARK_IO.Put_String (File,
                              " message(s) marked as expected", 0);
         if FailedMatches > 0 then
            SPARK_IO.Put_String (File,
                                 ", !!! Warning, ", 0);
            SPARK_IO.Put_Integer (File,
                                  FailedMatches,
                                  0,
                                  10);
            SPARK_IO.Put_String (File,
                                 " message(s) did not occur", 0);

         end if;
         SPARK_IO.Put_Char (File, '.');
         SPARK_IO.New_Line (File, 2);
      end PrintBriefListing;

   begin -- PrintJustifications
      if TableContainsEntries (WhichTable) then
         SPARK_IO.New_Line (File, 1);
         case CommandLineData.Content.JustificationOption is
            when CommandLineData.Full =>
               PrintFullListing;
            when CommandLineData.Brief =>
               PrintBriefListing;
            when CommandLineData.Ignore =>
               null;
         end case;
      end if;
   end PrintJustifications;

   -- Precondition: Must be called on a report file, at the correct location in the schema
   procedure PrintJustificationsXML (WhichTable : in     ErrorHandler.JustificationsDataTables;
                                     File       : in     SPARK_IO.File_Type)
   is
      procedure PrintFullListing
      --# global in     File;
      --#        in     LexTokenManager.State;
      --#        in     WhichTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                File,
      --#                                LexTokenManager.State,
      --#                                WhichTable,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                WhichTable;
      is
         NoMatchExplanation : EStrings.T;

         function PrintKindToString (TheType : in ErrorHandler.JustificationKinds) return EStrings.T
         is
            KindString : EStrings.T;
         begin
            case TheType is
               when ErrorHandler.FlowMessage =>
                  KindString := EStrings.Copy_String (Str => "Flow");
               when ErrorHandler.WarningMessage =>
                  KindString := EStrings.Copy_String (Str => "Warning");
            end case;
            return KindString;
         end PrintKindToString;

         function PrintLineNoToString (TheLine : in LexTokenManager.Line_Numbers) return EStrings.T
         is
            LineNoStr : EStrings.T;
            subtype StringLength is Integer range 1 .. 255;
            subtype TempString is String (StringLength);
            TmpString : TempString;
         begin
            if TheLine = 0 then
               LineNoStr := EStrings.Copy_String (Str => "end");
            else
               SPARK_IO.Put_Int_To_String (TmpString, Integer (TheLine), 1, 10);
               LineNoStr := EStrings.Copy_String (Str => TmpString);
            end if;
            return LineNoStr;
         end PrintLineNoToString;


      begin -- PrintFullListing
         NoMatchExplanation := EStrings.Copy_String (Str => NoMatchMessage);
         XMLReport.StartSection (XMLReport.SFullJustifications, File);

         for I in ErrorHandler.DataTableIndex range 1 .. WhichTable.CurrentSlot loop

            XMLReport.StartFullJustification (PrintKindToString (WhichTable.DataTable (I).Kind),
                                              WhichTable.DataTable (I).ErrNum,
                                              Integer (WhichTable.DataTable (I).StartLine),
                                              PrintLineNoToString (WhichTable.DataTable (I).EndLine),
                                              WhichTable.DataTable (I).MatchCount,
                                              Integer (WhichTable.DataTable (I).MatchLine),
                                              File);


            if WhichTable.DataTable (I).MatchCount = 0 then
               EStrings.Put_String (File  => File,
                                    E_Str => NoMatchExplanation);
            else
               EStrings.Put_String
                 (File  => File,
                  E_Str => XMLReport.FilterString
                    (LexTokenManager.Lex_String_To_String (Lex_Str => WhichTable.DataTable (I).Explanation)));
            end if;

            XMLReport.EndFullJustification (File);
         end loop;
         XMLReport.EndSection (XMLReport.SFullJustifications, File);
      end PrintFullListing;


      procedure PrintBriefListing
      --# global in     File;
      --#        in     WhichTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives SPARK_IO.File_Sys from *,
      --#                                File,
      --#                                WhichTable,
      --#                                XMLReport.State &
      --#         XMLReport.State   from *,
      --#                                WhichTable;
      is
         FailedMatches     : Natural := 0;
      begin
         for I in ErrorHandler.DataTableIndex range 1 .. WhichTable.CurrentSlot loop
            if WhichTable.DataTable (I).MatchCount = 0 then
               FailedMatches := FailedMatches + 1;
            end if;
         end loop;
         XMLReport.BriefJustifications (Natural (WhichTable.CurrentSlot), FailedMatches, File);
      end PrintBriefListing;

   begin -- PrintJustificationsXML
      if TableContainsEntries (WhichTable) then
         XMLReport.StartSection (XMLReport.SJustifications, File);
         case CommandLineData.Content.JustificationOption is
            when CommandLineData.Full =>
               PrintFullListing;
            when CommandLineData.Brief =>
               PrintBriefListing;
            when CommandLineData.Ignore =>
               null;
         end case;
         XMLReport.EndSection (XMLReport.SJustifications, File);
      end if;
   end PrintJustificationsXML;

end Justifications;
