-- $Id: errorhandler-errorbuffer.adb 15826 2010-01-28 18:09:51Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

separate (ErrorHandler)
package body ErrorBuffer
--# own Buffer is TheBuffer, CurrentFlowError;
is

   -------------------------------------------------------------------

   subtype BuffIndex is Integer range 1 .. ExaminerConstants.ErrorBufferSize;
   subtype BuffPtr   is Integer range 0 .. ExaminerConstants.ErrorBufferSize;
   type ErrorArray is array (BuffIndex) of Error_Types.NumericError;
   type Buffers is record
      ErrorList : ErrorArray;
      ErrPtr    : BuffPtr;
   end record;

   TheBuffer  : Buffers;

   CurrentFlowError : Error_Types.NumericError;

   --------------------------------------------------------------------------
   --                      Local Procedures
   --------------------------------------------------------------------------

   function NullError return Error_Types.NumericError
   --# global in Dictionary.Dict;
   is
   begin
      return Error_Types.NumericError'(ErrorType => Error_Types.NoErr,
                                       Position  => LexTokenManager.Token_Position'
                                         (Start_Line_No => 0,
                                          Start_Pos     => 0),
                                       Scope     => Dictionary.GlobalScope,
                                       ErrorNum  => 0,
                                       Reference => 0,
                                       Name1     => Error_Types.NoName,
                                       Name2     => Error_Types.NoName,
                                       Name3     => Error_Types.NoName);
   end NullError;

   --------------------------------------------------------------------

   procedure CreateTemp (F : out Error_IO.File_Type)
   --# global in out SPARK_IO.File_Sys;
   --# derives F,
   --#         SPARK_IO.File_Sys from SPARK_IO.File_Sys;
   is
      Ok     : SPARK_IO.File_Status;
      LocalF : Error_IO.File_Type := Error_IO.Null_File;
   begin
      Error_IO.Create (LocalF, Ok);
      if Ok /= SPARK_IO.Ok then
         SystemErrors.FatalError (SystemErrors.ErrorHandlerTemporaryFiles, "in ErrorBuffer.CreateTemp");
      end if;
      F := LocalF;
   end CreateTemp;

   --------------------------------------------------------------------------

   procedure ResetTemp (F : in out Error_IO.File_Type)
   --# global in out SPARK_IO.File_Sys;
   --# derives F,
   --#         SPARK_IO.File_Sys from *,
   --#                                F;
   is
      Ok : SPARK_IO.File_Status;
   begin
      Error_IO.Reset (F, SPARK_IO.In_File, Ok);
      if Ok /= SPARK_IO.Ok then
         SystemErrors.FatalError (SystemErrors.ErrorHandlerTemporaryFiles, "in ErrorBuffer.ResetTemp");
      end if;
   end ResetTemp;

   --------------------------------------------------------------------------

   procedure CloseTemp (F : in out Error_IO.File_Type)
   --# global in out SPARK_IO.File_Sys;
   --# derives F,
   --#         SPARK_IO.File_Sys from *,
   --#                                F;
   is
      Ok : SPARK_IO.File_Status;
   begin
      Error_IO.Close (F, Ok);
      if Ok /= SPARK_IO.Ok then
         SystemErrors.FatalError (SystemErrors.ErrorHandlerTemporaryFiles, "in ErrorBuffer.CloseTemp");
      end if;
   end CloseTemp;

   --------------------------------------------------------------------------

   procedure InitBuff
   --# global out TheBuffer;
   --# derives TheBuffer from ;
   is
   begin
      TheBuffer.ErrPtr := 0;
      --intentional failure to initialize array will cause flow error here
      --# accept F, 31,  TheBuffer.ErrorList, "Intentional incomplete initialization" &
      --#        F, 32,  TheBuffer.ErrorList, "Intentional incomplete initialization" &
      --#        F, 602, TheBuffer, TheBuffer.ErrorList, "Intentional incomplete initialization";
   end InitBuff; -- Init. is partial but effecive.  Expect 2 errs + 1 warning

   --------------------------------------------------------------------------

   procedure ResetErrorNum (ErrNum : in out Error_Types.NumericError)
   --# global in     Dictionary.Dict;
   --#        in out CurrentFlowError;
   --# derives CurrentFlowError from *,
   --#                               Dictionary.Dict,
   --#                               ErrNum &
   --#         ErrNum           from *,
   --#                               CurrentFlowError;

   is
   begin
      case ErrNum.ErrorType is
         when Error_Types.UncondDependencyErr          =>
            if ErrNum.ErrorNum = ErrorHandler.DependencyErrNumber (ErrorHandler.NotUsedNew) then
               if CurrentFlowError.ErrorNum = ErrorHandler.DependencyErrNumber (ErrorHandler.NotUsedNew) and then
                 (ErrNum.Name2 = CurrentFlowError.Name2 and
                    ErrNum.Scope = CurrentFlowError.Scope and
                    ErrNum.Position = CurrentFlowError.Position) then
                  -- Continuation
                  ErrNum.ErrorNum := ErrorHandler.DependencyErrNumber (ErrorHandler.NotUsedContinue);
               else
                  -- New Error;
                  CurrentFlowError := ErrNum;
               end if;
            else
               CurrentFlowError := NullError;
            end if;

         when Error_Types.CondlDependencyErr           =>
            if ErrNum.ErrorNum = ErrorHandler.DependencyErrNumber (ErrorHandler.MayBeUsedNew) then
               if  CurrentFlowError.ErrorNum = ErrorHandler.DependencyErrNumber (ErrorHandler.MayBeUsedNew) and then
                 (ErrNum.Name2 = CurrentFlowError.Name2 and
                    ErrNum.Scope = CurrentFlowError.Scope and
                    ErrNum.Position = CurrentFlowError.Position) then
                  -- Continuation
                  ErrNum.ErrorNum := ErrorHandler.DependencyErrNumber (ErrorHandler.MayBeUsedContinue);
               else
                  -- New Error;
                  CurrentFlowError := ErrNum;
               end if;
            else
               CurrentFlowError := NullError;
            end if;

         when others =>
            CurrentFlowError := NullError;
      end case;
   end ResetErrorNum;

   --------------------------------------------------------------------------
   --                      Exported Procedures
   --------------------------------------------------------------------------

   procedure Flush (ErrFile : in out Error_IO.File_Type)
   --# global in     Dictionary.Dict;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TheBuffer;
   --# derives ErrFile           from *,
   --#                                SPARK_IO.File_Sys,
   --#                                TheBuffer &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dictionary.Dict,
   --#                                ErrFile,
   --#                                TheBuffer &
   --#         TheBuffer         from ;
   is

      function IsLessThan (One, Two : LexTokenManager.Token_Position) return Boolean
      is
         Less_Than : Boolean;
      begin
         if One.Start_Line_No = Two.Start_Line_No then
            Less_Than := One.Start_Pos < Two.Start_Pos;
         else
            Less_Than := One.Start_Line_No < Two.Start_Line_No;
         end if;
         return Less_Than;
      end IsLessThan;

      --------------------------

      procedure SortBuff
      --# global in out TheBuffer;
      --# derives TheBuffer from *;
      is

         procedure Swap (X, Y : in BuffPtr)
         --# global in out TheBuffer;
         --# derives TheBuffer from *,
         --#                        X,
         --#                        Y;
         is
            T : Error_Types.NumericError;
         begin
            T := TheBuffer.ErrorList (X);
            TheBuffer.ErrorList (X) := TheBuffer.ErrorList (Y);
            TheBuffer.ErrorList (Y) := T;
         end Swap;

      begin --SortBuff
         for I in BuffPtr range 1 .. TheBuffer.ErrPtr loop
            for J in BuffPtr range I .. TheBuffer.ErrPtr loop
               if IsLessThan (TheBuffer.ErrorList (J).Position,
                              TheBuffer.ErrorList (I).Position) then
                  Swap (I, J);
               end if;
            end loop;
         end loop;
      end SortBuff;

      ------------------------------------------

      procedure Merge
      --# global in     Dictionary.Dict;
      --#        in     TheBuffer;
      --#        in out ErrFile;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrFile           from *,
      --#                                SPARK_IO.File_Sys,
      --#                                TheBuffer &
      --#         SPARK_IO.File_Sys from *,
      --#                                Dictionary.Dict,
      --#                                ErrFile,
      --#                                TheBuffer;
      is
         NewFile   : Error_IO.File_Type;
         Ptr       : BuffPtr;
         BufEmpty,
         FileEmpty : Boolean;
         BufEnt,
         --WriteEnt,
         FileEnt   : Error_Types.NumericError;

         --------------------------

         procedure GetBufferEntry (Ent   : out Error_Types.NumericError;
                                   Empty : out Boolean)
         --# global in     Dictionary.Dict;
         --#        in     TheBuffer;
         --#        in out Ptr;
         --# derives Empty,
         --#         Ptr   from Ptr,
         --#                    TheBuffer &
         --#         Ent   from Dictionary.Dict,
         --#                    Ptr,
         --#                    TheBuffer;
         is
         begin
            if Ptr = TheBuffer.ErrPtr then
               Ent := NullError;
               Empty := True;
            else
               Ptr := Ptr + 1;
               Ent := TheBuffer.ErrorList (Ptr);
               Empty := False;
            end if;
         end GetBufferEntry;

         --------------------------

      begin --Merge
         Ptr := 0;
         GetBufferEntry (BufEnt, BufEmpty);
         if not BufEmpty then --only merge sort if buffer contains some entries
            CreateTemp (NewFile);
            ResetTemp (ErrFile);

            Error_IO.Get_Numeric_Error (ErrFile, FileEnt);
            FileEmpty := (FileEnt = Error_Types.Empty_NumericError);

            while not (BufEmpty and FileEmpty) loop
               if FileEmpty then
                  Error_IO.Put_Numeric_Error (NewFile, BufEnt);
                  GetBufferEntry (BufEnt, BufEmpty);

               elsif BufEmpty then
                  Error_IO.Put_Numeric_Error (NewFile, FileEnt);
                  Error_IO.Get_Numeric_Error (ErrFile, FileEnt);
                  FileEmpty := (FileEnt = Error_Types.Empty_NumericError);

               else --neither empty
                  if IsLessThan (BufEnt.Position, FileEnt.Position) then
                     Error_IO.Put_Numeric_Error (NewFile, BufEnt);
                     GetBufferEntry (BufEnt, BufEmpty);

                  else
                     Error_IO.Put_Numeric_Error (NewFile, FileEnt);
                     Error_IO.Get_Numeric_Error (ErrFile, FileEnt);
                     FileEmpty := (FileEnt = Error_Types.Empty_NumericError);

                  end if;
               end if;
            end loop;
            --# accept Flow, 10, ErrFile, "Expected ineffective assignment";
            CloseTemp (ErrFile); -- Ineffective assignment as immediately overwritten
            --# end accept;
            ErrFile := NewFile;
         end if;
      end Merge;

      ------------------------------------------

   begin --Flush
      SortBuff;
      Merge;
      InitBuff;
   end Flush;

   --------------------------------------------------------------------------

   procedure Add (ErrFile     : in out Error_IO.File_Type;
                  ErrType     : in     Error_Types.Error_Class;
                  Pos         : in     LexTokenManager.Token_Position;
                  Scope       : in     Dictionary.Scopes;
                  ErrorNumber : in     Natural;
                  Reference   : in     Natural;
                  Name1,
                  Name2,
                  Name3       : in     Error_Types.Names;
                  EchoStr     :    out Error_Types.StringError)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out Conversions.State;
   --#        in out CurrentFlowError;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TheBuffer;
   --# derives Conversions.State,
   --#         TheBuffer         from *,
   --#                                CommandLineData.Content,
   --#                                CurrentFlowError,
   --#                                ErrorNumber,
   --#                                ErrType,
   --#                                Name1,
   --#                                Name2,
   --#                                Name3,
   --#                                Pos,
   --#                                Reference,
   --#                                Scope &
   --#         CurrentFlowError  from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                ErrorNumber,
   --#                                ErrType,
   --#                                Name1,
   --#                                Name2,
   --#                                Name3,
   --#                                Pos,
   --#                                Reference,
   --#                                Scope &
   --#         EchoStr           from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                CurrentFlowError,
   --#                                Dictionary.Dict,
   --#                                ErrorNumber,
   --#                                ErrType,
   --#                                LexTokenManager.State,
   --#                                Name1,
   --#                                Name2,
   --#                                Name3,
   --#                                Pos,
   --#                                Reference,
   --#                                Scope &
   --#         ErrFile           from *,
   --#                                CommandLineData.Content,
   --#                                CurrentFlowError,
   --#                                ErrorNumber,
   --#                                ErrType,
   --#                                Name1,
   --#                                Name2,
   --#                                Name3,
   --#                                Pos,
   --#                                Reference,
   --#                                Scope,
   --#                                SPARK_IO.File_Sys,
   --#                                TheBuffer &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                CurrentFlowError,
   --#                                Dictionary.Dict,
   --#                                ErrFile,
   --#                                ErrorNumber,
   --#                                ErrType,
   --#                                Name1,
   --#                                Name2,
   --#                                Name3,
   --#                                Pos,
   --#                                Reference,
   --#                                Scope,
   --#                                TheBuffer;
   is
      NewEntry : Error_Types.NumericError;

      procedure SetToNewErrors (NewEntry : in out Error_Types.NumericError)
      --# derives NewEntry from *;
      is
      begin
         if (NewEntry.ErrorNum = ErrorHandler.DependencyErrNumber (ErrorHandler.MayBeUsed)
           and NewEntry.ErrorType = Error_Types.CondlDependencyErr) then
            NewEntry.ErrorNum := ErrorHandler.DependencyErrNumber (ErrorHandler.MayBeUsedNew);
         elsif (NewEntry.ErrorNum = ErrorHandler.DependencyErrNumber (ErrorHandler.NotUsed)
           and NewEntry.ErrorType = Error_Types.UncondDependencyErr) then
            NewEntry.ErrorNum := ErrorHandler.DependencyErrNumber (ErrorHandler.NotUsedNew);
         end if;

      end SetToNewErrors;

   begin
      NewEntry := Error_Types.NumericError'(ErrorType => ErrType,
                                            Position  => Pos,
                                            Scope     => Scope,
                                            ErrorNum  => ErrorNumber,
                                            Reference => Reference,
                                            Name1     => Name1,
                                            Name2     => Name2,
                                            Name3     => Name3);
      if not CommandLineData.Content.LegacyErrors then
         SetToNewErrors (NewEntry);
      end if;
      ResetErrorNum (NewEntry);
      Conversions.ToString (NewEntry, Error_Types.ForScreen, EchoStr);
      TheBuffer.ErrPtr := TheBuffer.ErrPtr + 1;
      TheBuffer.ErrorList (TheBuffer.ErrPtr) := NewEntry;
      if TheBuffer.ErrPtr = ExaminerConstants.ErrorBufferSize then
         Flush (ErrFile);
      end if;
   end Add;

begin --init
   TheBuffer.ErrPtr := 0;
   CurrentFlowError.ErrorType := Error_Types.NoErr;
   CurrentFlowError.Position  := LexTokenManager.Token_Position'
     (Start_Line_No => 0,
      Start_Pos     => 0);
   CurrentFlowError.ErrorNum  := 0;
   CurrentFlowError.Reference := 0;
   CurrentFlowError.Name1     := Error_Types.NoName;
   CurrentFlowError.Name2     := Error_Types.NoName;
   CurrentFlowError.Name3     := Error_Types.NoName;
   --intentional non-initialization of array will cause flow error here
   --# accept F, 31,  TheBuffer.ErrorList, "Intentional incomplete initialization" &
   --#        F, 32,  TheBuffer.ErrorList, "Intentional incomplete initialization" &
   --#        F, 602, TheBuffer, TheBuffer.ErrorList, "Intentional incomplete initialization" &
   --#        F, 31,  CurrentFlowError.Scope, "Intentional incomplete initialization" &
   --#        F, 32,  CurrentFlowError.Scope, "Intentional incomplete initialization" &
   --#        F, 602, CurrentFlowError, CurrentFlowError.Scope, "Intentional incomplete initialization";
end ErrorBuffer; -- Init. is partial but effective.  Expect 4 errs + 2 warnings.
