-- $Id: errorhandler-conversions-tostring-warningwithoutposition.adb 16021 2010-02-10 15:26:52Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (ErrorHandler.Conversions.ToString)
procedure WarningWithoutPosition
  (ErrNum          : in     Error_Types.NumericError;
   WithExplanation : in     Boolean;
   EStr            : in out ELStrings.T)
is
   UnitTyp : SPSymbols.SPSymbol;
   procedure WarningWithoutPositionExpl (EStr   : in out ELStrings.T)
   --# global in ErrNum;
   --# derives EStr from *,
   --#                   ErrNum;
      is separate;
   -- Note that the parameter names for this subunit are chosen to make it as easy as
   --      possible to auto-generate the subunit from this, its parent, file.  The
   --      generation requires copying the case statement below, stripping out the
   --      current Append'Thing' statements and adding an AppendString for the
   --      explanatory text that is delineated by --! comments.

   procedure AppendExplanation
   --# global in     ErrNum;
   --#        in     WithExplanation;
   --#        in out EStr;
   --# derives EStr from *,
   --#                   ErrNum,
   --#                   WithExplanation;
   is
      ExplanationString : ELStrings.T := ELStrings.Empty_String;
   begin
      if WithExplanation then
         -- we need to at least look for an explanation
         WarningWithoutPositionExpl (ExplanationString);
         if ELStrings.Get_Length (E_Str => ExplanationString) > 0 then
            -- there actually is one
            ELStrings.Append_String (E_Str => EStr,
                                     Str   => ErrorHandler.ExplanationPrefix);
            ELStrings.Append_Examiner_Long_String (E_Str1 => EStr,
                                                   E_Str2 => ExplanationString);
            ELStrings.Append_String (E_Str => EStr,
                                     Str   => ErrorHandler.ExplanationPostfix);
         end if;
      end if;
   end AppendExplanation;

begin

   -- HTML Directives
   --! <NameFormat> <"warning-"><Name>
   --! <ErrorFormat> <"--- Warning : "><Name><" : "><Error>

   case ErrNum.ErrorNum is
      when 9 =>
         UnitTyp := SPSymbols.SPSymbol'Val (ErrNum.Name2.Pos);
         --! <Error> The body of XXX has a hidden exception handler - analysis and verification of contracts for this handler have not been performed.
         --! Issued when a --# hide XXX annotation is used to hide a user-defined exception handler.  (warning control file keyword:<b> handler_parts</b>)
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "The body of ");
         case UnitTyp is
            when SPSymbols.entry_body =>
               ELStrings.Append_String
                 (E_Str => EStr,
                  Str   => "entry ");
            when SPSymbols.subprogram_implementation =>
               ELStrings.Append_String
                 (E_Str => EStr,
                  Str   => "subprogram ");
            when SPSymbols.task_body =>
               ELStrings.Append_String
                 (E_Str => EStr,
                  Str   => "task ");
            when others =>
               ELStrings.Append_String
                 (E_Str => EStr,
                  Str   => "unknown_node_type ");
         end case;
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " has a hidden exception handler - analysis and verification of contracts " &
             "for this handler have not been performed");

      when 10 =>
         UnitTyp := SPSymbols.SPSymbol'Val (ErrNum.Name2.Pos);
         --! <Error> XXX is hidden - hidden text is ignored by the SPARK Examiner
         --! Issued when a --# hide XXX annotation is used.  (warning control file keyword:<b> hidden_parts</b>)
         case UnitTyp is
            when SPSymbols.subprogram_implementation =>
               ELStrings.Append_String
                 (E_Str => EStr,
                  Str   => "The body of subprogram ");
            when SPSymbols.private_part =>
               ELStrings.Append_String
                 (E_Str => EStr,
                  Str   => "The private part of package ");
            when SPSymbols.package_implementation =>
               ELStrings.Append_String
                 (E_Str => EStr,
                  Str   => "The body of package ");
            when SPSymbols.package_initialization =>
               ELStrings.Append_String
                 (E_Str => EStr,
                  Str   => "The initialization of package ");
            when SPSymbols.protected_type_declaration =>
               ELStrings.Append_String
                 (E_Str => EStr,
                  Str   => "The private part of protected type  ");
            when others =>
               null;  -- never happens
         end case;
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " is hidden - hidden text is ignored " &
             "by the SPARK Examiner");

      when 400 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "Variable ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " is declared but not used");
         --! Issued when a variable declared in a subprogram is neither
         --! referenced, nor updated.
         --! (warning control file keyword: <b>unused_variables</b>)

      when 402 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "Default assertion planted to cut loop");
         --! In order to prove properties of code containing loops, the
         --! loop must be &quot;cut&quot; with
         --! a suitable assertion statement.  When generating run-time checks,
         --! the Examiner
         --! inserts a simple assertion to cut any loops which do not have one
         --! supplied
         --! by the user.  The assertion is placed at the point where this
         --! warning appears in
         --! the listing file.  The default assertion asserts that the
         --! subprogram's precondition
         --! (if any) is satisfied, that all imports to it are in their
         --! subtypes and that any for
         --! loop counter is in its subtype.  In many cases this provides
         --! sufficient information
         --! to complete a proof of absence of run-time errors.  If more
         --! information is required,
         --! then the user can supply an assertion and the Examiner will
         --! append the above information
         --! to it. (warning control file keyword: <b>default_loop_assertions</b>)

      when 403 =>
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " is declared as a variable but used as a constant");
         --! XXX is a variable which was initialized at declaration but
         --! whose value is only ever
         --! read not updated; it could therefore have been declared as
         --! a constant. (warning control
         --! file keyword: <b>constant_variables</b>)

      when 404 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "Subprogram imports variables of abstract" &
             " types for which run-time checks cannot be generated");

      when 405 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "VCs for statements including real numbers are approximate");
         --! The Examiner generates VCs associated with
         --! real numbers using perfect arithmetic rather than the machine
         --! approximations used on the
         --! target platform.  It is possible that rounding errors might
         --! cause a Constraint_Error even
         --! if these run-time check proofs are completed satisfactorily.
         --! (warning control file keyword: <b>real_rtcs</b>)

      when 406 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "VC Generator unable to create output files. Permission is required to " &
             "create directories and files in the output directory");
         --! This message is echoed to the screen if the Examiner is unable
         --! to create output files for the VCs being generated
         --! (for instance, if the user does not have write
         --! permission for the output directory).

      when 407 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "This package requires a body.  Care should be taken to " &
               "provide one " &
               "because an Ada compiler will not detect its omission");
         --! Issued where SPARK own variable and initialization annotations
         --! make it clear that a
         --! package requires a body but where no Ada requirement for a body
         --! exists.

      when 408 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "VCs could not be generated for this subprogram owing to " &
               "semantic errors in its " &
               "specification or body.  Unprovable (False) VC generated");
         --! Semantic errors prevent VC Generation, so a single False VC
         --! is produced. This will be detected and reported by POGS.

      when 409 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "VCs could not be generated for this subprogram due to " &
               "its size and/or complexity " &
               "exceeding the capacity of the VC Generator.  Unprovable (False) VC generated");
         --! A subprogram which has excessive complexity of data structure
         --! or number of paths may cause the VC Generator to exceed its capacity.
         --! A single False VC is generated in this case to make sure this
         --! error is detected in subsequent proof and analysis with POGS

      when 410 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "Task or interrupt handler ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " is either unavailable (hidden) or has semantic errors in " &
               "its specification which prevent " &
               "partition-wide flow analysis being carried out");
         --! Partition-wide flow analysis is performed by checking all
         --! packages withed by the main program for
         --! tasks and interrupt handlers and constructing an overall flow
         --! relation that captures their cumulative
         --! effect.  It is for this reason that SPARK requires task and
         --! protected types to be declared in package
         --! specifications.  If a task or protected type which contains
         --! an interrupt handler, is hidden from the
         --! Examiner (in a hidden package private part) or contains errors
         --! in it specification, the partition-wide
         --! flow analysis cannot be
         --! constructed correctly and is therefore suppressed.  Correct the
         --! specification of the affected tasks
         --! and (temporarily if desired) make them visible to the Examiner.

      when 411 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "Task type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " is unavailable and has not been considered in the shared " &
               "variable check");
         --! The Examiner checks that there is no potential sharing of
         --! unprotected data between tasks.  If a task type
         --! is hidden from the Examiner in a hidden package private
         --! part, then it is not possible to check whether that
         --! task may share unprotected data.

      when 412 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "Task type ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " is unavailable and has not been considered in the " &
               "max-one-in-a-queue check");
         --! The Examiner checks that no more than one task can suspend on
         --! a single object.  If a task
         --! is hidden from the Examiner in a hidden package private part,
         --! then it is not possible to check whether that
         --! task may suspend on the same object as another task.

      when 413 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "Task or main program ");
         AppendName
            (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " has errors in its annotations. The shared variable and " &
               "max-one-in-a-queue checks may be incomplete");
         --! The Examiner checks that no more than one task can suspend on a
         --! single object and that there is no
         --! potential sharing of unprotected data between tasks.  These checks
         --! depend on the accuracy of the annotations
         --! on the task types withed by the main program.  If these annotations
         --! contain errors, then any reported
         --! violations of the shared variable and max-one-in-a-queue checks will
         --! be correct; however, the check
         --! may be incomplete.  The errors in the task annotations should be corrected.

      when 414 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "Long output file name has been truncated");
         --! Raised if an output file name is longer than the
         --! limit imposed by the operating system and has been truncated.
         --! Section 4.7 of the Examiner User Manual describes how the output file names
         --! are constructed. If this message is seen there is a possibility
         --! that the output from two
         --! or more subprograms will be written to the same file name,
         --! if they have a sufficiently large number of characters in common.


      when 420 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "Instance of SEPR 2124 found. An extra VC will " &
              "be generated here and must be discharged to " &
              "ensure absence of run-time errors. Please seek advice " &
              "for assistance with this issue");
         --! In release 7.5 of the SPARK Examiner, a flaw in the VC generation
         --! was fixed such that subcomponents of records and elements of
         --! arrays when used as &quot;out&quot; or &quot;in out&quot;
         --! parameters will now generate an
         --! additional VC to verify absence of run-time errors. This warning
         --! flags an instance of this occurrence. Please read the release
         --! note and/or seek advice for assistance with this issue.

      when 430 =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "SLI generation abandoned owing to syntax or semantic errors " &
                     "or multiple units in a single source file");

      when others =>
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => "UNKNOWN ERROR NUMBER PASSED TO WarningWithoutPosition");

   end case;
   AppendExplanation;
   ELStrings.Append_String (E_Str => EStr,
                            Str   => ".");
end WarningWithoutPosition;
