-- $Id: errorhandler-conversions-tostring-usageerror.adb 14620 2009-10-28 13:40:30Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (ErrorHandler.Conversions.ToString)
procedure UsageError (ErrNum          : in     Error_Types.NumericError;
                      WithExplanation : in     Boolean;
                      EStr            : in out ELStrings.T)
is
   ErrType : ErrorHandler.UsageErrType;

   procedure UsageErrorExpl (EStr   : in out ELStrings.T)
   --# global in ErrType;
   --# derives EStr from *,
   --#                   ErrType;
      is separate;
   -- Note that the parameter names for this subunit are chosen to make it as easy as
   --      possible to auto-generate the subunit from this, its parent, file.  The
   --      generation requires copying the case statement below, stripping out the
   --      current Append'Thing' statements and adding an AppendString for the
   --      explanatory text that is delineated by --! comments.

   procedure AppendExplanation
   --# global in     ErrType;
   --#        in     WithExplanation;
   --#        in out EStr;
   --# derives EStr from *,
   --#                   ErrType,
   --#                   WithExplanation;
   is
      ExplanationString : ELStrings.T := ELStrings.Empty_String;
   begin
      if WithExplanation then
         -- we need to at least look for an explanation
         UsageErrorExpl (ExplanationString);
         if ELStrings.Get_Length (E_Str => ExplanationString) > 0 then
            -- there actually is one
            ELStrings.Append_String (E_Str => EStr,
                                     Str   => ErrorHandler.ExplanationPrefix);
            ELStrings.Append_Examiner_Long_String (E_Str1 => EStr,
                                                   E_Str2 => ExplanationString);
            ELStrings.Append_String (E_Str => EStr,
                                     Str   => ErrorHandler.ExplanationPostfix);
         end if;
      end if;
   end AppendExplanation;

begin
   ErrType := ErrorHandler.UsageErrType'Val (ErrNum.ErrorNum -
                                             Error_Types.UsageErrOffset);
   case ErrType is

      -- HTML Directives
      --! <NameFormat> <"flow-"><Name>
      --! <ErrorFormat> <"!!! Flow Error : "><Name><" : "><Error>

      when ErrorHandler.UnusedImport =>
         --! <Name> 30
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "The variable ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " is imported but neither referenced nor exported");

      when ErrorHandler.UndefinedExport =>
         --! <Name> 31

         if ErrNum.Name1 = Error_Types.NoName then
            ELStrings.Append_String
              (E_Str => EStr,
               Str   => "The returned function value is not defined");
         else
            ELStrings.Append_String (E_Str => EStr,
                                     Str   => "The variable ");
            AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
            ELStrings.Append_String
              (E_Str => EStr,
               Str   => " is exported but not (internally) defined");
         end if;
         --! <Error> The variable XXX is exported but not (internally) defined.

      when ErrorHandler.UndefinedVar =>
         --! <Name> 32
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "The variable ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => " is neither imported nor defined");

      when ErrorHandler.UnreferencedVar =>
         --! <Name> 33
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "The variable ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => " is neither referenced nor exported");

      when ErrorHandler.RedefinedImport =>
         --! <Name> 34
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "The imported, non-exported variable ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => " may be redefined");
         --! The updating of imported-only variables is forbidden under all
         --! circumstances.

      when ErrorHandler.IneffectiveImport =>
         --! <Name> 35
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "Importation of the initial value of variable ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => " is ineffective");
         --! The meaning of this message is explained in Section 4.2 of Appendix A

      when ErrorHandler.ReferencedButNotInPartition =>
         --! <Name> 36
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "The referencing of variable ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " by a task or interrupt handler has been omitted from the partition annotation");
         --! This message is only issued when processing the partition annotation.  The partition annotation
         --! must describe all the actions of the tasks and interrupt handlers making up the program.  Therefore,
         --! if a variable is imported somewhere in the program by a task or interrupt handler, then it must
         --! also be an import at the partition level.  As well as the omission of explicit imports, this message is also
         --! generated if the implicit imports of tasks and interrupt handlers are omitted.  For tasks this means
         --! any variable the task suspends on and for interrupt handlers it means the name of the protected
         --! object containing the handler or, if given, the name of the interrupt stream associated with the
         --! handler.

      when ErrorHandler.UpdatedButNotInPartition =>
         --! <Name> 37
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "The updating of variable ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " by a task or interrupt handler has been omitted from the partition annotation");
         --! This message is only issued when processing the partition annotation.  The partition annotation
         --! must describe all the actions of the tasks and interrupt handlers making up the program.  Therefore,
         --! if a variable is exported somewhere in the program by a task or interrupt handler, then it must
         --! also be an export at the partition level.

      when ErrorHandler.UninitializedProtectedElement =>
         --! <Name> 38
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "The protected element ");
         AppendName (EStr, ErrNum.Name1, ErrNum.Scope);
         ELStrings.Append_String
           (E_Str => EStr,
            Str   => " must be initialized at its point of declaration");
         --! To avoid potential race conditions during program startup, all
         --! elements of a protected type must be initialized with a constant value
         --! at the point of declaration.
   end case;
   AppendExplanation;
   ELStrings.Append_String (E_Str => EStr,
                            Str   => ".");
end UsageError;
