-- $Id: errorhandler-conversions-tostring-stabilityerror.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (ErrorHandler.Conversions.ToString)
procedure StabilityError (ErrNum          : in     Error_Types.NumericError;
                          WithExplanation : in     Boolean;
                          EStr            : in out ELStrings.T)
is
   StabIndex : ErrorHandler.IndexType;
   StabTyp   : ErrorHandler.StabilityErrType;

   procedure StabilityErrorExpl (EStr   : in out ELStrings.T)
   --# global in StabTyp;
   --# derives EStr from *,
   --#                   StabTyp;
      is separate;
   -- Note that the parameter names for this subunit are chosen to make it as easy as
   --      possible to auto-generate the subunit from this, its parent, file.  The
   --      generation requires copying the case statement below, stripping out the
   --      current Append'Thing' statements and adding an AppendString for the
   --      explanatory text that is delineated by --! comments.

   procedure AppendExplanation
   --# global in     StabTyp;
   --#        in     WithExplanation;
   --#        in out EStr;
   --# derives EStr from *,
   --#                   StabTyp,
   --#                   WithExplanation;
   is
      ExplanationString : ELStrings.T := ELStrings.Empty_String;
   begin
      if WithExplanation then
         -- we need to at least look for an explanation
         StabilityErrorExpl (ExplanationString);
         if ELStrings.Get_Length (E_Str => ExplanationString) > 0 then
            -- there actually is one
            ELStrings.Append_String (E_Str => EStr,
                                     Str   => ErrorHandler.ExplanationPrefix);
            ELStrings.Append_Examiner_Long_String (E_Str1 => EStr,
                                                   E_Str2 => ExplanationString);
            ELStrings.Append_String (E_Str => EStr,
                                     Str   => ErrorHandler.ExplanationPostfix);
         end if;
      end if;
   end AppendExplanation;

begin
   StabIndex := ErrorHandler.IndexType'Val (ErrNum.Name1.Pos);
   StabTyp   := ErrorHandler.StabilityErrType'Val (ErrNum.ErrorNum -
                                                   Error_Types.StabilityErrOffset);

   case StabTyp is

      -- HTML Directives
      --! <NameFormat> <"flow-"><Name>
      --! <ErrorFormat> <"!!! Flow Error : "><Name><" : "><Error>

      when ErrorHandler.StableExitCond =>
         --! <Name> 40

         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "Exit condition is stable, of index");

         --! <Error> Exit condition is stable, of index 0
         --! <Error2> Exit condition is stable, of index 1
         --! <Error3> Exit condition is stable, of index greater than 1

         --! In these cases the (loop) exit condition occurs in an iteration scheme,
         --! an exit statement, or an if-statement whose (unique) sequence of
         --! statements ends with an unconditional exit statement - see the SPARK
         --! Definition. The concept of loop stability is explained in Section
         --! 4.4 of Appendix A. A loop exit condition which is stable of index 0
         --! takes the same value at every iteration around the loop, and with a
         --! stability index of 1, it always takes the same value after the first
         --! iteration. Stability with indices greater
         --! than 0 does not necessarily indicate a program error, but the
         --! conditions for loop termination require careful consideration

      when ErrorHandler.StableForkCond =>
         --! <Name> 41

         ELStrings.Append_String (E_Str => EStr,
                                  Str   => "Expression is stable, of index");

         --! <Error> Expression is stable, of index 0
         --! <Error2> Expression is stable, of index 1
         --! <Error3> Expression is stable, of index greater than 1

         --! The expression, occurring within a loop, is either a case expression
         --! or a condition (Boolean-valued expression) associated with an
         --! if-statement, whose value determines the path taken through the body
         --! of the loop, but does not (directly) cause loop termination.
         --! Information flow analysis shows that the expression does not vary
         --! as the loop is executed, so the same branch of the case or if statement will
         --! be taken on every loop iteration. An Index of 0 means that the expression is
         --! immediately stable, 1 means it becomes stable after the first pass through the loop and so on.
         --! The stability index is given with reference to the loop most
         --! closely-containing the expression.  Stable conditionals are not necessarily
         --! an error but do require careful evaluation; they can often be removed by lifting them
         --! outside the loop.
   end case;

   case StabIndex is
      when ErrorHandler.IndexZero =>
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => " 0");
      when ErrorHandler.IndexOne =>
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => " 1");
      when ErrorHandler.LargerIndex =>
         ELStrings.Append_String (E_Str => EStr,
                                  Str   => " greater than 1");
   end case;
   AppendExplanation;
   ELStrings.Append_String (E_Str => EStr,
                            Str   => ".");
end StabilityError;
