-- $Id: errorhandler-appenderrors.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (ErrorHandler)
procedure AppendErrors (Report  : in SPARK_IO.File_Type;
                        Purpose : in Error_Types.ConversionRequestSource)
is
   ErrCount      : Natural;
   OK            : Boolean;
   NextError     : Error_Types.StringError;
   NumErr        : Error_Types.NumericError;
   Success       : SPARK_IO.File_Status;
   TempErrorFile : Error_IO.File_Type;
   Accumulator   : ErrorAccumulator.T := ErrorAccumulator.Clear;

   procedure PutSourceLine (ToFile : in SPARK_IO.File_Type;
                            LineNo : in LexTokenManager.Line_Numbers)
   --# global in     CommandLineData.Content;
   --#        in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorContextRec   from *,
   --#                                LineNo,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                ErrorContextRec,
   --#                                LineNo,
   --#                                ToFile;
   --  pre     ErrorContextRec.LineNo /= LineNo;
   --  post    ErrorContextRec.LineNo =  LineNo;
   is
   begin
      loop
         GetFileLine;
         exit when ErrorContextRec.LineNo >= LineNo;
      end loop;
      if not CommandLineData.Content.XML then
         PrintSourceLine (ToFile);
      end if;
   end PutSourceLine;

   procedure GetErrorSet (ErrorSet  :    out ErrorSets;
                          NextError : in out Error_Types.StringError;
                          OK        :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ErrorContextRec;
   --#        in     LexTokenManager.State;
   --#        in     Purpose;
   --#        in out Conversions.State;
   --#        in out ErrCount;
   --#        in out SPARK_IO.File_Sys;
   --# derives Conversions.State,
   --#         NextError,
   --#         OK,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.State,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys &
   --#         ErrCount,
   --#         ErrorSet          from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrCount,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.State,
   --#                                NextError,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys;
   --  post    not OK or
   --          ErrorContextRec.LineNo /= NextError.Position.StartLineNo or
   --          Errors.Length = ExaminerConstants.MaxErrorSetSize;
   is
      LOK    : Boolean;
      Errors : ErrorSets;
      NumErr : Error_Types.NumericError;

   begin
      Errors := EmptyErrorSet;
      loop
         Errors.Length := Errors.Length + 1;
         Errors.Content (Errors.Length).Error := NextError;
         if ErrorAccumulator.IsErrorContinuation (NextError)  then
            Errors.Content (Errors.Length).ErrNum := 0;
         else
            ErrCount := ErrCount + 1;
            Errors.Content (Errors.Length).ErrNum := ErrCount;
         end if;
         loop
            Error_IO.Get_Numeric_Error (ErrorContextRec.Errs, NumErr);
            Conversions.ToString (NumErr, Purpose, NextError);
            LOK := (NextError /= Error_Types.Empty_StringError);

            exit when not LOK;
            exit when NextError.ErrorType /= Error_Types.NoErr;
         end loop;
         exit when not LOK;
         exit when ErrorContextRec.LineNo /= NextError.Position.Start_Line_No;
         exit when Errors.Length = ExaminerConstants.MaxErrorSetSize;
      end loop;

      OK := LOK;
      ErrorSet := Errors;
   end GetErrorSet;

   procedure ProcessErrorSet (Listing   : in     SPARK_IO.File_Type;
                              NextError : in out Error_Types.StringError;
                              OK        :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ErrorContextRec;
   --#        in     LexTokenManager.State;
   --#        in     Purpose;
   --#        in out Accumulator;
   --#        in out Conversions.State;
   --#        in out ErrCount;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives Accumulator,
   --#         ErrCount          from *,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrCount,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.State,
   --#                                NextError,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys &
   --#         Conversions.State,
   --#         NextError,
   --#         OK                from CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.State,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys &
   --#         SPARK_IO.File_Sys from *,
   --#                                Accumulator,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrCount,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.State,
   --#                                Listing,
   --#                                NextError,
   --#                                Purpose,
   --#                                XMLReport.State &
   --#         XMLReport.State   from *,
   --#                                Accumulator,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrCount,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.State,
   --#                                NextError,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys;
   is
      ErrorSet : ErrorSets;
   begin
      GetErrorSet (ErrorSet, NextError, OK);
      if not CommandLineData.Content.XML then
         PutErrorPointers (Listing, ErrorSet);
         PutErrorMessages (Listing, ErrorSet, 29, Accumulator);
      else
         PutErrorMessagesXML (Listing, ErrorSet, 29, Accumulator);
      end if;

   end ProcessErrorSet;

   procedure ProcessErrorsOnLine (Listing   : in     SPARK_IO.File_Type;
                                  NextError : in out Error_Types.StringError;
                                  OK        :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ErrorContextRec;
   --#        in     LexTokenManager.State;
   --#        in     Purpose;
   --#        in out Accumulator;
   --#        in out Conversions.State;
   --#        in out ErrCount;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives Accumulator,
   --#         Conversions.State,
   --#         ErrCount,
   --#         NextError,
   --#         OK,
   --#         SPARK_IO.File_Sys,
   --#         XMLReport.State   from Accumulator,
   --#                                CommandLineData.Content,
   --#                                Conversions.State,
   --#                                Dictionary.Dict,
   --#                                ErrCount,
   --#                                ErrorContextRec,
   --#                                LexTokenManager.State,
   --#                                Listing,
   --#                                NextError,
   --#                                Purpose,
   --#                                SPARK_IO.File_Sys,
   --#                                XMLReport.State;
   is
      LOK : Boolean;
      AccumulatorWasActive : Boolean;
   begin
      if not ErrorHasPositionInline (NextError.ErrorType) and then
        not CommandLineData.Content.XML then
         SPARK_IO.New_Line (Listing, 1);
      end if;
      loop
         ProcessErrorSet (Listing, NextError, LOK);
         exit when not LOK;
         exit when ErrorContextRec.LineNo /= NextError.Position.Start_Line_No;
      end loop;
      OK := LOK;
      AccumulatorWasActive := ErrorAccumulator.IsActive (Accumulator);
      ErrorAccumulator.Flush (Accumulator, Listing);
      if AccumulatorWasActive then
         if CommandLineData.Content.XML then
            XMLReport.EndMessage (Listing);
         else
            New_Line (Listing, 1);
         end if;
      end if;
   end ProcessErrorsOnLine;

   procedure SetUpFiles (OK : out Boolean)
   --# global in out ErrorContextRec;
   --#        in out SPARK_IO.File_Sys;
   --#           out ErrCount;
   --# derives ErrCount          from  &
   --#         ErrorContextRec,
   --#         SPARK_IO.File_Sys from *,
   --#                                ErrorContextRec &
   --#         OK                from ErrorContextRec,
   --#                                SPARK_IO.File_Sys;
   is
      LOK        : Boolean;
      Success    : SPARK_IO.File_Status;
      SourceFile : SPARK_IO.File_Type;
      ErrorFile  : Error_IO.File_Type;
   begin
      SourceFile := ErrorContextRec.Source;
      SPARK_IO.Reset (SourceFile, SPARK_IO.In_File, Success);
      ErrorContextRec.Source := SourceFile;
      LOK := Success = SPARK_IO.Ok;
      ErrorContextRec.LineNo := 0;
      ErrCount := 0;
      ErrorFile := ErrorContextRec.Errs;
      Error_IO.Reset (ErrorFile, SPARK_IO.In_File, Success);
      ErrorContextRec.Errs   := ErrorFile;
      OK := LOK and Success = SPARK_IO.Ok;
   end SetUpFiles;

   procedure PutErrorCount (File : in SPARK_IO.File_Type;
                            Cnt  : in Natural)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                Cnt,
   --#                                File;
   is
   begin
      SPARK_IO.Put_Integer (File, Cnt, 0, 10);
      SPARK_IO.Put_Line (File, " error(s) or warning(s)", 0);
      SPARK_IO.New_Line (File, 1);
   end PutErrorCount;

begin
   if ErrorContextRec.NumErrs = 0 then
      if not CommandLineData.Content.XML then
         SPARK_IO.Put_Line (Report, "No errors found", 0);
         SPARK_IO.New_Line (Report, 1);
      end if;
   else
      ErrorContextRec.CurrentLine := EStrings.Empty_String;
      if not CommandLineData.Content.XML then
         PutErrorCount (Report, Integer (ErrorContextRec.NumErrs));
      end if;
      SetUpFiles (OK);
      if OK then
         if not CommandLineData.Content.XML then
            SPARK_IO.Put_Line (Report, "Line", 0);
         end if;
         loop
            Error_IO.Get_Numeric_Error (ErrorContextRec.Errs, NumErr);
            Conversions.ToString (NumErr, Purpose, NextError);
            OK := (NextError /= Error_Types.Empty_StringError);

            exit when not OK;
            exit when NextError.ErrorType /= Error_Types.NoErr;
         end loop;
         --    assert ErrorContextRec.LineNo = 0 and
         --         OK -> NextError.Position.StartLineNo >= 1;
         loop
            exit when not OK;
            --  assert ErrorContextRec.LineNo /= NextError.Position.StartLineNo;
            PutSourceLine (Report, NextError.Position.Start_Line_No);
            --  assert ErrorContextRec.LineNo = NextError.Position.StartLineNo;
            ProcessErrorsOnLine (Report, NextError, OK);
            --  assert OK -> ErrorContextRec.LineNo /= NextError.Position.StartLineNo;
            --# accept Flow, 41, "Expected stable expression";
            if not CommandLineData.Content.XML then
            --# end accept;
               SPARK_IO.New_Line (Report, 1);
            end if;
         end loop;
      else
         SPARK_IO.Put_Line (Report, "***      Bad error list, unable to report errors", 0);
         SPARK_IO.New_Line (Report, 1);
      end if;
   end if;

   if CommandLineData.Content.XML then
      Justifications.PrintJustificationsXML (ErrorContextRec.JustificationsDataTable,
                                             Report);
   else
      Justifications.PrintJustifications (ErrorContextRec.JustificationsDataTable,
                                          Report);
   end if;

   WarningStatus.ReportSuppressedWarnings (Report,
                                           ErrorContextRec.Counter);

   TempErrorFile := ErrorContextRec.Errs;
   --# accept Flow, 10, Success, "Expected ineffective assignment to Success";
   Error_IO.Close (TempErrorFile, Success);
   --# end accept;
   ErrorContextRec.Errs := TempErrorFile;
   --# accept Flow, 33, Success, "Expected Success to be neither referenced nor exported";
end AppendErrors;
