-- $Id: elstrings.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Ada.Characters.Handling;
with Ada.Characters.Latin_1;
with SystemErrors;

package body ELStrings
is

   function ToExaminerLongString (EStr : EStrings.T)
                                 return T
   is
      Result : T := Empty_String;
   begin
      for I in Natural range 1 .. EStrings.Get_Length (E_Str => EStr) loop
         Result.Content (I) := EStrings.Get_Element (E_Str => EStr,
                                                     Pos   => (I + EStrings.Positions'First) - 1); -- flow error OK, array init
      end loop;
      Result.Length := EStrings.Get_Length (E_Str => EStr);
      return Result;
   end ToExaminerLongString; -- Flow Error OK on UnusedLen

   function ToExaminerString (EStr : T)
                             return EStrings.T
   is
      L       : EStrings.Lengths;
      Result  : EStrings.T := EStrings.Empty_String;
      Success : Boolean;
   begin
      if EStr.Length > EStrings.Lengths'Last then
         L := EStrings.Lengths'Last;
      else
         L := EStr.Length;
      end if;
      for I in Natural range 1 .. L loop
         --# accept F, 10, Success, "Ineffective assignment here OK";
         EStrings.Append_Char (E_Str   => Result,
                               Ch      => EStr.Content ((I + Positions'First) - 1), -- flow error OK, array init
                               Success => Success);
         --# end accept;
      end loop;
      --# accept F, 33, Success, "Expect Success unused";
      return Result;
   end ToExaminerString;

   --------------------------------------------

   -- Case (in)sensitive comparison.  Returns true if and only if the strings
   -- are equal, given the specified lengths.
   function Eq_String_Local (Str1           : String;
                             Length1        : Natural;
                             Str2           : String;
                             Length2        : Natural;
                             Case_Sensitive : Boolean) return Boolean
   is
      Result : Boolean;

      function Get_Char (Ch             : Character;
                         Case_Sensitive : Boolean) return Character
      is
         Result : Character;
      begin
         if Case_Sensitive then
            Result := Ch;
         else
            Result := Ada.Characters.Handling.To_Lower (Ch);
         end if;
         return Result;
      end Get_Char;

   begin
      if Length1 = Length2 then
         Result := True;
         for I in Natural range 1 .. Length1 loop
            Result := Result and then
              Get_Char (Ch             => Str1 (I),
                        Case_Sensitive => Case_Sensitive) =
              Get_Char (Ch             => Str2 (I),
                        Case_Sensitive => Case_Sensitive);
            exit when not Result;
         end loop;
      else
         Result := False;
      end if;
      return Result;
   end Eq_String_Local;

   -- Case INsensitive comparison.
   function Eq_String (E_Str1, E_Str2 : T) return Boolean
   is
   begin
      return Eq_String_Local (Str1           => E_Str1.Content,
                              Length1        => E_Str1.Length,
                              Str2           => E_Str2.Content,
                              Length2        => E_Str2.Length,
                              Case_Sensitive => False);
   end Eq_String;

   -- Case INsensitive comparison.
   function Eq1_String (E_Str : T;
                        Str   : String) return Boolean
   is
   begin
      return Eq_String_Local (Str1           => E_Str.Content,
                              Length1        => E_Str.Length,
                              Str2           => Str,
                              Length2        => Str'Length,
                              Case_Sensitive => False);
   end Eq1_String;

   -- Case sensitive comparison.
   function Eq_CS_String (E_Str1, E_Str2 : T) return Boolean
   is
   begin
      return Eq_String_Local (Str1           => E_Str1.Content,
                              Length1        => E_Str1.Length,
                              Str2           => E_Str2.Content,
                              Length2        => E_Str2.Length,
                              Case_Sensitive => True);
   end Eq_CS_String;

   function Is_Empty (E_Str : T) return Boolean
   is
   begin
      return Eq_String (E_Str1 => E_Str,
                        E_Str2 => Empty_String);
   end Is_Empty;

   function Get_Length (E_Str : T) return Lengths
   is
   begin
      return E_Str.Length;
   end Get_Length;

   function Get_Element (E_Str : T;
                         Pos   : Positions) return Character
   is
      Return_Value : Character;
   begin
      if Pos <= E_Str.Length then
         Return_Value := E_Str.Content (Pos);
      else
         Return_Value := ' ';
      end if;
      return Return_Value;
   end Get_Element;

   function Copy_String (Str : String) return T
   is
      E_Str : T := Empty_String;

      procedure Fatal_String_Overflow_Error
      --# derives ;
      is
         --# hide Fatal_String_Overflow_Error;
      begin
         SystemErrors.FatalError (SystemErrors.StringOverFlow, "");
      end Fatal_String_Overflow_Error;

   begin
      --# accept Flow, 10, "Effective as triggers Fatal Error";
      if E_Str.Content'Length < Str'Length then
         Fatal_String_Overflow_Error;
      end if;
      --# end accept;

      for I in Natural range 1 .. Str'Length loop
         E_Str.Content (I) := Str ((I + Str'First) - 1); -- flow error OK, array init
      end loop;
      E_Str.Length := Str'Length;

      return E_Str;
   end Copy_String;

   procedure Append_String_Local (Str1    : in out String;
                                  Length1 : in out Natural;
                                  Str2    : in     String;
                                  Length2 : in     Natural)
   --# derives Str1    from Length1,
   --#                      Length2,
   --#                      Str1,
   --#                      Str2 &
   --#         Length1 from Length1,
   --#                      Length2,
   --#                      Str1;
   is
      Limit           : Natural;
      Chars_To_Append : Natural;
   begin
      -- If the combined lengths exceed the maximum length
      -- of the result string then limit the length of the
      -- result and only copy as many characters as will fit,
      -- ie the difference between the current length of the
      -- result string and the max length of the result string.
      Limit := Length1 + Length2;
      if Limit > Str1'Length then
         Limit := Str1'Length;
         Chars_To_Append := Str1'Length - Length1;
      else
         Chars_To_Append := Length2;
      end if;

      for I in Natural range 1 .. Chars_To_Append loop
         Str1 (Length1 + I) := Str2 ((I + Str2'First) - 1);
      end loop;
      Length1 := Limit;
   end Append_String_Local;

   procedure Append_String (E_Str : in out T;
                            Str   : in     String)
   is
      Result_String : Contents;
      Result_Length : Lengths;
   begin
      Result_String := E_Str.Content;
      Result_Length := E_Str.Length;

      Append_String_Local (Str1    => Result_String,
                           Length1 => Result_Length,
                           Str2    => Str,
                           Length2 => Str'Length);

      E_Str.Content := Result_String;
      E_Str.Length  := Result_Length;
   end Append_String;

   procedure Append_Examiner_String (E_Str1 : in out T;
                                     E_Str2 : in     EStrings.T)
   is
      Result_String : Contents;
      Result_Length : Lengths;
      Tmp_Str       : T;
   begin
      Result_String := E_Str1.Content;
      Result_Length := E_Str1.Length;
      Tmp_Str := ToExaminerLongString (EStr => E_Str2);

      Append_String_Local (Str1    => Result_String,
                           Length1 => Result_Length,
                           Str2    => Tmp_Str.Content,
                           Length2 => Tmp_Str.Length);

      E_Str1.Content := Result_String;
      E_Str1.Length  := Result_Length;
   end Append_Examiner_String;


   procedure Append_Examiner_Long_String (E_Str1 : in out T;
                                          E_Str2 : in     T)
   is
      Result_String : Contents;
      Result_Length : Lengths;
   begin
      Result_String := E_Str1.Content;
      Result_Length := E_Str1.Length;

      Append_String_Local (Str1    => Result_String,
                           Length1 => Result_Length,
                           Str2    => E_Str2.Content,
                           Length2 => E_Str2.Length);

      E_Str1.Content := Result_String;
      E_Str1.Length  := Result_Length;
   end Append_Examiner_Long_String;

   function Lower_Case (E_Str : T) return T
   is
      Res : T := Empty_String;
   begin
      for I in Natural range 1 .. E_Str.Length loop
         Res.Content (I) := Ada.Characters.Handling.To_Lower (E_Str.Content (I));
      end loop;
      Res.Length := E_Str.Length;
      return Res;
   end Lower_Case;

   function Upper_Case (E_Str : T) return T
   is
      Res : T := Empty_String;
   begin
      for I in Natural range 1 .. E_Str.Length loop
         Res.Content (I) := Ada.Characters.Handling.To_Upper (E_Str.Content (I));
      end loop;
      Res.Length := E_Str.Length;
      return Res;
   end Upper_Case;

   function Lower_Case_Char (E_Str : T;
                             Pos   : Positions) return T
   is
      Result : T;
   begin
      Result := E_Str;
      Result.Content (Pos) := Ada.Characters.Handling.To_Lower (Result.Content (Pos));
      return Result;
   end Lower_Case_Char;

   function Upper_Case_Char (E_Str : T;
                             Pos   : Positions) return T
   is
      Result : T;
   begin
      Result := E_Str;
      Result.Content (Pos) := Ada.Characters.Handling.To_Upper (Result.Content (Pos));
      return Result;
   end Upper_Case_Char;

   function Translate (E_Str     : T;
                       From_Char : Character;
                       To_Char   : Character) return T
   is
      Result : T;
   begin
      Result := E_Str;
      for I in Positions range 1 .. Result.Length loop
         if Result.Content (I) = From_Char then
            Result.Content (I) := To_Char;
         end if;
      end loop;
      return Result;
   end Translate;

   procedure Append_Char (E_Str   : in out T;
                          Ch      : in     Character;
                          Success :    out Boolean)
   is
   begin
      if E_Str.Length = E_Str.Content'Length then
         Success := False;
      else
         Success := True;
         E_Str.Length := E_Str.Length + 1;
         E_Str.Content (E_Str.Length) := Ch;
      end if;
   end Append_Char;

   -----------------------------------------------------------------------
   -- FindSubStringAfter:
   -- find the specified SearchString, starting at the specified position in
   -- the given String
   procedure Find_Sub_String_After_Local (Str               : in     String;
                                          Str_Length        : in     Integer;
                                          Search_Start      : in     Positions;
                                          Search_String     : in     String;
                                          Search_Str_Length : in     Integer;
                                          String_Found      :    out Boolean;
                                          String_Start      :    out Positions)
   --# derives String_Found,
   --#         String_Start from Search_Start,
   --#                           Search_String,
   --#                           Search_Str_Length,
   --#                           Str,
   --#                           Str_Length;
   is
      Current_Start_Pos      : Positions;
      Current_Compare_Offset : Integer;
      Could_Be_Here          : Boolean;
      Found                  : Boolean := False;
      Last_Start_Pos         : Positions;
   begin
      Current_Start_Pos := Search_Start;
      if Search_Str_Length <= Str_Length then
         Last_Start_Pos := (Str_Length + 1) - Search_Str_Length;

         while not Found and then
           Current_Start_Pos <= Last_Start_Pos loop

            Current_Compare_Offset := 0;
            Could_Be_Here := True;

            while not Found and then
              Could_Be_Here and then
              Current_Compare_Offset < Search_Str_Length loop

               if Str (Current_Start_Pos + Current_Compare_Offset) /=
                 Search_String (1 + Current_Compare_Offset) then
                  Could_Be_Here := False;
               else
                  if Current_Compare_Offset = Search_Str_Length - 1 then
                     Found := True;
                  else
                     Current_Compare_Offset := Current_Compare_Offset + 1;
                  end if;
               end if;
            end loop;

            if not Found then
               Current_Start_Pos := Current_Start_Pos + 1;
            end if;
         end loop;
      end if;

      String_Found := Found;
      String_Start := Current_Start_Pos;

   end Find_Sub_String_After_Local;

   -------------------------------------------------------------------------
   -- Find_Sub_String_After: for use in summary tool
   -- find the specified SearchString, starting at the specified position in
   -- the given T
   -------------------------------------------------------------------------
   procedure Find_Sub_String_After (E_Str         : in     T;
                                    Search_Start  : in     Positions;
                                    Search_String : in     String;
                                    String_Found  :    out Boolean;
                                    String_Start  :    out Positions)
   is
   begin
      Find_Sub_String_After_Local (Str               => E_Str.Content,
                                   Str_Length        => E_Str.Length,
                                   Search_Start      => Search_Start,
                                   Search_String     => Search_String,
                                   Search_Str_Length => Search_String'Length,
                                   String_Found      => String_Found,
                                   String_Start      => String_Start);
   end Find_Sub_String_After;

   --------------------------------------------------------------------------
   -- Find_Sub_String: for use in summary tool
   -- find specified SearchString in the given T
   --------------------------------------------------------------------------
   procedure Find_Sub_String (E_Str         : in     T;
                              Search_String : in     String;
                              String_Found  :    out Boolean;
                              String_Start  :    out Positions)
   is
   begin
      Find_Sub_String_After_Local (Str               => E_Str.Content,
                                   Str_Length        => E_Str.Length,
                                   Search_Start      => 1,
                                   Search_String     => Search_String,
                                   Search_Str_Length => Search_String'Length,
                                   String_Found      => String_Found,
                                   String_Start      => String_Start);
   end Find_Sub_String;

   procedure Find_Examiner_Sub_String (E_Str         : in     T;
                                       Search_String : in     T;
                                       String_Found  :    out Boolean;
                                       String_Start  :    out Positions)
   is
   begin
      Find_Sub_String_After_Local (Str               => E_Str.Content,
                                   Str_Length        => E_Str.Length,
                                   Search_Start      => 1,
                                   Search_String     => Search_String.Content,
                                   Search_Str_Length => Search_String.Length,
                                   String_Found      => String_Found,
                                   String_Start      => String_Start);
   end Find_Examiner_Sub_String;

   procedure Pop_Char (E_Str : in out T;
                       Char  :    out Character)
   is
   begin
      if E_Str.Length = 0 then
         Char := ' ';
      else
         Char := E_Str.Content (E_Str.Content'First);
         for I in Natural range E_Str.Content'First + 1 .. E_Str.Content'Length loop
            E_Str.Content (I - 1) := E_Str.Content (I);
         end loop;
         E_Str.Length := E_Str.Length - 1;
      end if;
   end Pop_Char;

   ---------------------------------------------------------------------------
   -- Find_Char_After: find specified Character in EStr, starting at specified
   -- position
   ---------------------------------------------------------------------------
   procedure Find_Char_After (E_Str        : in     T;
                              Search_Start : in     Positions;
                              Search_Char  : in     Character;
                              Char_Found   :    out Boolean;
                              Char_Pos     :    out Positions)
   is
      Position : Positions;
      Found    : Boolean := False;
      Finished : Boolean := False;
   begin
      Position := Search_Start;
      while not Finished loop
         if Position <= E_Str.Length and then
           E_Str.Content (Position) = Search_Char then
            Found    := True;
            Finished := True;
         elsif Position < E_Str.Length then
            Position := Position + 1;
         else
            Finished := True;
            Found    := False;
         end if;
      end loop;

      Char_Pos   := Position;
      Char_Found := Found;
   end Find_Char_After;

   --------------------------------------------------------------------------
   -- Find_Char: find first occurrence of specified Character in EStr
   --------------------------------------------------------------------------
   procedure Find_Char (E_Str       : in     T;
                        Search_Char : in     Character;
                        Char_Found  :    out Boolean;
                        Char_Pos    :    out Positions)
   is
   begin
      Find_Char_After (E_Str        => E_Str,
                       Search_Start => 1,
                       Search_Char  => Search_Char,
                       Char_Found   => Char_Found,
                       Char_Pos     => Char_Pos);
   end Find_Char;

   --------------------------------------------------------------------------
   function Lex_Order (First_Name, Second_Name : T)
                      return Order_Types
   is
      Pos                         : Positions   := 1;
      Finished                    : Boolean     := False;
      Lex_Order_Result            : Order_Types := Neither_First;
      Rank_Of_Next_Char_In_First  : Integer;
      Rank_Of_Next_Char_In_Second : Integer;
   begin

      while not Finished loop
         -- if the remainder of only one of the strings is empty then
         -- that is the first in order
         -- if the remainder of both is empty then neither is first
         if Pos > First_Name.Length and Pos <= Second_Name.Length then
            Lex_Order_Result := First_One_First;
            Finished := True;
         elsif Pos > Second_Name.Length and Pos <= First_Name.Length then
            Lex_Order_Result := Second_One_First;
            Finished := True;
         elsif Pos > First_Name.Length and Pos > Second_Name.Length then
            Lex_Order_Result := Neither_First;
            Finished := True;
         else
            -- decide using the current Character in the string
            Rank_Of_Next_Char_In_First  := Character'Pos (First_Name.Content (Pos));
            Rank_Of_Next_Char_In_Second := Character'Pos (Second_Name.Content (Pos));

            if Rank_Of_Next_Char_In_First < Rank_Of_Next_Char_In_Second then
               Lex_Order_Result := First_One_First;
               Finished := True;
            elsif Rank_Of_Next_Char_In_Second < Rank_Of_Next_Char_In_First then
               Lex_Order_Result := Second_One_First;
               Finished := True;
            else
               if Pos < Positions'Last then
                  Pos := Pos + 1;
               else
                  Lex_Order_Result := Neither_First;
                  Finished := True;
               end if;
            end if;
         end if;
      end loop;

      return Lex_Order_Result;
   end Lex_Order;

   ----------------------------------------------------------------------------
   -- Section added for use in summary tool
   -- returns the specified subsection of the string
   -- if the subsection lies outside the string, empty string returned
   ----------------------------------------------------------------------------
   function Section (E_Str     : T;
                     Start_Pos : Positions;
                     Length    : Lengths) return T
   is
      Finished : Boolean;
      Pos      : Positions;
      Result   : T := Empty_String;
   begin
      if (Start_Pos + Length) - 1 <= E_Str.Length then
         Pos := Start_Pos;
         Finished := False;
         while not Finished and then Pos < Start_Pos + Length loop
            Result.Content ((Pos + 1) - Start_Pos) := E_Str.Content (Pos);
            if not (Pos = Positions'Last) then
               Pos := Pos + 1;
            else
               Finished := True;
            end if;
         end loop;
         Result.Length := Length;
      end if;
      return Result;
   end Section;

   ----------------------------------------------------------------------------
   -- Trim added for use in summary tool
   -- removes ' ', Latin_1.HT, Latin_1.LF, Latin_1.CR from either end of the string
   ----------------------------------------------------------------------------
   function Trim (E_Str : T) return T
   is
      Start_Pos      : Positions;
      End_Pos        : Positions;
      Finished       : Boolean;
      Trimmed_Length : Lengths;
      Trimmed_String : T;

      -----------------------------------------------------------------------
      function Is_White_Space (C : Character) return Boolean
      is
      begin
         return C = ' ' or else
           C = Ada.Characters.Latin_1.HT or else
           C = Ada.Characters.Latin_1.LF or else
           C = Ada.Characters.Latin_1.CR;
      end Is_White_Space;

      -----------------------------------------------------------------------
   begin -- Trim
      -- find first non space Character
      Start_Pos := 1;
      Finished := False;
      while not Finished loop
         if Start_Pos > E_Str.Length or else
           not Is_White_Space (E_Str.Content (Start_Pos)) then
            Finished := True;
         else
            Start_Pos := Start_Pos + 1;
         end if;
      end loop;

      -- if end of string then return empty string
      if Start_Pos > E_Str.Length then
         Trimmed_String := Empty_String;
      else
         -- find last non space character
         End_Pos := E_Str.Length;
         Finished := False;
         while not Finished loop
            if End_Pos = Start_Pos or else
              not Is_White_Space (E_Str.Content (End_Pos)) then
               Finished := True;
            else
               End_Pos := End_Pos - 1;
            end if;
         end loop;

         -- calculate trimmed string length
         Trimmed_Length := (End_Pos - Start_Pos) + 1;

         Trimmed_String := Section (E_Str, Start_Pos, Trimmed_Length);
      end if;

      return Trimmed_String;
   end Trim;

   procedure Get_Int_From_String (Source   : in     T;
                                  Item     :    out Integer;
                                  Start_Pt : in     Positive;
                                  Stop     :    out Natural)
   is
      --# hide Get_Int_From_String;
   begin
      SPARK_IO.Get_Int_From_String (Source   => Source.Content (Start_Pt .. Source.Length),
                                    Item     => Item,
                                    Start_Pt => Start_Pt,
                                    Stop     => Stop);
   end Get_Int_From_String;

   procedure Create (File         : in out SPARK_IO.File_Type;
                     Name_Of_File : in     T;
                     Form_Of_File : in     String;
                     Status       :    out SPARK_IO.File_Status)
   is
   begin
      SPARK_IO.Create (File         => File,
                       Name_Length  => Name_Of_File.Length,
                       Name_Of_File => Name_Of_File.Content,
                       Form_Of_File => Form_Of_File,
                       Status       => Status);
   end Create;

   procedure Open (File         : in out SPARK_IO.File_Type;
                   Mode_Of_File : in     SPARK_IO.File_Mode;
                   Name_Of_File : in     T;
                   Form_Of_File : in     String;
                   Status       :    out SPARK_IO.File_Status)
   is
   begin
      SPARK_IO.Open (File         => File,
                     Mode_Of_File => Mode_Of_File,
                     Name_Length  => Name_Of_File.Length,
                     Name_Of_File => Name_Of_File.Content,
                     Form_Of_File => Form_Of_File,
                     Status       => Status);
   end Open;

   procedure Put_String (File  : in SPARK_IO.File_Type;
                         E_Str : in T)
   is
   begin
      if E_Str.Length /= 0 then
         SPARK_IO.Put_String (File => File,
                              Item => E_Str.Content,
                              Stop => E_Str.Length);
      end if;
   end Put_String;

   procedure Put_Line (File  : in SPARK_IO.File_Type;
                       E_Str : in T)
   is
   begin
      if E_Str.Length = 0 then
         SPARK_IO.New_Line (File    => File,
                            Spacing => 1);
      else
         SPARK_IO.Put_Line (File => File,
                            Item => E_Str.Content,
                            Stop => E_Str.Length);
      end if;
   end Put_Line;

   procedure Get_Line (File  : in     SPARK_IO.File_Type;
                       E_Str :    out T)
   is
      E_Str_Content : Contents;
      E_Str_Length  : Lengths;
   begin
      SPARK_IO.Get_Line (File => File,
                         Item => E_Str_Content,
                         Stop => E_Str_Length);
      E_Str := T'(Content => E_Str_Content,
                  Length  => E_Str_Length);
   end Get_Line;

end ELStrings;
