-- $Id: dictionary.adb 16567 2010-03-25 16:09:40Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SystemErrors;
with Statistics;
with Maths;
with ELStrings;
with CommandLineData;

use type Maths.ErrorCode;
use type CommandLineData.LanguageProfiles;

package body Dictionary
is

   -----------------------------------------------------------------------------
   --                                   TYPES                                 --
   -----------------------------------------------------------------------------

   type SymbolDiscriminant is (NullSym,
                               TypeSymbol,
                               EnumerationLiteralSymbol,
                               ArrayIndexSymbol,
                               RecordComponentSymbol,
                               SubcomponentSymbol,
                               VariableSymbol,
                               QuantifiedVariableSymbol,
                               ImplicitReturnVariableSymbol,
                               OwnVariableSymbol,
                               OwnTaskSymbol,
                               ConstituentSymbol,
                               RulePolicySymbol,
                               ConstantSymbol,
                               PackageSymbol,
                               ContextClauseSymbol,
                               UseTypeClauseSymbol,
                               SubprogramSymbol,
                               ImplicitProofFunctionSymbol,
                               SubprogramParameterSymbol,
                               ParameterConstraintSymbol, -- index information for unconstrained formal parameters
                               GenericParameterSymbol,
                               GenericAssociationSymbol,
                               GlobalVariableSymbol,
                               DependencySymbol,
                               SuspendsListItemSymbol,
                               InterruptStreamMappingSymbol,
                               VirtualElementSymbol,
                               ProtectedImplicitInStreamSymbol, -- notional volatile source for reads of protected vars
                               OperatorSymbol,
                               LoopSymbol,
                               LoopParameterSymbol,
                               LoopEntryVariableSymbol,
                               DeclarationSymbol,
                               ProtectedInfoSymbol,
                               TaskInfoSymbol,
                               KnownDiscriminantSymbol,
                               SubtypeSymbol,                  -- extra info on Task/Protected subtypes
                               DiscriminantConstraintSymbol);  -- constrains for above

   type TypeDiscriminant is (UnknownType,
                             EnumerationType,
                             IntegerType,
                             ModularType,
                             FloatingPointType,
                             FixedPointType,
                             ArrayType,
                             RecordType,
                             AbstractProofType,
                             ProtectedType,
                             TaskType,
                             AccessType,
                              -- generic types
                             GenericLimitedType,
                             GenericPrivateType,
                             GenericOrderedDiscreteType,   -- excludes Boolean
                             GenericUnorderedDiscreteType,
                             GenericIntegerType,
                             GenericModularType,
                             GenericFloatingPointType,
                             GenericFixedPointType,
                             GenericArrayType);

   subtype GenericType is TypeDiscriminant range GenericLimitedType .. GenericArrayType;
   subtype GenericDiscreteType is GenericType range GenericOrderedDiscreteType .. GenericUnorderedDiscreteType;


   type TriState is (Never, Sometimes, Always);

   type DeclarationDiscriminant is (ConstantDeclaration,
                                    DeferredConstantDeclaration,
                                    VariableDeclaration,
                                    TypeAnnouncement,
                                    TypeDeclaration,
                                    SubtypeDeclaration,
                                    PrivateTypeDeclaration,
                                    PackageSpecification,
                                    SubprogramSpecification,
                                    RenamingDeclaration);

   type ProofStatementDiscriminant is (AssertStatement, CheckStatement);

   type ReferenceDiscriminant is (ReadRef, WriteRef, CallRef, OtherRef);

   -- RefType needs to be big enough to be suitable for Unchecked_Conversion
   -- from/to an access type, so...
   type RefType is range
     -2**(ExaminerConstants.Address_Size - 1) .. 2**(ExaminerConstants.Address_Size - 1) - 1;

   -- If conducting proof, it may be useful to assert the base type of RefType.
   -- This base type will be platform dependent. On 32 bit systems the correct
   -- assertion may look like:
   -- --# assert RefType'Base is Integer;
   -- While on 64 bit systems the correct assertion may look like:
   -- --# assert RefType'Base is Long_Long_Integer;

   for RefType'Size use ExaminerConstants.Address_Size;

   type SymbolInfo is
      record
         Discriminant : SymbolDiscriminant;
         Comp_Unit    : ContextManager.UnitDescriptors;
         Loc          : LexTokenManager.Token_Position;
         Ref          : RefType;
      end record;
   -- pragma pack (SymbolInfo)

   subtype ValidSymbol is Symbol range 1 .. Symbol'Last;

   type SymbolTable is array (ValidSymbol) of SymbolInfo;
   -- pragma pack (SymbolTable)

   type SymbolSet is
      record
         Current : Symbol;
         List    : SymbolTable;
      end record;
   -- pragma pack (SymbolSet)

   type UniversalTypes is (UniversalIntegerType,
                           UniversalRealType,
                           UniversalFixedType);

   type UniversalTypeList is array (UniversalTypes) of Symbol;
   -- pragma pack (UniversalTypeList)

   type PredefinedTypes is (PredefinedBooleanType,
                            PredefinedIntegerType,
                            PredefinedLongIntegerType,
                            PredefinedFloatType,
                            PredefinedLongFloatType,
                            PredefinedCharacterType,
                            PredefinedStringType,
                            PredefinedNaturalSubtype,
                            PredefinedPositiveSubtype,
                            PredefinedDurationType,
                            PredefinedTimeType,
                            PredefinedTimeSpanType,
                            PredefinedSuspensionObjectType);

   type PredefinedTypeList is array (PredefinedTypes) of Symbol;
   -- pragma pack (PredefinedTypeList)

--     type PredefinedGenericTypes is (PredefinedGenericLimitedType,
--                                     PredefinedGenericPrivateType,
--                                     PredefinedGenericOrderedDiscreteType,
--                                     PredefinedGenericUnorderedDiscreteType,
--                                     PredefinedGenericIntegerType,
--                                     PredefinedGenericModularType,
--                                     PredefinedGenericFloatingPointType,
--                                     PredefinedGenericFixedPointType,
--                                     PredefinedGenericArrayType);

--     type PredefinedGenericTypeList is array (PredefinedGenericTypes) of Symbol;

   type PredefinedLiterals is (PredefinedFalse, PredefinedTrue);

   type PredefinedLiteralList is array (PredefinedLiterals) of Symbol;
   -- pragma pack (PredefinedLiteralList)

   type TypeSet is
      record
         UnknownTypeMark           : Symbol;
         TheUniversalTypes         : UniversalTypeList;
         ThePredefinedTypes        : PredefinedTypeList;
--         ThePredefinedGenericTypes : PredefinedGenericTypeList;
         ThePredefinedLiterals     : PredefinedLiteralList;
      end record;
   -- pragma pack (TypeSet)

   type SubprogramSet is
      record
         STC_SuspendUntilTrue : Symbol; -- STC: Synchronous Task Control
         AdaRealTimeClock     : Symbol;
         UncheckedConversion  : Symbol;
      end record;
   -- pragma pack (SubprogramSet)

   type MainProgramSet is
      record
         Subprogram     : Symbol;
         InheritClauses : Symbol;
         PriorityGiven  : Boolean;
         ThePriority    : LexTokenManager.Lex_String;
      end record;
   -- pragma pack (MainProgramSet)

   ------------------------------------------------------------------
   -- Symbols for the predefined packages.  In SPARK83 mode,
   -- these are set to NullSymbol where such packages do not
   -- exist.
   --
   -- Note - package System is NOT included here, since some
   -- older SPARK83 projects supply it explicitly as a shadow
   -- specification which gets read in via the normal index file
   -- mechanism.  More recent projects specify System in the config
   -- file.  In that case, System is added in configfile.adb
   ------------------------------------------------------------------
   type PackageSet is
      record
         PackageStandard               : Symbol; -- All
         PackageASCII                  : Symbol; -- All (but considered obsolete in SPARK95)
         PackageAda                    : Symbol; -- SPARK95 and RavenSPARK only
         PackageAdaCharacters          : Symbol; -- SPARK95 nad RavenSPARK only
         PackageAdaCharactersLatin1    : Symbol; -- SPARK95 nad RavenSPARK only
         PackageRealTime               : Symbol; -- RavenSPARK only
         PackageSynchronousTaskControl : Symbol; -- RavenSPARK only
         PackageInterrupts             : Symbol; -- RavenSPARK only
      end record;
   -- pragma pack (PackageSet)

   type List is
      record
         Head, Tail : Symbol;
      end record;
   pragma Unreferenced (List); -- Why?  What is List for?
   -- pragma pack (List)

   type Dictionaries is
      record
         FileName        : EStrings.T;
         TemporaryFile   : SPARK_IO.File_Type;
         Symbols         : SymbolSet;
         Types           : TypeSet;
         Main            : MainProgramSet;
         Packages        : PackageSet;
         Subprograms     : SubprogramSet;
         NullVariable    : Symbol; -- an own variable of package standard used as "data sink"
         ThePartition    : Symbol; -- a subprogram record used to store the partition-wide flow relation
      end record;
   -- pragma pack (Dictionaries)

   --------------------------------------------------------------------------------
   --                                 VARIABLES                                  --
   --------------------------------------------------------------------------------

   Dict : Dictionaries;

   --------------------------------------------------------------------------------
   --                              RAW DICTIONARY                                --
   --------------------------------------------------------------------------------

   --# inherit ContextManager,
   --#         Dictionary,
   --#         ExaminerConstants,
   --#         LexTokenManager,
   --#         SPSymbols;
   package RawDict
   is

      function GetSymbolDiscriminant (Item : Dictionary.Symbol) return Dictionary.SymbolDiscriminant;
      --# global in Dictionary.Dict;
      pragma Inline (GetSymbolDiscriminant);

      function Get_Symbol_Compilation_Unit (Item : Dictionary.Symbol) return ContextManager.UnitDescriptors;
      --# global in Dictionary.Dict;
      pragma Inline (Get_Symbol_Compilation_Unit);

      function Get_Symbol_Location (Item : Dictionary.Symbol) return LexTokenManager.Token_Position;
      --# global in Dictionary.Dict;
      pragma Inline (Get_Symbol_Location);

      procedure Set_Symbol_Location (Item     : in Dictionary.Symbol;
                                     Location : in LexTokenManager.Token_Position);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Item,
      --#                              Location;
      pragma Inline (Set_Symbol_Location);

      -- types ---------------------------------------------------------------------

      procedure CreateType (Name      : in     LexTokenManager.Lex_String;
                            Comp_Unit : in     ContextManager.UnitDescriptors;
                            Loc       : in     LexTokenManager.Token_Position;
                            TypeMark  :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Name &
      --#         TypeMark        from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateType);

      procedure CreateProtectedInfo (Comp_Unit : in     ContextManager.UnitDescriptors;
                                     Loc       : in     LexTokenManager.Token_Position;
                                     InfoSym   :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         InfoSym         from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateProtectedInfo);

      procedure CreateProtectedImplicitInStream (TheOwnVar   : in     Dictionary.Symbol;
                                                 Comp_Unit   : in     ContextManager.UnitDescriptors;
                                                 Loc         : in     LexTokenManager.Token_Position;
                                                 InStreamSym :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         InStreamSym     from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc,
      --#                              TheOwnVar;
      pragma Inline (CreateProtectedImplicitInStream);

      procedure CreateTaskInfo (Comp_Unit : in     ContextManager.UnitDescriptors;
                                Loc       : in     LexTokenManager.Token_Position;
                                InfoSym   :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         InfoSym         from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateTaskInfo);

      procedure SetTypeDiscriminant (TypeMark     : in Dictionary.Symbol;
                                     Discriminant : in Dictionary.TypeDiscriminant);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant,
      --#                              TypeMark;
      pragma Inline (SetTypeDiscriminant);

      procedure SetTypePrivate (TypeMark  : in Dictionary.Symbol;
                                IsPrivate : in Dictionary.TriState);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              IsPrivate,
      --#                              TypeMark;
      pragma Inline (SetTypePrivate);

      procedure SetTypeLimited (TypeMark  : in Dictionary.Symbol;
                                IsLimited : in Dictionary.TriState);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              IsLimited,
      --#                              TypeMark;
      pragma Inline (SetTypeLimited);

      procedure SetTypeIsTagged (TypeMark  : in Dictionary.Symbol;
                                 IsTagged  : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              IsTagged,
      --#                              TypeMark;
      pragma Inline (SetTypeIsTagged);

      procedure SetTypeExtends (TypeMark  : in Dictionary.Symbol;
                                RootType  : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              RootType,
      --#                              TypeMark;
      pragma Inline (SetTypeExtends);

      procedure SetTypeAccesses (TheType   : in Dictionary.Symbol;
                                 TheAccess : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheAccess,
      --#                              TheType;
      pragma Inline (SetTypeAccesses);

      procedure SetTypeIsAbstract (TypeMark   : in Dictionary.Symbol;
                                   IsAbstract : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              IsAbstract,
      --#                              TypeMark;
      pragma Inline (SetTypeIsAbstract);

      procedure SetTypeLimitedPrivate (TypeMark  : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TypeMark;
      pragma Inline (SetTypeLimitedPrivate);

      procedure SetTypeDerived (TypeMark : in Dictionary.Symbol;
                                Derived  : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Derived,
      --#                              TypeMark;
      pragma Inline (SetTypeDerived);

      procedure SetTypeEqualityDefined (TypeMark        : in Dictionary.Symbol;
                                        EqualityDefined : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              EqualityDefined,
      --#                              TypeMark;
      pragma Inline (SetTypeEqualityDefined);

      procedure SetTypeContainsFloat (TypeMark      : in Dictionary.Symbol;
                                      ContainsFloat : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ContainsFloat,
      --#                              TypeMark;
      pragma Inline (SetTypeContainsFloat);

      procedure SetTypeConstrained (ArrayType   : in Dictionary.Symbol;
                                    Constrained : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ArrayType,
      --#                              Constrained;
      pragma Inline (SetTypeConstrained);

      procedure SetTypeStatic (TypeMark : in Dictionary.Symbol;
                               Static   : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Static,
      --#                              TypeMark;
      pragma Inline (SetTypeStatic);

      procedure SetTypeWellformed (TypeMark   : in Dictionary.Symbol;
                                   Wellformed : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TypeMark,
      --#                              Wellformed;
      pragma Inline (SetTypeWellformed);

      procedure SetTypeLower (TypeMark : in Dictionary.Symbol;
                              Lower    : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Lower,
      --#                              TypeMark;
      pragma Inline (SetTypeLower);

      procedure SetTypeUpper (TypeMark : in Dictionary.Symbol;
                              Upper    : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TypeMark,
      --#                              Upper;
      pragma Inline (SetTypeUpper);

      procedure SetTypeModulus (TypeMark : in Dictionary.Symbol;
                                Modulus  : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Modulus,
      --#                              TypeMark;
      pragma Inline (SetTypeModulus);

      procedure SetTypeErrorBound (TypeMark   : in Dictionary.Symbol;
                                   ErrorBound : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ErrorBound,
      --#                              TypeMark;
      pragma Inline (SetTypeErrorBound);

      procedure SetTypeParent (TheSubtype, Parent : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Parent,
      --#                              TheSubtype;
      pragma Inline (SetTypeParent);

      procedure SetTypeBaseType (TypeMark, BaseType : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              BaseType,
      --#                              TypeMark;
      pragma Inline (SetTypeBaseType);

      procedure SetTypeFirstEnumerationLiteral (EnumerationType, EnumerationLiteral : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              EnumerationLiteral,
      --#                              EnumerationType;
      pragma Inline (SetTypeFirstEnumerationLiteral);

      procedure SetTypeLastEnumerationLiteral (EnumerationType, EnumerationLiteral : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              EnumerationLiteral,
      --#                              EnumerationType;
      pragma Inline (SetTypeLastEnumerationLiteral);

      procedure SetTypeFirstArrayIndex (ArrayType, ArrayIndex : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ArrayIndex,
      --#                              ArrayType;
      pragma Inline (SetTypeFirstArrayIndex);

      procedure SetTypeLastArrayIndex (ArrayType, ArrayIndex : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ArrayIndex,
      --#                              ArrayType;
      pragma Inline (SetTypeLastArrayIndex);

      procedure SetTypeArrayComponent (ArrayType, ComponentType : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ArrayType,
      --#                              ComponentType;
      pragma Inline (SetTypeArrayComponent);

      procedure SetTypeFirstRecordComponent (RecordType, Component : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Component,
      --#                              RecordType;
      pragma Inline (SetTypeFirstRecordComponent);

      procedure SetTypeLastRecordComponent (RecordType, Component : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Component,
      --#                              RecordType;
      pragma Inline (SetTypeLastRecordComponent);

      procedure SetTypeAnnouncement (TypeMark, Announcement : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Announcement,
      --#                              TypeMark;
      pragma Inline (SetTypeAnnouncement);

      procedure SetTypeDeclaration (TypeMark, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TypeMark;
      pragma Inline (SetTypeDeclaration);

      procedure SetTypePrivateTypeDeclaration (PrivateType, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              PrivateType;
      pragma Inline (SetTypePrivateTypeDeclaration);

      procedure SetTypeAncilliaryFields (TheType, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TheType;
      pragma Inline (SetTypeAncilliaryFields);

      function GetTypeName (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeName);

      function GetTypeDiscriminant (TypeMark : Dictionary.Symbol) return Dictionary.TypeDiscriminant;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeDiscriminant);

      function GetTypePrivate (TypeMark : Dictionary.Symbol) return Dictionary.TriState;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypePrivate);

      function GetTypeLimited (TypeMark : Dictionary.Symbol) return Dictionary.TriState;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeLimited);

      function GetTypeLimitedPrivate (TypeMark : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeLimitedPrivate);

      function GetTypeDerived (TypeMark : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeDerived);

      function GetTypeIsAbstract (TypeMark : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeIsAbstract);

      function GetTypeIsTagged (TypeMark : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeIsTagged);

      function GetTypeExtends (TypeMark : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeExtends);

      function GetTypeAccesses (TypeMark : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeAccesses);

      function GetTypeEqualityDefined (TypeMark : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeEqualityDefined);

      function GetTypeContainsFloat (TypeMark : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeContainsFloat);

      function GetTypeConstrained (ArrayType : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeConstrained);

      function GetTypeStatic (TypeMark : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeStatic);

      function GetTypeWellformed (TypeMark : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeWellformed);

      function GetTypeParent (TypeMark : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeParent);

      function GetTypeBaseType (TypeMark : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeBaseType);

      function GetTypeLower (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeLower);

      function GetTypeUpper (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeUpper);

      function GetTypeModulus (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeModulus);

      function GetTypeErrorBound (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeErrorBound);

      function GetTypeFirstEnumerationLiteral (EnumerationType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeFirstEnumerationLiteral);

      function GetTypeLastEnumerationLiteral (EnumerationType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeLastEnumerationLiteral);

      function GetTypeFirstArrayIndex (ArrayType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeFirstArrayIndex);

      function GetTypeLastArrayIndex (ArrayType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeLastArrayIndex);

      function GetTypeArrayComponent (ArrayType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeArrayComponent);

      function GetTypeFirstRecordComponent (RecordType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeFirstRecordComponent);

      function GetTypeLastRecordComponent (RecordType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeLastRecordComponent);

      function GetTypeAnnouncement (TypeMark : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeAnnouncement);

      function GetTypeDeclaration (TypeMark : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeDeclaration);

      function GetTypePrivateTypeDeclaration (TypeMark : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypePrivateTypeDeclaration);

      function GetTypeAncilliaryFields (TypeMark : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeAncilliaryFields);

      procedure CreateEnumerationLiteral (Name               : in     LexTokenManager.Lex_String;
                                          Position           : in     LexTokenManager.Lex_String;
                                          EnumerationType    : in     Dictionary.Symbol;
                                          Comp_Unit          : in     ContextManager.UnitDescriptors;
                                          Loc                : in     LexTokenManager.Token_Position;
                                          EnumerationLiteral :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict    from *,
      --#                                 Comp_Unit,
      --#                                 EnumerationType,
      --#                                 Loc,
      --#                                 Name,
      --#                                 Position &
      --#         EnumerationLiteral from Comp_Unit,
      --#                                 Dictionary.Dict,
      --#                                 Loc;
      pragma Inline (CreateEnumerationLiteral);

      procedure SetNextEnumerationLiteral (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextEnumerationLiteral);

      function GetEnumerationLiteralName (Literal : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetEnumerationLiteralName);

      function GetEnumerationLiteralPosition (Literal : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetEnumerationLiteralPosition);

      function GetEnumerationLiteralType (Literal : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetEnumerationLiteralType);

      function GetNextEnumerationLiteral (Literal : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextEnumerationLiteral);

      procedure CreateArrayIndex (IndexType  : in     Dictionary.Symbol;
                                  Comp_Unit  : in     ContextManager.UnitDescriptors;
                                  Loc        : in     LexTokenManager.Token_Position;
                                  ArrayIndex :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives ArrayIndex      from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc &
      --#         Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              IndexType,
      --#                              Loc;
      pragma Inline (CreateArrayIndex);

      procedure SetNextArrayIndex (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextArrayIndex);

      function GetArrayIndexType (ArrayIndex : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetArrayIndexType);

      function GetNextArrayIndex (ArrayIndex : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextArrayIndex);

      procedure CreateRecordComponent (Name           : in     LexTokenManager.Lex_String;
                                       RecordType     : in     Dictionary.Symbol;
                                       ComponentType  : in     Dictionary.Symbol;
                                       InheritedField : in     Boolean;
                                       Comp_Unit      : in     ContextManager.UnitDescriptors;
                                       Loc            : in     LexTokenManager.Token_Position;
                                       Component      :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Component       from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc &
      --#         Dictionary.Dict from *,
      --#                              ComponentType,
      --#                              Comp_Unit,
      --#                              InheritedField,
      --#                              Loc,
      --#                              Name,
      --#                              RecordType;
      pragma Inline (CreateRecordComponent);

      procedure SetNextRecordComponent (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextRecordComponent);

      function GetRecordComponentName (Component : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetRecordComponentName);

      function GetRecordComponentRecordType (Component : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetRecordComponentRecordType);

      function GetRecordComponentType (Component : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetRecordComponentType);

      function GetRecordComponentInheritedField (Component : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetRecordComponentInheritedField);

      function GetNextRecordComponent (Component : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextRecordComponent);

      procedure CreateSubcomponent (Object       : in     Dictionary.Symbol;
                                    Component    : in     Dictionary.Symbol;
                                    Comp_Unit    : in     ContextManager.UnitDescriptors;
                                    Loc          : in     LexTokenManager.Token_Position;
                                    Subcomponent :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Component,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Object &
      --#         Subcomponent    from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateSubcomponent);

      procedure SetNextSubcomponent (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextSubcomponent);

      procedure SetSubcomponentSubcomponents (Parent, Sibling : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Parent,
      --#                              Sibling;
      pragma Inline (SetSubcomponentSubcomponents);

      function GetSubcomponentComponent (Subcomponent : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubcomponentComponent);

      function GetSubcomponentObject (Subcomponent : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubcomponentObject);

      function GetNextSubcomponent (Subcomponent : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextSubcomponent);

      function GetSubcomponentSubcomponents (Subcomponent : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubcomponentSubcomponents);

      function GetGlobalMode (Item : Dictionary.Symbol) return Dictionary.Modes;
      --# global in Dictionary.Dict;
      pragma Inline (GetGlobalMode);

      -- rule policies (for constants) -------------------------------------------------

      procedure CreateRulePolicy (Scope         : in     Dictionary.Scopes;
                                  Value         : in     Dictionary.RulePolicies;
                                  Comp_Unit     : in     ContextManager.UnitDescriptors;
                                  Loc           : in     LexTokenManager.Token_Position;
                                  TheRulePolicy :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Scope,
      --#                              Value &
      --#         TheRulePolicy   from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateRulePolicy);


      procedure SetNextRulePolicy (TheRulePolicy : in    Dictionary.Symbol;
                                   Next          : in    Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              TheRulePolicy;
      pragma Inline (SetNextRulePolicy);

      function GetNextRulePolicy (TheRulePolicy : in    Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextRulePolicy);

      function GetRulePolicyScope (TheRulePolicy : in    Dictionary.Symbol) return Dictionary.Scopes;
      --# global in Dictionary.Dict;
      pragma Inline (GetRulePolicyScope);

      function GetRulePolicyValue (TheRulePolicy : in    Dictionary.Symbol) return Dictionary.RulePolicies;
      --# global in Dictionary.Dict;
      pragma Inline (GetRulePolicyValue);

      -- constants ---------------------------------------------------------------------

      procedure CreateConstant (Name        : in     LexTokenManager.Lex_String;
                                TypeMark    : in     Dictionary.Symbol;
                                Static      : in     Boolean;
                                Comp_Unit   : in     ContextManager.UnitDescriptors;
                                Loc         : in     LexTokenManager.Token_Position;
                                TheConstant :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Name,
      --#                              Static,
      --#                              TypeMark &
      --#         TheConstant     from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateConstant);

      procedure SetConstantValue (TheConstant : in Dictionary.Symbol;
                                  Value       : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstant,
      --#                              Value;
      pragma Inline (SetConstantValue);

      procedure SetConstantExpNode (TheConstant     : in Dictionary.Symbol;
                                    ExpIsWellFormed : in Boolean;
                                    ExpNode         : in ExaminerConstants.RefType);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ExpIsWellFormed,
      --#                              ExpNode,
      --#                              TheConstant;
      pragma Inline (SetConstantExpNode);

      procedure SetConstantStatic (TheConstant : in Dictionary.Symbol;
                                   Static      : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Static,
      --#                              TheConstant;
      pragma Inline (SetConstantStatic);

      procedure SetConstantDeclaration (TheConstant, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TheConstant;
      pragma Inline (SetConstantDeclaration);

      procedure SetConstantDeferredConstantDeclaration (DeferredConstant, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              DeferredConstant;
      pragma Inline (SetConstantDeferredConstantDeclaration);

      procedure SetConstantAssociatedGenericParameter (TheConstant, TheGenericParameter : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstant,
      --#                              TheGenericParameter;
      pragma Inline (SetConstantAssociatedGenericParameter);

      procedure SetConstantFirstRulePolicy (TheConstant : in Dictionary.Symbol;
                                            ThePolicy   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstant,
      --#                              ThePolicy;
      pragma Inline (SetConstantFirstRulePolicy);

      procedure SetConstantLastRulePolicy (TheConstant : in Dictionary.Symbol;
                                           ThePolicy   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstant,
      --#                              ThePolicy;
      pragma Inline (SetConstantLastRulePolicy);

      function GetConstantName (TheConstant : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstantName);

      function GetConstantType (TheConstant : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstantType);

      function GetConstantValue (TheConstant : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstantValue);

      function GetConstantFirstRulePolicy (TheConstant : Dictionary.Symbol)
                                       return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstantFirstRulePolicy);

      function GetConstantLastRulePolicy (TheConstant : Dictionary.Symbol)
                                       return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstantLastRulePolicy);

      function GetConstantExpNode (TheConstant : Dictionary.Symbol) return ExaminerConstants.RefType;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstantExpNode);

      function ConstantExpIsWellFormed (TheConstant : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (ConstantExpIsWellFormed);


      function GetConstantStatic (TheConstant : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstantStatic);

      function GetConstantDeclaration (TheConstant : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstantDeclaration);

      function GetConstantDeferredConstantDeclaration (TheConstant : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstantDeferredConstantDeclaration);

      function GetConstantAssociatedGenericParameter (TheConstant : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstantAssociatedGenericParameter);

      -- variabless ---------------------------------------------------------------------

      procedure CreateVariable (Name      : in     LexTokenManager.Lex_String;
                                Comp_Unit : in     ContextManager.UnitDescriptors;
                                Loc       : in     LexTokenManager.Token_Position;
                                Variable  :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Name &
      --#         Variable        from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateVariable);

      procedure SetVariableType (Variable, TypeMark : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TypeMark,
      --#                              Variable;
      pragma Inline (SetVariableType);

      procedure SetVariableInitialized (Variable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Variable;
      pragma Inline (SetVariableInitialized);

      procedure SetVariableHasAddressClause (Variable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Variable;
      pragma Inline (SetVariableHasAddressClause);

      procedure SetVariableHasPragmaImport (Variable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Variable;
      pragma Inline (SetVariableHasPragmaImport);

      procedure SetTypeSizeAttribute (TypeMark : in Dictionary.Symbol; SizeVal : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              SizeVal,
      --#                              TypeMark;
      pragma Inline (SetTypeSizeAttribute);

      procedure SetVariableIsAliased (Variable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Variable;
      pragma Inline (SetVariableIsAliased);

      procedure SetVariableMarkedValid (Variable   : in Dictionary.Symbol;
                                        Val        : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Val,
      --#                              Variable;
      pragma Inline (SetVariableMarkedValid);

      procedure SetSubcomponentMarkedValid (Subcomponent   : in Dictionary.Symbol;
                                            Val            : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subcomponent,
      --#                              Val;
      pragma Inline (SetSubcomponentMarkedValid);

      procedure SetTypeAtomic (TypeMark : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TypeMark;
      pragma Inline (SetTypeAtomic);

      procedure SetVariableDeclaration (Variable, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              Variable;
      pragma Inline (SetVariableDeclaration);

      procedure SetVariableExpNode (Variable : in Dictionary.Symbol;
                                    ExpNode  : in ExaminerConstants.RefType);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ExpNode,
      --#                              Variable;
      pragma Inline (SetVariableExpNode);

      procedure SetVariableOwnTask (Variable : in Dictionary.Symbol;
                                    OwnTask  : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnTask,
      --#                              Variable;

      procedure SetVariableVirtualElement (Variable    : in Dictionary.Symbol;
                                           VirtualElement : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Variable,
      --#                              VirtualElement;
      pragma Inline (SetVariableVirtualElement);

      procedure SetVariableGlobalReferences (Abstraction : in Dictionary.Abstractions;
                                             Variable    : in Dictionary.Symbol;
                                             Reference   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Reference,
      --#                              Variable;
      pragma Inline (SetVariableGlobalReferences);

      procedure SetVariableOwnVariable (Variable, OwnVariable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnVariable,
      --#                              Variable;
      pragma Inline (SetVariableOwnVariable);

      procedure SetVariableConstituent (Variable, Constituent : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Constituent,
      --#                              Variable;
      pragma Inline (SetVariableConstituent);

      procedure SetVariableSubComponents (Variable, Subcomponents : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subcomponents,
      --#                              Variable;
      pragma Inline (SetVariableSubComponents);

      function GetVariableName (Variable : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableName);

      function GetVariableType (Variable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableType);

      function GetVariableInitialized (Variable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableInitialized);

      function GetVariableHasAddressClause (Variable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableHasAddressClause);

      function GetVariableHasPragmaImport (Variable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableHasPragmaImport);

      function GetVariableIsAliased (Variable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableIsAliased);

      function GetVariableMarkedValid (Variable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableMarkedValid);

      function GetSubcomponentMarkedValid (Subcomponent : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubcomponentMarkedValid);

      function IsAtomic (TheSymbol : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (IsAtomic);

      function IsPragmaAtomic (TheSymbol : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (IsPragmaAtomic);

      function GetVariableDeclaration (Variable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableDeclaration);

      function GetVariableExpNode (Variable : Dictionary.Symbol) return ExaminerConstants.RefType;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableExpNode);

      function GetVariableGlobalReferences (Abstraction : Dictionary.Abstractions;
                                            Variable    : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableGlobalReferences);

      function GetVariableOwnVariable (Variable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableOwnVariable);

      function GetVariableOwnTask (Variable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableOwnTask);

      function GetVariableVirtualElement (Variable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableVirtualElement);

      function GetVariableConstituent (Variable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableConstituent);

      function GetVariableSubComponents (Variable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVariableSubComponents);

      function GetTypeSizeAttribute (TypeMark : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetTypeSizeAttribute);

      -- quantifiers ---------------------------------------------------------------------

      procedure CreateQuantifiedVariable (Name      : in     LexTokenManager.Lex_String;
                                          TypeMark  : in     Dictionary.Symbol;
                                          Region    : in     Dictionary.Symbol;
                                          Comp_Unit : in     ContextManager.UnitDescriptors;
                                          Loc       : in     LexTokenManager.Token_Position;
                                          Variable  :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Name,
      --#                              Region,
      --#                              TypeMark &
      --#         Variable        from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateQuantifiedVariable);

      function GetQuantifiedVariableName (Variable : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetQuantifiedVariableName);

      function GetQuantifiedVariableType (Variable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetQuantifiedVariableType);

      function GetQuantifiedVariableRegion (Variable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetQuantifiedVariableRegion);

      -- implicit return variables -------------------------------------------------------------------

      procedure CreateImplicitReturnVariable (Name        : in     LexTokenManager.Lex_String;
                                              TheFunction : in     Dictionary.Symbol;
                                              Comp_Unit   : in     ContextManager.UnitDescriptors;
                                              Loc         : in     LexTokenManager.Token_Position;
                                              Variable    :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Name,
      --#                              TheFunction &
      --#         Variable        from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateImplicitReturnVariable);

      function GetImplicitReturnVariableName (Variable : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetImplicitReturnVariableName);

      function GetImplicitReturnVariableFunction (Variable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetImplicitReturnVariableFunction);

      -- own variables -------------------------------------------------------------------

      procedure CreateOwnVariable (Variable    : in     Dictionary.Symbol;
                                   Owner       : in     Dictionary.Symbol;
                                   Comp_Unit   : in     ContextManager.UnitDescriptors;
                                   Loc         : in     LexTokenManager.Token_Position;
                                   OwnVariable :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Owner,
      --#                              Variable &
      --#         OwnVariable     from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateOwnVariable);

      procedure SetOwnVariableAnnounced (OwnVariable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnVariable;
      pragma Inline (SetOwnVariableAnnounced);

      procedure SetOwnVariableTyped (OwnVariable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnVariable;
      pragma Inline (SetOwnVariableTyped);

      procedure SetOwnVariableInitialized (OwnVariable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnVariable;
      pragma Inline (SetOwnVariableInitialized);

      procedure SetOwnVariableConstituents (Subject, Constituent : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Constituent,
      --#                              Subject;
      pragma Inline (SetOwnVariableConstituents);

      procedure SetOwnVariableMode (OwnVariable : in Dictionary.Symbol;
                                    Mode        : in Dictionary.Modes);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Mode,
      --#                              OwnVariable;
      pragma Inline (SetOwnVariableMode);

      procedure SetOwnVariableProtected (OwnVariable : in Dictionary.Symbol;
                                         IsProtected : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              IsProtected,
      --#                              OwnVariable;
      pragma Inline (SetOwnVariableProtected);

      procedure SetNextOwnVariable (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextOwnVariable);

      function GetOwnVariableVariable (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableVariable);

      function GetOwnVariableOwner (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableOwner);

      function GetOwnVariableAnnounced (OwnVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableAnnounced);

      function GetOwnVariableTyped (OwnVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableTyped);

      function GetOwnVariableInitialized (OwnVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableInitialized);

      function GetOwnVariableConstituents (Subject : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableConstituents);

      function GetOwnVariableMode (OwnVariable : Dictionary.Symbol) return Dictionary.Modes;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableMode);

      function GetOwnVariableProtected (OwnVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableProtected);

      function GetOwnVariableImplicitInStream (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableImplicitInStream);

      -- essentially a reciprocal of the above function
      function GetImplicitInStreamOwnVariable (TheStream : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetImplicitInStreamOwnVariable);

      function GetNextOwnVariable (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextOwnVariable);

      procedure CreateOwnTask (Variable  : in     Dictionary.Symbol;
                               Owner     : in     Dictionary.Symbol;
                               Comp_Unit : in     ContextManager.UnitDescriptors;
                               Loc       : in     LexTokenManager.Token_Position;
                               OwnTask   :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Owner,
      --#                              Variable &
      --#         OwnTask         from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;

      pragma Inline (CreateOwnTask);

      procedure SetNextOwnTask (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextOwnTask);

      function GetNextOwnTask (Current : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextOwnTask);

      function GetOwnTaskVariable (OwnTask : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnTaskVariable);

      function GetOwnTaskOwner (OwnTask : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnTaskOwner);

      procedure CreateConstituent (Subject     : in     Dictionary.Symbol;
                                   Variable    : in     Dictionary.Symbol;
                                   Comp_Unit   : in     ContextManager.UnitDescriptors;
                                   Loc         : in     LexTokenManager.Token_Position;
                                   Constituent :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Constituent     from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc &
      --#         Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Subject,
      --#                              Variable;
      pragma Inline (CreateConstituent);

      procedure SetConstituentMode (Constituent : in Dictionary.Symbol;
                                    Mode        : in Dictionary.Modes);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Constituent,
      --#                              Mode;
      pragma Inline (SetConstituentMode);

      procedure SetNextConstituent (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextConstituent);

      function GetConstituentSubject (Constituent : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstituentSubject);

      function GetConstituentVariable (Constituent : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstituentVariable);

      function GetConstituentMode (Constituent : Dictionary.Symbol) return Dictionary.Modes;
      --# global in Dictionary.Dict;
      pragma Inline (GetConstituentMode);

      function GetNextConstituent (Constituent : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextConstituent);

      -- packages ---------------------------------------------------------------------

      procedure CreatePackage (Name       : in     LexTokenManager.Lex_String;
                               Comp_Unit  : in     ContextManager.UnitDescriptors;
                               Loc        : in     LexTokenManager.Token_Position;
                               ThePackage :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Name &
      --#         ThePackage      from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreatePackage);

      procedure SetPackagePostcondition (Abstraction   : in Dictionary.Abstractions;
                                         ThePackage    : in Dictionary.Symbol;
                                         Postcondition : in ExaminerConstants.RefType);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Postcondition,
      --#                              ThePackage;
      pragma Inline (SetPackagePostcondition);

      procedure SetPackageIsGeneric (ThePackage : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePackage;
      pragma Inline (SetPackageIsGeneric);

      procedure SetPackageBody (ThePackage, TheBody : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheBody,
      --#                              ThePackage;
      pragma Inline (SetPackageBody);

      procedure SetPackageHasProperBody (ThePackage : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePackage;
      pragma Inline (SetPackageHasProperBody);

      procedure SetPackageElaborateBodyFound (ThePackage : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePackage;
      pragma Inline (SetPackageElaborateBodyFound);

      procedure SetPackageInheritClauses (ThePackage, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              ThePackage;
      pragma Inline (SetPackageInheritClauses);

      procedure SetPackageVisibleWithClauses (ThePackage, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              ThePackage;
      pragma Inline (SetPackageVisibleWithClauses);

      procedure SetPackageLocalWithClauses (ThePackage, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              ThePackage;
      pragma Inline (SetPackageLocalWithClauses);

      procedure SetPackageVisibleUseTypeClauses (ThePackage, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              ThePackage;
      pragma Inline (SetPackageVisibleUseTypeClauses);

      procedure SetPackageLocalUseTypeClauses (ThePackage, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              ThePackage;
      pragma Inline (SetPackageLocalUseTypeClauses);

      procedure SetPackageOwnVariables (ThePackage, OwnVariable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnVariable,
      --#                              ThePackage;
      pragma Inline (SetPackageOwnVariables);

      procedure SetPackageTaskList (ThePackage, OwnTask : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnTask,
      --#                              ThePackage;
      pragma Inline (SetPackageTaskList);

      procedure SetPackageFirstVisibleDeclaration (ThePackage, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              ThePackage;
      pragma Inline (SetPackageFirstVisibleDeclaration);

      procedure SetPackageLastVisibleDeclaration (ThePackage, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              ThePackage;
      pragma Inline (SetPackageLastVisibleDeclaration);

      procedure SetPackageVisibleRenamingDeclarations (ThePackage, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              ThePackage;
      pragma Inline (SetPackageVisibleRenamingDeclarations);

      procedure SetPackageLocalRenamingDeclarations (ThePackage, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              ThePackage;
      pragma Inline (SetPackageLocalRenamingDeclarations);

      procedure SetPackageFirstLocalDeclaration (ThePackage, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              ThePackage;
      pragma Inline (SetPackageFirstLocalDeclaration);

      procedure SetPackageLastLocalDeclaration (ThePackage, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              ThePackage;
      pragma Inline (SetPackageLastLocalDeclaration);

      procedure SetPackageFirstPrivateDeclaration (ThePackage, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              ThePackage;
      pragma Inline (SetPackageFirstPrivateDeclaration);

      procedure SetPackageLastPrivateDeclaration (ThePackage, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              ThePackage;
      pragma Inline (SetPackageLastPrivateDeclaration);

      procedure SetPackageParent (ThePackage, TheParent : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePackage,
      --#                              TheParent;
      pragma Inline (SetPackageParent);

      procedure SetPackageSibling (ThePackage, TheSibling : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePackage,
      --#                              TheSibling;
      pragma Inline (SetPackageSibling);

      procedure SetPackageExtends (ThePackage, ThePackageItExtends : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePackage,
      --#                              ThePackageItExtends;
      pragma Inline (SetPackageExtends);

      procedure SetPackageDeclaresTaggedType (ThePackage : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePackage;
      pragma Inline (SetPackageDeclaresTaggedType);

      procedure SetPackageFirstChild (ThePackage,
                                         TheChild   : in Dictionary.Symbol;
                                      NameSpace  : in Dictionary.PackageSort);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              NameSpace,
      --#                              TheChild,
      --#                              ThePackage;
      pragma Inline (SetPackageFirstChild);

      procedure SetPackageLastChild (ThePackage,
                                        TheChild   : in Dictionary.Symbol;
                                     NameSpace  : in Dictionary.PackageSort);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              NameSpace,
      --#                              TheChild,
      --#                              ThePackage;
      pragma Inline (SetPackageLastChild);

      procedure SetPackageIsPrivate (ThePackage : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePackage;
      pragma Inline (SetPackageIsPrivate);

      procedure SetPackageFirstLoop (ThePackage, TheLoop : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheLoop,
      --#                              ThePackage;
      pragma Inline (SetPackageFirstLoop);

      procedure SetPackageLastLoop (ThePackage, TheLoop : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheLoop,
      --#                              ThePackage;
      pragma Inline (SetPackageLastLoop);

      procedure SetPackageFirstGenericParameter (ThePackage, TheParameter : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePackage,
      --#                              TheParameter;
      pragma Inline (SetPackageFirstGenericParameter);

      procedure SetPackageLastGenericParameter (ThePackage, TheParameter : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePackage,
      --#                              TheParameter;
      pragma Inline (SetPackageLastGenericParameter);

      procedure SetPackageSpecification (ThePackage, Specification : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Specification,
      --#                              ThePackage;
      pragma Inline (SetPackageSpecification);

      -- protected types ---------------------------------------------------------------------

      procedure SetProtectedTypeOwnVariable (TheProtectedType, OwnVariable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnVariable,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeOwnVariable);

      procedure SetProtectedTypeElementsHidden (TheProtectedType : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeElementsHidden);

      procedure SetProtectedTypeFirstVisibleDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeFirstVisibleDeclaration);

      procedure SetProtectedTypeLastVisibleDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeLastVisibleDeclaration);

      procedure SetProtectedTypeFirstPrivateDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeFirstPrivateDeclaration);

      procedure SetProtectedTypeLastPrivateDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeLastPrivateDeclaration);

      procedure SetProtectedTypeFirstLocalDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeFirstLocalDeclaration);

      procedure SetProtectedTypeLastLocalDeclaration (TheProtectedType, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeLastLocalDeclaration);

      procedure SetProtectedTypeTheEntry (TheProtectedType, TheEntry : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheEntry,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeTheEntry);

      procedure SetProtectedTypeFirstDiscriminant (TheProtectedType, Discriminant : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeFirstDiscriminant);

      procedure SetProtectedTypeLastDiscriminant (TheProtectedType, Discriminant : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeLastDiscriminant);

      procedure SetProtectedTypeHasPragma (TheProtectedType : in Dictionary.Symbol;
                                           ThePragma        : in Dictionary.RavenscarPragmas);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePragma,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeHasPragma);

      procedure SetProtectedTypePragmaValue (TheProtectedType : in Dictionary.Symbol;
                                             ThePragma        : in Dictionary.RavenscarPragmasWithValue;
                                             TheValue         : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePragma,
      --#                              TheProtectedType,
      --#                              TheValue;
      pragma Inline (SetProtectedTypePragmaValue);

      procedure SetProtectedTypeBody (TheProtectedType, TheBody : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheBody,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeBody);

      procedure SetProtectedTypeHasProperBody (TheProtectedType : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeHasProperBody);

      procedure SetProtectedTypeWithClauses (TheProtectedType, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeWithClauses);

      procedure SetProtectedTypeUseTypeClauses (TheProtectedType, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              TheProtectedType;
      pragma Inline (SetProtectedTypeUseTypeClauses);

      -- task types ---------------------------------------------------------------------

      procedure SetTaskTypeHasSecondAnnotation (TheTaskType : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeHasSecondAnnotation);

      procedure SetTaskTypeFirstLocalDeclaration (TheTaskType, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeFirstLocalDeclaration);

      procedure SetTaskTypeLastLocalDeclaration (TheTaskType, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeLastLocalDeclaration);

      procedure SetTaskTypeWithClauses (TheTaskType, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeWithClauses);

      procedure SetTaskTypeUseTypeClauses (TheTaskType, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeUseTypeClauses);

      procedure SetTaskTypeFirstDiscriminant (TheTaskType, Discriminant : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeFirstDiscriminant);

      procedure SetTaskTypeLastDiscriminant (TheTaskType, Discriminant : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeLastDiscriminant);

      procedure SetTaskTypeFirstGlobalVariable (Abstraction : in Dictionary.Abstractions;
                                                TheTaskType,
                                                TheGlobal   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              TheGlobal,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeFirstGlobalVariable);

      procedure SetTaskTypeLastGlobalVariable (Abstraction : in Dictionary.Abstractions;
                                               TheTaskType,
                                               TheGlobal   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              TheGlobal,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeLastGlobalVariable);

      procedure SetTaskTypeHasPragma (TheTaskType : in Dictionary.Symbol;
                                      ThePragma   : in Dictionary.RavenscarPragmas);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePragma,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeHasPragma);

      procedure SetTaskTypePragmaValue (TheTaskType : in Dictionary.Symbol;
                                        ThePragma   : in Dictionary.RavenscarPragmasWithValue;
                                        TheValue    : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ThePragma,
      --#                              TheTaskType,
      --#                              TheValue;
      pragma Inline (SetTaskTypePragmaValue);

      procedure SetTaskTypeFirstLoop (TheTaskType, TheLoop : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheLoop,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeFirstLoop);

      procedure SetTaskTypeLastLoop (TheTaskType, TheLoop : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheLoop,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeLastLoop);

      procedure SetVirtualElementList (TheProtectedType : in Dictionary.Symbol;
                                       TheList          : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheList,
      --#                              TheProtectedType;
      pragma Inline (SetVirtualElementList);

      procedure SetTaskTypeSuspendsList (TheTaskType     : in Dictionary.Symbol;
                                         TheSuspendsList : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheSuspendsList,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeSuspendsList);

      procedure SetProcedureSuspendsList (TheProcedure    : in Dictionary.Symbol;
                                          TheSuspendsList : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheProcedure,
      --#                              TheSuspendsList;
      pragma Inline (SetProcedureSuspendsList);

      procedure SetTaskTypeBody (TheTaskType, TheBody : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheBody,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeBody);

      procedure SetTaskTypeHasProperBody (TheTaskType : in Dictionary.Symbol;
                                          IsHidden    : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              IsHidden,
      --#                              TheTaskType;
      pragma Inline (SetTaskTypeHasProperBody);

      -- Accessor functions ----------------------------------------------------------------------
      -- packages --------------------------------------------------------------------------------

      function GetPackageName (ThePackage : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageName);

      function GetPackagePostcondition (Abstraction : Dictionary.Abstractions;
                                        ThePackage  : Dictionary.Symbol) return ExaminerConstants.RefType;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackagePostcondition);

      function GetPackageBody (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageBody);

      function GetPackageHasProperBody (ThePackage : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageHasProperBody);

      function GetPackageElaborateBodyFound (ThePackage : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageElaborateBodyFound);

      function GetPackageInheritClauses (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageInheritClauses);

      function GetPackageVisibleWithClauses (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageVisibleWithClauses);

      function GetPackageLocalWithClauses (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageLocalWithClauses);

      function GetPackageVisibleUseTypeClauses (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageVisibleUseTypeClauses);

      function GetPackageLocalUseTypeClauses (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageLocalUseTypeClauses);

      function GetPackageOwnVariables (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageOwnVariables);

      function GetPackageTaskList (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageTaskList);

      function GetPackageIsGeneric (ThePackage : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageIsGeneric);

      function GetPackageFirstVisibleDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageFirstVisibleDeclaration);

      function GetPackageLastVisibleDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageLastVisibleDeclaration);

      function GetPackageVisibleRenamingDeclarations (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageVisibleRenamingDeclarations);

      function GetPackageLocalRenamingDeclarations (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageLocalRenamingDeclarations);

      function GetPackageFirstLocalDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageFirstLocalDeclaration);

      function GetPackageLastLocalDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageLastLocalDeclaration);

      function GetPackageFirstPrivateDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageFirstPrivateDeclaration);

      function GetPackageLastPrivateDeclaration (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageLastPrivateDeclaration);

      function GetPackageFirstGenericParameter (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageFirstGenericParameter);

      function GetPackageLastGenericParameter (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageLastGenericParameter);

      function GetPackageParent (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageParent);

      function GetPackageSibling (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageSibling);

      function GetPackageExtends (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageExtends);

      function GetPackageDeclaresTaggedType (ThePackage : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageDeclaresTaggedType);

      function GetPackageFirstChild (ThePackage : Dictionary.Symbol;
                                     NameSpace  : Dictionary.PackageSort) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageFirstChild);

      function GetPackageLastChild (ThePackage : Dictionary.Symbol;
                                    NameSpace  : Dictionary.PackageSort) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageLastChild);

      function GetPackageIsPrivate (ThePackage : in Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageIsPrivate);

      function GetPackageFirstLoop (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageFirstLoop);

      function GetPackageLastLoop (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageLastLoop);

      function GetPackageSpecification (ThePackage : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetPackageSpecification);

      -- protected types ----------------------------------------------------------------

      function GetProtectedTypeOwnVariable (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeOwnVariable);

      function GetProtectedTypeElementsHidden (TheProtectedType : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeElementsHidden);

      function GetProtectedTypeHasEntry (TheProtectedType : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeHasEntry);

      function GetProtectedTypeFirstVisibleDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeFirstVisibleDeclaration);

      function GetProtectedTypeLastVisibleDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeLastVisibleDeclaration);

      function GetProtectedTypeFirstPrivateDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeFirstPrivateDeclaration);

      function GetProtectedTypeLastPrivateDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeLastPrivateDeclaration);

      function GetProtectedTypeFirstLocalDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeFirstLocalDeclaration);

      function GetProtectedTypeLastLocalDeclaration (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeLastLocalDeclaration);

      function GetProtectedTypeTheEntry (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeTheEntry);
      pragma Unreferenced (GetProtectedTypeTheEntry); -- not used at present

      function GetProtectedTypeFirstDiscriminant (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeFirstDiscriminant);

      function GetProtectedTypeLastDiscriminant (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeLastDiscriminant);

      function GetProtectedTypeHasPragma (TheProtectedType : Dictionary.Symbol;
                                          ThePragma        : Dictionary.RavenscarPragmas) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeHasPragma);

      function GetProtectedTypePragmaValue (TheProtectedType : Dictionary.Symbol;
                                            ThePragma        : Dictionary.RavenscarPragmasWithValue)
                                           return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypePragmaValue);

      function GetProtectedTypeBody (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeBody);

      function GetProtectedTypeHasProperBody (TheProtectedType : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeHasProperBody);

      function GetProtectedTypeWithClauses (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeWithClauses);

      function GetProtectedTypeUseTypeClauses (TheProtectedType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProtectedTypeUseTypeClauses);

      function GetTaskTypeFirstLocalDeclaration (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeFirstLocalDeclaration);

      function GetTaskTypeLastLocalDeclaration (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeLastLocalDeclaration);

      function GetTaskTypeWithClauses (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeWithClauses);

      function GetTaskTypeUseTypeClauses (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeUseTypeClauses);

      function GetTaskTypeFirstDiscriminant (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeFirstDiscriminant);

      function GetTaskTypeLastDiscriminant (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeLastDiscriminant);

      function GetTaskTypeFirstGlobalVariable (Abstraction : Dictionary.Abstractions;
                                               TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeFirstGlobalVariable);

      function GetTaskTypeLastGlobalVariable (Abstraction : Dictionary.Abstractions;
                                              TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeLastGlobalVariable);

      function GetTaskTypeHasPragma (TheTaskType : Dictionary.Symbol;
                                     ThePragma   : Dictionary.RavenscarPragmas) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeHasPragma);

      function GetTaskTypePragmaValue (TheTaskType : Dictionary.Symbol;
                                       ThePragma   : Dictionary.RavenscarPragmasWithValue)
                                      return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypePragmaValue);

      function GetTaskTypeFirstLoop (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeFirstLoop);

      function GetTaskTypeLastLoop (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeLastLoop);

      function GetVirtualElementList (TheProtectedType : Dictionary.Symbol)
                                     return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVirtualElementList);

      function GetTaskTypeSuspendsList (TheTaskType : Dictionary.Symbol)
                                       return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeSuspendsList);

      function GetProcedureSuspendsList (TheProcedure : Dictionary.Symbol)
                                        return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetProcedureSuspendsList);

      function GetTaskTypeBody (TheTaskType : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeBody);

      function GetTaskTypeHasProperBody (TheTaskType : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeHasProperBody);

      procedure CreateContextClause (Scope         : in     Dictionary.Scopes;
                                     ThePackage    : in     Dictionary.Symbol;
                                     Explicit      : in     Boolean;
                                     Comp_Unit     : in     ContextManager.UnitDescriptors;
                                     Loc           : in     LexTokenManager.Token_Position;
                                     ContextClause :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives ContextClause   from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc &
      --#         Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Explicit,
      --#                              Loc,
      --#                              Scope,
      --#                              ThePackage;
      pragma Inline (CreateContextClause);

      procedure CreateUseTypeClause (Scope         : in     Dictionary.Scopes;
                                     TheType       : in     Dictionary.Symbol;
                                     Comp_Unit     : in     ContextManager.UnitDescriptors;
                                     Loc           : in     LexTokenManager.Token_Position;
                                     UseTypeClause :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Scope,
      --#                              TheType &
      --#         UseTypeClause   from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateUseTypeClause);

      procedure SetNextContextClause (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextContextClause);

      procedure SetContextClauseExplicit (Current  : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current;
      pragma Inline (SetContextClauseExplicit);

      procedure SetNextUseTypeClause (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextUseTypeClause);

      function GetContextClauseScope (ContextClause : Dictionary.Symbol) return Dictionary.Scopes;
      --# global in Dictionary.Dict;
      pragma Inline (GetContextClauseScope);
      pragma Unreferenced (GetContextClauseScope); -- not used at present

      function GetContextClauseExplicit (ContextClause : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetContextClauseExplicit);

      function GetContextClausePackage (ContextClause : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetContextClausePackage);

      function GetUseTypeClauseType (UseTypeClause : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetUseTypeClauseType);

      function GetNextContextClause (ContextClause : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextContextClause);

      function GetNextUseTypeClause (UseTypeClause : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextUseTypeClause);

      -- subprograms ---------------------------------------------------------------------------

      procedure CreateSubprogram (Name       : in     LexTokenManager.Lex_String;
                                  Comp_Unit  : in     ContextManager.UnitDescriptors;
                                  Loc        : in     LexTokenManager.Token_Position;
                                  Subprogram :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Name &
      --#         Subprogram      from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateSubprogram);

      procedure SetSubprogramImplicitProofFunction (Abstraction   : in Dictionary.Abstractions;
                                                    AdaFunction   : in Dictionary.Symbol;
                                                    ProofFunction : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              AdaFunction,
      --#                              ProofFunction;
      pragma Inline (SetSubprogramImplicitProofFunction);

      procedure SetSubprogramImplicitReturnVariable (Abstraction : in Dictionary.Abstractions;
                                                     TheFunction : in Dictionary.Symbol;
                                                     Variable    : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              TheFunction,
      --#                              Variable;
      pragma Inline (SetSubprogramImplicitReturnVariable);

      procedure SetSubprogramSignatureNotWellformed (Abstraction  : in Dictionary.Abstractions;
                                                     Subprogram   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Subprogram;
      pragma Inline (SetSubprogramSignatureNotWellformed);

      procedure SetTaskTypeSignatureNotWellformed (Abstraction  : in Dictionary.Abstractions;
                                                   TaskType     : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              TaskType;
      pragma Inline (SetTaskTypeSignatureNotWellformed);

      procedure SetSubprogramIsEntry (Subprogram   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram;
      pragma Inline (SetSubprogramIsEntry);

      procedure SetSubprogramEntryBarrier (Subprogram, TheBarrier : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram,
      --#                              TheBarrier;
      pragma Inline (SetSubprogramEntryBarrier);

      procedure SetIsInterruptHandler (Subprogram   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram;
      pragma Inline (SetIsInterruptHandler);

      procedure SetSubprogramUsesUncheckedConversion (Subprogram : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram;
      pragma Inline (SetSubprogramUsesUncheckedConversion);


      procedure SetSubprogramAssignsFromExternal (Subprogram : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram;
      pragma Inline (SetSubprogramAssignsFromExternal);

      procedure SetSubprogramPrecondition (Abstraction  : in Dictionary.Abstractions;
                                           Subprogram   : in Dictionary.Symbol;
                                           Precondition : in ExaminerConstants.RefType);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Precondition,
      --#                              Subprogram;
      pragma Inline (SetSubprogramPrecondition);

      procedure SetSubprogramPostcondition (Abstraction   : in Dictionary.Abstractions;
                                            Subprogram    : in Dictionary.Symbol;
                                            Postcondition : in ExaminerConstants.RefType);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Postcondition,
      --#                              Subprogram;
      pragma Inline (SetSubprogramPostcondition);

      procedure SetSubprogramBody (Subprogram, TheBody : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram,
      --#                              TheBody;
      pragma Inline (SetSubprogramBody);

      procedure SetSubprogramInstantiationOf (Subprogram, TheGeneric : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram,
      --#                              TheGeneric;
      pragma Inline (SetSubprogramInstantiationOf);

      procedure SetSubprogramHasProperBody (Subprogram : in Dictionary.Symbol;
                                            IsHidden   : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              IsHidden,
      --#                              Subprogram;
      pragma Inline (SetSubprogramHasProperBody);

      procedure SetSubprogramIsGeneric (Subprogram : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram;
      pragma Inline (SetSubprogramIsGeneric);

      procedure SetSubprogramHasSecondAnnotation (Subprogram : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram;
      pragma Inline (SetSubprogramHasSecondAnnotation);

      procedure SetSubprogramHasSecondConstraint (Subprogram : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram;
      pragma Inline (SetSubprogramHasSecondConstraint);

      procedure SetSubprogramReturnType (TheFunction, TypeMark : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheFunction,
      --#                              TypeMark;
      pragma Inline (SetSubprogramReturnType);

      procedure SetSubprogramWithClauses (Subprogram, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              Subprogram;
      pragma Inline (SetSubprogramWithClauses);

      procedure SetSubprogramUseTypeClauses (Subprogram, Clause : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Clause,
      --#                              Subprogram;
      pragma Inline (SetSubprogramUseTypeClauses);

      procedure SetSubprogramFirstParameter (Subprogram, Parameter : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Parameter,
      --#                              Subprogram;
      pragma Inline (SetSubprogramFirstParameter);

      procedure SetSubprogramLastParameter (Subprogram, Parameter : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Parameter,
      --#                              Subprogram;
      pragma Inline (SetSubprogramLastParameter);

      procedure SetSubprogramFirstGenericParameter (Subprogram, Parameter : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Parameter,
      --#                              Subprogram;
      pragma Inline (SetSubprogramFirstGenericParameter);

      procedure SetSubprogramLastGenericParameter (Subprogram, Parameter : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Parameter,
      --#                              Subprogram;
      pragma Inline (SetSubprogramLastGenericParameter);

      procedure SetSubprogramFirstGenericAssociation (Subprogram, TheAssociation : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram,
      --#                              TheAssociation;
      pragma Inline (SetSubprogramFirstGenericAssociation);

      procedure SetSubprogramLastGenericAssociation (Subprogram, TheAssociation : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram,
      --#                              TheAssociation;
      pragma Inline (SetSubprogramLastGenericAssociation);

      procedure SetSubprogramFirstGlobalVariable (Abstraction : in Dictionary.Abstractions;
                                                  Subprogram  : in Dictionary.Symbol;
                                                  Variable    : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Subprogram,
      --#                              Variable;
      pragma Inline (SetSubprogramFirstGlobalVariable);

      procedure SetSubprogramLastGlobalVariable (Abstraction : in Dictionary.Abstractions;
                                                 Subprogram  : in Dictionary.Symbol;
                                                 Variable    : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Subprogram,
      --#                              Variable;
      pragma Inline (SetSubprogramLastGlobalVariable);

      procedure SetSubprogramRenamingDeclarations (Subprogram, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              Subprogram;
      pragma Inline (SetSubprogramRenamingDeclarations);

      procedure SetSubprogramFirstDeclaration (Subprogram, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              Subprogram;
      pragma Inline (SetSubprogramFirstDeclaration);

      procedure SetSubprogramLastDeclaration (Subprogram, Declaration : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              Subprogram;
      pragma Inline (SetSubprogramLastDeclaration);

      procedure SetSubprogramFirstLoop (Subprogram, TheLoop : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram,
      --#                              TheLoop;
      pragma Inline (SetSubprogramFirstLoop);

      procedure SetSubprogramLastLoop (Subprogram, TheLoop : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Subprogram,
      --#                              TheLoop;
      pragma Inline (SetSubprogramLastLoop);

      procedure SetSubprogramSpecification (Subprogram, Specification : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Specification,
      --#                              Subprogram;
      pragma Inline (SetSubprogramSpecification);

      function GetSubprogramName (Subprogram : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramName);

      function GetSubprogramImplicitProofFunction (Abstraction : Dictionary.Abstractions;
                                                   AdaFunction : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramImplicitProofFunction);

      function GetSubprogramImplicitReturnVariable (Abstraction : Dictionary.Abstractions;
                                                    TheFunction : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramImplicitReturnVariable);

      function GetSubprogramSignatureIsWellformed (Abstraction : Dictionary.Abstractions;
                                                   Subprogram  : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramSignatureIsWellformed);

      function GetTaskTypeSignatureIsWellformed (Abstraction : Dictionary.Abstractions;
                                                 TaskType    : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeSignatureIsWellformed);

      function GetSubprogramIsEntry (Subprogram  : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramIsEntry);

      function GetSubprogramEntryBarrier (Subprogram  : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramEntryBarrier);

      function IsInterruptHandler (Subprogram  : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (IsInterruptHandler);

      function GetSubprogramUsesUncheckedConversion (Subprogram : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramUsesUncheckedConversion);

      function GetSubprogramAssignsFromExternal (Subprogram : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramAssignsFromExternal);

      function GetSubprogramPrecondition (Abstraction : Dictionary.Abstractions;
                                          Subprogram  : Dictionary.Symbol) return ExaminerConstants.RefType;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramPrecondition);

      function GetSubprogramPostcondition (Abstraction : Dictionary.Abstractions;
                                           Subprogram  : Dictionary.Symbol) return ExaminerConstants.RefType;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramPostcondition);

      function GetSubprogramBody (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramBody);

      function GetSubprogramInstantiationOf (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramInstantiationOf);

      function GetSubprogramHasProperBody (Subprogram : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramHasProperBody);

      function GetSubprogramIsGeneric (Subprogram : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramIsGeneric);

      function GetSubprogramHasSecondAnnotation (Subprogram : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramHasSecondAnnotation);

      function GetSubprogramHasSecondConstraint (Subprogram : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramHasSecondConstraint);

      function GetTaskTypeHasSecondAnnotation (TaskType : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetTaskTypeHasSecondAnnotation);

      function GetSubprogramReturnType (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramReturnType);

      function GetSubprogramWithClauses (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramWithClauses);

      function GetSubprogramUseTypeClauses (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramUseTypeClauses);

      function GetSubprogramFirstParameter (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramFirstParameter);

      function GetSubprogramLastParameter (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramLastParameter);

      function GetSubprogramFirstGenericParameter (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramFirstGenericParameter);

      function GetSubprogramLastGenericParameter (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramLastGenericParameter);

      function GetSubprogramFirstGenericAssociation (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramFirstGenericAssociation);

      function GetSubprogramLastGenericAssociation (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramLastGenericAssociation);

      function GetSubprogramFirstGlobalVariable (Abstraction : Dictionary.Abstractions;
                                                 Subprogram  : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramFirstGlobalVariable);

      function GetSubprogramLastGlobalVariable (Abstraction : Dictionary.Abstractions;
                                                Subprogram  : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramLastGlobalVariable);

      function GetSubprogramRenamingDeclarations (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramRenamingDeclarations);

      function GetSubprogramFirstDeclaration (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramFirstDeclaration);

      function GetSubprogramLastDeclaration (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramLastDeclaration);

      function GetSubprogramFirstLoop (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramFirstLoop);

      function GetSubprogramLastLoop (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramLastLoop);

      function GetSubprogramSpecification (Subprogram : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramSpecification);

      procedure CreateImplicitProofFunction (AdaFunction   : in     Dictionary.Symbol;
                                             Comp_Unit     : in     ContextManager.UnitDescriptors;
                                             Loc           : in     LexTokenManager.Token_Position;
                                             ProofFunction :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              AdaFunction,
      --#                              Comp_Unit,
      --#                              Loc &
      --#         ProofFunction   from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateImplicitProofFunction);

      function GetImplicitProofFunctionAdaFunction (ProofFunction : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetImplicitProofFunctionAdaFunction);

      -- subprogram parameters -----------------------------------------------------------------

      procedure CreateSubprogramParameter (Name       : in     LexTokenManager.Lex_String;
                                           Subprogram : in     Dictionary.Symbol;
                                           TypeMark   : in     Dictionary.Symbol;
                                           Mode       : in     Dictionary.Modes;
                                           Comp_Unit  : in     ContextManager.UnitDescriptors;
                                           Loc        : in     LexTokenManager.Token_Position;
                                           Parameter  :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Mode,
      --#                              Name,
      --#                              Subprogram,
      --#                              TypeMark &
      --#         Parameter       from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateSubprogramParameter);

      procedure SetSubprogramParameterExported (Abstraction : in Dictionary.Abstractions;
                                                Parameter   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Parameter;
      pragma Inline (SetSubprogramParameterExported);

      procedure SetSubprogramParameterImported (Abstraction : in Dictionary.Abstractions;
                                                Parameter   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Parameter;
      pragma Inline (SetSubprogramParameterImported);

      procedure SetSubprogramParameterDependencies (Abstraction : in Dictionary.Abstractions;
                                                    Parameter   : in Dictionary.Symbol;
                                                    Dependency  : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Dependency,
      --#                              Parameter;
      pragma Inline (SetSubprogramParameterDependencies);

      procedure SetSubprogramParameterGlobalReferences (Abstraction : in Dictionary.Abstractions;
                                                        Parameter   : in Dictionary.Symbol;
                                                        Reference   : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Parameter,
      --#                              Reference;
      pragma Inline (SetSubprogramParameterGlobalReferences);

      procedure SetSubprogramParameterSubComponents (Parameter, Subcomponents : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Parameter,
      --#                              Subcomponents;
      pragma Inline (SetSubprogramParameterSubComponents);

      procedure SetSubprogramParameterIndexConstraints (Parameter, IndexConstraints : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              IndexConstraints,
      --#                              Parameter;
      pragma Inline (SetSubprogramParameterIndexConstraints);

      procedure SetNextSubprogramParameter (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextSubprogramParameter);

      function GetSubprogramParameterName (Parameter : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramParameterName);

      function GetSubprogramParameterSubprogram (Parameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramParameterSubprogram);

      function GetSubprogramParameterType (Parameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramParameterType);

      function GetSubprogramParameterMode (Parameter : Dictionary.Symbol) return Dictionary.Modes;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramParameterMode);

      function GetSubprogramParameterExported (Abstraction : Dictionary.Abstractions;
                                               Parameter   : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramParameterExported);

      function GetSubprogramParameterImported (Abstraction : Dictionary.Abstractions;
                                               Parameter   : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramParameterImported);

      function GetSubprogramParameterDependencies (Abstraction : Dictionary.Abstractions;
                                                   Parameter   : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramParameterDependencies);

      function GetSubprogramParameterGlobalReferences (Abstraction : Dictionary.Abstractions;
                                                       Parameter   : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramParameterGlobalReferences);

      function GetSubprogramParameterSubComponents (Parameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramParameterSubComponents);

      function GetSubprogramParameterIndexConstraints (Parameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubprogramParameterIndexConstraints);

      function GetNextSubprogramParameter (Parameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextSubprogramParameter);

      -- symbols for index constraints of unconstrained formal parameters ---------------------------------

      procedure CreateParameterConstraint (TheParameter     : in     Dictionary.Symbol;
                                           Dimension        : in     Positive;
                                           Comp_Unit        : in     ContextManager.UnitDescriptors;
                                           Loc              : in     LexTokenManager.Token_Position;
                                           ConstraintSymbol :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives ConstraintSymbol from Comp_Unit,
      --#                               Dictionary.Dict,
      --#                               Loc &
      --#         Dictionary.Dict  from *,
      --#                               Comp_Unit,
      --#                               Dimension,
      --#                               Loc,
      --#                               TheParameter;

      function GetParameterConstraintParameter (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetParameterConstraintParameter);

      function GetParameterConstraintDimension (TheConstraint : Dictionary.Symbol) return Positive;
      --# global in Dictionary.Dict;
      pragma Inline (GetParameterConstraintDimension);

      function GetParameterConstraintNext (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetParameterConstraintNext);

      procedure SetParameterConstraintNext (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetParameterConstraintNext);

      -- generic parameters -----------------------------------------------------------------

      procedure CreateGenericParameter (OwningGeneric : in     Dictionary.Symbol;
                                        TypeOrObject  : in     Dictionary.Symbol;
                                        Kind          : in     Dictionary.GenericParameterKind;
                                        Comp_Unit     : in     ContextManager.UnitDescriptors;
                                        Loc           : in     LexTokenManager.Token_Position;
                                        Parameter     :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Kind,
      --#                              Loc,
      --#                              OwningGeneric,
      --#                              TypeOrObject &
      --#         Parameter       from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateGenericParameter);

      procedure SetNextGenericParameter (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextGenericParameter);

      function GetGenericParameterName (Parameter : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetGenericParameterName);

      function GetGenericParameterTypeOrObject (Parameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetGenericParameterTypeOrObject);

      function GetGenericParameterKind (Parameter : Dictionary.Symbol) return Dictionary.GenericParameterKind;
      --# global in Dictionary.Dict;
      pragma Inline (GetGenericParameterKind);

      function GetGenericParameterNext (Parameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetGenericParameterNext);

      function GetGenericParameterOwningGeneric (Parameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetGenericParameterOwningGeneric);

      -- generic associations -----------------------------------------------------------------

      procedure CreateGenericAssociation (Formal       : in     Dictionary.Symbol;
                                          ActualSymbol : in     Dictionary.Symbol;
                                          Comp_Unit    : in     ContextManager.UnitDescriptors;
                                          Loc          : in     LexTokenManager.Token_Position;
                                          Association  :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Association     from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc &
      --#         Dictionary.Dict from *,
      --#                              ActualSymbol,
      --#                              Comp_Unit,
      --#                              Formal,
      --#                              Loc;
      pragma Inline (CreateGenericAssociation);

      procedure SetNextGenericAssociation (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextGenericAssociation);

      function GetGenericAssociationFormal (Association : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetGenericAssociationFormal);

      function GetGenericAssociationActualSymbol (Association : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetGenericAssociationActualSymbol);

      function GetGenericAssociationNext (Association : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetGenericAssociationNext);

      -- discriminants -----------------------------------------------------------------------

      procedure CreateKnownDiscriminant (Name              : in     LexTokenManager.Lex_String;
                                         ProtectedType     : in     Dictionary.Symbol;
                                         TypeMark          : in     Dictionary.Symbol;
                                         Comp_Unit         : in     ContextManager.UnitDescriptors;
                                         Loc               : in     LexTokenManager.Token_Position;
                                         KnownDiscriminant :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict   from *,
      --#                                Comp_Unit,
      --#                                Loc,
      --#                                Name,
      --#                                ProtectedType,
      --#                                TypeMark &
      --#         KnownDiscriminant from Comp_Unit,
      --#                                Dictionary.Dict,
      --#                                Loc;
      pragma Inline (CreateKnownDiscriminant);

      procedure SetNextDiscriminant (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextDiscriminant);

      procedure SetDiscriminantSetsPriority (Discriminant : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Discriminant;
      pragma Inline (SetDiscriminantSetsPriority);

      function GetNextDiscriminant (Discriminant : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextDiscriminant);

      function GetDiscriminantName (Discriminant : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetDiscriminantName);

      function GetDiscriminantTypeMark (Discriminant : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetDiscriminantTypeMark);

      function GetDiscriminantProtectedType (Discriminant : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetDiscriminantProtectedType);

      function GetDiscriminantSetsPriority (Discriminant : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetDiscriminantSetsPriority);

      -- subtype here means extra infor for task and protected subtypes only
      procedure CreateSubtype (Comp_Unit  : in     ContextManager.UnitDescriptors;
                               Loc        : in     LexTokenManager.Token_Position;
                               TheSubtype :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         TheSubtype      from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateSubtype);

      procedure SetSubtypeInfoPriority (TheSubtype : in Dictionary.Symbol;
                                       Priority   : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Priority,
      --#                              TheSubtype;
      pragma Inline (SetSubtypeInfoPriority);

      procedure SetSubtypeInfoFirstConstraint (TheSubtype    : in Dictionary.Symbol;
                                              TheConstraint : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstraint,
      --#                              TheSubtype;
      pragma Inline (SetSubtypeInfoFirstConstraint);

      procedure SetSubtypeInfoLastConstraint (TheSubtype    : in Dictionary.Symbol;
                                             TheConstraint : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstraint,
      --#                              TheSubtype;
      pragma Inline (SetSubtypeInfoLastConstraint);

      function GetSubtypeInfoPriority (TheSubtype : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubtypeInfoPriority);

      function GetSubtypeInfoFirstConstraint (TheSubtype : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubtypeInfoFirstConstraint);

      function GetSubtypeInfoLastConstraint (TheSubtype : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSubtypeInfoLastConstraint);

      procedure CreateDiscriminantConstraint (Comp_Unit                 : in     ContextManager.UnitDescriptors;
                                              Loc                       : in     LexTokenManager.Token_Position;
                                              TheDiscriminantConstraint :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict,
      --#         TheDiscriminantConstraint from Comp_Unit,
      --#                                        Dictionary.Dict,
      --#                                        Loc;
      pragma Inline (CreateDiscriminantConstraint);

      procedure SetDiscriminantConstraintStaticValue (TheConstraint : in Dictionary.Symbol;
                                                     TheValue      : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstraint,
      --#                              TheValue;
      pragma Inline (SetDiscriminantConstraintStaticValue);

      procedure SetDiscriminantConstraintAccessedObject (TheConstraint : in Dictionary.Symbol;
                                                        TheObject     : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheConstraint,
      --#                              TheObject;
      pragma Inline (SetDiscriminantConstraintAccessedObject);

      procedure SetNextDiscriminantConstraint (TheConstraint : in Dictionary.Symbol;
                                              Next          : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              TheConstraint;
      pragma Inline (SetNextDiscriminantConstraint);

      function GetDiscriminantConstraintStaticValue (TheConstraint : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetDiscriminantConstraintStaticValue);

      function GetDiscriminantConstraintAccessedObject (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetDiscriminantConstraintAccessedObject);

      function GetNextDiscriminantConstraint (TheConstraint : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextDiscriminantConstraint);

      procedure CreateGlobalVariable (Subprogram     : in     Dictionary.Symbol;
                                      Variable       : in     Dictionary.Symbol;
                                      Mode           : in     Dictionary.Modes;
                                      PrefixNeeded   : in     Boolean;
                                      NextSubprogram : in     Dictionary.Symbol;
                                      Comp_Unit      : in     ContextManager.UnitDescriptors;
                                      Loc            : in     LexTokenManager.Token_Position;
                                      GlobalVariable :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Mode,
      --#                              NextSubprogram,
      --#                              PrefixNeeded,
      --#                              Subprogram,
      --#                              Variable &
      --#         GlobalVariable  from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateGlobalVariable);

      procedure SetGlobalVariableExported (GlobalVariable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              GlobalVariable;
      pragma Inline (SetGlobalVariableExported);

      procedure SetGlobalVariableImported (GlobalVariable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              GlobalVariable;
      pragma Inline (SetGlobalVariableImported);

      procedure SetGlobalVariableDependencies (Abstraction    : in Dictionary.Abstractions;
                                               GlobalVariable : in Dictionary.Symbol;
                                               Dependency     : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Abstraction,
      --#                              Dependency,
      --#                              GlobalVariable;
      pragma Inline (SetGlobalVariableDependencies);

      procedure SetGlobalVariableNextVariable (Previous, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Next,
      --#                              Previous;
      pragma Inline (SetGlobalVariableNextVariable);

      function GetGlobalVariableSubprogram (GlobalVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetGlobalVariableSubprogram);

      function GetGlobalVariableVariable (GlobalVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetGlobalVariableVariable);

      function GetGlobalVariablePrefixNeeded (GlobalVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetGlobalVariablePrefixNeeded);

      function GetGlobalVariableExported (GlobalVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetGlobalVariableExported);

      function GetGlobalVariableImported (GlobalVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetGlobalVariableImported);

      function GetGlobalVariableDependencies (Abstraction    : Dictionary.Abstractions;
                                              GlobalVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetGlobalVariableDependencies);

      function GetGlobalVariableNextSubprogram (GlobalVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetGlobalVariableNextSubprogram);

      function GetNextGlobalVariable (GlobalVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextGlobalVariable);

      procedure CreateDependency (TheImport  : in     Dictionary.Symbol;
                                  Comp_Unit  : in     ContextManager.UnitDescriptors;
                                  Loc        : in     LexTokenManager.Token_Position;
                                  Dependency :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dependency      from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc &
      --#         Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              TheImport;
      pragma Inline (CreateDependency);

      procedure SetNextDependency (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextDependency);

      function GetDependencyImport (Dependency : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetDependencyImport);

      function GetNextDependency (Dependency : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextDependency);

      procedure CreateInterruptStreamMapping
        (TheHandler         : in     LexTokenManager.Lex_String;
         TheInterruptStream : in     LexTokenManager.Lex_String;
         Comp_Unit          : in     ContextManager.UnitDescriptors;
         Loc                : in     LexTokenManager.Token_Position;
         TheMapping         :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              TheHandler,
      --#                              TheInterruptStream &
      --#         TheMapping      from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateInterruptStreamMapping);

      procedure SetNextInterruptStreamMapping (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextInterruptStreamMapping);

      function GetNextInterruptStreamMapping
        (TheMapping : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextInterruptStreamMapping);

      procedure SetOwnVariableInterruptStreamMappings
        (TheOwnVariable             : in Dictionary.Symbol;
         TheInterruptStreamMappings : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheInterruptStreamMappings,
      --#                              TheOwnVariable;
      pragma Inline (SetOwnVariableInterruptStreamMappings);

      function GetOwnVariableInterruptStreamMappings
        (TheOwnVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableInterruptStreamMappings);

      procedure SetOwnVariableIsInterruptStream
        (TheOwnVariable    : in Dictionary.Symbol;
         IsInterruptStream : in Boolean);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              IsInterruptStream,
      --#                              TheOwnVariable;
      pragma Inline (SetOwnVariableIsInterruptStream);

      procedure SetOwnVariableImplicitInStream
        (TheOwnVariable      : in Dictionary.Symbol;
         TheImplicitInStream : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheImplicitInStream,
      --#                              TheOwnVariable;
      pragma Inline (SetOwnVariableImplicitInStream);

      function GetOwnVariableIsInterruptStream
        (TheOwnVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetOwnVariableIsInterruptStream);

      function GetInterruptStreamMappingHandler (TheMapping : in Dictionary.Symbol)
                                                return LexTokenManager.Lex_String;

      function GetInterruptStreamMappingStream (TheMapping : in Dictionary.Symbol)
                                               return LexTokenManager.Lex_String;

      procedure CreateSuspendsListItem (ThePOorSO        : in     Dictionary.Symbol;
                                        Comp_Unit        : in     ContextManager.UnitDescriptors;
                                        Loc              : in     LexTokenManager.Token_Position;
                                        SuspendsListItem :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict  from *,
      --#                               Comp_Unit,
      --#                               Loc,
      --#                               ThePOorSO &
      --#         SuspendsListItem from Comp_Unit,
      --#                               Dictionary.Dict,
      --#                               Loc;
      pragma Inline (CreateSuspendsListItem);

      procedure SetNextSuspendsListItem (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextSuspendsListItem);

      function GetSuspendsListItem (SuspendsListItem : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetSuspendsListItem);

      procedure SetSuspendsListItemIsAccountedFor (SuspendsListItem : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              SuspendsListItem;
      pragma Inline (SetSuspendsListItemIsAccountedFor);

      function GetSuspendsListItemIsAccountedFor (SuspendsListItem : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetSuspendsListItemIsAccountedFor);

      function GetNextSuspendsListItem (SuspendsListItem : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextSuspendsListItem);

      procedure CreateVirtualElement (TheVariable       : in     Dictionary.Symbol;
                                      TheOwner          : in     Dictionary.Symbol;
                                      Comp_Unit         : in     ContextManager.UnitDescriptors;
                                      Loc               : in     LexTokenManager.Token_Position;
                                      TheVirtualElement :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict   from *,
      --#                                Comp_Unit,
      --#                                Loc,
      --#                                TheOwner,
      --#                                TheVariable &
      --#         TheVirtualElement from Comp_Unit,
      --#                                Dictionary.Dict,
      --#                                Loc;
      pragma Inline (CreateVirtualElement);

      function GetVirtualElementVariable (VirtualElement : Dictionary.Symbol)
                                         return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVirtualElementVariable);

      function GetVirtualElementOwner (VirtualElement : Dictionary.Symbol)
                                      return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetVirtualElementOwner);

      procedure SetVirtualElementSeenByOwner (VirtualElement : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              VirtualElement;
      pragma Inline (SetVirtualElementSeenByOwner);

      function GetVirtualElementSeenByOwner (VirtualElement : Dictionary.Symbol)
                                            return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetVirtualElementSeenByOwner);

      procedure SetNextVirtualElement (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;

      pragma Inline (SetNextVirtualElement);

      function GetNextVirtualElement (VirtualElement : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextVirtualElement);

      procedure CreateUnaryOperator (Name      : in     SPSymbols.SPSymbol;
                                     Operand   : in     Dictionary.Symbol;
                                     Comp_Unit : in     ContextManager.UnitDescriptors;
                                     Loc       : in     LexTokenManager.Token_Position;
                                     Operator  :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Name,
      --#                              Operand &
      --#         Operator        from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateUnaryOperator);

      procedure CreateBinaryOperator (Name        : in     SPSymbols.SPSymbol;
                                      Left, Right : in     Dictionary.Symbol;
                                      Comp_Unit   : in     ContextManager.UnitDescriptors;
                                      Loc         : in     LexTokenManager.Token_Position;
                                      Operator    :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Left,
      --#                              Loc,
      --#                              Name,
      --#                              Right &
      --#         Operator        from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateBinaryOperator);

      function GetOperatorName (Operator : Dictionary.Symbol) return SPSymbols.SPSymbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetOperatorName);

      function GetOperatorLeftOperand (Operator : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetOperatorLeftOperand);

      function GetOperatorRightOperand (Operator : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetOperatorRightOperand);

      procedure CreateLoop (Region    : in     Dictionary.Symbol;
                            Comp_Unit : in     ContextManager.UnitDescriptors;
                            Loc       : in     LexTokenManager.Token_Position;
                            TheLoop   :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Loc,
      --#                              Region &
      --#         TheLoop         from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateLoop);

      procedure SetLoopName (Name    : in LexTokenManager.Lex_String;
                             TheLoop : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Name,
      --#                              TheLoop;
      pragma Inline (SetLoopName);

      procedure SetLoopParameter (ForLoop, LoopParameter : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ForLoop,
      --#                              LoopParameter;
      pragma Inline (SetLoopParameter);

      procedure SetLoopOnEntryVars (ForLoop, OnEntryVars : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ForLoop,
      --#                              OnEntryVars;
      pragma Inline (SetLoopOnEntryVars);

      procedure SetLoopExitExpn (ForLoop : in Dictionary.Symbol;
                                 Expn    : in Natural);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Expn,
      --#                              ForLoop;
      pragma Inline (SetLoopExitExpn);

      procedure SetLoopEntryExpn (ForLoop : in Dictionary.Symbol;
                                  Expn    : in Natural);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Expn,
      --#                              ForLoop;
      pragma Inline (SetLoopEntryExpn);

      procedure SetLoopHasExits (TheLoop : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheLoop;
      pragma Inline (SetLoopHasExits);

      procedure SetNextLoop (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextLoop);

      function GetLoopName (TheLoop : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopName);

      function GetLoopRegion (TheLoop : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopRegion);

      function GetLoopOnEntryVars (TheLoop : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopOnEntryVars);

      function GetLoopExitExpn (TheLoop : Dictionary.Symbol) return Natural;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopExitExpn);

      function GetLoopEntryExpn (TheLoop : Dictionary.Symbol) return Natural;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopEntryExpn);

      function GetLoopParameter (TheLoop : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopParameter);

      function GetLoopHasExits (TheLoop : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopHasExits);

      function GetNextLoop (TheLoop : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextLoop);

      procedure CreateLoopParameter (Name           : in     LexTokenManager.Lex_String;
                                     TypeMark       : in     Dictionary.Symbol;
                                     TheLoop        : in     Dictionary.Symbol;
                                     HasStaticRange : in     Boolean;
                                     IsReverse      : in     Boolean;
                                     Comp_Unit      : in     ContextManager.UnitDescriptors;
                                     Loc            : in     LexTokenManager.Token_Position;
                                     LoopParameter  :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              HasStaticRange,
      --#                              IsReverse,
      --#                              Loc,
      --#                              Name,
      --#                              TheLoop,
      --#                              TypeMark &
      --#         LoopParameter   from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc;
      pragma Inline (CreateLoopParameter);

      function GetLoopParameterName (LoopParameter : Dictionary.Symbol) return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopParameterName);

      function GetLoopParameterType (LoopParameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopParameterType);

      function GetLoopParameterHasStaticRange (LoopParameter : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopParameterHasStaticRange);

      function GetLoopParameterIsReverse (LoopParameter : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopParameterIsReverse);

      function GetLoopParameterLoop (LoopParameter : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopParameterLoop);

      procedure CreateLoopEntryVariable (OriginalVar       : in     Dictionary.Symbol;
                                         TheLoop           : in     Dictionary.Symbol;
                                         Comp_Unit         : in     ContextManager.UnitDescriptors;
                                         Loc               : in     LexTokenManager.Token_Position;
                                         LoopEntryVariable :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict   from *,
      --#                                Comp_Unit,
      --#                                Loc,
      --#                                OriginalVar,
      --#                                TheLoop &
      --#         LoopEntryVariable from Comp_Unit,
      --#                                Dictionary.Dict,
      --#                                Loc;
      pragma Inline (CreateLoopEntryVariable);

      procedure SetLoopEntryVariableNext (LoopEntryVariable : in Dictionary.Symbol;
                                          Next              : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              LoopEntryVariable,
      --#                              Next;
      pragma Inline (SetLoopEntryVariableNext);

      function GetLoopEntryVariableOriginalVar (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopEntryVariableOriginalVar);

      function GetLoopEntryVariableTheLoop (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopEntryVariableTheLoop);

      function GetLoopEntryVariableNext (LoopEntryVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetLoopEntryVariableNext);

      procedure CreateDeclaration (Item        : in     Dictionary.Symbol;
                                   Context     : in     Dictionary.Contexts;
                                   Comp_Unit   : in     ContextManager.UnitDescriptors;
                                   Loc         : in     LexTokenManager.Token_Position;
                                   Declaration :    out Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Declaration     from Comp_Unit,
      --#                              Dictionary.Dict,
      --#                              Loc &
      --#         Dictionary.Dict from *,
      --#                              Comp_Unit,
      --#                              Context,
      --#                              Item,
      --#                              Loc;
      pragma Inline (CreateDeclaration);

      procedure SetDeclarationScope (Declaration : in Dictionary.Symbol;
                                     Scope       : in Dictionary.Scopes);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Declaration,
      --#                              Scope;
      pragma Inline (SetDeclarationScope);

      procedure SetDeclarationContext (Declaration : in Dictionary.Symbol;
                                       Context     : in Dictionary.Contexts);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Context,
      --#                              Declaration;
      pragma Inline (SetDeclarationContext);

      procedure SetNextDeclaration (Current, Next : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              Current,
      --#                              Next;
      pragma Inline (SetNextDeclaration);

      function GetDeclarationItem (Declaration : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetDeclarationItem);

      function GetDeclarationScope (Declaration : Dictionary.Symbol) return Dictionary.Scopes;
      --# global in Dictionary.Dict;
      pragma Inline (GetDeclarationScope);

      function GetDeclarationContext (Declaration : Dictionary.Symbol) return Dictionary.Contexts;
      --# global in Dictionary.Dict;
      pragma Inline (GetDeclarationContext);

      function GetNextDeclaration (Declaration : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;
      pragma Inline (GetNextDeclaration);

      procedure SetHasDelayProperty (TheProcedure : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheProcedure;
      -- pre IsProcedure (TheProcedure);

      procedure SetTaskUsesUnprotectedVariables (TheTaskType : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheTaskType;
      -- pre IsTaskType (TheTaskType);

      procedure SetTaskUsesUncheckedConversion (TheTaskType : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheTaskType;
      -- pre IsTaskType (SetTaskUsesUncheckedConversion);

      procedure SetTaskAssignsFromExternal (TheTaskType : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheTaskType;
      -- pre IsTaskType (SetTaskUsesUncheckedConversion);

      procedure SetSubprogUsesUnprotectedVariables (SubprogSym : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              SubprogSym;
      --  pre IsSubprogram (SubprogSym)

      procedure SetUnprotectedReference (OwnVariable : in Dictionary.Symbol;
                                         ByThread    : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ByThread,
      --#                              OwnVariable;
      -- pre IsOwnVariable (OwnVariable);

      procedure SetSuspendsReference (OwnVariable : in Dictionary.Symbol;
                                      ByThread    : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              ByThread,
      --#                              OwnVariable;
      -- pre IsOwnVariable (OwnVariable);

      function HasDelayProperty (TheProcedure : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function DelayPropertyIsAccountedFor (TheProcedure : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      procedure MarkAccountsForDelay (TheProcedure : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              TheProcedure;

      procedure SetIsSuspendable (OwnVariable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnVariable;

      procedure SetHasInterruptProperty (OwnVariable : in Dictionary.Symbol);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnVariable;

      function GetIsSuspendable (OwnVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function GetHasInterruptProperty (OwnVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      procedure SetPriorityProperty (OwnVariable : in Dictionary.Symbol;
                                     TheValue    : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnVariable,
      --#                              TheValue;

      procedure SetIntegrityProperty (OwnVariable : in Dictionary.Symbol;
                                      TheValue    : in LexTokenManager.Lex_String);
      --# global in out Dictionary.Dict;
      --# derives Dictionary.Dict from *,
      --#                              OwnVariable,
      --#                              TheValue;

      function GetPriorityProperty (OwnVariable : Dictionary.Symbol)
                                   return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function GetIntegrityProperty (OwnVariable : Dictionary.Symbol)
                                   return LexTokenManager.Lex_String;
      --# global in Dictionary.Dict;

      function HasValidPriorityProperty (OwnVariable : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function TaskUsesUnprotectedVariables (TheTaskType : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function GetTaskUsesUncheckedConversion (TheTaskType : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function GetTaskAssignsFromExternal (TheTaskType : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function SubprogUsesUnprotectedVariables (SubprogSym : Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function GetUnprotectedReference (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function GetSuspendsReference (OwnVariable : Dictionary.Symbol) return Dictionary.Symbol;
      --# global in Dictionary.Dict;

      function ProcedureBodyIsHidden (Subprogram : in Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

      function TaskBodyIsHidden (TheTask : in Dictionary.Symbol) return Boolean;
      --# global in Dictionary.Dict;

   end RawDict;

   ------------------------------------------------------------------------
   --# inherit CommandLineData,
   --#         CommandLineHandler,
   --#         Dictionary,
   --#         ELStrings,
   --#         EStrings,
   --#         FileSystem,
   --#         LexTokenManager,
   --#         Maths,
   --#         RawDict,
   --#         ScreenEcho,
   --#         SPARK_IO,
   --#         SystemErrors,
   --#         XMLReport;
   package TargetData
   is
      procedure ReadTargetDataFile;
      --# global in     CommandLineData.Content;
      --#        in out Dictionary.Dict;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    LexTokenManager.State,
      --#                                    SPARK_IO.File_Sys;

      procedure OutputTargetDataFile (ToFile : in SPARK_IO.File_Type);
      --# global in     CommandLineData.Content;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives LexTokenManager.State,
      --#         SPARK_IO.File_Sys,
      --#         XMLReport.State       from CommandLineData.Content,
      --#                                    LexTokenManager.State,
      --#                                    SPARK_IO.File_Sys,
      --#                                    ToFile,
      --#                                    XMLReport.State;

   end TargetData;

   ---------------------------------------------------------------------------

   package body RawDict is separate;


   --------------------------------------------------------------------------------
   --                                SYMBOLS and ORDERING                        --
   --------------------------------------------------------------------------------

   function Declared_Before (Left, Right : in Symbol) return Boolean
   is
   begin
      return Left < Right;
   end Declared_Before;

   --------------------------------------------------------------------------------
   --                                SUBPROGRAMS                                 --
   --------------------------------------------------------------------------------

   procedure SetCurrentFileName (FileName : in EStrings.T)
   is
   begin
      Dict.FileName := FileName;
   end SetCurrentFileName;

   --------------------------------------------------------------------------------

   function GetRegion (Scope : Scopes) return Symbol
   is
   begin
      return Scope.Region;
   end GetRegion;

   --------------------------------------------------------------------------------

   procedure AddDeclaration (Item        : in     Symbol;
                             Comp_Unit   : in     ContextManager.UnitDescriptors;
                             Loc         : in     Location;
                             Scope       : in     Scopes;
                             Context     : in     Contexts;
                             Declaration :    out Symbol)
   --# global in out Dict;
   --# derives Declaration from Comp_Unit,
   --#                          Dict,
   --#                          Loc &
   --#         Dict        from *,
   --#                          Comp_Unit,
   --#                          Context,
   --#                          Item,
   --#                          Loc,
   --#                          Scope;
      is separate;

   --------------------------------------------------------------------------------

   function IsVisibleScope (Scope : Scopes) return Boolean
   is
   begin
      return Scope.TypeOfScope = Visible;
   end IsVisibleScope;

   --------------------------------------------------------------------------------

   function IsLocalScope (Scope : Scopes) return Boolean
   is
   begin
      return Scope.TypeOfScope = Local;
   end IsLocalScope;

   --------------------------------------------------------------------------------

   function IsPrivateScope (Scope : Scopes) return Boolean
   is
   begin
      return Scope.TypeOfScope = Privat;
   end IsPrivateScope;

   --------------------------------------------------------------------------------

   function IsLocalOrPrivateScope (Scope : Scopes) return Boolean
   is
   begin
      return  Scope.TypeOfScope = Local or else Scope.TypeOfScope = Privat;
   end IsLocalOrPrivateScope;

   --------------------------------------------------------------------------------

   function LocalScope (Region : Symbol) return Scopes
   is
   begin
      return Scopes'(Local, Region);
   end LocalScope;

   --------------------------------------------------------------------------------

   function PrivateScope (Region : Symbol) return Scopes
   is
   begin
      return Scopes'(Privat, Region);
   end PrivateScope;

   --------------------------------------------------------------------------------

   function TypeIsPrivate (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypePrivateTypeDeclaration (TheType) /= NullSymbol;
   end TypeIsPrivate;

   --------------------------------------------------------------------------------

   function TypeIsTagged (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeIsTagged (TheType) or else
        RawDict.GetTypeExtends (TheType) /= NullSymbol;
      -- derived type of tagged type also tagged
   end TypeIsTagged;

   --------------------------------------------------------------------------------

   function TypeIsExtendedTagged (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeExtends (TheType) /= NullSymbol;
   end TypeIsExtendedTagged;

   --------------------------------------------------------------------------------

   function TypeIsProtected (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TheType) = ProtectedType;
   end TypeIsProtected;

   --------------------------------------------------------------------------------

   function TypeIsTask (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TheType) = TaskType;
   end TypeIsTask;

   --------------------------------------------------------------------------------

   function TypeIsAccess (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TheType) = AccessType;
   end TypeIsAccess;

   --------------------------------------------------------------------------------

   function IsTypeMark (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = TypeSymbol;
   end IsTypeMark;

   --------------------------------------------------------------------------------

   function IsAnExtensionOf (RootType, ExtendedType : Symbol) return Boolean
   is
      Result        : Boolean := False;
      ExtendedLocal : Symbol;
   begin
      if TypeIsTagged (RootType) and then
        IsTypeMark (ExtendedType) and then             -- to trap null symbol
        TypeIsExtendedTagged (ExtendedType) then
         ExtendedLocal := ExtendedType;
         loop
            Result := RootType = ExtendedLocal;
            exit when Result;

            ExtendedLocal := RawDict.GetTypeExtends (ExtendedLocal);
            exit when ExtendedLocal = NullSymbol;
         end loop;
      end if;
      return Result;
   end IsAnExtensionOf;

   --------------------------------------------------------------------------------

   function TypeIsAbstract (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeIsAbstract (TheType);
   end TypeIsAbstract;

   --------------------------------------------------------------------------------

   function TypeIsAnnounced (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeAnnouncement (TheType) /= NullSymbol;
   end TypeIsAnnounced;

   --------------------------------------------------------------------------------

   function ConstantIsDeferred (TheConstant : Symbol) return Boolean
      --# global in Dict;
   is
      Result : Boolean := False;
   begin
      -- added if clause to esnure that generic ofrmal objects (that we regard as being constant)
      -- aren't checked to see if they are deferred, because they can't be
      if RawDict.GetSymbolDiscriminant (TheConstant) = ConstantSymbol then
         Result := RawDict.GetConstantDeferredConstantDeclaration (TheConstant) /= NullSymbol;
      end if;
      return Result;
   end ConstantIsDeferred;

   --------------------------------------------------------------------------------

   function GetDeclaration (Item : Symbol) return Symbol
      --# global in Dict;
      is separate;

   --------------------------------------------------------------------------------

   function GetParentType (TheSubtype : Symbol) return Symbol
      --# global in Dict;
   is
   begin
      return RawDict.GetTypeParent (TheSubtype);
   end GetParentType;

   --------------------------------------------------------------------------------

   function SubprogramSignatureIsWellformed (Abstraction : Abstractions;
                                             Subprogram  : Symbol) return Boolean
   is
      Result : Boolean;
   begin
      if RawDict.GetSymbolDiscriminant (Subprogram) = SubprogramSymbol then
         Result := RawDict.GetSubprogramSignatureIsWellformed (Abstraction, Subprogram);
      else
         Result := RawDict.GetTaskTypeSignatureIsWellformed (Abstraction, Subprogram);
      end if;
      return Result;
   end SubprogramSignatureIsWellformed;

   --------------------------------------------------------------------------------

   function GetAdaFunction (ProofFunction : Symbol) return Symbol
   is
   begin
      return RawDict.GetImplicitProofFunctionAdaFunction (ProofFunction);
   end GetAdaFunction;

   --------------------------------------------------------------------------------

   function GetImplicitProofFunction (Abstraction : Abstractions;
                                      TheFunction : Symbol) return Symbol
   is
   begin
      return RawDict.GetSubprogramImplicitProofFunction (Abstraction,
                                                         TheFunction);
   end GetImplicitProofFunction;

   --------------------------------------------------------------------------------

   function HasImplicitReturnVariable (Abstraction : Abstractions;
                                       TheFunction : Symbol) return Boolean
   is
   begin
      return RawDict.GetSubprogramImplicitReturnVariable (Abstraction,
                                                          TheFunction) /= NullSymbol;
   end HasImplicitReturnVariable;

   --------------------------------------------------------------------------------

   function GetImplicitReturnVariable (Abstraction : Abstractions;
                                       TheFunction : Symbol) return Symbol
   is
   begin
      return RawDict.GetSubprogramImplicitReturnVariable (Abstraction, TheFunction);
   end GetImplicitReturnVariable;

   --------------------------------------------------------------------------------

   function GetType (TheSymbol : Symbol) return Symbol
   is
      Result      : Symbol;
      LocalSymbol : Symbol;

      -- hidden procedure to call SystemErrors.  Hidden because we want GetType to remain a
      -- function and the call would have a side effect.  It seems reasonable to have a side
      -- effect in a function if execution terminates immediately thereafter
      procedure InvalidGetTypeRequest
      --# derives ;
      is
         --# hide InvalidGetTypeRequest;
      begin
         SystemErrors.FatalError (SystemErrors.InvalidSymbolTable, "in InvalidGetTypeRequest");
      end InvalidGetTypeRequest;

      function GetParameterConstraintType (TheSymbol : Symbol) return Symbol
      --# global in Dict;
      is
         TheParameter : Symbol;
         TheDimension : Positive;
         TheParameterType : Symbol;
         TheIndex : Symbol;
      begin
         -- Parameter constraint symbols are only used to communicate information about
         -- unconstrained parameters bewteen the WFFs and te VCG.  As such they don't really
         -- have a type other than for FDL declaration purposes where they behave as if they
         -- were subtypes.  For this purpose it is enough to return the type of matching index of the
         -- uncontrained type of teh obhject associated with the constraint symbol.
         -- So if parameter X is an unconstrained array of type T then the type of X__index__subtype__2 is
         -- the type of the 2nd dimension of T.

         -- The rather cumbersome use of RawDict calls here is caused by ordering problems between GetType and the
         -- higher-level Dictionary iterator calls that could otehrwise be used.

         TheParameter := RawDict.GetParameterConstraintParameter (TheSymbol);
         TheParameterType := RawDict.GetSubprogramParameterType (TheParameter);
         TheDimension := RawDict.GetParameterConstraintDimension (TheSymbol);

         -- Follow linked list of indexes
         TheIndex := RawDict.GetTypeFirstArrayIndex (TheParameterType);
         while TheDimension > 1 loop
            TheIndex := RawDict.GetNextArrayIndex (TheIndex);
            TheDimension := TheDimension - 1;
         end loop;

         return RawDict.GetArrayIndexType (TheIndex);
      end GetParameterConstraintType;

   begin -- GetType
      LocalSymbol := TheSymbol;

      -- If the symbol is a special variable used to store values on entry to a for loop
      -- then we "dereference" it back to the original variable before seeking the type
      if RawDict.GetSymbolDiscriminant (LocalSymbol) = LoopEntryVariableSymbol then
         LocalSymbol := RawDict.GetLoopEntryVariableOriginalVar (TheSymbol);
      end if;

      case RawDict.GetSymbolDiscriminant (LocalSymbol) is
         when TypeSymbol =>
            Result := GetParentType (LocalSymbol);
         when EnumerationLiteralSymbol =>
            Result := RawDict.GetEnumerationLiteralType (LocalSymbol);
         when RecordComponentSymbol =>
            Result := RawDict.GetRecordComponentType (LocalSymbol);
         when SubcomponentSymbol =>
            Result := RawDict.GetRecordComponentType (RawDict.GetSubcomponentComponent (LocalSymbol));
         when VariableSymbol =>
            Result := RawDict.GetVariableType (LocalSymbol);
         when QuantifiedVariableSymbol =>
            Result := RawDict.GetQuantifiedVariableType (LocalSymbol);
         when ImplicitReturnVariableSymbol =>
            Result := RawDict.GetSubprogramReturnType (RawDict.GetImplicitReturnVariableFunction (LocalSymbol));
         when ConstantSymbol =>
            Result := RawDict.GetConstantType (LocalSymbol);
         when SubprogramSymbol =>
            Result := RawDict.GetSubprogramReturnType (LocalSymbol);
         when ImplicitProofFunctionSymbol =>
            Result := RawDict.GetSubprogramReturnType (GetAdaFunction (LocalSymbol));
         when SubprogramParameterSymbol =>
            Result := RawDict.GetSubprogramParameterType (LocalSymbol);
         when ParameterConstraintSymbol =>
            Result := GetParameterConstraintType (LocalSymbol);
         when GenericParameterSymbol =>
            Result := RawDict.GetGenericParameterTypeOrObject (LocalSymbol);
            -- if its a type parameter then this is enough, we have the type mark
            if RawDict.GetGenericParameterKind (LocalSymbol) = GenericObjectParameter then
               -- its an object so above gave us a constant and next line gets its type
               Result := RawDict.GetConstantType (Result);
            end if;
         when LoopParameterSymbol =>
            Result := RawDict.GetLoopParameterType (LocalSymbol);
         when KnownDiscriminantSymbol =>
            Result := RawDict.GetDiscriminantTypeMark (LocalSymbol);
         when others =>
            InvalidGetTypeRequest; -- terminates execution
            Result := NullSymbol;  -- never returned, just there for DFA completeness
      end case;

      return Result;

   end GetType;

   ------------------------------------------------------------------------------

   function GetAccess (TheProtectedType : Symbol) return Symbol
   is
   begin
      return RawDict.GetTypeAccesses (TheProtectedType);
   end GetAccess;

   ------------------------------------------------------------------------------

   function DeReference (TheAccessType : Symbol) return Symbol
   is
   begin
      return RawDict.GetTypeAccesses (TheAccessType);
   end DeReference;

   ------------------------------------------------------------------------------

   function GetSubprogram (Parameter : Symbol) return Symbol
      --# global in Dict;
   is
   begin
      return RawDict.GetSubprogramParameterSubprogram (Parameter);
   end GetSubprogram;

   --------------------------------------------------------------------------------

   function GetContext (TheSymbol : Symbol) return Contexts is separate;

   --------------------------------------------------------------------------------

   function IsPackage (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = PackageSymbol;
   end IsPackage;

   --------------------------------------------------------------------------------

   function IsGenericPackage (TheSymbol : Symbol) return Boolean
   is
   begin
      return (RawDict.GetSymbolDiscriminant (TheSymbol) = PackageSymbol and then
                RawDict.GetPackageIsGeneric (TheSymbol));
   end IsGenericPackage;

   --------------------------------------------------------------------------------
   function GetRootPackage (ThePackage  : Symbol) return Symbol
   is
      CurrentPackage,
      NextPackage  : Symbol;
   begin
      SystemErrors.RTAssert (IsPackage (ThePackage),
                             SystemErrors.PreconditionFailure,
                             "In call to GetRootPackage");
      NextPackage := ThePackage;
      loop
         CurrentPackage := NextPackage;
         NextPackage := RawDict.GetPackageParent (CurrentPackage);
         exit when NextPackage = NullSymbol;
      end loop;
      return CurrentPackage;
   end GetRootPackage;

   --------------------------------------------------------------------------------

   function IsPrivatePackage (ThePackage : Symbol) return Boolean
   is
   begin
      SystemErrors.RTAssert (IsPackage (ThePackage),
                             SystemErrors.PreconditionFailure,
                             "In call to IsPrivatePackage");
      return RawDict.GetPackageIsPrivate (ThePackage);
   end IsPrivatePackage;

   --------------------------------------------------------------------------------

   function GetPackageParent (ThePackage : Symbol) return Symbol
   is
   begin
      SystemErrors.RTAssert (IsPackage (ThePackage),
                             SystemErrors.PreconditionFailure,
                             "In call to GetPackageParent");
      return RawDict.GetPackageParent (ThePackage);
   end GetPackageParent;

   --------------------------------------------------------------------------------

   function PackageDeclaresTaggedType (ThePackage : Symbol) return Boolean
   is
   begin
      SystemErrors.RTAssert (IsPackage (ThePackage),
                             SystemErrors.PreconditionFailure,
                             "In call to GetPackageDeclaresTaggedType");
      return RawDict.GetPackageDeclaresTaggedType (ThePackage);
   end PackageDeclaresTaggedType;

   --------------------------------------------------------------------------------

   function PackageExtendsAnotherPackage (ThePackage : Symbol) return Boolean
   is
   begin
      SystemErrors.RTAssert (IsPackage (ThePackage),
                             SystemErrors.PreconditionFailure,
                             "In call to PackageExtendsAnotherPackage");
      return RawDict.GetPackageExtends (ThePackage) /= NullSymbol;
   end PackageExtendsAnotherPackage;

   --------------------------------------------------------------------------------

   function GetPackageThatIsExtended (ThePackage : Symbol) return Symbol
   is
   begin
      SystemErrors.RTAssert (IsPackage (ThePackage),
                             SystemErrors.PreconditionFailure,
                             "In call to GetPackageThatIsExtended");
      return RawDict.GetPackageExtends (ThePackage);
   end GetPackageThatIsExtended;

   --------------------------------------------------------------------------------

   -- function GetPackageExtendedType (ThePackage : Symbol) return Symbol
   -- declared later after GetType which it uses

   --------------------------------------------------------------------------------

   function GetPackageOwner (ThePackage : Symbol) return Symbol
   is
      Current : Symbol;
   begin
      SystemErrors.RTAssert (IsPackage (ThePackage),
                             SystemErrors.PreconditionFailure,
                             "In call to GetPackageOwner");
      Current := ThePackage;
      loop
         exit when IsPrivatePackage (Current);
         Current := RawDict.GetPackageParent (Current);
         exit when Current = NullSymbol;
      end loop;

      if Current /= NullSymbol then
         Current := RawDict.GetPackageParent (Current);
      end if;

      return Current;
   end GetPackageOwner;

   --------------------------------------------------------------------------------

   function IsProperDescendent (Inner, Outer : Symbol) return Boolean
   is
      Current : Symbol;
   begin
      SystemErrors.RTAssert (IsPackage (Inner) and then IsPackage (Outer),
                             SystemErrors.PreconditionFailure,
                             "In call to IsProperDescendent");
      Current := Inner;
      loop
         Current := RawDict.GetPackageParent (Current);
         exit when Current = Outer;
         exit when Current = NullSymbol;
      end loop;

      return Current = Outer;
   end IsProperDescendent;

   --------------------------------------------------------------------------------

   function IsPrivateDescendent (Inner, Outer : Symbol) return Boolean
   is
      Current   : Symbol;
      IsPrivate : Boolean := False;
   begin
      SystemErrors.RTAssert (IsPackage (Inner) and then IsPackage (Outer),
                             SystemErrors.PreconditionFailure,
                             "In call to IsPrivateDescendent");
      Current := Inner;
      loop
         IsPrivate := IsPrivate or else IsPrivatePackage (Current);
         Current := RawDict.GetPackageParent (Current);
         exit when Current = Outer;
         exit when Current = NullSymbol;
      end loop;

      return Current = Outer and IsPrivate;
   end IsPrivateDescendent;

   --------------------------------------------------------------------------------

   function IsDescendentOfPrivateSibling (Candidate, ThePackage : Symbol)
                                        return Boolean
   is
      TheParent, Current, Next : Symbol;
   begin
      SystemErrors.RTAssert (IsPackage (ThePackage),
                             SystemErrors.PreconditionFailure,
                             "In call to IsDescendentOfPrivateSibling");
      TheParent := RawDict.GetPackageParent (ThePackage);
      Current := Candidate;
      loop
         Next := RawDict.GetPackageParent (Current);
         exit when Next = TheParent;
         exit when Next = NullSymbol;
         Current := Next;
      end loop;
      return Next = TheParent and IsPrivatePackage (Current);
   end IsDescendentOfPrivateSibling;

   --------------------------------------------------------------------------------

   function IsDeclared (Item : Symbol) return Boolean
   is
      Result : Boolean;
   begin

      case RawDict.GetSymbolDiscriminant (Item) is
         when TypeSymbol =>
            Result := RawDict.GetTypeDeclaration (Item) /= NullSymbol;
         when VariableSymbol =>
            Result := RawDict.GetVariableDeclaration (Item) /= NullSymbol;
         when ConstantSymbol =>
            Result := RawDict.GetConstantDeclaration (Item) /= NullSymbol;
         when QuantifiedVariableSymbol | SubprogramSymbol | PackageSymbol |
            LoopParameterSymbol | EnumerationLiteralSymbol |
            RecordComponentSymbol | ImplicitProofFunctionSymbol |
            SubcomponentSymbol |
            ImplicitReturnVariableSymbol | SubprogramParameterSymbol =>
            Result := True;
         when others =>
            Result := False;
      end case;

      return Result;

   end IsDeclared;

   --------------------------------------------------------------------------------

   function GetValue (TheConstant : Symbol) return LexTokenManager.Lex_String
   is
      Result : LexTokenManager.Lex_String;
   begin
      case RawDict.GetSymbolDiscriminant (TheConstant) is
         when ConstantSymbol =>
            Result := RawDict.GetConstantValue (TheConstant);
         when GenericParameterSymbol =>
            Result := LexTokenManager.Null_String;
         when others =>
            Result := LexTokenManager.Null_String;
      end case;
      return Result;
   end GetValue;

   --------------------------------------------------------------------------------

   function GetConstantExpNode (TheConstant : Symbol) return ExaminerConstants.RefType
   is
   begin
      return RawDict.GetConstantExpNode (TheConstant);
   end GetConstantExpNode;

   --------------------------------------------------------------------------------

   function ConstantExpIsWellFormed (TheConstant : Symbol) return Boolean
   is
   begin
      return RawDict.ConstantExpIsWellFormed (TheConstant);
   end ConstantExpIsWellFormed;

   --------------------------------------------------------------------------------

   function IsOwnVariable (Variable : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (Variable) = VariableSymbol and then
         RawDict.GetVariableOwnVariable (Variable) /= NullSymbol;
   end IsOwnVariable;

   --------------------------------------------------------------------------------

   function IsOwnTask (Variable : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (Variable) = VariableSymbol and then
         RawDict.GetVariableOwnTask (Variable) /= NullSymbol;
   end IsOwnTask;

   --------------------------------------------------------------------------------

   function GetOwner (Variable : Symbol) return Symbol
   is
      Result      : Symbol;
   begin
      if IsOwnVariable (Variable) then
         Result := RawDict.GetOwnVariableOwner
           (RawDict.GetVariableOwnVariable (Variable));
      elsif IsOwnTask (Variable) then
         Result := RawDict.GetOwnTaskOwner
           (RawDict.GetVariableOwnTask (Variable));
      else
         Result := RawDict.GetOwnVariableOwner
           (RawDict.GetConstituentSubject
              (RawDict.GetVariableConstituent (Variable)));
      end if;

      return Result;

   end GetOwner;

   --------------------------------------------------------------------------------

   function TypeIsFixedPoint (TypeMark : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TypeMark) = FixedPointType or else
        RawDict.GetTypeDiscriminant (TypeMark) = GenericFixedPointType;
   end TypeIsFixedPoint;

   --------------------------------------------------------------------------------

   function TypeIsInteger (TypeMark : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TypeMark) = IntegerType or else
        RawDict.GetTypeDiscriminant (TypeMark) = GenericIntegerType;
   end TypeIsInteger;

   --------------------------------------------------------------------------------

   function TypeIsModular (TypeMark : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TypeMark) = ModularType or else
        RawDict.GetTypeDiscriminant (TypeMark) = GenericModularType;
   end TypeIsModular;

   --------------------------------------------------------------------------------

   function GetPredefinedBooleanType return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedBooleanType);
   end GetPredefinedBooleanType;

   --------------------------------------------------------------------------------

   function GetFalse return Symbol
   is
   begin
      return Dict.Types.ThePredefinedLiterals (PredefinedFalse);
   end GetFalse;

   --------------------------------------------------------------------------------

   function GetTrue return Symbol
   is
   begin
      return Dict.Types.ThePredefinedLiterals (PredefinedTrue);
   end GetTrue;

   --------------------------------------------------------------------------------

   function GetUnknownTypeMark return Symbol
   is
   begin
      return Dict.Types.UnknownTypeMark;
   end GetUnknownTypeMark;

   --------------------------------------------------------------------------------

   function IsUnknownTypeMark (TheSymbol : Symbol) return Boolean
   is
   begin
      return TheSymbol = GetUnknownTypeMark;
   end IsUnknownTypeMark;

   --------------------------------------------------------------------------------

   function TypeIsUnknown (TypeMark : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return IsUnknownTypeMark (TypeMark);
   end TypeIsUnknown;

   --------------------------------------------------------------------------------

   function GetUniversalRealType return Symbol
   is
   begin
      return Dict.Types.TheUniversalTypes (UniversalRealType);
   end GetUniversalRealType;

   --------------------------------------------------------------------------------

   function GetUniversalFixedType return Symbol
   is
   begin
      return Dict.Types.TheUniversalTypes (UniversalFixedType);
   end GetUniversalFixedType;

   --------------------------------------------------------------------------------

   function IsUniversalRealType (TheSymbol : Symbol) return Boolean
   is
   begin
      return TheSymbol = GetUniversalRealType;
   end IsUniversalRealType;

   --------------------------------------------------------------------------------

   function GetUniversalIntegerType return Symbol
   is
   begin
      return Dict.Types.TheUniversalTypes (UniversalIntegerType);
   end GetUniversalIntegerType;

   --------------------------------------------------------------------------------

   function IsUniversalIntegerType (TheSymbol : Symbol) return Boolean
   is
   begin
      return TheSymbol = GetUniversalIntegerType;
   end IsUniversalIntegerType;

   --------------------------------------------------------------------------------

   function GetPredefinedIntegerType return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedIntegerType);
   end GetPredefinedIntegerType;

   --------------------------------------------------------------------------------

   function GetPredefinedLongIntegerType return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedLongIntegerType);
   end GetPredefinedLongIntegerType;

   --------------------------------------------------------------------------------

   function GetPredefinedFloatType return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedFloatType);
   end GetPredefinedFloatType;

   --------------------------------------------------------------------------------

   function GetPredefinedLongFloatType return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedLongFloatType);
   end GetPredefinedLongFloatType;

   --------------------------------------------------------------------------------

   function GetPredefinedDurationType return Symbol
   --# global in Dict;
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedDurationType);
   end GetPredefinedDurationType;
   pragma Unreferenced (GetPredefinedDurationType); -- not used at present

   --------------------------------------------------------------------------------

   function GetPredefinedTimeType return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedTimeType);
   end GetPredefinedTimeType;

   --------------------------------------------------------------------------------

   function GetPredefinedTimeSpanType return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedTimeSpanType);
   end GetPredefinedTimeSpanType;

   --------------------------------------------------------------------------------

   function GetPredefinedSuspensionObjectType return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedSuspensionObjectType);
   end GetPredefinedSuspensionObjectType;


--     -- Predefined generic types ----------------------------------------------

--     function GetPredefinedGenericLimitedType return Symbol
--     is
--     begin
--        return Dict.Types.ThePredefinedGenericTypes (PredefinedGenericLimitedType);
--     end GetPredefinedGenericLimitedType;

--     --------------------------------------------------------------------------------

--     function GetPredefinedGenericPrivateType return Symbol
--     is
--     begin
--        return Dict.Types.ThePredefinedGenericTypes (PredefinedGenericPrivateType);
--     end GetPredefinedGenericPrivateType;

--     --------------------------------------------------------------------------------

--     function GetPredefinedGenericOrderedDiscreteType return Symbol
--     is
--     begin
--        return Dict.Types.ThePredefinedGenericTypes (PredefinedGenericOrderedDiscreteType);
--     end GetPredefinedGenericOrderedDiscreteType;

--     --------------------------------------------------------------------------------

--     function GetPredefinedGenericUnorderedDiscreteType return Symbol
--     is
--     begin
--        return Dict.Types.ThePredefinedGenericTypes (PredefinedGenericUnorderedDiscreteType);
--     end GetPredefinedGenericUnorderedDiscreteType;

--     --------------------------------------------------------------------------------

--     function GetPredefinedGenericIntegerType return Symbol
--     is
--     begin
--        return Dict.Types.ThePredefinedGenericTypes (PredefinedGenericIntegerType);
--     end GetPredefinedGenericIntegerType;

--     --------------------------------------------------------------------------------

--     function GetPredefinedGenericModularType return Symbol
--     is
--     begin
--        return Dict.Types.ThePredefinedGenericTypes (PredefinedGenericModularType);
--     end GetPredefinedGenericModularType;

--     --------------------------------------------------------------------------------

--     function GetPredefinedGenericFloatingPointType return Symbol
--     is
--     begin
--        return Dict.Types.ThePredefinedGenericTypes (PredefinedGenericFloatingPointType);
--     end GetPredefinedGenericFloatingPointType;

--     --------------------------------------------------------------------------------

--     function GetPredefinedGenericFixedPointType return Symbol
--     is
--     begin
--        return Dict.Types.ThePredefinedGenericTypes (PredefinedGenericFixedPointType);
--     end GetPredefinedGenericFixedPointType;

--     --------------------------------------------------------------------------------

--     function GetPredefinedGenericArrayType return Symbol
--     is
--     begin
--        return Dict.Types.ThePredefinedGenericTypes (PredefinedGenericArrayType);
--     end GetPredefinedGenericArrayType;

--     --------------------------------------------------------------------------

   function GetNullVariable return Symbol
   is
   begin
      return Dict.NullVariable;
   end GetNullVariable;

   --------------------------------------------------------------------------------

   function GetThePartition return Symbol
   is
   begin
      return Dict.ThePartition;
   end GetThePartition;

   --------------------------------------------------------------------------------

   function IsPredefinedIntegerType (TheSymbol : Symbol) return Boolean
   is
   begin
      return TheSymbol = GetPredefinedIntegerType;
   end IsPredefinedIntegerType;

   --------------------------------------------------------------------------------

   function IsPredefinedFloatType (TheSymbol : Symbol) return Boolean
   is
   begin
      return TheSymbol = GetPredefinedFloatType;
   end IsPredefinedFloatType;

   --------------------------------------------------------------------------------

   function IsPredefinedTimeType (TheSymbol : Symbol) return Boolean
   is
   begin
      return TheSymbol = GetPredefinedTimeType or else
             TheSymbol = GetPredefinedTimeSpanType;
   end IsPredefinedTimeType;

   --------------------------------------------------------------------------------

   function IsPredefinedSuspensionObjectType (TheSymbol : Symbol) return Boolean
   is
   begin
      return
        CommandLineData.RavenscarSelected and then
        TheSymbol = GetPredefinedSuspensionObjectType;
   end IsPredefinedSuspensionObjectType;

   --------------------------------------------------------------------------------

   function GetBinaryOperatorType (Name  : SPSymbols.SPSymbol;
                                   Left  : Symbol;
                                   Right : Symbol) return Symbol
      is separate;

   --------------------------------------------------------------------------------

   function IsType (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         not TypeIsUnknown (TheSymbol) and then
         GetParentType (TheSymbol) = NullSymbol;
   end IsType;

   --------------------------------------------------------------------------------

   function IsTaskType (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
        TypeIsTask (TheSymbol);
   end IsTaskType;

   --------------------------------------------------------------------------------

   function IsProtectedType (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
        TypeIsProtected (TheSymbol);
   end IsProtectedType;

   --------------------------------------------------------------------------------

   -- declared somewhat out of position because it uses and must follow GetType
   function GetPackageExtendedType (ThePackage : Symbol) return Symbol
   is
      Declaration  : Symbol;
      Item         : Symbol;
      Result       : Symbol := NullSymbol;
   begin
      SystemErrors.RTAssert (IsPackage (ThePackage),
                             SystemErrors.PreconditionFailure,
                             "In call to GetPackageExtendedType");
      if RawDict.GetPackageExtends (ThePackage) /= NullSymbol then
         -- search for a tagged type
         Declaration := RawDict.GetPackageFirstVisibleDeclaration (ThePackage);
         loop
            exit when Declaration = NullSymbol;

            Item := RawDict.GetDeclarationItem (Declaration);
            if IsType (Item) and then
              TypeIsTagged (Item) then
               Result := Item;
               exit;
            end if;
            Declaration := RawDict.GetNextDeclaration (Declaration);
         end loop;
         -- if we haven't succeeded already then check private declarations
         if Result = NullSymbol then
            Declaration := RawDict.GetPackageFirstPrivateDeclaration (ThePackage);
            loop
               exit when Declaration = NullSymbol;

               Item := RawDict.GetDeclarationItem (Declaration);
               if IsType (Item) and then
                 TypeIsTagged (Item) then
                  Result := Item;
                  exit;
               end if;
               Declaration := RawDict.GetNextDeclaration (Declaration);
            end loop;
         end if;
      end if;
      return Result;
   end GetPackageExtendedType;

   --------------------------------------------------------------------------------

   function TypeIsFloatingPoint (TypeMark : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TypeMark) = FloatingPointType or else
        RawDict.GetTypeDiscriminant (TypeMark) = GenericFloatingPointType;
   end TypeIsFloatingPoint;

   --------------------------------------------------------------------------------

   function VisibleScope (ThePackage : Symbol) return Scopes
   is
   begin
      return Scopes'(Visible, ThePackage);
   end VisibleScope;

   --------------------------------------------------------------------------------

   function CurrentSymbol (CurrentIterator : Iterator) return Symbol
   is
      Current : Symbol;
   begin

      case CurrentIterator.Discriminant is
         when NullSymIterator .. EmbeddedPackageIterator =>
            Current := CurrentIterator.Current;
         when OwnVariableIterator .. AbstractOwnVariableIterator =>
            Current := RawDict.GetOwnVariableVariable (CurrentIterator.Current);
         when ConstituentIterator =>
            Current := RawDict.GetConstituentVariable (CurrentIterator.Current);
      end case;

      return Current;

   end CurrentSymbol;

   --------------------------------------------------------------------------------

   function GetEnclosingObject (Object : Symbol) return Symbol
   is
   begin
      return RawDict.GetSubcomponentObject (Object);
   end GetEnclosingObject;

   --------------------------------------------------------------------------------

   function GetMostEnclosingObject (Object : Symbol) return Symbol
   is
      Current : Symbol;
   begin
      Current := Object;
      loop
         exit when RawDict.GetSymbolDiscriminant (Current) /= SubcomponentSymbol;
         Current := GetEnclosingObject (Current);
      end loop;
      return Current;
   end GetMostEnclosingObject;

   --------------------------------------------------------------------------------

   function IsConstant (TheSymbol : Symbol) return Boolean
   is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (TheSymbol) is
         when ConstantSymbol =>
            Result := True;
         when GenericParameterSymbol =>
            -- we can regard a geenric formal object as a constant since we will only allow
            -- it to be instantiated with a constant
            Result := RawDict.GetGenericParameterKind (TheSymbol) = GenericObjectParameter;
         when others =>
            Result := False;
      end case;
      return Result;
   end IsConstant;

   --------------------------------------------------------------------------------

   function IsDeferredConstant (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsConstant (TheSymbol) and then ConstantIsDeferred (TheSymbol);
   end IsDeferredConstant;

   --------------------------------------------------------------------------------

   function FirstGlobalVariable (Abstraction : Abstractions;
                                 Subprogram  : Symbol) return Iterator
   is
      GlobalVariable  : Symbol;
      GlobalVariables : Iterator;
   begin

      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when SubprogramSymbol =>
            GlobalVariable := RawDict.GetSubprogramFirstGlobalVariable (Abstraction,
                                                                        Subprogram);
         when TypeSymbol => -- must be a task type
            GlobalVariable := RawDict.GetTaskTypeFirstGlobalVariable (Abstraction,
                                                                      Subprogram);
         when others =>
            GlobalVariable := NullSymbol;
      end case;
      if GlobalVariable = NullSymbol then
         GlobalVariables := NullIterator;
      else
         GlobalVariables := Iterator'(GlobalVariableIterator,
                                      IsAbstract,
                                      RawDict.GetGlobalVariableVariable (GlobalVariable),
                                      GlobalVariable);
      end if;
         --when others =>
         --   GlobalVariables := NullIterator;
         --end case;

      return GlobalVariables;

   end FirstGlobalVariable;

   --------------------------------------------------------------------------------

   function IsVariable (TheSymbol : Symbol) return Boolean
   is
      Result : Boolean;
   begin

      case RawDict.GetSymbolDiscriminant (TheSymbol) is
         when VariableSymbol | LoopEntryVariableSymbol | QuantifiedVariableSymbol |
           ImplicitReturnVariableSymbol | SubprogramParameterSymbol |
           LoopParameterSymbol =>
            Result := True;
         when others =>
            Result := False;
      end case;

      return Result;

   end IsVariable;

   --------------------------------------------------------------------------------

   function IsVariableOrSubcomponent (TheSymbol : Symbol) return Boolean
   is
      Result : Boolean;
   begin

      case RawDict.GetSymbolDiscriminant (TheSymbol) is
         when VariableSymbol | QuantifiedVariableSymbol |
            ImplicitReturnVariableSymbol | SubprogramParameterSymbol |
            LoopParameterSymbol | SubcomponentSymbol =>
            Result := True;
         when others =>
            Result := False;
      end case;

      return Result;

   end IsVariableOrSubcomponent;

   --------------------------------------------------------------------------------

   function GetFirstRecordSubcomponent (TheSymbol : Symbol) return Symbol
   is
      TheSubcomponent : Symbol;
   begin
      case RawDict.GetSymbolDiscriminant (TheSymbol) is
         when SubcomponentSymbol =>
            TheSubcomponent := RawDict.GetSubcomponentSubcomponents (TheSymbol);
         when VariableSymbol =>
            TheSubcomponent := RawDict.GetVariableSubComponents (TheSymbol);
         when SubprogramParameterSymbol =>
            TheSubcomponent := RawDict.GetSubprogramParameterSubComponents (TheSymbol);
         when others =>
            TheSubcomponent := NullSymbol;
      end case;
      return TheSubcomponent;
   end GetFirstRecordSubcomponent;

   --------------------------------------------------------------------------------

   function GetNextRecordSubcomponent (TheSubcomponent : Symbol) return Symbol
   is
   begin
      return RawDict.GetNextSubcomponent (TheSubcomponent);
   end GetNextRecordSubcomponent;

   --------------------------------------------------------------------------------

   function VariableIsInitialized (Variable : Symbol) return Boolean
   is
   begin
      return RawDict.GetVariableInitialized (Variable);
   end VariableIsInitialized;

   --------------------------------------------------------------------------------

   function VariableHasAddressClause (Variable : Symbol) return Boolean
   is
   begin
      return RawDict.GetVariableHasAddressClause (Variable);
   end VariableHasAddressClause;

   --------------------------------------------------------------------------------

   function VariableHasPragmaImport (Variable : Symbol) return Boolean
   is
   begin
      return RawDict.GetVariableHasPragmaImport (Variable);
   end VariableHasPragmaImport;

   --------------------------------------------------------------------------------

   function TypeSizeAttribute (TypeMark : Symbol) return LexTokenManager.Lex_String
   is
   begin
      return RawDict.GetTypeSizeAttribute (TypeMark);
   end TypeSizeAttribute;

   --------------------------------------------------------------------------------

   function VariableIsAliased (Variable : Symbol) return Boolean
   is
   begin
      return RawDict.GetVariableIsAliased (Variable);
   end VariableIsAliased;

   --------------------------------------------------------------------------------

   function VariableIsMarkedValid (TheVariable : Symbol) return Boolean
   is
   begin
      return RawDict.GetVariableMarkedValid (TheVariable);
   end VariableIsMarkedValid;

   --------------------------------------------------------------------------------

   function SubcomponentIsMarkedValid (TheSubcomponent : Symbol) return Boolean
   is
   begin
      return RawDict.GetSubcomponentMarkedValid (TheSubcomponent);
   end SubcomponentIsMarkedValid;

   --------------------------------------------------------------------------------

   function VariableOrSubcomponentIsMarkedValid (TheSym : Symbol) return Boolean
   is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (TheSym) is
         when VariableSymbol =>
            Result := VariableIsMarkedValid (TheSym);
         when SubcomponentSymbol =>
            Result := SubcomponentIsMarkedValid (TheSym);
         when LoopEntryVariableSymbol | QuantifiedVariableSymbol |
           ImplicitReturnVariableSymbol | SubprogramParameterSymbol |
           LoopParameterSymbol =>
            Result := True;
         when others =>
            SystemErrors.RTAssert (False,
                         SystemErrors.AssertionFailure,
                         "Unexpected case in VariableOrSubcomponentIsMarkedValid");
            Result := False;
      end case;
      return Result;
   end VariableOrSubcomponentIsMarkedValid;

   --------------------------------------------------------------------------------

   function IsAtomic (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.IsAtomic (TheSymbol);
   end IsAtomic;

   --------------------------------------------------------------------------------

   function IsPragmaAtomic (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.IsPragmaAtomic (TheSymbol);
   end IsPragmaAtomic;

   --------------------------------------------------------------------------------

   function IsKnownDiscriminant (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = KnownDiscriminantSymbol;
   end IsKnownDiscriminant;

   --------------------------------------------------------------------------------

   function SetsPriority (TheDiscriminant : Symbol) return Boolean
   is
   begin
      return RawDict.GetDiscriminantSetsPriority (TheDiscriminant);
   end SetsPriority;

   --------------------------------------------------------------------------------

   function IsSubprogramParameter (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = SubprogramParameterSymbol;
   end IsSubprogramParameter;

   --------------------------------------------------------------------------------

   function IsSubprogramParameterConstraint (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = ParameterConstraintSymbol;
   end IsSubprogramParameterConstraint;

   --------------------------------------------------------------------------------

   function IsFormalParameter (Subprogram, Parameter : Symbol) return Boolean
   is
   begin
      return IsSubprogramParameter (Parameter) and then
         GetSubprogram (Parameter) = Subprogram;
   end IsFormalParameter;

   --------------------------------------------------------------------------------

   function IsGenericFormalParameter (TheGeneric, Parameter : Symbol) return Boolean
   is
      ParameterOrConstant : Symbol;
      AssociatedGenericFormalOfConstant : Symbol;
   begin
      ParameterOrConstant := Parameter;
      -- if the generic formal is an object then a constant will have been declared in the visible
      -- scope of the geenric unit and linked to the list of its generic formal parameters.  In many
      -- cases, e.g. in the VCG, the symbol that we will find when looking up a generic formal object parameter
      -- will be the symbol of this implicitly-declared constant.  We want such constants to be correctly identified
      -- as generic formal parameters.  The following test checks for this case before going on to check whether
      -- the parameter supplied is the one we are looking for.
      if RawDict.GetSymbolDiscriminant (ParameterOrConstant) = ConstantSymbol then
         -- if its a constant then it might be an implicitly-declared constant associated with a generic formal
         AssociatedGenericFormalOfConstant := RawDict.GetConstantAssociatedGenericParameter (ParameterOrConstant);
         if AssociatedGenericFormalOfConstant /= NullSymbol then
            -- then it really is an implicitly-declared constant associated with the generic formal
            ParameterOrConstant := AssociatedGenericFormalOfConstant;
         end if;
      end if;
      -- now check whether Parameter, perhaps modified by code above, is actually a generic formal of the TheGeneric
      return RawDict.GetSymbolDiscriminant (ParameterOrConstant) = GenericParameterSymbol and then
         RawDict.GetGenericParameterOwningGeneric (ParameterOrConstant) = TheGeneric;
   end IsGenericFormalParameter;

   --------------------------------------------------------------------------------

   function IsGenericParameterSymbol (Parameter : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (Parameter) = GenericParameterSymbol;
   end IsGenericParameterSymbol;
   --------------------------------------------------------------------------------

   function IsSubprogram (TheSymbol : Symbol) return Boolean
   is
      Result : Boolean;
   begin

      case RawDict.GetSymbolDiscriminant (TheSymbol) is
         when SubprogramSymbol | ImplicitProofFunctionSymbol =>
            Result := True;
         when others =>
            Result := False;
      end case;

      return Result;

   end IsSubprogram;

   --------------------------------------------------------------------------------

   function IsGenericSubprogram (TheSymbol : Symbol) return Boolean
   is
      Result : Boolean;
   begin

      case RawDict.GetSymbolDiscriminant (TheSymbol) is
         when SubprogramSymbol =>
            Result := RawDict.GetSubprogramIsGeneric (TheSymbol);
         when ImplicitProofFunctionSymbol =>
            Result := RawDict.GetSubprogramIsGeneric
              (RawDict.GetImplicitProofFunctionAdaFunction (TheSymbol));
         when others =>
            Result := False;
      end case;

      return Result;

   end IsGenericSubprogram;

   --------------------------------------------------------------------------------

   function GetPrecondition (Abstraction : Abstractions;
                             Subprogram  : Symbol) return ExaminerConstants.RefType
   is
      Result : ExaminerConstants.RefType;
   begin
      if IsSubprogram (Subprogram) then
         Result :=  RawDict.GetSubprogramPrecondition (Abstraction, Subprogram);
      elsif IsTaskType (Subprogram) then
         Result := 0;
      else
         Result := 0; -- to avoid DF error
         SystemErrors.RTAssert (False,
                                SystemErrors.InvalidSymbolTable,
                                "Invalid call to GetPrecondition");
      end if;
      return Result;
   end GetPrecondition;

   --------------------------------------------------------------------------------

   function HasPrecondition (Abstraction : Abstractions;
                             Subprogram  : Symbol) return Boolean
   is
   begin
      return GetPrecondition (Abstraction, Subprogram) /= 0;
   end HasPrecondition;

   --------------------------------------------------------------------------------

   function GetPostcondition (Abstraction     : Abstractions;
                              CompilationUnit : Symbol) return ExaminerConstants.RefType
   is
      Postcondition : ExaminerConstants.RefType;
      ValidRequest  : Boolean;
   begin
      --# accept Flow, 10, "not ineffective, used in RTAssert (which derives null from it)";
      ValidRequest := True;
      --# end accept;

      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when PackageSymbol =>
            Postcondition := RawDict.GetPackagePostcondition (Abstraction,
                                                              CompilationUnit);
         when SubprogramSymbol =>
            Postcondition := RawDict.GetSubprogramPostcondition (Abstraction,
                                                                 CompilationUnit);
         when TypeSymbol =>
            if IsTaskType (CompilationUnit) then
               Postcondition := 0; -- valid but pointless request for a task's postcondition
            else
               Postcondition := 0; -- to avoid DF error
               --# accept Flow, 10, "not ineffective, used in RTAssert (which derives null from it)";
               ValidRequest := False;
               --# end accept;
            end if;
         when others =>
            Postcondition := 0; -- to avoid DF error
            --# accept Flow, 10, "not ineffective, used in RTAssert (which derives null from it)";
            ValidRequest := False;
            --# end accept;
      end case;
      SystemErrors.RTAssert (ValidRequest,
                             SystemErrors.InvalidSymbolTable,
                             "Invalid call to GetPostcondition");

      return Postcondition;
   end GetPostcondition;

   --------------------------------------------------------------------------------

   function HasPostcondition (Abstraction     : Abstractions;
                              CompilationUnit : Symbol) return Boolean
   is
   begin
      return GetPostcondition (Abstraction, CompilationUnit) /= 0;
   end HasPostcondition;

   --------------------------------------------------------------------------------

   function IsImport (Abstraction  : Abstractions;
                      TheProcedure : Symbol;
                      Variable     : Symbol) return Boolean is separate;

   --------------------------------------------------------------------------------

   function IsExport (Abstraction  : Abstractions;
                      TheProcedure : Symbol;
                      Variable     : Symbol) return Boolean is separate;

   --------------------------------------------------------------------------------

   function IsImportExport (Abstraction  : Abstractions;
                            TheProcedure : Symbol;
                            Variable     : Symbol) return Boolean
   is
   begin
      return IsImport (Abstraction, TheProcedure, Variable) and then
         IsExport (Abstraction, TheProcedure, Variable);
   end IsImportExport;

   --------------------------------------------------------------------------------

--     function IsPackage (TheSymbol : Symbol) return Boolean
--     is
--     begin
--        return RawDict.GetSymbolDiscriminant (TheSymbol) = PackageSymbol;
--     end IsPackage;

   --------------------------------------------------------------------------------

   function FirstRecordComponent (TheRecordType : Symbol) return Iterator
   is
      Component  : Symbol;
      Components : Iterator;
   begin

      Component := RawDict.GetTypeFirstRecordComponent (TheRecordType);

      if Component = NullSymbol then
         Components := NullIterator;
      else
         Components := Iterator'(RecordComponentIterator,
                                 IsAbstract,
                                 Component,
                                 NullSymbol);
      end if;

      return Components;

   end FirstRecordComponent;

   --------------------------------------------------------------------------------
   -- This function is to do with iterating over the fields of extended tagged records.
   -- It is declared here because it is used by FirstExtendedRecordComponent and NextSymbol
   function BackTrackUpInheritChain (StartSym, StopSym : Symbol) return Symbol
   --# global in Dict;
   is
      Result        : Symbol;
      CurrentRecord : Symbol;
   begin
      -- Starting at extended record StartSym follow the Inherit pointers to the
      -- record _preceding_ StopSym.  If StartSym = StopSym return NullSymbol.
      if StartSym = StopSym then
         Result := NullSymbol;
      else
         CurrentRecord := StartSym;
         loop
            exit when GetType (CurrentSymbol (FirstRecordComponent (CurrentRecord))) = StopSym;
            CurrentRecord := GetType (CurrentSymbol (FirstRecordComponent (CurrentRecord)));
         end loop;
         Result := CurrentRecord;
      end if;
      return Result;
   end BackTrackUpInheritChain;

   --------------------------------------------------------------------------------
   --Moved to below GetScope to avoid callingorder problem
   --function NextSymbol (Previous : Iterator) return Iterator is separate;
   --------------------------------------------------------------------------------

   function FirstEnumerationLiteral (EnumerationType : Symbol) return Iterator
   is
      Literal  : Symbol;
      Literals : Iterator;
   begin

      Literal := RawDict.GetTypeFirstEnumerationLiteral (EnumerationType);

      if Literal = NullSymbol then
         Literals := NullIterator;
      else
         Literals := Iterator'(EnumerationLiteralIterator,
                               IsAbstract,
                               Literal,
                               NullSymbol);
      end if;

      return Literals;

   end FirstEnumerationLiteral;

   --------------------------------------------------------------------------------

   function GetPositionNumber (Literal : Symbol) return LexTokenManager.Lex_String
   is
   begin
      return RawDict.GetEnumerationLiteralPosition (Literal);
   end GetPositionNumber;

   --------------------------------------------------------------------------------

   function GetRecordType (Component : Symbol) return Symbol
   is
      TheComponent : Symbol;
   begin
      case RawDict.GetSymbolDiscriminant (Component) is
         when RecordComponentSymbol =>
            TheComponent := Component;
         when others =>
            TheComponent := RawDict.GetSubcomponentComponent (Component);
      end case;
      return RawDict.GetRecordComponentRecordType (TheComponent);
   end GetRecordType;

   --------------------------------------------------------------------------------

   function GetLocalScope (Scope : Scopes) return Scopes
   is
   begin
      return LocalScope (GetRegion (Scope));
   end GetLocalScope;

   --------------------------------------------------------------------------------

   function GetPrivateScope (Scope : Scopes) return Scopes
   is
   begin
      return PrivateScope (GetRegion (Scope));
   end GetPrivateScope;
   pragma Unreferenced (GetPrivateScope); -- not used at present

   --------------------------------------------------------------------------------

   function GetPredefinedPackageStandard return Symbol
   is
   begin
      return Dict.Packages.PackageStandard;
   end GetPredefinedPackageStandard;

   --------------------------------------------------------------------------------

   function PredefinedScope return Scopes
      --# global in Dict;
   is
   begin
      return VisibleScope (GetPredefinedPackageStandard);
   end PredefinedScope;

   --------------------------------------------------------------------------------

   function GlobalScope return Scopes
   is
   begin
      return LocalScope (GetPredefinedPackageStandard);
   end GlobalScope;

   --------------------------------------------------------------------------------

   function GetScope (Item : Symbol) return Scopes is separate;

   --------------------------------------------------------------------------------

   function NextSymbol (Previous : Iterator) return Iterator is separate;

   --------------------------------------------------------------------------------

   function GetEnclosingScope (Scope : Scopes) return Scopes
   is
      Region : Symbol;
      EnclosingScope : Scopes;
   begin
      Region := GetRegion (Scope);
      EnclosingScope := GetScope (Region);
      if IsSubprogram (Region) and then
         IsLocalScope (Scope) and then
         (IsVisibleScope (EnclosingScope) or else IsPrivateScope (EnclosingScope)) then
         EnclosingScope := GetLocalScope (EnclosingScope);
      elsif IsProtectedType (Region) and then
        IsLocalScope (Scope) then
         EnclosingScope := GetLocalScope (EnclosingScope);
      elsif IsTaskType (Region) and then
        IsLocalScope (Scope) then
         EnclosingScope := GetLocalScope (EnclosingScope);
      end if;
      return EnclosingScope;
   end GetEnclosingScope;

   --------------------------------------------------------------------------------

   function IsPredefinedScope (Scope : Scopes) return Boolean
   is
   begin
      return Scope = PredefinedScope;
   end IsPredefinedScope;

   --------------------------------------------------------------------------------

   function IsGlobalScope (Scope : Scopes) return Boolean
   is
   begin
      return Scope = GlobalScope;
   end IsGlobalScope;

   --------------------------------------------------------------------------------

   function IsLibraryLevel (Scope : Scopes) return Boolean
   is
   begin
      return IsPackage (GetRegion (Scope)) and then
        (GetRegion (Scope) = GetPredefinedPackageStandard or else
           GetRegion (GetEnclosingScope (Scope)) = GetPredefinedPackageStandard);
   end IsLibraryLevel;

   --------------------------------------------------------------------------------

   function IsLocal (Inner, Outer : Scopes) return Boolean
   is
      CurrentScope  : Scopes;
      CurrentRegion : Symbol;
      OuterRegion   : Symbol;
   begin

      CurrentScope := Inner;
      OuterRegion := GetRegion (Outer);

      loop
         CurrentRegion := GetRegion (CurrentScope);
         exit when CurrentRegion = OuterRegion;
         exit when CurrentRegion = GetPredefinedPackageStandard;
         CurrentScope := GetEnclosingScope (CurrentScope);
      end loop;

      return CurrentRegion = OuterRegion;

   end IsLocal;

   --------------------------------------------------------------------------------

   function GetEnclosingPackage (Scope : Scopes) return Symbol
   is
      Current : Scopes;
      Region  : Symbol;
   begin

      Current := Scope;

      loop
         Region := GetRegion (Current);
         exit when IsPackage (Region);
         Current := GetEnclosingScope (Current);
      end loop;

      return Region;

   end GetEnclosingPackage;

   --------------------------------------------------------------------------------

   function GetLibraryPackage (Scope : Scopes) return Symbol
   is
      CurrPackage  : Symbol;
      EnclPackage  : Symbol;
   begin
      CurrPackage := GetEnclosingPackage (Scope);
      if CurrPackage /= GetPredefinedPackageStandard then
         loop
            exit when RawDict.GetPackageParent (CurrPackage) /= NullSymbol;
            EnclPackage := GetEnclosingPackage (GetScope (CurrPackage));
            exit when EnclPackage = GetPredefinedPackageStandard;
            CurrPackage := EnclPackage;
         end loop;
      end if;

      return CurrPackage;
   end GetLibraryPackage;


   --------------------------------------------------------------------------------
   function GetConstantRulePolicyForCurrentScope (TheConstant : Symbol;
                                                  TheScope    : Scopes) return RulePolicies
   --# global in Dict;
   is
      NextRP     : Symbol;
      RulePolicy : RulePolicies := Unspecified;
   begin
      NextRP := RawDict.GetConstantFirstRulePolicy (TheConstant);

      while NextRP /= NullSymbol loop
         if RawDict.GetRulePolicyScope (NextRP) = TheScope then
            RulePolicy := RawDict.GetRulePolicyValue (NextRP);
            exit;
         end if;

         NextRP := RawDict.GetNextRulePolicy (NextRP);
      end loop;
      return RulePolicy;
   end GetConstantRulePolicyForCurrentScope;

   --------------------------------------------------------------------------------

   function GetConstantRulePolicy (TheConstant : Symbol;
                                   TheScope    : Scopes) return RulePolicies
   is
      RulePolicy   : RulePolicies;
      CurrentScope : Scopes;

   begin
      CurrentScope := TheScope;

      -- check the list of policies for the current scope and any enclosing scope.
      loop
         RulePolicy := GetConstantRulePolicyForCurrentScope (TheConstant, CurrentScope);
         exit when RulePolicy /= Unspecified;

         exit when IsPredefinedScope (CurrentScope);

         -- if we have a local package scope check the visible scope for this package next
         if IsLocalScope (CurrentScope) and then
           RawDict.GetSymbolDiscriminant (GetRegion (CurrentScope)) = PackageSymbol then
            CurrentScope := VisibleScope (GetRegion (CurrentScope));
         else
            CurrentScope := GetEnclosingScope (CurrentScope);
         end if;
      end loop;

      -- if still unspecified then check the declaritive region for the constant.
      if RulePolicy = Unspecified then
         CurrentScope := GetScope (TheConstant);
         RulePolicy := GetConstantRulePolicyForCurrentScope (TheConstant, CurrentScope);
      end if;

      return RulePolicy;
   end GetConstantRulePolicy;

   --------------------------------------------------------------------------------

   function IsConstantRulePolicyPresent (TheConstant : Symbol;
                                         TheScope    : Scopes) return Boolean
   is
   begin
      return GetConstantRulePolicyForCurrentScope (TheConstant, TheScope) /= Unspecified;
   end IsConstantRulePolicyPresent;

   --------------------------------------------------------------------------------

   function TypeIsWellformed (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeWellformed (TheType);
   end TypeIsWellformed;

   --------------------------------------------------------------------------------

   function TypeIsIncompleteHere (TypeMark : Symbol; Scope : Scopes) return Boolean
   is
      Result : Boolean;

      function DescendentIsPrivate (Inner, Outer : Symbol) return Boolean
         --# global in Dict;
      is
         Current : Symbol;
      begin
         Current := Inner;
         loop
            exit when RawDict.GetPackageIsPrivate (Current);
            Current := RawDict.GetPackageParent (Current);
            exit when Current = Outer;
            exit when Current = NullSymbol;
         end loop;

         return Current /= Outer and Current /= NullSymbol;
      end DescendentIsPrivate;

      function IsPrivateSeeingDescendent (Scope      : Scopes;
                                         ThePackage : Symbol) return Boolean
         --# global in Dict;
      is
         LibPackage : Symbol;
      begin
         LibPackage := GetLibraryPackage (Scope);
         return IsProperDescendent (LibPackage, ThePackage) and then
            (Scope /= VisibleScope (LibPackage) or else
             DescendentIsPrivate (LibPackage, ThePackage));
      end IsPrivateSeeingDescendent;

   begin
      if IsDeclared (TypeMark) then
         Result :=
           (not ((not TypeIsPrivate (TypeMark) and then    -- TypeMark is declared in the visible part of package spec
                    IsVisibleScope (GetScope (TypeMark)))  -- and isn't private
                   or else
                   IsLocal (Scope, GetScope (TypeMark)) -- Scope is Local relative to TypeMark (e.g. in same package body)
                   or else
                   IsPrivateSeeingDescendent (Scope,
                                              GetRegion (GetScope (TypeMark)))) or else
               -- if we are in the visible scope of the place where the type is declared it can't be complete yet
              Scope = VisibleScope (GetRegion (GetScope (TypeMark))));
      else
         -- No Declaration - must be an announced type with no completion
         Result := True;
      end if;
      return Result;
   end TypeIsIncompleteHere;

   function TypeIsPrivateHere (TypeMark : Symbol;
                               Scope    : Scopes) return Boolean
   --# global in Dict;
   is

   begin
      return TypeIsPrivate (TypeMark) and then -- type must be private to be private here!
        TypeIsIncompleteHere (TypeMark, Scope);
   end TypeIsPrivateHere;

   -------------------------------------------------------------------------------

   function TypeIsReal (TypeMark : Symbol) return Boolean
   is
   begin
      return TypeIsFixedPoint (TypeMark) or else TypeIsFloatingPoint (TypeMark);
   end TypeIsReal;

   --------------------------------------------------------------------------------

   function IsRealTypeMark (TheSymbol : Symbol;
                            Scope     : Scopes) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         TypeIsReal (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsRealTypeMark;

   --------------------------------------------------------------------------------

   function GetRootType (TypeMark : Symbol) return Symbol
   is
      Current : Symbol;
   begin

      Current := TypeMark;

      if not TypeIsUnknown (Current) then -- don't try and get parent of unknown
         if IsSubprogramParameterConstraint (Current) then
            -- synthetic index type of unconst array params, first get type of the constraint
            Current := GetType (Current);
         end if;

         -- loop through possible chain of subtype to get actual type
         loop
            exit when IsType (Current);
            Current := GetParentType (Current);
         end loop;
      end if;

      return Current;

   end GetRootType;

   --------------------------------------------------------------------------------

   function GetBaseType (TypeMark : Symbol) return Symbol
   is
      Result : Symbol;
   begin
      Result := GetRootType (TypeMark); -- go to first named subtype
      if RawDict.GetTypeDerived (Result) then
         -- find base type
         Result := RawDict.GetTypeBaseType (Result);
      end if;
      return Result;
   end GetBaseType;

   --------------------------------------------------------------------------------

   function GetRootOfExtendedType (TypeMark : Symbol) return Symbol
   is
   begin
      return RawDict.GetTypeExtends (TypeMark);
   end GetRootOfExtendedType;

   --------------------------------------------------------------------------------

   function GetEnumerationLiteral (EnumerationType : Symbol;
                                   Position        : LexTokenManager.Lex_String) return Symbol
   is
      Literals : Iterator;
   begin
      Literals := FirstEnumerationLiteral (GetRootType (EnumerationType));
      loop
         exit when LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => GetPositionNumber (CurrentSymbol (Literals)),
                                                                        Lex_Str2 => Position) = LexTokenManager.Str_Eq;
         Literals := NextSymbol (Literals);
      end loop;
      return CurrentSymbol (Literals);
   end GetEnumerationLiteral;

   --------------------------------------------------------------------------------

   function IsIntegerTypeMark (TheSymbol : Symbol;
                               Scope     : Scopes) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         TypeIsInteger (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsIntegerTypeMark;

   --------------------------------------------------------------------------------

   function IsModularTypeMark (TheSymbol : Symbol;
                               Scope     : Scopes) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         TypeIsModular (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsModularTypeMark;

   --------------------------------------------------------------------------------

   function IsIntegerType (TheSymbol : Symbol; Scope : Scopes) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
         TypeIsInteger (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsIntegerType;

   --------------------------------------------------------------------------------

   function IsModularType (TheSymbol : Symbol; Scope : Scopes) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
         TypeIsModular (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsModularType;

   --------------------------------------------------------------------------------

   function IsSubtype (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         GetParentType (TheSymbol) /= NullSymbol;
   end IsSubtype;

   --------------------------------------------------------------------------------

   function CompatibleTypes (Scope : Scopes;
                             Left  : Symbol;
                             Right : Symbol) return Boolean
      is separate;

   --------------------------------------------------------------------------------

   function GetPredefinedCharacterType return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedCharacterType);
   end GetPredefinedCharacterType;

   --------------------------------------------------------------------------------

   function IsPredefinedCharacterType (TheSymbol : Symbol) return Boolean
   is
   begin
      return TheSymbol = GetPredefinedCharacterType;
   end IsPredefinedCharacterType;

   --------------------------------------------------------------------------------

   function TypeIsEnumeration (TypeMark : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TypeMark) = EnumerationType or else
        RawDict.GetTypeDiscriminant (TypeMark) = GenericOrderedDiscreteType;
   end TypeIsEnumeration;

   --------------------------------------------------------------------------------

   function TypeIsGeneric (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TheType) in GenericType;
   end TypeIsGeneric;

   --------------------------------------------------------------------------------

   function TypeIsGenericArray (TheType : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TheType) = GenericArrayType;
   end TypeIsGenericArray;

   --------------------------------------------------------------------------------

   function TypeIsGenericDiscrete (TypeMark : Symbol) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.GetTypeDiscriminant (TypeMark) in GenericDiscreteType;
   end TypeIsGenericDiscrete;

   --------------------------------------------------------------------------------

   function TypeIsGenericUnOrderedDiscrete (TypeMark : Symbol) return Boolean
   --# global in Dict;
   is
   begin
      return RawDict.GetTypeDiscriminant (TypeMark) = GenericUnorderedDiscreteType;
   end TypeIsGenericUnOrderedDiscrete;

   --------------------------------------------------------------------------------

   function TypeIsDiscrete (TypeMark : Symbol) return Boolean
   is
   begin
      return
        TypeIsEnumeration (TypeMark) or else
        TypeIsInteger (TypeMark) or else
        TypeIsModular (TypeMark) or else
        TypeIsGenericDiscrete (TypeMark);
   end TypeIsDiscrete;

   --------------------------------------------------------------------------------

   function TypeIsNumeric (TypeMark : Symbol) return Boolean
   is
   begin
      return
         TypeIsInteger (TypeMark) or else
         TypeIsReal (TypeMark) or else
         TypeIsModular (TypeMark);
   end TypeIsNumeric;

   --------------------------------------------------------------------------------

   function TypeIsScalar (TypeMark : Symbol) return Boolean
   is
   begin
      return TypeIsDiscrete (TypeMark) or else TypeIsNumeric (TypeMark);
   end TypeIsScalar;

   --------------------------------------------------------------------------------

   function FirstArrayIndex (TypeMark : Symbol) return Iterator
   is
      ArrayIndex   : Symbol;
      ArrayIndices : Iterator;
   begin

      ArrayIndex := RawDict.GetTypeFirstArrayIndex (TypeMark);

      if ArrayIndex = NullSymbol then
         ArrayIndices := NullIterator;
      else
         ArrayIndices := Iterator'(ArrayIndexIterator,
                                   IsAbstract,
                                   RawDict.GetArrayIndexType (ArrayIndex),
                                   ArrayIndex);
      end if;

      return ArrayIndices;

   end FirstArrayIndex;

   --------------------------------------------------------------------------------

   function IsPrivateType (TheSymbol : Symbol; Scope : Scopes) return Boolean
   is
   begin
      return IsType (TheSymbol) and then TypeIsPrivateHere (TheSymbol, Scope);
   end IsPrivateType;

   --------------------------------------------------------------------------------

   function IsLimitedPrivateType (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
         RawDict.GetTypeLimitedPrivate (TheSymbol);
   end IsLimitedPrivateType;

   --------------------------------------------------------------------------------

   function IsQuantifiedVariable (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = QuantifiedVariableSymbol;
   end IsQuantifiedVariable;

   --------------------------------------------------------------------------------

   function TypeIsAbstractProof (TypeMark : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TypeMark) = AbstractProofType;
   end TypeIsAbstractProof;

   --------------------------------------------------------------------------------

   function TypeIsArray (TypeMark : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TypeMark) = ArrayType or else
        RawDict.GetTypeDiscriminant (TypeMark) = GenericArrayType;
   end TypeIsArray;

   --------------------------------------------------------------------------------

   function ArrayTypeIsConstrained (TypeMark : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return RawDict.GetTypeConstrained (TypeMark);
   end ArrayTypeIsConstrained;

   --------------------------------------------------------------------------------

   function IsUnconstrainedArrayType (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
         TypeIsArray (TheSymbol) and then
         not ArrayTypeIsConstrained (TheSymbol);
   end IsUnconstrainedArrayType;

   --------------------------------------------------------------------------------

   function IsUnconstrainedTaskType (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsTaskType (TheSymbol) and then
         not RawDict.GetTypeConstrained (TheSymbol);
   end IsUnconstrainedTaskType;

   --------------------------------------------------------------------------------

   function IsUnconstrainedProtectedType (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsProtectedType (TheSymbol) and then
         not RawDict.GetTypeConstrained (TheSymbol);
   end IsUnconstrainedProtectedType;

   --------------------------------------------------------------------------------

   function IsArrayAttribute (Name     : LexTokenManager.Lex_String;
                              TypeMark : Symbol) return Boolean
   is
      Result : Boolean;
   begin

      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                              Lex_Str2 => LexTokenManager.First_Token) = LexTokenManager.Str_Eq or else
        LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                             Lex_Str2 => LexTokenManager.Last_Token) = LexTokenManager.Str_Eq then
         Result := TypeIsArray (TypeMark);
      elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                                 Lex_Str2 => LexTokenManager.Length_Token) = LexTokenManager.Str_Eq or else
        LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                             Lex_Str2 => LexTokenManager.Range_Token) = LexTokenManager.Str_Eq or else
        LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                             Lex_Str2 => LexTokenManager.Component_Size_Token) = LexTokenManager.Str_Eq then
         Result := True;
      else
         Result := False;
      end if;

      return Result;

   end IsArrayAttribute;

   --------------------------------------------------------------------------------

   function OwnVariableHasType (OwnVariable : Symbol;
                                Scope       : Scopes) return Boolean
   is
      TypeMark : Symbol;
      Result   : Boolean;
   begin

      TypeMark := GetType (OwnVariable);

      if TypeIsUnknown (TypeMark) then
         Result := False;
      elsif TypeIsAbstractProof (TypeMark) then
         Result := False;
      elsif RawDict.GetOwnVariableTyped (RawDict.GetVariableOwnVariable (OwnVariable)) then
         Result := True;
      else
         Result := IsVisibleScope (GetScope (TypeMark)) or else
            IsLocal (Scope, GetScope (TypeMark));
      end if;

      return Result;

   end OwnVariableHasType;

   --------------------------------------------------------------------------------

   function OwnVariableIsAnnounced (Variable : Symbol) return Boolean
   is
   begin
      return RawDict.GetOwnVariableAnnounced (RawDict.GetVariableOwnVariable (Variable));
   end OwnVariableIsAnnounced;

   --------------------------------------------------------------------------------

   function OwnVariableIsInitialized (Variable : Symbol) return Boolean
   is
   begin
      return RawDict.GetOwnVariableInitialized (RawDict.GetVariableOwnVariable (Variable));
   end OwnVariableIsInitialized;

   --------------------------------------------------------------------------------

   function OwnVariableIsAbstract (Variable : Symbol) return Boolean
   is
   begin
      return RawDict.GetOwnVariableConstituents (RawDict.GetVariableOwnVariable (Variable)) /= NullSymbol;
   end OwnVariableIsAbstract;

   --------------------------------------------------------------------------------

   function FirstConstituent (Subject : Symbol) return Iterator
   is
      Constituent  : Symbol;
      Constituents : Iterator;
   begin

      Constituent := RawDict.GetOwnVariableConstituents (RawDict.GetVariableOwnVariable (Subject));

      if Constituent = NullSymbol then
         Constituents := NullIterator;
      else
         Constituents := Iterator'(ConstituentIterator,
                                   IsAbstract,
                                   Constituent,
                                   NullSymbol);
      end if;

      return Constituents;

   end FirstConstituent;

   --------------------------------------------------------------------------------

   function GetMainProgram return Symbol
   is
   begin
      return Dict.Main.Subprogram;
   end GetMainProgram;

   --------------------------------------------------------------------------------

   -- special handling that affects the main subprogram also affects the "subprogram"
   -- that handles partition-wide stuff so this function returns true for either

   function IsMainProgram (Subprogram : Symbol) return Boolean
   is
   begin
      return Subprogram /= NullSymbol and then
         (Subprogram = GetMainProgram or else Subprogram = GetThePartition);
   end IsMainProgram;

   --------------------------------------------------------------------------------

   -- this can be used to distinguish the main program symbol from the partition sym.
   function IsThePartition (Subprogram : Symbol) return Boolean
   is
   begin
      return Subprogram = GetThePartition;
   end IsThePartition;

   --------------------------------------------------------------------------------

   function FirstInheritsClause (Sym : Symbol) return Iterator
   is
      InheritClause  : Symbol;
      InheritClauses : Iterator;
   begin
      if IsMainProgram (Sym) then
         InheritClause := Dict.Main.InheritClauses;
      else
         InheritClause := RawDict.GetPackageInheritClauses (Sym);
      end if;

      if InheritClause = NullSymbol then
         InheritClauses := NullIterator;
      else
         InheritClauses := Iterator'(InheritedPackageIterator,
                                     IsAbstract,
                                     RawDict.GetContextClausePackage (InheritClause),
                                     InheritClause);
      end if;
      return InheritClauses;
   end FirstInheritsClause;

   --------------------------------------------------------------------------------

   function OwnVariableIsConcrete (Variable : Symbol) return Boolean
   is
   begin
      return IsDeclared (Variable);
   end OwnVariableIsConcrete;

   --------------------------------------------------------------------------------

   function FirstLocalVariable (Subprogram  : Symbol) return Iterator
   is
      Declaration : Symbol;
      Item        : Symbol;
      Variables   : Iterator;
   begin

      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when SubprogramSymbol =>
            Declaration := RawDict.GetSubprogramFirstDeclaration (Subprogram);
            loop
               if Declaration = NullSymbol then
                  Variables := NullIterator;
                  exit;
               end if;
               Item := RawDict.GetDeclarationItem (Declaration);
               if IsVariable (Item) then
                  Variables := Iterator'(LocalVariableIterator,
                                         IsAbstract,
                                         Item,
                                         Declaration);
                  exit;
               end if;
               Declaration := RawDict.GetNextDeclaration (Declaration);
            end loop;
         when TypeSymbol =>
            if RawDict.GetTypeDiscriminant (Subprogram) = TaskType then
               Declaration := RawDict.GetTaskTypeFirstLocalDeclaration (Subprogram);
               loop
                  if Declaration = NullSymbol then
                     Variables := NullIterator;
                     exit;
                  end if;
                  Item := RawDict.GetDeclarationItem (Declaration);
                  if IsVariable (Item) then
                     Variables := Iterator'(LocalVariableIterator,
                                            IsAbstract,
                                            Item,
                                            Declaration);
                     exit;
                  end if;
                  Declaration := RawDict.GetNextDeclaration (Declaration);
               end loop;
            else
               Variables := NullIterator;
            end if;
         when others =>
            Variables := NullIterator;
      end case;

      return Variables;

   end FirstLocalVariable;

   --------------------------------------------------------------------------------

   function FirstInitializedVariable (Subprogram  : Symbol) return Iterator
   is
      Declaration : Symbol;
      Item        : Symbol;
      Variables   : Iterator;
   begin

      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when SubprogramSymbol =>
            Declaration := RawDict.GetSubprogramFirstDeclaration (Subprogram);
            loop
               if Declaration = NullSymbol then
                  Variables := NullIterator;
                  exit;
               end if;
               Item := RawDict.GetDeclarationItem (Declaration);
               if IsVariable (Item) and then VariableIsInitialized (Item) then
                  Variables := Iterator'(InitializedVariableIterator,
                                         IsAbstract,
                                         Item,
                                         Declaration);
                  exit;
               end if;
               Declaration := RawDict.GetNextDeclaration (Declaration);
            end loop;
         when TypeSymbol =>
            if RawDict.GetTypeDiscriminant (Subprogram) = TaskType then
               Declaration := RawDict.GetTaskTypeFirstLocalDeclaration (Subprogram);
               loop
                  if Declaration = NullSymbol then
                     Variables := NullIterator;
                     exit;
                  end if;
                  Item := RawDict.GetDeclarationItem (Declaration);
                  if IsVariable (Item) and then VariableIsInitialized (Item) then
                     Variables := Iterator'(InitializedVariableIterator,
                                            IsAbstract,
                                            Item,
                                            Declaration);
                     exit;
                  end if;
                  Declaration := RawDict.GetNextDeclaration (Declaration);
               end loop;
            else
               Variables := NullIterator;
            end if;
         when others =>
            Variables := NullIterator;
      end case;

      return Variables;

   end FirstInitializedVariable;

   --------------------------------------------------------------------------------

   function IsNullIterator (Current : Iterator) return Boolean
   is
   begin
      return Current.Discriminant = NullSymIterator;
   end IsNullIterator;

   --------------------------------------------------------------------------------

   function GetNumberOfGlobalVariables (Abstraction : Abstractions;
                                        Subprogram  : Symbol) return Natural
   is
      GlobalVariable : Iterator;
      Count          : Natural;
   begin
      GlobalVariable := FirstGlobalVariable (Abstraction, Subprogram);
      Count := 0;
      loop
         exit when IsNullIterator (GlobalVariable);
         GlobalVariable := NextSymbol (GlobalVariable);
         Count := Count + 1;
      end loop;
      return Count;
   end GetNumberOfGlobalVariables;

   --------------------------------------------------------------------------------

   function GetArrayIndex (TypeMark : Symbol; Dimension : Positive) return Symbol
   is
      ArrayIndex : Iterator;
   begin
      ArrayIndex := FirstArrayIndex (TypeMark);
      for Dim in Positive range 1 .. Dimension - 1 loop
         ArrayIndex := NextSymbol (ArrayIndex);
      end loop;
      return CurrentSymbol (ArrayIndex);
   end GetArrayIndex;

   --------------------------------------------------------------------------------

   function GetSimpleName (Item : Symbol) return LexTokenManager.Lex_String
   is
      Name      : LexTokenManager.Lex_String;
      ItemLocal : Symbol;

      -- hidden procedure to call SystemErrors.  Side effect in function acceptable because
      -- execution terminates immediately
      procedure InvalidGetSimpleName
      --# derives ;
      is
         --# hide InvalidGetSimpleName;
      begin
         SystemErrors.FatalError (SystemErrors.InvalidSymbolTable, "in InvalidGetSimpleName");
      end InvalidGetSimpleName;

   begin
      ItemLocal := Item;
      -- A special symbol kind is used to store the value of variables on entry to a for
      -- loop.  We "dereference" these back to the original variable before attempting
      -- to retrieve the name
      if RawDict.GetSymbolDiscriminant (ItemLocal) = LoopEntryVariableSymbol then
         ItemLocal := RawDict.GetLoopEntryVariableOriginalVar (Item);
      end if;

      case RawDict.GetSymbolDiscriminant (ItemLocal) is
         when NullSym =>
            Name := LexTokenManager.Null_String;
         when TypeSymbol =>
            Name := RawDict.GetTypeName (ItemLocal);
         when EnumerationLiteralSymbol =>
            Name := RawDict.GetEnumerationLiteralName (ItemLocal);
         when RecordComponentSymbol =>
            Name := RawDict.GetRecordComponentName (ItemLocal);
         when SubcomponentSymbol =>
            Name := RawDict.GetRecordComponentName (RawDict.GetSubcomponentComponent (ItemLocal));
         when VariableSymbol =>
            Name := RawDict.GetVariableName (ItemLocal);
         when QuantifiedVariableSymbol =>
            Name := RawDict.GetQuantifiedVariableName (ItemLocal);
         when ConstantSymbol =>
            Name := RawDict.GetConstantName (ItemLocal);
         when PackageSymbol =>
            Name := RawDict.GetPackageName (ItemLocal);
         when SubprogramSymbol =>
            Name := RawDict.GetSubprogramName (ItemLocal);
         when ImplicitProofFunctionSymbol =>
            Name := RawDict.GetSubprogramName (GetAdaFunction (ItemLocal));
         when ImplicitReturnVariableSymbol =>
            Name := RawDict.GetImplicitReturnVariableName (ItemLocal);
         when SubprogramParameterSymbol =>
            Name := RawDict.GetSubprogramParameterName (ItemLocal);
         when ParameterConstraintSymbol =>
            -- Parameter constraint symbols don't have name of their own.  We return the name of the parameter
            -- with which they are associated.  Dictionary subunit GenerateSimpleName creates a synthetic name
            -- for teh constraint using the parameter name as a prefix
            Name := RawDict.GetSubprogramParameterName (RawDict.GetParameterConstraintParameter (ItemLocal));
         when LoopSymbol =>
            Name := RawDict.GetLoopName (ItemLocal);
         when KnownDiscriminantSymbol =>
            Name := RawDict.GetDiscriminantName (ItemLocal);
         when LoopParameterSymbol =>
            Name := RawDict.GetLoopParameterName (ItemLocal);
         when GenericParameterSymbol =>
            Name := RawDict.GetGenericParameterName (ItemLocal);
         --1606 Item below added by JEB as part of generic package work
         when ConstituentSymbol =>
            Name := RawDict.GetVariableName (RawDict.GetConstituentVariable (ItemLocal));
         when others =>
            InvalidGetSimpleName;               -- calls SystemErrors
            Name := LexTokenManager.Null_String; -- Unreachable, there for DFA completeness
      end case;

      return Name;

   end GetSimpleName;

   --------------------------------------------------------------------------------

   function GetScalarAttributeType (Name     : LexTokenManager.Lex_String;
                                    TypeMark : Symbol) return Symbol
      is separate;

   --------------------------------------------------------------------------------

   function GetScalarAttributeValue (Base     : Boolean;
                                     Name     : LexTokenManager.Lex_String;
                                     TypeMark : Symbol) return LexTokenManager.Lex_String
      is separate;

   --------------------------------------------------------------------------------

   function ContainsPrivate (TypeMark : Symbol;
                             Scope    : Scopes) return Boolean
   is
      Result : Boolean;
   begin

      case RawDict.GetTypePrivate (TypeMark) is
         when Never =>
            Result := False;
         when Sometimes =>
            Result := not IsLocal (Scope, GetScope (TypeMark));
         when Always =>
            Result := True;
      end case;

      return Result;

   end ContainsPrivate;

   --------------------------------------------------------------------------------
   function ContainsFloat (TypeMark : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeContainsFloat (TypeMark);
   end ContainsFloat;

   --------------------------------------------------------------------------------

   function TypeIsLimited (TypeMark : Symbol;
                           Scope    : Scopes) return Boolean
   is
      Result : Boolean;
   begin

      case RawDict.GetTypeLimited (TypeMark) is
         when Never =>
            Result := False;
         when Sometimes =>
            Result := not IsLocal (Scope, GetScope (TypeMark));
         when Always =>
            Result := True;
      end case;

      return Result;

   end TypeIsLimited;

   --------------------------------------------------------------------------------

   function EqualityDefined (TypeMark : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeEqualityDefined (TypeMark);
   end EqualityDefined;

   --------------------------------------------------------------------------------

   procedure AddIntegerTypeMark (TypeMark : in Symbol;
                                 Static   : in Boolean;
                                 Lower    : in LexTokenManager.Lex_String;
                                 Upper    : in LexTokenManager.Lex_String)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Lower,
      --#                   Static,
      --#                   TypeMark,
      --#                   Upper;
   is
   begin
      RawDict.SetTypeDiscriminant (TypeMark     => TypeMark,
                                   Discriminant => IntegerType);
      RawDict.SetTypeStatic (TypeMark => TypeMark,
                             Static   => Static);
      RawDict.SetTypeLower (TypeMark => TypeMark,
                            Lower    => Lower);
      RawDict.SetTypeUpper (TypeMark => TypeMark,
                            Upper    => Upper);
   end AddIntegerTypeMark;

   --------------------------------------------------------------------------------

   procedure AddModularTypeMark (TypeMark : in Symbol;
                                 Static   : in Boolean;
                                 Lower    : in LexTokenManager.Lex_String;
                                 Upper    : in LexTokenManager.Lex_String;
                                 Modulus  : in LexTokenManager.Lex_String)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Lower,
   --#                   Modulus,
   --#                   Static,
   --#                   TypeMark,
   --#                   Upper;
   is
   begin
      RawDict.SetTypeDiscriminant (TypeMark     => TypeMark,
                                   Discriminant => ModularType);
      RawDict.SetTypeStatic (TypeMark => TypeMark,
                             Static   => Static);
      RawDict.SetTypeLower (TypeMark => TypeMark,
                            Lower    => Lower);
      RawDict.SetTypeUpper (TypeMark => TypeMark,
                            Upper    => Upper);
      RawDict.SetTypeModulus (TypeMark, Modulus);
   end AddModularTypeMark;

   --------------------------------------------------------------------------------

   procedure AddFloatingPointTypeMark (TypeMark : in Symbol;
                                       Static   : in Boolean)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Static,
      --#                   TypeMark;
   is
   begin
      RawDict.SetTypeDiscriminant (TypeMark     => TypeMark,
                                   Discriminant => FloatingPointType);
      RawDict.SetTypeStatic (TypeMark => TypeMark,
                             Static   => Static);
      RawDict.SetTypeContainsFloat (TypeMark, True);
   end AddFloatingPointTypeMark;

   --------------------------------------------------------------------------------

   procedure AddFixedPointTypeMark (TypeMark : in Symbol;
                                    Static   : in Boolean)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Static,
      --#                   TypeMark;
   is
   begin
      RawDict.SetTypeDiscriminant (TypeMark     => TypeMark,
                                   Discriminant => FixedPointType);
      RawDict.SetTypeStatic (TypeMark => TypeMark,
                             Static   => Static);
   end AddFixedPointTypeMark;

   --------------------------------------------------------------------------------

   procedure AddArrayTypeMark (TypeMark      : in Symbol;
                               ComponentType : in Symbol;
                               Static        : in Boolean)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   ComponentType,
      --#                   Static,
      --#                   TypeMark;
   is
      IsPrivate, IsLimited : TriState;
   begin

      case RawDict.GetTypePrivate (ComponentType) is
         when Never =>
            IsPrivate := Never;
         when Sometimes =>
            if IsLocal (GetScope (TypeMark), GetScope (ComponentType)) then
               IsPrivate := Sometimes;
            else
               IsPrivate := Always;
            end if;
         when Always =>
            IsPrivate := Always;
      end case;

      case RawDict.GetTypeLimited (ComponentType) is
         when Never =>
            IsLimited := Never;
         when Sometimes =>
            if IsLocal (GetScope (TypeMark), GetScope (ComponentType)) then
               IsLimited := Sometimes;
            else
               IsLimited := Always;
            end if;
         when Always =>
            IsLimited := Always;
      end case;

      RawDict.SetTypeDiscriminant (TypeMark     => TypeMark,
                                   Discriminant => ArrayType);
      RawDict.SetTypeStatic (TypeMark => TypeMark,
                             Static   => Static);
      RawDict.SetTypePrivate (TypeMark, IsPrivate);
      RawDict.SetTypeLimited (TypeMark  => TypeMark,
                              IsLimited => IsLimited);
      RawDict.SetTypeEqualityDefined (TypeMark        => TypeMark,
                                      EqualityDefined => EqualityDefined (ComponentType));
      RawDict.SetTypeContainsFloat (TypeMark, ContainsFloat (ComponentType));
      RawDict.SetTypeArrayComponent (TypeMark, ComponentType);

   end AddArrayTypeMark;

   --------------------------------------------------------------------------------

   procedure AddSubtype (TheSubtype : in Symbol;
                         Parent     : in Symbol)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Parent,
      --#                   TheSubtype;
   is
   begin
      RawDict.SetTypeParent (TheSubtype => TheSubtype,
                             Parent     => Parent);
   end AddSubtype;

   --------------------------------------------------------------------------------

   procedure AddNewArrayIndex (TheArrayType, IndexType : in Symbol;
                               Comp_Unit               : in ContextManager.UnitDescriptors;
                               Declaration             : in Location)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Comp_Unit,
   --#                   Declaration,
   --#                   IndexType,
   --#                   TheArrayType;
   is
      Previous, Current : Symbol;
   begin

      RawDict.CreateArrayIndex (IndexType  => IndexType,
                                Comp_Unit  => Comp_Unit,
                                Loc        => Declaration.StartPosition,
                                ArrayIndex => Current);

      Previous := RawDict.GetTypeLastArrayIndex (TheArrayType);

      if Previous = NullSymbol then
         RawDict.SetTypeFirstArrayIndex (TheArrayType, Current);
      else
         RawDict.SetNextArrayIndex (Previous, Current);
      end if;

      RawDict.SetTypeLastArrayIndex (TheArrayType, Current);

   end AddNewArrayIndex;

   --------------------------------------------------------------------------------

   procedure AddEnumerationTypeMark (TypeMark : in Symbol;
                                     Static   : in Boolean)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Static,
      --#                   TypeMark;
   is
   begin
      RawDict.SetTypeDiscriminant (TypeMark     => TypeMark,
                                   Discriminant => EnumerationType);
      RawDict.SetTypeStatic (TypeMark => TypeMark,
                             Static   => Static);
   end AddEnumerationTypeMark;

   --------------------------------------------------------------------------------

   procedure AddProtectedTypeMark (TypeMark    : in Symbol;
                                   Comp_Unit   : in ContextManager.UnitDescriptors;
                                   Declaration : in Location)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Comp_Unit,
   --#                   Declaration,
   --#                   TypeMark;
   is
      ProtectedInfoSym : Symbol;
   begin
      RawDict.CreateProtectedInfo (Comp_Unit => Comp_Unit,
                                   Loc       => Declaration.StartPosition,
                                   InfoSym   => ProtectedInfoSym);
      RawDict.SetTypeAncilliaryFields (TheType     => TypeMark,
                                       Declaration => ProtectedInfoSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeMark,
                                   Discriminant => ProtectedType);
      RawDict.SetTypeStatic (TypeMark => TypeMark,
                             Static   => False);
      RawDict.SetTypeEqualityDefined (TypeMark        => TypeMark,
                                      EqualityDefined => False);
      RawDict.SetTypeLimited (TypeMark  => TypeMark,
                              IsLimited => Always);
      RawDict.SetTypeAtomic (TypeMark => TypeMark);
   end AddProtectedTypeMark;

   --------------------------------------------------------------------------------

   procedure AddTaskTypeMark (TypeMark    : in Symbol;
                              Comp_Unit   : in ContextManager.UnitDescriptors;
                              Declaration : in Location)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Comp_Unit,
   --#                   Declaration,
   --#                   TypeMark;
   is
      TaskInfoSym : Symbol;
   begin
      RawDict.CreateTaskInfo (Comp_Unit => Comp_Unit,
                              Loc       => Declaration.StartPosition,
                              InfoSym   => TaskInfoSym);
      RawDict.SetTypeAncilliaryFields (TheType     => TypeMark,
                                       Declaration => TaskInfoSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeMark,
                                   Discriminant => TaskType);
      RawDict.SetTypeStatic (TypeMark => TypeMark,
                             Static   => False);
      RawDict.SetTypeEqualityDefined (TypeMark        => TypeMark,
                                      EqualityDefined => False);
      RawDict.SetTypeLimited (TypeMark  => TypeMark,
                              IsLimited => Always);
   end AddTaskTypeMark;

   --------------------------------------------------------------------------------

   procedure AddEnumerationLiteral (Name        : in LexTokenManager.Lex_String;
                                    Comp_Unit   : in ContextManager.UnitDescriptors;
                                    Declaration : in Location;
                                    Position    : in LexTokenManager.Lex_String;
                                    TheType     : in Symbol)
   is
      Literal  : Symbol;
      Previous : Symbol;
   begin

      RawDict.CreateEnumerationLiteral (Name               => Name,
                                        Position           => Position,
                                        EnumerationType    => TheType,
                                        Comp_Unit          => Comp_Unit,
                                        Loc                => Declaration.StartPosition,
                                        EnumerationLiteral => Literal);

      Previous := RawDict.GetTypeLastEnumerationLiteral (TheType);

      if Previous = NullSymbol then
         RawDict.SetTypeFirstEnumerationLiteral (TheType, Literal);
         RawDict.SetTypeLower (TypeMark => TheType,
                               Lower    => Position);
      else
         RawDict.SetNextEnumerationLiteral (Previous, Literal);
      end if;

      RawDict.SetTypeLastEnumerationLiteral (TheType, Literal);
      RawDict.SetTypeUpper (TypeMark => TheType,
                            Upper    => Position);
   end AddEnumerationLiteral;

   --------------------------------------------------------------------------------

   function GetPredefinedPositiveSubtype return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedPositiveSubtype);
   end GetPredefinedPositiveSubtype;

   --------------------------------------------------------------------------------

   function GetPredefinedPackageASCII return Symbol
   is
   begin
      return Dict.Packages.PackageASCII;
   end GetPredefinedPackageASCII;

   --------------------------------------------------------------------------------

   function GetPredefinedPackageAda return Symbol
      --# global in Dict;
   is
   begin
      return Dict.Packages.PackageAda;
   end GetPredefinedPackageAda;

   --------------------------------------------------------------------------------

   function GetPredefinedPackageAdaCharacters return Symbol
      --# global in Dict;
   is
   begin
      return Dict.Packages.PackageAdaCharacters;
   end GetPredefinedPackageAdaCharacters;

   --------------------------------------------------------------------------------

   function GetPredefinedPackageAdaCharactersLatin1 return Symbol
      --# global in Dict;
   is
   begin
      return Dict.Packages.PackageAdaCharactersLatin1;
   end GetPredefinedPackageAdaCharactersLatin1;

   --------------------------------------------------------------------------------

   function GetPredefinedPackageAdaRealTime return Symbol
      --# global in Dict;
   is
   begin
      return Dict.Packages.PackageRealTime;
   end GetPredefinedPackageAdaRealTime;

   --------------------------------------------------------------------------------

   function GetPredefinedPackageAdaSynchronousTaskControl return Symbol
      --# global in Dict;
   is
   begin
      return Dict.Packages.PackageSynchronousTaskControl;
   end GetPredefinedPackageAdaSynchronousTaskControl;

   --------------------------------------------------------------------------------

   function GetPredefinedPackageAdaInterrupts return Symbol
      --# global in Dict;
   is
   begin
      return Dict.Packages.PackageInterrupts;
   end GetPredefinedPackageAdaInterrupts;

   --------------------------------------------------------------------------------

   function TypeIsRecord (TypeMark : Symbol) return Boolean
   is
   begin
      return RawDict.GetTypeDiscriminant (TypeMark) = RecordType;
   end TypeIsRecord;

   --------------------------------------------------------------------------------

   function GetArrayComponent (TypeMark : Symbol) return Symbol
   is
      Result : Symbol;
   begin
      if IsUnknownTypeMark (TypeMark) then
         Result := TypeMark;
      else
         Result := RawDict.GetTypeArrayComponent (GetRootType (TypeMark));
      end if;
      return Result;
   end GetArrayComponent;

   --------------------------------------------------------------------------------

   function GetNumberOfDimensions (TypeMark : Symbol) return Positive
   is
      ArrayIndex : Iterator;
      Count      : Positive;
   begin
      ArrayIndex := FirstArrayIndex (TypeMark);
      Count := 1;
      loop
         ArrayIndex := NextSymbol (ArrayIndex);
         exit when IsNullIterator (ArrayIndex);
         Count := Count + 1;
      end loop;
      return Count;
   end GetNumberOfDimensions;

   --------------------------------------------------------------------------------

   function IsPredefinedBooleanType (TheSymbol : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TheSymbol = GetPredefinedBooleanType;
   end IsPredefinedBooleanType;

   --------------------------------------------------------------------------------

   function TypeIsBoolean (TypeMark : Symbol) return Boolean
   is
   begin
      return IsPredefinedBooleanType (GetRootType (TypeMark));
   end TypeIsBoolean;

   --------------------------------------------------------------------------------

   function TypeIsBooleanArray (Operand : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsArray (Operand) and then
         TypeIsBoolean (GetArrayComponent (Operand)) and then
         GetNumberOfDimensions (Operand) = 1;
   end TypeIsBooleanArray;

   --------------------------------------------------------------------------------

   procedure WriteSpace (File : in SPARK_IO.File_Type)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                File;
   is
   begin
      SPARK_IO.Put_Char (File, ' ');
   end WriteSpace;

   --------------------------------------------------------------------------------

   procedure WriteString (File : in SPARK_IO.File_Type; Str : in String)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                File,
      --#                                Str;
   is
   begin
      SPARK_IO.Put_String (File, Str, 0);
   end WriteString;

   --------------------------------------------------------------------------------

   procedure WriteLine (File : in SPARK_IO.File_Type; Str : in String)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                File,
      --#                                Str;
   is
   begin
      SPARK_IO.Put_Line (File, Str, 0);
   end WriteLine;

   --------------------------------------------------------------------------------

   function IsLoop (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = LoopSymbol;
   end IsLoop;

   --------------------------------------------------------------------------------

   function LoopHasName (TheLoop : Symbol) return Boolean
   is
   begin
      return LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => GetSimpleName (TheLoop),
                                                                  Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq;
   end LoopHasName;

   --------------------------------------------------------------------------------

   function FirstLoop (CompilationUnit : Symbol) return Iterator
   --# global in Dict;
   is
      TheLoop : Symbol;
      Loops   : Iterator;
   begin

      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when SubprogramSymbol =>
            TheLoop := RawDict.GetSubprogramFirstLoop (CompilationUnit);
         when TypeSymbol =>
            TheLoop := RawDict.GetTaskTypeFirstLoop (CompilationUnit);
         when others =>
            TheLoop := RawDict.GetPackageFirstLoop (CompilationUnit);
      end case;

      if TheLoop = NullSymbol then
         Loops := NullIterator;
      else
         Loops := Iterator'(LoopIterator, IsAbstract, TheLoop, NullSymbol);
      end if;

      return Loops;

   end FirstLoop;

   --------------------------------------------------------------------------------

   function FirstLoopOnEntryVar (TheLoop : Symbol) return Iterator
   is
      OnEntryVars : Iterator;
      TheVar      : Symbol;
   begin
      TheVar := RawDict.GetLoopOnEntryVars (TheLoop);
      if TheVar = NullSymbol then
         OnEntryVars := NullIterator;
      else
         OnEntryVars := Iterator'(LoopOnEntryVarIterator, IsAbstract, TheVar, NullSymbol);
      end if;

      return OnEntryVars;

   end FirstLoopOnEntryVar;

   --------------------------------------------------------------------------------

   function IsCompilationUnit (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsSubprogram (TheSymbol) or else
        IsPackage (TheSymbol);
   end IsCompilationUnit;

   --------------------------------------------------------------------------------

   procedure AddUsesUncheckedConversion (TheUnit : in Symbol)
   is
   begin
      SystemErrors.RTAssert (IsCompilationUnit (TheUnit) or else IsTaskType (TheUnit),
                             SystemErrors.PreconditionFailure,
                             "in AddUsesUncheckedConversion");

      case RawDict.GetSymbolDiscriminant (TheUnit) is
         when SubprogramSymbol =>
            -- covers procedures, functions and entries
            RawDict.SetSubprogramUsesUncheckedConversion (TheUnit);

         when TypeSymbol =>
            -- covers task bodies
            RawDict.SetTaskUsesUncheckedConversion (TheUnit);

         when PackageSymbol =>
            -- a package could only use U_C in elabotration code where it would be illegal
            null;

         when others =>
            -- unreachable if precondition observed
            null;
      end case;
   end AddUsesUncheckedConversion;

   procedure AddAssignsFromExternal (TheUnit : in Symbol)
   is
   begin
      SystemErrors.RTAssert (IsCompilationUnit (TheUnit) or else IsTaskType (TheUnit),
                             SystemErrors.PreconditionFailure,
                             "in AddAssignsFromExternal");

      case RawDict.GetSymbolDiscriminant (TheUnit) is
         when SubprogramSymbol =>
            -- covers procedures, functions and entries
            RawDict.SetSubprogramAssignsFromExternal (TheUnit);

         when TypeSymbol =>
            -- covers task bodies
            RawDict.SetTaskAssignsFromExternal (TheUnit);

         when PackageSymbol =>
            -- a package could only read external variable in elabotration code
            -- where it would be illegal
            null;

         when others =>
            -- unreachable if precondition observed
            null;
      end case;
   end AddAssignsFromExternal;

   --------------------------------------------------------------------------------

   function LastMostEnclosingLoop (CompilationUnit : Symbol) return Symbol
   is
      Sym       : Symbol;
      TheResult : Symbol;
   begin
      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when SubprogramSymbol =>
            Sym := RawDict.GetSubprogramLastLoop (CompilationUnit);
         when TypeSymbol =>
            Sym := RawDict.GetTaskTypeLastLoop (CompilationUnit);
         when others =>
            Sym := RawDict.GetPackageLastLoop (CompilationUnit);
      end case;
      TheResult := Sym;
      while not IsCompilationUnit (Sym) loop
         Sym := GetRegion (GetScope (Sym));
         if IsLoop (Sym) then
            TheResult := Sym;
         end if;
      end loop;
      return TheResult;
   end LastMostEnclosingLoop;

   --------------------------------------------------------------------------------

   function GetEnclosingCompilationUnit (Scope : Scopes) return Symbol
   is
      Current : Scopes;
      Region  : Symbol;
   begin

      Current := Scope;

      loop
         Region := GetRegion (Current);
         exit when IsCompilationUnit (Region);
         exit when IsTaskType (Region); -- to deal with task bodies
         Current := GetEnclosingScope (Current);
      end loop;

      return Region;

   end GetEnclosingCompilationUnit;

   --------------------------------------------------------------------------------

   function GetLoopNumber (TheLoop : Symbol) return Positive
   is
      Loops  : Iterator;
      Number : Positive;
   begin

      Loops := FirstLoop (GetEnclosingCompilationUnit (LocalScope (TheLoop)));
      Number := 1;

      loop
         exit when CurrentSymbol (Loops) = TheLoop;
         Loops := NextSymbol (Loops);
         Number := Number + 1;
      end loop;

      return Number;
   end GetLoopNumber;

   --------------------------------------------------------------------------------

   function IsRecordSubcomponent (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = SubcomponentSymbol;
   end IsRecordSubcomponent;

   --------------------------------------------------------------------------------

   function RecordComponentIsInherited (TheComponent : Symbol) return Boolean
   is
   begin
      return RawDict.GetRecordComponentInheritedField (TheComponent);
   end RecordComponentIsInherited;

   --------------------------------------------------------------------------------

   function GetOwnVariableOfProtectedImplicitInStream (TheImplicitInStream : Symbol) return Symbol
   --# global in Dict;
   is
   begin
      return RawDict.GetOwnVariableVariable (RawDict.GetImplicitInStreamOwnVariable (TheImplicitInStream));
   end GetOwnVariableOfProtectedImplicitInStream;

   --------------------------------------------------------------------------------
   -- moved here for declaration/calling order reasons
   function GetSubprogramParameterConstraintDimension (TheConstraint : Symbol) return Positive
   is
   begin
      return RawDict.GetParameterConstraintDimension (TheConstraint);
   end GetSubprogramParameterConstraintDimension;

   --------------------------------------------------------------------------------

   procedure GenerateSimpleName (Item      : in     Symbol;
                                 Separator : in     String;
                                 Name      :    out EStrings.T)
      is separate;

   --------------------------------------------------------------------------------

   function IsImplicitProofFunction (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = ImplicitProofFunctionSymbol;
   end IsImplicitProofFunction;

   --------------------------------------------------------------------------------

   procedure GetAnyPrefixNeeded (Sym       : in     Symbol;
                                 Scope     : in     Scopes;
                                 Separator : in     String;
                                 Prefix    :    out EStrings.T)
      is separate;

   --------------------------------------------------------------------------------

   function IsValidGenericTypeAssociation (Formal, Actual : Symbol;
                                           Scope          : Scopes) return Boolean
      is separate;

   --------------------------------------------------------------------------------

   function Get_Symbol_Compilation_Unit (Item : Symbol) return ContextManager.UnitDescriptors is
   begin
      return RawDict.Get_Symbol_Compilation_Unit (Item);
   end Get_Symbol_Compilation_Unit;

   --------------------------------------------------------------------------------

   function Get_Symbol_Location (Item : Symbol) return LexTokenManager.Token_Position is
   begin
      return RawDict.Get_Symbol_Location (Item);
   end Get_Symbol_Location;

   --------------------------------------------------------------------------------

   procedure Get_SLI_Type (Item   : in     Symbol;
                           Result :    out SLI_Type) is

      Type_Item : Symbol;
      Str       : EStrings.T;

      procedure Get_Type_Discriminant (Type_Item : in     Symbol;
                                       Is_Type   : in     Boolean;
                                       Result    :    out SLI_Type)
      --# global in     CommandLineData.Content;
      --#        in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives Result            from Dict,
      --#                                Is_Type,
      --#                                Type_Item &
      --#         SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                Dict,
      --#                                Is_Type,
      --#                                LexTokenManager.State,
      --#                                Type_Item;
      is
         Str : EStrings.T;
      begin
         if CommandLineData.Content.Debug.SLI then
            --  Debug
            GenerateSimpleName (Item      => Type_Item,
                                Separator => ".",
                                Name      => Str);
            SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                 Item => " Type symbol = ",
                                 Stop => 0);
            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => Str);
            SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                 Item => " : ",
                                 Stop => 0);
         end if;
         if RawDict.GetSymbolDiscriminant (Item => Type_Item) = TypeSymbol then
            case RawDict.GetTypeDiscriminant (TypeMark => Type_Item) is
               when UnknownType =>
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Unknown_Type",
                                          Stop => 0);
                  end if;
                  Result := Unknown_Type;
               when EnumerationType =>
                  if Is_Type then
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Enumeration_Type",
                                             Stop => 0);
                     end if;
                     Result := Enumeration_Type;
                  else
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Enumeration_Object",
                                             Stop => 0);
                     end if;
                     Result := Enumeration_Object;
                  end if;
               when IntegerType =>
                  if Is_Type then
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Signed_Integer_Type",
                                             Stop => 0);
                     end if;
                     Result := Signed_Integer_Type;
                  else
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Signed_Integer_Object",
                                             Stop => 0);
                     end if;
                     Result := Signed_Integer_Object;
                  end if;
               when ModularType =>
                  if Is_Type then
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Modular_Integer_Type",
                                             Stop => 0);
                     end if;
                     Result := Modular_Integer_Type;
                  else
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Modular_Integer_Object",
                                             Stop => 0);
                     end if;
                     Result := Modular_Integer_Object;
                  end if;
               when FloatingPointType =>
                  if Is_Type then
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Floating_Point_Type",
                                             Stop => 0);
                     end if;
                     Result := Floating_Point_Type;
                  else
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Floating_Point_Object",
                                             Stop => 0);
                     end if;
                     Result := Floating_Point_Object;
                  end if;
               when FixedPointType =>
                  if Is_Type then
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Fixed_Point_Type",
                                             Stop => 0);
                     end if;
                     Result := Fixed_Point_Type;
                  else
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Fixed_Point_Object",
                                             Stop => 0);
                     end if;
                     Result := Fixed_Point_Object;
                  end if;
               when ArrayType =>
                  if Is_Type then
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Array_Type",
                                             Stop => 0);
                     end if;
                     Result := Array_Type;
                  else
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Array_Object",
                                             Stop => 0);
                     end if;
                     Result := Array_Object;
                  end if;
               when RecordType =>
                  if Is_Type then
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Record_Type",
                                             Stop => 0);
                     end if;
                     Result := Record_Type;
                  else
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Record_Object",
                                             Stop => 0);
                     end if;
                     Result := Record_Object;
                  end if;
               when AbstractProofType =>
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Abstract_Type",
                                          Stop => 0);
                  end if;
                  Result := Abstract_Type;
               when ProtectedType =>
                  if Is_Type then
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Protected_Type",
                                             Stop => 0);
                     end if;
                     Result := Protected_Type;
                  else
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Protected_Object",
                                             Stop => 0);
                     end if;
                     Result := Protected_Object;
                  end if;
               when TaskType =>
                  if Is_Type then
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Task_Type",
                                             Stop => 0);
                     end if;
                     Result := Task_Type;
                  else
                     if CommandLineData.Content.Debug.SLI then
                        --  Debug
                        SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                             Item => "Task_Object",
                                             Stop => 0);
                     end if;
                     Result := Task_Object;
                  end if;
               when others =>
                  if CommandLineData.Content.Debug.SLI then
                     --  Debug
                     SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                          Item => "Unknown_Type",
                                          Stop => 0);
                  end if;
                  Result := Unknown_Type;
                  SystemErrors.FatalError (SysErr => SystemErrors.OtherInternalError,
                                           Msg    => "DICTIONARY.GET_TYPE_DISCRIMINANT : PROGRAM ERRROR");
            end case;
         else
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => "Unknown_Type",
                                    Stop => 0);
            end if;
            Result := Unknown_Type;
            SystemErrors.FatalError (SysErr => SystemErrors.OtherInternalError,
                                     Msg    => "DICTIONARY.GET_TYPE_DISCRIMINANT : PROGRAM ERRROR");
         end if;
      end Get_Type_Discriminant;

   begin
      if CommandLineData.Content.Debug.SLI then
         --  Debug
         GenerateSimpleName (Item      => Item,
                             Separator => ".",
                             Name      => Str);
         SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                              Item => " [DICTIONARY.GET_SLI_TYPE : Symbol = ",
                              Stop => 0);
         EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                              E_Str => Str);
      end if;
      case RawDict.GetSymbolDiscriminant (Item => Item) is
         when TypeSymbol =>
            Get_Type_Discriminant (Type_Item => Item,
                                   Is_Type   => True,
                                   Result    => Result);
         when EnumerationLiteralSymbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Enumeration_Literal",
                                    Stop => 0);
            end if;
            Result := Enumeration_Literal;
         when RecordComponentSymbol =>
            Get_Type_Discriminant (Type_Item => RawDict.GetRecordComponentType (Component => Item),
                                   Is_Type   => False,
                                   Result    => Result);
         when SubcomponentSymbol =>
            Get_Type_Discriminant (Type_Item => RawDict.GetRecordComponentType (Component => RawDict.GetSubcomponentComponent (Subcomponent => Item)),
                                   Is_Type   => False,
                                   Result    => Result);
         when VariableSymbol =>
            Get_Type_Discriminant (Type_Item => RawDict.GetVariableType (Variable => Item),
                                   Is_Type   => False,
                                   Result    => Result);
         when QuantifiedVariableSymbol =>
            Get_Type_Discriminant (Type_Item => RawDict.GetQuantifiedVariableType (Variable => Item),
                                   Is_Type   => False,
                                   Result    => Result);
         when ImplicitReturnVariableSymbol =>
            Get_Type_Discriminant (Type_Item => RawDict.GetSubprogramReturnType (Subprogram => RawDict.GetImplicitReturnVariableFunction (Variable => Item)),
                                   Is_Type   => False,
                                   Result    => Result);
         when OwnVariableSymbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Record_Object",
                                    Stop => 0);
            end if;
            Result := Record_Object;
         when OwnTaskSymbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Task_Object",
                                    Stop => 0);
            end if;
            Result := Task_Object;
         when ConstantSymbol =>
            Type_Item := RawDict.GetConstantType (TheConstant => Item);
            if Type_Item = GetUniversalIntegerType or
              Type_Item = GetUniversalRealType then
               if CommandLineData.Content.Debug.SLI then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => " : Named_Number",
                                       Stop => 0);
               end if;
               Result := Named_Number;
            elsif Type_Item = GetUnknownTypeMark then
               if CommandLineData.Content.Debug.SLI then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => " : Unknown_Type",
                                       Stop => 0);
               end if;
               Result := Unknown_Type;
            else
               Get_Type_Discriminant (Type_Item => Type_Item,
                                      Is_Type   => False,
                                      Result    => Result);
            end if;
         when PackageSymbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Package_Type",
                                    Stop => 0);
            end if;
            Result := Package_Type;
         when SubprogramSymbol =>
            Type_Item := RawDict.GetSubprogramReturnType (Subprogram => Item);
            if Type_Item = NullSymbol then
               if CommandLineData.Content.Debug.SLI then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => " : Procedure_Type",
                                       Stop => 0);
               end if;
               Result := Procedure_Type;
            else
               if CommandLineData.Content.Debug.SLI then
                  --  Debug
                  SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                       Item => " : Function_Op",
                                       Stop => 0);
               end if;
               Result := Function_Op;
            end if;
         when ImplicitProofFunctionSymbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Function_Op",
                                    Stop => 0);
            end if;
            Result := Function_Op;
         when SubprogramParameterSymbol =>
            Get_Type_Discriminant (Type_Item => RawDict.GetSubprogramParameterType (Parameter => Item),
                                   Is_Type   => False,
                                   Result    => Result);
         when OperatorSymbol =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Function_Op",
                                    Stop => 0);
            end if;
            Result := Function_Op;
         when others =>
            if CommandLineData.Content.Debug.SLI then
               --  Debug
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => " : Unknown_Type",
                                    Stop => 0);
            end if;
            Result := Unknown_Type;
      end case;
      if CommandLineData.Content.Debug.SLI then
         SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                              Item => "] ",
                              Stop => 0);
      end if;
   end Get_SLI_Type;

   --------------------------------------------------------------------------------

   procedure WriteSimpleName (File : in SPARK_IO.File_Type;
                              Item : in Symbol)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                File,
      --#                                Item,
      --#                                LexTokenManager.State;
   is
      Name : EStrings.T;
   begin
      -- These statements put out the raw symbol number before its name (useful for debugging)
      -- SPARK_IO.Put_Char (File, '(');
      -- SPARK_IO.Put_Integer (File, Integer (Dictionary.SymbolRef (Item)), 4, 10);
      -- SPARK_IO.Put_Char (File, ')');
      -- end of numeric output lines

      GenerateSimpleName (Item, ".", Name);
      EStrings.Put_String (File  => File,
                           E_Str => Name);
   end WriteSimpleName;

   --------------------------------------------------------------------------------

   procedure WriteName (File : in SPARK_IO.File_Type;
                        Item : in Symbol)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                File,
      --#                                Item,
      --#                                LexTokenManager.State;
      is separate;

   --------------------------------------------------------------------------------

   procedure WriteScope (File  : in SPARK_IO.File_Type;
                         Scope : in Scopes)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                File,
      --#                                LexTokenManager.State,
      --#                                Scope;
   is
   begin
      if IsVisibleScope (Scope) then
         WriteString (File, "specification");
      elsif IsPrivateScope (Scope) then
         WriteString (File, "private part of");
      else
         WriteString (File, "body");
      end if;
      WriteString (File, " of ");
      WriteName (File, GetRegion (Scope));
   end WriteScope;

   --------------------------------------------------------------------------------

   procedure WriteInteger (File : in SPARK_IO.File_Type; Int : in Integer)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                File,
      --#                                Int;
   is
   begin
      SPARK_IO.Put_Integer (File, Int, 0, 10);
   end WriteInteger;

   --------------------------------------------------------------------------------

   function FirstOwnVariable (ThePackage : Symbol) return Iterator
   is
      OwnVariable  : Symbol;
      OwnVariables : Iterator;
   begin

      OwnVariable := RawDict.GetPackageOwnVariables (ThePackage);

      if OwnVariable = NullSymbol then
         OwnVariables := NullIterator;
      else
         OwnVariables := Iterator'(OwnVariableIterator,
                                   IsAbstract,
                                   OwnVariable,
                                   NullSymbol);
      end if;

      return OwnVariables;

   end FirstOwnVariable;

   --------------------------------------------------------------------------------

   function FirstAbstractOwnVariable (ThePackage : Symbol) return Iterator
      --# global in Dict;
   is
      OwnVariables, AbstractOwnVariables : Iterator;
   begin

      OwnVariables := FirstOwnVariable (ThePackage);

      loop
         if IsNullIterator (OwnVariables) then
            AbstractOwnVariables := NullIterator;
            exit;
         end if;
         if OwnVariableIsAbstract (CurrentSymbol (OwnVariables)) then
            AbstractOwnVariables := Iterator'(AbstractOwnVariableIterator,
                                              IsAbstract,
                                              OwnVariables.Current,
                                              NullSymbol);
            exit;
         end if;
         OwnVariables := NextSymbol (OwnVariables);
      end loop;

      return AbstractOwnVariables;

   end FirstAbstractOwnVariable;

   --------------------------------------------------------------------------------

   function GetLoopParameter (TheLoop : Symbol) return Symbol
   is
   begin
      return RawDict.GetLoopParameter (TheLoop);
   end GetLoopParameter;

   --------------------------------------------------------------------------------

   function GetLoopExitExpn (TheLoop : Symbol) return Natural
   is
   begin
      return RawDict.GetLoopExitExpn (TheLoop);
   end GetLoopExitExpn;

   --------------------------------------------------------------------------------

   function GetLoopEntryExpn (TheLoop : Symbol) return Natural
   is
   begin
      return RawDict.GetLoopEntryExpn (TheLoop);
   end GetLoopEntryExpn;

   --------------------------------------------------------------------------------

   function GetLoopHasExits (TheLoop : Symbol) return Boolean
   is
   begin
      return RawDict.GetLoopHasExits (TheLoop);
   end GetLoopHasExits;

   --------------------------------------------------------------------------------

   function IsForLoop (TheSymbol : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return IsLoop (TheSymbol) and then
         GetLoopParameter (TheSymbol) /= NullSymbol;
   end IsForLoop;

   --------------------------------------------------------------------------------

   function LoopParameterHasStaticRange (TheLoopParameter : Symbol) return Boolean
   is
   begin
      return RawDict.GetLoopParameterHasStaticRange (TheLoopParameter);
   end LoopParameterHasStaticRange;

   --------------------------------------------------------------------------------

   function LoopParameterMovesInReverse (TheLoopParameter : Symbol) return Boolean
   is
   begin
      return RawDict.GetLoopParameterIsReverse (TheLoopParameter);
   end LoopParameterMovesInReverse;

   --------------------------------------------------------------------------------

   function FirstGenericFormalParameter (TheGeneric : Symbol) return Iterator
      is separate;

   --------------------------------------------------------------------------------

   procedure Write (FileName : in     EStrings.T;
                    Status   :    out SPARK_IO.File_Status)
      is separate;

   --------------------------------------------------------------------------------

   function IsLoopName (Name  : LexTokenManager.Lex_String;
                        Scope : Scopes) return Boolean
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      Current : Iterator;
   begin

      Current := FirstLoop (GetEnclosingCompilationUnit (Scope));

      loop
         exit when IsNullIterator (Current);
         exit when LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => GetSimpleName (CurrentSymbol (Current)),
                                                                        Lex_Str2 => Name) = LexTokenManager.Str_Eq;
         Current := NextSymbol (Current);
      end loop;

      return not IsNullIterator (Current);

   end IsLoopName;

   --------------------------------------------------------------------------------

   function IsAbstractOwnVariable (Variable : Symbol) return Boolean
   is
   begin
      return IsOwnVariable (Variable) and then OwnVariableIsAbstract (Variable);
   end IsAbstractOwnVariable;

   --------------------------------------------------------------------------------

   function IsProcedure (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsSubprogram (TheSymbol) and then GetType (TheSymbol) = NullSymbol;
   end IsProcedure;

   --------------------------------------------------------------------------------

   function GetLoop (CompilationUnit : Symbol;
                     Number          : Positive) return Symbol
   is
      TheLoop : Iterator;
   begin
      TheLoop := FirstLoop (CompilationUnit);
      for No in Positive range 1 .. Number - 1 loop
         TheLoop := NextSymbol (TheLoop);
      end loop;
      return CurrentSymbol (TheLoop);
   end GetLoop;

   --------------------------------------------------------------------------------

   function GetAbstraction (Subprogram : Symbol;
                            Scope      : Scopes) return Abstractions
   is
      DeclaredScope       : Scopes;
      Abstraction         : Abstractions;
      HasSecondAnnotation : Boolean;
   begin
      if RawDict.GetSymbolDiscriminant (Subprogram) = SubprogramSymbol then
         HasSecondAnnotation := RawDict.GetSubprogramHasSecondAnnotation (Subprogram);
      else
         -- must be task type
         HasSecondAnnotation := RawDict.GetTaskTypeHasSecondAnnotation (Subprogram);
      end if;

      if HasSecondAnnotation then
         DeclaredScope := GetScope (Subprogram);

         if (IsVisibleScope (DeclaredScope) or else IsPrivateScope (DeclaredScope)) and then
            IsLocal (Scope, GetLocalScope (DeclaredScope)) then
            Abstraction := IsRefined;
         else
            Abstraction := IsAbstract;
         end if;

      else
         Abstraction := IsAbstract;
      end if;

      return Abstraction;

   end GetAbstraction;

   --------------------------------------------------------------------------------

   -- Similar to preceding function but is concerned with whether there is a second
   -- constraint rather than a second flow annotation.  If there is a second flow
   -- anno then it is assumed there is a second constraint (even if the user has just
   -- left it as True).  If there is NO second flow anno then we still need to check
   -- whether there is a second constraint (to cover the private type refinement case)
   function GetConstraintAbstraction (Subprogram : Symbol;
                                      Scope      : Scopes) return Abstractions
   is
      DeclaredScope       : Scopes;
      Abstraction         : Abstractions;
      HasSecondAnnotation : Boolean;
   begin
      Abstraction := GetAbstraction (Subprogram, Scope);
      if Abstraction = IsAbstract and then
        RawDict.GetSymbolDiscriminant (Subprogram) = SubprogramSymbol then
         -- No second flow anno, so check for second constraint.  Second half of guard protects against
         -- looking for a second proof annotation on a task or protected type
         HasSecondAnnotation := RawDict.GetSubprogramHasSecondConstraint (Subprogram);

         if HasSecondAnnotation then
            DeclaredScope := GetScope (Subprogram);

            if (IsVisibleScope (DeclaredScope) or else IsPrivateScope (DeclaredScope)) and then
              IsLocal (Scope, GetLocalScope (DeclaredScope)) then
               Abstraction := IsRefined;
            else
               Abstraction := IsAbstract;
            end if;

         else
            Abstraction := IsAbstract;
         end if;
      end if;
      return Abstraction;
   end GetConstraintAbstraction;

   --------------------------------------------------------------------------------

   procedure AdjustTypeUpperBound (TypeMark : Symbol;
                                   NewBound : LexTokenManager.Lex_String)
   is
   begin
      RawDict.SetTypeUpper (TypeMark => TypeMark,
                            Upper    => NewBound);
   end AdjustTypeUpperBound;

   procedure AdjustTypeLowerBound (TypeMark : Symbol;
                                   NewBound : LexTokenManager.Lex_String)
   is
   begin
      RawDict.SetTypeLower (TypeMark => TypeMark,
                            Lower    => NewBound);
   end AdjustTypeLowerBound;

   procedure AdjustTypeErrorBound (TypeMark : Symbol;
                                   NewBound : LexTokenManager.Lex_String)
   is
   begin
      RawDict.SetTypeErrorBound (TypeMark   => TypeMark,
                                 ErrorBound => NewBound);
   end AdjustTypeErrorBound;

   --------------------------------------------------------------------------------

   function GetMainProgramPriority return LexTokenManager.Lex_String
   is
   begin
      return Dict.Main.ThePriority; -- NullString if not PriorityGiven or invalid value
   end GetMainProgramPriority;

   --------------------------------------------------------------------------------

   function IsFunction (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsSubprogram (TheSymbol) and then GetType (TheSymbol) /= NullSymbol;
   end IsFunction;

   --------------------------------------------------------------------------------


   function IsProtectedFunction (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsFunction (TheSymbol) and then
        IsProtectedType (GetRegion (GetScope (TheSymbol)));
   end IsProtectedFunction;

   --------------------------------------------------------------------------------

   function IsEntry (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsSubprogram (TheSymbol) and then RawDict.GetSubprogramIsEntry (TheSymbol);
   end IsEntry;

   --------------------------------------------------------------------------------

   function IsProgramUnit (Sym : Symbol) return Boolean
   is
   begin
      return (Sym /= NullSymbol) and then
        (IsPackage (Sym) or
           IsTaskType (Sym) or
           IsProtectedType (Sym) or
           IsEntry (Sym) or
           IsGenericSubprogram (Sym) or
           IsGenericPackage (Sym) or
           IsSubprogram (Sym));
   end IsProgramUnit;



   --------------------------------------------------------------------------------

   function GetSubprogramEntryBarrier (Subprogram  : Symbol) return Symbol
   is
      Result : Symbol := NullSymbol;
   begin
      if IsEntry (Subprogram) then
         Result := RawDict.GetSubprogramEntryBarrier (Subprogram);
      end if;
      return Result;
   end GetSubprogramEntryBarrier;

   --------------------------------------------------------------------------------

   function IsInterruptHandler (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsSubprogram (TheSymbol) and then RawDict.IsInterruptHandler (TheSymbol);
   end IsInterruptHandler;

   --------------------------------------------------------------------------------

   function IsAdaFunction (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsFunction (TheSymbol) and then
         GetContext (TheSymbol) = ProgramContext;
   end IsAdaFunction;

   --------------------------------------------------------------------------------

   function IsAnUncheckedConversion (TheSymbol : Symbol) return Boolean
   is
   begin
      return (IsAdaFunction (TheSymbol) and then
                RawDict.GetSubprogramInstantiationOf (TheSymbol) = Dict.Subprograms.UncheckedConversion);
   end IsAnUncheckedConversion;

   --------------------------------------------------------------------------------

   function UsesUncheckedConversion (TheUnit : Symbol) return Boolean
   is
      Result : Boolean;
   begin
      SystemErrors.RTAssert (IsCompilationUnit (TheUnit) or else IsTaskType (TheUnit),
                             SystemErrors.PreconditionFailure,
                             "in function UsesUncheckedConversion");

      case RawDict.GetSymbolDiscriminant (TheUnit) is
         when SubprogramSymbol =>
            -- covers procedures, functions and entries
            Result := RawDict.GetSubprogramUsesUncheckedConversion (TheUnit);

         when TypeSymbol =>
            -- covers task bodies
            Result := RawDict.GetTaskUsesUncheckedConversion (TheUnit);

         when PackageSymbol =>
            -- a package could only use U_C in elabotration code where it would be illegal
            Result := False;

         when others =>
            -- unreachable if precondition observed
            Result := False;
      end case;
      return Result;
   end UsesUncheckedConversion;

   --------------------------------------------------------------------------------

   function AssignsFromExternal (TheUnit : Symbol) return Boolean
   is
      Result : Boolean;
   begin
      SystemErrors.RTAssert (IsCompilationUnit (TheUnit) or else IsTaskType (TheUnit),
                             SystemErrors.PreconditionFailure,
                             "in function AssignsFromExternal");

      case RawDict.GetSymbolDiscriminant (TheUnit) is
         when SubprogramSymbol =>
            -- covers procedures, functions and entries
            Result := RawDict.GetSubprogramAssignsFromExternal (TheUnit);

         when TypeSymbol =>
            -- covers task bodies
            Result := RawDict.GetTaskAssignsFromExternal (TheUnit);

         when PackageSymbol =>
            -- a package could only read external variable in elabotration code
            -- where it would be illegal
            Result := False;

         when others =>
            -- unreachable if precondition observed
            Result := False;
      end case;
      return Result;
   end AssignsFromExternal;

   --------------------------------------------------------------------------------
   function IsObject (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsConstant (TheSymbol) or else IsVariable (TheSymbol);
   end IsObject;

   --------------------------------------------------------------------------------

   function PrefixAllowed (Prefix : Symbol; Scope : Scopes) return Boolean
   is
      Result : Boolean;
   begin

      if IsPackage (Prefix) then
         Result := Prefix /= GetEnclosingPackage (Scope);
      elsif IsTypeMark (Prefix) then
         Result := TypeIsRecord (Prefix);
      elsif IsObject (Prefix) then
         Result := TypeIsRecord (GetType (Prefix)) or
           TypeIsProtected (GetType (Prefix));
      else
         Result := False;
      end if;

      return Result;

   end PrefixAllowed;

   --------------------------------------------------------------------------------

   function PrefixRequired (Item : Symbol; Scope : Scopes) return Boolean
   is
   begin
      SystemErrors.RTAssert (not IsUnknownTypeMark (Item),
                             SystemErrors.PreconditionFailure,
                             "In call to PrefixRequired");
      return GetRegion (GetScope (Item)) /= GetEnclosingPackage (Scope);
   end PrefixRequired;

   --------------------------------------------------------------------------------

   function GetPrefix (Item : Symbol) return LexTokenManager.Lex_String
   is
   begin
      SystemErrors.RTAssert (not IsUnknownTypeMark (Item),
                             SystemErrors.PreconditionFailure,
                             "In call to GetPrefix");
      return GetSimpleName (GetRegion (GetScope (Item)));
   end GetPrefix;

   --------------------------------------------------------------------------------

   function GetEnclosingProtectedRegion (Scope : Scopes) return Symbol
   is
      CurrentRegion : Symbol;
      CurrentScope  : Scopes;
      Result        : Symbol;
   begin
      Result := NullSymbol;
      CurrentScope := Scope;
      loop
         CurrentRegion := GetRegion (CurrentScope);
         if IsTypeMark (CurrentRegion) and then TypeIsProtected (CurrentRegion) then
            Result := CurrentRegion;
            exit;
         end if;
         exit when IsLibraryLevel (CurrentScope);
         CurrentScope := GetEnclosingScope (CurrentScope);
      end loop;
      return Result;
   end GetEnclosingProtectedRegion;

   --------------------------------------------------------------------------------

   function IsOrIsInProtectedScope (Scope : Scopes) return Boolean
   is
   begin
      return GetEnclosingProtectedRegion (Scope) /= NullSymbol;
   end IsOrIsInProtectedScope;

   --------------------------------------------------------------------------------
   -- needs to be above LookUpScope
   function FirstProtectedElement (TheProtectedType : Symbol) return Iterator
   is
      Declaration        : Symbol;
      Item              : Symbol;
      ProtectedElements : Iterator;
   begin
      Declaration := RawDict.GetProtectedTypeFirstPrivateDeclaration (TheProtectedType);
      if Declaration = NullSymbol then
         ProtectedElements := NullIterator;
      else
         Item := RawDict.GetDeclarationItem (Declaration);
         ProtectedElements := Iterator'(ProtectedElementIterator,
                                        IsAbstract,
                                        Item,
                                        Declaration);

      end if;
      return ProtectedElements;
   end FirstProtectedElement;

   --------------------------------------------------------------------------------
   -- needs to be above LookUpScope
   function GetProtectedTypeOwnVariable (TheProtectedType : Symbol) return Symbol
   is
   begin
      return RawDict.GetOwnVariableVariable
        (RawDict.GetProtectedTypeOwnVariable (TheProtectedType));
   end GetProtectedTypeOwnVariable;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------
   -- Trace routines for dictionary look up operations.  These are invisible to the
   -- flow analyser. They are enabled by -debug=l (for Lookup)

   procedure TraceMsg (Msg : in String)
   --# derives null from Msg;
   is
      --# hide TraceMsg;
   begin
      if CommandLineData.Content.Debug.LookupTrace then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Msg, 0);
      end if;
   end TraceMsg;

   procedure TraceLexStr (Msg : in String;
                          L   : in LexTokenManager.Lex_String)
   --# derives null from L,
   --#                   Msg;
   is
      --# hide TraceLexStr;
   begin
      if CommandLineData.Content.Debug.LookupTrace then
         SPARK_IO.Put_String (SPARK_IO.Standard_Output, Msg, 0);
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => L,
                                                                 Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "Null String", 0);
         else
            EStrings.Put_Line
              (File  => SPARK_IO.Standard_Output,
               E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => L));
         end if;
      end if;
   end TraceLexStr;

   procedure TraceSym (Msg   : in String;
                       Sym   : in Symbol;
                       Scope : in Scopes)
   --# derives null from Msg,
   --#                   Scope,
   --#                   Sym;
   is
      --# hide TraceSym;
      Str : EStrings.T;
   begin
      if CommandLineData.Content.Debug.LookupTrace then
         SPARK_IO.Put_String (SPARK_IO.Standard_Output, Msg, 0);
         if Sym = NullSymbol then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "Null Symbol", 0);
         else
            GetAnyPrefixNeeded (Sym, Scope, ".", Str);
            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => Str);
            SPARK_IO.Put_Char (SPARK_IO.Standard_Output, '.');
            GenerateSimpleName (Sym, ".", Str);
            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => Str);
         end if;
      end if;
   end TraceSym;

   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------

   procedure LookupScope (Name         : in     LexTokenManager.Lex_String;
                          StopAt       : in     LexTokenManager.Lex_String;
                          Scope        : in     Scopes;
                          CallingScope : in     Scopes;
                          Context      : in     Contexts;
                          Item         :    out Symbol;
                          IsVisible    :    out Boolean)
   --# global in Dict;
   --#        in LexTokenManager.State;
   --# derives IsVisible,
   --#         Item      from CallingSCope,
   --#                        Context,
   --#                        Dict,
   --#                        LexTokenManager.State,
   --#                        Name,
   --#                        Scope,
   --#                        StopAt;
      is separate;

   --------------------------------------------------------------------------------

   function LookupImmediateScope (Name    : LexTokenManager.Lex_String;
                                  Scope   : Scopes;
                                  Context : Contexts) return Symbol
   is
      Item      : Symbol;
      IsVisible : Boolean;
   begin
      TraceLexStr ("In LookupImmediateScope, seeking ", Name);
      LookupScope (Name,
                   LexTokenManager.Null_String,
                   Scope,
                   Scope,
                   Context,
                   Item,
                   IsVisible);
      if not IsVisible then
         Item := NullSymbol;
      end if;
      return Item;
   end LookupImmediateScope;

   --------------------------------------------------------------------------------

   function IsDirectlyDefined (Name    : LexTokenManager.Lex_String;
                               Scope   : Scopes;
                               Context : Contexts) return Boolean
   is
   begin
      return IsLoopName (Name, Scope) or else
         LookupImmediateScope (Name, Scope, Context) /= NullSymbol;
   end IsDirectlyDefined;

   --------------------------------------------------------------------------------

   function IsWithed (ThePackage : Symbol; Scope : Scopes) return Boolean
   is
      Region     : Symbol;
      WithClause : Symbol;
   begin

      Region := GetRegion (Scope);

      case Scope.TypeOfScope is
         when Visible | Privat =>
            case RawDict.GetSymbolDiscriminant (Region) is
               when PackageSymbol =>
                  WithClause := RawDict.GetPackageVisibleWithClauses (Region);
               when TypeSymbol =>
                  -- we must be in a protected type declaration that is in the visible
                  -- part of its enclosing package
                  WithClause := RawDict.GetPackageVisibleWithClauses (GetEnclosingPackage (Scope));
               when others =>
                  WithClause := NullSymbol;
            end case;
         when Local =>
            case RawDict.GetSymbolDiscriminant (Region) is
               when PackageSymbol =>
                  WithClause := RawDict.GetPackageLocalWithClauses (Region);
               when TypeSymbol =>
                  -- we must be in a task or protected type body or subunit which must be in the body
                  -- part of its enclosing package
                  if RawDict.GetTypeDiscriminant (Region) = ProtectedType then
                     WithClause := RawDict.GetProtectedTypeWithClauses (Region);
                  elsif RawDict.GetTypeDiscriminant (Region) = TaskType then
                     WithClause := RawDict.GetTaskTypeWithClauses (Region);
                  else
                     WithClause := NullSymbol;
                  end if;
               when SubprogramSymbol =>
                  WithClause := RawDict.GetSubprogramWithClauses (Region);
               when others =>
                  WithClause := NullSymbol;
            end case;
      end case;

      loop
         exit when WithClause = NullSymbol;
         exit when RawDict.GetContextClausePackage (WithClause) = ThePackage;
         WithClause := RawDict.GetNextContextClause (WithClause);
      end loop;

      if WithClause = NullSymbol and then
         IsPackage (Region) and then
         IsLocalScope (Scope) then

         WithClause := RawDict.GetPackageVisibleWithClauses (Region);

         loop
            exit when WithClause = NullSymbol;
            exit when RawDict.GetContextClausePackage (WithClause) = ThePackage;
            WithClause := RawDict.GetNextContextClause (WithClause);
         end loop;

      end if;

      return WithClause /= NullSymbol;

   end IsWithed;

   --------------------------------------------------------------------------------
   function IsWithedLocally (ThePackage : Symbol; Scope : Scopes) return Boolean
   is
      Region     : Symbol;
      WithClause : Symbol;
   begin

      Region := GetRegion (Scope);

      case Scope.TypeOfScope is
         when Visible | Privat =>
            WithClause := RawDict.GetPackageVisibleWithClauses (Region);
         when Local =>
            case RawDict.GetSymbolDiscriminant (Region) is
               when PackageSymbol =>
                  WithClause := RawDict.GetPackageLocalWithClauses (Region);
               when SubprogramSymbol =>
                  WithClause := RawDict.GetSubprogramWithClauses (Region);
               when TypeSymbol =>
                  if RawDict.GetTypeDiscriminant (Region) = ProtectedType then
                     WithClause := RawDict.GetProtectedTypeWithClauses (Region);
                  elsif RawDict.GetTypeDiscriminant (Region) = TaskType then
                     WithClause := RawDict.GetTaskTypeWithClauses (Region);
                  else
                     WithClause := NullSymbol; -- non-exec
                  end if;
               when others =>
                  WithClause := NullSymbol;
            end case;
      end case;

      loop
         exit when WithClause = NullSymbol;
         exit when RawDict.GetContextClausePackage (WithClause) = ThePackage;
         WithClause := RawDict.GetNextContextClause (WithClause);
      end loop;

      return WithClause /= NullSymbol;

   end IsWithedLocally;

   --------------------------------------------------------------------------------
   function IsUsedLocally (TheType : Symbol; Scope : Scopes) return Boolean
   is
      Region     : Symbol;
      UseTypeClause : Symbol;
   begin

      Region := GetRegion (Scope);

      case Scope.TypeOfScope is
         when Visible | Privat =>
            case RawDict.GetSymbolDiscriminant (Region) is
               when PackageSymbol =>
                  -- check use type clause on package spec
                  UseTypeClause := RawDict.GetPackageVisibleUseTypeClauses (Region);
               when TypeSymbol =>
                  -- here we are dealing with a task or protected type declaration; these don't
                  -- have use type clauses so the one to check is the one on the spec of the enclosing package
                  if RawDict.GetTypeDiscriminant (Region) = ProtectedType then
                     UseTypeClause := RawDict.GetPackageVisibleUseTypeClauses (GetEnclosingPackage (Scope));
                  elsif RawDict.GetTypeDiscriminant (Region) = TaskType then
                     UseTypeClause := RawDict.GetPackageVisibleUseTypeClauses (GetEnclosingPackage (Scope));
                  else
                     UseTypeClause := NullSymbol; -- non-exec
                  end if;
               when others =>
                  UseTypeClause := NullSymbol;
            end case;

         when Local  =>
            case RawDict.GetSymbolDiscriminant (Region) is
               when PackageSymbol =>
                  -- check use type clause on package body
                  UseTypeClause := RawDict.GetPackageLocalUseTypeClauses (Region);
               when TypeSymbol =>
                  -- here we must be in protected or task body subunit which may have its own use type clause
                  if RawDict.GetTypeDiscriminant (Region) = ProtectedType then
                     UseTypeClause := RawDict.GetProtectedTypeUseTypeClauses (Region);
                  elsif RawDict.GetTypeDiscriminant (Region) = TaskType then
                     UseTypeClause := RawDict.GetTaskTypeUseTypeClauses (Region);
                  else
                     UseTypeClause := NullSymbol; -- non-exec
                  end if;
               when SubprogramSymbol =>
                  -- and this must be a subprgoram subunit
                  UseTypeClause := RawDict.GetSubprogramUseTypeClauses (Region);
               when others =>
                  UseTypeClause := NullSymbol;
            end case;
      end case;

      loop
         exit when UseTypeClause = NullSymbol;
         exit when RawDict.GetUseTypeClauseType (UseTypeClause) = TheType;
         UseTypeClause := RawDict.GetNextUseTypeClause (UseTypeClause);
      end loop;

      return UseTypeClause /= NullSymbol;

   end IsUsedLocally;

   --------------------------------------------------------------------------------

   function IsEmbeddedPackage (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsPackage (TheSymbol) and then
         GetRegion (GetScope (TheSymbol)) /= GetPredefinedPackageStandard;
   end IsEmbeddedPackage;

   --------------------------------------------------------------------------------

   function IsDescendentOfPrivateChild (Candidate, ThePackage : Symbol)
                                      return Boolean
      --# global in Dict;
   is
      Current, Next : Symbol;
   begin
      Current := Candidate;
      loop
         Next := RawDict.GetPackageParent (Current);
         exit when Next = ThePackage;
         exit when Next = NullSymbol;
         Current := Next;
      end loop;
      return Next = ThePackage and IsPrivatePackage (Current);
   end IsDescendentOfPrivateChild;

   --------------------------------------------------------------------------------

   function IsGlobalVariable (Abstraction : Abstractions;
                              Subprogram  : Symbol;
                              Variable    : Symbol) return Boolean
   is
      Current : Symbol;
   begin

      case RawDict.GetSymbolDiscriminant (Subprogram) is
         when SubprogramSymbol =>
            Current := RawDict.GetSubprogramFirstGlobalVariable (Abstraction,
                                                                 Subprogram);
         when TypeSymbol => -- must be a task type
            Current := RawDict.GetTaskTypeFirstGlobalVariable (Abstraction,
                                                               Subprogram);
         when others =>
            Current := NullSymbol;
      end case;

      loop
         exit when Current = NullSymbol;
         exit when RawDict.GetGlobalVariableVariable (Current) = Variable;

         Current := RawDict.GetNextGlobalVariable (Current);
      end loop;

      return Current /= NullSymbol;

   end IsGlobalVariable;

   --------------------------------------------------------------------------------

   function LookupItem (Name    : LexTokenManager.Lex_String;
                        Scope   : Scopes;
                        Context : Contexts) return Symbol is separate;

   --------------------------------------------------------------------------------

   function IsRenamed (Subprogram : Symbol; Scope : Scopes) return Boolean
      is separate;

   --------------------------------------------------------------------------------

   function IsCallable (Subprogram   : Symbol;
                        PrefixNeeded : Boolean;
                        Scope        : Scopes) return Boolean is separate;

   --------------------------------------------------------------------------------

   function IsDefined (Name    : LexTokenManager.Lex_String;
                       Scope   : Scopes;
                       Context : Contexts) return Boolean
   is
   begin
      return IsLoopName (Name, Scope) or else
         LookupItem (Name, Scope, Context) /= NullSymbol;
   end IsDefined;

   --------------------------------------------------------------------------------

   function UnaryOperatorIsDefined (Name    : SPSymbols.SPSymbol;
                                    Operand : Symbol) return Boolean
   is
      Result : Boolean;
   begin
      SystemErrors.RTAssert (IsType (Operand) or else IsUnknownTypeMark (Operand),
                             SystemErrors.PreconditionFailure,
                             "In call to UnaryOperatorIsDefined");
      case Name is
         when SPSymbols.plus | SPSymbols.minus | SPSymbols.RWabs =>
            -- Unary arithmetic operators on modular types are not
            -- allowed in SPARK at present.  "abs" and "+" are just no-op
            -- for modular type, so they're useless.  Unary "-" is a potential
            -- danger, since "-1" for a modular type is confusing and can always
            -- be written as T'Last or 16#FFFF# or similar.
            --
            -- For time type Time_Span, unary "abs" and "-" are defined but "+" is not.
            Result :=
              TypeIsInteger (Operand) or else
              TypeIsReal (Operand) or else
              (CommandLineData.RavenscarSelected and then
                 Operand = GetPredefinedTimeSpanType and then
                 Name /= SPSymbols.plus) or else
              TypeIsUnknown (Operand);

         when SPSymbols.RWnot =>
            -- Note - unary "not" is allowed for modular types.
            Result :=
               TypeIsBoolean (Operand) or else
               TypeIsModular (Operand) or else
               IsUniversalIntegerType (Operand) or else
               TypeIsBooleanArray (Operand) or else
               TypeIsUnknown (Operand);

         when others =>
            Result := False;
      end case;

      return Result;
   end UnaryOperatorIsDefined;

   --------------------------------------------------------------------------------

   function GetPredefinedStringType return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedStringType);
   end GetPredefinedStringType;

   --------------------------------------------------------------------------------

   function GetPredefinedNaturalSubtype return Symbol
   is
   begin
      return Dict.Types.ThePredefinedTypes (PredefinedNaturalSubtype);
   end GetPredefinedNaturalSubtype;

   --------------------------------------------------------------------------------

   function IsPredefinedStringType (TheSymbol : Symbol) return Boolean
   is
   begin
      return TheSymbol = GetPredefinedStringType;
   end IsPredefinedStringType;

   --------------------------------------------------------------------------------

   function BinaryOperatorIsDefined (Name  : SPSymbols.SPSymbol;
                                     Left  : Symbol;
                                     Right : Symbol) return Boolean
      is separate;

   --------------------------------------------------------------------------------

   function DefinedInPackageStandard (TheSymbol : Symbol) return Boolean
   is
      Result : Boolean;
   begin
      if IsUnknownTypeMark (TheSymbol) then
         Result := False;
      else
         Result :=
           GetScope (TheSymbol) = VisibleScope (GetPredefinedPackageStandard);
      end if;
      return Result;
   end DefinedInPackageStandard;

   --------------------------------------------------------------------------------

   function IsPredefined (TheSymbol : Symbol) return Boolean
   is
      Result : Boolean;
      Scope  : Scopes;
   begin

      if IsUnknownTypeMark (TheSymbol) then
         Result := False;
      else
         Scope := GetScope (TheSymbol);
         case CommandLineData.Content.LanguageProfile is
            when CommandLineData.SPARK83 =>
               Result :=
                 Scope = VisibleScope (GetPredefinedPackageStandard) or else
                 Scope = VisibleScope (GetPredefinedPackageASCII);

            when CommandLineData.SPARK95 |
              CommandLineData.SPARK2005 =>

               Result :=
                 Scope = VisibleScope (GetPredefinedPackageStandard) or else
                 Scope = VisibleScope (GetPredefinedPackageASCII) or else
                 Scope = VisibleScope (GetPredefinedPackageAda) or else
                 Scope = VisibleScope (GetPredefinedPackageAdaCharacters) or else
                 Scope = VisibleScope (GetPredefinedPackageAdaCharactersLatin1) or else
                 (CommandLineData.RavenscarSelected and then
                    (Scope = VisibleScope (GetPredefinedPackageAdaSynchronousTaskControl) or else
                       Scope = VisibleScope (GetPredefinedPackageAdaRealTime) or else
                       Scope = VisibleScope (GetPredefinedPackageAdaInterrupts)));
         end case;


      end if;

      return Result;

   end IsPredefined;

   --------------------------------------------------------------------------------

   function DeclarationsAreVisible (Inner, Outer : Scopes) return Boolean
      --# global in Dict;
   is
      CurrentScope  : Scopes;
      CurrentRegion : Symbol;
      OuterRegion   : Symbol;
   begin

      CurrentScope := Inner;
      OuterRegion := GetRegion (Outer);

      loop
         CurrentRegion := GetRegion (CurrentScope);
         exit when CurrentRegion = OuterRegion;
         exit when IsPackage (CurrentRegion);
         CurrentScope := GetEnclosingScope (CurrentScope);
      end loop;

      return CurrentRegion = OuterRegion;

   end DeclarationsAreVisible;

   --------------------------------------------------------------------------------

   function TypeExportsOperators (TypeMark : Symbol;
                                  Scope    : Scopes) return Boolean
      --# global in CommandLineData.Content;
      --#        in Dict;
   is
   begin
      return not TypeIsPrivateHere (TypeMark, Scope) or else
            (CommandLineData.RavenscarSelected and then IsPredefinedTimeType (TypeMark));
   end TypeExportsOperators;

   --------------------------------------------------------------------------------

   function UnaryOperatorIsVisible (Name    : SPSymbols.SPSymbol;
                                    Operand : Symbol;
                                    Scope   : Scopes) return Boolean
      is separate;

   --------------------------------------------------------------------------------

   function BinaryOperatorIsVisible (Name  : SPSymbols.SPSymbol;
                                     Left  : Symbol;
                                     Right : Symbol;
                                     Scope : Scopes) return Boolean
      is separate;

   --------------------------------------------------------------------------------

   function IsFixedPointTypeMark (TheSymbol : Symbol;
                                  Scope     : Scopes) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         TypeIsFixedPoint (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsFixedPointTypeMark;

   --------------------------------------------------------------------------------

   function IsFloatingPointTypeMark (TheSymbol : Symbol;
                                     Scope     : Scopes) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         TypeIsFloatingPoint (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsFloatingPointTypeMark;

   --------------------------------------------------------------------------------

   function IsScalarTypeMark (TheSymbol : Symbol;
                              Scope     : Scopes) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         TypeIsScalar (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsScalarTypeMark;

   --------------------------------------------------------------------------------

   function IsArrayTypeMark (TheSymbol : Symbol;
                             Scope     : Scopes) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         TypeIsArray (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsArrayTypeMark;

   --------------------------------------------------------------------------------

   function IsDiscreteTypeMark (TheSymbol : Symbol;
                                Scope     : Scopes) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         TypeIsDiscrete (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsDiscreteTypeMark;

   --------------------------------------------------------------------------------

   function IsConstrainedArrayType (TheSymbol : Symbol;
                                    Scope     : Scopes) return Boolean
      is separate;

   --------------------------------------------------------------------------------

   function AttributeIsVisible (Name     : LexTokenManager.Lex_String;
                                Prefix   : PrefixSort;
                                TypeMark : Symbol;
                                Scope    : Scopes) return Boolean
      is separate;

   --------------------------------------------------------------------------------

   function AttributeIsVisibleButObselete (Name     : LexTokenManager.Lex_String;
                                           Prefix   : PrefixSort;
                                           TypeMark : Symbol;
                                           Scope    : Scopes) return Boolean
      is separate;

   --------------------------------------------------------------------------------

   procedure WriteLocation (File : in SPARK_IO.File_Type;
                            Loc  : in Location)
      --# global in     Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                File,
      --#                                Loc;
      is separate;

   --------------------------------------------------------------------------------

   procedure WriteReference (Discriminant    : in ReferenceDiscriminant;
                             Item            : in Symbol;
                             CompilationUnit : in Symbol;
                             Reference       : in Location)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CompilationUnit,
      --#                                Dict,
      --#                                Discriminant,
      --#                                Item,
      --#                                LexTokenManager.State,
      --#                                Reference;
      is separate;

   --------------------------------------------------------------------------------

   procedure AddOtherReference (Item, CompilationUnit : in Symbol;
                                Reference             : in Location)
   is
   begin
      WriteReference (OtherRef, Item, CompilationUnit, Reference);
   end AddOtherReference;

   --------------------------------------------------------------------------------

   procedure WriteDeclaration (Discriminant : in DeclarationDiscriminant;
                               Item         : in Symbol;
                               Scope        : in Scopes;
                               Declaration  : in Location)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Declaration,
      --#                                Dict,
      --#                                Discriminant,
      --#                                Item,
      --#                                LexTokenManager.State,
      --#                                Scope;
      is separate;

   --------------------------------------------------------------------------------

   procedure AddDeferredConstant (Name          : in LexTokenManager.Lex_String;
                                  TypeMark      : in Symbol;
                                  TypeReference : in Location;
                                  Comp_Unit     : in ContextManager.UnitDescriptors;
                                  Declaration   : in Location;
                                  ThePackage    : in Symbol)
   is
      TheConstant, TheDeclaration : Symbol;
   begin

      RawDict.CreateConstant (Name        => Name,
                              TypeMark    => TypeMark,
                              Static      => False,
                              Comp_Unit   => Comp_Unit,
                              Loc         => Declaration.StartPosition,
                              TheConstant => TheConstant);
      AddDeclaration (Item        => TheConstant,
                      Comp_Unit   => Comp_Unit,
                      Loc         => Declaration,
                      Scope       => VisibleScope (ThePackage),
                      Context     => ProgramContext,
                      Declaration => TheDeclaration);
      RawDict.SetConstantDeferredConstantDeclaration (TheConstant, TheDeclaration);

      if not TypeIsUnknown (TypeMark) then
         AddOtherReference (TypeMark, ThePackage, TypeReference);
      end if;

      WriteDeclaration (Discriminant => DeferredConstantDeclaration,
                        Item         => TheConstant,
                        Scope        => VisibleScope (ThePackage),
                        Declaration  => Declaration);

   end AddDeferredConstant;


   procedure AddConstantDeclaration (Name            : in LexTokenManager.Lex_String;
                                     TypeMark        : in Symbol;
                                     TypeReference   : in Location;
                                     Value           : in LexTokenManager.Lex_String;
                                     ExpIsWellFormed : in Boolean;
                                     ExpNode         : in ExaminerConstants.RefType;
                                     Static          : in Boolean;
                                     Comp_Unit       : in ContextManager.UnitDescriptors;
                                     Declaration     : in Location;
                                     Scope           : in Scopes;
                                     Context         : in Contexts)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                ExpIsWellFormed,
   --#                                ExpNode,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope,
   --#                                Static,
   --#                                TypeMark,
   --#                                Value &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope,
   --#                                Static,
   --#                                TypeMark,
   --#                                TypeReference,
   --#                                Value;
   is
      Region, TheConstant, TheDeclaration : Symbol;
   begin

      Region := GetRegion (Scope);

      if (TypeIsPrivate (GetRootType (TypeMark)) or else
          CommandLineData.Content.LanguageProfile /= CommandLineData.SPARK83) and then
         IsPrivateScope (Scope) and then
         IsPackage (Region) and then
         Context = ProgramContext then
         TheConstant := LookupImmediateScope (Name,
                                              VisibleScope (Region),
                                              Context);
      else
         TheConstant := NullSymbol;
      end if;

      if TheConstant = NullSymbol then
         RawDict.CreateConstant (Name        => Name,
                                 TypeMark    => TypeMark,
                                 Static      => Static,
                                 Comp_Unit   => Comp_Unit,
                                 Loc         => Declaration.StartPosition,
                                 TheConstant => TheConstant);
      else
         RawDict.SetConstantStatic (TheConstant, Static);
      end if;

      RawDict.SetConstantValue (TheConstant, Value);
      RawDict.SetConstantExpNode (TheConstant, ExpIsWellFormed, ExpNode);
      AddDeclaration (Item        => TheConstant,
                      Comp_Unit   => Comp_Unit,
                      Loc         => Declaration,
                      Scope       => Scope,
                      Context     => Context,
                      Declaration => TheDeclaration);
      RawDict.SetConstantDeclaration (TheConstant, TheDeclaration);

      if not TypeIsUnknown (TypeMark) then
         AddOtherReference (TypeMark, Region, TypeReference);
      end if;

      WriteDeclaration (Discriminant => ConstantDeclaration,
                        Item         => TheConstant,
                        Scope        => Scope,
                        Declaration  => Declaration);
      --# accept Flow, 601, SPARK_IO.File_Sys, ExpNode, "Spurious data coupling via Dictionary" &
      --#        Flow, 601, SPARK_IO.File_Sys, ExpIsWellFormed, "Spurious data coupling via Dictionary";
   end AddConstantDeclaration;   -- Spurious data coupling betw ExpNode, ExpIsWellFormed and
                                 -- File_Sys via Dictionary

   --------------------------------------------------------------------------------

   -- This procedure is only use in 1 special case - to promote the
   -- standard declaration of System.Default_Bit_Order from a deferred
   -- constant to a normal constant in the case where the full declaration
   -- is supplied in the configuration file.  (Some compilers, particularly GNAT,
   -- do give such a full constant declaration)
   procedure PromoteDeferredToFullConstant (TheConstant : in Symbol;
                                            Comp_Unit   : in ContextManager.UnitDescriptors;
                                            Declaration : in Location;
                                            Value       : in LexTokenManager.Lex_String;
                                            ExpNode     : in ExaminerConstants.RefType;
                                            ThePackage  : in Symbol)
   is
      TheDeclaration : Symbol;
   begin
      AddDeclaration (Item        => TheConstant,
                      Comp_Unit   => Comp_Unit,
                      Loc         => Declaration,
                      Scope       => VisibleScope (ThePackage),
                      Context     => ProgramContext,
                      Declaration => TheDeclaration);

      RawDict.SetConstantValue (TheConstant, Value);
      RawDict.SetConstantExpNode (TheConstant, True, ExpNode);
      RawDict.SetConstantStatic (TheConstant, True);
      RawDict.SetConstantDeclaration (TheConstant, TheDeclaration);
      RawDict.SetConstantDeferredConstantDeclaration (TheConstant, NullSymbol);
   end PromoteDeferredToFullConstant;

   --------------------------------------------------------------------------------

   procedure AddScalarConstant (Name            : in LexTokenManager.Lex_String;
                                TypeMark        : in Symbol;
                                TypeReference   : in Location;
                                Value           : in LexTokenManager.Lex_String;
                                ExpIsWellFormed : in Boolean;
                                ExpNode         : in ExaminerConstants.RefType;
                                Static          : in Boolean;
                                Comp_Unit       : in ContextManager.UnitDescriptors;
                                Declaration     : in Location;
                                Scope           : in Scopes;
                                Context         : in Contexts)
   is
   begin
      AddConstantDeclaration (Name            => Name,
                              TypeMark        => TypeMark,
                              TypeReference   => TypeReference,
                              Value           => Value,
                              ExpIsWellFormed => ExpIsWellFormed,
                              ExpNode         => ExpNode,
                              Static          => Static,
                              Comp_Unit       => Comp_Unit,
                              Declaration     => Declaration,
                              Scope           => Scope,
                              Context         => Context);
   end AddScalarConstant;

   --------------------------------------------------------------------------------

   procedure AddArrayConstant (Name            : in LexTokenManager.Lex_String;
                               TypeMark        : in Symbol;
                               TypeReference   : in Location;
                               ExpIsWellFormed : in Boolean;
                               ExpNode         : in ExaminerConstants.RefType;
                               Static          : in Boolean;
                               Comp_Unit       : in ContextManager.UnitDescriptors;
                               Declaration     : in Location;
                               Scope           : in Scopes;
                               Context         : in Contexts)
   is
   begin
      AddConstantDeclaration (Name            => Name,
                              TypeMark        => TypeMark,
                              TypeReference   => TypeReference,
                              Value           => LexTokenManager.Null_String,
                              ExpIsWellFormed => ExpIsWellFormed,
                              ExpNode         => ExpNode,
                              Static          => Static,
                              Comp_Unit       => Comp_Unit,
                              Declaration     => Declaration,
                              Scope           => Scope,
                              Context         => Context);
   end AddArrayConstant;

   --------------------------------------------------------------------------------

   procedure AddRecordConstant (Name            : in LexTokenManager.Lex_String;
                                TheType         : in Symbol;
                                TypeReference   : in Location;
                                ExpIsWellFormed : in Boolean;
                                ExpNode         : in ExaminerConstants.RefType;
                                Comp_Unit       : in ContextManager.UnitDescriptors;
                                Declaration     : in Location;
                                Scope           : in Scopes;
                                Context         : in Contexts)
   is
   begin
      AddConstantDeclaration (Name            => Name,
                              TypeMark        => TheType,
                              TypeReference   => TypeReference,
                              Value           => LexTokenManager.Null_String,
                              ExpIsWellFormed => ExpIsWellFormed,
                              ExpNode         => ExpNode,
                              Static          => False,
                              Comp_Unit       => Comp_Unit,
                              Declaration     => Declaration,
                              Scope           => Scope,
                              Context         => Context);
   end AddRecordConstant;

   -------------------------------------------------------------------------------

   procedure AddConstantRulePolicy (TheConstant : in Symbol;
                                    Comp_Unit   : in ContextManager.UnitDescriptors;
                                    Declaration : in Location;
                                    TheScope    : in Scopes;
                                    ThePolicy   : in RulePolicies)
   is
      Next : Symbol;
   begin

      RawDict.CreateRulePolicy (Scope         => TheScope,
                                Value         => ThePolicy,
                                Comp_Unit     => Comp_Unit,
                                Loc           => Declaration.StartPosition,
                                TheRulePolicy => Next);

      if RawDict.GetConstantFirstRulePolicy (TheConstant) = NullSymbol then
         RawDict.SetConstantFirstRulePolicy (TheConstant, Next);
      else
         RawDict.SetNextRulePolicy (RawDict.GetConstantLastRulePolicy (TheConstant), Next);
      end if;
      RawDict.SetConstantLastRulePolicy (TheConstant, Next);

   end AddConstantRulePolicy;

   -------------------------------------------------------------------------------

   procedure AddVariable (Name          : in     LexTokenManager.Lex_String;
                          TypeMark      : in     Symbol;
                          Initialized   : in     Boolean;
                          IsAliased     : in     Boolean;
                          ExpNode       : in     ExaminerConstants.RefType;
                          TypeReference : in     Location;
                          Comp_Unit     : in     ContextManager.UnitDescriptors;
                          Declaration   : in     Location;
                          Scope         : in     Scopes;
                          Var_Symbol    :    out Symbol)
   is
      Region, Variable : Symbol;
   begin
      Region := GetRegion (Scope);
      if IsPackage (Region) then
         Variable := LookupImmediateScope (Name, Scope, ProofContext);
      elsif (IsType (Region) and then TypeIsProtected (Region)) then
         -- the "refinement constituents" of a PT are its protected elements in its private part
         Variable := LookupImmediateScope (Name, PrivateScope (Region), ProofContext);
      else
         Variable := NullSymbol;
      end if;

      if Variable = NullSymbol then
         RawDict.CreateVariable (Name      => Name,
                                 Comp_Unit => Comp_Unit,
                                 Loc       => Declaration.StartPosition,
                                 Variable  => Variable);
      elsif RawDict.Get_Symbol_Compilation_Unit (Item => Variable) = Comp_Unit then
         --  There was a usage of the variable before its actual
         --  declaration -> update the declaration position. The
         --  declaration must be in the same compilation unit as the
         --  usage (in the package specification) to prevent to get a
         --  cross-reference from the package specification to the
         --  package body.
         RawDict.Set_Symbol_Location (Item     => Variable,
                                      Location => Declaration.StartPosition);
      end if;

      AddDeclaration (Item        => Variable,
                      Comp_Unit   => Comp_Unit,
                      Loc         => Declaration,
                      Scope       => Scope,
                      Context     => ProgramContext,
                      Declaration => Var_Symbol);
      RawDict.SetVariableDeclaration (Variable    => Variable,
                                      Declaration => Var_Symbol);
      RawDict.SetVariableExpNode (Variable, ExpNode);
      RawDict.SetVariableType (Variable => Variable,
                               TypeMark => TypeMark);

      if IsAliased then
         RawDict.SetVariableIsAliased (Variable);
      end if;

      if Initialized then
         RawDict.SetVariableInitialized (Variable);
      end if;

      if not TypeIsUnknown (TypeMark) then
         AddOtherReference (TypeMark, Region, TypeReference);
      end if;

      WriteDeclaration (Discriminant => VariableDeclaration,
                        Item         => Variable,
                        Scope        => Scope,
                        Declaration  => Declaration);
      --# accept Flow, 601, SPARK_IO.File_Sys, ExpNode, "Spurious data coupling via Dictionary";
   end AddVariable;  -- Spurious data coupling betw ExpNode and
                     -- File_Sys via Dictionary

   -------------------------------------------------------------------------------

   procedure AddVariableAddressClause (Variable : in Symbol)
   is
   begin
      RawDict.SetVariableHasAddressClause (Variable);
   end AddVariableAddressClause;

   -------------------------------------------------------------------------------

   procedure AddVariablePragmaImport (Variable : in Symbol)
   is
   begin
      RawDict.SetVariableHasPragmaImport (Variable);
      -- Ada LRM states that variable with pragma import is deemed to be initialized by external code segment
      RawDict.SetVariableInitialized (Variable);  -- so we can mark it as initialized
   end AddVariablePragmaImport;

   -------------------------------------------------------------------------------

   procedure AddTypeSizeAttribute (TypeMark : in Symbol; SizeVal : in LexTokenManager.Lex_String)
   is
   begin
      RawDict.SetTypeSizeAttribute (TypeMark, SizeVal);
   end AddTypeSizeAttribute;

   -------------------------------------------------------------------------------

   procedure SetVariableMarkedValid (Variable : in Symbol; Val : in Boolean)
   is
   begin
      RawDict.SetVariableMarkedValid (Variable, Val);
   end SetVariableMarkedValid;

   -------------------------------------------------------------------------------

   procedure SetSubcomponentMarkedValid (Subcomponent : in Symbol; Val : in Boolean)
   is
   begin
      RawDict.SetSubcomponentMarkedValid (Subcomponent, Val);
   end SetSubcomponentMarkedValid;

   -------------------------------------------------------------------------------

   procedure SetTypeAtomic (TypeMark : in Symbol)
   is
   begin
      RawDict.SetTypeAtomic (TypeMark => TypeMark);
   end SetTypeAtomic;

   -------------------------------------------------------------------------------

   procedure AddRecordSubcomponent (Prefix       : in     Symbol;
                                    Component    : in     Symbol;
                                    Comp_Unit    : in     ContextManager.UnitDescriptors;
                                    Subcomponent :    out Symbol)
      is separate;

   -------------------------------------------------------------------------------

   procedure AddQuantifiedVariable (Name        : in     LexTokenManager.Lex_String;
                                    Comp_Unit   : in     ContextManager.UnitDescriptors;
                                    Declaration : in     Location;
                                    TypeMark    : in     Symbol;
                                    Region      : in     Symbol;
                                    Variable    :    out Symbol)
   is
   begin
      RawDict.CreateQuantifiedVariable (Name      => Name,
                                        TypeMark  => TypeMark,
                                        Region    => Region,
                                        Comp_Unit => Comp_Unit,
                                        Loc       => Declaration.StartPosition,
                                        Variable  => Variable);
      if not TypeIsUnknown (TypeMark) then
         AddOtherReference (TypeMark, Region, Declaration);
      end if;
   end AddQuantifiedVariable;

   --------------------------------------------------------------------------------

   procedure AddRecordTypeMark (TypeMark       : in Symbol;
                                IsTaggedType   : in Boolean;
                                IsAbstractType : in Boolean;
                                Extends        : in Symbol)
      --# global in out Dict;
      --# derives Dict from *,
      --#                   Extends,
      --#                   IsAbstractType,
      --#                   IsTaggedType,
      --#                   TypeMark;
   is
   begin
      RawDict.SetTypeDiscriminant (TypeMark     => TypeMark,
                                   Discriminant => RecordType);
      RawDict.SetTypeStatic (TypeMark => TypeMark,
                             Static   => False);
      RawDict.SetTypeIsTagged (TypeMark, IsTaggedType);
      RawDict.SetTypeIsAbstract (TypeMark, IsAbstractType);
      RawDict.SetTypeExtends (TypeMark, Extends);
   end AddRecordTypeMark;

   --------------------------------------------------------------------------------

   procedure AddTypeAnnouncement (Name         : in     LexTokenManager.Lex_String;
                                  Comp_Unit    : in     ContextManager.UnitDescriptors;
                                  Announcement : in     Location;
                                  ThePackage   : in     Symbol;
                                  TypeMark     :    out Symbol)
   is
      NewTypeMark, TheAnnouncement : Symbol;
   begin
      RawDict.CreateType (Name      => Name,
                          Comp_Unit => Comp_Unit,
                          Loc       => Announcement.StartPosition,
                          TypeMark  => NewTypeMark);
      AddDeclaration (Item        => NewTypeMark,
                      Comp_Unit   => Comp_Unit,
                      Loc         => Announcement,
                      Scope       => VisibleScope (ThePackage),
                      Context     => ProofContext,
                      Declaration => TheAnnouncement);
      RawDict.SetTypeAnnouncement (NewTypeMark, TheAnnouncement);
      WriteDeclaration (Discriminant => TypeAnnouncement,
                        Item         => NewTypeMark,
                        Scope        => VisibleScope (ThePackage),
                        Declaration  => Announcement);
      TypeMark := NewTypeMark;
   end AddTypeAnnouncement;

   --------------------------------------------------------------------------------

   procedure AddPrivateType (Name           : in LexTokenManager.Lex_String;
                             Comp_Unit      : in ContextManager.UnitDescriptors;
                             Declaration    : in Location;
                             ThePackage     : in Symbol;
                             IsLimited      : in Boolean;
                             IsTaggedType   : in Boolean;
                             IsAbstractType : in Boolean;
                             Extends        : in Symbol)
   is
      TheType, TheDeclaration : Symbol;
   begin

      TheType := LookupImmediateScope (Name,
                                       VisibleScope (ThePackage),
                                       ProofContext);

      if TheType = NullSymbol then
         RawDict.CreateType (Name      => Name,
                             Comp_Unit => Comp_Unit,
                             Loc       => Declaration.StartPosition,
                             TypeMark  => TheType);
      elsif RawDict.Get_Symbol_Compilation_Unit (Item => TheType) = Comp_Unit then
         --  There was a usage of the type before its actual
         --  declaration -> update the declaration position. The
         --  declaration must be in the same compilation unit as the
         --  usage (in the package specification) to prevent to get a
         --  cross-reference from the package specification to the
         --  package body.
         RawDict.Set_Symbol_Location (Item     => TheType,
                                      Location => Declaration.StartPosition);
      end if;

      AddDeclaration (Item        => TheType,
                      Comp_Unit   => Comp_Unit,
                      Loc         => Declaration,
                      Scope       => VisibleScope (ThePackage),
                      Context     => ProgramContext,
                      Declaration => TheDeclaration);

      RawDict.SetTypePrivate (TheType, Sometimes);
      RawDict.SetTypePrivateTypeDeclaration (TheType, TheDeclaration);
      RawDict.SetTypeExtends (TheType, Extends);

      if IsLimited then
         RawDict.SetTypeLimited (TypeMark  => TheType,
                                 IsLimited => Sometimes);
         RawDict.SetTypeLimitedPrivate (TheType);
      else
         RawDict.SetTypeLimited (TypeMark  => TheType,
                                 IsLimited => Never);
      end if;

      if IsTaggedType then
         RawDict.SetPackageDeclaresTaggedType (ThePackage);
      end if;

      RawDict.SetTypeIsTagged (TheType, IsTaggedType);
      RawDict.SetTypeIsAbstract (TheType, IsAbstractType);

      WriteDeclaration (Discriminant => PrivateTypeDeclaration,
                        Item         => TheType,
                        Scope        => VisibleScope (ThePackage),
                        Declaration  => Declaration);

      --# accept Flow, 601, SPARK_IO.File_Sys, IsTaggedType, "Spurious data coupling via Dictionary" &
      --#        Flow, 601, SPARK_IO.File_Sys, IsAbstractType, "Spurious data coupling via Dictionary" &
      --#        Flow, 601, SPARK_IO.File_Sys, Extends, "Spurious data coupling via Dictionary";
   end AddPrivateType; -- false coupling, IsTagged etc & SPARKIO

   --------------------------------------------------------------------------------

   procedure AddTypeDeclaration (Name        : in     LexTokenManager.Lex_String;
                                 Comp_Unit   : in     ContextManager.UnitDescriptors;
                                 Declaration : in     Location;
                                 Scope       : in     Scopes;
                                 Context     : in     Contexts;
                                 TypeMark    :    out Symbol)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives TypeMark          from Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope &
   --#         Dict,
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope;
   is
      Region,
      NewType,
      OldType,
      TheDeclaration : Symbol;
   begin
      Region := GetRegion (Scope);

      if IsPackage (Region) and then Context = ProgramContext then
         -------------------------------------------------------------
         -- If declaring a type within a package in ProgramContext,
         -- then that type may have already been declared by way of a
         -- private type declaration, a type-annoucement or both.
         -------------------------------------------------------------
         OldType := LookupImmediateScope (Name,
                                          VisibleScope (Region),
                                          ProofContext);

         if OldType /= NullSymbol then
            ------------------------------------------------------------
            -- Type already exists in the Dictionary, so it MUST be
            -- either private, announced or both.
            ------------------------------------------------------------
            SystemErrors.RTAssert
              (TypeIsAnnounced (OldType) or TypeIsPrivate (OldType),
               SystemErrors.AssertionFailure,
               "Re-declaration of a non-announced, non-private type");

            AddDeclaration (Item        => OldType,
                            Comp_Unit   => Comp_Unit,
                            Loc         => Declaration,
                            Scope       => Scope,
                            Context     => Context,
                            Declaration => TheDeclaration);
            RawDict.SetTypeDeclaration (OldType, TheDeclaration);
            WriteDeclaration (Discriminant => TypeDeclaration,
                              Item         => OldType,
                              Scope        => Scope,
                              Declaration  => Declaration);
            TypeMark := OldType;
            if RawDict.Get_Symbol_Compilation_Unit (Item => OldType) = Comp_Unit then
               --  There was a usage of the type before its actual
               --  declaration -> update the declaration position. The
               --  declaration must be in the same compilation unit as
               --  the usage (in the package specification) to prevent
               --  to get a cross-reference from the package
               --  specification to the package body.
               RawDict.Set_Symbol_Location (Item     => OldType,
                                            Location => Declaration.StartPosition);
            end if;
         else
            ------------------------------------------------------------
            -- Type does not already exist in Dictionary, so create it
            ------------------------------------------------------------
            RawDict.CreateType (Name      => Name,
                                Comp_Unit => Comp_Unit,
                                Loc       => Declaration.StartPosition,
                                TypeMark  => NewType);
            AddDeclaration (Item        => NewType,
                            Comp_Unit   => Comp_Unit,
                            Loc         => Declaration,
                            Scope       => Scope,
                            Context     => Context,
                            Declaration => TheDeclaration);
            RawDict.SetTypeDeclaration (NewType, TheDeclaration);
            WriteDeclaration (Discriminant => TypeDeclaration,
                              Item         => NewType,
                              Scope        => Scope,
                              Declaration  => Declaration);
            TypeMark := NewType;
         end if;

      else
         ------------------------------------------------------------
         -- Not in a package, so create the type from scratch
         ------------------------------------------------------------
         RawDict.CreateType (Name      => Name,
                             Comp_Unit => Comp_Unit,
                             Loc       => Declaration.StartPosition,
                             TypeMark  => NewType);
         AddDeclaration (Item        => NewType,
                         Comp_Unit   => Comp_Unit,
                         Loc         => Declaration,
                         Scope       => Scope,
                         Context     => Context,
                         Declaration => TheDeclaration);
         RawDict.SetTypeDeclaration (NewType, TheDeclaration);
         WriteDeclaration (Discriminant => TypeDeclaration,
                           Item         => NewType,
                           Scope        => Scope,
                           Declaration  => Declaration);
         TypeMark := NewType;
      end if;
   end AddTypeDeclaration;

   --------------------------------------------------------------------------------

   procedure AddEnumerationType (Name            : in     LexTokenManager.Lex_String;
                                 Comp_Unit       : in     ContextManager.UnitDescriptors;
                                 Declaration     : in     Location;
                                 Scope           : in     Scopes;
                                 Context         : in     Contexts;
                                 EnumerationType :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => EnumerationType);
      AddEnumerationTypeMark (TypeMark => EnumerationType,
                              Static   => True);
   end AddEnumerationType;

   --------------------------------------------------------------------------------

   procedure AddAccessType (AccessedType : in Symbol;
                            Comp_Unit    : in ContextManager.UnitDescriptors;
                            Declaration  : in Location)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   AccessedType,
   --#                   Comp_Unit,
   --#                   Declaration;
   is
      NewAccessType     : Symbol;
   begin
      -- Create a type record with the name "Access" to ensure we never look it up
      -- by mistake (access being reserved word).
      RawDict.CreateType (Name      => LexTokenManager.Access_Token,
                          Comp_Unit => Comp_Unit,
                          Loc       => Declaration.StartPosition,
                          TypeMark  => NewAccessType);
      -- mark it as an access type
      RawDict.SetTypeDiscriminant (TypeMark     => NewAccessType,
                                   Discriminant => AccessType);
      -- set other fields to "safe" values
      RawDict.SetTypeStatic (TypeMark => NewAccessType,
                             Static   => False);
      RawDict.SetTypeEqualityDefined (TypeMark        => NewAccessType,
                                      EqualityDefined => False);
      RawDict.SetTypeLimited (TypeMark  => NewAccessType,
                              IsLimited => Always);
      -- link type and access type together
      RawDict.SetTypeAccesses (AccessedType, NewAccessType);
      RawDict.SetTypeAccesses (NewAccessType, AccessedType);
   end AddAccessType;

   --------------------------------------------------------------------------------

   procedure AddProtectedType (Name          : in     LexTokenManager.Lex_String;
                               Comp_Unit     : in     ContextManager.UnitDescriptors;
                               Declaration   : in     Location;
                               Scope         : in     Scopes;
                               Context       : in     Contexts;
                               Mode          : in     Modes;
                               Constrained   : in     Boolean;
                               ProtectedType :    out Symbol)
   is
      NewVariable : Symbol;
      OwnVariable : Symbol;
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => ProtectedType);
      AddProtectedTypeMark (TypeMark    => ProtectedType,
                            Comp_Unit   => Comp_Unit,
                            Declaration => Declaration);

      -- We also declare an abstract own variable with the same name as the protected type
      -- at this point.  The existence of such a variable simplifies the use of the protected
      -- type name in the annotations of its operations
      RawDict.CreateVariable (Name      => Name,
                              Comp_Unit => Comp_Unit,
                              Loc       => Declaration.StartPosition,
                              Variable  => NewVariable);
      RawDict.CreateOwnVariable (Variable    => NewVariable,
                                 Owner       => ProtectedType,
                                 Comp_Unit   => Comp_Unit,
                                 Loc         => Declaration.StartPosition,
                                 OwnVariable => OwnVariable);
      RawDict.SetVariableOwnVariable (NewVariable, OwnVariable);
      RawDict.SetProtectedTypeOwnVariable (TheProtectedType => ProtectedType,
                                           OwnVariable      => OwnVariable);
      RawDict.SetTypeConstrained (ArrayType   => ProtectedType,
                                  Constrained => Constrained);

      -- the type of this fictitious abstract own variable is the type itself
      RawDict.SetVariableType (Variable => NewVariable,
                               TypeMark => ProtectedType);
      RawDict.SetOwnVariableTyped (OwnVariable);

      -- set some sensible values for the new own variable
      RawDict.SetOwnVariableAnnounced (OwnVariable);
      RawDict.SetOwnVariableInitialized (OwnVariable);
      RawDict.SetOwnVariableMode (OwnVariable, Mode);
      RawDict.SetOwnVariableProtected (OwnVariable, True);

      -- implicitly declare an access type to go with the PT
      AddAccessType (AccessedType => ProtectedType,
                     Comp_Unit    => Comp_Unit,
                     Declaration  => Declaration);
   end AddProtectedType;

   --------------------------------------------------------------------------------

   procedure AddTaskType (Name        : in     LexTokenManager.Lex_String;
                          Comp_Unit   : in     ContextManager.UnitDescriptors;
                          Declaration : in     Location;
                          Scope       : in     Scopes;
                          Context     : in     Contexts;
                          Constrained : in     Boolean;
                          TaskType    :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => TaskType);
      AddTaskTypeMark (TypeMark    => TaskType,
                       Comp_Unit   => Comp_Unit,
                       Declaration => Declaration);
      RawDict.SetTypeConstrained (ArrayType   => TaskType,
                                  Constrained => Constrained);
   end AddTaskType;

   --------------------------------------------------------------------------------

   procedure AddTaskOrProtectedSubtype (Name        : in     LexTokenManager.Lex_String;
                                        Parent      : in     Symbol;
                                        Comp_Unit   : in     ContextManager.UnitDescriptors;
                                        Declaration : in     Location;
                                        Scope       : in     Scopes;
                                        Context     : in     Contexts;
                                        TheSubtype  :    out Symbol)
   is
      ExtraInfo : Symbol;
   begin
      SystemErrors.RTAssert (IsTaskType (Parent) or else IsProtectedType (Parent),
                             SystemErrors.PreconditionFailure,
                             "In call to AddTaskOrProtectedSubtype");

      -- add a "basic" type record making sure we use any existing announced type
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => TheSubtype);
      -- set its fields to suitable values:
      -- subtype is same sort of thing as its parent
      RawDict.SetTypeDiscriminant (TypeMark     => TheSubtype,
                                   Discriminant => RawDict.GetTypeDiscriminant (Parent));
      -- make types as limited as possible
      RawDict.SetTypeStatic (TypeMark => TheSubtype,
                             Static   => False);
      RawDict.SetTypeEqualityDefined (TypeMark        => TheSubtype,
                                      EqualityDefined => False);
      RawDict.SetTypeLimited (TypeMark  => TheSubtype,
                              IsLimited => Always);
      if RawDict.IsAtomic (Parent) then
         RawDict.SetTypeAtomic (TheSubtype);
      end if;
      -- link subtype to parent
      RawDict.SetTypeParent (TheSubtype => TheSubtype,
                             Parent     => Parent);
      -- a legal subtype must have had its discriminants constrained
      RawDict.SetTypeConstrained (ArrayType   => TheSubtype,
                                  Constrained => True);

      -- create subtype info record and hook it to AncilliaryFields of subtype
      RawDict.CreateSubtype (Comp_Unit  => Comp_Unit,
                             Loc        => Declaration.StartPosition,
                             TheSubtype => ExtraInfo);
      RawDict.SetTypeAncilliaryFields (TheType     => TheSubtype,
                                       Declaration => ExtraInfo);
      WriteDeclaration (Discriminant => TypeDeclaration,
                        Item         => TheSubtype,
                        Scope        => Scope,
                        Declaration  => Declaration);
   end AddTaskOrProtectedSubtype;

   --------------------------------------------------------------------------------

   procedure WriteRepresentationClause (TheType : in Symbol;
                                        Clause  : in Location)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Clause,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                TheType;
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile, "representation clause of ");
         WriteName (Dict.TemporaryFile, TheType);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Clause);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end WriteRepresentationClause;

   --------------------------------------------------------------------------------

   procedure AddEnumerationRepresentationClause (TheType : in Symbol;
                                                 Clause  : in Location)
   is
   begin
      WriteRepresentationClause (TheType, Clause);
   end AddEnumerationRepresentationClause;

   --------------------------------------------------------------------------------

   procedure AddEnumerationLiteralRepresentation (Literal : in Symbol;
                                                  Code    : in Integer)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile,
                      "enumeration literal representation of ");
         WriteName (Dict.TemporaryFile, Literal);
         WriteString (Dict.TemporaryFile, " is ");
         WriteInteger (Dict.TemporaryFile, Code);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddEnumerationLiteralRepresentation;

   --------------------------------------------------------------------------------

   procedure AddIntegerType (Name        : in     LexTokenManager.Lex_String;
                             Comp_Unit   : in     ContextManager.UnitDescriptors;
                             Declaration : in     Location;
                             Lower       : in     LexTokenManager.Lex_String;
                             Upper       : in     LexTokenManager.Lex_String;
                             Scope       : in     Scopes;
                             Context     : in     Contexts;
                             Type_Symbol :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => Type_Symbol);
      AddIntegerTypeMark (TypeMark => Type_Symbol,
                          Static   => True,
                          Lower    => Lower,
                          Upper    => Upper);
      RawDict.SetTypeDerived (TypeMark => Type_Symbol,
                              Derived  => True);
   end AddIntegerType;

   --------------------------------------------------------------------------------
   procedure AddPreDefIntegerType (Name        : in LexTokenManager.Lex_String;
                                   Comp_Unit   : in ContextManager.UnitDescriptors;
                                   Declaration : in Location;
                                   Lower       : in LexTokenManager.Lex_String;
                                   Upper       : in LexTokenManager.Lex_String;
                                   Scope       : in Scopes;
                                   Context     : in Contexts)
   is
      TheType : Symbol;
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => TheType);
      AddIntegerTypeMark (TypeMark => TheType,
                          Static   => True,
                          Lower    => Lower,
                          Upper    => Upper);
   end AddPreDefIntegerType;

   --------------------------------------------------------------------------------

   procedure AddModularType (Name        : in     LexTokenManager.Lex_String;
                             Comp_Unit   : in     ContextManager.UnitDescriptors;
                             Declaration : in     Location;
                             Modulus     : in     LexTokenManager.Lex_String;
                             Scope       : in     Scopes;
                             Context     : in     Contexts;
                             Type_Symbol :    out Symbol)
   is
      Upper      : LexTokenManager.Lex_String;
      UpperValue : Maths.Value;
      Ok         : Maths.ErrorCode;
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => Type_Symbol);

      -- T'Last is (T'Modulus - 1), so...
      Maths.Subtract (Maths.ValueRep (Modulus),
                      Maths.OneInteger,
                      -- to get
                      UpperValue,
                      Ok);

      if Ok = Maths.NoError then
         Maths.StorageRep (UpperValue, Upper);
      else
         Upper := LexTokenManager.Null_String;
      end if;

      AddModularTypeMark (TypeMark => Type_Symbol,
                          Static   => True,
                          Lower    => LexTokenManager.Zero_Value,
                          Upper    => Upper,
                          Modulus  => Modulus);
   end AddModularType;

   --------------------------------------------------------------------------------

   procedure AddAbstractProofType (Name        : in     LexTokenManager.Lex_String;
                                   Comp_Unit   : in     ContextManager.UnitDescriptors;
                                   Declaration : in     Location;
                                   Scope       : in     Scopes;
                                   TypeSym     :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => ProgramContext, --set to this so we find any announcements
                          TypeMark    => TypeSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeSym,
                                   Discriminant => AbstractProofType);
   end AddAbstractProofType;

   --------------------------------------------------------------------------------

   procedure AddDefaultAbstractProofType (OwnVarName  : in     LexTokenManager.Lex_String;
                                          Comp_Unit   : in     ContextManager.UnitDescriptors;
                                          Declaration : in     Location;
                                          Scope       : in     Scopes;
                                          TypeSym     :    out Symbol)
   is
      Name     : EStrings.T;
      TypeName : LexTokenManager.Lex_String;
   begin --AddDefaultAbstractProofType
         --create name for type by appending to __type to own variable name (e.g. state__type)
      Name := LexTokenManager.Lex_String_To_String (Lex_Str => OwnVarName);
      EStrings.Append_String (E_Str => Name,
                              Str   => "__type");
      LexTokenManager.Insert_Examiner_String
        (Str     => Name,
         Lex_Str => TypeName);
      AddAbstractProofType (Name        => TypeName,
                            Comp_Unit   => Comp_Unit,
                            Declaration => Declaration,
                            Scope       => Scope,
                            TypeSym     => TypeSym);
   end AddDefaultAbstractProofType;

   --------------------------------------------------------------------------------

   procedure AddFloatingPointType (Name        : in     LexTokenManager.Lex_String;
                                   Comp_Unit   : in     ContextManager.UnitDescriptors;
                                   Declaration : in     Location;
                                   Lower       : in     LexTokenManager.Lex_String;
                                   Upper       : in     LexTokenManager.Lex_String;
                                   ErrorBound  : in     LexTokenManager.Lex_String;
                                   Scope       : in     Scopes;
                                   Context     : in     Contexts;
                                   Type_Symbol :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => Type_Symbol);
      AddFloatingPointTypeMark (TypeMark => Type_Symbol,
                                Static   => True);
      RawDict.SetTypeLower (TypeMark => Type_Symbol,
                            Lower    => Lower);
      RawDict.SetTypeUpper (TypeMark => Type_Symbol,
                            Upper    => Upper);
      RawDict.SetTypeErrorBound (TypeMark   => Type_Symbol,
                                 ErrorBound => ErrorBound);
      RawDict.SetTypeDerived (TypeMark => Type_Symbol,
                              Derived  => True);
   end AddFloatingPointType;

   --------------------------------------------------------------------------------

   procedure AddPreDefFloatingPointType
     (Name        : in LexTokenManager.Lex_String;
      Comp_Unit   : in ContextManager.UnitDescriptors;
      Declaration : in Location;
      Lower       : in LexTokenManager.Lex_String;
      Upper       : in LexTokenManager.Lex_String;
      ErrorBound  : in LexTokenManager.Lex_String;
      Scope       : in Scopes;
      Context     : in Contexts)
   is
      TheType : Symbol;
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => TheType);
      AddFloatingPointTypeMark (TypeMark => TheType,
                                Static   => True);
      RawDict.SetTypeLower (TypeMark => TheType,
                            Lower    => Lower);
      RawDict.SetTypeUpper (TypeMark => TheType,
                            Upper    => Upper);
      RawDict.SetTypeErrorBound (TypeMark   => TheType,
                                 ErrorBound => ErrorBound);
   end AddPreDefFloatingPointType;

   --------------------------------------------------------------------------------

   procedure SetTypeIsPredefined (TypeMark : in Symbol) is
   begin
      RawDict.SetTypeDerived (TypeMark => TypeMark,
                              Derived  => False);
   end SetTypeIsPredefined;

   --------------------------------------------------------------------------------

   procedure SetTypeIsWellformed (TypeMark   : in Symbol;
                                  Wellformed : in Boolean) is
   begin
      RawDict.SetTypeWellformed (TypeMark, Wellformed);
   end SetTypeIsWellformed;

   --------------------------------------------------------------------------------

   procedure SetBaseType (TypeMark, BaseType : in Symbol) is
   begin
      RawDict.SetTypeBaseType (TypeMark, BaseType);
   end SetBaseType;

   --------------------------------------------------------------------------------

   procedure AddFixedPointType (Name        : in     LexTokenManager.Lex_String;
                                Comp_Unit   : in     ContextManager.UnitDescriptors;
                                Declaration : in     Location;
                                Lower       : in     LexTokenManager.Lex_String;
                                Upper       : in     LexTokenManager.Lex_String;
                                ErrorBound  : in     LexTokenManager.Lex_String;
                                Scope       : in     Scopes;
                                Context     : in     Contexts;
                                Type_Symbol :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => Type_Symbol);
      AddFixedPointTypeMark (TypeMark => Type_Symbol,
                             Static   => True);
      RawDict.SetTypeLower (TypeMark => Type_Symbol,
                            Lower    => Lower);
      RawDict.SetTypeUpper (TypeMark => Type_Symbol,
                            Upper    => Upper);
      RawDict.SetTypeErrorBound (TypeMark   => Type_Symbol,
                                 ErrorBound => ErrorBound);
   end AddFixedPointType;

   --------------------------------------------------------------------------------

   procedure AddArrayType (Name                   : in     LexTokenManager.Lex_String;
                           Comp_Unit              : in     ContextManager.UnitDescriptors;
                           Declaration            : in     Location;
                           Scope                  : in     Scopes;
                           Context                : in     Contexts;
                           Constrained            : in     Boolean;
                           ComponentType          : in     Symbol;
                           ComponentTypeReference : in     Location;
                           IsGeneric              : in     Boolean;
                           TheArrayType           :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => TheArrayType);
      AddArrayTypeMark (TypeMark      => TheArrayType,
                        ComponentType => ComponentType,
                        Static        => False);
      -- Above call will have set type discriminant to ArrayType
      -- If we are adding a generic array we overide that setting thus:
      if IsGeneric then
         RawDict.SetTypeDiscriminant (TypeMark     => TheArrayType,
                                      Discriminant => GenericArrayType);
      end if;

      RawDict.SetTypeConstrained (ArrayType   => TheArrayType,
                                  Constrained => Constrained);

      if not TypeIsUnknown (ComponentType) then
         AddOtherReference (ComponentType,
                            GetRegion (Scope),
                            ComponentTypeReference);
      end if;
   end AddArrayType;

   --------------------------------------------------------------------------------

   procedure AddArrayIndex (TheArrayType       : in Symbol;
                            TheIndexType       : in Symbol;
                            Comp_Unit          : in ContextManager.UnitDescriptors;
                            IndexTypeReference : in Location)
   is
   begin

      AddNewArrayIndex (TheArrayType => TheArrayType,
                        IndexType    => TheIndexType,
                        Comp_Unit    => Comp_Unit,
                        Declaration  => IndexTypeReference);

      if not TypeIsUnknown (TheIndexType) then
         AddOtherReference (TheIndexType,
                            GetRegion (GetScope (TheArrayType)),
                            IndexTypeReference);
      end if;

   end AddArrayIndex;

   --------------------------------------------------------------------------------

   procedure AddRecordType (Name           : in     LexTokenManager.Lex_String;
                            IsTaggedType   : in     Boolean;
                            IsAbstractType : in     Boolean;
                            Extends        : in     Symbol;
                            Comp_Unit      : in     ContextManager.UnitDescriptors;
                            Declaration    : in     Location;
                            Scope          : in     Scopes;
                            Context        : in     Contexts;
                            TheRecordType  :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => Context,
                          TypeMark    => TheRecordType);
      AddRecordTypeMark (TypeMark       => TheRecordType,
                         IsTaggedType   => IsTaggedType,
                         IsAbstractType => IsAbstractType,
                         Extends        => Extends);
      -- mark package as containing a tagged type if necessary
      if IsTaggedType then
         RawDict.SetPackageDeclaresTaggedType (GetRegion (Scope));
      end if;
   end AddRecordType;

   --------------------------------------------------------------------------------

   procedure AddRecordComponent (Name                   : in LexTokenManager.Lex_String;
                                 Comp_Unit              : in ContextManager.UnitDescriptors;
                                 Declaration            : in Location;
                                 TheRecordType          : in Symbol;
                                 TheComponentType       : in Symbol;
                                 InheritedField         : in Boolean;
                                 ComponentTypeReference : in Location)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddGenericLimitedType (Name        : in     LexTokenManager.Lex_String;
                                    Comp_Unit   : in     ContextManager.UnitDescriptors;
                                    Declaration : in     Location;
                                    Scope       : in     Scopes;
                                    TypeSym     :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => ProgramContext,
                          -- to get
                          TypeMark    => TypeSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeSym,
                                   Discriminant => GenericLimitedType);
   end AddGenericLimitedType;

   --------------------------------------------------------------------------------

   procedure AddGenericPrivateType (Name        : in     LexTokenManager.Lex_String;
                                    Comp_Unit   : in     ContextManager.UnitDescriptors;
                                    Declaration : in     Location;
                                    Scope       : in     Scopes;
                                    TypeSym     :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => ProgramContext,
                          -- to get
                          TypeMark    => TypeSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeSym,
                                   Discriminant => GenericPrivateType);
   end AddGenericPrivateType;

   --------------------------------------------------------------------------------

   procedure AddGenericOrderedDiscreteType (Name        : in     LexTokenManager.Lex_String;
                                            Comp_Unit   : in     ContextManager.UnitDescriptors;
                                            Declaration : in     Location;
                                            Scope       : in     Scopes;
                                            TypeSym     :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => ProgramContext,
                          -- to get
                          TypeMark    => TypeSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeSym,
                                   Discriminant => GenericOrderedDiscreteType);
   end AddGenericOrderedDiscreteType;

   --------------------------------------------------------------------------------

   procedure AddGenericUnorderedDiscreteType (Name        : in     LexTokenManager.Lex_String;
                                              Comp_Unit   : in     ContextManager.UnitDescriptors;
                                              Declaration : in     Location;
                                              Scope       : in     Scopes;
                                              TypeSym     :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => ProgramContext,
                          -- to get
                          TypeMark    => TypeSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeSym,
                                   Discriminant => GenericUnorderedDiscreteType);
   end AddGenericUnorderedDiscreteType;

   --------------------------------------------------------------------------------

   procedure AddGenericIntegerType (Name        : in     LexTokenManager.Lex_String;
                                    Comp_Unit   : in     ContextManager.UnitDescriptors;
                                    Declaration : in     Location;
                                    Scope       : in     Scopes;
                                    TypeSym     :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => ProgramContext,
                          -- to get
                          TypeMark    => TypeSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeSym,
                                   Discriminant => GenericIntegerType);
   end AddGenericIntegerType;

   --------------------------------------------------------------------------------

   procedure AddGenericModularType (Name        : in     LexTokenManager.Lex_String;
                                    Comp_Unit   : in     ContextManager.UnitDescriptors;
                                    Declaration : in     Location;
                                    Scope       : in     Scopes;
                                    TypeSym     :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => ProgramContext,
                          -- to get
                          TypeMark    => TypeSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeSym,
                                   Discriminant => GenericModularType);
   end AddGenericModularType;

   --------------------------------------------------------------------------------

   procedure AddGenericFloatingPointType (Name        : in     LexTokenManager.Lex_String;
                                          Comp_Unit   : in     ContextManager.UnitDescriptors;
                                          Declaration : in     Location;
                                          Scope       : in     Scopes;
                                          TypeSym     :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => ProgramContext,
                          -- to get
                          TypeMark    => TypeSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeSym,
                                   Discriminant => GenericFloatingPointType);
   end AddGenericFloatingPointType;

   --------------------------------------------------------------------------------

   procedure AddGenericFixedPointType (Name        : in     LexTokenManager.Lex_String;
                                       Comp_Unit   : in     ContextManager.UnitDescriptors;
                                       Declaration : in     Location;
                                       Scope       : in     Scopes;
                                       TypeSym     :    out Symbol)
   is
   begin
      AddTypeDeclaration (Name        => Name,
                          Comp_Unit   => Comp_Unit,
                          Declaration => Declaration,
                          Scope       => Scope,
                          Context     => ProgramContext,
                          -- to get
                          TypeMark    => TypeSym);
      RawDict.SetTypeDiscriminant (TypeMark     => TypeSym,
                                   Discriminant => GenericFixedPointType);
   end AddGenericFixedPointType;

   --------------------------------------------------------------------------------

   procedure AddRecordRepresentationClause (TheType : in Symbol;
                                            Clause  : in Location)
   is
   begin
      WriteRepresentationClause (TheType, Clause);
   end AddRecordRepresentationClause;

   --------------------------------------------------------------------------------

   procedure AddRecordComponentRepresentation (Component        : in Symbol;
                                               Clause           : in Location;
                                               RelativeAddress  : in Natural;
                                               FirstBitPosition : in Natural;
                                               LastBitPosition  : in Natural)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile, "record component representation of ");
         WriteName (Dict.TemporaryFile, Component);
         WriteSpace (Dict.TemporaryFile);
         WriteInteger (Dict.TemporaryFile, RelativeAddress);
         WriteString (Dict.TemporaryFile, " range ");
         WriteInteger (Dict.TemporaryFile, FirstBitPosition);
         WriteString (Dict.TemporaryFile, " ... ");
         WriteInteger (Dict.TemporaryFile, LastBitPosition);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Clause);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddRecordComponentRepresentation;

   --------------------------------------------------------------------------------

   procedure AddAlignmentClause (TheType : in Symbol;
                                 Clause  : in Location)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile, "alignment clause of ");
         WriteName (Dict.TemporaryFile, TheType);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Clause);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddAlignmentClause;

   --------------------------------------------------------------------------------

   procedure AddLoop (Scope         : in     Scopes;
                      Comp_Unit     : in     ContextManager.UnitDescriptors;
                      LoopStatement : in     Location;
                      TheLoop       :    out Symbol)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddLoopName (Name    : in LexTokenManager.Lex_String;
                          TheLoop : in Symbol)
   is
   begin
      RawDict.SetLoopName (Name, TheLoop);
   end AddLoopName;

   --------------------------------------------------------------------------------

   procedure AddLoopParameter (TheLoop       : in Symbol;
                               Comp_Unit     : in ContextManager.UnitDescriptors;
                               Declaration   : in Location;
                               Name          : in LexTokenManager.Lex_String;
                               TypeMark      : in Symbol;
                               StaticRange   : in Boolean;
                               IsReverse     : in Boolean;
                               TypeReference : in Location)
   is
      LoopParameter : Symbol;
   begin
      RawDict.CreateLoopParameter (Name           => Name,
                                   TypeMark       => TypeMark,
                                   TheLoop        => TheLoop,
                                   HasStaticRange => StaticRange,
                                   IsReverse      => IsReverse,
                                   Comp_Unit      => Comp_Unit,
                                   Loc            => Declaration.StartPosition,
                                   -- to get
                                   LoopParameter  => LoopParameter);
      RawDict.SetLoopParameter (TheLoop, LoopParameter);

      if not TypeIsUnknown (TypeMark) then
         AddOtherReference (TypeMark, TheLoop, TypeReference);
      end if;
   end AddLoopParameter;

   --------------------------------------------------------------------------------

   procedure MarkLoopHasExits (TheLoop : in Symbol)
   is
   begin
      if IsLoop (TheLoop) then   -- 904
         RawDict.SetLoopHasExits (TheLoop);
      end if;
      -- if it is not a loop then the exit statement must be illegally placed so we do nothing
   end MarkLoopHasExits;

   --------------------------------------------------------------------------------

   procedure SetLoopExitExpn (ForLoop : in Symbol;
                              Expn    : in Natural)
   is
   begin
      RawDict.SetLoopExitExpn (ForLoop, Expn);
   end SetLoopExitExpn;

   --------------------------------------------------------------------------------

   procedure SetLoopEntryExpn (ForLoop : in Symbol;
                               Expn    : in Natural)
   is
   begin
      RawDict.SetLoopEntryExpn (ForLoop, Expn);
   end SetLoopEntryExpn;

   --------------------------------------------------------------------------------

   function LoopVarOnlyAnImport (OriginalVariable : in     Symbol;
                                 TheLoop          : in     Symbol) return Boolean
   --# global in Dict;
   is
      CurrentRegion : Symbol;
      IsOnlyAnImport : Boolean;
   begin
      CurrentRegion := RawDict.GetLoopRegion (TheLoop);
      loop
         exit when IsSubprogram (CurrentRegion) or IsTaskType (CurrentRegion);

         CurrentRegion := RawDict.GetLoopRegion (CurrentRegion);
      end loop;

      if (IsImport (IsAbstract, CurrentRegion, OriginalVariable) or else
          IsImport (IsRefined, CurrentRegion, OriginalVariable)) and
         not (IsExport (IsAbstract, CurrentRegion, OriginalVariable) or else
              IsExport (IsRefined, CurrentRegion, OriginalVariable)) then
         IsOnlyAnImport := True;
      else
         IsOnlyAnImport := False;
      end if;

      return IsOnlyAnImport;
   end LoopVarOnlyAnImport;

   --------------------------------------------------------------------------------

   procedure IdempotentCreateLoopOnEntryVariable (OriginalVariable : in     Symbol;
                                                  TheLoop          : in     Symbol;
                                                  OnEntryVariable  :    out Symbol)
   is
      Found : Boolean := False;
      OnEntryVariableToTry : Symbol;
      NewOnEntryVariable   : Symbol;
   begin
      -- Before creating a new variable, check whether the standard variable
      -- itself is acceptable in this context due to non-modification.
      if LoopVarOnlyAnImport (OriginalVariable, TheLoop) then
         OnEntryVariable := OriginalVariable;
      else

         -- Search linked list of OnEntryVariables associated with TheLoop.  If an entry matching
         -- OriginalVariable is found then current search posiiton; this is the
         -- required OnEntryVariable symbol.  Otherwise, create a new entry and link it in to list.

         OnEntryVariableToTry := RawDict.GetLoopOnEntryVars (TheLoop);
         while OnEntryVariableToTry /= NullSymbol loop
            Found := RawDict.GetLoopEntryVariableOriginalVar (OnEntryVariableToTry) = OriginalVariable;
            exit when Found;

            OnEntryVariableToTry := RawDict.GetLoopEntryVariableNext (OnEntryVariableToTry);
         end loop;

         if Found then
            -- Return existing var
            OnEntryVariable := OnEntryVariableToTry;
         else
            RawDict.CreateLoopEntryVariable (OriginalVar       => OriginalVariable,
                                             TheLoop           => TheLoop,
                                             Comp_Unit         => ContextManager.NullUnit,
                                             Loc               => LexTokenManager.Null_Token_Position,
                                             -- to get
                                             LoopEntryVariable => NewOnEntryVariable);
            -- Insert it at the head of the list
            RawDict.SetLoopEntryVariableNext (NewOnEntryVariable,
                                              RawDict.GetLoopOnEntryVars (TheLoop));
            RawDict.SetLoopOnEntryVars (TheLoop,
                                        NewOnEntryVariable);
            -- Return newly created var
            OnEntryVariable := NewOnEntryVariable;
         end if;

      end if;
   end IdempotentCreateLoopOnEntryVariable;

   --------------------------------------------------------------------------------

   function GetLoopOnEntryVariable (OriginalVariable : Symbol;
                                    TheLoop          : Symbol) return Symbol
   is
      Result : Symbol := NullSymbol; -- Default result for error case only
      OnEntryVariableToTry : Symbol;

   begin
      if LoopVarOnlyAnImport (OriginalVariable, TheLoop) then
         Result := OriginalVariable;
      else
         OnEntryVariableToTry := RawDict.GetLoopOnEntryVars (TheLoop);
         while OnEntryVariableToTry /= NullSymbol loop
            if RawDict.GetLoopEntryVariableOriginalVar (OnEntryVariableToTry) = OriginalVariable then
               Result := OnEntryVariableToTry;
               exit;
            end if;
            OnEntryVariableToTry := RawDict.GetLoopEntryVariableNext (OnEntryVariableToTry);
         end loop;
      end if;
      return Result;
   end GetLoopOnEntryVariable;

   --------------------------------------------------------------------------------

   procedure AddSubtypeDeclaration (Name            : in     LexTokenManager.Lex_String;
                                    Parent          : in     Symbol;
                                    ParentReference : in     Location;
                                    Comp_Unit       : in     ContextManager.UnitDescriptors;
                                    Declaration     : in     Location;
                                    Scope           : in     Scopes;
                                    Context         : in     Contexts;
                                    TheSubtype      :    out Symbol)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from *,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Parent,
   --#                                Scope &
   --#         SPARK_IO.File_Sys from *,
   --#                                Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Parent,
   --#                                ParentReference,
   --#                                Scope &
   --#         TheSubtype        from Comp_Unit,
   --#                                Context,
   --#                                Declaration,
   --#                                Dict,
   --#                                LexTokenManager.State,
   --#                                Name,
   --#                                Scope;
   is
      NewSubtype : Symbol;
      Decl       : Symbol;
      Region     : Symbol;
   begin
      Region := GetRegion (Scope);

      if IsPackage (Region) and then Context = ProgramContext then
         NewSubtype := LookupImmediateScope (Name,
                                             VisibleScope (Region),
                                             ProofContext);
      else
         NewSubtype := NullSymbol;
      end if;

      if NewSubtype = NullSymbol then
         RawDict.CreateType (Name      => Name,
                             Comp_Unit => Comp_Unit,
                             Loc       => Declaration.StartPosition,
                             TypeMark  => NewSubtype);
      elsif RawDict.Get_Symbol_Compilation_Unit (Item => NewSubtype) = Comp_Unit then
         --  There was a usage of the subtype before its actual
         --  declaration -> update the declaration position. The
         --  declaration must be in the same compilation unit as the
         --  usage (in the package specification) to prevent to get a
         --  cross-reference from the package specification to the
         --  package body.
         RawDict.Set_Symbol_Location (Item     => NewSubtype,
                                      Location => Declaration.StartPosition);
      end if;
      AddDeclaration (Item        => NewSubtype,
                      Comp_Unit   => Comp_Unit,
                      Loc         => Declaration,
                      Scope       => Scope,
                      Context     => Context,
                      Declaration => Decl);
      RawDict.SetTypeDeclaration (NewSubtype, Decl);
      AddSubtype (NewSubtype, Parent);

      if not TypeIsUnknown (Parent) then
         AddOtherReference (Parent, GetRegion (Scope), ParentReference);
      end if;

      WriteDeclaration (Discriminant => TypeDeclaration,
                        Item         => NewSubtype,
                        Scope        => Scope,
                        Declaration  => Declaration);

      TheSubtype := NewSubtype;

   end AddSubtypeDeclaration;

   --------------------------------------------------------------------------------

   procedure AddEnumerationSubtype (Name            : in     LexTokenManager.Lex_String;
                                    Static          : in     Boolean;
                                    Parent          : in     Symbol;
                                    ParentReference : in     Location;
                                    Lower           : in     LexTokenManager.Lex_String;
                                    Upper           : in     LexTokenManager.Lex_String;
                                    Comp_Unit       : in     ContextManager.UnitDescriptors;
                                    Declaration     : in     Location;
                                    Scope           : in     Scopes;
                                    Context         : in     Contexts;
                                    Subtype_Symbol  :    out Symbol)
   is
   begin
      AddSubtypeDeclaration (Name            => Name,
                             Parent          => Parent,
                             ParentReference => ParentReference,
                             Comp_Unit       => Comp_Unit,
                             Declaration     => Declaration,
                             Scope           => Scope,
                             Context         => Context,
                             TheSubtype      => Subtype_Symbol);
      AddEnumerationTypeMark (TypeMark => Subtype_Symbol,
                              Static   => Static);
      RawDict.SetTypeLower (TypeMark => Subtype_Symbol,
                            Lower    => Lower);
      RawDict.SetTypeUpper (TypeMark => Subtype_Symbol,
                            Upper    => Upper);
   end AddEnumerationSubtype;

   --------------------------------------------------------------------------------

   procedure AddIntegerSubtype (Name            : in     LexTokenManager.Lex_String;
                                Static          : in     Boolean;
                                Parent          : in     Symbol;
                                ParentReference : in     Location;
                                Lower           : in     LexTokenManager.Lex_String;
                                Upper           : in     LexTokenManager.Lex_String;
                                Comp_Unit       : in     ContextManager.UnitDescriptors;
                                Declaration     : in     Location;
                                Scope           : in     Scopes;
                                Context         : in     Contexts;
                                Subtype_Symbol  :    out Symbol)
   is
   begin
      AddSubtypeDeclaration (Name            => Name,
                             Parent          => Parent,
                             ParentReference => ParentReference,
                             Comp_Unit       => Comp_Unit,
                             Declaration     => Declaration,
                             Scope           => Scope,
                             Context         => Context,
                             TheSubtype      => Subtype_Symbol);
      AddIntegerTypeMark (TypeMark => Subtype_Symbol,
                          Static   => Static,
                          Lower    => Lower,
                          Upper    => Upper);
   end AddIntegerSubtype;

   --------------------------------------------------------------------------------

   procedure AddRecordSubtype (Name            : in     LexTokenManager.Lex_String;
                               Parent          : in     Symbol;
                               ParentReference : in     Location;
                               Comp_Unit       : in     ContextManager.UnitDescriptors;
                               Declaration     : in     Location;
                               Scope           : in     Scopes;
                               Context         : in     Contexts;
                               Subtype_Symbol  :    out Symbol)
   is
   begin
      SystemErrors.RTAssert (not TypeIsTagged (Parent),
                             SystemErrors.PreconditionFailure,
                             "Illegal attempt to add a tagged record subtype.");
      AddSubtypeDeclaration (Name            => Name,
                             Parent          => Parent,
                             ParentReference => ParentReference,
                             Comp_Unit       => Comp_Unit,
                             Declaration     => Declaration,
                             Scope           => Scope,
                             Context         => Context,
                             TheSubtype      => Subtype_Symbol);
      AddRecordTypeMark (TypeMark       => Subtype_Symbol,
                           -- Tagged record subtypes not allowed, so...
                         IsTaggedType   => False,
                         IsAbstractType => False,
                         Extends        => NullSymbol);
   end AddRecordSubtype;

   --------------------------------------------------------------------------------

   procedure AddModularSubtype (Name            : in     LexTokenManager.Lex_String;
                                Parent          : in     Symbol;
                                ParentReference : in     Location;
                                Lower           : in     LexTokenManager.Lex_String;
                                Upper           : in     LexTokenManager.Lex_String;
                                Comp_Unit       : in     ContextManager.UnitDescriptors;
                                Declaration     : in     Location;
                                Scope           : in     Scopes;
                                Context         : in     Contexts;
                                Subtype_Symbol  :    out Symbol)
   is
      Modulus : LexTokenManager.Lex_String;
   begin
      Modulus := GetScalarAttributeValue (False, LexTokenManager.Modulus_Token, Parent);
      AddSubtypeDeclaration (Name            => Name,
                             Parent          => Parent,
                             ParentReference => ParentReference,
                             Comp_Unit       => Comp_Unit,
                             Declaration     => Declaration,
                             Scope           => Scope,
                             Context         => Context,
                             TheSubtype      => Subtype_Symbol);
      AddModularTypeMark (TypeMark => Subtype_Symbol,
                          Static   => True,
                          Lower    => Lower,
                          Upper    => Upper,
                          Modulus  => Modulus);
   end AddModularSubtype;

   --------------------------------------------------------------------------------

   procedure AddFloatingPointSubtype (Name            : in     LexTokenManager.Lex_String;
                                      Static          : in     Boolean;
                                      Parent          : in     Symbol;
                                      ParentReference : in     Location;
                                      Lower           : in     LexTokenManager.Lex_String;
                                      Upper           : in     LexTokenManager.Lex_String;
                                      ErrorBound      : in     LexTokenManager.Lex_String;
                                      Comp_Unit       : in     ContextManager.UnitDescriptors;
                                      Declaration     : in     Location;
                                      Scope           : in     Scopes;
                                      Context         : in     Contexts;
                                      Subtype_Symbol  :    out Symbol)
   is
   begin
      AddSubtypeDeclaration (Name            => Name,
                             Parent          => Parent,
                             ParentReference => ParentReference,
                             Comp_Unit       => Comp_Unit,
                             Declaration     => Declaration,
                             Scope           => Scope,
                             Context         => Context,
                             TheSubtype      => Subtype_Symbol);
      AddFloatingPointTypeMark (TypeMark => Subtype_Symbol,
                                Static   => Static);
      RawDict.SetTypeLower (TypeMark => Subtype_Symbol,
                            Lower    => Lower);
      RawDict.SetTypeUpper (TypeMark => Subtype_Symbol,
                            Upper    => Upper);
      RawDict.SetTypeErrorBound (TypeMark   => Subtype_Symbol,
                                 ErrorBound => ErrorBound);
   end AddFloatingPointSubtype;

   --------------------------------------------------------------------------------

   procedure AddFixedPointSubtype (Name            : in     LexTokenManager.Lex_String;
                                   Static          : in     Boolean;
                                   Parent          : in     Symbol;
                                   ParentReference : in     Location;
                                   Lower           : in     LexTokenManager.Lex_String;
                                   Upper           : in     LexTokenManager.Lex_String;
                                   ErrorBound      : in     LexTokenManager.Lex_String;
                                   Comp_Unit       : in     ContextManager.UnitDescriptors;
                                   Declaration     : in     Location;
                                   Scope           : in     Scopes;
                                   Context         : in     Contexts;
                                   Subtype_Symbol  :    out Symbol)
   is
   begin
      AddSubtypeDeclaration (Name            => Name,
                             Parent          => Parent,
                             ParentReference => ParentReference,
                             Comp_Unit       => Comp_Unit,
                             Declaration     => Declaration,
                             Scope           => Scope,
                             Context         => Context,
                             TheSubtype      => Subtype_Symbol);
      AddFixedPointTypeMark (TypeMark => Subtype_Symbol,
                             Static   => Static);
      RawDict.SetTypeLower (TypeMark => Subtype_Symbol,
                            Lower    => Lower);
      RawDict.SetTypeUpper (TypeMark => Subtype_Symbol,
                            Upper    => Upper);
      RawDict.SetTypeErrorBound (TypeMark   => Subtype_Symbol,
                                 ErrorBound => ErrorBound);
   end AddFixedPointSubtype;

   --------------------------------------------------------------------------------

   procedure AddArraySubtype (Name            : in     LexTokenManager.Lex_String;
                              Parent          : in     Symbol;
                              ParentReference : in     Location;
                              Comp_Unit       : in     ContextManager.UnitDescriptors;
                              Declaration     : in     Location;
                              Scope           : in     Scopes;
                              Context         : in     Contexts;
                              Static          : in     Boolean;
                              ArraySubtype    :    out Symbol)
   is
   begin
      AddSubtypeDeclaration (Name            => Name,
                             Parent          => Parent,
                             ParentReference => ParentReference,
                             Comp_Unit       => Comp_Unit,
                             Declaration     => Declaration,
                             Scope           => Scope,
                             Context         => Context,
                             TheSubtype      => ArraySubtype);
      AddArrayTypeMark (TypeMark      => ArraySubtype,
                        ComponentType => GetArrayComponent (GetParentType (ArraySubtype)),
                        Static        => Static);
      RawDict.SetTypeConstrained (ArrayType   => ArraySubtype,
                                  Constrained => True);
   end AddArraySubtype;

   --------------------------------------------------------------------------------

   procedure WriteProofStatement (Discriminant : in ProofStatementDiscriminant;
                                  Statement    : in Location)
      --# global in     Dict;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                Discriminant,
      --#                                Statement;
      is separate;

   --------------------------------------------------------------------------------

   procedure AddAssertStatement (Statement : in Location)
   is
   begin
      WriteProofStatement (AssertStatement, Statement);
   end AddAssertStatement;

   --------------------------------------------------------------------------------

   procedure AddCheckStatement (Statement : in Location)
   is
   begin
      WriteProofStatement (CheckStatement, Statement);
   end AddCheckStatement;

   --------------------------------------------------------------------------------

   procedure AddPrecondition (Abstraction   : in Abstractions;
                              Subprogram    : in Symbol;
                              Predicate     : in ExaminerConstants.RefType;
                              Precondition  : in Location)
   is
   begin
      -- mark subprog as having second proof constraint (independent of whether it has second flow anno)
      if Abstraction = IsRefined then
         RawDict.SetSubprogramHasSecondConstraint (Subprogram);
      end if;

      RawDict.SetSubprogramPrecondition (Abstraction, Subprogram, Predicate);

      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Abstraction = IsRefined then
            WriteString (Dict.TemporaryFile, "refined ");
         end if;
         WriteString (Dict.TemporaryFile, "precondition of ");
         WriteName (Dict.TemporaryFile, Subprogram);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Precondition);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;

   end AddPrecondition;

   --------------------------------------------------------------------------------

   procedure AddPostcondition (Abstraction     : in Abstractions;
                               CompilationUnit : in Symbol;
                               Predicate       : in ExaminerConstants.RefType;
                               Postcondition   : in Location)
   is
   begin
      -- mark subprog as having second proof constraint (independent of whether it has second flow anno)
      if Abstraction = IsRefined then
         RawDict.SetSubprogramHasSecondConstraint (CompilationUnit);
      end if;

      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when PackageSymbol =>
            RawDict.SetPackagePostcondition (Abstraction,
                                             CompilationUnit,
                                             Predicate);
         when others =>
            RawDict.SetSubprogramPostcondition (Abstraction,
                                                CompilationUnit,
                                                Predicate);
      end case;

      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Abstraction = IsRefined then
            WriteString (Dict.TemporaryFile, "refined ");
         end if;
         WriteString (Dict.TemporaryFile, "postcondition of ");
         WriteName (Dict.TemporaryFile, CompilationUnit);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Postcondition);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;

   end AddPostcondition;

   --------------------------------------------------------------------------------

   procedure AddSubprogram (Name          : in     LexTokenManager.Lex_String;
                            Comp_Unit     : in     ContextManager.UnitDescriptors;
                            Specification : in     Location;
                            Scope         : in     Scopes;
                            Context       : in     Contexts;
                            Subprogram    :    out Symbol)
   is
      NewSubprogram : Symbol;
      Declaration   : Symbol;
   begin
      RawDict.CreateSubprogram (Name       => Name,
                                Comp_Unit  => Comp_Unit,
                                Loc        => Specification.StartPosition,
                                Subprogram => NewSubprogram);
      AddDeclaration (Item        => NewSubprogram,
                      Comp_Unit   => Comp_Unit,
                      Loc         => Specification,
                      Scope       => Scope,
                      Context     => Context,
                      Declaration => Declaration);
      RawDict.SetSubprogramSpecification (NewSubprogram, Declaration);
      WriteDeclaration (Discriminant => SubprogramSpecification,
                        Item         => NewSubprogram,
                        Scope        => Scope,
                        Declaration  => Specification);
      Subprogram := NewSubprogram;
   end AddSubprogram;

   --------------------------------------------------------------------------------

   function ActualOfGenericFormal (TheGenericFormalSym : in Symbol;
                                   ActualSubprogramSym : in Symbol) return Symbol

   is
      AssociationToTry : Symbol;
      ParameterOrConstant : Symbol;
      AssociatedGenericFormalOfConstant : Symbol;

      Result : Symbol := NullSymbol; -- should never be returned
   begin
      ParameterOrConstant := TheGenericFormalSym;
      -- if the generic formal is an object then a constant will have been declared in the visible
      -- scope of the geenric unit and linked to the list of its generic formal parameters.  In many
      -- cases, e.g. in the VCG, the symbol that we will find when looking up a generic formal object parameter
      -- will be the symbol of this implicitly-declared constant.  We want such constants to be correctly identified
      -- as generic formal parameters.  The following test checks for this case before going on to check whether
      -- the parameter supplied is the one we are looking for.
      if RawDict.GetSymbolDiscriminant (ParameterOrConstant) = ConstantSymbol then
         -- if its a constant then it might be an implicitly-declared constant associated with a generic formal
         AssociatedGenericFormalOfConstant := RawDict.GetConstantAssociatedGenericParameter (ParameterOrConstant);
         if AssociatedGenericFormalOfConstant /= NullSymbol then
            -- then it really is an implicitly-declared constant associated with the generic formal
            ParameterOrConstant := AssociatedGenericFormalOfConstant;
         end if;
      end if;

      -- now that we have the actual generic formal sym we can look for it in the linked list of parameters
      AssociationToTry := RawDict.GetSubprogramFirstGenericAssociation (ActualSubprogramSym);
      while AssociationToTry /= NullSymbol loop
         if RawDict.GetGenericAssociationFormal (AssociationToTry) = ParameterOrConstant then
            -- match found
            Result :=  RawDict.GetGenericAssociationActualSymbol (AssociationToTry);
            exit;
         end if;
         AssociationToTry := RawDict.GetGenericAssociationNext (AssociationToTry);
      end loop;
      return Result;
   end ActualOfGenericFormal;

   --------------------------------------------------------------------------------

   -- function ActualParameterOfGenericParameter (TheParameter        : in Symbol;
   --                                             ActualSubprogramSym : in Symbol) return Symbol
   -- moved lower for calling order reasons

   --------------------------------------------------------------------------------

   procedure AddSubprogramInstantiation (Name          : in     LexTokenManager.Lex_String;
                                         Comp_Unit     : in     ContextManager.UnitDescriptors;
                                         Declaration   : in     Location;
                                         TheGeneric    : in     Symbol;
                                         Specification : in     Location;
                                         Scope         : in     Scopes;
                                         Context       : in     Contexts;
                                         Subprogram    :    out Symbol)
   is
      NewSubprogram,
      NewSubprogramDeclaration : Symbol;
      ScopeL : Scopes;
   begin
      AddSubprogram (Name          => Name,
                     Comp_Unit     => Comp_Unit,
                     Specification => Specification,
                     Scope         => Scope,
                     Context       => Context,
                     -- to get
                     Subprogram    => NewSubprogram);
      RawDict.SetSubprogramInstantiationOf (NewSubprogram, TheGeneric);
      -- Mark as having hidden body so Examiner won't demand a body later on

      -- We always add the body to the local scope (because bodies can't exits in visible or
      -- private scope
      ScopeL := Scope;
      if IsVisibleScope (Scope) or else
        IsPrivateScope (Scope) then
         ScopeL := GetLocalScope (Scope);
      end if;

      AddDeclaration (Item        => NewSubprogram,
                      Comp_Unit   => Comp_Unit,
                      Loc         => Declaration,
                      Scope       => ScopeL,
                      Context     => ProgramContext,
                      -- to get
                      Declaration => NewSubprogramDeclaration);
      RawDict.SetSubprogramBody (NewSubprogram, NewSubprogramDeclaration);
      RawDict.SetSubprogramHasProperBody (Subprogram => NewSubprogram,
                                          IsHidden   => True);

      Subprogram := NewSubprogram;
   end AddSubprogramInstantiation;

   --------------------------------------------------------------------------------

   function GetGenericOfInstantiation (PackageOrSubProgram : Symbol) return Symbol
   is
      Result : Symbol;
   begin
      if IsPackage (PackageOrSubProgram) then
         Result := NullSymbol; -- GenericTBD
      elsif IsSubprogram (PackageOrSubProgram) then
         Result := RawDict.GetSubprogramInstantiationOf (PackageOrSubProgram);
      else
         Result := NullSymbol;
      end if;
      return Result;
   end GetGenericOfInstantiation;

   --------------------------------------------------------------------------------

   function IsInstantiation (PackageOrSubProgram : Symbol) return Boolean
   is
   begin
      return GetGenericOfInstantiation (PackageOrSubProgram) /= NullSymbol;
   end IsInstantiation;

   --------------------------------------------------------------------------------
   --procedure InstantiateSubprogramParameters (GenericSubprogramSym : in Symbol;
   --                                           ActualSubprogramSym  : in Symbol)
   --is separate;

   --------------------------------------------------------------------------------

   procedure SetSubprogramSignatureNotWellformed (Abstraction   : in Abstractions;
                                                 Subprogram    : in Symbol)
   is
   begin
      if RawDict.GetSymbolDiscriminant (Subprogram) = SubprogramSymbol then
         RawDict.SetSubprogramSignatureNotWellformed (Abstraction, Subprogram);
      else
         RawDict.SetTaskTypeSignatureNotWellformed (Abstraction, Subprogram);
      end if;
   end SetSubprogramSignatureNotWellformed;

   --------------------------------------------------------------------------------

   procedure SetSubprogramIsEntry (Subprogram    : in Symbol)
   is
   begin
      RawDict.SetSubprogramIsEntry (Subprogram);
   end SetSubprogramIsEntry;

   --------------------------------------------------------------------------------

   procedure SetSubprogramIsGeneric (Subprogram    : in Symbol)
   is
   begin
      RawDict.SetSubprogramIsGeneric (Subprogram);
   end SetSubprogramIsGeneric;

   --------------------------------------------------------------------------------

   procedure SetSubprogramEntryBarrier (Subprogram, TheBarrier  : in Symbol)
   is
   begin
      RawDict.SetSubprogramEntryBarrier (Subprogram, TheBarrier);
   end SetSubprogramEntryBarrier;

   --------------------------------------------------------------------------------

   procedure SetIsInterruptHandler (Subprogram    : in Symbol)
   is
   begin
      RawDict.SetIsInterruptHandler (Subprogram);
   end SetIsInterruptHandler;

   --------------------------------------------------------------------------------

   procedure SetProtectedTypeElementsHidden (TheProtectedType : in Symbol)
   is
   begin
      RawDict.SetProtectedTypeElementsHidden (TheProtectedType);
   end SetProtectedTypeElementsHidden;

   --------------------------------------------------------------------------------

   procedure SetProtectedTypeEntry (TheProtectedType, TheEntry : in Symbol)
   is
   begin
      RawDict.SetProtectedTypeTheEntry (TheProtectedType, TheEntry);
   end SetProtectedTypeEntry;

   --------------------------------------------------------------------------------

   procedure SetTypeHasPragma (TheProtectedOrTaskType : in Symbol;
                               ThePragma              : in RavenscarPragmas)
   is
   begin
      if RawDict.GetTypeDiscriminant (TheProtectedOrTaskType) = ProtectedType then
         RawDict.SetProtectedTypeHasPragma (TheProtectedOrTaskType, ThePragma);
      else
         RawDict.SetTaskTypeHasPragma (TheProtectedOrTaskType, ThePragma);
      end if;
   end SetTypeHasPragma;

   --------------------------------------------------------------------------------

   procedure SetTypePragmaValue (TheProtectedOrTaskType : in Symbol;
                                 ThePragma              : in RavenscarPragmasWithValue;
                                 TheValue               : in LexTokenManager.Lex_String)
   is
   begin
      if RawDict.GetTypeDiscriminant (TheProtectedOrTaskType) = ProtectedType then
         RawDict.SetProtectedTypePragmaValue (TheProtectedOrTaskType, ThePragma, TheValue);
      else
         RawDict.SetTaskTypePragmaValue (TheProtectedOrTaskType, ThePragma, TheValue);
      end if;
   end SetTypePragmaValue;

   --------------------------------------------------------------------------------

   procedure SetMainProgramPriority (TheValue : in LexTokenManager.Lex_String)
   is
   begin
      Dict.Main.PriorityGiven := True;
      Dict.Main.ThePriority := TheValue;
      null;
   end SetMainProgramPriority;

   --------------------------------------------------------------------------------

   function IsPredefinedSuspendUntilTrueOperation (TheProcedure : Symbol) return Boolean
   is
   begin
      return TheProcedure = Dict.Subprograms.STC_SuspendUntilTrue;
   end IsPredefinedSuspendUntilTrueOperation;

   --------------------------------------------------------------------------------

   function IsPredefinedRealTimeClockOperation (TheProcedure : Symbol) return Boolean
   is
   begin
      return TheProcedure = Dict.Subprograms.AdaRealTimeClock;
   end IsPredefinedRealTimeClockOperation;

   --------------------------------------------------------------------------------

   procedure AddReturnType (TheFunction   : in Symbol;
                            TypeMark      : in Symbol;
                            Comp_Unit     : in ContextManager.UnitDescriptors;
                            TypeReference : in Location)
   is
      ProofFunction : Symbol;
   begin

      RawDict.SetSubprogramReturnType (TheFunction, TypeMark);

      if GetContext (TheFunction) = ProgramContext then
         RawDict.CreateImplicitProofFunction (AdaFunction   => TheFunction,
                                              Comp_Unit     => Comp_Unit,
                                              Loc           => TypeReference.StartPosition,
                                              ProofFunction => ProofFunction);
         RawDict.SetSubprogramImplicitProofFunction (IsAbstract,
                                                     TheFunction,
                                                     ProofFunction);

      end if;

      if not TypeIsUnknown (TypeMark) then
         AddOtherReference (TypeMark, TheFunction, TypeReference);
      end if;

   end AddReturnType;

   -------------------------------------------------------------------------------

   procedure AddImplicitReturnVariable (Abstraction : in     Abstractions;
                                        Comp_Unit   : in     ContextManager.UnitDescriptors;
                                        Declaration : in     Location;
                                        Name        : in     LexTokenManager.Lex_String;
                                        TheFunction : in     Symbol;
                                        Variable    :    out Symbol)
   is
      ReturnVariable : Symbol;
   begin
      RawDict.CreateImplicitReturnVariable (Name        => Name,
                                            TheFunction => TheFunction,
                                            Comp_Unit   => Comp_Unit,
                                            Loc         => Declaration.StartPosition,
                                            Variable    => ReturnVariable);
      RawDict.SetSubprogramImplicitReturnVariable (Abstraction,
                                                   TheFunction,
                                                   ReturnVariable);
      Variable := ReturnVariable;
   end AddImplicitReturnVariable;

   --------------------------------------------------------------------------------

   procedure AddSubprogramParameter (Name          : in LexTokenManager.Lex_String;
                                     Subprogram    : in Symbol;
                                     TypeMark      : in Symbol;
                                     TypeReference : in Location;
                                     Mode          : in Modes;
                                     Comp_Unit     : in ContextManager.UnitDescriptors;
                                     Specification : in Location)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddGenericFormalParameter (Name                : in LexTokenManager.Lex_String;
                                        Comp_Unit           : in ContextManager.UnitDescriptors;
                                        Declaration         : in Location;
                                        SubprogramOrPackage : in Symbol;
                                        TypeMark            : in Symbol;
                                        Kind                : in GenericParameterKind)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddGenericAssociation (SubprogramOrPackage : in Symbol;
                                    Comp_Unit           : in ContextManager.UnitDescriptors;
                                    Declaration         : in Location;
                                    FormalSym           : in Symbol;
                                    ActualSym           : in Symbol)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddKnownDiscriminant (Name                : in LexTokenManager.Lex_String;
                                   Comp_Unit           : in ContextManager.UnitDescriptors;
                                   Declaration         : in Location;
                                   ProtectedOrTaskType : in Symbol;
                                   TypeMark            : in Symbol)
   is
      Discriminant : Symbol;
      Previous     : Symbol;
   begin
      RawDict.CreateKnownDiscriminant (Name              => Name,
                                       ProtectedType     => ProtectedOrTaskType,
                                       TypeMark          => TypeMark,
                                       Comp_Unit         => Comp_Unit,
                                       Loc               => Declaration.StartPosition,
                                       KnownDiscriminant => Discriminant);
      if RawDict.GetTypeDiscriminant (ProtectedOrTaskType) = ProtectedType then
         Previous := RawDict.GetProtectedTypeLastDiscriminant (ProtectedOrTaskType);
         if Previous = NullSymbol then
            RawDict.SetProtectedTypeFirstDiscriminant (ProtectedOrTaskType, Discriminant);
         else
            RawDict.SetNextDiscriminant (Previous, Discriminant);
         end if;
         RawDict.SetProtectedTypeLastDiscriminant (ProtectedOrTaskType, Discriminant);
      else
         Previous := RawDict.GetTaskTypeLastDiscriminant (ProtectedOrTaskType);
         if Previous = NullSymbol then
            RawDict.SetTaskTypeFirstDiscriminant (ProtectedOrTaskType, Discriminant);
         else
            RawDict.SetNextDiscriminant (Previous, Discriminant);
         end if;
         RawDict.SetTaskTypeLastDiscriminant (ProtectedOrTaskType, Discriminant);
      end if;
   end AddKnownDiscriminant;

   --------------------------------------------------------------------------------

   procedure SetDiscriminantSetsPriority (TheDiscriminant : in Symbol)
   is
   begin
      RawDict.SetDiscriminantSetsPriority (TheDiscriminant);
   end SetDiscriminantSetsPriority;

   --------------------------------------------------------------------------------

   procedure AddDiscriminantConstraintStaticValue (ProtectedOrTaskSubtype : in Symbol;
                                                   Comp_Unit              : in ContextManager.UnitDescriptors;
                                                   Declaration            : in Location;
                                                   TheValue               : in LexTokenManager.Lex_String)
   is
      DiscriminantConstraint : Symbol;
      SubtypeInfo            : Symbol;
      Previous               : Symbol;
   begin
      -- create DiscrimianntConstraint record
      RawDict.CreateDiscriminantConstraint (Comp_Unit                 => Comp_Unit,
                                            Loc                       => Declaration.StartPosition,
                                            TheDiscriminantConstraint => DiscriminantConstraint);
      RawDict.SetDiscriminantConstraintStaticValue (DiscriminantConstraint, TheValue);
      -- link to subtype
      SubtypeInfo := RawDict.GetTypeAncilliaryFields (ProtectedOrTaskSubtype);
      -- assuming precondition met then SubtypeInfo is symbol of record containing priority, linked
      -- list of constraints etc.
      Previous := RawDict.GetSubtypeInfoLastConstraint (SubtypeInfo);
      if Previous = NullSymbol then
         RawDict.SetSubtypeInfoFirstConstraint (SubtypeInfo, DiscriminantConstraint);
      else
         RawDict.SetNextDiscriminantConstraint (Previous, DiscriminantConstraint);
      end if;
      RawDict.SetSubtypeInfoLastConstraint (SubtypeInfo, DiscriminantConstraint);
   end AddDiscriminantConstraintStaticValue;

   --------------------------------------------------------------------------------

   procedure AddDiscriminantConstraintAccessedObject (ProtectedOrTaskSubtype : in Symbol;
                                                      Comp_Unit              : in ContextManager.UnitDescriptors;
                                                      Declaration            : in Location;
                                                      TheObject              : in Symbol)
   is
      DiscriminantConstraint : Symbol;
      SubtypeInfo            : Symbol;
      Previous               : Symbol;
   begin
      -- create DiscrimianntConstraint record
      RawDict.CreateDiscriminantConstraint (Comp_Unit                 => Comp_Unit,
                                            Loc                       => Declaration.StartPosition,
                                            TheDiscriminantConstraint => DiscriminantConstraint);
      RawDict.SetDiscriminantConstraintAccessedObject (DiscriminantConstraint, TheObject);
      -- link to subtype
      SubtypeInfo := RawDict.GetTypeAncilliaryFields (ProtectedOrTaskSubtype);
      -- assuming precondition met then SubtypeInfo is symbol of record containing priority, linked
      -- list of constraints etc.
      Previous := RawDict.GetSubtypeInfoLastConstraint (SubtypeInfo);
      if Previous = NullSymbol then
         RawDict.SetSubtypeInfoFirstConstraint (SubtypeInfo, DiscriminantConstraint);
      else
         RawDict.SetNextDiscriminantConstraint (Previous, DiscriminantConstraint);
      end if;
      RawDict.SetSubtypeInfoLastConstraint (SubtypeInfo, DiscriminantConstraint);
   end AddDiscriminantConstraintAccessedObject;

   --------------------------------------------------------------------------------

   procedure SetSubtypePriority (ProtectedOrTaskSubtype : in Symbol;
                                 ThePriority            : in LexTokenManager.Lex_String)
   is
      SubtypeInfo            : Symbol;
   begin
      SubtypeInfo := RawDict.GetTypeAncilliaryFields (ProtectedOrTaskSubtype);
      RawDict.SetSubtypeInfoPriority (SubtypeInfo,
                                      ThePriority);

   end SetSubtypePriority;

   --------------------------------------------------------------------------------

   function HasBodyStub (CompilationUnit : Symbol) return Boolean
   is
      Result : Boolean;
   begin

      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when PackageSymbol =>
            Result := RawDict.GetPackageBody (CompilationUnit) /= NullSymbol and then
              not RawDict.GetPackageHasProperBody (CompilationUnit);
         when TypeSymbol =>
            if RawDict.GetTypeDiscriminant (CompilationUnit) = ProtectedType then
               Result := RawDict.GetProtectedTypeBody (CompilationUnit) /= NullSymbol and then
                 not RawDict.GetProtectedTypeHasProperBody (CompilationUnit);
            elsif RawDict.GetTypeDiscriminant (CompilationUnit) = TaskType then
               Result := RawDict.GetTaskTypeBody (CompilationUnit) /= NullSymbol and then
                 not RawDict.GetTaskTypeHasProperBody (CompilationUnit);
            else
               Result := False;
            end if;
         when  SubprogramSymbol =>
            Result := RawDict.GetSubprogramBody (CompilationUnit) /= NullSymbol and then
              not RawDict.GetSubprogramHasProperBody (CompilationUnit);
         when others =>
            Result := False;
      end case;

      return Result;

   end HasBodyStub;

   --------------------------------------------------------------------------------

   procedure AddBody (CompilationUnit : in Symbol;
                      Comp_Unit       : in ContextManager.UnitDescriptors;
                      TheBody         : in Location;
                      Hidden          : in Boolean)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddBodyStub (CompilationUnit : in Symbol;
                          Comp_Unit       : in ContextManager.UnitDescriptors;
                          BodyStub        : in Location)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddMainProgram (Subprogram : in Symbol;
                             Annotation : in Location)
   is
   begin
      Dict.Main.Subprogram := Subprogram;
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile, "main program annotation is at ");
         WriteLocation (Dict.TemporaryFile, Annotation);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddMainProgram;

   --------------------------------------------------------------------------------

   procedure AddDependencyRelation (Abstraction        : in Abstractions;
                                    TheProcedure       : in Symbol;
                                    DependencyRelation : in Location)
   is
   begin
      if Abstraction = IsRefined then
         if RawDict.GetSymbolDiscriminant (TheProcedure) = SubprogramSymbol then
            RawDict.SetSubprogramHasSecondAnnotation (TheProcedure);
         else
            -- must be task body
            RawDict.SetTaskTypeHasSecondAnnotation (TheProcedure);
         end if;
      end if;
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Abstraction = IsRefined then
            WriteString (Dict.TemporaryFile, "refined ");
         end if;
         WriteString (Dict.TemporaryFile, "dependency relation of ");
         WriteName (Dict.TemporaryFile, TheProcedure);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, DependencyRelation);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddDependencyRelation;

   --------------------------------------------------------------------------------

   procedure AddRenamingDeclaration (Subprogram  : in Symbol;
                                     Comp_Unit   : in ContextManager.UnitDescriptors;
                                     Declaration : in Location;
                                     Scope       : in Scopes)
   --# global in out Dict;
   --# derives Dict from *,
   --#                   Comp_Unit,
   --#                   Declaration,
   --#                   Scope,
   --#                   Subprogram;
      is separate;

   --------------------------------------------------------------------------------

   procedure RenameSubprogram (Subprogram          : in Symbol;
                               SubprogramReference : in Location;
                               Comp_Unit           : in ContextManager.UnitDescriptors;
                               Declaration         : in Location;
                               Scope               : in Scopes)
   is
   begin
      AddRenamingDeclaration (Subprogram  => Subprogram,
                              Comp_Unit   => Comp_Unit,
                              Declaration => Declaration,
                              Scope       => Scope);
      WriteDeclaration (Discriminant => RenamingDeclaration,
                        Item         => Subprogram,
                        Scope        => Scope,
                        Declaration  => Declaration);
      AddOtherReference (Subprogram, GetRegion (Scope), SubprogramReference);
   end RenameSubprogram;

   --------------------------------------------------------------------------------

   procedure WriteOperatorRenamingDeclaration (Operator : in Symbol;
                                               Scope    : in Scopes)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                LexTokenManager.State,
      --#                                Operator,
      --#                                Scope;
      is separate;

   --------------------------------------------------------------------------------

   procedure RenameUnaryOperator (Name        : in SPSymbols.SPSymbol;
                                  Comp_Unit   : in ContextManager.UnitDescriptors;
                                  Declaration : in Location;
                                  Operand     : in Symbol;
                                  Scope       : in Scopes)
   is
      Operator : Symbol;
   begin
      RawDict.CreateUnaryOperator (Name      => Name,
                                   Operand   => Operand,
                                   Comp_Unit => Comp_Unit,
                                   Loc       => Declaration.StartPosition,
                                   Operator  => Operator);
      AddRenamingDeclaration (Subprogram  => Operator,
                              Comp_Unit   => Comp_Unit,
                              Declaration => Declaration,
                              Scope       => Scope);
      WriteOperatorRenamingDeclaration (Operator, Scope);
   end RenameUnaryOperator;

   --------------------------------------------------------------------------------

   procedure RenameBinaryOperator (Name        : in SPSymbols.SPSymbol;
                                   Comp_Unit   : in ContextManager.UnitDescriptors;
                                   Declaration : in Location;
                                   Left        : in Symbol;
                                   Right       : in Symbol;
                                   Scope       : in Scopes)
   is
      Operator : Symbol;
   begin
      RawDict.CreateBinaryOperator (Name      => Name,
                                    Left      => Left,
                                    Right     => Right,
                                    Comp_Unit => Comp_Unit,
                                    Loc       => Declaration.StartPosition,
                                    Operator  => Operator);
      AddRenamingDeclaration (Subprogram  => Operator,
                              Comp_Unit   => Comp_Unit,
                              Declaration => Declaration,
                              Scope       => Scope);
      WriteOperatorRenamingDeclaration (Operator, Scope);
   end RenameBinaryOperator;

   --------------------------------------------------------------------------------

   procedure AddGlobalAnnotation (Abstraction : in Abstractions;
                                  Subprogram  : in Symbol;
                                  Comp_Unit   : in ContextManager.UnitDescriptors;
                                  Annotation  : in Location)
   is
      ProofFunction : Symbol;
   begin

      if Abstraction = IsRefined and then
        IsSubprogram (Subprogram) and then
        not RawDict.GetSubprogramHasSecondAnnotation (Subprogram) then
         RawDict.SetSubprogramHasSecondAnnotation (Subprogram);
         if IsFunction (Subprogram) and then
            GetContext (Subprogram) = ProgramContext then
            RawDict.CreateImplicitProofFunction (AdaFunction   => Subprogram,
                                                 Comp_Unit     => Comp_Unit,
                                                 Loc           => Annotation.StartPosition,
                                                 ProofFunction => ProofFunction);
            RawDict.SetSubprogramImplicitProofFunction (IsRefined,
                                                        Subprogram,
                                                        ProofFunction);
         end if;
      end if;

      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Abstraction = IsRefined then
            WriteString (Dict.TemporaryFile, "refined ");
         end if;
         WriteString (Dict.TemporaryFile, "global annotation of ");
         WriteName (Dict.TemporaryFile, Subprogram);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Annotation);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;

   end AddGlobalAnnotation;

   --------------------------------------------------------------------------------

   procedure AddGlobalVariable (Abstraction       : in Abstractions;
                                Subprogram        : in Symbol;
                                Variable          : in Symbol;
                                Mode              : in Modes;
                                PrefixNeeded      : in Boolean;
                                Comp_Unit         : in ContextManager.UnitDescriptors;
                                VariableReference : in Location)
   is
      Previous, Next : Symbol;
   begin

      case RawDict.GetSymbolDiscriminant (Variable) is
         when SubprogramParameterSymbol =>
            Previous := RawDict.GetSubprogramParameterGlobalReferences (Abstraction,
                                                                        Variable);
         when others =>
            Previous := RawDict.GetVariableGlobalReferences (Abstraction, Variable);
      end case;

      RawDict.CreateGlobalVariable (Subprogram     => Subprogram,
                                    Variable       => Variable,
                                    Mode           => Mode,
                                    PrefixNeeded   => PrefixNeeded,
                                    NextSubprogram => Previous,
                                    Comp_Unit      => Comp_Unit,
                                    Loc            => VariableReference.StartPosition,
                                    GlobalVariable => Next);

      case RawDict.GetSymbolDiscriminant (Variable) is
         when SubprogramParameterSymbol =>
            RawDict.SetSubprogramParameterGlobalReferences (Abstraction,
                                                            Variable,
                                                            Next);
         when others =>
            RawDict.SetVariableGlobalReferences (Abstraction, Variable, Next);
      end case;


      if RawDict.GetSymbolDiscriminant (Subprogram) = SubprogramSymbol then
         if RawDict.GetSubprogramFirstGlobalVariable (Abstraction,
                                                      Subprogram) = NullSymbol then
            RawDict.SetSubprogramFirstGlobalVariable (Abstraction, Subprogram, Next);
         else
            RawDict.SetGlobalVariableNextVariable (RawDict.GetSubprogramLastGlobalVariable (Abstraction,
                                                                                            Subprogram),
                                                   Next);
         end if;
         RawDict.SetSubprogramLastGlobalVariable (Abstraction, Subprogram, Next);
      else -- must be task type
         if RawDict.GetTaskTypeFirstGlobalVariable (Abstraction,
                                                    Subprogram) = NullSymbol then
            RawDict.SetTaskTypeFirstGlobalVariable (Abstraction, Subprogram, Next);
         else
            RawDict.SetGlobalVariableNextVariable (RawDict.GetTaskTypeLastGlobalVariable (Abstraction,
                                                                                          Subprogram),
                                                   Next);
         end if;
         RawDict.SetTaskTypeLastGlobalVariable (Abstraction, Subprogram, Next);
      end if;
      AddOtherReference (Variable, Subprogram, VariableReference);

   end AddGlobalVariable;

   --------------------------------------------------------------------------------

   function GetGlobalVariable (Abstraction : Abstractions;
                               Subprogram  : Symbol;
                               Variable    : Symbol) return Symbol
      --# global in Dict;
   is
      GlobalVariable : Symbol;
   begin

      if IsSubprogramParameter (Variable) then
         GlobalVariable := RawDict.GetSubprogramParameterGlobalReferences (Abstraction, Variable);
      else
         GlobalVariable := RawDict.GetVariableGlobalReferences (Abstraction, Variable);
      end if;

      loop
         exit when RawDict.GetGlobalVariableSubprogram (GlobalVariable) = Subprogram;
         GlobalVariable := RawDict.GetGlobalVariableNextSubprogram (GlobalVariable);
      end loop;

      return GlobalVariable;

   end GetGlobalVariable;

   --------------------------------------------------------------------------------

   function GetGlobalMode (Abstraction : Abstractions;
                           Subprogram  : Symbol;
                           Variable    : Symbol) return Modes
   is
   begin
      return RawDict.GetGlobalMode (GetGlobalVariable (Abstraction,
                                                       Subprogram,
                                                       Variable));
   end GetGlobalMode;

   --------------------------------------------------------------------------------

   function GetOwnVariableMode (Variable : Symbol) return Modes
   is
   begin
      SystemErrors.RTAssert (IsOwnVariable (Variable),
                             SystemErrors.PreconditionFailure,
                             "In call to GetOwnVariableMode");
      return RawDict.GetOwnVariableMode (RawDict.GetVariableOwnVariable (Variable));
   end GetOwnVariableMode;

   --------------------------------------------------------------------------------

   function GetOwnVariableProtected (Variable : Symbol) return Boolean
   is
   begin
      SystemErrors.RTAssert (IsOwnVariable (Variable),
                             SystemErrors.PreconditionFailure,
                             "In call to GetOwnVariableProtected");
      return RawDict.GetOwnVariableProtected (RawDict.GetVariableOwnVariable (Variable));
   end GetOwnVariableProtected;

   --------------------------------------------------------------------------------

   function GetOwnVariableIsInterruptStream (Variable : Symbol) return Boolean
   is
   begin
      SystemErrors.RTAssert (IsOwnVariable (Variable),
                             SystemErrors.PreconditionFailure,
                             "In call to GetOwnVariableIsInterruptStream");
      return RawDict.GetOwnVariableIsInterruptStream (RawDict.GetVariableOwnVariable (Variable));
   end GetOwnVariableIsInterruptStream;

   --------------------------------------------------------------------------------

   function GetConstituentMode (Variable : Symbol) return Modes
   is
   begin
      return RawDict.GetConstituentMode (RawDict.GetVariableConstituent (Variable));
   end GetConstituentMode;

   --------------------------------------------------------------------------------

   procedure AddExport (Abstraction     : in Abstractions;
                        TheProcedure    : in Symbol;
                        TheExport       : in Symbol;
                        ExportReference : in Location;
                        Annotation      : in Location)
   is
   begin

      if IsFormalParameter (TheProcedure, TheExport) then
         RawDict.SetSubprogramParameterExported (Abstraction, TheExport);
      else
         RawDict.SetGlobalVariableExported (GetGlobalVariable (Abstraction,
                                                               TheProcedure,
                                                               TheExport));
      end if;

      AddOtherReference (TheExport, TheProcedure, ExportReference);

      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Abstraction = IsRefined then
            WriteString (Dict.TemporaryFile, "refined ");
         end if;
         WriteString (Dict.TemporaryFile, "export of ");
         WriteName (Dict.TemporaryFile, TheExport);
         WriteString (Dict.TemporaryFile, " from ");
         WriteName (Dict.TemporaryFile, TheProcedure);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Annotation);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;

   end AddExport;

   --------------------------------------------------------------------------------

   procedure ForceImport (Abstraction     : in Abstractions;
                          TheProcedure    : in Symbol;
                          TheImport       : in Symbol;
                          ImportReference : in Location)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddDependency (Abstraction     : in Abstractions;
                            Comp_Unit       : in ContextManager.UnitDescriptors;
                            TheProcedure    : in Symbol;
                            TheExport       : in Symbol;
                            TheImport       : in Symbol;
                            ImportReference : in Location)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddVirtualElement (ToProtectedType   : in Symbol;
                                Comp_Unit         : in ContextManager.UnitDescriptors;
                                Declaration       : in Location;
                                TheVirtualElement : in Symbol;
                                TheOwner          : in Symbol)
   is
      NewVirtualElement : Symbol;
   begin
      RawDict.CreateVirtualElement (TheVariable       => TheVirtualElement,
                                    TheOwner          => TheOwner,
                                    Comp_Unit         => Comp_Unit,
                                    Loc               => Declaration.StartPosition,
                                    TheVirtualElement => NewVirtualElement);
      RawDict.SetNextVirtualElement (NewVirtualElement,
                                     RawDict.GetVirtualElementList (ToProtectedType));
      RawDict.SetVirtualElementList (ToProtectedType, NewVirtualElement);
      RawDict.SetVariableVirtualElement (Variable => TheVirtualElement,
                                         VirtualElement => NewVirtualElement);
   end AddVirtualElement;

   --------------------------------------------------------------------------------

   procedure AddPOorSOToSuspendsList (TheTaskOrProc : in Symbol;
                                      Comp_Unit     : in ContextManager.UnitDescriptors;
                                      Declaration   : in Location;
                                      ThePOorSO     : in Symbol)
   is
      SuspendsListItem : Symbol;
   begin
      RawDict.CreateSuspendsListItem (ThePOorSO        => ThePOorSO,
                                      Comp_Unit        => Comp_Unit,
                                      Loc              => Declaration.StartPosition,
                                      SuspendsListItem => SuspendsListItem);
      if IsSubprogram (TheTaskOrProc) then
         RawDict.SetNextSuspendsListItem (SuspendsListItem,
                                          RawDict.GetProcedureSuspendsList (TheTaskOrProc));
         RawDict.SetProcedureSuspendsList (TheTaskOrProc, SuspendsListItem);
      else
         RawDict.SetNextSuspendsListItem (SuspendsListItem,
                                          RawDict.GetTaskTypeSuspendsList (TheTaskOrProc));
         RawDict.SetTaskTypeSuspendsList (TheTaskOrProc, SuspendsListItem);
      end if;
   end AddPOorSOToSuspendsList;

   --------------------------------------------------------------------------------

   procedure AddInterruptStreamMapping
     (Subject             : in Symbol;
      Comp_Unit           : in ContextManager.UnitDescriptors;
      Declaration         : in Location;
      TheInterruptHandler : in LexTokenManager.Lex_String;
      TheInterruptStream  : in LexTokenManager.Lex_String)
   is
      NewSym             : Symbol;
      SubjectOwnVariable : Symbol;
   begin
      SubjectOwnVariable := RawDict.GetVariableOwnVariable (Subject);

      RawDict.CreateInterruptStreamMapping
        (TheHandler         => TheInterruptHandler,
         TheInterruptStream => TheInterruptStream,
         Comp_Unit          => Comp_Unit,
         Loc                => Declaration.StartPosition,
         TheMapping         => NewSym);

      RawDict.SetNextInterruptStreamMapping
        (Current => NewSym,
         Next    => RawDict.GetOwnVariableInterruptStreamMappings (SubjectOwnVariable));

      RawDict.SetOwnVariableInterruptStreamMappings
        (TheOwnVariable => SubjectOwnVariable,
         TheInterruptStreamMappings => NewSym);
   end AddInterruptStreamMapping;

   --------------------------------------------------------------------------------

   procedure CopyDependencyList (Abstraction  : in Abstractions;
                                 TheProcedure : in Symbol;
                                 FromExport   : in Symbol;
                                 ToExport     : in Symbol)
   is
      Dependency : Symbol;
   begin

      if IsFormalParameter (TheProcedure, FromExport) then
         Dependency := RawDict.GetSubprogramParameterDependencies (Abstraction,
                                                                   FromExport);
      else
         Dependency := RawDict.GetGlobalVariableDependencies (Abstraction,
                                                              GetGlobalVariable (Abstraction,
                                                                                 TheProcedure,
                                                                                 FromExport));
      end if;

      if IsFormalParameter (TheProcedure, ToExport) then
         RawDict.SetSubprogramParameterDependencies (Abstraction,
                                                     ToExport,
                                                     Dependency);
      else
         RawDict.SetGlobalVariableDependencies (Abstraction,
                                                GetGlobalVariable (Abstraction,
                                                                   TheProcedure,
                                                                   ToExport),
                                                Dependency);
      end if;

   end CopyDependencyList;

   --------------------------------------------------------------------------------

   procedure AddNewPackage (Name        : in     LexTokenManager.Lex_String;
                            Comp_Unit   : in     ContextManager.UnitDescriptors;
                            Declaration : in     Location;
                            Scope       : in     Scopes;
                            TheParent   : in     Symbol;
                            ChildSort   : in     PackageSort;
                            ThePackage  :    out Symbol)
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --# derives Dict       from *,
   --#                         ChildSort,
   --#                         Comp_Unit,
   --#                         Declaration,
   --#                         LexTokenManager.State,
   --#                         Name,
   --#                         Scope,
   --#                         TheParent &
   --#         ThePackage from Comp_Unit,
   --#                         Declaration,
   --#                         Dict,
   --#                         LexTokenManager.State,
   --#                         Name,
   --#                         Scope,
   --#                         TheParent;
   is
      NewPackage, TheSpecification : Symbol;

      procedure MakeChild (TheParent,
                             TheChild   : in     Symbol;
                          ChildSort  : in     PackageSort)
         --# global in out Dict;
         --# derives Dict from *,
         --#                   ChildSort,
         --#                   TheChild,
         --#                   TheParent;
      is
      begin
         RawDict.SetPackageParent (TheChild, TheParent);
         if RawDict.GetPackageFirstChild (TheParent, ChildSort) = NullSymbol then
            -- Adding first child
            RawDict.SetPackageFirstChild (TheParent, TheChild, ChildSort);
            RawDict.SetPackageLastChild (TheParent, TheChild, ChildSort);
         else
            -- Adding a sibling to an existing child
            RawDict.SetPackageSibling (RawDict.GetPackageLastChild (TheParent, ChildSort), TheChild);
            RawDict.SetPackageLastChild (TheParent, TheChild, ChildSort);
         end if;
      end MakeChild;

   begin --AddNewPackage

      if TheParent = NullSymbol and then
         IsLocalScope (Scope) then
         NewPackage := LookupImmediateScope (Name, Scope, ProofContext);
      else
         NewPackage := NullSymbol;
      end if;

      if NewPackage = NullSymbol then
         RawDict.CreatePackage (Name       => Name,
                                Comp_Unit  => Comp_Unit,
                                Loc        => Declaration.StartPosition,
                                ThePackage => NewPackage);
         if TheParent = NullSymbol then
            AddDeclaration (Item        => NewPackage,
                            Comp_Unit   => Comp_Unit,
                            Loc         => Declaration,
                            Scope       => Scope,
                            Context     => ProgramContext,
                            Declaration => TheSpecification);
         else
            RawDict.CreateDeclaration (Item        => NewPackage,
                                       Context     => ProgramContext,
                                       Comp_Unit   => Comp_Unit,
                                       Loc         => Declaration.StartPosition,
                                       Declaration => TheSpecification);
            RawDict.SetDeclarationScope (TheSpecification, Scope);
         end if;
         RawDict.SetPackageSpecification (NewPackage, TheSpecification);
      else
         RawDict.SetDeclarationContext (RawDict.GetPackageSpecification (NewPackage),
                                        ProgramContext);
         if RawDict.Get_Symbol_Compilation_Unit (Item => NewPackage) = Comp_Unit then
            --  There was a usage of the package before its actual
            --  declaration -> update the declaration position. The
            --  declaration must be in the same compilation unit as
            --  the usage (in the package specification) to prevent to
            --  get a cross-reference from the package specification
            --  to the package body.
            RawDict.Set_Symbol_Location (Item     => NewPackage,
                                         Location => Declaration.StartPosition);
         end if;
      end if;

      if TheParent /= NullSymbol then
         MakeChild (TheParent, NewPackage, ChildSort);
      end if;

      if ChildSort = PrivateChild then
         RawDict.SetPackageIsPrivate (NewPackage);
      end if;

      ThePackage := NewPackage;

   end AddNewPackage;

   --------------------------------------------------------------------------------

   procedure AddPackage (Name          : in     LexTokenManager.Lex_String;
                         Comp_Unit     : in     ContextManager.UnitDescriptors;
                         Specification : in     Location;
                         Scope         : in     Scopes;
                         ThePackage    :    out Symbol)
   is
      NewPackage : Symbol;
   begin
      AddNewPackage (Name        => Name,
                     Comp_Unit   => Comp_Unit,
                     Declaration => Specification,
                     Scope       => Scope,
                     TheParent   => NullSymbol,
                     ChildSort   => Public,
                     --to get
                     ThePackage  => NewPackage);
      WriteDeclaration (Discriminant => PackageSpecification,
                        Item         => NewPackage,
                        Scope        => Scope,
                        Declaration  => Specification);
      ThePackage := NewPackage;
   end AddPackage;

   --------------------------------------------------------------------------------

   procedure AddChildPackage (TheParent     : in     Symbol;
                              Sort          : in     PackageSort;
                              Name          : in     LexTokenManager.Lex_String;
                              Comp_Unit     : in     ContextManager.UnitDescriptors;
                              Specification : in     Location;
                              Scope         : in     Scopes;
                              ThePackage    :    out Symbol)
   is
      NewPackage : Symbol;
   begin
      AddNewPackage (Name        => Name,
                     Comp_Unit   => Comp_Unit,
                     Declaration => Specification,
                     Scope       => Scope,
                     TheParent   => TheParent,
                     ChildSort   => Sort,
                     --to get
                     ThePackage  => NewPackage);
      WriteDeclaration (Discriminant => PackageSpecification,
                        Item         => NewPackage,
                        Scope        => Scope,
                        Declaration  => Specification);
      ThePackage := NewPackage;
   end AddChildPackage;

   --------------------------------------------------------------------------------

   procedure AddPrivatePackage (Name          : in     LexTokenManager.Lex_String;
                                Comp_Unit     : in     ContextManager.UnitDescriptors;
                                Specification : in     Location;
                                Scope         : in     Scopes;
                                ThePackage    :    out Symbol)
   is
      NewPackage : Symbol;
   begin
      AddNewPackage (Name        => Name,
                     Comp_Unit   => Comp_Unit,
                     Declaration => Specification,
                     Scope       => Scope,
                     TheParent   => NullSymbol,
                     ChildSort   => PrivateChild,
                     --to get
                     ThePackage  => NewPackage);
      WriteDeclaration (Discriminant => PackageSpecification,
                        Item         => NewPackage,
                        Scope        => Scope,
                        Declaration  => Specification);
      ThePackage := NewPackage;
   end AddPrivatePackage;

   --------------------------------------------------------------------------------

   procedure SetPackageIsGeneric (ThePackage    : in Symbol)
   is
   begin
      RawDict.SetPackageIsGeneric (ThePackage);
   end SetPackageIsGeneric;

   --------------------------------------------------------------------------------

   procedure SetPackageElaborateBodyFound (ThePackage : in Symbol)
   is
   begin
      RawDict.SetPackageElaborateBodyFound (ThePackage);
   end SetPackageElaborateBodyFound;

   --------------------------------------------------------------------------------

   procedure SetPackageAsExtendingAnother (ThePackage          : in Symbol;
                                           ThePackageItExtends : in Symbol)
   is
   begin
      RawDict.SetPackageExtends (ThePackage, ThePackageItExtends);
   end SetPackageAsExtendingAnother;

   --------------------------------------------------------------------------------

   procedure AddPrivatePart (ThePackage  : in Symbol;
                             PrivatePart : in Location;
                             Hidden      : in Boolean)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Hidden then
            WriteString (Dict.TemporaryFile, "hidden ");
         end if;
         WriteString (Dict.TemporaryFile, "private part of ");
         WriteName (Dict.TemporaryFile, ThePackage);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, PrivatePart);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddPrivatePart;

   --------------------------------------------------------------------------------

   procedure AddOwnAnnotation (ThePackage : in Symbol;
                               Annotation : in Location)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile, "own annotation of ");
         WriteName (Dict.TemporaryFile, ThePackage);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Annotation);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddOwnAnnotation;

   --------------------------------------------------------------------------------

   procedure AddOwnVariable (Name              : in     LexTokenManager.Lex_String;
                             ThePackage        : in     Symbol;
                             Mode              : in     Modes;
                             IsProtected       : in     Boolean;
                             IsInterruptStream : in     Boolean;
                             Comp_Unit         : in     ContextManager.UnitDescriptors;
                             Reference         : in     Location;
                             Variable          :    out Symbol)
   is
      NewVariable, OwnVariable, ImplicitInStream : Symbol;
   begin
      if IsLocalOrPrivateScope (GetScope (ThePackage)) then
         NewVariable := LookupImmediateScope (Name,
                                              VisibleScope (ThePackage),
                                              ProofContext);
      else
         NewVariable := NullSymbol;
      end if;

      if NewVariable = NullSymbol then
         RawDict.CreateVariable (Name      => Name,
                                 Comp_Unit => Comp_Unit,
                                 Loc       => Reference.StartPosition,
                                 Variable  => NewVariable);
         RawDict.CreateOwnVariable (Variable    => NewVariable,
                                    Owner       => ThePackage,
                                    Comp_Unit   => Comp_Unit,
                                    Loc         => Reference.StartPosition,
                                    OwnVariable => OwnVariable);
         RawDict.SetVariableOwnVariable (NewVariable, OwnVariable);
         RawDict.SetNextOwnVariable (OwnVariable,
                                     RawDict.GetPackageOwnVariables (ThePackage));
         RawDict.SetPackageOwnVariables (ThePackage, OwnVariable);
      else
         if RawDict.Get_Symbol_Compilation_Unit (Item => NewVariable) = Comp_Unit then
            --  There was a usage of the variable before its actual
            --  declaration -> update the declaration position. The
            --  declaration must be in the same compilation unit as
            --  the usage (in the package specification) to prevent to
            --  get a cross-reference from the package specification
            --  to the package body.
            RawDict.Set_Symbol_Location (Item     => NewVariable,
                                         Location => Reference.StartPosition);
         end if;
         OwnVariable := RawDict.GetVariableOwnVariable (NewVariable);
      end if;

      RawDict.SetOwnVariableAnnounced (OwnVariable);
      RawDict.SetOwnVariableMode (OwnVariable, Mode);
      RawDict.SetOwnVariableProtected (OwnVariable, IsProtected);
      RawDict.SetOwnVariableIsInterruptStream (OwnVariable, IsInterruptStream);
      AddOtherReference (NewVariable, ThePackage, Reference);

      -- For protected state that has no mode, we need to create an implicit in stream
      -- from which reads of the protected state are deemed to come; this facilitiates
      -- effective flow analysis of shared protected state.
      if IsProtected and then Mode = DefaultMode then
         RawDict.CreateProtectedImplicitInStream (TheOwnVar   => OwnVariable,
                                                  Comp_Unit   => Comp_Unit,
                                                  Loc         => Reference.StartPosition,
                                                  InStreamSym => ImplicitInStream);
         RawDict.SetOwnVariableImplicitInStream (OwnVariable,
                                                 ImplicitInStream);
      end if;

      Variable := NewVariable;

   end AddOwnVariable;

   --------------------------------------------------------------------------------

   procedure AddOwnVariableType (OwnVariable   : in Symbol;
                                 TypeMark      : in Symbol;
                                 TypeReference : in Location)
   is
   begin

      RawDict.SetVariableType (Variable => OwnVariable,
                               TypeMark => TypeMark);
      RawDict.SetOwnVariableTyped (RawDict.GetVariableOwnVariable (OwnVariable));

      if not TypeIsUnknown (TypeMark) then
         AddOtherReference (TypeMark, GetOwner (OwnVariable), TypeReference);
      end if;

   end AddOwnVariableType;

   --------------------------------------------------------------------------------

   procedure AddOwnTask (Name        : in     LexTokenManager.Lex_String;
                         Comp_Unit   : in     ContextManager.UnitDescriptors;
                         Declaration : in     Location;
                         TypeMark    : in     Symbol;
                         ThePackage  : in     Symbol;
                         TaskSym     :    out Symbol) is
      NewVariable, OwnTask : Symbol;
   begin
      RawDict.CreateVariable (Name      => Name,
                              Comp_Unit => Comp_Unit,
                              Loc       => Declaration.StartPosition,
                              Variable  => NewVariable);
      RawDict.SetVariableType (Variable => NewVariable,
                               TypeMark => TypeMark);
      RawDict.CreateOwnTask (Variable  => NewVariable,
                             Owner     => ThePackage,
                             Comp_Unit => Comp_Unit,
                             Loc       => Declaration.StartPosition,
                             OwnTask   => OwnTask);
      RawDict.SetVariableOwnTask (NewVariable, OwnTask);
      RawDict.SetNextOwnTask (OwnTask, RawDict.GetPackageTaskList (ThePackage));
      RawDict.SetPackageTaskList (ThePackage, OwnTask);
      TaskSym := NewVariable;
   end AddOwnTask;

   --------------------------------------------------------------------------------

   procedure AddRefinementDefinition (ThePackage : in Symbol;
                                      Annotation : in Location)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile, "refinement definition of ");
         WriteName (Dict.TemporaryFile, ThePackage);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Annotation);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddRefinementDefinition;

   --------------------------------------------------------------------------------

   procedure AddConstituentSym (ConstituentVariable  : in Symbol;
                                Subject              : in Symbol;
                                Comp_Unit            : in ContextManager.UnitDescriptors;
                                ConstituentReference : in Location)
   is
      ThePackage             : Symbol;
      SubjectOwnVariable     : Symbol;
      Constituent            : Symbol;
   begin
      SubjectOwnVariable := RawDict.GetVariableOwnVariable (Subject);
      ThePackage := RawDict.GetOwnVariableOwner (SubjectOwnVariable);

      RawDict.CreateConstituent (Subject     => SubjectOwnVariable,
                                 Variable    => ConstituentVariable,
                                 Comp_Unit   => Comp_Unit,
                                 Loc         => ConstituentReference.StartPosition,
                                 Constituent => Constituent);
      RawDict.SetConstituentMode (Constituent, GetOwnVariableMode (ConstituentVariable));
      RawDict.SetVariableConstituent (ConstituentVariable, Constituent);

      RawDict.SetNextConstituent (Constituent,
                                  RawDict.GetOwnVariableConstituents (SubjectOwnVariable));
      RawDict.SetOwnVariableConstituents (SubjectOwnVariable, Constituent);

      AddOtherReference (Subject, ThePackage, Null_Location);
      AddOtherReference (ConstituentVariable, ThePackage, ConstituentReference);
   end AddConstituentSym;

   --------------------------------------------------------------------------------

   procedure AddConstituent (Name                 : in LexTokenManager.Lex_String;
                             Subject              : in Symbol;
                             Mode                 : in Modes;
                             SubjectReference     : in Location;
                             Comp_Unit            : in ContextManager.UnitDescriptors;
                             ConstituentReference : in Location)
   is
      ThePackage             : Symbol;
      SubjectOwnVariable     : Symbol;
      ConstituentVariable    : Symbol;
      Constituent            : Symbol;
   begin
      SubjectOwnVariable := RawDict.GetVariableOwnVariable (Subject);
      ThePackage := RawDict.GetOwnVariableOwner (SubjectOwnVariable);

      RawDict.CreateVariable (Name      => Name,
                              Comp_Unit => Comp_Unit,
                              Loc       => ConstituentReference.StartPosition,
                              Variable  => ConstituentVariable);
      RawDict.CreateConstituent (Subject     => SubjectOwnVariable,
                                 Variable    => ConstituentVariable,
                                 Comp_Unit   => Comp_Unit,
                                 Loc         => ConstituentReference.StartPosition,
                                 Constituent => Constituent);
      RawDict.SetConstituentMode (Constituent, Mode);
      RawDict.SetVariableConstituent (ConstituentVariable, Constituent);

      RawDict.SetNextConstituent (Constituent,
                                  RawDict.GetOwnVariableConstituents (SubjectOwnVariable));
      RawDict.SetOwnVariableConstituents (SubjectOwnVariable, Constituent);

      AddOtherReference (Subject, ThePackage, SubjectReference);
      AddOtherReference (ConstituentVariable, ThePackage, ConstituentReference);

      --# accept Flow, 601, SPARK_IO.File_Sys, Mode, "Spurious data coupling via Dictionary";
   end AddConstituent; -- spurious coupling between Mode and File_Sys

   --------------------------------------------------------------------------------

   procedure AddEmbeddedConstituent (PackageName          : in LexTokenManager.Lex_String;
                                     VariableName         : in LexTokenManager.Lex_String;
                                     Subject              : in Symbol;
                                     Mode                 : in Modes;
                                     SubjectReference     : in Location;
                                     Comp_Unit            : in ContextManager.UnitDescriptors;
                                     ConstituentReference : in Location)
   is
      OuterPackage           : Symbol;
      InnerPackage           : Symbol;
      Specification          : Symbol;
      SubjectOwnVariable     : Symbol;
      ConstituentVariable    : Symbol;
      ConstituentOwnVariable : Symbol;
      Constituent            : Symbol;
   begin

      SubjectOwnVariable := RawDict.GetVariableOwnVariable (Subject);
      OuterPackage := RawDict.GetOwnVariableOwner (SubjectOwnVariable);

      InnerPackage := LookupImmediateScope (PackageName,
                                            LocalScope (OuterPackage),
                                            ProofContext);

      if InnerPackage = NullSymbol then
         RawDict.CreatePackage (Name       => PackageName,
                                Comp_Unit  => Comp_Unit,
                                Loc        => ConstituentReference.StartPosition,
                                ThePackage => InnerPackage);
         AddDeclaration (Item        => InnerPackage,
                         Comp_Unit   => Comp_Unit,
                         Loc         => ConstituentReference,
                         Scope       => LocalScope (OuterPackage),
                         Context     => ProofContext,
                         Declaration => Specification);
         RawDict.SetPackageSpecification (InnerPackage, Specification);
      end if;

      RawDict.CreateVariable (Name      => VariableName,
                              Comp_Unit => Comp_Unit,
                              Loc       => ConstituentReference.StartPosition,
                              Variable  => ConstituentVariable);
      RawDict.CreateOwnVariable (Variable    => ConstituentVariable,
                                 Owner       => InnerPackage,
                                 Comp_Unit   => Comp_Unit,
                                 Loc         => ConstituentReference.StartPosition,
                                 OwnVariable => ConstituentOwnVariable);
      RawDict.SetVariableOwnVariable (ConstituentVariable, ConstituentOwnVariable);

      RawDict.SetNextOwnVariable (ConstituentOwnVariable,
                                  RawDict.GetPackageOwnVariables (InnerPackage));
      RawDict.SetPackageOwnVariables (InnerPackage, ConstituentOwnVariable);

      RawDict.CreateConstituent (Subject     => SubjectOwnVariable,
                                 Variable    => ConstituentVariable,
                                 Comp_Unit   => Comp_Unit,
                                 Loc         => ConstituentReference.StartPosition,
                                 Constituent => Constituent);
      RawDict.SetConstituentMode (Constituent, Mode);
      RawDict.SetVariableConstituent (ConstituentVariable, Constituent);

      RawDict.SetNextConstituent (Constituent,
                                  RawDict.GetOwnVariableConstituents (SubjectOwnVariable));
      RawDict.SetOwnVariableConstituents (SubjectOwnVariable, Constituent);

      AddOtherReference (Subject, OuterPackage, SubjectReference);
      AddOtherReference (Constituent, OuterPackage, ConstituentReference);

      --# accept Flow, 601, SPARK_IO.File_Sys, Mode, "Spurious data coupling via Dictionary";
   end AddEmbeddedConstituent; -- spurious coupling between Mode and File_Sys

   --------------------------------------------------------------------------------

   procedure AddChildConstituent (Variable             : in Symbol;
                                  Subject              : in Symbol;
                                  Mode                 : in Modes;
                                  SubjectReference     : in Location;
                                  Comp_Unit            : in ContextManager.UnitDescriptors;
                                  ConstituentReference : in Location)
   is
      ThePackage             : Symbol;
      SubjectOwnVariable     : Symbol;
      Constituent            : Symbol;
   begin

      SubjectOwnVariable := RawDict.GetVariableOwnVariable (Subject);
      ThePackage := RawDict.GetOwnVariableOwner (SubjectOwnVariable);

      RawDict.CreateConstituent (Subject     => SubjectOwnVariable,
                                 Variable    => Variable,
                                 Comp_Unit   => Comp_Unit,
                                 Loc         => ConstituentReference.StartPosition,
                                 Constituent => Constituent);
      RawDict.SetConstituentMode (Constituent, Mode);
      RawDict.SetVariableConstituent (Variable, Constituent);

      RawDict.SetNextConstituent (Constituent,
                                  RawDict.GetOwnVariableConstituents (SubjectOwnVariable));
      RawDict.SetOwnVariableConstituents (SubjectOwnVariable, Constituent);

      AddOtherReference (Subject, ThePackage, SubjectReference);
      AddOtherReference (Variable, ThePackage, ConstituentReference);

      --# accept Flow, 601, SPARK_IO.File_Sys, Mode, "Spurious data coupling via Dictionary";
   end AddChildConstituent; -- spurious coupling between Mode and File_Sys

   --------------------------------------------------------------------------------

   procedure AddInitializationSpecification (ThePackage : in Symbol;
                                             Annotation : in Location)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile,
                      "initialization specification of ");
         WriteName (Dict.TemporaryFile, ThePackage);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Annotation);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddInitializationSpecification;

   --------------------------------------------------------------------------------

   procedure AddInitializedOwnVariable (Variable          : in Symbol;
                                        VariableReference : in Location)
   is
      OwnVariable : Symbol;
   begin
      OwnVariable := RawDict.GetVariableOwnVariable (Variable);
      RawDict.SetOwnVariableInitialized (OwnVariable);
      AddOtherReference (Variable, RawDict.GetOwnVariableOwner (OwnVariable), VariableReference);
   end AddInitializedOwnVariable;

   --------------------------------------------------------------------------------

   procedure AddPackageInitialization (ThePackage     : in Symbol;
                                       Initialization : in Location;
                                       Hidden         : in Boolean)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         if Hidden then
            WriteString (Dict.TemporaryFile, "hidden ");
         end if;
         WriteString (Dict.TemporaryFile, "package initialization of ");
         WriteName (Dict.TemporaryFile, ThePackage);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Initialization);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddPackageInitialization;

   --------------------------------------------------------------------------------

   procedure AddWithClause (Scope  : in Scopes;
                            Clause : in Location)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile, "with clause of ");
         WriteScope (Dict.TemporaryFile, Scope);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Clause);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddWithClause;

   --------------------------------------------------------------------------------
   procedure AddUseTypeClause (Scope  : in Scopes;
                               Clause : in Location)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile, "use type clause of ");
         WriteScope (Dict.TemporaryFile, Scope);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Clause);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddUseTypeClause;

   --------------------------------------------------------------------------------

   procedure AddWithReference (Scope            : in     Scopes;
                               ThePackage       : in     Symbol;
                               Explicit         : in     Boolean;
                               Comp_Unit        : in     ContextManager.UnitDescriptors;
                               PackageReference : in     Location;
                               AlreadyPresent   :    out Boolean)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddUseTypeReference (Scope         : in Scopes;
                                  TheType       : in Symbol;
                                  Comp_Unit     : in ContextManager.UnitDescriptors;
                                  TypeReference : in Location)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddInheritsAnnotation (CompilationUnit : in Symbol;
                                    Annotation      : in Location)
   is
   begin
      if SPARK_IO.Is_Open (Dict.TemporaryFile) then
         WriteString (Dict.TemporaryFile, "inherits annotation of ");
         WriteName (Dict.TemporaryFile, CompilationUnit);
         WriteString (Dict.TemporaryFile, " is at ");
         WriteLocation (Dict.TemporaryFile, Annotation);
         WriteLine (Dict.TemporaryFile, " ;");
      end if;
   end AddInheritsAnnotation;

   --------------------------------------------------------------------------------

   procedure AddInheritsReference (CompilationUnit  : in     Symbol;
                                   ThePackage       : in     Symbol;
                                   Explicit         : in     Boolean;
                                   Comp_Unit        : in     ContextManager.UnitDescriptors;
                                   PackageReference : in     Location;
                                   AlreadyPresent   :    out Boolean)
      is separate;

   --------------------------------------------------------------------------------

   procedure AddWriteReference (Variable, CompilationUnit : in Symbol;
                                Reference                 : in Location)
   is
   begin
      WriteReference (WriteRef, Variable, CompilationUnit, Reference);
   end AddWriteReference;

   --------------------------------------------------------------------------------

   procedure AddReadReference (Object, CompilationUnit : in Symbol;
                               Reference               : in Location)
   is
   begin
      WriteReference (ReadRef, Object, CompilationUnit, Reference);
   end AddReadReference;

   --------------------------------------------------------------------------------

   procedure AddSubprogramCall (Subprogram, CompilationUnit : in Symbol;
                                Call                        : in Location)
   is
   begin
      WriteReference (CallRef, Subprogram, CompilationUnit, Call);
   end AddSubprogramCall;

   --------------------------------------------------------------------------------

   function SymbolRef (Item : Symbol) return ExaminerConstants.RefType
   is
   begin
      return ExaminerConstants.RefType (Item);
   end SymbolRef;

   --------------------------------------------------------------------------------

   function ConvertSymbolRef (Ref : ExaminerConstants.RefType) return Symbol
   is
   begin
      return Symbol (Ref);
   end ConvertSymbolRef;

   --------------------------------------------------------------------------------

   function ConvertScope (S : Scopes) return ScopeReference
   is
   begin
      return ScopeReference'(TypeOfScope => Visibility'Pos (S.TypeOfScope),
                             Region => Integer (SymbolRef (S.Region)));
   end ConvertScope;

   --------------------------------------------------------------------------------

   function ConvertScopeReference (SR : ScopeReference) return Scopes
   is
   begin
      return Scopes'(TypeOfScope => Visibility'Val (SR.TypeOfScope),
                     Region => ConvertSymbolRef (ExaminerConstants.RefType (SR.Region)));
   end ConvertScopeReference;

   --------------------------------------------------------------------------------

   function FirstExtendedRecordComponent (TheRecordType : Symbol) return Iterator
   is
      Component     : Symbol;
      Components    : Iterator;
      CurrentRecord : Symbol;
   begin
      CurrentRecord := TheRecordType;
      -- find root record
      loop
         CurrentRecord := GetType (CurrentSymbol (FirstRecordComponent (CurrentRecord)));
         exit when not TypeIsExtendedTagged (CurrentRecord);
      end loop;
      -- CurrentRecord now points at Root record
      -- Assuming this record has some fields, we need the first one.
      Component := RawDict.GetTypeFirstRecordComponent (CurrentRecord);
      loop
         if Component /= NullSymbol then
            -- found a field
            Components := Iterator'(ExtendedRecordComponentIterator,
                                    IsAbstract,
                                    Component,
                                    TheRecordType);
            exit;
         end if;

         -- if we get here then the root record had no fields so we need to try the 1st extension
         CurrentRecord := BackTrackUpInheritChain (TheRecordType, CurrentRecord);
         if CurrentRecord = NullSymbol then
            -- we didn't find ANY fields in the entire extended record structure!
            Components := NullIterator;
            exit;
         end if;

         -- if we get here then we have successfully backtracked one place up the inherit chain
         -- and can have another go.  Since we are now in an extension we need the second field
         -- so as to skip the Inherit field
         Component := RawDict.GetNextRecordComponent (RawDict.GetTypeFirstRecordComponent (CurrentRecord));
      end loop;
      return Components;
   end FirstExtendedRecordComponent;

   --------------------------------------------------------------------------------

   function ExtendedTaggedHasPrivateAncestors (TheType : Symbol;
                                               Scope   : Scopes) return Boolean
   is
      Result : Boolean := False;
      CurrentRecord : Symbol;
   begin
      CurrentRecord := TheType;
      loop
         -- follow chain of Inherit field pointers
         CurrentRecord := CurrentSymbol (FirstRecordComponent (CurrentRecord));
         exit when CurrentRecord = NullSymbol; -- root record is a null record

         CurrentRecord := GetType (CurrentRecord);
         --exit when not TypeIsRecord (CurrentRecord); -- all fields checked - false result exit
         exit when not TypeIsTagged (CurrentRecord); -- all fields checked - false result exit

         if IsPrivateType (CurrentRecord, Scope) then
            Result := True;
            exit;        -- true result exit
         end if;
      end loop;
      return Result;
   end ExtendedTaggedHasPrivateAncestors;

   --------------------------------------------------------------------------------

   function GetNumberOfNonExtendedComponents (TheRecordType : Symbol) return Natural
   is
      Component : Iterator;
      Count     : Natural;
   begin
      SystemErrors.RTAssert (not IsSubtype (TheRecordType),
                             SystemErrors.PreconditionFailure,
                             "GetNumberOfNonExtendedComponents called with record subtype");

      Component := FirstRecordComponent (TheRecordType);
      if CurrentSymbol (Component) = NullSymbol then
         -- record has no components perhaps because it is a private type
         -- and the private part is hidden or perhaps because it is a
         -- tagged null record
         Count := 0;
      else
         Count := 1;
         loop
            Component := NextSymbol (Component);
            exit when IsNullIterator (Component);
            Count := Count + 1;
         end loop;
      end if;
      return Count;
   end GetNumberOfNonExtendedComponents;

   --------------------------------------------------------------------------------

   -- As above but ignore Inherit fields of extended tagged records
   function GetNumberOfActualComponents (TheRecordType : Symbol) return Natural
   is
      Result : Natural;
   begin
      Result := GetNumberOfNonExtendedComponents (TheRecordType);
      if TypeIsExtendedTagged (TheRecordType) then
         Result := Result - 1; -- ignore Inherit field
      end if;
      return Result;
   end GetNumberOfActualComponents;

   --------------------------------------------------------------------------------

   function GetNumberOfExtendedComponents (TheRecordType : Symbol) return Natural
   --# global in Dict;
   is
      Count         : Natural := 0;
      CurrentRecord : Symbol;
   begin
      CurrentRecord := TheRecordType;
      loop
         Count := Count + GetNumberOfActualComponents (CurrentRecord);

         -- Move down chain of Inherit pointers
         CurrentRecord := CurrentSymbol (FirstRecordComponent (CurrentRecord));
         exit when CurrentRecord = NullSymbol; -- root record is null record

         CurrentRecord := GetType (CurrentRecord);
         -- exit when not TypeIsRecord (CurrentRecord); -- all inherited fields checked
         exit when not TypeIsTagged (CurrentRecord); -- all inherited fields checked

      end loop;
      return Count;
   end GetNumberOfExtendedComponents;

   --------------------------------------------------------------------------------

   function GetNumberOfComponents (TheRecordType : Symbol) return Natural
   is
      Result : Natural;
   begin
      SystemErrors.RTAssert (not IsSubtype (TheRecordType),
                             SystemErrors.PreconditionFailure,
                             "Call to GetNumberOfComponents for a record subtype");

      if TypeIsExtendedTagged (TheRecordType) then
         Result := GetNumberOfExtendedComponents (TheRecordType);
      else
         Result := GetNumberOfNonExtendedComponents (TheRecordType);
      end if;
      return Result;
   end GetNumberOfComponents;

   --------------------------------------------------------------------------------

   function NoFieldsBelowThisRecord (RecordSym : Symbol) return Boolean
   is
      Result        : Boolean := True;
      CurrentRecord : Symbol;
   begin
      CurrentRecord := GetRootOfExtendedType (RecordSym);
      loop
         -- stop if not an extended record and therefore nowhere else to check
         exit when CurrentRecord = NullSymbol;

         -- if we find some fields then function returns false
         if GetNumberOfActualComponents (CurrentRecord) > 0 then
            Result := False;
            exit;
         end if;
         -- move down inherit chain
         CurrentRecord := GetRootOfExtendedType (CurrentRecord);
      end loop;
      return Result;
   end NoFieldsBelowThisRecord;

   --------------------------------------------------------------------------------

   function RecordHasSomeFields (RecordSym : Symbol) return Boolean
   is
      Result   : Boolean;
      RootType : Symbol;
   begin
      -- a record need to be declared if:
      -- (1) it has some actual fields; or
      -- (2) it is an extended record and an inherited record has some actual fields
      --
      -- The inverse is easier to calculate.  We don't need a declaration if:
      -- (1) the record has no fields (even an Inherit one); or
      -- (2) it is extended and has no actual fields and there are no fields below it.

      -- RecordSym might be a record subtype, so...
      RootType := GetRootType (RecordSym);

      Result := (GetNumberOfNonExtendedComponents (RootType) = 0) or else
        (TypeIsExtendedTagged (RootType) and then
         GetNumberOfActualComponents (RootType) = 0 and then
         NoFieldsBelowThisRecord (RootType));
      return not Result;
   end RecordHasSomeFields;

   --------------------------------------------------------------------------------

   function GetNonExtendedRecordComponent (TheRecordType : Symbol;
                                           Number        : Positive) return Symbol
   is
      Component : Iterator;
   begin
      Component := FirstRecordComponent (TheRecordType);
      for No in Positive range 1 .. Number - 1 loop
         Component := NextSymbol (Component);
      end loop;
      return CurrentSymbol (Component);
   end GetNonExtendedRecordComponent;

   --------------------------------------------------------------------------------

   function GetRecordComponent (TheRecordType : Symbol;
                                Number        : Positive) return Symbol
   is
      Result : Symbol;

      function GetExtendedRecordComponent (TheRecordType : Symbol;
                                           Number        : Positive) return Symbol
      --# global in Dict;
      is
         Component : Iterator;
         CurrentRecord : Symbol;
         Low, High     : Positive;
      begin
         CurrentRecord := TheRecordType;
         High := GetNumberOfExtendedComponents (CurrentRecord);
         loop
            Low := (High - GetNumberOfActualComponents (CurrentRecord)) + 1;
            exit when Number in Low .. High; -- found which bit of the record model holds field

            -- move down Inherit chain
            CurrentRecord := GetType (CurrentSymbol (FirstRecordComponent (CurrentRecord)));
            High := Low - 1;
         end loop;
         -- on exit, CurrentRecord indicates the record that holds field number is the
         -- range Low .. High
         Component := FirstRecordComponent (CurrentRecord);
         -- if type is extended then the above returns the Inherit field which we don't
         -- want so we skip it thus:
         if TypeIsExtendedTagged (CurrentRecord) then
            Component := NextSymbol (Component);
         end if;
         -- now loop through to find desired field
         for I in Positive range Low .. Number - 1 loop
            Component := NextSymbol (Component);
         end loop;
         return CurrentSymbol (Component);
      end GetExtendedRecordComponent;

      -------------------------------------

   begin -- GetRecordComponent
      if TypeIsExtendedTagged (TheRecordType) then
         Result := GetExtendedRecordComponent (TheRecordType, Number);
      else
         Result := GetNonExtendedRecordComponent (TheRecordType => TheRecordType,
                                                  Number        => Number);
      end if;
      return Result;
   end GetRecordComponent;

   --------------------------------------------------------------------------------

   function FirstSubprogramParameter (Subprogram : Symbol) return Iterator
      is separate;

   --------------------------------------------------------------------------------

   -- function FirstGenericFormalParameter (TheGeneric : Symbol) return Iterator
   --    is separate;

   --------------------------------------------------------------------------------

   function ActualParameterOfGenericParameter (TheParameter        : in Symbol;
                                               ActualSubprogramSym : in Symbol) return Symbol

   is
      GenericIt : Iterator;
      ActualIt  : Iterator;
      Result : Symbol := NullSymbol;

   begin
      -- traverse generic and instantiation subprogram parameters lists in parallel
      GenericIt := FirstSubprogramParameter (GetSubprogram (TheParameter));
      ActualIt := FirstSubprogramParameter (ActualSubprogramSym);
      loop
         if CurrentSymbol (GenericIt) = TheParameter then
            -- found it
            Result := CurrentSymbol (ActualIt);
            exit;
         end if;

         -- this exit should never be reached, we should find what we are looking for above
         exit when IsNullIterator (GenericIt);

         GenericIt := NextSymbol (GenericIt);
         ActualIt := NextSymbol (ActualIt);
      end loop;
      return Result;
   end ActualParameterOfGenericParameter;

   --------------------------------------------------------------------------------

   function FirstKnownDiscriminant (ProtectedOrTaskType : Symbol) return Iterator
   is
      Discriminant  : Symbol;
      Discriminants : Iterator;
   begin
      if RawDict.GetTypeDiscriminant (ProtectedOrTaskType) = ProtectedType then
         Discriminant := RawDict.GetProtectedTypeFirstDiscriminant (ProtectedOrTaskType);
      else
         Discriminant := RawDict.GetTaskTypeFirstDiscriminant (ProtectedOrTaskType);
      end if;

      if Discriminant = NullSymbol then
         Discriminants := NullIterator;
      else
         Discriminants := Iterator'(KnownDiscriminantIterator,
                                    IsAbstract,
                                    Discriminant,
                                    NullSymbol);
      end if;
      return Discriminants;
   end FirstKnownDiscriminant;

   --------------------------------------------------------------------------------

   function FirstDiscriminantConstraint (ProtectedOrTaskSubtype : Symbol) return Iterator
   is
      Discriminant  : Symbol;
      ExtraInfo     : Symbol;
      Discriminants : Iterator;
   begin
      ExtraInfo := RawDict.GetTypeAncilliaryFields (ProtectedOrTaskSubtype);
      Discriminant := RawDict.GetSubtypeInfoFirstConstraint (ExtraInfo);

      if Discriminant = NullSymbol then
         Discriminants := NullIterator;
      else
         Discriminants := Iterator'(DiscriminantConstraintIterator,
                                    IsAbstract,
                                    Discriminant,
                                    NullSymbol);
      end if;
      return Discriminants;
   end FirstDiscriminantConstraint;

   --------------------------------------------------------------------------------
   function GetVariableExpNode (Variable : Symbol) return ExaminerConstants.RefType
   is
   begin
      return RawDict.GetVariableExpNode (Variable);
   end GetVariableExpNode;

   --------------------------------------------------------------------------------

   function GetSubprogramParameterNumber (Parameter : Symbol) return Positive
   is
      Current : Iterator;
      Number  : Positive;
   begin
      Current := FirstSubprogramParameter (GetSubprogram (Parameter));
      Number := 1;
      loop
         exit when CurrentSymbol (Current) = Parameter;
         Current := NextSymbol (Current);
         Number := Number + 1;
      end loop;
      return Number;
   end GetSubprogramParameterNumber;

   --------------------------------------------------------------------------------

   function GetSubprogramParameterMode (Parameter : Symbol) return Modes
   is
   begin
      return RawDict.GetSubprogramParameterMode (Parameter);
   end GetSubprogramParameterMode;

   --------------------------------------------------------------------------------

   function GetSubprogramParameterConstraint (Parameter : Symbol;
                                              Dimension : Positive) return Symbol
   is
      SymbolToTry : Symbol;
   begin
      if IsSubprogramParameter (Parameter) then
         -- Precondition is met so:
         --    traverse linked list of constraints until right dimension found
         SymbolToTry := RawDict.GetSubprogramParameterIndexConstraints (Parameter);
         loop
            -- error exit (prevent infinite loop in worst case)
            exit when SymbolToTry = NullSymbol;
            -- normal exit
            exit when RawDict.GetParameterConstraintDimension (SymbolToTry) = Dimension;

            SymbolToTry := RawDict.GetParameterConstraintNext (SymbolToTry);
         end loop;
      else
         -- Precondition not met - only possible if source being processed by Examiner contains semantic error
         SymbolToTry := GetUnknownTypeMark;
      end if;
      return SymbolToTry;
   end GetSubprogramParameterConstraint;

   --------------------------------------------------------------------------------

   function GetNumberOfSubprogramParameters (Subprogram : Symbol) return Natural
   is
      Parameter : Iterator;
      Count     : Natural;
   begin
      Parameter := FirstSubprogramParameter (Subprogram);
      Count := 0;
      loop
         exit when IsNullIterator (Parameter);
         Parameter := NextSymbol (Parameter);
         Count := Count + 1;
      end loop;
      return Count;
   end GetNumberOfSubprogramParameters;

   --------------------------------------------------------------------------------

   function GetNumberOfGenericFormalParameters (TheGeneric : Symbol) return Natural
   is
      Parameter : Iterator;
      Count     : Natural;
   begin
      Parameter := FirstGenericFormalParameter (TheGeneric);
      Count := 0;
      loop
         exit when IsNullIterator (Parameter);
         Parameter := NextSymbol (Parameter);
         Count := Count + 1;
      end loop;
      return Count;
   end GetNumberOfGenericFormalParameters;

   --------------------------------------------------------------------------------

   function GetGenericFormalParameterKind (TheGenericFormalParameter : Symbol) return GenericParameterKind
   is
   begin
      return RawDict.GetGenericParameterKind (TheGenericFormalParameter);
   end GetGenericFormalParameterKind;

   --------------------------------------------------------------------------------
   -- moved above GenerateSimpleName
   -- function GetSubprogramParameterConstraintDimension (TheConstraint : Symbol) return Positive
   --------------------------------------------------------------------------------

   function GetParameterAssociatedWithParameterConstraint (TheConstraint : Symbol) return Symbol
   is
   begin
      return RawDict.GetParameterConstraintParameter (TheConstraint);
   end GetParameterAssociatedWithParameterConstraint;

   --------------------------------------------------------------------------------

   function GetSubprogramParameter (Subprogram : Symbol;
                                    Number     : Positive) return Symbol
   is
      Parameter : Iterator;
   begin
      Parameter := FirstSubprogramParameter (Subprogram);
      for No in Positive range 1 .. Number - 1 loop
         Parameter := NextSymbol (Parameter);
      end loop;
      return CurrentSymbol (Parameter);
   end GetSubprogramParameter;

   --------------------------------------------------------------------------------

   function GetArrayAttributeType (Name      : LexTokenManager.Lex_String;
                                   TypeMark  : Symbol;
                                   Dimension : Positive) return Symbol
   is
      Result : Symbol;
   begin
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                              Lex_Str2 => LexTokenManager.Length_Token) = LexTokenManager.Str_Eq or else
        LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                             Lex_Str2 => LexTokenManager.Component_Size_Token) = LexTokenManager.Str_Eq then
         Result := GetUniversalIntegerType;
      else
         Result := GetArrayIndex (TypeMark, Dimension);
      end if;
      return Result;
   end GetArrayAttributeType;

   --------------------------------------------------------------------------------

   function GetArrayAttributeValue (Name      : LexTokenManager.Lex_String;
                                    TypeMark  : Symbol;
                                    Dimension : Positive) return LexTokenManager.Lex_String
   is
      Index  : Symbol;
      Result : LexTokenManager.Lex_String;
   begin

      Index := GetArrayIndex (TypeMark, Dimension);

      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                              Lex_Str2 => LexTokenManager.First_Token) = LexTokenManager.Str_Eq then
         Result := RawDict.GetTypeLower (Index);
      elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Name,
                                                                 Lex_Str2 => LexTokenManager.Last_Token) = LexTokenManager.Str_Eq then
         Result := RawDict.GetTypeUpper (Index);
      else
         Result := LexTokenManager.Null_String;
      end if;

      return Result;

   end GetArrayAttributeValue;

   --------------------------------------------------------------------------------

   function IsEnumerationLiteral (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = EnumerationLiteralSymbol;
   end IsEnumerationLiteral;

   --------------------------------------------------------------------------------

   function IsLoopParameter (TheSymbol : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (TheSymbol) = LoopParameterSymbol;
   end IsLoopParameter;

   --------------------------------------------------------------------------------

   function HasBody (CompilationUnit : Symbol) return Boolean
   is
      Result : Boolean;
   begin

      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when PackageSymbol =>
            Result := RawDict.GetPackageHasProperBody (CompilationUnit);
         when TypeSymbol =>
            if RawDict.GetTypeDiscriminant (CompilationUnit) = ProtectedType then
               Result := RawDict.GetProtectedTypeHasProperBody (CompilationUnit);
            elsif RawDict.GetTypeDiscriminant (CompilationUnit) = TaskType then
               Result := RawDict.GetTaskTypeHasProperBody (CompilationUnit);
            else
               Result := False;
            end if;
         when others =>
            Result := RawDict.GetSubprogramHasProperBody (CompilationUnit);
      end case;

      return Result;

   end HasBody;

   --------------------------------------------------------------------------------

   function ConstantIsDeferredHere (TheConstant : Symbol;
                                    Scope       : Scopes) return Boolean
   is
   begin
      return ConstantIsDeferred (TheConstant) and then
         not IsLocal (Scope, GetScope (TheConstant));
   end ConstantIsDeferredHere;

   --------------------------------------------------------------------------------

   function ConstantIsScalar (TheConstant : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsScalar (GetType (TheConstant));
   end ConstantIsScalar;
   pragma Unreferenced (ConstantIsScalar); -- not used at present

   --------------------------------------------------------------------------------

   function ConstantIsDiscrete (TheConstant : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsDiscrete (GetType (TheConstant));
   end ConstantIsDiscrete;
   pragma Unreferenced (ConstantIsDiscrete); -- not used at present

   --------------------------------------------------------------------------------

   function ConstantIsNumeric (TheConstant : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsNumeric (GetType (TheConstant));
   end ConstantIsNumeric;
   pragma Unreferenced (ConstantIsNumeric); -- not used at present

   --------------------------------------------------------------------------------

   function ConstantIsBoolean (TheConstant : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsBoolean (GetType (TheConstant));
   end ConstantIsBoolean;
   pragma Unreferenced (ConstantIsBoolean); -- not used at present

   --------------------------------------------------------------------------------

   function TypeIsCharacter (TypeMark : Symbol) return Boolean
   is
   begin
      return IsPredefinedCharacterType (GetRootType (TypeMark));
   end TypeIsCharacter;

   --------------------------------------------------------------------------------

   function ConstantIsCharacter (TheConstant : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsCharacter (GetType (TheConstant));
   end ConstantIsCharacter;
   pragma Unreferenced (ConstantIsCharacter); -- not used at present

   --------------------------------------------------------------------------------

   function ConstantIsEnumeration (TheSymbol : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsEnumeration (GetType (TheSymbol));
   end ConstantIsEnumeration;
   pragma Unreferenced (ConstantIsEnumeration); -- not used at present

   --------------------------------------------------------------------------------

   function ConstantIsInteger (TheConstant : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsInteger (GetType (TheConstant));
   end ConstantIsInteger;
   pragma Unreferenced (ConstantIsInteger); -- not used at present

   --------------------------------------------------------------------------------

   function ConstantIsReal (TheConstant : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsReal (GetType (TheConstant));
   end ConstantIsReal;
   pragma Unreferenced (ConstantIsReal); -- not used at present

   -------------------------------------------------------------------------------

   function ConstantIsArray (TheConstant : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsArray (GetType (TheConstant));
   end ConstantIsArray;
   pragma Unreferenced (ConstantIsArray); -- not used at present

   --------------------------------------------------------------------------------

   function ConstantIsRecord (TheConstant : Symbol) return Boolean
      --# global in Dict;
   is
   begin
      return TypeIsRecord (GetType (TheConstant));
   end ConstantIsRecord;
   pragma Unreferenced (ConstantIsRecord); -- not used at present

   --------------------------------------------------------------------------------

   function IsBooleanTypeMark (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsPredefinedBooleanType (GetRootType (TheSymbol));
   end IsBooleanTypeMark;

   --------------------------------------------------------------------------------

   function IsRecordTypeMark (TheSymbol : Symbol;
                              Scope     : Scopes) return Boolean
   is
   begin
      return IsTypeMark (TheSymbol) and then
         TypeIsRecord (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsRecordTypeMark;

   --------------------------------------------------------------------------------

   function IsStatic (Item  : Symbol;
                      Scope : Scopes) return Boolean
   is
      Static : Boolean;
   begin

      case RawDict.GetSymbolDiscriminant (Item) is
         when ConstantSymbol =>
            Static := RawDict.GetConstantStatic (Item) and then
               not ConstantIsDeferredHere (Item, Scope);
         when TypeSymbol =>
            Static := RawDict.GetTypeStatic (Item) and then
               not TypeIsPrivateHere (Item, Scope);
         when others =>
            Static := False;
      end case;

      return Static;

   end IsStatic;

   --------------------------------------------------------------------------------

   function IsScalarType (TheSymbol : Symbol;
                          Scope     : Scopes) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
         TypeIsScalar (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsScalarType;

   --------------------------------------------------------------------------------

   function IsNumericType (TheSymbol : Symbol; Scope : Scopes) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
         TypeIsNumeric (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsNumericType;

   --------------------------------------------------------------------------------

   function IsFixedPointType (TheSymbol : Symbol;
                              Scope     : Scopes) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
         TypeIsFixedPoint (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsFixedPointType;

   --------------------------------------------------------------------------------

   function IsFloatingPointType (TheSymbol : Symbol;
                                 Scope     : Scopes) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
         TypeIsFloatingPoint (TheSymbol) and then
         not TypeIsPrivateHere (TheSymbol, Scope);
   end IsFloatingPointType;

   --------------------------------------------------------------------------------

   function IsRecordComponent (TheSymbol : Symbol) return Boolean
   is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (TheSymbol) is
         when RecordComponentSymbol | SubcomponentSymbol =>
            Result := True;
         when others =>
            Result := False;
      end case;
      return Result;
   end IsRecordComponent;

   --------------------------------------------------------------------------------

   function IsInherited (ThePackage, CompilationUnit : Symbol) return Boolean
   is
      Current : Symbol;
   begin

      if IsPackage (CompilationUnit) then
         Current := RawDict.GetPackageInheritClauses (CompilationUnit);
      elsif IsMainProgram (CompilationUnit) then
         Current := Dict.Main.InheritClauses;
      else
         Current := NullSymbol;
      end if;

      loop
         exit when Current = NullSymbol;
         exit when RawDict.GetContextClausePackage (Current) = ThePackage;
         Current := RawDict.GetNextContextClause (Current);
      end loop;

      return Current /= NullSymbol;

   end IsInherited;

   --------------------------------------------------------------------------------

   -- moved above LookUpSelectedItem so that it can be called
   function FirstVisibleSubprogram (ThePackageOrType : Symbol) return Iterator
   is
      Declaration        : Symbol;
      Item               : Symbol;
      VisibleSubprograms : Iterator;
   begin

      if IsPackage (ThePackageOrType) then
         Declaration := RawDict.GetPackageFirstVisibleDeclaration (ThePackageOrType);
      else -- precondiition means it must be a protected type
         Declaration := RawDict.GetProtectedTypeFirstVisibleDeclaration (ThePackageOrType);
      end if;

      loop
         if Declaration = NullSymbol then
            VisibleSubprograms := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if RawDict.GetSymbolDiscriminant (Item) = SubprogramSymbol and then
            Declaration /= RawDict.GetSubprogramBody (Item) then
            VisibleSubprograms := Iterator'(VisibleSubprogramIterator,
                                            IsAbstract,
                                            Item,
                                            Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return VisibleSubprograms;

   end FirstVisibleSubprogram;

   --------------------------------------------------------------------------------

   function FirstPrivateSubprogram (ThePackage : Symbol) return Iterator

   is
      Declaration        : Symbol;
      Item               : Symbol;
      PrivateSubprograms : Iterator;
   begin

      Declaration := RawDict.GetPackageFirstPrivateDeclaration (ThePackage);
      loop
         if Declaration = NullSymbol then
            PrivateSubprograms := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if RawDict.GetSymbolDiscriminant (Item) = SubprogramSymbol and then
            Declaration /= RawDict.GetSubprogramBody (Item) then
            PrivateSubprograms := Iterator'(PrivateSubprogramIterator,
                                            IsAbstract,
                                            Item,
                                            Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return PrivateSubprograms;

   end FirstPrivateSubprogram;

   --------------------------------------------------------------------------------

   function LookupSelectedItem (Prefix   : Symbol;
                                Selector : LexTokenManager.Lex_String;
                                Scope    : Scopes;
                                Context  : Contexts) return Symbol
      is separate;
   --------------------------------------------------------------------------------

   procedure SearchForInheritedOperations (Name             : in     LexTokenManager.Lex_String;
                                           Scope            : in     Scopes;
                                           Prefix           : in     Symbol;
                                           Context          : in     Contexts;
                                           OpSym            :    out Symbol;
                                           KindOfOp         :    out KindsOfOp;
                                           ActualTaggedType :    out Symbol)
      is separate;

   --------------------------------------------------------------------------------

   function GetSubprogramControllingType (Subprogram  : Symbol) return Symbol
      is separate;

   --------------------------------------------------------------------------------

   function GetOverriddenSubprogram (Subprogram  : Symbol) return Symbol
   is
      DeclaringRegion : Symbol;
      Result : Symbol := NullSymbol;
   begin
      SystemErrors.RTAssert (IsSubprogram (Subprogram),
                             SystemErrors.PreconditionFailure,
                             "In call to GetOverriddenSubprogram");

      DeclaringRegion := GetRegion (GetScope (Subprogram));
      if IsPackage (DeclaringRegion) and then
        PackageExtendsAnotherPackage (DeclaringRegion) then
         -- It is possible that the subprogram overrides since it
         -- is declared in a package that extends another.

         -- Overriding has taken place if there is a subprogram of
         -- the same name in the package which has been extended; this
         -- test is strong enough because we know that a name cannot
         -- be re-used unless the overriding is succesful (overloading
         -- not being permitted).

         Result := LookupImmediateScope (GetSimpleName (Subprogram),
                                         VisibleScope (GetPackageThatIsExtended (DeclaringRegion)),
                                         ProgramContext);

         -- Result will either be the overridden subprogram if present or null symbol if not
      end if;
      return Result;
   end GetOverriddenSubprogram;

   --------------------------------------------------------------------------------

   function IsProofType (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsType (TheSymbol) and then
        ((GetContext (TheSymbol) = ProofContext) or else
           (RawDict.GetTypeDiscriminant (TheSymbol) = AbstractProofType));
   end IsProofType;

   --------------------------------------------------------------------------------

   function IsProofConstant (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsConstant (TheSymbol) and then
         GetContext (TheSymbol) = ProofContext;
   end IsProofConstant;

   --------------------------------------------------------------------------------

   function IsProofFunction (TheSymbol : Symbol) return Boolean
   is
   begin
      return IsFunction (TheSymbol) and then
         GetContext (TheSymbol) = ProofContext;
   end IsProofFunction;

   --------------------------------------------------------------------------------

   function IsConcreteOwnVariable (Variable : Symbol) return Boolean
   is
   begin
      return IsOwnVariable (Variable) and then OwnVariableIsConcrete (Variable);
   end IsConcreteOwnVariable;

   --------------------------------------------------------------------------------

   function IsConstituent (Variable : Symbol) return Boolean
   is
      Result : Boolean;
   begin
      case RawDict.GetSymbolDiscriminant (Variable) is
         when VariableSymbol =>
            Result := RawDict.GetVariableConstituent (Variable) /= NullSymbol;
         when others =>
            Result := False;
      end case;

      return Result;

   end IsConstituent;

   --------------------------------------------------------------------------------

   function GetSubject (Constituent : Symbol) return Symbol
   is
      TheSubject, TheConstituent : Symbol;
   begin

      TheConstituent := RawDict.GetVariableConstituent (Constituent);
      TheSubject := RawDict.GetConstituentSubject (TheConstituent);

      return RawDict.GetOwnVariableVariable (TheSubject);

   end GetSubject;

   --------------------------------------------------------------------------------

   function IsRefinement (Subject, Constituent : Symbol) return Boolean
   is
   begin
      return IsConstituent (Constituent) and then
         GetSubject (Constituent) = Subject;
   end IsRefinement;

   --------------------------------------------------------------------------------

   function IsRefinementConstituent (ThePackage, Variable : Symbol) return Boolean
   is
   begin
      return IsConstituent (Variable) and then
         GetOwner (GetSubject (Variable)) = ThePackage;
   end IsRefinementConstituent;

   --------------------------------------------------------------------------------

   function GetOwnVariableOrConstituentMode (Variable : Symbol) return Modes
   is
      Result : Modes;
   begin
      if IsConstituent (Variable) then
         Result := GetConstituentMode (Variable);
      elsif IsOwnVariable (Variable) then
         Result := GetOwnVariableMode (Variable);
      else
         Result := DefaultMode;
      end if;
      return Result;
   end GetOwnVariableOrConstituentMode;

   --------------------------------------------------------------------------------

   function IsOwnVariableOrConstituentWithMode (Variable : Symbol) return Boolean
   is
   begin
      return GetOwnVariableOrConstituentMode (Variable) /= DefaultMode;
   end IsOwnVariableOrConstituentWithMode;

   --------------------------------------------------------------------------------

   function IsProtectedImplicitInStream (Sym : Symbol) return Boolean
   is
   begin
      return RawDict.GetSymbolDiscriminant (Sym) = ProtectedImplicitInStreamSymbol;
   end IsProtectedImplicitInStream;

   --------------------------------------------------------------------------------

   function IsUnmodedProtectedOwnVariable (Sym : Symbol) return Boolean
   is
   begin
      return IsOwnVariable (Sym) and then
        GetOwnVariableProtected (Sym) and then
        GetOwnVariableMode (Sym) = DefaultMode;
   end IsUnmodedProtectedOwnVariable;

   --------------------------------------------------------------------------------

   function GetProtectedImplicitInStream (TheProtectedOwnVar : Symbol) return Symbol
   is
   begin
      return RawDict.GetOwnVariableImplicitInStream (RawDict.GetVariableOwnVariable (TheProtectedOwnVar));
   end GetProtectedImplicitInStream;

   --------------------------------------------------------------------------------

   function GetProtectedTypeElementsHidden (TheProtectedType : in Symbol) return Boolean
   is
   begin
      return RawDict.GetProtectedTypeElementsHidden (TheProtectedType);
   end GetProtectedTypeElementsHidden;

   --------------------------------------------------------------------------------

   function GetProtectedTypeHasEntry (TheProtectedType : Symbol) return Boolean
   is
   begin
      return RawDict.GetProtectedTypeHasEntry (TheProtectedType);
   end GetProtectedTypeHasEntry;

   --------------------------------------------------------------------------------

   function GetTypeHasPragma (TheProtectedOrTaskType : in Symbol;
                              ThePragma              : in RavenscarPragmas) return Boolean
   is
      Result : Boolean;
   begin
      if RawDict.GetTypeDiscriminant (TheProtectedOrTaskType) = ProtectedType then
         Result := RawDict.GetProtectedTypeHasPragma (TheProtectedOrTaskType, ThePragma);
      else
         Result := RawDict.GetTaskTypeHasPragma (TheProtectedOrTaskType, ThePragma);
      end if;
      return Result;
   end GetTypeHasPragma;

   --------------------------------------------------------------------------------

   function GetTypePragmaValue (TheProtectedOrTaskType : in Symbol;
                                ThePragma              : in RavenscarPragmasWithValue)
                               return LexTokenManager.Lex_String
   is
      Result : LexTokenManager.Lex_String;
   begin
      if RawDict.GetTypeDiscriminant (TheProtectedOrTaskType) = ProtectedType then
         Result := RawDict.GetProtectedTypePragmaValue (TheProtectedOrTaskType, ThePragma);
      else
         Result := RawDict.GetTaskTypePragmaValue (TheProtectedOrTaskType, ThePragma);
      end if;
      return Result;
   end GetTypePragmaValue;

   --------------------------------------------------------------------------------

   function GetTypePriority (TheProtectedOrTaskType : in Symbol)
                            return LexTokenManager.Lex_String
   is
      Result : LexTokenManager.Lex_String;
   begin
      if RawDict.GetTypeParent (TheProtectedOrTaskType) = NullSymbol then
         -- get priority of type
         if GetTypeHasPragma (TheProtectedOrTaskType, Priority) then
            Result := GetTypePragmaValue (TheProtectedOrTaskType, Priority);
         else
            -- must be
            Result := GetTypePragmaValue (TheProtectedOrTaskType, InterruptPriority);
         end if;
      else
         -- get priority of subtype
         Result := RawDict.GetSubtypeInfoPriority (RawDict.GetTypeAncilliaryFields (TheProtectedOrTaskType));
      end if;
      return Result;
   end GetTypePriority;

   --------------------------------------------------------------------------------

   function MainProgramExists return Boolean
   is
   begin
      return Dict.Main.Subprogram /= NullSymbol;
   end MainProgramExists;

   --------------------------------------------------------------------------------

   function MainProgramPrioritySupplied return Boolean
   is begin
      return Dict.Main.PriorityGiven;
   end MainProgramPrioritySupplied;

   --------------------------------------------------------------------------------

   function FirstDeferredConstant (ThePackage : Symbol) return Iterator
   is
      Declaration       : Symbol;
      Item              : Symbol;
      DeferredConstants : Iterator;
   begin

      Declaration := RawDict.GetPackageFirstVisibleDeclaration (ThePackage);

      loop
         if Declaration = NullSymbol then
            DeferredConstants := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsDeferredConstant (Item) then
            DeferredConstants := Iterator'(DeferredConstantIterator,
                                           IsAbstract,
                                           Item,
                                           Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return DeferredConstants;

   end FirstDeferredConstant;

   --------------------------------------------------------------------------------

   function FirstUndeclaredType (ThePackage : Symbol) return Iterator
   is
      Declaration     : Symbol;
      Item            : Symbol;
      UndeclaredTypes : Iterator;
   begin

      Declaration := RawDict.GetPackageFirstVisibleDeclaration (ThePackage);

      loop
         if Declaration = NullSymbol then
            UndeclaredTypes := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsType (Item) and then
            RawDict.GetTypeDiscriminant (Item) = UnknownType then
            UndeclaredTypes := Iterator'(UndeclaredTypeIterator,
                                         IsAbstract,
                                         Item,
                                         Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return UndeclaredTypes;

   end FirstUndeclaredType;

   --------------------------------------------------------------------------------

   function FirstPrivateType (ThePackage : Symbol) return Iterator
   is
      Declaration  : Symbol;
      Item         : Symbol;
      PrivateTypes : Iterator;
   begin

      Declaration := RawDict.GetPackageFirstVisibleDeclaration (ThePackage);

      loop
         if Declaration = NullSymbol then
            PrivateTypes := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsType (Item) and then
            TypeIsPrivate (Item) and then
            RawDict.GetDeclarationContext (Declaration) = ProgramContext then
            PrivateTypes := Iterator'(PrivateTypeIterator,
                                      IsAbstract,
                                      Item,
                                      Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return PrivateTypes;

   end FirstPrivateType;

   --------------------------------------------------------------------------------

   function FirstVisibleTaskType (ThePackage : Symbol) return Iterator
   is
      Declaration        : Symbol;
      Item               : Symbol;
      TaskTypes          : Iterator;
   begin
      Declaration := RawDict.GetPackageFirstVisibleDeclaration (ThePackage);
      loop
         if Declaration = NullSymbol then
            TaskTypes := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsTaskType (Item) then
            TaskTypes := Iterator'(TaskTypeIterator,
                                   IsAbstract,
                                   Item,
                                   Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return TaskTypes;

   end FirstVisibleTaskType;

   --------------------------------------------------------------------------------

   function FirstPrivateTaskType (ThePackage : Symbol) return Iterator
   is
      Declaration        : Symbol;
      Item               : Symbol;
      TaskTypes          : Iterator;
   begin
      Declaration := RawDict.GetPackageFirstPrivateDeclaration (ThePackage);
      loop
         if Declaration = NullSymbol then
            TaskTypes := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsTaskType (Item) then
            TaskTypes := Iterator'(TaskTypeIterator,
                                   IsAbstract,
                                   Item,
                                   Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return TaskTypes;

   end FirstPrivateTaskType;

   --------------------------------------------------------------------------------

   function FirstOwnTask (ThePackage : Symbol) return Iterator is
      Context : Symbol;
      Result : Iterator;
   begin
      Context := RawDict.GetPackageTaskList (ThePackage);
      if Context = NullSymbol then
         Result := NullIterator;
      else
         Result := Iterator'(Discriminant => OwnTaskIterator,
                             Abstraction => IsAbstract,
                             Current => RawDict.GetOwnTaskVariable (Context),
                             Context => Context);
      end if;
      return Result;
   end FirstOwnTask;

   --------------------------------------------------------------------------------

   function FirstVisibleProtectedType (ThePackage : Symbol) return Iterator
   is
      Declaration        : Symbol;
      Item               : Symbol;
      ProtectedTypes     : Iterator;
   begin
      Declaration := RawDict.GetPackageFirstVisibleDeclaration (ThePackage);
      loop
         if Declaration = NullSymbol then
            ProtectedTypes := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         -- filter items to leave just protected types that are actually declared (not just announced)
         if IsProtectedType (Item) and then IsVisibleScope (GetScope (Item)) then
            ProtectedTypes := Iterator'(ProtectedTypeIterator,
                                        IsAbstract,
                                        Item,
                                        Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return ProtectedTypes;

   end FirstVisibleProtectedType;

   --------------------------------------------------------------------------------

   function FirstPrivateProtectedType (ThePackage : Symbol) return Iterator
   is
      Declaration        : Symbol;
      Item               : Symbol;
      ProtectedTypes     : Iterator;
   begin
      Declaration := RawDict.GetPackageFirstPrivateDeclaration (ThePackage);
      loop
         if Declaration = NullSymbol then
            ProtectedTypes := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsProtectedType (Item) then
            ProtectedTypes := Iterator'(ProtectedTypeIterator,
                                        IsAbstract,
                                        Item,
                                        Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return ProtectedTypes;

   end FirstPrivateProtectedType;

   --------------------------------------------------------------------------------

   function PackageRequiresBody (ThePackage : Symbol) return Boolean
   is
      It         : Iterator;
      SubProgSym : Symbol;
      Result     : Boolean;
   begin
      -- look for a visible subprgoram that needs a body
      if RawDict.GetPackageElaborateBodyFound (ThePackage) then
         Result := True;
      else
         Result := False;
         It := FirstVisibleSubprogram (ThePackage);
         while not IsNullIterator (It) loop
            SubProgSym := CurrentSymbol (It);
            if not (IsProofFunction (SubProgSym) or else
                    HasBody (SubProgSym))
            then
               Result := True;
               exit;
            end if;
            It := NextSymbol (It);
         end loop;
      end if;

      if not Result then
         -- look for a private subprogram that needs a body
         It := FirstPrivateSubprogram (ThePackage);
         while not IsNullIterator (It) loop
            SubProgSym := CurrentSymbol (It);
            if not (IsProofFunction (SubProgSym) or else
                      HasBody (SubProgSym))
            then
               Result := True;
               exit;
            end if;
            It := NextSymbol (It);
         end loop;
      end if;
      -- if we have not already seen enough to know a body is needed and we are
      -- in Ravenscar profile then see if there are any task or protected types to
      -- complete
      if not Result and then
        CommandLineData.RavenscarSelected then
         Result := FirstVisibleTaskType (ThePackage) /= NullIterator or else
           FirstPrivateTaskType (ThePackage) /= NullIterator or else
           FirstVisibleProtectedType (ThePackage) /= NullIterator or else
           FirstPrivateProtectedType (ThePackage) /= NullIterator;
      end if;
      return Result;
   end PackageRequiresBody;

   --------------------------------------------------------------------------------

   function FirstInterruptStreamMapping (Sym : Symbol) return Iterator is
      TheMappings : Symbol;
      It          : Iterator;
   begin
      TheMappings := RawDict.GetOwnVariableInterruptStreamMappings
        (RawDict.GetVariableOwnVariable (Sym));

      if TheMappings = NullSymbol then
         It := NullIterator;
      else
         It := Iterator'(InterruptStreamMappingIterator,
                         IsAbstract,
                         TheMappings,
                         TheMappings);
      end if;
      return It;
   end FirstInterruptStreamMapping;

   --------------------------------------------------------------------------------

   function GetInterruptStreamMappingHandler (TheMapping : in Symbol) return LexTokenManager.Lex_String is
   begin
      return RawDict.GetInterruptStreamMappingHandler (TheMapping);
   end GetInterruptStreamMappingHandler;

   --------------------------------------------------------------------------------

   function GetInterruptStreamMappingStream (TheMapping : in Symbol) return LexTokenManager.Lex_String is
   begin
      return RawDict.GetInterruptStreamMappingStream (TheMapping);
   end GetInterruptStreamMappingStream;

   --------------------------------------------------------------------------------

   function GetInterruptStreamVariable
     (ProtectedObject  : in Symbol;
      InterruptHandler : in Symbol) return Symbol
   is
      It        : Iterator;
      TheResult : Symbol;
   begin
      TheResult := NullSymbol;

      -- Check the PO/interrupt handler pair is valid
      if GetType (ProtectedObject) =
        GetRegion (GetScope (InterruptHandler)) and
        IsInterruptHandler (InterruptHandler) then

         -- Look for the handler name in the interrupt property mappings
         It := FirstInterruptStreamMapping (ProtectedObject);
         while not IsNullIterator (It) loop

            if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => GetInterruptStreamMappingHandler (CurrentSymbol (It)),
                                                                    Lex_Str2 => GetSimpleName (InterruptHandler)) = LexTokenManager.Str_Eq then

               -- Found it. Lookup the the stream variable name.
               TheResult := LookupItem (Name    => GetInterruptStreamMappingStream (CurrentSymbol (It)),
                                        Scope   => GetScope (ProtectedObject),
                                        Context => ProofContext);
               exit;
            end if;
            It := NextSymbol (It);
         end loop;

         if TheResult = NullSymbol then
            -- The default return value is the protected object itself.
            -- This is used if no maping exists for the given interrupt handler.
            TheResult := ProtectedObject;
         end if;

      else
         SystemErrors.FatalError (SystemErrors.OtherInternalError, "in GetInterruptStreamVariable");
      end if;
      return TheResult;
   end GetInterruptStreamVariable;

   --------------------------------------------------------------------------------

   function PartitionElaborationPolicyIsConcurrent return Boolean is
   begin
      return True;
   end PartitionElaborationPolicyIsConcurrent;

   --------------------------------------------------------------------------------


   -- TBD not yet usable for task types
   function FirstImportExport (Abstraction  : Abstractions;
                               TheProcedure : Symbol) return Iterator
   is
      SubprogramParameter : Symbol;
      GlobalVariable      : Symbol;
      Variable            : Symbol;
      ImportExports       : Iterator;
   begin

      SubprogramParameter := RawDict.GetSubprogramFirstParameter (TheProcedure);

      loop
         exit when SubprogramParameter = NullSymbol;
         exit when IsImportExport (Abstraction, TheProcedure, SubprogramParameter);
         SubprogramParameter := RawDict.GetNextSubprogramParameter (SubprogramParameter);
      end loop;

      if SubprogramParameter = NullSymbol then

         GlobalVariable := RawDict.GetSubprogramFirstGlobalVariable (Abstraction,
                                                                     TheProcedure);

         loop
            if GlobalVariable = NullSymbol then
               ImportExports := NullIterator;
               exit;
            end if;
            Variable := RawDict.GetGlobalVariableVariable (GlobalVariable);
            if IsImportExport (Abstraction, TheProcedure, Variable) then
               ImportExports := Iterator'(ImportExportIterator,
                                          Abstraction,
                                          Variable,
                                          GlobalVariable);
               exit;
            end if;
            GlobalVariable := RawDict.GetNextGlobalVariable (GlobalVariable);
         end loop;

      else
         ImportExports := Iterator'(ImportExportIterator,
                                    Abstraction,
                                    SubprogramParameter,
                                    NullSymbol);
      end if;

      return ImportExports;

   end FirstImportExport;

   --------------------------------------------------------------------------------

   function FirstExport (Abstraction  : Abstractions;
                         TheProcedure : Symbol) return Iterator
   is
      TheProcedureLocal   : Symbol;
      --PossibleGeneric     : Symbol;
      SubprogramParameter : Symbol;
      GlobalVariable      : Symbol;
      Variable            : Symbol;
      Exports             : Iterator;
   begin
      -- can now be called for task types as well and these have no parameters

      TheProcedureLocal := TheProcedure;
      if IsSubprogram (TheProcedureLocal) then
         -- if the procedure is an instantaion then use the identical flow relation of its generic source
         --PossibleGeneric := RawDict.GetSubprogramInstantiationOf (TheProcedureLocal);
         --if PossibleGeneric /= NullSymbol then
         --   TheProcedureLocal := PossibleGeneric;
         --end if;

         -- seek exports etc. as before
         SubprogramParameter := RawDict.GetSubprogramFirstParameter (TheProcedureLocal);
         GlobalVariable := RawDict.GetSubprogramFirstGlobalVariable (Abstraction,
                                                                     TheProcedureLocal);
      else -- task type
         SubprogramParameter := NullSymbol;
         GlobalVariable := RawDict.GetTaskTypeFirstGlobalVariable (Abstraction,
                                                                   TheProcedureLocal);
      end if;

      if SubprogramParameter /= NullSymbol then

         Exports := Iterator'(ExportIterator,
                              Abstraction,
                              SubprogramParameter,
                              NullSymbol);

         if not IsExport (Abstraction, TheProcedureLocal, SubprogramParameter) then
            Exports := NextSymbol (Exports);
         end if;

      elsif GlobalVariable /= NullSymbol then

         Variable := RawDict.GetGlobalVariableVariable (GlobalVariable);
         Exports := Iterator'(ExportIterator,
                              Abstraction,
                              Variable,
                              GlobalVariable);

         if not IsExport (Abstraction, TheProcedureLocal, Variable) then
            Exports := NextSymbol (Exports);
         end if;

      else
         Exports := NullIterator;
      end if;

      return Exports;

   end FirstExport;

   --------------------------------------------------------------------------------

   function FirstImport (Abstraction  : Abstractions;
                         TheProcedure : Symbol) return Iterator
   is
      SubprogramParameter : Symbol;
      GlobalVariable      : Symbol;
      Variable            : Symbol;
      Imports             : Iterator;
   begin

      -- can now be called for task types as well and these have no parameters
      if IsSubprogram (TheProcedure) then
         SubprogramParameter := RawDict.GetSubprogramFirstParameter (TheProcedure);
         GlobalVariable := RawDict.GetSubprogramFirstGlobalVariable (Abstraction,
                                                                     TheProcedure);
      else -- task type
         SubprogramParameter := NullSymbol;
         GlobalVariable := RawDict.GetTaskTypeFirstGlobalVariable (Abstraction,
                                                                   TheProcedure);
      end if;

      if SubprogramParameter /= NullSymbol then

         Imports := Iterator'(ImportIterator,
                              Abstraction,
                              SubprogramParameter,
                              NullSymbol);

         if not IsImport (Abstraction, TheProcedure, SubprogramParameter) then
            Imports := NextSymbol (Imports);
         end if;

      elsif GlobalVariable /= NullSymbol then

         Variable := RawDict.GetGlobalVariableVariable (GlobalVariable);
         Imports := Iterator'(ImportIterator,
                              Abstraction,
                              Variable,
                              GlobalVariable);

         if not IsImport (Abstraction, TheProcedure, Variable) then
            Imports := NextSymbol (Imports);
         end if;

      else
         Imports := NullIterator;
      end if;

      return Imports;

   end FirstImport;

   --------------------------------------------------------------------------------

   function FirstDependency (Abstraction  : Abstractions;
                             TheProcedure : Symbol;
                             TheExport    : Symbol) return Iterator
   is
      Dependency   : Symbol;
      Dependencies : Iterator;
      TheProcedureLocal   : Symbol;
      --PossibleGeneric     : Symbol;
   begin
      -- Before searching for dependencies, check if proc is an instnatiation of a generic.  If so,
      -- use the flow relations of the generic procedure instead.
      TheProcedureLocal := TheProcedure;
      --if IsSubprogram (TheProcedureLocal) then
         --PossibleGeneric := RawDict.GetSubprogramInstantiationOf (TheProcedureLocal);
         --if PossibleGeneric /= NullSymbol then
         --   TheProcedureLocal := PossibleGeneric;
         --end if;
      --end if;

      -- look up dependencies as before
      if IsFormalParameter (TheProcedureLocal, TheExport) then
         Dependency := RawDict.GetSubprogramParameterDependencies (Abstraction,
                                                                   TheExport);
      else
         Dependency := RawDict.GetGlobalVariableDependencies (Abstraction,
                                                              GetGlobalVariable (Abstraction,
                                                                                 TheProcedureLocal,
                                                                                 TheExport));
      end if;

      if Dependency = NullSymbol then
         Dependencies := NullIterator;
      else
         Dependencies := Iterator'(DependencyIterator,
                                   IsAbstract,
                                   RawDict.GetDependencyImport (Dependency),
                                   Dependency);
      end if;

      return Dependencies;

   end FirstDependency;

   --------------------------------------------------------------------------------

   procedure InstantiateSubprogramParameters (GenericSubprogramSym : in Symbol;
                                              ActualSubprogramSym  : in Symbol;
                                              Comp_Unit            : in ContextManager.UnitDescriptors;
                                              Declaration          : in Location)
   is separate;

   --------------------------------------------------------------------------------


   function FirstOwnedPackage (ThePackage : Symbol) return Iterator
   is
      Item          : Symbol;
      Descendent    : Symbol;
      OwnedPackages : Iterator;
   begin
      Item := RawDict.GetPackageFirstChild (ThePackage, PrivateChild);

      if Item = NullSymbol then
         OwnedPackages := NullIterator;
      else
         loop
            Descendent := RawDict.GetPackageFirstChild (Item, Public);
            exit when Descendent = NullSymbol;
            Item := Descendent;
         end loop;
         OwnedPackages := Iterator'(OwnedPackageIterator,
                                    IsAbstract,
                                    Item,
                                    ThePackage);
      end if;

      return OwnedPackages;

   end FirstOwnedPackage;

   --------------------------------------------------------------------------------

   function FirstEmbeddedPackage (CompilationUnit : Symbol) return Iterator
   is
      Declaration      : Symbol;
      Item             : Symbol;
      EmbeddedPackages : Iterator;
   begin

      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when PackageSymbol =>
            Declaration := RawDict.GetPackageFirstLocalDeclaration (CompilationUnit);
         when SubprogramSymbol =>
            Declaration := RawDict.GetSubprogramFirstDeclaration (CompilationUnit);
         when TypeSymbol =>
            if RawDict.GetTypeDiscriminant (CompilationUnit) = TaskType then
               Declaration := RawDict.GetTaskTypeFirstLocalDeclaration (CompilationUnit);
            else
               Declaration := NullSymbol;
            end if;
         when others =>
            Declaration := NullSymbol;
      end case;

      loop
         if Declaration = NullSymbol then
            EmbeddedPackages := NullIterator;
            exit;
         end if;
         Item := RawDict.GetDeclarationItem (Declaration);
         if IsPackage (Item) and then
            Declaration /= RawDict.GetPackageBody (Item) then
            EmbeddedPackages := Iterator'(EmbeddedPackageIterator,
                                          IsAbstract,
                                          Item,
                                          Declaration);
            exit;
         end if;
         Declaration := RawDict.GetNextDeclaration (Declaration);
      end loop;

      return EmbeddedPackages;

   end FirstEmbeddedPackage;

   --------------------------------------------------------------------------------

   function FirstInitializedOwnVariable (ThePackage : Symbol) return Iterator
   is
      OwnVariables, InitializedOwnVariables : Iterator;
   begin

      OwnVariables := FirstOwnVariable (ThePackage);

      loop
         if IsNullIterator (OwnVariables) then
            InitializedOwnVariables := NullIterator;
            exit;
         end if;
         if OwnVariableIsInitialized (CurrentSymbol (OwnVariables)) then
            InitializedOwnVariables := Iterator'(InitializedOwnVariableIterator,
                                                 IsAbstract,
                                                 OwnVariables.Current,
                                                 NullSymbol);
            exit;
         end if;
         OwnVariables := NextSymbol (OwnVariables);
      end loop;

      return InitializedOwnVariables;

   end FirstInitializedOwnVariable;

   --------------------------------------------------------------------------------

   function FirstSuspendsListItem (TheTaskOrProc : Symbol) return Iterator
   is
      SuspendsList : Symbol;
      It           : Iterator;
   begin
      if TheTaskOrProc = NullSymbol then
         -- It might be Null owing to a previous error, such as a mal-formed
         -- stub.
         It := NullIterator;
      else

         if IsSubprogram (TheTaskOrProc) then
            SuspendsList := RawDict.GetProcedureSuspendsList (TheTaskOrProc);
         else
            SuspendsList := RawDict.GetTaskTypeSuspendsList (TheTaskOrProc);
         end if;

         if SuspendsList = NullSymbol then
            It := NullIterator;
         else
            It := Iterator'(SuspendsListItemIterator,
                            IsAbstract,
                            RawDict.GetSuspendsListItem (SuspendsList),
                            SuspendsList);
         end if;
      end if;
      return It;
   end FirstSuspendsListItem;

   --------------------------------------------------------------------------------

   function FirstVirtualElement (TheProtectedType : Symbol) return Iterator
   is
      VirtualElementList : Symbol;
      It                 : Iterator;
   begin
      VirtualElementList := RawDict.GetVirtualElementList (TheProtectedType);

      if VirtualElementList = NullSymbol then
         It := NullIterator;
      else
         It := Iterator'(VirtualElementIterator,
                         IsAbstract,
                         RawDict.GetVirtualElementVariable (VirtualElementList),
                         VirtualElementList);
      end if;
      return It;
   end FirstVirtualElement;

   --------------------------------------------------------------------------------

   procedure SetHasDelayProperty (TheProcedure : in Symbol)
   is
   begin
      RawDict.SetHasDelayProperty (TheProcedure);
   end SetHasDelayProperty;

   --------------------------------------------------------------------------------

   procedure SetUsesUnprotectedVariables (Sym : in Symbol)
   is
   begin
      if RawDict.GetSymbolDiscriminant (Sym) = SubprogramSymbol then
         RawDict.SetSubprogUsesUnprotectedVariables (Sym);
      else
         RawDict.SetTaskUsesUnprotectedVariables (Sym);
      end if;
   end SetUsesUnprotectedVariables;

   --------------------------------------------------------------------------------

   procedure SetUnprotectedReference (Variable : in Symbol;
                                      ByThread : in Symbol) is
   begin
      RawDict.SetUnprotectedReference (RawDict.GetVariableOwnVariable (Variable), ByThread);
   end SetUnprotectedReference;

   --------------------------------------------------------------------------------

   procedure SetSuspendsReference (Variable : in Symbol;
                                   ByThread : in Symbol) is
   begin
      RawDict.SetSuspendsReference (RawDict.GetVariableOwnVariable (Variable), ByThread);
   end SetSuspendsReference;

   --------------------------------------------------------------------------------

   procedure SetVirtualElementSeenByOwner (TheVariable : in Symbol)
   is
   begin
      RawDict.SetVirtualElementSeenByOwner
        (RawDict.GetVariableVirtualElement (TheVariable));
   end SetVirtualElementSeenByOwner;

   --------------------------------------------------------------------------------

   function HasDelayProperty (TheProcedure : Symbol) return Boolean
   is
   begin
      return RawDict.HasDelayProperty (TheProcedure);
   end HasDelayProperty;

   --------------------------------------------------------------------------------

   function SubprogramMayBlock (Subprogram : Symbol) return Boolean
   is
   begin
      return IsPredefinedSuspendUntilTrueOperation (Subprogram) or
        IsEntry (Subprogram) or
        HasDelayProperty (Subprogram) or
        not IsNullIterator (FirstSuspendsListItem (Subprogram));
   end SubprogramMayBlock;

   --------------------------------------------------------------------------------

   function BodyIsHidden (Sym : Symbol) return Boolean
   is
      Result : Boolean;
   begin
      if IsSubprogram (Sym) then
         Result := RawDict.ProcedureBodyIsHidden (Sym);
      else
         Result := RawDict.TaskBodyIsHidden (Sym);
      end if;
      return Result;
   end BodyIsHidden;

   --------------------------------------------------------------------------------

   function DelayPropertyIsAccountedFor (TheProcedure : Symbol) return Boolean
   is
   begin
      return RawDict.DelayPropertyIsAccountedFor (TheProcedure);
   end DelayPropertyIsAccountedFor;

   --------------------------------------------------------------------------------

   function UltimateSubject (OwnVariable : Symbol) return Symbol
   --# global in Dict;
   is
      TempSym : Symbol;
   begin
      TempSym := OwnVariable;
      while IsConstituent (TempSym) loop
         TempSym := GetSubject (TempSym);
      end loop;
      return TempSym;
   end UltimateSubject;

   --------------------------------------------------------------------------------

   procedure SetIsSuspendable (Variable : in Symbol)
   is
   begin
      RawDict.SetIsSuspendable (RawDict.GetVariableOwnVariable (Variable));
   end SetIsSuspendable;

   --------------------------------------------------------------------------------

   function GetIsSuspendable (Variable : in Symbol) return Boolean
   is
   begin
      return RawDict.GetIsSuspendable (RawDict.GetVariableOwnVariable (UltimateSubject (Variable)));
   end GetIsSuspendable;

   --------------------------------------------------------------------------------

   procedure SetHasInterruptProperty (Variable : in Symbol)
   is
   begin
      RawDict.SetHasInterruptProperty (RawDict.GetVariableOwnVariable (Variable));
   end SetHasInterruptProperty;

   --------------------------------------------------------------------------------

   function GetHasInterruptProperty (Variable : in Symbol) return Boolean
   is
   begin
      return RawDict.GetHasInterruptProperty
        (RawDict.GetVariableOwnVariable (UltimateSubject (Variable)));
   end GetHasInterruptProperty;

   --------------------------------------------------------------------------------

   function GetVirtualElementOwner (Variable : in Symbol) return Symbol
   is
   begin
      return RawDict.GetVirtualElementOwner
        (RawDict.GetVariableVirtualElement (Variable));
   end GetVirtualElementOwner;

   --------------------------------------------------------------------------------

   function IsVirtualElement (Variable : in Symbol) return Boolean
   is
   begin
      return RawDict.GetVariableVirtualElement (Variable) /= NullSymbol;
   end IsVirtualElement;

   --------------------------------------------------------------------------------

   function IsVirtualElementForType (TheVariable      : in Symbol;
                                     TheProtectedType : in Symbol) return Boolean is
   begin
      return IsVirtualElement (TheVariable) and then
        GetRootType (GetType (GetVirtualElementOwner (TheVariable))) =
        TheProtectedType;
   end IsVirtualElementForType;

   --------------------------------------------------------------------------------

   function VirtualElementSeenByOwner (Variable : in Symbol) return Boolean
   is
   begin
      return RawDict.GetVirtualElementSeenByOwner
        (RawDict.GetVariableVirtualElement (Variable));
   end VirtualElementSeenByOwner;

   --------------------------------------------------------------------------------

   procedure SetPriorityProperty (OwnVariable : in Symbol;
                                  TheValue    : in LexTokenManager.Lex_String)
   is
   begin
      RawDict.SetPriorityProperty (RawDict.GetVariableOwnVariable (OwnVariable), TheValue);
   end SetPriorityProperty;

   --------------------------------------------------------------------------------

   procedure SetIntegrityProperty (OwnVariable : in Symbol;
                                   TheValue    : in LexTokenManager.Lex_String)
   is
   begin
      RawDict.SetIntegrityProperty (RawDict.GetVariableOwnVariable (OwnVariable), TheValue);
   end SetIntegrityProperty;

   --------------------------------------------------------------------------------

   function GetPriorityProperty (OwnVariable : in Symbol)
                                return LexTokenManager.Lex_String
   is
   begin
      return RawDict.GetPriorityProperty
        (RawDict.GetVariableOwnVariable (UltimateSubject (OwnVariable)));
   end GetPriorityProperty;

   --------------------------------------------------------------------------------

   function GetIntegrityProperty (S : in Symbol)
                                 return LexTokenManager.Lex_String
   is
      R : LexTokenManager.Lex_String;
   begin
      if IsOwnVariable (S) then
         R := RawDict.GetIntegrityProperty (RawDict.GetVariableOwnVariable (S));
      else
         R := LexTokenManager.Null_String;
      end if;
      return R;
   end GetIntegrityProperty;


   function RelationViolatesInfoFlowPolicy (TheExport : in Symbol;
                                            TheImport : in Symbol) return Boolean
   is
      ExpILexToken  : LexTokenManager.Lex_String;
      ImpILexToken  : LexTokenManager.Lex_String;
      ExpIVal       : Maths.Value;
      ImpIVal       : Maths.Value;
      ExpI          : Natural;
      ImpI          : Natural;
      UnusedFlag    : Maths.ErrorCode;
      Result        : Boolean;
      DefinedPolicy : CommandLineData.DefinedInfoFlowPolicies;

      type DefaultIntegrities is array (CommandLineData.DefinedInfoFlowPolicies) of
        Natural;

      DefaultExportIntegrities : constant DefaultIntegrities := DefaultIntegrities'
        (CommandLineData.Safety   => Natural'Last,
         CommandLineData.Security => Natural'First);

      DefaultImportIntegrities : constant DefaultIntegrities := DefaultIntegrities'
        (CommandLineData.Safety   => Natural'First,
         CommandLineData.Security => Natural'Last);

   begin
      -- Only compare integrity levels for own variables.  This avoids
      -- generating many spurious errors for interior, hidden
      -- information flows involving local variables.  These flows, of course,
      -- exhibit themselves in the checking of a subprograms imports
      -- and exports, so are not lost.
      if IsOwnVariable (TheImport) and
        IsOwnVariable (TheExport) then

         case CommandLineData.Content.InfoFlowPolicy is
            when CommandLineData.None =>
               Result := False;
            when CommandLineData.Safety |
              CommandLineData.Security =>

               DefinedPolicy := CommandLineData.Content.InfoFlowPolicy;

               ExpILexToken := GetIntegrityProperty (TheExport);
               ImpILexToken := GetIntegrityProperty (TheImport);

               if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ExpILexToken,
                                                                       Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq and then
                 LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ImpILexToken,
                                                                      Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
                  -- Both unspecified local variables, so anything goes...
                  Result := False;
               else

                  if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ExpILexToken,
                                                                          Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
                     ExpI := DefaultExportIntegrities (DefinedPolicy);
                  else
                     ExpIVal := Maths.ValueRep (ExpILexToken);
                     --# accept Flow, 10, UnusedFlag, "Expected ineffective assignment to UnusedFlag";
                     Maths.ValueToInteger (ExpIVal, ExpI, UnusedFlag);
                     --# end accept;
                  end if;

                  if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ImpILexToken,
                                                                          Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
                     ImpI := DefaultImportIntegrities (DefinedPolicy);
                  else
                     ImpIVal := Maths.ValueRep (ImpILexToken);
                     --# accept Flow, 10, UnusedFlag, "Expected ineffective assignment to UnusedFlag";
                     Maths.ValueToInteger (ImpIVal, ImpI, UnusedFlag);
                     --# end accept;
                  end if;

                  case DefinedPolicy is
                     when CommandLineData.Security =>
                        Result := ExpI < ImpI;
                     when CommandLineData.Safety =>
                        Result := ExpI > ImpI;
                  end case;
               end if;
         end case;
      else
         Result := False;
      end if;
      --# accept Flow, 33, UnusedFlag, "Expected to be neither referenced nor exported";
      return Result;
   end RelationViolatesInfoFlowPolicy;



   --------------------------------------------------------------------------------

   function HasValidPriorityProperty (OwnVariable : Symbol) return Boolean
   is
   begin
      return RawDict.HasValidPriorityProperty
        (RawDict.GetVariableOwnVariable (UltimateSubject (OwnVariable)));
   end HasValidPriorityProperty;

   --------------------------------------------------------------------------------

   function IsThread (Sym : in Symbol) return Boolean
   is
   begin
      return IsTaskType (Sym) or
        IsMainProgram (Sym) or
        IsInterruptHandler (Sym);
   end IsThread;

   --------------------------------------------------------------------------------

   function UsesUnprotectedVariables (Sym : in Symbol) return Boolean
   is
      Result : Boolean;
   begin
      if RawDict.GetSymbolDiscriminant (Sym) = SubprogramSymbol then
         Result := RawDict.SubprogUsesUnprotectedVariables (Sym);
      else
         Result := RawDict.TaskUsesUnprotectedVariables (Sym);
      end if;
      return Result;
   end UsesUnprotectedVariables;

   --------------------------------------------------------------------------------

   function GetUnprotectedReference (Variable : in Symbol) return Symbol
   is
   begin
      return RawDict.GetUnprotectedReference (RawDict.GetVariableOwnVariable (Variable));
   end GetUnprotectedReference;

   --------------------------------------------------------------------------------

   function GetSuspendsReference (Variable : in Symbol) return Symbol
   is
   begin
      return RawDict.GetSuspendsReference (RawDict.GetVariableOwnVariable (Variable));
   end GetSuspendsReference;

   --------------------------------------------------------------------------------

   function IsOrIsRefinement (Sym1, Sym2 : Symbol) return Boolean
   --# global in Dict;
   is
      -- Returns True if Sym1 and Sym2 are the same symbol or Sym2 is a refinement of Sym1
   begin
      return (Sym1 = Sym2) or else
        (IsOwnVariable (Sym1) and then
           IsVariable (Sym2) and then
           IsRefinement (Subject => Sym1,
                         Constituent => Sym2));
   end IsOrIsRefinement;

   --------------------------------------------------------------------------------

   function GetActual (TheFormal  : Symbol;
                       TheSubtype : Symbol) return Symbol
   is
      Result     : Symbol;
      FormalsIt  : Iterator;
      ActualsIt  : Iterator;
   begin
      Result := NullSymbol;
      FormalsIt := FirstKnownDiscriminant (TheSubtype);
      ActualsIt := FirstDiscriminantConstraint (TheSubtype);
      while not IsNullIterator (FormalsIt) loop
         if CurrentSymbol (FormalsIt) = TheFormal then
            Result := CurrentSymbol (ActualsIt);
            exit;
         end if;
         FormalsIt := NextSymbol (FormalsIt);
         ActualsIt := NextSymbol (ActualsIt);
      end loop;
      return Result;
   end GetActual;

   --------------------------------------------------------------------------------

   function SuspendsOn (TheTaskOrProc : Symbol;
                        ThePOorSO     : Symbol) return Boolean
   is
      Result     : Boolean;
      It         : Iterator;
   begin
      Result := False;
      It := FirstSuspendsListItem (TheTaskOrProc);
      while not IsNullIterator (It) loop
         Result := IsOrIsRefinement (CurrentSymbol (It), ThePOorSO);
         exit when Result;
         It := NextSymbol (It);
      end loop;
      return Result;
   end SuspendsOn;

   --------------------------------------------------------------------------------

   function SuspendsListItemIsAccountedFor (TheTaskOrProc : Symbol;
                                            ThePOorSO     : Symbol) return Boolean
   is
      Result : Boolean;
      It     : Iterator;
   begin
      Result := False;
      It     := FirstSuspendsListItem (TheTaskOrProc);
      while not IsNullIterator (It) loop
         if IsOrIsRefinement (CurrentSymbol (It), ThePOorSO) then
            Result := RawDict.GetSuspendsListItemIsAccountedFor (It.Context);
            exit;
         end if;
         It := NextSymbol (It);
      end loop;
      return Result;
   end SuspendsListItemIsAccountedFor;

   --------------------------------------------------------------------------------

   procedure MarkAccountsForSuspendsListItem (TheTaskOrProc : in Symbol;
                                              ThePOorSO     : in Symbol)
   is
      It : Iterator;
   begin
      It := FirstSuspendsListItem (TheTaskOrProc);
      while not IsNullIterator (It) loop
         if IsOrIsRefinement (CurrentSymbol (It), ThePOorSO) then
            RawDict.SetSuspendsListItemIsAccountedFor (It.Context);
            exit;
         end if;
         It := NextSymbol (It);
      end loop;
   end MarkAccountsForSuspendsListItem;

   --------------------------------------------------------------------------------

   procedure MarkAccountsForSuspendsListItems (TheTaskOrProc       : in Symbol;
                                               TheItemsInProcedure : in Symbol)
   is
      It : Iterator;
   begin
      It := FirstSuspendsListItem (TheItemsInProcedure);
      while not IsNullIterator (It) loop
         MarkAccountsForSuspendsListItem (TheTaskOrProc, CurrentSymbol (It));
         It := NextSymbol (It);
      end loop;
   end MarkAccountsForSuspendsListItems;

   --------------------------------------------------------------------------------

   procedure MarkAccountsForDelay (TheProcedure : in Symbol)
   is
   begin
      RawDict.MarkAccountsForDelay (TheProcedure);
   end MarkAccountsForDelay;

   --------------------------------------------------------------------------------

   function SuspendsListIsPropagated (FromProcedure : Symbol;
                                      ToTaskOrProc  : Symbol) return Boolean
   is
      Result : Boolean;
      It : Iterator;
   begin
      Result := True;
      It := FirstSuspendsListItem (FromProcedure);
      while not IsNullIterator (It) loop
         Result := SuspendsOn (TheTaskOrProc => ToTaskOrProc,
                               ThePOorSO => CurrentSymbol (It));
         exit when not Result;
         It := NextSymbol (It);
      end loop;
      return Result;
   end SuspendsListIsPropagated;

   --------------------------------------------------------------------------------

   function GetInheritDepth (FieldName    : LexTokenManager.Lex_String;
                             RecordType   : Symbol) return Natural
   is separate;

   --------------------------------------------------------------------------------

   procedure ReportUsage
   is
   begin
      -- dictionary symbols are never recycled, so the 'Current' symbol
      -- gives the maximum usage
      Statistics.SetTableUsage (Statistics.SymbolTable,
                                Integer (Dict.Symbols.Current));
   end ReportUsage;

   --------------------------------------------------------------------------------

   procedure Initialize (WriteToFile : in Boolean) is separate;

   --------------------------------------------------------------------------------

   package body TargetData is separate;

   --------------------------------------------------------------------------------

   procedure ReadTargetDataFile
   is
   begin
      TargetData.ReadTargetDataFile;
   end ReadTargetDataFile;

   procedure OutputTargetDataFile (ToFile : in SPARK_IO.File_Type)
   is
   begin
      TargetData.OutputTargetDataFile (ToFile);
   end OutputTargetDataFile;

end Dictionary;
