-- $Id: dictionary-write.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Maths;
with FileSystem;
with ELStrings;

separate (Dictionary)
procedure Write (FileName : in     EStrings.T;
                 Status   :    out SPARK_IO.File_Status)
is

   File          : SPARK_IO.File_Type := SPARK_IO.Null_File;
   FileStatus    : SPARK_IO.File_Status;
   LocalFileName : EStrings.T;

   --------------------------------------------------------------------------------

   procedure WriteLibraryUnits (File : in SPARK_IO.File_Type)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                File,
      --#                                LexTokenManager.State;
   is

      LibraryUnits : Iterator;

      procedure WriteGenericFormalParameters (File       : in SPARK_IO.File_Type;
                                              TheGeneric : in Symbol)
      --# global in     Dict;
      --#        in     LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Dict,
      --#                                File,
      --#                                LexTokenManager.State,
      --#                                TheGeneric;
      is

         ParameterIt : Iterator;
         Number      : Positive;

         --------------------------------------------------------------------------------

         procedure WriteGenericFormalParameter (File       : in SPARK_IO.File_Type;
                                                TheGeneric : in Symbol;
                                                Number     : in Positive;
                                                Parameter  : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                Number,
         --#                                Parameter,
         --#                                TheGeneric;
         is

            procedure WriteGenericType (File      : in SPARK_IO.File_Type;
                                        Parameter : in Symbol)

            --# global in     Dict;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                Parameter;
            is
               TheType : Symbol;
            begin
               TheType := RawDict.GetGenericParameterTypeOrObject (Parameter);
               case RawDict.GetTypeDiscriminant (TheType) is
                  when GenericLimitedType =>
                     WriteString (File, "generic limited ");
                  when GenericPrivateType =>
                     WriteString (File, "generic private ");
                  when GenericOrderedDiscreteType =>
                     WriteString (File, "generic dicrete (ordered) ");
                  when GenericUnorderedDiscreteType =>
                     WriteString (File, "generic discrete ");
                  when GenericIntegerType =>
                     WriteString (File, "generic integer ");
                  when GenericModularType =>
                     WriteString (File, "generic modular ");
                  when GenericFloatingPointType =>
                     WriteString (File, "generic floating point ");
                  when GenericFixedPointType =>
                     WriteString (File, "generic fixed point ");
                  when GenericArrayType         =>
                     WriteString (File, "generic array ");
                  when others =>
                     WriteString (File, "invalid generic type ");
               end case;
            end WriteGenericType;

         begin -- WriteGenericFormalParameter
            WriteString (File, "generic formal parameter # ");
            WriteInteger (File, Number);
            WriteString (File, " of ");
            WriteName (File, TheGeneric);
            WriteString (File, " is ");
            WriteSimpleName (File, Parameter);
            WriteString (File, " which is ");
            case RawDict.GetGenericParameterKind (Parameter) is
               when GenericTypeParameter =>
                  WriteString (File, "a ");
                  WriteGenericType (File, Parameter);
                  WriteString (File, "type parameter");
               when GenericObjectParameter =>
                  WriteString (File, "a generic object parameter of type ");
                  WriteName (File, GetType (RawDict.GetGenericParameterTypeOrObject (Parameter)));
            end case;
            WriteLine (File, " ;");
         end WriteGenericFormalParameter;

         --------------------------------------------------------------------------------

      begin
         ParameterIt := FirstGenericFormalParameter (TheGeneric);
         Number := 1;
         loop
            exit when IsNullIterator (ParameterIt);
            WriteGenericFormalParameter (File,
                                         TheGeneric,
                                         Number,
                                         CurrentSymbol (ParameterIt));
            ParameterIt := NextSymbol (ParameterIt);
            Number := Number + 1;
         end loop;
      end WriteGenericFormalParameters;

      --------------------------------------------------------------------------------

      function FirstLibraryUnit return Iterator
         --# global in Dict;
      is
         Declaration  : Symbol;
         Item         : Symbol;
         Found        : Boolean;
         LibraryUnits : Iterator;
      begin

         Declaration := RawDict.GetPackageFirstLocalDeclaration (GetPredefinedPackageStandard);

         loop
            if Declaration = NullSymbol then
               LibraryUnits := NullIterator;
               exit;
            end if;
            Item := RawDict.GetDeclarationItem (Declaration);
            case RawDict.GetSymbolDiscriminant (Item) is
               when PackageSymbol =>
                  Found := Declaration /= RawDict.GetPackageBody (Item);
               when others =>
                  Found := Declaration /= RawDict.GetSubprogramBody (Item);
            end case;
            if Found then
               LibraryUnits := Iterator'(LibraryUnitIterator,
                                         IsAbstract,
                                         Item,
                                         Declaration);
               exit;
            end if;
            Declaration := RawDict.GetNextDeclaration (Declaration);
         end loop;

         return LibraryUnits;

      end FirstLibraryUnit;

      --------------------------------------------------------------------------------

      procedure WriteLibraryUnit (File : in SPARK_IO.File_Type;
                                  Unit : in Symbol)
         --# global in     Dict;
         --#        in     LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Dict,
         --#                                File,
         --#                                LexTokenManager.State,
         --#                                Unit;
      is
         --# hide WriteLibraryUnit;

         --------------------------------------------------------------------------------

         procedure WriteWithReferences (File  : in SPARK_IO.File_Type;
                                        Scope : in Scopes)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                Scope;
         is

            WithReference : Iterator;

            --------------------------------------------------------------------------------

            function FirstWithedPackage (Scope : Scopes) return Iterator
               --# global in Dict;
            is
               ContextClause  : Symbol;
               Region         : Symbol;
               WithedPackages : Iterator;
            begin

               case Scope.TypeOfScope is
                  when Visible =>
                     ContextClause := RawDict.GetPackageVisibleWithClauses (GetRegion (Scope));
                  when others =>
                     Region := GetRegion (Scope);
                     case RawDict.GetSymbolDiscriminant (Region) is
                        when PackageSymbol =>
                           ContextClause := RawDict.GetPackageLocalWithClauses (Region);
                        when others =>
                           ContextClause := RawDict.GetSubprogramWithClauses (Region);
                     end case;
               end case;

               if ContextClause = NullSymbol then
                  WithedPackages := NullIterator;
               else
                  WithedPackages := Iterator'(WithedPackageIterator,
                                              IsAbstract,
                                              RawDict.GetContextClausePackage (ContextClause),
                                              ContextClause);
               end if;

               return WithedPackages;

            end FirstWithedPackage;

            --------------------------------------------------------------------------------

            procedure WriteWithReference (File       : in SPARK_IO.File_Type;
                                          ThePackage : in Symbol;
                                          Scope      : in Scopes)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                Scope,
               --#                                ThePackage;
            is
            begin
               WriteString (File, "with reference to ");
               WriteSimpleName (File, ThePackage);
               WriteString (File, " in ");
               WriteScope (File, Scope);
               WriteLine (File, " ;");
            end WriteWithReference;

            --------------------------------------------------------------------------------

         begin
            WithReference := FirstWithedPackage (Scope);
            loop
               exit when IsNullIterator (WithReference);
               WriteWithReference (File, CurrentSymbol (WithReference), Scope);
               WithReference := NextSymbol (WithReference);
            end loop;
         end WriteWithReferences;

         --------------------------------------------------------------------------------

         procedure WriteInheritsReferences (File            : in SPARK_IO.File_Type;
                                            CompilationUnit : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                CompilationUnit,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State;
         is

            InheritsReference : Iterator;

            --------------------------------------------------------------------------------

            function FirstInheritedPackage (CompilationUnit : Symbol) return Iterator
               --# global in Dict;
            is
               ContextClause     : Symbol;
               InheritedPackages : Iterator;
            begin

               if IsPackage (CompilationUnit) then
                  ContextClause := RawDict.GetPackageInheritClauses (CompilationUnit);
               elsif IsMainProgram (CompilationUnit) then
                  ContextClause := Dict.Main.InheritClauses;
               else
                  ContextClause := NullSymbol;
               end if;

               if ContextClause = NullSymbol then
                  InheritedPackages := NullIterator;
               else
                  InheritedPackages := Iterator'(InheritedPackageIterator,
                                                 IsAbstract,
                                                 RawDict.GetContextClausePackage (ContextClause),
                                                 ContextClause);
               end if;

               return InheritedPackages;

            end FirstInheritedPackage;

            --------------------------------------------------------------------------------

            procedure WriteInheritsReference (File            : in SPARK_IO.File_Type;
                                              ThePackage      : in Symbol;
                                              CompilationUnit : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                CompilationUnit,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                ThePackage;
            is
            begin
               WriteString (File, "inherits reference to ");
               WriteSimpleName (File, ThePackage);
               WriteString (File, " in ");
               WriteName (File, CompilationUnit);
               WriteLine (File, " ;");
            end WriteInheritsReference;

            --------------------------------------------------------------------------------

         begin
            InheritsReference := FirstInheritedPackage (CompilationUnit);
            loop
               exit when IsNullIterator (InheritsReference);
               WriteInheritsReference (File,
                                       CurrentSymbol (InheritsReference),
                                       CompilationUnit);
               InheritsReference := NextSymbol (InheritsReference);
            end loop;
         end WriteInheritsReferences;

         --------------------------------------------------------------------------------

         procedure WritePackageInfo (File       : in SPARK_IO.File_Type;
                                     ThePackage : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                ThePackage;
         is
            Scope : Scopes;
         begin
            if IsGenericPackage (ThePackage) then
               WriteString (File, "generic ");
            end if;

            if RawDict.GetPackageParent (ThePackage) /= NullSymbol then
               WriteString (File, "child ");
            end if;

            WriteString (File, "package named ");
            WriteSimpleName (File, ThePackage);
            Scope := GetScope (ThePackage);
            if GetRegion (Scope) /= GetPredefinedPackageStandard then
               WriteString (File, " declared in ");
               WriteScope (File, Scope);
            end if;
            WriteLine (File, " ;");
         end WritePackageInfo;

         --------------------------------------------------------------------------------

         procedure WriteSubprogram (File       : in SPARK_IO.File_Type;
                                    Subprogram : in Symbol);

         --------------------------------------------------------------------------------

         procedure WritePackage (File       : in SPARK_IO.File_Type;
                                 ThePackage : in Symbol);

         --------------------------------------------------------------------------------

         procedure WriteGlobalVariables (Abstraction : in Abstractions;
                                         File        : in SPARK_IO.File_Type;
                                         Subprogram  : in Symbol);

         --------------------------------------------------------------------------------

         procedure WriteDependencyClauses (Abstraction  : in Abstractions;
                                           File         : in SPARK_IO.File_Type;
                                           TheProcedure : in Symbol);

         --------------------------------------------------------------------------------

         function FirstDeclarativeItem (Scope : Scopes) return Iterator
            --# global in Dict;
         is
            Declaration      : Symbol;
            Region           : Symbol;
            Item             : Symbol;
            Found            : Boolean;
            DeclarativeItems : Iterator;
         begin

            case Scope.TypeOfScope is
               when Visible =>
                  Declaration := RawDict.GetPackageFirstVisibleDeclaration (GetRegion (Scope));
               when Local =>
                  Region := GetRegion (Scope);
                  case RawDict.GetSymbolDiscriminant (Region) is
                     when PackageSymbol =>
                        Declaration := RawDict.GetPackageFirstLocalDeclaration (Region);
                     when SubprogramSymbol =>
                        Declaration := RawDict.GetSubprogramFirstDeclaration (Region);
                     when others =>
                        Declaration := NullSymbol;
                  end case;
               when Privat =>
                  Declaration := RawDict.GetPackageFirstPrivateDeclaration (GetRegion (Scope));
            end case;

            loop
               exit when Declaration = NullSymbol;
               Item := RawDict.GetDeclarationItem (Declaration);
               case RawDict.GetSymbolDiscriminant (Item) is
                  when PackageSymbol =>
                     Found := Declaration /= RawDict.GetPackageBody (Item);
                  when SubprogramSymbol =>
                     Found := Declaration /= RawDict.GetSubprogramBody (Item);
                  when others =>
                     Found := True;
               end case;
               exit when Found;
               Declaration := RawDict.GetNextDeclaration (Declaration);
            end loop;

            if Declaration = NullSymbol then
               DeclarativeItems := NullIterator;
            else
               DeclarativeItems := Iterator'(DeclarativeItemIterator,
                                             IsAbstract,
                                             Item,
                                             Declaration);
            end if;

            return DeclarativeItems;

         end FirstDeclarativeItem;

         --------------------------------------------------------------------------------

         procedure WriteDeclarativeItems (File  : in SPARK_IO.File_Type;
                                          Scope : in Scopes)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                Scope;
         is

            DeclarativeItems : Iterator;
            DeclarativeItem  : Symbol;

            --------------------------------------------------------------------------------

            procedure WriteLexString (File   : in SPARK_IO.File_Type;
                                      LexStr : in LexTokenManager.Lex_String)
               --# global in     LexTokenManager.LexStringTable;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                File,
               --#                                LexStr,
               --#                                LexTokenManager.LexStringTable;
            is
            begin
               EStrings.Put_String
                 (File  => File,
                  E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => LexStr));
            end WriteLexString;

            --------------------------------------------------------------------------------

            procedure WriteStaticValue (File     : in SPARK_IO.File_Type;
                                        Value    : in LexTokenManager.Lex_String;
                                        TypeMark : in Symbol)
               --# global Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                TypeMark,
               --#                                Value;
            is
            begin
               if IsUnknownTypeMark (TypeMark) then
                  WriteString (File, " unknown ");
               elsif TypeIsNumeric (TypeMark) or else TypeIsCharacter (TypeMark) then
                  ELStrings.Put_String (File  => File,
                                        E_Str => Maths.ValueToString (Maths.ValueRep (Value)));
               elsif TypeIsBoolean (TypeMark) then
                  WriteLexString (File, Value);
               else
                  WriteName (File, GetEnumerationLiteral (TypeMark, Value));
               end if;
            end WriteStaticValue;

            --------------------------------------------------------------------------------

            procedure WriteType (File    : in SPARK_IO.File_Type;
                                 TheType : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                TheType;
            is

               --------------------------------------------------------------------------------

               procedure WriteTypeInfo (File    : in SPARK_IO.File_Type;
                                        TheType : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheType;
               is

                  --------------------------------------------------------------------------------

                  procedure WriteDiscriminant (File    : in SPARK_IO.File_Type;
                                               TheType : in Symbol)
                     --# global in     Dict;
                     --#        in out SPARK_IO.File_Sys;
                     --# derives SPARK_IO.File_Sys from *,
                     --#                                Dict,
                     --#                                File,
                     --#                                TheType;
                  is
                  begin
                     case RawDict.GetTypeDiscriminant (TheType) is
                        when UnknownType =>
                           WriteString (File, "unknown");
                        when EnumerationType =>
                           WriteString (File, "enumeration");
                        when IntegerType =>
                           WriteString (File, "integer");
                        when ModularType =>
                           WriteString (File, "modular");
                        when FloatingPointType =>
                           WriteString (File, "floating point");
                        when FixedPointType =>
                           WriteString (File, "fixed point");
                        when ArrayType =>
                           WriteString (File, "array");
                        when RecordType =>
                           WriteString (File, "record");
                        when AbstractProofType =>
                           WriteString (File, "abstract ");
                        when ProtectedType =>
                           WriteString (File, "protected");
                        when TaskType =>
                           WriteString (File, "task");
                        when AccessType =>
                           WriteString (File, "access ");
                           WriteName (File, RawDict.GetTypeAccesses (TheType));
                        when GenericLimitedType =>
                           WriteString (File, "generic limited ");
                        when GenericPrivateType =>
                           WriteString (File, "generic private ");
                        when GenericOrderedDiscreteType =>
                           WriteString (File, "generic dicrete (ordered) ");
                        when GenericUnorderedDiscreteType =>
                           WriteString (File, "generic discrete ");
                        when GenericIntegerType =>
                           WriteString (File, "generic integer ");
                        when GenericModularType =>
                           WriteString (File, "generic modular ");
                        when GenericFloatingPointType =>
                           WriteString (File, "generic floating point ");
                        when GenericFixedPointType =>
                           WriteString (File, "generic fixed point ");
                        when GenericArrayType         =>
                           WriteString (File, "generic array ");
                     end case;
                  end WriteDiscriminant;

                  -------------------------------------------------------------------------

                  procedure WriteBound (File    : in SPARK_IO.File_Type;
                                        Bound   : in LexTokenManager.Lex_String;
                                        TheType : in Symbol)
                     --# global in     Dict;
                     --#        in out SPARK_IO.File_Sys;
                     --# derives SPARK_IO.File_Sys from *,
                     --#                                Bound,
                     --#                                Dict,
                     --#                                File,
                     --#                                TheType;
                  is
                  begin
                     if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Bound,
                                                                             Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
                        SPARK_IO.Put_String (File, "unknown", 7);
                     else
                        WriteStaticValue (File, Bound, TheType);
                     end if;
                  end WriteBound;

                  --------------------------------------------------------------------------------

                  function TypeIsLimitedPrivate (TheType : Symbol) return Boolean
                     --# global in Dict;
                  is
                  begin
                     return RawDict.GetTypeLimited (TheType) /= Never;
                  end TypeIsLimitedPrivate;

                  --------------------------------------------------------------------------------

                  procedure WriteProtectedRefinement (File    : in SPARK_IO.File_Type;
                                                      TheType : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheType;
                  is
                     It : Iterator;
                  begin
                     WriteString (File, "The implicit state variable of type ");
                     WriteSimpleName (File, TheType);
                     WriteLine (File, " has the following refinement constituents:");
                     It := FirstConstituent (GetProtectedTypeOwnVariable (TheType));
                     while not IsNullIterator (It) loop
                        WriteString (File, "      ");
                        WriteSimpleName (File, CurrentSymbol (It));
                        WriteLine (File, ",");
                        It := NextSymbol (It);
                     end loop;
                     WriteLine (File, ";");
                  end WriteProtectedRefinement;

                  --------------------------------------------------------------------------------

                  procedure WriteTypeDiscriminants (File    : in SPARK_IO.File_Type;
                                                    TheType : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheType;
                  is
                     It : Iterator;
                  begin
                     WriteString (File, "Type ");
                     WriteSimpleName (File, TheType);
                     WriteLine (File, " has the following known discriminants:");
                     It := FirstKnownDiscriminant (TheType);
                     if IsNullIterator (It) then
                        WriteLine (File, "None;");
                     else
                        while not IsNullIterator (It) loop
                           WriteString (File, "      ");
                           WriteSimpleName (File, CurrentSymbol (It));
                           WriteString (File, " which is of type ");
                           WriteName (File, GetType (CurrentSymbol (It)));
                           if SetsPriority (CurrentSymbol (It)) then
                              WriteString (File, " and is used to set the priority");
                           end if;
                           WriteLine (File, ";");
                           It := NextSymbol (It);
                        end loop;
                     end if;
                  end WriteTypeDiscriminants;

                  --------------------------------------------------------------------------------

                  procedure WriteDiscriminantConstraint (File    : in SPARK_IO.File_Type;
                                                          TheType : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheType;
                  is
                     ConstraintIt : Iterator;
                     KnownIt      : Iterator;

                     procedure WriteConstraintValue (File          : in SPARK_IO.File_Type;
                                                     TheConstraint : in Symbol)
                     --# global in     Dict;
                     --#        in     LexTokenManager.State;
                     --#        in out SPARK_IO.File_Sys;
                     --# derives SPARK_IO.File_Sys from *,
                     --#                                Dict,
                     --#                                File,
                     --#                                LexTokenManager.State,
                     --#                                TheConstraint;
                     is
                        Val : LexTokenManager.Lex_String;
                     begin
                        Val := RawDict.GetDiscriminantConstraintStaticValue (TheConstraint);
                        if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Val,
                                                                                Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq then
                           WriteStaticValue (File, Val, GetPredefinedIntegerType);
                        elsif RawDict.GetDiscriminantConstraintAccessedObject (TheConstraint) /= NullSymbol then
                           -- if no static value then must be access to PO
                           WriteName (File, RawDict.GetDiscriminantConstraintAccessedObject (TheConstraint));
                        end if;
                     end WriteConstraintValue;

                  begin -- WriteDiscriminantConstraint
                     WriteString (File, "Type ");
                     WriteSimpleName (File, TheType);
                     WriteLine (File, " has the following discriminant constraints:");
                     ConstraintIt := FirstDiscriminantConstraint (TheType);
                     if IsNullIterator (ConstraintIt) then
                        WriteLine (File, "None;");
                     else
                        KnownIt := FirstKnownDiscriminant (GetParentType (TheType));
                        while not IsNullIterator (ConstraintIt) and then
                          not IsNullIterator (KnownIt) loop
                           WriteString (File, "      ");
                           WriteSimpleName (File, CurrentSymbol (KnownIt));
                           WriteString (File, " => ");
                           WriteConstraintValue (File, CurrentSymbol (ConstraintIt));
                           if SetsPriority (CurrentSymbol (KnownIt)) then
                              WriteString (File, " which is used to set the priority");
                           end if;
                           WriteLine (File, ";");
                           KnownIt := NextSymbol (KnownIt);
                           ConstraintIt := NextSymbol (ConstraintIt);
                        end loop;
                     end if;
                  end WriteDiscriminantConstraint;

                  --------------------------------------------------------------------------------

                  procedure WriteTypePriority (File    : in SPARK_IO.File_Type;
                                               TheType : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheType;
                  is
                     Val : LexTokenManager.Lex_String;
                  begin
                     if GetTypeHasPragma (TheType, Priority) then
                        WriteString (File, "Type ");
                        WriteSimpleName (File, TheType);
                        WriteString (File, " has pragma Priority");
                        Val := GetTypePragmaValue (TheType, Priority);
                        if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Val,
                                                                                Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
                           WriteLine (File, " of unknown value;");
                        else
                           WriteString (File, " of value ");
                           WriteStaticValue (File, Val, GetPredefinedIntegerType);
                           WriteLine (File, ";");
                        end if;

                     elsif GetTypeHasPragma (TheType, InterruptPriority) then
                        WriteString (File, "Type ");
                        WriteSimpleName (File, TheType);
                        WriteString (File, " has pragma Interrupt_Priority");
                        Val := GetTypePragmaValue (TheType, InterruptPriority);
                        if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Val,
                                                                                Lex_Str2 => LexTokenManager.Null_String) = LexTokenManager.Str_Eq then
                           WriteLine (File, " of unknown value;");
                        else
                           WriteString (File, " of value ");
                           WriteStaticValue (File, Val, GetPredefinedIntegerType);
                           WriteLine (File, ";");
                        end if;
                     end if;
                  end WriteTypePriority;

                  --------------------------------------------------------------------------------
                  procedure WritePriority (File    : in SPARK_IO.File_Type;
                                           TheType : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheType;
                  is
                  begin
                     WriteString (File, "Type ");
                     WriteName (File, TheType);
                     WriteString (File, " has a priority of ");
                     WriteStaticValue (File,
                                       GetTypePriority (TheType),
                                       GetPredefinedIntegerType);
                     WriteLine (File, ";");
                  end WritePriority;

                  --------------------------------------------------------------------------------

                  procedure WriteSuspendsList (File    : in SPARK_IO.File_Type;
                                               TheTask : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheTask;
                  is
                     It : Iterator;
                  begin
                     WriteString (File, "Type ");
                     WriteSimpleName (File, TheTask);
                     WriteLine (File, " has the following PO or SO names in its Suspend list");
                     It := FirstSuspendsListItem (TheTask);
                     if IsNullIterator (It) then
                        WriteLine (File, "None;");
                     else
                        while not IsNullIterator (It) loop
                           WriteString (File, "      ");
                           WriteName (File, CurrentSymbol (It));
                           WriteLine (File, ";");
                           It := NextSymbol (It);
                        end loop;
                     end if;
                  end WriteSuspendsList;

                  --------------------------------------------------------------------------------

               begin
                  WriteString (File, "type named ");
                  WriteSimpleName (File, TheType);
                  WriteString (File, " is ");

                  if GetRootOfExtendedType (TheType) /= NullSymbol then
                     WriteString (File, "an extension of type ");
                     WriteName (File, GetRootOfExtendedType (TheType));
                     WriteString (File, " and is ");
                  end if;

                  if TypeIsAbstract (TheType) then
                     WriteString (File, "abstract ");
                  end if;
                  if TypeIsTagged (TheType) then
                     WriteString (File, "tagged ");
                  end if;
                  if TypeIsPrivate (TheType) then
                     if TypeIsLimitedPrivate (TheType) then
                        WriteString (File, "limited ");
                     end if;
                     WriteString (File, "private ");
                  end if;
                  if IsUnconstrainedArrayType (TheType) then
                     WriteString (File, "unconstrained ");
                  end if;
                  if IsStatic (TheType, GetScope (TheType)) then
                     WriteString (File, "static ");
                  end if;
                  WriteDiscriminant (File, TheType);
                  if IsProofType (TheType) then
                     WriteString (File, "proof ");
                  end if;
                  WriteSpace (File);
                  if IsType (TheType) then
                     WriteString (File, "type");
                  else
                     WriteString (File, "subtype of ");
                     WriteName (File, GetParentType (TheType));
                  end if;
                  if TypeIsScalar (TheType) then
                     WriteString (File, " range ");
                     WriteBound (File, RawDict.GetTypeLower (TheType), TheType);
                     WriteString (File, " .. ");
                     WriteBound (File, RawDict.GetTypeUpper (TheType), TheType);
                     if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => RawDict.GetTypeErrorBound (TheType),
                                                                             Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq then
                        if TypeIsFloatingPoint (TheType) then
                           WriteString (File, " digits ");
                        else
                           WriteString (File, " delta ");
                        end if;
                        WriteStaticValue (File,
                                          RawDict.GetTypeErrorBound (TheType),
                                          TheType);
                     end if;
                  elsif TypeIsArray (TheType) then
                     WriteString (File, " of ");
                     WriteName (File, GetArrayComponent (TheType));
                  elsif IsType (TheType) and then
                    TypeIsProtected (TheType) and then
                    GetProtectedTypeElementsHidden (TheType) then
                     WriteString (File, " with hidden elements ");
                  end if;
                  if TypeIsAccess (TheType) then
                     WriteString (File, " implicitly");
                  end if;
                  WriteString (File, " declared in ");
                  WriteScope (File, GetScope (TheType));
                  WriteLine (File, " ;");
                  if TypeIsProtected (TheType) then
                     if IsType (TheType) then
                        WriteTypeDiscriminants (File, TheType);
                        WriteProtectedRefinement (File, TheType);
                        WriteTypePriority (File, TheType);
                     else -- subtype
                        WriteDiscriminantConstraint (File, TheType);
                     end if;
                     WritePriority (File, TheType);
                  end if;
                  if TypeIsTask (TheType) then
                     if IsType (TheType) then
                        WriteTypeDiscriminants (File, TheType);
                        WriteGlobalVariables (IsAbstract, File, TheType);
                        WriteDependencyClauses (IsAbstract, File, TheType);
                        WriteSuspendsList (File, TheType);
                        WriteTypePriority (File, TheType);
                        if not RawDict.GetTaskTypeSignatureIsWellformed (IsAbstract, TheType) then
                           WriteLine (File, "Task type signature contains errors");
                        end if;
                     else --subtype
                        WriteDiscriminantConstraint (File, TheType);
                     end if;
                     WritePriority (File, TheType);
                  end if;
               end WriteTypeInfo;

               --------------------------------------------------------------------------------

               procedure WriteEnumerationLiterals (File            : in SPARK_IO.File_Type;
                                                   EnumerationType : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                EnumerationType,
                  --#                                File,
                  --#                                LexTokenManager.State;
               is

                  Literal : Iterator;
                  Value   : Natural;

                  --------------------------------------------------------------------------------

                  procedure WriteEnumerationLiteral (File               : in SPARK_IO.File_Type;
                                                     EnumerationType    : in Symbol;
                                                     EnumerationValue   : in Natural;
                                                     EnumerationLiteral : in Symbol)
                     --# global in     Dict;
                     --#        in     LexTokenManager.State;
                     --#        in out SPARK_IO.File_Sys;
                     --# derives SPARK_IO.File_Sys from *,
                     --#                                Dict,
                     --#                                EnumerationLiteral,
                     --#                                EnumerationType,
                     --#                                EnumerationValue,
                     --#                                File,
                     --#                                LexTokenManager.State;
                  is
                  begin
                     WriteString (File, "enumeration literal # ");
                     WriteInteger (File, EnumerationValue);
                     WriteString (File, " of ");
                     WriteName (File, EnumerationType);
                     WriteString (File, " is ");
                     WriteSimpleName (File, EnumerationLiteral);
                     WriteLine (File, " ;");
                  end WriteEnumerationLiteral;

                  --------------------------------------------------------------------------------

               begin
                  Literal := FirstEnumerationLiteral (EnumerationType);
                  Value := 0;
                  loop
                     exit when IsNullIterator (Literal);
                     WriteEnumerationLiteral (File,
                                              EnumerationType,
                                              Value,
                                              CurrentSymbol (Literal));
                     Literal := NextSymbol (Literal);
                     Value := Value + 1;
                  end loop;
               end WriteEnumerationLiterals;

               --------------------------------------------------------------------------------

               procedure WriteRecordComponents (File       : in SPARK_IO.File_Type;
                                                RecordType : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                RecordType;
               is

                  Component : Iterator;
                  Number    : Positive;

                  --------------------------------------------------------------------------------

                  procedure WriteRecordComponent (File            : in SPARK_IO.File_Type;
                                                  RecordType      : in Symbol;
                                                  Number          : in Positive;
                                                  RecordComponent : in Symbol)
                     --# global in     Dict;
                     --#        in     LexTokenManager.State;
                     --#        in out SPARK_IO.File_Sys;
                     --# derives SPARK_IO.File_Sys from *,
                     --#                                Dict,
                     --#                                File,
                     --#                                LexTokenManager.State,
                     --#                                Number,
                     --#                                RecordComponent,
                     --#                                RecordType;
                  is
                  begin
                     WriteString (File, "record component # ");
                     WriteInteger (File, Number);
                     WriteString (File, " of ");
                     WriteName (File, RecordType);
                     WriteString (File, " is ");
                     WriteSimpleName (File, RecordComponent);
                     WriteString (File, " of type ");
                     WriteName (File, GetType (RecordComponent));
                     WriteLine (File, " ;");
                  end WriteRecordComponent;

                  --------------------------------------------------------------------------------

               begin
                  Component := FirstRecordComponent (RecordType);
                  Number := 1;
                  loop
                     exit when IsNullIterator (Component);
                     WriteRecordComponent (File,
                                           RecordType,
                                           Number,
                                           CurrentSymbol (Component));
                     Component := NextSymbol (Component);
                     Number := Number + 1;
                  end loop;
               end WriteRecordComponents;

               --------------------------------------------------------------------------------

               procedure WriteArrayIndices (File      : in SPARK_IO.File_Type;
                                            ArrayType : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                ArrayType,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State;
               is

                  ArrayIndex : Iterator;
                  Dimension  : Positive;

                  --------------------------------------------------------------------------------

                  procedure WriteArrayIndex (File      : in SPARK_IO.File_Type;
                                             ArrayType : in Symbol;
                                             Dimension : in Positive;
                                             IndexType : in Symbol)
                     --# global in     Dict;
                     --#        in     LexTokenManager.State;
                     --#        in out SPARK_IO.File_Sys;
                     --# derives SPARK_IO.File_Sys from *,
                     --#                                ArrayType,
                     --#                                Dict,
                     --#                                Dimension,
                     --#                                File,
                     --#                                IndexType,
                     --#                                LexTokenManager.State;
                  is
                  begin
                     WriteString (File, "index # ");
                     WriteInteger (File, Dimension);
                     WriteString (File, " of ");
                     WriteName (File, ArrayType);
                     WriteString (File, " is ");
                     WriteName (File, IndexType);
                     WriteLine (File, " ;");
                  end WriteArrayIndex;

                  --------------------------------------------------------------------------------

               begin
                  ArrayIndex := FirstArrayIndex (ArrayType);
                  Dimension := 1;
                  loop
                     exit when IsNullIterator (ArrayIndex);
                     WriteArrayIndex (File,
                                      ArrayType,
                                      Dimension,
                                      CurrentSymbol (ArrayIndex));
                     ArrayIndex := NextSymbol (ArrayIndex);
                     Dimension := Dimension + 1;
                  end loop;
               end WriteArrayIndices;

               --------------------------------------------------------------------------------

               procedure WriteProtectedDeclarations (File    : in     SPARK_IO.File_Type;
                                                     TheType : in     Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                TheType;
               is
                  It : Iterator;

                  procedure WriteProtectedElement (File       : in     SPARK_IO.File_Type;
                                                   TheType    : in     Symbol;
                                                   TheElement : in     Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheElement,
                  --#                                TheType;
                  is
                  begin
                     WriteString (File, "protected element named ");
                     WriteSimpleName (File, TheElement);
                     WriteString (File, "  of ");
                     WriteName (File, GetType (TheElement));
                     WriteString (File, " declared in ");
                     WriteName (File, TheType);
                     WriteLine (File, " ;");
                  end WriteProtectedElement;

               begin -- WriteProtectedDeclarations
                  It := FirstVisibleSubprogram (TheType);
                  while not IsNullIterator (It) loop
                     WriteSubprogram (File, CurrentSymbol (It));
                     It := NextSymbol (It);
                  end loop;
                  It := FirstProtectedElement (TheType);
                  while not IsNullIterator (It) loop
                     WriteProtectedElement (File,
                                            TheType,
                                            CurrentSymbol (It));
                     It := NextSymbol (It);
                  end loop;
               end WriteProtectedDeclarations;

               --------------------------------------------------------------------------------

            begin
               WriteTypeInfo (File, TheType);
               if TypeIsEnumeration (TheType) and then IsType (TheType) then
                  WriteEnumerationLiterals (File, TheType);
               elsif TypeIsRecord (TheType) then
                  WriteRecordComponents (File, TheType);
               elsif TypeIsArray (TheType) then
                  WriteArrayIndices (File, TheType);
               elsif IsType (TheType) and then TypeIsProtected (TheType) then
                  WriteProtectedDeclarations (File, TheType);
               end if;
            end WriteType;

            --------------------------------------------------------------------------------

            procedure WriteVariable (File     : in SPARK_IO.File_Type;
                                     Variable : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                Variable;
            is
            begin
               WriteString (File, "variable named ");
               WriteSimpleName (File, Variable);
               WriteString (File, " is");
               if VariableIsInitialized (Variable) then
                  WriteString (File, " initialized");
               end if;
               if VariableIsAliased (Variable) then
                  WriteString (File, " aliased");
               end if;
               WriteString (File, " variable of ");
               WriteName (File, GetType (Variable));
               WriteString (File, " declared in ");
               WriteScope (File, GetScope (Variable));
               if VariableHasPragmaImport (Variable) then
                  WriteString (File, " and completed by a pragma Import");
               end if;
               WriteLine (File, " ;");
            end WriteVariable;

            --------------------------------------------------------------------------------

            procedure WriteConstant (File        : in SPARK_IO.File_Type;
                                     TheConstant : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                TheConstant;
            is
            begin
               WriteString (File, "constant named ");
               WriteSimpleName (File, TheConstant);
               WriteString (File, " is ");
               if ConstantIsDeferred (TheConstant) then
                  WriteString (File, "deferred ");
               end if;
               if IsStatic (TheConstant, GetScope (TheConstant)) then
                  WriteString (File, "static ");
               end if;
               if IsProofConstant (TheConstant) then
                  WriteString (File, "proof ");
               end if;
               WriteString (File, "constant of ");
               WriteName (File, GetType (TheConstant));
               if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => GetValue (TheConstant),
                                                                       Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq then
                  WriteString (File, " value ");
                  WriteStaticValue (File,
                                    GetValue (TheConstant),
                                    GetType (TheConstant));
               end if;
               WriteString (File, " declared in ");
               WriteScope (File, GetScope (TheConstant));
               WriteLine (File, " ;");
            end WriteConstant;

            --------------------------------------------------------------------------------

            procedure WriteDeclarativeItem (File : in SPARK_IO.File_Type;
                                            Item : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                Item,
               --#                                LexTokenManager.State;
            is
            begin
               case RawDict.GetSymbolDiscriminant (Item) is
                  when TypeSymbol =>
                     WriteType (File, Item);
                  when VariableSymbol =>
                     WriteVariable (File, Item);
                  when ConstantSymbol =>
                     WriteConstant (File, Item);
                  when SubprogramSymbol =>
                     WriteSubprogram (File, Item);
                  when others =>
                     WritePackage (File, Item);
               end case;
            end WriteDeclarativeItem;

            --------------------------------------------------------------------------------

         begin

            DeclarativeItems := FirstDeclarativeItem (Scope);

            loop

               exit when IsNullIterator (DeclarativeItems);

               DeclarativeItem := CurrentSymbol (DeclarativeItems);

               if IsType (DeclarativeItem) and then
                  TypeIsPrivate (DeclarativeItem) and then
                  IsLocalOrPrivateScope (Scope) then
                  null;
               elsif IsLocalOrPrivateScope (Scope) and then
                  IsDeferredConstant (DeclarativeItem) then
                  null;
               else
                  WriteDeclarativeItem (File, DeclarativeItem);
               end if;

               DeclarativeItems := NextSymbol (DeclarativeItems);

            end loop;

         end WriteDeclarativeItems;

         --------------------------------------------------------------------------------

         procedure WriteLoops (File            : in SPARK_IO.File_Type;
                               CompilationUnit : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                CompilationUnit,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State;
         is

            TheLoop : Iterator;

            --------------------------------------------------------------------------------

            procedure WriteLoop (File    : in SPARK_IO.File_Type;
                                 TheLoop : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                TheLoop;
            is

               --------------------------------------------------------------------------------

               procedure WriteLoopInfo (File    : in SPARK_IO.File_Type;
                                        TheLoop : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                TheLoop;
               is
               begin
                  WriteString (File, "loop named ");
                  WriteSimpleName (File, TheLoop);
                  WriteString (File, " declared in ");
                  WriteScope (File, GetScope (TheLoop));
                  WriteLine (File, " ;");
               end WriteLoopInfo;

               --------------------------------------------------------------------------------

               procedure WriteLoopParameter (File          : in SPARK_IO.File_Type;
                                             LoopParameter : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                LoopParameter;
               is
               begin
                  WriteString (File, "loop parameter of ");
                  WriteName (File, GetRegion (GetScope (LoopParameter)));
                  WriteString (File, " named ");
                  WriteSimpleName (File, LoopParameter);
                  WriteString (File, " is of type ");
                  WriteName (File, GetType (LoopParameter));
                  WriteLine (File, " ;");
               end WriteLoopParameter;

               --------------------------------------------------------------------------------

            begin
               WriteLoopInfo (File, TheLoop);
               if IsForLoop (TheLoop) then
                  WriteLoopParameter (File, GetLoopParameter (TheLoop));
               end if;
            end WriteLoop;

            --------------------------------------------------------------------------------

         begin
            TheLoop := FirstLoop (CompilationUnit);
            loop
               exit when IsNullIterator (TheLoop);
               WriteLoop (File, CurrentSymbol (TheLoop));
               TheLoop := NextSymbol (TheLoop);
            end loop;
         end WriteLoops;

         --------------------------------------------------------------------------------

         procedure WriteGlobalVariables (Abstraction : in Abstractions;
                                         File        : in SPARK_IO.File_Type;
                                         Subprogram  : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Abstraction,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                Subprogram;
         is

            GlobalVariable : Iterator;

            --------------------------------------------------------------------------------

            procedure WriteGlobalVariable (Abstraction    : in Abstractions;
                                           File           : in SPARK_IO.File_Type;
                                           Subprogram     : in Symbol;
                                           GlobalVariable : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Abstraction,
               --#                                Dict,
               --#                                File,
               --#                                GlobalVariable,
               --#                                LexTokenManager.State,
               --#                                Subprogram;
            is
            begin
               if Abstraction = IsRefined then
                  WriteString (File, "refined ");
               end if;
               WriteString (File, "global variable named ");
               WriteName (File, GlobalVariable);

               WriteString (File, " of mode ");
               case GetGlobalMode (Abstraction, Subprogram, GlobalVariable) is
                  when DefaultMode =>
                     WriteString (File, "default");
                  when InMode =>
                     WriteString (File, "in");
                  when OutMode =>
                     WriteString (File, "out");
                  when InOutMode =>
                     WriteString (File, "in out");
                  when InvalidMode =>
                     WriteString (File, "invalid");
               end case;
               WriteString (File, " is referenced by ");
               WriteName (File, Subprogram);
               WriteLine (File, " ;");
            end WriteGlobalVariable;

            --------------------------------------------------------------------------------

         begin
            GlobalVariable := FirstGlobalVariable (Abstraction, Subprogram);
            loop
               exit when IsNullIterator (GlobalVariable);
               WriteGlobalVariable (Abstraction,
                                    File,
                                    Subprogram,
                                    CurrentSymbol (GlobalVariable));
               GlobalVariable := NextSymbol (GlobalVariable);
            end loop;
         end WriteGlobalVariables;

         --------------------------------------------------------------------------------

         procedure WriteDependencyClauses (Abstraction  : in Abstractions;
                                           File         : in SPARK_IO.File_Type;
                                           TheProcedure : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Abstraction,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                TheProcedure;
         is

            Export : Iterator;

            --------------------------------------------------------------------------------

            procedure WriteDependencies (Abstraction  : in Abstractions;
                                         File         : in SPARK_IO.File_Type;
                                         TheProcedure : in Symbol;
                                         Export       : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Abstraction,
               --#                                Dict,
               --#                                Export,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                TheProcedure;
            is

               Dependency : Iterator;

               --------------------------------------------------------------------------------

               procedure WriteDependency (Abstraction  : in Abstractions;
                                          File         : in SPARK_IO.File_Type;
                                          TheProcedure : in Symbol;
                                          Export       : in Symbol;
                                          Import       : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Abstraction,
                  --#                                Dict,
                  --#                                Export,
                  --#                                File,
                  --#                                Import,
                  --#                                LexTokenManager.State,
                  --#                                TheProcedure;
               is
               begin
                  if Abstraction = IsRefined then
                     WriteString (File, "refined ");
                  end if;
                  if GetOwnVariableOrConstituentMode (Export) = InMode then
                     WriteString (File, "implicit ");
                  end if;
                  WriteString (File, "export named ");
                  WriteName (File, Export);
                  WriteString (File, " is ");
                  if GetOwnVariableOrConstituentMode (Import) = OutMode then
                     WriteString (File, "implicitly ");
                  end if;
                  WriteString (File, "derived from ");
                  if Import = NullSymbol then
                     WriteString (File, "nothing");
                  else
                     WriteName (File, Import);
                  end if;
                  WriteString (File, " in ");
                  WriteName (File, TheProcedure);
                  WriteLine (File, " ;");
               end WriteDependency;

               --------------------------------------------------------------------------------

            begin
               Dependency := FirstDependency (Abstraction, TheProcedure, Export);
               if IsNullIterator (Dependency) then
                  WriteDependency (Abstraction,
                                   File,
                                   TheProcedure,
                                   Export,
                                   NullSymbol);
               else
                  loop
                     exit when IsNullIterator (Dependency);
                     WriteDependency (Abstraction,
                                      File,
                                      TheProcedure,
                                      Export,
                                      CurrentSymbol (Dependency));
                     Dependency := NextSymbol (Dependency);
                  end loop;
               end if;
            end WriteDependencies;

            --------------------------------------------------------------------------------

         begin
            Export := FirstExport (Abstraction, TheProcedure);
            loop
               exit when IsNullIterator (Export);
               WriteDependencies (Abstraction,
                                  File,
                                  TheProcedure,
                                  CurrentSymbol (Export));
               Export := NextSymbol (Export);
            end loop;
         end WriteDependencyClauses;

         --------------------------------------------------------------------------------

         procedure WritePackage (File       : in SPARK_IO.File_Type;
                                 ThePackage : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                ThePackage;
         is

            --------------------------------------------------------------------------------

            procedure WriteOwnVariables (File       : in SPARK_IO.File_Type;
                                         ThePackage : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                ThePackage;
            is

               OwnVariables : Iterator;
               OwnVariable  : Symbol;

               --------------------------------------------------------------------------------

               procedure WriteOwnVariable (File       : in SPARK_IO.File_Type;
                                           ThePackage : in Symbol;
                                           Variable   : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                ThePackage,
                  --#                                Variable;
               is
               begin
                  case GetOwnVariableMode (Variable) is
                     when DefaultMode =>
                        WriteString (File, "default");
                     when InMode =>
                        WriteString (File, "in");
                     when OutMode =>
                        WriteString (File, "out");
                     when InOutMode =>
                        WriteString (File, "in out");
                     when InvalidMode =>
                        WriteString (File, "invalid");
                  end case;
                  WriteString (File, " mode ");
                  if GetOwnVariableProtected (Variable) then
                     WriteString (File, "protected ");
                  end if;
                  WriteString (File, "own variable ");
                  WriteSimpleName (File, Variable);
                  if RawDict.GetOwnVariableTyped (RawDict.GetVariableOwnVariable (Variable)) then
                     WriteString (File, " of type ");
                     WriteName (File, GetType (Variable));
                  end if;
                  WriteString (File, " is owned ");
                  if OwnVariableIsInitialized (Variable) then
                     WriteString (File, "and initialized ");
                  end if;
                  WriteString (File, "by ");
                  WriteName (File, ThePackage);
                  WriteLine (File, " ;");
               end WriteOwnVariable;

               --------------------------------------------------------------------------------

            begin
               OwnVariables := FirstOwnVariable (ThePackage);
               loop
                  exit when IsNullIterator (OwnVariables);
                  OwnVariable := CurrentSymbol (OwnVariables);
                  WriteOwnVariable (File, ThePackage, OwnVariable);
                  OwnVariables := NextSymbol (OwnVariables);
               end loop;
            end WriteOwnVariables;

            --------------------------------------------------------------------------------

            procedure WriteEmbeddedPackages (File       : in SPARK_IO.File_Type;
                                             ThePackage : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                ThePackage;
            is
               Packages : Iterator;
            begin
               Packages := FirstEmbeddedPackage (ThePackage);
               loop
                  exit when IsNullIterator (Packages);
                  WritePackageInfo (File, CurrentSymbol (Packages));
                  Packages := NextSymbol (Packages);
               end loop;
            end WriteEmbeddedPackages;

            --------------------------------------------------------------------------------
            --1606 added by JEB
            procedure WriteOwnedPackages (File       : in SPARK_IO.File_Type;
                                             ThePackage : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                ThePackage;
            is
               Packages : Iterator;
            begin
               Packages := FirstOwnedPackage (ThePackage);
               loop
                  exit when IsNullIterator (Packages);
                  WritePackageInfo (File, CurrentSymbol (Packages));
                  WritePackage (File, CurrentSymbol (Packages));
                  Packages := NextSymbol (Packages);
               end loop;
            end WriteOwnedPackages;

            --------------------------------------------------------------------------------

            procedure WriteAbstractOwnVariables (File       : in SPARK_IO.File_Type;
                                                 ThePackage : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                ThePackage;
            is

               AbstractOwnVariables : Iterator;

               --------------------------------------------------------------------------------

               procedure WriteAbstractOwnVariable (File    : in SPARK_IO.File_Type;
                                                   Subject : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                Subject;
               is

                  Constituents : Iterator;
                  Constituent  : Symbol;

                  --------------------------------------------------------------------------------

                  procedure WriteConstituent (File        : in SPARK_IO.File_Type;
                                              Subject     : in Symbol;
                                              Constituent : in Symbol)
                     --# global in     Dict;
                     --#        in     LexTokenManager.State;
                     --#        in out SPARK_IO.File_Sys;
                     --# derives SPARK_IO.File_Sys from *,
                     --#                                Constituent,
                     --#                                Dict,
                     --#                                File,
                     --#                                LexTokenManager.State,
                     --#                                Subject;
                  is
                  begin
                     case GetConstituentMode (Constituent) is
                        when DefaultMode =>
                           WriteString (File, "default");
                        when InMode =>
                           WriteString (File, "in");
                        when OutMode =>
                           WriteString (File, "out");
                        when InOutMode =>
                           WriteString (File, "in out");
                        when InvalidMode =>
                           WriteString (File, "invalid");
                     end case;
                     WriteString (File, " mode ");
                     WriteString (File, "constituent of ");
                     WriteName (File, Subject);
                     WriteString (File, " is ");
                     WriteSimpleName (File, Constituent);
                     if GetOwner (Subject) /= GetOwner (Constituent) then
                        WriteString (File, " declared in ");
                        WriteName (File, GetOwner (Constituent));
                     end if;
                     WriteLine (File, " ;");
                  end WriteConstituent;

                  --------------------------------------------------------------------------------

               begin
                  Constituents := FirstConstituent (Subject);
                  loop
                     exit when IsNullIterator (Constituents);
                     Constituent := CurrentSymbol (Constituents);
                     WriteConstituent (File, Subject, Constituent);
                     Constituents := NextSymbol (Constituents);
                  end loop;
               end WriteAbstractOwnVariable;

               --------------------------------------------------------------------------------

            begin
               AbstractOwnVariables := FirstAbstractOwnVariable (ThePackage);
               loop
                  exit when IsNullIterator (AbstractOwnVariables);
                  WriteAbstractOwnVariable (File,
                                            CurrentSymbol (AbstractOwnVariables));
                  AbstractOwnVariables := NextSymbol (AbstractOwnVariables);
               end loop;
            end WriteAbstractOwnVariables;

            --------------------------------------------------------------------------------

            procedure WriteSubprogramBodies (File  : in SPARK_IO.File_Type;
                                             Scope : in Scopes)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                Scope;
            is
               DeclarativeItems : Iterator;
               DeclarativeItem  : Symbol;
            begin

               DeclarativeItems := FirstDeclarativeItem (Scope);

               loop
                  exit when IsNullIterator (DeclarativeItems);
                  DeclarativeItem := CurrentSymbol (DeclarativeItems);
                  if IsSubprogram (DeclarativeItem) then
                     WriteGlobalVariables (IsRefined, File, DeclarativeItem);
                     if IsProcedure (DeclarativeItem) then
                        WriteDependencyClauses (IsRefined, File, DeclarativeItem);
                     end if;
                  end if;

                  DeclarativeItems := NextSymbol (DeclarativeItems);

               end loop;

            end WriteSubprogramBodies;

            --------------------------------------------------------------------------------

         begin
            WriteWithReferences (File, VisibleScope (ThePackage));
            WriteWithReferences (File, LocalScope (ThePackage));
            WriteInheritsReferences (File, ThePackage);
            WriteOwnVariables (File, ThePackage);
            if IsGenericPackage (ThePackage) then
               WriteGenericFormalParameters (File, ThePackage);
            end if;
            WriteEmbeddedPackages (File, ThePackage);
            WriteAbstractOwnVariables (File, ThePackage);
            WriteDeclarativeItems (File, VisibleScope (ThePackage));
            WriteDeclarativeItems (File, PrivateScope (ThePackage));
            WriteDeclarativeItems (File, LocalScope (ThePackage));
            WriteSubprogramBodies (File, VisibleScope (ThePackage));
            WriteSubprogramBodies (File, LocalScope (ThePackage));
            WriteLoops (File, ThePackage);
            WriteOwnedPackages (File, ThePackage);
         end WritePackage;

         --------------------------------------------------------------------------------

         procedure WriteLibraryPackage (File       : in SPARK_IO.File_Type;
                                        ThePackage : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                ThePackage;
         is
         begin
            if GetRootPackage (ThePackage) /= GetPredefinedPackageAda then
               WritePackageInfo (File, ThePackage);
               WritePackage (File, ThePackage);
            end if;
         end WriteLibraryPackage;

         --------------------------------------------------------------------------------

         procedure WriteSubprogram (File       : in SPARK_IO.File_Type;
                                    Subprogram : in Symbol)
            --# global in     Dict;
            --#        in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Dict,
            --#                                File,
            --#                                LexTokenManager.State,
            --#                                Subprogram;
         is

            --------------------------------------------------------------------------------

            procedure WriteSubprogramInfo (File       : in SPARK_IO.File_Type;
                                           Subprogram : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                Subprogram;
            is
            begin
               WriteString (File, "subprogram named ");
               WriteSimpleName (File, Subprogram);
               WriteString (File, " is ");
               if IsGenericSubprogram (Subprogram) then
                  WriteString (File, "generic ");
               end if;
               if IsEntry (Subprogram) then
                  WriteString (File, "entry");
               elsif IsProcedure (Subprogram) then
                  WriteString (File, "procedure");
               else
                  if IsProofFunction (Subprogram) then
                     WriteString (File, "proof ");
                  end if;
                  WriteString (File, "function of ");
                  WriteName (File, GetType (Subprogram));
               end if;
               -- don't try and print declaration region for library-level units
               if not (IsMainProgram (Subprogram) or else
                         (Subprogram = GetThePartition) or else
                         IsGenericSubprogram (Subprogram)) then
                  WriteString (File, " declared in ");
                  WriteScope (File, GetScope (Subprogram));
               end if;
               if RawDict.GetSubprogramInstantiationOf (Subprogram) /= NullSymbol then
                  WriteString (File, " and which is an instantiation of ");
                  WriteName (File, RawDict.GetSubprogramInstantiationOf (Subprogram));
               end if;
               if IsInterruptHandler (Subprogram) then
                  WriteString (File, " and is an interrupt handler");
               end if;
               if IsThePartition (Subprogram) then
                  WriteString (File, " and is the partition table of the whole program");
               end if;
               WriteLine (File, " ;");
            end WriteSubprogramInfo;

            --------------------------------------------------------------------------------

            procedure WriteSubprogramParameters (File       : in SPARK_IO.File_Type;
                                                 Subprogram : in Symbol)
               --# global in     Dict;
               --#        in     LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives SPARK_IO.File_Sys from *,
               --#                                Dict,
               --#                                File,
               --#                                LexTokenManager.State,
               --#                                Subprogram;
            is

               Parameter : Iterator;
               Number    : Positive;

               --------------------------------------------------------------------------------

               procedure WriteSubprogramParameter (File       : in SPARK_IO.File_Type;
                                                   Subprogram : in Symbol;
                                                   Number     : in Positive;
                                                   Parameter  : in Symbol)
                  --# global in     Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                Dict,
                  --#                                File,
                  --#                                LexTokenManager.State,
                  --#                                Number,
                  --#                                Parameter,
                  --#                                Subprogram;
               is
               begin
                  WriteString (File, "subprogram parameter # ");
                  WriteInteger (File, Number);
                  WriteString (File, " of ");
                  WriteName (File, Subprogram);
                  WriteString (File, " is ");
                  WriteSimpleName (File, Parameter);
                  WriteString (File, " which is ");
                  case GetSubprogramParameterMode (Parameter) is
                     when DefaultMode =>
                        WriteString (File, "default");
                     when InMode =>
                        WriteString (File, "in");
                     when OutMode =>
                        WriteString (File, "out");
                     when InOutMode =>
                        WriteString (File, "in out");
                     when InvalidMode =>
                        WriteString (File, "invalid");
                  end case;
                  WriteString (File, " parameter of type ");
                  WriteName (File, GetType (Parameter));
                  WriteLine (File, " ;");
               end WriteSubprogramParameter;

               --------------------------------------------------------------------------------

            begin
               Parameter := FirstSubprogramParameter (Subprogram);
               Number := 1;
               loop
                  exit when IsNullIterator (Parameter);
                  WriteSubprogramParameter (File,
                                            Subprogram,
                                            Number,
                                            CurrentSymbol (Parameter));
                  Parameter := NextSymbol (Parameter);
                  Number := Number + 1;
               end loop;
            end WriteSubprogramParameters;

            --------------------------------------------------------------------------------

         begin
            WriteSubprogramInfo (File, Subprogram);
            WriteWithReferences (File, LocalScope (Subprogram));
            WriteInheritsReferences (File, Subprogram);
            WriteSubprogramParameters (File, Subprogram);
            WriteGlobalVariables (IsAbstract, File, Subprogram);
            if IsProcedure (Subprogram) then
               WriteDependencyClauses (IsAbstract, File, Subprogram);
            end if;
            if IsGenericSubprogram (Subprogram) then
               WriteGenericFormalParameters (File, Subprogram);
            end if;
            WriteDeclarativeItems (File, LocalScope (Subprogram));
            WriteLoops (File, Subprogram);
         end WriteSubprogram;

         --------------------------------------------------------------------------------

      begin
         if IsPackage (Unit) then
            WriteLibraryPackage (File, Unit);
         else
            WriteSubprogram (File, Unit);
         end if;
      end WriteLibraryUnit; -- Hidden body expected

   --------------------------------------------------------------------------------

   begin
      LibraryUnits := FirstLibraryUnit;
      loop
         exit when IsNullIterator (LibraryUnits);
         WriteLibraryUnit (File, CurrentSymbol (LibraryUnits));
         LibraryUnits := NextSymbol (LibraryUnits);
      end loop;
      -- If we are in Ravencar mode, then a pseudo procedure called main_program will have been
      -- created as a place to store partition-wide flow anno data.  We print these details out now.
      if GetThePartition /= NullSymbol then
         WriteLibraryUnit (File, GetThePartition);
      end if;
   end WriteLibraryUnits;

   --------------------------------------------------------------------------------

   procedure AppendViewerSpecificInfo (File   : in SPARK_IO.File_Type)
   --# global in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict              from * &
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                File;
   is
      MaxLineLength : constant Positive := 512;
      subtype LineIndex is Positive range 1 .. MaxLineLength;
      subtype LineCount is Natural  range 1 .. MaxLineLength;
      subtype Lines is String (LineIndex);
      Line : Lines;
      Length : LineCount;
      FileStatus : SPARK_IO.File_Status;
      TemporaryFile : SPARK_IO.File_Type;
   begin
      TemporaryFile := Dict.TemporaryFile;
      SPARK_IO.Reset (TemporaryFile, SPARK_IO.In_File, FileStatus);
      if FileStatus = SPARK_IO.Ok then
         loop
            exit when SPARK_IO.End_Of_File (TemporaryFile);
            SPARK_IO.Get_Line (TemporaryFile, Line, Length);
            SPARK_IO.Put_Line (File, Line, Length);
         end loop;
      end if;
      Dict.TemporaryFile := TemporaryFile;
   end AppendViewerSpecificInfo;

begin

   LocalFileName := FileSystem.CaseOfFilesForCreate (FileName);
   EStrings.Create (File         => File,
                    Name_Of_File => LocalFileName,
                    Form_Of_File => "",
                    Status       => FileStatus);

   if FileStatus = SPARK_IO.Ok then
      -- File := SPARK_IO.Standard_Output; -- useful debug statement
      WriteLibraryUnits (File);
      AppendViewerSpecificInfo (File);
      --# accept Flow, 10, File, "Expected ineffective assignment";
      SPARK_IO.Close (File, FileStatus); -- Flow error expected
      --# end accept;
   end if;

   Status := FileStatus;

end Write;
