-- $Id: dictionary-targetdata.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with CommandLineData;
with CommandLineHandler;
with ELStrings;
with FileSystem;
with ScreenEcho;
with SystemErrors;
with XMLReport;

separate (Dictionary)
package body TargetData
is

   -- Types-------------------------------------------------------------------------------

   type ValStatus is (OkVal, MissingVal, IllegalVal);
   type ValSort   is (IntegerVal, RealVal);

   -- Local Subprograms-------------------------------------------------------------------

   procedure OpenFile (DataFile : out SPARK_IO.File_Type;
                       FileOk   : out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives DataFile,
   --#         FileOk,
   --#         SPARK_IO.File_Sys from CommandLineData.Content,
   --#                                SPARK_IO.File_Sys;
   is
      FileName       : EStrings.T;
      FileSpecStatus : FileSystem.TypFileSpecStatus;
      FileStatus     : SPARK_IO.File_Status;
      LocalFile      : SPARK_IO.File_Type;
   begin
      LocalFile := SPARK_IO.Null_File;

      --# accept Flow, 10, FileSpecStatus, "Expected ineffective assignment";
      FileSystem.FindFullFileName -- 782 expect flow error FileSpecStatus not used.
         (CommandLineData.Content.TargetDataFile,
          FileSpecStatus,  -- not used: flow error
          FileName);
      --# end accept;
      EStrings.Open (File         => LocalFile,
                     Mode_Of_File => SPARK_IO.In_File,
                     Name_Of_File => FileName,
                     Form_Of_File => "",
                     Status       => FileStatus);

      if FileStatus = SPARK_IO.Ok then
         FileOk := True;
      else
         FileOk := False;
         ScreenEcho.Put_String ("Cannot open file ");
         if CommandLineData.Content.PlainOutput then
            ScreenEcho.Put_ExaminerLine
               (EStrings.Lower_Case
                (E_Str => FileSystem.JustFile (FileName, True)));
         else
            ScreenEcho.Put_ExaminerLine (FileName);
         end if;
      end if;
      DataFile := LocalFile;
      --# accept Flow, 33, FileSpecStatus, "Expected to be neither referenced nor exported";
   end OpenFile;

   ------------------------------------------------

   procedure CloseFile (DataFile : in out SPARK_IO.File_Type)
      --# global in out SPARK_IO.File_Sys;
      --# derives DataFile,
      --#         SPARK_IO.File_Sys from *,
      --#                                DataFile;
   is
      FileStatus     : SPARK_IO.File_Status;
   begin
      --# accept Flow, 10, FileStatus, "Expected ineffective assignment";
      SPARK_IO.Close (DataFile, -- 782 Expect 2 ineffective assignments
                      FileStatus); -- not used
      --# accept Flow, 33, FileStatus, "Expected to be neither referenced nor exported";
   end CloseFile;

   ------------------------------------------------

   procedure GetString (File : in     SPARK_IO.File_Type;
                        Str  :    out EStrings.T)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         Str               from File,
      --#                                SPARK_IO.File_Sys;
   is
      CharOk   : Boolean;
      Ch       : Character;
      AppendOK : Boolean;
      EStr     : EStrings.T;

      procedure GetChar (File : in     SPARK_IO.File_Type;
                         Ch   :    out Character;
                         OK   :    out Boolean)
         --# global in out SPARK_IO.File_Sys;
         --# derives Ch,
         --#         OK,
         --#         SPARK_IO.File_Sys from File,
         --#                                SPARK_IO.File_Sys;
      is
      begin
         if SPARK_IO.End_Of_File (File) then
            OK := False;
            Ch := ' ';
         elsif SPARK_IO.End_Of_Line (File) then
            SPARK_IO.Skip_Line (File, 1);
            OK := True;
            Ch := ' ';
         else
            SPARK_IO.Get_Char (File, Ch);
            OK := True;
         end if;
      end GetChar;

   begin --GetString
      EStr := EStrings.Empty_String;
      --skip leading white space
      loop
         GetChar (File, Ch, CharOk);
         exit when Ch /= ' ';
         exit when not CharOk;
      end loop;

      if CharOk then
         loop
            EStrings.Append_Char (E_Str   => EStr,
                                  Ch      => Ch,
                                  Success => AppendOK);
            if not AppendOK then
               SystemErrors.FatalError (SystemErrors.WarningNameTooLong, "in TargetData.GetString");
            end if;
            GetChar (File, Ch, CharOk);
            exit when Ch = ' ';
            exit when not CharOk;
         end loop;
      end if;

      Str := EStr;
   end GetString;

   -------------------------------

   procedure SkipEquals (DataFile : SPARK_IO.File_Type)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                DataFile;
   is
      UnusedString : EStrings.T;
      pragma Unreferenced (UnusedString);
   begin
      --# accept Flow, 10, UnusedString, "Expected ineffective assignment" &
      --#        Flow, 33, UnusedString, "Expected to be neither referenced nor exported";
      GetString (DataFile,
                  -- to get
                 UnusedString);
   end SkipEquals;

   --------------------------------

   procedure GetDataValue (DataFile : in       SPARK_IO.File_Type;
                           Sort     : in       ValSort;
                           Val      :      out LexTokenManager.Lex_String;
                           Status   :      out ValStatus)
      --# global in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives LexTokenManager.State,
      --#         Status,
      --#         Val                   from DataFile,
      --#                                    LexTokenManager.State,
      --#                                    Sort,
      --#                                    SPARK_IO.File_Sys &
      --#         SPARK_IO.File_Sys     from *,
      --#                                    DataFile;
   is
      ValueString : EStrings.T;
      LexVal      : LexTokenManager.Lex_String;
      ConvOk      : Maths.ErrorCode;
      Num         : Maths.Value;
      IsNegative  : Boolean;
      Dummy_Char  : Character;
   begin --GetDataValue
      SkipEquals (DataFile);
      GetString (DataFile,
                  -- to get
                 ValueString);
      if EStrings.Get_Length (E_Str => ValueString) /= 0 then
         if EStrings.Get_Element (E_Str => ValueString,
                                  Pos   => 1) = '-' then
            --# accept F, 10, Dummy_Char, "Ineffective assignment here OK";
            EStrings.Pop_Char (E_Str => ValueString,
                               Char  => Dummy_Char);
            --# end accept;
            IsNegative := True;
         else
            IsNegative := False;
         end if;
         LexTokenManager.Insert_Examiner_String (Str     => ValueString,
                                                 Lex_Str => LexVal);
         Maths.LiteralToValue (LexVal,
                                 -- to get
                               Num,
                               ConvOk);
         if ConvOk = Maths.NoError then
            if Sort = IntegerVal and not Maths.IsIntegerValue (Num) then
               Val := LexTokenManager.Null_String;
               Status := IllegalVal;
            elsif Sort = RealVal and not Maths.IsRealValue (Num) then
               Val := LexTokenManager.Null_String;
               Status := IllegalVal;
            else
               if IsNegative then
                  Maths.Negate (Num);
               end if;
               Maths.StorageRep (Num, Val);
               Status := OkVal;
            end if;
         else
            Val := LexTokenManager.Null_String;
            Status := IllegalVal;
         end if;
      else
         Val := LexTokenManager.Null_String;
         Status := MissingVal;
      end if;
      --# accept F, 33, Dummy_Char, "Dummy_Char not referenced here";
   end GetDataValue;

   -----------------------------------------------------------

   procedure EchoError (Status : in     ValStatus)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Status;
   is
   begin
      case Status is
         when OkVal =>
            null;
         when IllegalVal =>
            ScreenEcho.Put_Line ("Illegal value");
         when MissingVal =>
            ScreenEcho.Put_Line ("Value missing");
      end case;
   end EchoError;

   --Exported Subprograms-----------------------------------------------------------------

   procedure ReadTargetDataFile
   is
      Option      : EStrings.T;
      FileOk      : Boolean;
      DataFile    : SPARK_IO.File_Type;

      procedure InvalidOption (Opt : EStrings.T)
         --# global in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Opt;
      is
      begin
         ScreenEcho.Put_String ("Invalid target compiler data item: ");
         ScreenEcho.Put_ExaminerLine (Opt);
      end InvalidOption;

      -------------------------------

      procedure ProcessOption (Opt : EStrings.T)
         --# global in     DataFile;
         --#        in out Dictionary.Dict;
         --#        in out LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --# derives Dictionary.Dict,
         --#         LexTokenManager.State,
         --#         SPARK_IO.File_Sys     from *,
         --#                                    DataFile,
         --#                                    LexTokenManager.State,
         --#                                    Opt,
         --#                                    SPARK_IO.File_Sys;
      is
         OptionMatch : Boolean;
         Val         : LexTokenManager.Lex_String;
         Status      : ValStatus;

      begin --ProcessOption
         OptionMatch := False;
         case EStrings.Get_Element (E_Str => Opt,
                                    Pos   => 1) is
            when 'i' | 'I' =>
               case EStrings.Get_Element (E_Str => Opt,
                                          Pos   => 9) is
                  when 'f' | 'F' =>
                     CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                           Str      => "integer'first",
                                                           OK       => OptionMatch);
                     if OptionMatch then
                        GetDataValue (DataFile, IntegerVal, Val, Status);
                        EchoError (Status);
                        RawDict.SetTypeLower (Dictionary.GetPredefinedIntegerType, Val);
                     end if;

                  when 'l' | 'L' =>
                     CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                           Str      => "integer'last",
                                                           OK       => OptionMatch);
                     if OptionMatch then
                        GetDataValue (DataFile, IntegerVal, Val, Status);
                        EchoError (Status);
                        RawDict.SetTypeUpper (Dictionary.GetPredefinedIntegerType, Val);
                        RawDict.SetTypeUpper (Dictionary.GetPredefinedPositiveSubtype, Val);
                        RawDict.SetTypeUpper (Dictionary.GetPredefinedNaturalSubtype, Val);
                     end if;

                  when others =>
                     null;

               end case;

            when 'l' | 'L' =>
               case EStrings.Get_Element (E_Str => Opt,
                                          Pos   => 14) is
                  when 'f' | 'F' =>
                     CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                           Str      => "long_integer'first",
                                                           OK       => OptionMatch);
                     if OptionMatch then
                        GetDataValue (DataFile, IntegerVal, Val, Status);
                        EchoError (Status);
                        RawDict.SetTypeLower (Dictionary.GetPredefinedLongIntegerType, Val);
                     end if;

                  when 'l' | 'L' =>
                     CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                           Str      => "long_integer'last",
                                                           OK       => OptionMatch);
                     if OptionMatch then
                        GetDataValue (DataFile, IntegerVal, Val, Status);
                        EchoError (Status);
                        RawDict.SetTypeUpper (Dictionary.GetPredefinedLongIntegerType, Val);
                     end if;

                  when others =>
                     null;

               end case;

            when others =>
               null;
         end case;

         if not OptionMatch then
            InvalidOption (Opt);
         end if;
      end ProcessOption;

      -------------------------------

   begin --ReadTargetDataFile
      if CommandLineData.Content.TargetData then
         OpenFile (DataFile,
                     -- to get
                   FileOk);
         if FileOk then
            if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
               ScreenEcho.New_Line (1);
               ScreenEcho.Put_Line ("           Reading target compiler data ...");
            end if;
            loop
               GetString (DataFile,
                           -- to get
                          Option);
               exit when EStrings.Get_Length (E_Str => Option) = 0;
               ProcessOption (Option);
            end loop;
            --# accept Flow, 10, DataFile, "Expected ineffective assignment";
            CloseFile (DataFile); -- Expect ineff assignment here
            --# end accept;
         end if;
      end if;
   end ReadTargetDataFile;

   ------------------------------------------------------------------------------

   procedure OutputTargetDataFile (ToFile : in SPARK_IO.File_Type)
   is
      Option      : EStrings.T;
      FileOk      : Boolean;
      DataFile    : SPARK_IO.File_Type;

      --------------------------------

      procedure InvalidOption (ToFile : SPARK_IO.File_Type;
                               Opt : EStrings.T)
         --# global in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Opt,
         --#                                ToFile;
      is
      begin
         SPARK_IO.Put_String (ToFile, "Invalid target compiler data item: ", 0);
         EStrings.Put_String (File  => ToFile,
                              E_Str => Opt);
         SPARK_IO.Put_Line (ToFile, " has been ignored", 0);
      end InvalidOption;

      -------------------------------

      procedure ProcessOption (Opt    : EStrings.T;
                               ToFile : SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     DataFile;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out XMLReport.State;
      --# derives LexTokenManager.State from *,
      --#                                    CommandLineData.Content,
      --#                                    DataFile,
      --#                                    Opt,
      --#                                    SPARK_IO.File_Sys,
      --#                                    ToFile &
      --#         SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    DataFile,
      --#                                    LexTokenManager.State,
      --#                                    Opt,
      --#                                    ToFile,
      --#                                    XMLReport.State &
      --#         XMLReport.State       from *,
      --#                                    CommandLineData.Content,
      --#                                    Opt;
      is
         OptionMatch : Boolean;
         --Val         : LexTokenManager.Lex_String;
         --Status      : ValStatus;

         procedure Margin
            --# global in     ToFile;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                ToFile;
         is
         begin
            SPARK_IO.Put_String (ToFile, "   ", 0);
         end Margin;

         ----------------------------------------

         procedure Separator
            --# global in     ToFile;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                ToFile;
         is
         begin
            SPARK_IO.Put_String (ToFile, " = ", 0);
         end Separator;

         ----------------------------------------

         procedure PutVal (ToFile : in     SPARK_IO.File_Type;
                           Val    : in     LexTokenManager.Lex_String;
                           Status : in     ValStatus)
            --# global in     LexTokenManager.State;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                LexTokenManager.State,
            --#                                Status,
            --#                                ToFile,
            --#                                Val;
         is
         begin
            case Status is
               when OkVal =>
                  ELStrings.Put_Line (File  => ToFile,
                                      E_Str => Maths.ValueToString (Maths.ValueRep (Val)));
               when IllegalVal =>
                  SPARK_IO.Put_Line (ToFile, "Illegal value - ignored", 0);
               when MissingVal =>
                  SPARK_IO.Put_Line (ToFile, "Value missing - ignored", 0);
            end case;
         end PutVal;


         function GetVal (Val    : in LexTokenManager.Lex_String;
                          Status : in ValStatus) return ELStrings.T
         --# global in LexTokenManager.State;
         is
            TmpString : ELStrings.T;
         begin
            case Status is
               when OkVal =>
                  TmpString := Maths.ValueToString (Maths.ValueRep (Val));
               when IllegalVal =>
                  TmpString := ELStrings.Copy_String (Str => "Illegal value - ignored");
               when MissingVal =>
                  TmpString := ELStrings.Copy_String (Str => "Value missing - ignored");
            end case;
            return TmpString;
         end GetVal;



         ---------------------------------------

         procedure OutputValue (DataFile : in     SPARK_IO.File_Type;
                                Opt      : in     EStrings.T;
                                Valid    : in     Boolean)
         --# global in     CommandLineData.Content;
         --#        in     ToFile;
         --#        in out LexTokenManager.State;
         --#        in out SPARK_IO.File_Sys;
         --#        in out XMLReport.State;
         --# derives LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    DataFile,
         --#                                    Opt,
         --#                                    SPARK_IO.File_Sys,
         --#                                    ToFile,
         --#                                    Valid &
         --#         SPARK_IO.File_Sys     from *,
         --#                                    CommandLineData.Content,
         --#                                    DataFile,
         --#                                    LexTokenManager.State,
         --#                                    Opt,
         --#                                    ToFile,
         --#                                    Valid,
         --#                                    XMLReport.State &
         --#         XMLReport.State       from *,
         --#                                    CommandLineData.Content,
         --#                                    Opt,
         --#                                    Valid;
         is
            Val    : LexTokenManager.Lex_String;
            Status : ValStatus;
         begin
            if Valid then
               if CommandLineData.Content.XML then
                  GetDataValue (DataFile, IntegerVal, Val, Status);
                  XMLReport.LongCompilerItem (Opt,
                                              GetVal (Val, Status),
                                              ToFile);
               else
                  Margin;
                  EStrings.Put_String (File  => ToFile,
                                       E_Str => Opt);
                  Separator;
                  GetDataValue (DataFile, IntegerVal, Val, Status);
                  PutVal (ToFile, Val, Status);
               end if;
            end if;
         end OutputValue;

         ---------------------------------------

      begin --ProcessOption
         OptionMatch := False;
         case EStrings.Get_Element (E_Str => Opt,
                                    Pos   => 1) is
            when 'i' | 'I' =>
               case EStrings.Get_Element (E_Str => Opt,
                                          Pos   => 9) is
                  when 'f' | 'F' =>
                     CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                           Str      => "integer'first",
                                                           OK       => OptionMatch);
                     OutputValue (DataFile, Opt, OptionMatch);

                  when 'l' | 'L' =>
                     CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                           Str      => "integer'last",
                                                           OK       => OptionMatch);
                     OutputValue (DataFile, Opt, OptionMatch);

                  when others =>
                     null;
               end case;

            when 'l' | 'L' =>
               case EStrings.Get_Element (E_Str => Opt,
                                          Pos   => 14) is
                  when 'f' | 'F' =>
                     CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                           Str      => "long_integer'first",
                                                           OK       => OptionMatch);
                     OutputValue (DataFile, Opt, OptionMatch);

                  when 'l' | 'L' =>
                     CommandLineHandler.Check_Option_Name (Opt_Name => Opt,
                                                           Str      => "long_integer'last",
                                                           OK       => OptionMatch);
                     OutputValue (DataFile, Opt, OptionMatch);

                  when others =>
                     null;
               end case;

            when others =>
               null;
         end case;

         if not OptionMatch then
            Margin;
            InvalidOption (ToFile, Opt);
         end if;
      end ProcessOption;

      -------------------------------

   begin --OutputTargetDataFile
      if CommandLineData.Content.TargetData then
         OpenFile (DataFile,
                   FileOk);
         if FileOk then
            if CommandLineData.Content.XML then
               XMLReport.StartSection (XMLReport.SCompilerData,
                                       ToFile);
               loop
                  GetString (DataFile,
                              -- to get
                             Option);
                  exit when EStrings.Get_Length (E_Str => Option) = 0;
                  ProcessOption (Option, ToFile);
               end loop;
               XMLReport.EndSection (XMLReport.SCompilerData,
                                     ToFile);
               --# accept Flow, 10, DataFile, "Expected ineffective assignment";
               CloseFile (DataFile); -- Expect ineff assignment here
               --# end accept;
            else
               SPARK_IO.New_Line (ToFile, 2);
               SPARK_IO.Put_Line (ToFile, "Target compiler data:", 0);
               loop
                  GetString (DataFile,
                              -- to get
                             Option);
                  exit when EStrings.Get_Length (E_Str => Option) = 0;
                  ProcessOption (Option, ToFile);
               end loop;
               --# accept Flow, 10, DataFile, "Expected ineffective assignment";
               CloseFile (DataFile); -- Expect ineff assignment here
               --# end accept;
            end if; -- XML
         end if;
      end if;
   end OutputTargetDataFile;

   ------------------------------------------------------------------------------

end TargetData;
