-- $Id: dictionary-initialize.adb 16567 2010-03-25 16:09:40Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Maths,
     CommandLineData;

separate (Dictionary)
procedure Initialize (WriteToFile : in Boolean)
is

   File        : SPARK_IO.File_Type := SPARK_IO.Null_File;
   Status      : SPARK_IO.File_Status;

   --------------------------------------------------------------------------------

   procedure AssignInitialValue
      --# global out Dict;
      --# derives Dict from ;
   is
   begin
      Dict.FileName := EStrings.Empty_String;
      Dict.TemporaryFile   := SPARK_IO.Null_File;
      Dict.Symbols.Current := NullSymbol;
      Dict.Main := MainProgramSet'(Subprogram     => NullSymbol,
                                   InheritClauses => NullSymbol,
                                   PriorityGiven  => False,
                                   ThePriority    => LexTokenManager.Null_String);
      Dict.Packages := PackageSet'(NullSymbol, NullSymbol, NullSymbol, NullSymbol,
                                   NullSymbol, NullSymbol, NullSymbol, NullSymbol);
      --# accept Flow, 32, Dict.Symbols.List, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.Symbols.List, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.Symbols.List, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.Types, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.Types, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.Types, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.Subprograms, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.Subprograms, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.Subprograms, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.NullVariable, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.NullVariable, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.NullVariable, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 32, Dict.ThePartition, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 31, Dict.ThePartition, "Flow errors due to undefined components of Dict OK" &
      --#        Flow, 602, Dict, Dict.ThePartition, "Flow errors due to undefined components of Dict OK";
   end AssignInitialValue;

   --------------------------------------------------------------------------------

   procedure Insert_Lex_String (Name  : in     String;
                                Token :    out LexTokenManager.Lex_String)
   --# global in out LexTokenManager.State;
   --# derives LexTokenManager.State,
   --#         Token                 from LexTokenManager.State,
   --#                                    Name;
   is
   begin
      LexTokenManager.Insert_Examiner_String
        (Str     => EStrings.Copy_String (Str => Name),
         Lex_Str => Token);
   end Insert_Lex_String;

   --------------------------------------------------------------------------------

   procedure AddUnknownType
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    LexTokenManager.State;
   is
      Token   : LexTokenManager.Lex_String;
      TheType : Symbol;
   begin
      Insert_Lex_String (Name  => "unknown",
                         Token => Token);
      RawDict.CreateType (Name      => Token,
                          Comp_Unit => ContextManager.NullUnit,
                          Loc       => LexTokenManager.Null_Token_Position,
                          TypeMark  => TheType);
      RawDict.SetTypeDiscriminant (TheType, UnknownType);
      RawDict.SetTypeStatic (TheType, True);
      Dict.Types.UnknownTypeMark := TheType;
   end AddUnknownType;

   --------------------------------------------------------------------------------

   procedure AddPredefinedPackageStandard
   --# global in out Dict;
   --# derives Dict from *;
   is
      NewPackage, Specification : Symbol;
   begin
      RawDict.CreatePackage (Name       => LexTokenManager.Standard_Token,
                             Comp_Unit  => ContextManager.NullUnit,
                             Loc        => LexTokenManager.Null_Token_Position,
                             ThePackage => NewPackage);
      RawDict.CreateDeclaration (Item        => NewPackage,
                                 Context     => ProgramContext,
                                 Comp_Unit   => ContextManager.NullUnit,
                                 Loc         => LexTokenManager.Null_Token_Position,
                                 Declaration => Specification);
      RawDict.SetPackageSpecification (NewPackage, Specification);
      Dict.Packages.PackageStandard := NewPackage;
   end AddPredefinedPackageStandard;

   --------------------------------------------------------------------------------

   procedure AddPackageStandard
   --# global in     CommandLineData.Content;
   --#        in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    CommandLineData.Content,
   --#                                    LexTokenManager.State;
   is

      procedure Add_Predefined_Type (Token    : in     LexTokenManager.Lex_String;
                                     Context  : in     Contexts;
                                     The_Type :    out Symbol)
      --# global in out Dict;
      --# derives Dict     from *,
      --#                       Context,
      --#                       Token &
      --#         The_Type from Dict;
      is
         New_Type    : Symbol;
         Declaration : Symbol;
      begin
         RawDict.CreateType (Name      => Token,
                             Comp_Unit => ContextManager.NullUnit,
                             Loc       => LexTokenManager.Null_Token_Position,
                             TypeMark  => New_Type);
         AddDeclaration (Item        => New_Type,
                         Comp_Unit   => ContextManager.NullUnit,
                         Loc         => Null_Location,
                         Scope       => PredefinedScope,
                         Context     => Context,
                         Declaration => Declaration);
         RawDict.SetTypeDeclaration (New_Type, Declaration);
         The_Type := New_Type;
      end Add_Predefined_Type;

      --------------------------------------------------------------------------------

      procedure AddUniversalTypes
      --# global in out Dict;
      --# derives Dict from *;
      is

         --------------------------------------------------------------------------------

         procedure AddUniversalIntegerType
         --# global in out Dict;
         --# derives Dict from *;
         is
            TheType : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Universal_Integer_Token,
                                 Context  => ProofContext,
                                 The_Type => TheType);
            AddIntegerTypeMark (TheType,
                                True,
                                LexTokenManager.Null_String,
                                LexTokenManager.Null_String);
            Dict.Types.TheUniversalTypes (UniversalIntegerType) := TheType;
         end AddUniversalIntegerType;

         --------------------------------------------------------------------------------

         procedure AddUniversalRealType
         --# global in out Dict;
         --# derives Dict from *;
         is
            TheType : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Universal_Real_Token,
                                 Context  => ProofContext,
                                 The_Type => TheType);
            AddFloatingPointTypeMark (TheType, True);
            Dict.Types.TheUniversalTypes (UniversalRealType) := TheType;
         end AddUniversalRealType;

         --------------------------------------------------------------------------------

         procedure AddUniversalFixedType
         --# global in out Dict;
         --# derives Dict from *;
         is
            TheType : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Universal_Fixed_Token,
                                 Context  => ProofContext,
                                 The_Type => TheType);
            AddFixedPointTypeMark (TheType, True);
            Dict.Types.TheUniversalTypes (UniversalFixedType) := TheType;
         end AddUniversalFixedType;

         --------------------------------------------------------------------------------

      begin
         AddUniversalIntegerType;
         AddUniversalRealType;
         AddUniversalFixedType;
      end AddUniversalTypes;

      --------------------------------------------------------------------------------

      procedure AddPredefinedTypeMarks
      --# global in     CommandLineData.Content;
      --#        in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict,
      --#         LexTokenManager.State from *,
      --#                                    CommandLineData.Content,
      --#                                    LexTokenManager.State;
      is

         --------------------------------------------------------------------------------

         procedure AddPredefinedCharacterType
         --# global in     CommandLineData.Content;
         --#        in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    LexTokenManager.State;
         is
            TheType  : Symbol;
            StoreRep : LexTokenManager.Lex_String;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Character_Token,
                                 Context  => ProgramContext,
                                 The_Type => TheType);
            AddEnumerationTypeMark (TheType, True);
            -- Set bounds for character type for use with 'first, 'last
            RawDict.SetTypeLower (TheType, LexTokenManager.Zero_Value);

            case CommandLineData.Content.LanguageProfile is
               when CommandLineData.SPARK83 =>
                  Maths.StorageRep (Maths.IntegerToValue (127), StoreRep);
                  RawDict.SetTypeUpper (TheType, StoreRep);

               when CommandLineData.SPARK95 |
                 CommandLineData.SPARK2005 =>
                  Maths.StorageRep (Maths.IntegerToValue (255), StoreRep);
                  RawDict.SetTypeUpper (TheType, StoreRep);
            end case;

            Dict.Types.ThePredefinedTypes (PredefinedCharacterType) := TheType;
         end AddPredefinedCharacterType;

         --------------------------------------------------------------------------------

         procedure AddPredefinedBooleanType
         --# global in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    LexTokenManager.State;
         is

            TheType : Symbol;

            --------------------------------------------------------------------------------

            procedure AddPredefinedEnumerationLiteral (Name    : in String;
                                                       Value   : in LexTokenManager.Lex_String;
                                                       TheType : in Symbol)
            --# global in out Dict;
            --#        in out LexTokenManager.State;
            --# derives Dict                  from *,
            --#                                    LexTokenManager.State,
            --#                                    Name,
            --#                                    TheType,
            --#                                    Value &
            --#         LexTokenManager.State from *,
            --#                                    Name;
            is
               Token : LexTokenManager.Lex_String;
            begin
               Insert_Lex_String (Name  => Name,
                                  Token => Token);
               AddEnumerationLiteral (Name        => Token,
                                      Comp_Unit   => ContextManager.NullUnit,
                                      Declaration => Null_Location,
                                      Position    => Value,
                                      TheType     => TheType);
            end AddPredefinedEnumerationLiteral;

            --------------------------------------------------------------------------------

         begin
            Add_Predefined_Type (Token    => LexTokenManager.Boolean_Token,
                                 Context  => ProgramContext,
                                 The_Type => TheType);
            AddEnumerationTypeMark (TheType, True);
            AddPredefinedEnumerationLiteral ("False", LexTokenManager.False_Token, TheType);
            AddPredefinedEnumerationLiteral ("True", LexTokenManager.True_Token, TheType);
            Dict.Types.ThePredefinedTypes (PredefinedBooleanType) := TheType;
            Dict.Types.ThePredefinedLiterals (PredefinedFalse) :=
               RawDict.GetTypeFirstEnumerationLiteral (TheType);
            Dict.Types.ThePredefinedLiterals (PredefinedTrue) :=
               RawDict.GetTypeLastEnumerationLiteral (TheType);
         end AddPredefinedBooleanType;

         --------------------------------------------------------------------------------

         procedure AddPredefinedIntegerType
         --# global in     CommandLineData.Content;
         --#        in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    LexTokenManager.State;
         is
            Token   : LexTokenManager.Lex_String;
            TheType : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Integer_Token,
                                 Context  => ProgramContext,
                                 The_Type => TheType);
            AddIntegerTypeMark (TheType,
                                True,
                                LexTokenManager.Null_String,
                                LexTokenManager.Null_String);
            SetTypeIsPredefined (TheType);
            Dict.Types.ThePredefinedTypes (PredefinedIntegerType) := TheType;
            if CommandLineData.Content.TargetData then
               Insert_Lex_String (Name  => "Long_Integer",
                                  Token => Token);
               Add_Predefined_Type (Token    => Token,
                                    Context  => ProgramContext,
                                    The_Type => TheType);
               AddIntegerTypeMark (TheType,
                                   True,
                                   LexTokenManager.Null_String,
                                   LexTokenManager.Null_String);
               SetTypeIsPredefined (TheType);
               Dict.Types.ThePredefinedTypes (PredefinedLongIntegerType) := TheType;
            else
               Dict.Types.ThePredefinedTypes (PredefinedLongIntegerType) := NullSymbol;
            end if;
         end AddPredefinedIntegerType;

         --------------------------------------------------------------------------------

         procedure AddPredefinedFloatType
         --# global in     CommandLineData.Content;
         --#        in out Dict;
         --#        in out LexTokenManager.State;
         --# derives Dict,
         --#         LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    LexTokenManager.State;
         is
            Token   : LexTokenManager.Lex_String;
            TheType : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Float_Token,
                                 Context  => ProgramContext,
                                 The_Type => TheType);
            AddFloatingPointTypeMark (TheType, True);
            SetTypeIsPredefined (TheType);
            Dict.Types.ThePredefinedTypes (PredefinedFloatType) := TheType;
            if CommandLineData.Content.TargetData then
               Insert_Lex_String (Name  => "Long_Float",
                                  Token => Token);
               Add_Predefined_Type (Token    => Token,
                                    Context  => ProgramContext,
                                    The_Type => TheType);
               AddFloatingPointTypeMark (TheType, True);
               SetTypeIsPredefined (TheType);
               Dict.Types.ThePredefinedTypes (PredefinedLongFloatType) := TheType;
            else
               Dict.Types.ThePredefinedTypes (PredefinedLongFloatType) := NullSymbol;
            end if;
         end AddPredefinedFloatType;

         --------------------------------------------------------------------------------

         procedure AddDuration
         --# global in out Dict;
         --# derives Dict from *;
         is
            TheType : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Duration_Token,
                                 Context  => ProgramContext,
                                 The_Type => TheType);
            AddFixedPointTypeMark (TheType, True);
            Dict.Types.ThePredefinedTypes (PredefinedDurationType) := TheType;
         end AddDuration;

         --------------------------------------------------------------------------------

         procedure AddPredefinedStringType
         --# global in out Dict;
         --# derives Dict from *;
         is
            TheType : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.String_Token,
                                 Context  => ProgramContext,
                                 The_Type => TheType);
            AddArrayTypeMark (TheType, GetPredefinedCharacterType, False);
            AddNewArrayIndex (TheArrayType => TheType,
                              IndexType    => GetPredefinedPositiveSubtype,
                              Comp_Unit    => ContextManager.NullUnit,
                              Declaration  => Null_Location);
            Dict.Types.ThePredefinedTypes (PredefinedStringType) := TheType;
         end AddPredefinedStringType;

         --------------------------------------------------------------------------------

         procedure AddPredefinedNaturalSubtype
         --# global in out Dict;
         --# derives Dict from *;
         is
            TheType : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Natural_Token,
                                 Context  => ProgramContext,
                                 The_Type => TheType);
            AddSubtype (TheType, GetPredefinedIntegerType);
            AddIntegerTypeMark (TheType,
                                True,
                                LexTokenManager.Zero_Value,
                                LexTokenManager.Null_String);
            Dict.Types.ThePredefinedTypes (PredefinedNaturalSubtype) := TheType;
         end AddPredefinedNaturalSubtype;

         --------------------------------------------------------------------------------

         procedure AddPredefinedPositiveSubtype
         --# global in out Dict;
         --# derives Dict from *;
         is
            TheType : Symbol;
         begin
            Add_Predefined_Type (Token    => LexTokenManager.Positive_Token,
                                 Context  => ProgramContext,
                                 The_Type => TheType);
            AddSubtype (TheType, GetPredefinedIntegerType);
            AddIntegerTypeMark (TheType,
                                True,
                                LexTokenManager.One_Value,
                                LexTokenManager.Null_String);
            Dict.Types.ThePredefinedTypes (PredefinedPositiveSubtype) := TheType;
         end AddPredefinedPositiveSubtype;

      begin
         AddPredefinedBooleanType;
         AddPredefinedIntegerType;
         AddPredefinedFloatType;
         AddPredefinedCharacterType;
         AddPredefinedNaturalSubtype;
         AddPredefinedPositiveSubtype;
         AddPredefinedStringType;
         if not CommandLineData.Content.NoDuration then
            AddDuration;
         end if;
      end AddPredefinedTypeMarks;

      --------------------------------------------------------------------------------

      procedure AddNullOwnVariable
      --# global in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict,
      --#         LexTokenManager.State from *,
      --#                                    LexTokenManager.State;
      is
         NewVariable,
         OwnVariable,
         PackageStandard : Symbol;
         NullStr         : LexTokenManager.Lex_String;

      begin -- AddNullOwnVariable
         PackageStandard := Dict.Packages.PackageStandard;
         Insert_Lex_String (Name  => "null",
                            Token => NullStr);
         RawDict.CreateVariable (Name      => NullStr,
                                 Comp_Unit => ContextManager.NullUnit,
                                 Loc       => LexTokenManager.Null_Token_Position,
                                 Variable  => NewVariable);
         RawDict.CreateOwnVariable (Variable    => NewVariable,
                                    Owner       => PackageStandard,
                                    Comp_Unit   => ContextManager.NullUnit,
                                    Loc         => LexTokenManager.Null_Token_Position,
                                    OwnVariable => OwnVariable);
         RawDict.SetVariableOwnVariable (NewVariable, OwnVariable);
         RawDict.SetPackageOwnVariables (PackageStandard, OwnVariable);
         RawDict.SetOwnVariableMode (OwnVariable, OutMode);
         Dict.NullVariable := NewVariable;
      end AddNullOwnVariable;

      --------------------------------------------------------------------------------

   begin
      AddPredefinedPackageStandard;
      AddUniversalTypes;
      AddPredefinedTypeMarks;
      -- package standard has a mode out own variable called "Null" which is used
      -- as a "data sink" for operations that affect only things outside the SPARK
      -- boundary such as delay statements or calls to procedures which derive null
      -- from something.
      AddNullOwnVariable;
   end AddPackageStandard;

   --------------------------------------------------------------------------------


   procedure AddPredefinedConstant (Name       : in String;
                                    AscCode    : in Natural;
                                    ThePackage : in Symbol)
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict                  from *,
   --#                                    AscCode,
   --#                                    LexTokenManager.State,
   --#                                    Name,
   --#                                    ThePackage &
   --#         LexTokenManager.State from *,
   --#                                    AscCode,
   --#                                    Name;
   is
      Token       : LexTokenManager.Lex_String;
      TheConstant : Symbol;
      Declaration : Symbol;
      StoreRep    : LexTokenManager.Lex_String;
   begin
      Insert_Lex_String (Name  => Name,
                         Token => Token);
      RawDict.CreateConstant (Name        => Token,
                              TypeMark    => GetPredefinedCharacterType,
                              Static      => True,
                              Comp_Unit   => ContextManager.NullUnit,
                              Loc         => LexTokenManager.Null_Token_Position,
                              TheConstant => TheConstant);
      -- Add values for predefined constants
      Maths.StorageRep (Maths.IntegerToValue (AscCode), StoreRep);
      RawDict.SetConstantValue (TheConstant, StoreRep);
      AddDeclaration (Item        => TheConstant,
                      Comp_Unit   => ContextManager.NullUnit,
                      Loc         => Null_Location,
                      Scope       => VisibleScope (ThePackage),
                      Context     => ProgramContext,
                      Declaration => Declaration);
      RawDict.SetConstantDeclaration (TheConstant, Declaration);
   end AddPredefinedConstant;

   -------------------------------------

   -- Adds Character constants which are common to both
   -- SPARK83 and SPARK95 _AND_ have the same identifier - this
   -- is a subset of the first 127 Characters.
   procedure AddCommonCharacters (ToThePackage : in Symbol)
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict                  from *,
   --#                                    LexTokenManager.State,
   --#                                    ToThePackage &
   --#         LexTokenManager.State from *;
   is
   begin
      AddPredefinedConstant ("NUL", 0, ToThePackage);
      AddPredefinedConstant ("SOH", 1, ToThePackage);
      AddPredefinedConstant ("STX", 2, ToThePackage);
      AddPredefinedConstant ("ETX", 3, ToThePackage);
      AddPredefinedConstant ("EOT", 4, ToThePackage);
      AddPredefinedConstant ("ENQ", 5, ToThePackage);
      AddPredefinedConstant ("ACK", 6, ToThePackage);
      AddPredefinedConstant ("BEL", 7, ToThePackage);
      AddPredefinedConstant ("BS", 8, ToThePackage);
      AddPredefinedConstant ("HT", 9, ToThePackage);
      AddPredefinedConstant ("LF", 10, ToThePackage);
      AddPredefinedConstant ("VT", 11, ToThePackage);
      AddPredefinedConstant ("FF", 12, ToThePackage);
      AddPredefinedConstant ("CR", 13, ToThePackage);
      AddPredefinedConstant ("SO", 14, ToThePackage);
      AddPredefinedConstant ("SI", 15, ToThePackage);
      AddPredefinedConstant ("DLE", 16, ToThePackage);
      AddPredefinedConstant ("DC1", 17, ToThePackage);
      AddPredefinedConstant ("DC2", 18, ToThePackage);
      AddPredefinedConstant ("DC3", 19, ToThePackage);
      AddPredefinedConstant ("DC4", 20, ToThePackage);
      AddPredefinedConstant ("NAK", 21, ToThePackage);
      AddPredefinedConstant ("SYN", 22, ToThePackage);
      AddPredefinedConstant ("ETB", 23, ToThePackage);
      AddPredefinedConstant ("CAN", 24, ToThePackage);
      AddPredefinedConstant ("EM", 25, ToThePackage);
      AddPredefinedConstant ("SUB", 26, ToThePackage);
      AddPredefinedConstant ("ESC", 27, ToThePackage);
      AddPredefinedConstant ("FS", 28, ToThePackage);
      AddPredefinedConstant ("GS", 29, ToThePackage);
      AddPredefinedConstant ("RS", 30, ToThePackage);
      AddPredefinedConstant ("US", 31, ToThePackage);

      --# assert True;

      AddPredefinedConstant ("LC_A", 97, ToThePackage);
      AddPredefinedConstant ("LC_B", 98, ToThePackage);
      AddPredefinedConstant ("LC_C", 99, ToThePackage);
      AddPredefinedConstant ("LC_D", 100, ToThePackage);
      AddPredefinedConstant ("LC_E", 101, ToThePackage);
      AddPredefinedConstant ("LC_F", 102, ToThePackage);
      AddPredefinedConstant ("LC_G", 103, ToThePackage);
      AddPredefinedConstant ("LC_H", 104, ToThePackage);
      AddPredefinedConstant ("LC_I", 105, ToThePackage);
      AddPredefinedConstant ("LC_J", 106, ToThePackage);
      AddPredefinedConstant ("LC_K", 107, ToThePackage);
      AddPredefinedConstant ("LC_L", 108, ToThePackage);
      AddPredefinedConstant ("LC_M", 109, ToThePackage);
      AddPredefinedConstant ("LC_N", 110, ToThePackage);
      AddPredefinedConstant ("LC_O", 111, ToThePackage);
      AddPredefinedConstant ("LC_P", 112, ToThePackage);
      AddPredefinedConstant ("LC_Q", 113, ToThePackage);
      AddPredefinedConstant ("LC_R", 114, ToThePackage);
      AddPredefinedConstant ("LC_S", 115, ToThePackage);
      AddPredefinedConstant ("LC_T", 116, ToThePackage);
      AddPredefinedConstant ("LC_U", 117, ToThePackage);
      AddPredefinedConstant ("LC_V", 118, ToThePackage);
      AddPredefinedConstant ("LC_W", 119, ToThePackage);
      AddPredefinedConstant ("LC_X", 120, ToThePackage);
      AddPredefinedConstant ("LC_Y", 121, ToThePackage);
      AddPredefinedConstant ("LC_Z", 122, ToThePackage);

      AddPredefinedConstant ("DEL", 127, ToThePackage);
   end AddCommonCharacters;

   --------------------------------------------------------------------------------

   procedure AddPackageASCII
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    LexTokenManager.State;
   is
      Token        : LexTokenManager.Lex_String;
      PackageASCII : Symbol;
   begin
      Insert_Lex_String (Name  => "ASCII",
                         Token => Token);
      AddNewPackage (Name        => Token,
                     Comp_Unit   => ContextManager.NullUnit,
                     Declaration => Null_Location,
                     Scope       => PredefinedScope,
                     TheParent   => NullSymbol,
                     ChildSort   => Public,
                     ThePackage  => PackageASCII);
      Dict.Packages.PackageASCII := PackageASCII;

      AddCommonCharacters (PackageASCII);

      -- Ada83 Characters whose identiifers differ from those used in Ada95.
      AddPredefinedConstant ("Exclam", 33, PackageASCII);
      AddPredefinedConstant ("Quotation", 34, PackageASCII);
      AddPredefinedConstant ("Sharp", 35, PackageASCII);
      AddPredefinedConstant ("Dollar", 36, PackageASCII);
      AddPredefinedConstant ("Percent", 37, PackageASCII);
      AddPredefinedConstant ("Ampersand", 38, PackageASCII);
      AddPredefinedConstant ("Colon", 58, PackageASCII);
      AddPredefinedConstant ("Semicolon", 59, PackageASCII);
      AddPredefinedConstant ("Query", 63, PackageASCII);
      AddPredefinedConstant ("At_Sign", 64, PackageASCII);
      AddPredefinedConstant ("L_Bracket", 91, PackageASCII);
      AddPredefinedConstant ("Back_Slash", 92, PackageASCII);
      AddPredefinedConstant ("R_Bracket", 93, PackageASCII);
      AddPredefinedConstant ("Circumflex", 94, PackageASCII);
      AddPredefinedConstant ("Underline", 95, PackageASCII);
      AddPredefinedConstant ("Grave", 96, PackageASCII);
      AddPredefinedConstant ("L_Brace", 123, PackageASCII);
      AddPredefinedConstant ("Bar", 124, PackageASCII);
      AddPredefinedConstant ("R_Brace", 125, PackageASCII);
      AddPredefinedConstant ("Tilde", 126, PackageASCII);
   end AddPackageASCII;

   --------------------------------------------------------------------------------

   procedure AddPackageAda
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --# derives Dict,
   --#         LexTokenManager.State from *,
   --#                                    LexTokenManager.State;
   is
      Token              : LexTokenManager.Lex_String;
      PackageAda,
      PackageCharacters,
      PackageLatin1      : Symbol;


      procedure AddLowerHalfControlAndGraphicCharacters
      --# global in     PackageLatin1;
      --#        in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict                  from *,
      --#                                    LexTokenManager.State,
      --#                                    PackageLatin1 &
      --#         LexTokenManager.State from *;
      is
      begin
         --ISO 6429 control characters
         AddPredefinedConstant ("IS4", 28, PackageLatin1);
         AddPredefinedConstant ("IS3", 29, PackageLatin1);
         AddPredefinedConstant ("IS2", 30, PackageLatin1);
         AddPredefinedConstant ("IS1", 31, PackageLatin1);

         --ISO 646 graphic characters
         AddPredefinedConstant ("Space", 32, PackageLatin1);
         AddPredefinedConstant ("Exclamation", 33, PackageLatin1);
         AddPredefinedConstant ("Quotation", 34, PackageLatin1);
         AddPredefinedConstant ("Number_Sign", 35, PackageLatin1);
         AddPredefinedConstant ("Dollar_Sign", 36, PackageLatin1);
         AddPredefinedConstant ("Percent_Sign", 37, PackageLatin1); --596
         AddPredefinedConstant ("Ampersand", 38, PackageLatin1);
         AddPredefinedConstant ("Apostrophe", 39, PackageLatin1);
         AddPredefinedConstant ("Left_Parenthesis", 40, PackageLatin1);
         AddPredefinedConstant ("Right_Parenthesis", 41, PackageLatin1);
         AddPredefinedConstant ("Asterisk", 42, PackageLatin1);
         AddPredefinedConstant ("Plus_Sign", 43, PackageLatin1);
         AddPredefinedConstant ("Comma", 44, PackageLatin1);
         AddPredefinedConstant ("Hyphen", 45, PackageLatin1);
         AddPredefinedConstant ("Minus_Sign", 45, PackageLatin1);
         AddPredefinedConstant ("Full_Stop", 46, PackageLatin1);
         AddPredefinedConstant ("Solidus", 47, PackageLatin1);

         --Decimal digits '0' through '9 are at positions 48 through 57

         AddPredefinedConstant ("Colon", 58, PackageLatin1);
         AddPredefinedConstant ("Semicolon", 59, PackageLatin1);
         AddPredefinedConstant ("Less_Than_Sign", 60, PackageLatin1);
         AddPredefinedConstant ("Equals_Sign", 61, PackageLatin1);
         AddPredefinedConstant ("Greater_Than_Sign", 62, PackageLatin1);
         AddPredefinedConstant ("Question", 63, PackageLatin1);
         AddPredefinedConstant ("Commercial_At", 64, PackageLatin1);

         --Letters 'A' through 'Z' are at positions 65 through 90

         AddPredefinedConstant ("Left_Square_Bracket", 91, PackageLatin1);
         AddPredefinedConstant ("Reverse_Solidus", 92, PackageLatin1);
         AddPredefinedConstant ("Right_Square_Bracket", 93, PackageLatin1);
         AddPredefinedConstant ("Circumflex", 94, PackageLatin1);
         AddPredefinedConstant ("Low_Line", 95, PackageLatin1);
         AddPredefinedConstant ("Grave", 96, PackageLatin1);

         AddPredefinedConstant ("Left_Curly_Bracket", 123, PackageLatin1);
         AddPredefinedConstant ("Vertical_Line", 124, PackageLatin1);
         AddPredefinedConstant ("Right_Curly_Bracket", 125, PackageLatin1);
         AddPredefinedConstant ("Tilde", 126, PackageLatin1);

      end AddLowerHalfControlAndGraphicCharacters;

      procedure AddUpperHalfAda95Characters
      --# global in     PackageLatin1;
      --#        in out Dict;
      --#        in out LexTokenManager.State;
      --# derives Dict                  from *,
      --#                                    LexTokenManager.State,
      --#                                    PackageLatin1 &
      --#         LexTokenManager.State from *;
      is
      begin
         AddPredefinedConstant ("Reserved_128", 128, PackageLatin1);
         AddPredefinedConstant ("Reserved_129", 129, PackageLatin1);
         AddPredefinedConstant ("BPH", 130, PackageLatin1);
         AddPredefinedConstant ("NBH", 131, PackageLatin1);
         AddPredefinedConstant ("Reserved_132", 132, PackageLatin1);
         AddPredefinedConstant ("NEL", 133, PackageLatin1);
         AddPredefinedConstant ("SSA", 134, PackageLatin1);
         AddPredefinedConstant ("ESA", 135, PackageLatin1);
         AddPredefinedConstant ("HTS", 136, PackageLatin1);
         AddPredefinedConstant ("HTJ", 137, PackageLatin1);
         AddPredefinedConstant ("VTS", 138, PackageLatin1);
         AddPredefinedConstant ("PLD", 139, PackageLatin1);
         AddPredefinedConstant ("PLU", 140, PackageLatin1);
         AddPredefinedConstant ("RI", 141, PackageLatin1);
         AddPredefinedConstant ("SS2", 142, PackageLatin1);
         AddPredefinedConstant ("SS3", 143, PackageLatin1);
         AddPredefinedConstant ("DCS", 144, PackageLatin1);
         AddPredefinedConstant ("PU1", 145, PackageLatin1);
         AddPredefinedConstant ("PU2", 146, PackageLatin1);
         AddPredefinedConstant ("STS", 147, PackageLatin1);
         AddPredefinedConstant ("CCH", 148, PackageLatin1);
         AddPredefinedConstant ("MW", 149, PackageLatin1);
         AddPredefinedConstant ("SPA", 150, PackageLatin1);
         AddPredefinedConstant ("EPA", 151, PackageLatin1);
         AddPredefinedConstant ("SOS", 152, PackageLatin1);
         AddPredefinedConstant ("Reserved_153", 153, PackageLatin1);
         AddPredefinedConstant ("SCI", 154, PackageLatin1);
         AddPredefinedConstant ("CSI", 155, PackageLatin1);
         AddPredefinedConstant ("ST", 156, PackageLatin1);
         AddPredefinedConstant ("OSC", 157, PackageLatin1);
         AddPredefinedConstant ("PM", 158, PackageLatin1);
         AddPredefinedConstant ("APC", 159, PackageLatin1);

         --# assert True;

         --other graphic characters
         --character positions 160 .. 175
         AddPredefinedConstant ("No_Break_Space", 160, PackageLatin1);
         AddPredefinedConstant ("NBSP", 160, PackageLatin1);
         AddPredefinedConstant ("Inverted_Exclamation", 161, PackageLatin1);
         AddPredefinedConstant ("Cent_Sign", 162, PackageLatin1);
         AddPredefinedConstant ("Pound_Sign", 163, PackageLatin1);
         AddPredefinedConstant ("Currency_Sign", 164, PackageLatin1);
         AddPredefinedConstant ("Yen_Sign", 165, PackageLatin1);
         AddPredefinedConstant ("Broken_Bar", 166, PackageLatin1);
         AddPredefinedConstant ("Section_Sign", 167, PackageLatin1);
         AddPredefinedConstant ("Diaeresis", 168, PackageLatin1);
         AddPredefinedConstant ("Copyright_Sign", 169, PackageLatin1);
         AddPredefinedConstant ("Feminine_Ordinal_Indicator", 170, PackageLatin1);
         AddPredefinedConstant ("Left_Angle_Quotation", 171, PackageLatin1);
         AddPredefinedConstant ("Not_Sign", 172, PackageLatin1);
         AddPredefinedConstant ("Soft_Hyphen", 173, PackageLatin1);
         AddPredefinedConstant ("Registered_Trade_Mark_Sign", 174, PackageLatin1);
         AddPredefinedConstant ("Macron", 175, PackageLatin1);

         --# assert True;

         --character positions 176 .. 191
         AddPredefinedConstant ("Degree_Sign", 176, PackageLatin1);
         AddPredefinedConstant ("Ring_Above", 177, PackageLatin1);
         AddPredefinedConstant ("Plus_Minus_Sign", 177, PackageLatin1);
         AddPredefinedConstant ("Superscript_Two", 178, PackageLatin1);
         AddPredefinedConstant ("Superscript_Three", 179, PackageLatin1);
         AddPredefinedConstant ("Acute", 180, PackageLatin1);
         AddPredefinedConstant ("Micro_Sign", 181, PackageLatin1);
         AddPredefinedConstant ("Pilcrow_Sign", 182, PackageLatin1);
         AddPredefinedConstant ("Paragraph_Sign", 182, PackageLatin1);
         AddPredefinedConstant ("Middle_Dot", 183, PackageLatin1);
         AddPredefinedConstant ("Cedilla", 184, PackageLatin1);
         AddPredefinedConstant ("Superscript_One", 185, PackageLatin1);
         AddPredefinedConstant ("Masculine_Ordinal_Indicator", 186, PackageLatin1);
         AddPredefinedConstant ("Right_Angle_Quotation", 187, PackageLatin1);
         AddPredefinedConstant ("Fraction_One_Quarter", 188, PackageLatin1);
         AddPredefinedConstant ("Fraction_One_Half", 189, PackageLatin1);
         AddPredefinedConstant ("Fraction_Three_Quarters", 190, PackageLatin1);
         AddPredefinedConstant ("Inverted_Question", 191, PackageLatin1);

         --# assert True;

         --character positions 192 .. 207
         AddPredefinedConstant ("UC_A_Grave", 192, PackageLatin1);
         AddPredefinedConstant ("UC_A_Acute", 193, PackageLatin1);
         AddPredefinedConstant ("UC_A_Circumflex", 194, PackageLatin1);
         AddPredefinedConstant ("UC_A_Tilde", 195, PackageLatin1);
         AddPredefinedConstant ("UC_A_Diaeresis", 196, PackageLatin1);
         AddPredefinedConstant ("UC_A_Ring", 197, PackageLatin1);
         AddPredefinedConstant ("UC_AE_Diphthong", 198, PackageLatin1);
         AddPredefinedConstant ("UC_C_Cedilla", 199, PackageLatin1);
         AddPredefinedConstant ("UC_E_Grave", 200, PackageLatin1);
         AddPredefinedConstant ("UC_E_Acute", 201, PackageLatin1);
         AddPredefinedConstant ("UC_E_Circumflex", 202, PackageLatin1);
         AddPredefinedConstant ("UC_E_Diaeresis", 203, PackageLatin1);
         AddPredefinedConstant ("UC_I_Grave", 204, PackageLatin1);
         AddPredefinedConstant ("UC_I_Acute", 205, PackageLatin1);
         AddPredefinedConstant ("UC_I_Circumflex", 206, PackageLatin1);
         AddPredefinedConstant ("UC_I_Diaeresis", 207, PackageLatin1);

         --# assert True;

         --character positions 208 .. 223
         AddPredefinedConstant ("UC_Icelandic_Eth", 208, PackageLatin1);
         AddPredefinedConstant ("UC_N_Tilde", 209, PackageLatin1);
         AddPredefinedConstant ("UC_O_Grave", 210, PackageLatin1);
         AddPredefinedConstant ("UC_O_Acute", 211, PackageLatin1);
         AddPredefinedConstant ("UC_O_Circumflex", 212, PackageLatin1);
         AddPredefinedConstant ("UC_O_Tilde", 213, PackageLatin1);
         AddPredefinedConstant ("UC_O_Diaeresis", 214, PackageLatin1);
         AddPredefinedConstant ("Multiplication_Sign", 215, PackageLatin1);
         AddPredefinedConstant ("UC_O_Oblique_Stroke", 216, PackageLatin1);
         AddPredefinedConstant ("UC_U_Grave", 217, PackageLatin1);
         AddPredefinedConstant ("UC_U_Acute", 218, PackageLatin1);
         AddPredefinedConstant ("UC_U_Circumflex", 219, PackageLatin1);
         AddPredefinedConstant ("UC_U_Diaeresis", 220, PackageLatin1);
         AddPredefinedConstant ("UC_Y_Acute", 221, PackageLatin1);
         AddPredefinedConstant ("UC_Icelandic_Thorn", 222, PackageLatin1);
         AddPredefinedConstant ("LC_German_Sharp_S", 223, PackageLatin1);

         --# assert True;

         --character positions 224 .. 239
         AddPredefinedConstant ("LC_A_Grave", 224, PackageLatin1);
         AddPredefinedConstant ("LC_A_Acute", 225, PackageLatin1);
         AddPredefinedConstant ("LC_A_Circumflex", 226, PackageLatin1);
         AddPredefinedConstant ("LC_A_Tilde", 227, PackageLatin1);
         AddPredefinedConstant ("LC_A_Diaeresis", 228, PackageLatin1);
         AddPredefinedConstant ("LC_A_Ring", 229, PackageLatin1);
         AddPredefinedConstant ("LC_AE_Diphthong", 230, PackageLatin1);
         AddPredefinedConstant ("LC_C_Cedilla", 231, PackageLatin1);
         AddPredefinedConstant ("LC_E_Grave", 232, PackageLatin1);
         AddPredefinedConstant ("LC_E_Acute", 233, PackageLatin1);
         AddPredefinedConstant ("LC_E_Circumflex", 234, PackageLatin1);
         AddPredefinedConstant ("LC_E_Diaeresis", 235, PackageLatin1);
         AddPredefinedConstant ("LC_I_Grave", 236, PackageLatin1);
         AddPredefinedConstant ("LC_I_Acute", 237, PackageLatin1);
         AddPredefinedConstant ("LC_I_Circumflex", 238, PackageLatin1);
         AddPredefinedConstant ("LC_I_Diaeresis", 239, PackageLatin1);

         --# assert True;

         --character positions 240 .. 255
         AddPredefinedConstant ("LC_Icelandic_Eth", 240, PackageLatin1);
         AddPredefinedConstant ("LC_N_Tilde", 241, PackageLatin1);
         AddPredefinedConstant ("LC_O_Grave", 242, PackageLatin1);
         AddPredefinedConstant ("LC_O_Acute", 243, PackageLatin1);
         AddPredefinedConstant ("LC_O_Circumflex", 244, PackageLatin1);
         AddPredefinedConstant ("LC_O_Tilde", 245, PackageLatin1);
         AddPredefinedConstant ("LC_O_Diaeresis", 246, PackageLatin1);
         AddPredefinedConstant ("Division_Sign", 247, PackageLatin1);
         AddPredefinedConstant ("LC_O_Oblique_Stroke", 248, PackageLatin1);
         AddPredefinedConstant ("LC_U_Grave", 249, PackageLatin1);
         AddPredefinedConstant ("LC_U_Acute", 250, PackageLatin1);
         AddPredefinedConstant ("LC_U_Circumflex", 251, PackageLatin1);
         AddPredefinedConstant ("LC_U_Diaeresis", 252, PackageLatin1);
         AddPredefinedConstant ("LC_Y_Acute", 253, PackageLatin1);
         AddPredefinedConstant ("LC_Icelandic_Thorn", 254, PackageLatin1);
         AddPredefinedConstant ("LC_Y_Diaeresis", 255, PackageLatin1);
      end AddUpperHalfAda95Characters;

      ----------------------------------

   begin --AddPackageAda

      Insert_Lex_String (Name  => "Ada",
                         Token => Token);
      AddNewPackage (Name        => Token,
                     Comp_Unit   => ContextManager.NullUnit,
                     Declaration => Null_Location,
                     Scope       => GlobalScope,
                     TheParent   => NullSymbol,
                     ChildSort   => Public,
                     ThePackage  => PackageAda);
      Dict.Packages.PackageAda := PackageAda;
      Insert_Lex_String (Name  => "Characters",
                         Token => Token);
      AddNewPackage (Name        => Token,
                     Comp_Unit   => ContextManager.NullUnit,
                     Declaration => Null_Location,
                     Scope       => GlobalScope,
                     TheParent   => PackageAda,
                     ChildSort   => Public,
                     ThePackage  => PackageCharacters);
      Dict.Packages.PackageAdaCharacters := PackageCharacters;
      Insert_Lex_String (Name  => "Latin_1",
                         Token => Token);
      AddNewPackage (Name        => Token,
                     Comp_Unit   => ContextManager.NullUnit,
                     Declaration => Null_Location,
                     Scope       => GlobalScope,
                     TheParent   => PackageCharacters,
                     ChildSort   => Public,
                     ThePackage  => PackageLatin1);
      Dict.Packages.PackageAdaCharactersLatin1 := PackageLatin1;
      --control characters

      AddCommonCharacters (PackageLatin1);

      AddLowerHalfControlAndGraphicCharacters;

      AddUpperHalfAda95Characters;

   end AddPackageAda;

   --------------------------------------------------------------------------------


   procedure AddPackageRealTime
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;
   is
      Token,
      SC_Token,
      TS_Token        : LexTokenManager.Lex_String;
      PackageRealTime,
      ClockTime,
      Time_Type,
      Time_Span_Type,
      Seconds_Count,
      T_Param,
      SC_Param,
      TS_Param,
      Subprog         : Symbol;
      PackageAda      : Symbol;
      RealTime_Scope  : Scopes;
      Arg             : Iterator;

      -------------------------------------

   begin --AddPackageRealTime
      PackageAda := Dict.Packages.PackageAda;
      --
      -- Create the package as a child of package Ada
      --
      AddNewPackage (Name        => LexTokenManager.Real_Time_Token,
                     Comp_Unit   => ContextManager.NullUnit,
                     Declaration => Null_Location,
                     Scope       => GlobalScope,
                     TheParent   => PackageAda,
                     ChildSort   => Public,
                     ThePackage  => PackageRealTime);
      Dict.Packages.PackageRealTime := PackageRealTime;
      RealTime_Scope := VisibleScope (PackageRealTime);

      --
      -- Add type definitions for private types Time and Time_Span
      --
      Insert_Lex_String (Name  => "Time",
                         Token => Token);
      AddPrivateType (Name           => Token,
                      Comp_Unit      => ContextManager.NullUnit,
                      Declaration    => Null_Location,
                      ThePackage     => PackageRealTime,
                      IsLimited      => False,
                      IsTaggedType   => False,
                      IsAbstractType => False,
                      Extends        => NullSymbol);
      -- Annoyingly, the symbol is not returned so we must look it up
      Time_Type :=  LookupImmediateScope (Name    => Token,
                                          Scope   => RealTime_Scope,
                                          Context => ProgramContext);
      Dict.Types.ThePredefinedTypes (PredefinedTimeType) := Time_Type;
      Insert_Lex_String (Name  => "Time_Span",
                         Token => Token);
      AddPrivateType (Name           => Token,
                      Comp_Unit      => ContextManager.NullUnit,
                      Declaration    => Null_Location,
                      ThePackage     => PackageRealTime,
                      IsLimited      => False,
                      IsTaggedType   => False,
                      IsAbstractType => False,
                      Extends        => NullSymbol);
      Time_Span_Type := LookupImmediateScope (Name    => Token,
                                              Scope   => RealTime_Scope,
                                              Context => ProgramContext);
      Dict.Types.ThePredefinedTypes (PredefinedTimeSpanType) := Time_Span_Type;

      --
      -- Create annotation "own protected in ClockTime : Time" for external time returned by Clock.
      --
      -- Note that the type-announcement is needed here so that the signature
      -- of the proof function associated with function Clock is well-formed
      -- in FDL.
      Insert_Lex_String (Name  => "ClockTime",
                         Token => Token);
      AddOwnVariable (Name              => Token,
                      ThePackage        => PackageRealTime,
                      Mode              => InMode,
                      IsProtected       => True,
                      IsInterruptStream => False,
                      Comp_Unit         => ContextManager.NullUnit,
                      Reference         => Null_Location,
                      Variable          => ClockTime);

      AddOwnVariableType (OwnVariable   => ClockTime,
                          TypeMark      => Time_Type,
                          TypeReference => Null_Location);

      --
      -- Add the seven deferred constants. Note that Time_Unit is not supported.
      --
      Insert_Lex_String (Name  => "Time_First",
                         Token => Token);
      AddDeferredConstant (Name          => Token,
                           TypeMark      => Time_Type,
                           TypeReference => Null_Location,
                           Comp_Unit     => ContextManager.NullUnit,
                           Declaration   => Null_Location,
                           ThePackage    => PackageRealTime);
      Insert_Lex_String (Name  => "Time_Last",
                         Token => Token);
      AddDeferredConstant (Name          => Token,
                           TypeMark      => Time_Type,
                           TypeReference => Null_Location,
                           Comp_Unit     => ContextManager.NullUnit,
                           Declaration   => Null_Location,
                           ThePackage    => PackageRealTime);
      Insert_Lex_String (Name  => "Time_Span_First",
                         Token => Token);
      AddDeferredConstant (Name          => Token,
                           TypeMark      => Time_Span_Type,
                           TypeReference => Null_Location,
                           Comp_Unit     => ContextManager.NullUnit,
                           Declaration   => Null_Location,
                           ThePackage    => PackageRealTime);
      Insert_Lex_String (Name  => "Time_Span_Last",
                         Token => Token);
      AddDeferredConstant (Name          => Token,
                           TypeMark      => Time_Span_Type,
                           TypeReference => Null_Location,
                           Comp_Unit     => ContextManager.NullUnit,
                           Declaration   => Null_Location,
                           ThePackage    => PackageRealTime);
      Insert_Lex_String (Name  => "Time_Span_Zero",
                         Token => Token);
      AddDeferredConstant (Name          => Token,
                           TypeMark      => Time_Span_Type,
                           TypeReference => Null_Location,
                           Comp_Unit     => ContextManager.NullUnit,
                           Declaration   => Null_Location,
                           ThePackage    => PackageRealTime);
      Insert_Lex_String (Name  => "Time_Span_Unit",
                         Token => Token);
      AddDeferredConstant (Name          => Token,
                           TypeMark      => Time_Span_Type,
                           TypeReference => Null_Location,
                           Comp_Unit     => ContextManager.NullUnit,
                           Declaration   => Null_Location,
                           ThePackage    => PackageRealTime);
      Insert_Lex_String (Name  => "Tick",
                         Token => Token);
      AddDeferredConstant (Name          => Token,
                           TypeMark      => Time_Span_Type,
                           TypeReference => Null_Location,
                           Comp_Unit     => ContextManager.NullUnit,
                           Declaration   => Null_Location,
                           ThePackage    => PackageRealTime);
      --
      -- Add function Clock return Time.
      --
      Insert_Lex_String (Name  => "Clock",
                         Token => Token);
      AddSubprogram (Name          => Token,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => RealTime_Scope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => Subprog);
      AddReturnType (TheFunction   => Subprog,
                     TypeMark      => Time_Type,
                     Comp_Unit     => ContextManager.NullUnit,
                     TypeReference => Null_Location);
      Dict.Subprograms.AdaRealTimeClock := Subprog;
      -- Clock requires annotation "global in ClockTime"
      AddGlobalVariable (Abstraction       => IsAbstract,
                         Subprogram        => Subprog,
                         Variable          => ClockTime,
                         Mode              => InMode,
                         PrefixNeeded      => False,
                         Comp_Unit         => ContextManager.NullUnit,
                         VariableReference => Null_Location);
      --
      -- The arithmetic and relational operators are not in the dictionary.
      -- Functions To_Duration and To_Time_Span are not supported.
      --
      -- Add functions Nanoseconds / Microseconds / Milliseconds. These take a
      -- parameter of type Integer and return Time_Span.
      --
      Insert_Lex_String (Name  => "Nanoseconds",
                         Token => Token);
      AddSubprogram (Name          => Token,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => RealTime_Scope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => Subprog);
      Insert_Lex_String (Name  => "NS",
                         Token => Token);
      AddSubprogramParameter (Name          => Token,
                              Subprogram    => Subprog,
                              TypeMark      => GetPredefinedIntegerType,
                              TypeReference => Null_Location,
                              Mode          => InMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);
      AddReturnType (TheFunction   => Subprog,
                     TypeMark      => Time_Span_Type,
                     Comp_Unit     => ContextManager.NullUnit,
                     TypeReference => Null_Location);

      Insert_Lex_String (Name  => "Microseconds",
                         Token => Token);
      AddSubprogram (Name          => Token,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => RealTime_Scope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => Subprog);
      Insert_Lex_String (Name  => "US",
                         Token => Token);
      AddSubprogramParameter (Name          => Token,
                              Subprogram    => Subprog,
                              TypeMark      => GetPredefinedIntegerType,
                              TypeReference => Null_Location,
                              Mode          => InMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);
      AddReturnType (TheFunction   => Subprog,
                     TypeMark      => Time_Span_Type,
                     Comp_Unit     => ContextManager.NullUnit,
                     TypeReference => Null_Location);

      Insert_Lex_String (Name  => "Milliseconds",
                         Token => Token);
      AddSubprogram (Name          => Token,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => RealTime_Scope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => Subprog);
      Insert_Lex_String (Name  => "MS",
                         Token => Token);
      AddSubprogramParameter (Name          => Token,
                              Subprogram    => Subprog,
                              TypeMark      => GetPredefinedIntegerType,
                              TypeReference => Null_Location,
                              Mode          => InMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);
      AddReturnType (TheFunction   => Subprog,
                     TypeMark      => Time_Span_Type,
                     Comp_Unit     => ContextManager.NullUnit,
                     TypeReference => Null_Location);
      --
      -- Add type Seconds_Count. This is an integer type with implementation-
      -- defined range. The range is left blank at this stage, and may be
      -- overridden by values supplied in the configuration file.
      --
      AddIntegerType (Name        => LexTokenManager.Seconds_Count_Token,
                      Comp_Unit   => ContextManager.NullUnit,
                      Declaration => Null_Location,
                      Lower       => LexTokenManager.Null_String,
                      Upper       => LexTokenManager.Null_String,
                      Scope       => RealTime_Scope,
                      Context     => ProgramContext,
                      Type_Symbol => Seconds_Count);
      --
      -- Add procedure Split (T:Time; SC:out Seconds_Count; TS:out Time_Span)
      --
      Insert_Lex_String (Name  => "Split",
                         Token => Token);
      AddSubprogram (Name          => Token,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => RealTime_Scope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => Subprog);
      Insert_Lex_String (Name  => "T",
                         Token => Token);
      AddSubprogramParameter (Name          => Token,
                              Subprogram    => Subprog,
                              TypeMark      => Time_Type,
                              TypeReference => Null_Location,
                              Mode          => InMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);
      Insert_Lex_String (Name  => "SC",
                         Token => SC_Token); -- Remember the token for further use of "SC"
      AddSubprogramParameter (Name          => SC_Token,
                              Subprogram    => Subprog,
                              TypeMark      => Seconds_Count,
                              TypeReference => Null_Location,
                              Mode          => OutMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);
      Insert_Lex_String (Name  => "TS",
                         Token => TS_Token); -- Remember the token for further use of "TS"
      AddSubprogramParameter (Name          => TS_Token,
                              Subprogram    => Subprog,
                              TypeMark      => Time_Span_Type,
                              TypeReference => Null_Location,
                              Mode          => OutMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);

      -- Split requires annotation "derives SC, TS from T", so we need the
      -- symbols for the three parameters.
      Arg := FirstSubprogramParameter (Subprog);
      T_Param := CurrentSymbol (Arg);
      Arg := NextSymbol (Arg);
      SC_Param := CurrentSymbol (Arg);
      Arg := NextSymbol (Arg);
      TS_Param := CurrentSymbol (Arg);

      AddExport (Abstraction     => IsAbstract,
                 TheProcedure    => Subprog,
                 TheExport       => SC_Param,
                 ExportReference => Null_Location,
                 Annotation      => Null_Location);
      AddExport (Abstraction     => IsAbstract,
                 TheProcedure    => Subprog,
                 TheExport       => TS_Param,
                 ExportReference => Null_Location,
                 Annotation      => Null_Location);
      AddDependency (Abstraction     => IsAbstract,
                     Comp_Unit       => ContextManager.NullUnit,
                     TheProcedure    => Subprog,
                     TheExport       => SC_Param,
                     TheImport       => T_Param,
                     ImportReference => Null_Location);
      AddDependency (Abstraction     => IsAbstract,
                     Comp_Unit       => ContextManager.NullUnit,
                     TheProcedure    => Subprog,
                     TheExport       => TS_Param,
                     TheImport       => T_Param,
                     ImportReference => Null_Location);
      --
      -- Add function Time_Of (SC: Seconds_Count; TS: Time_Span) return Time
      --
      Insert_Lex_String (Name  => "Time_Of",
                         Token => Token);
      AddSubprogram (Name          => Token,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => RealTime_Scope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => Subprog);
      AddSubprogramParameter (Name          => SC_Token,
                              Subprogram    => Subprog,
                              TypeMark      => Seconds_Count,
                              TypeReference => Null_Location,
                              Mode          => InMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);
      AddSubprogramParameter (Name          => TS_Token,
                              Subprogram    => Subprog,
                              TypeMark      => Time_Span_Type,
                              TypeReference => Null_Location,
                              Mode          => InMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);
      AddReturnType (TheFunction   => Subprog,
                     TypeMark      => Time_Type,
                     Comp_Unit     => ContextManager.NullUnit,
                     TypeReference => Null_Location);

   end AddPackageRealTime;


   --------------------------------------------------------------------------------

   procedure AddPackageSynchronousTaskControl
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;
   is
      Token,
      S_Token                      : LexTokenManager.Lex_String;
      PackageSynchronousTaskControl,
      SuspensionObject_Type,
      Subprog,
      S_Param                      : Symbol;
      PackageAda                   : Symbol;
      SynchronousTaskControl_Scope : Scopes;
      Arg                          : Iterator;

      -------------------------------------

   begin --AddPackageSynchronousTaskControl
      PackageAda := Dict.Packages.PackageAda;
      --
      -- Create the package as a child of package Ada
      --
      AddNewPackage (Name        => LexTokenManager.Synchronous_Task_Control_Token,
                     Comp_Unit   => ContextManager.NullUnit,
                     Declaration => Null_Location,
                     Scope       => GlobalScope,
                     TheParent   => PackageAda,
                     ChildSort   => Public,
                     ThePackage  => PackageSynchronousTaskControl);
      Dict.Packages.PackageSynchronousTaskControl := PackageSynchronousTaskControl;
      SynchronousTaskControl_Scope := VisibleScope (PackageSynchronousTaskControl);

      --
      -- Add type definitions for limited private type Suspension_Object
      --
      Insert_Lex_String (Name  => "Suspension_Object",
                         Token => Token);
      AddPrivateType (Name           => Token,
                      Comp_Unit      => ContextManager.NullUnit,
                      Declaration    => Null_Location,
                      ThePackage     => PackageSynchronousTaskControl,
                      IsLimited      => True,
                      IsTaggedType   => False,
                      IsAbstractType => False,
                      Extends        => NullSymbol);

      -- Annoyingly, the symbol is not returned so we must look it up
      SuspensionObject_Type := LookupImmediateScope (Name    => Token,
                                                     Scope   => SynchronousTaskControl_Scope,
                                                     Context => ProgramContext);

      -- Store the predefined type for easy lookup
      Dict.Types.ThePredefinedTypes (PredefinedSuspensionObjectType) := SuspensionObject_Type;

      -- Mark the type as Atomic
      SetTypeAtomic (SuspensionObject_Type);

      --
      -- Add procedure Set_True (S: in out Suspension_Object);
      --               --# derives S from ;
      --
      Insert_Lex_String (Name  => "Set_True",
                         Token => Token);
      AddSubprogram (Name          => Token,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => SynchronousTaskControl_Scope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => Subprog);
      Insert_Lex_String (Name  => "S",
                         Token => S_Token); -- used in all subroutine definitions for this package.
      AddSubprogramParameter (Name          => S_Token,
                              Subprogram    => Subprog,
                              TypeMark      => SuspensionObject_Type,
                              TypeReference => Null_Location,
                              Mode          => InOutMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);

      -- The annotation
      Arg := FirstSubprogramParameter (Subprog);
      S_Param := CurrentSymbol (Arg);

      AddExport (Abstraction     => IsAbstract,
                 TheProcedure    => Subprog,
                 TheExport       => S_Param,
                 ExportReference => Null_Location,
                 Annotation      => Null_Location);

      --
      -- Add procedure Set_False (S: in out Suspension_Object);
      --               --# derives S from ;
      --
      Insert_Lex_String (Name  => "Set_False",
                         Token => Token);
      AddSubprogram (Name          => Token,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => SynchronousTaskControl_Scope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => Subprog);
      AddSubprogramParameter (Name          => S_Token,
                              Subprogram    => Subprog,
                              TypeMark      => SuspensionObject_Type,
                              TypeReference => Null_Location,
                              Mode          => InOutMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);
      -- The annotation
      Arg := FirstSubprogramParameter (Subprog);
      S_Param := CurrentSymbol (Arg);

      AddExport (Abstraction     => IsAbstract,
                 TheProcedure    => Subprog,
                 TheExport       => S_Param,
                 ExportReference => Null_Location,
                 Annotation      => Null_Location);

      --
      -- Add procedure Suspend_Until_True (S: in out Suspension_Object);
      --               --# derives S from ;
      Insert_Lex_String (Name  => "Suspend_Until_True",
                         Token => Token);
      AddSubprogram (Name          => Token,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => SynchronousTaskControl_Scope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => Subprog);

      -- Record the symbol for later retrieval by wellformedness checker.
      Dict.Subprograms.STC_SuspendUntilTrue := Subprog;

      AddSubprogramParameter (Name          => S_Token,
                              Subprogram    => Subprog,
                              TypeMark      => SuspensionObject_Type,
                              TypeReference => Null_Location,
                              Mode          => InOutMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);
      -- The annotations
      Arg := FirstSubprogramParameter (Subprog);
      S_Param := CurrentSymbol (Arg);

      AddExport (Abstraction     => IsAbstract,
                 TheProcedure    => Subprog,
                 TheExport       => S_Param,
                 ExportReference => Null_Location,
                 Annotation      => Null_Location);

   end AddPackageSynchronousTaskControl;

   --------------------------------------------------------------------------------

   procedure AddPackageInterrupts
   --# global in     LexTokenManager.State;
   --#        in out Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys from *,
   --#                                Dict,
   --#                                LexTokenManager.State;
   is
      New_Symbol       : Symbol;
      PackageAda       : Symbol;
      Interrupts_Scope : Scopes;
   begin --AddPackageInterrupts
      PackageAda := Dict.Packages.PackageAda;
      --
      -- Create the package as a child of package Ada
      --
      AddNewPackage (Name        => LexTokenManager.Interrupts_Token,
                     Comp_Unit   => ContextManager.NullUnit,
                     Declaration => Null_Location,
                     Scope       => GlobalScope,
                     TheParent   => PackageAda,
                     ChildSort   => Public,
                     ThePackage  => New_Symbol);
      Dict.Packages.PackageInterrupts := New_Symbol;
      Interrupts_Scope := VisibleScope (New_Symbol);

      --
      -- Add type Interrupt_ID. This is an integer type with implementation-
      -- defined range. The range is left blank at this stage, and may be
      -- overridden by values supplied in the configuration file.
      --
      --# accept F, 10, New_Symbol, "Expected ineffective assignment to New_Symbol";
      AddIntegerType (Name        => LexTokenManager.Interrupt_ID_Token,
                      Comp_Unit   => ContextManager.NullUnit,
                      Declaration => Null_Location,
                      Lower       => LexTokenManager.Null_String,
                      Upper       => LexTokenManager.Null_String,
                      Scope       => Interrupts_Scope,
                      Context     => ProgramContext,
                      Type_Symbol => New_Symbol);
      --# end accept;
      pragma Warnings (Off, New_Symbol);
   end AddPackageInterrupts;

   --------------------------------------------------------------------------------

   -- This procedure creates a subprogram record with which we can associate globals
   -- and dependencies from the partition annotation
   procedure AddThePartition
   --# global in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;

   is
      Token : LexTokenManager.Lex_String;
      Sym   : Symbol;
   begin
      Insert_Lex_String (Name  => "main_program",
                         Token => Token); -- suitable reserved name for the subprog
      AddSubprogram (Name          => Token,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => GlobalScope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => Sym);
      Dict.ThePartition := Sym;
   end AddThePartition;

   --------------------------------------------------------------------------------

--     procedure AddGenericTypes
--     --# global in out Dict;
--     --#        in out LexTokenManager.State;
--     --# derives Dict,
--     --#         LexTokenManager.State from *,
--     --#                                          LexTokenManager.State;
--     is
--        Sym : Symbol;

--        procedure AddGenericType (Name    : in     String;
--                                  Discrim : in     TypeDiscriminant;
--                                  Sym     :    out Symbol)
--        --# global in out Dict;
--        --#        in out LexTokenManager.State;
--        --# derives Dict                        from *,
--        --#                                          LexTokenManager.State,
--        --#                                          Name,
--        --#                                          Discrim &
--        --#         LexTokenManager.State from *,
--        --#                                          Name &
--        --#         Sym                         from Dict;
--        is
--           Token   : LexTokenManager.Lex_String;
--           TheType : Symbol;
--        begin
--           InsertLexString (Name, Token);
--           RawDict.CreateType (Token, TheType);
--           RawDict.SetTypeDiscriminant (TheType, Discrim);
--           Sym := TheType;
--        end AddGenericType;

--     begin -- AddGenericTypes
--        AddGenericType ("GenericLimitedType", GenericLimitedType, Sym);
--        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericLimitedType) := Sym;
--        AddGenericType ("GenericPrivateType", GenericPrivateType, Sym);
--        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericPrivateType) := Sym;
--        AddGenericType ("GenericOrderedDiscreteType", GenericOrderedDiscreteType, Sym);
--        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericOrderedDiscreteType) := Sym;
--        AddGenericType ("GenericUnorderedDiscreteType", GenericUnorderedDiscreteType, Sym);
--        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericUnorderedDiscreteType) := Sym;
--        AddGenericType ("GenericIntegerType", GenericIntegerType, Sym);
--        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericIntegerType) := Sym;
--        AddGenericType ("GenericModularType", GenericModularType, Sym);
--        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericModularType) := Sym;
--        AddGenericType ("GenericFloatingPointType", GenericFloatingPointType, Sym);
--        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericFloatingPointType) := Sym;
--        AddGenericType ("GenericFixedPointType", GenericFixedPointType, Sym);
--        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericFixedPointType) := Sym;
--        AddGenericType ("GenericArrayType", GenericArrayType, Sym);
--        Dict.Types.ThePredefinedGenericTypes (PredefinedGenericArrayType) := Sym;
--     end AddGenericTypes;

   --------------------------------------------------------------------------------

   procedure AddUncheckedConversion
   --# global in     CommandLineData.Content;
   --#        in out Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    Dict,
   --#                                    LexTokenManager.State &
   --#         LexTokenManager.State from *;
   is
      SubprogramToken,
      SourceToken,
      TargetToken,
      SToken          : LexTokenManager.Lex_String;
      SubprogramSym,
      SourceTypeMark,
      TargetTypeMark,
      UnusedDeclaration : Symbol;
   begin -- AddUncheckedConversion
      -- add the function itself
      Insert_Lex_String (Name  => "Unchecked_Conversion",
                         Token => SubprogramToken);
      AddSubprogram (Name          => SubprogramToken,
                     Comp_Unit     => ContextManager.NullUnit,
                     Specification => Null_Location,
                     Scope         => GlobalScope,
                     Context       => ProgramContext,
                     -- to get
                     Subprogram    => SubprogramSym);
      -- mark predefined generic unit as having a body
      AddBody (CompilationUnit => SubprogramSym,
               Comp_Unit       => ContextManager.NullUnit,
               TheBody         => Null_Location,
               Hidden          => True);
      -- make it generic
      RawDict.SetSubprogramIsGeneric (SubprogramSym);
      Dict.Subprograms.UncheckedConversion := SubprogramSym;

      -- also add a declaration so that Ada.Unchecked_Conversion can be looked up
      case CommandLineData.Content.LanguageProfile is
         when CommandLineData.SPARK83 =>
            null;

         when CommandLineData.SPARK95 |
           CommandLineData.SPARK2005 =>

            --# accept Flow, 10, UnusedDeclaration, "Expected ineffective assignment to UnusedDeclaration";
            AddDeclaration (Item        => SubprogramSym,
                            Comp_Unit   => ContextManager.NullUnit,
                            Loc         => Null_Location,
                            Scope       => VisibleScope (Dict.Packages.PackageAda),
                            Context     => ProgramContext,
                            Declaration => UnusedDeclaration);
            --# end accept;
      end case;

      -- create the generic types
      Insert_Lex_String (Name  => "Source",
                         Token => SourceToken);
      AddTypeDeclaration (Name        => SourceToken,
                          Comp_Unit   => ContextManager.NullUnit,
                          Declaration => Null_Location,
                          Scope       => LocalScope (SubprogramSym),
                          Context     => ProgramContext,
                          -- to get
                          TypeMark    => SourceTypeMark);
      RawDict.SetTypeDiscriminant (SourceTypeMark,
                                   GenericLimitedType);
      -- PNA should we set Static false for generic type?

      Insert_Lex_String (Name  => "Target",
                         Token => TargetToken);
      AddTypeDeclaration (Name        => TargetToken,
                          Comp_Unit   => ContextManager.NullUnit,
                          Declaration => Null_Location,
                          Scope       => LocalScope (SubprogramSym),
                          Context     => ProgramContext,
                          -- to get
                          TypeMark    => TargetTypeMark);
      RawDict.SetTypeDiscriminant (TargetTypeMark,
                                   GenericLimitedType);
      -- PNA should we set Static false for generic type?

      -- create generic parameters
      AddGenericFormalParameter (Name                => LexTokenManager.Null_String,
                                 Comp_Unit           => ContextManager.NullUnit,
                                 Declaration         => Null_Location,
                                 SubprogramOrPackage => SubprogramSym,
                                 TypeMark            => SourceTypeMark,
                                 Kind                => GenericTypeParameter);

      AddGenericFormalParameter (Name                => LexTokenManager.Null_String,
                                 Comp_Unit           => ContextManager.NullUnit,
                                 Declaration         => Null_Location,
                                 SubprogramOrPackage => SubprogramSym,
                                 TypeMark            => TargetTypeMark,
                                 Kind                => GenericTypeParameter);

      -- add subprogram parameter
      Insert_Lex_String (Name  => "S",
                         Token => SToken);
      AddSubprogramParameter (Name          => SToken,
                              Subprogram    => SubprogramSym,
                              TypeMark      => SourceTypeMark,
                              TypeReference => Null_Location,
                              Mode          => InMode,
                              Comp_Unit     => ContextManager.NullUnit,
                              Specification => Null_Location);

      -- add return type
      AddReturnType (TheFunction   => SubprogramSym,
                     TypeMark      => TargetTypeMark,
                     Comp_Unit     => ContextManager.NullUnit,
                     TypeReference => Null_Location);

      --# accept Flow, 33, UnusedDeclaration, "Expected to be neither referenced nor exported";
   end AddUncheckedConversion;

   --------------------------------------------------------------------------------

begin -- Initialize
   AssignInitialValue;
   AddUnknownType;
   AddPackageStandard;
   AddPackageASCII;


   case CommandLineData.Content.LanguageProfile is
      when CommandLineData.SPARK83 =>

         Dict.Packages.PackageAda := NullSymbol;

      when CommandLineData.SPARK95 |
        CommandLineData.SPARK2005 =>

         AddPackageAda;
   end case;

   --AddGenericTypes;
   AddUncheckedConversion;
   if CommandLineData.RavenscarSelected then
      AddPackageRealTime;
      AddPackageSynchronousTaskControl;
      AddPackageInterrupts;
      AddThePartition;
   else
      Dict.Packages.PackageRealTime := NullSymbol;
      Dict.Packages.PackageSynchronousTaskControl := NullSymbol;
      Dict.Packages.PackageInterrupts := NullSymbol;
      Dict.ThePartition := NullSymbol;
   end if;
   if WriteToFile then
      SPARK_IO.Create (File, 0, "", "", Status);    -- Expected flow err
      if Status = SPARK_IO.Ok then
         Dict.TemporaryFile := File;
      else
         SystemErrors.FatalError (SystemErrors.InvalidInit, "in Dictionary.Initialize");
         Dict.TemporaryFile := SPARK_IO.Null_File;
      end if;
   else
      Dict.TemporaryFile := SPARK_IO.Null_File;
   end if;
   LexTokenManager.Set_Last_Token;
end Initialize;
