-- $Id: dictionary-getanyprefixneeded.adb 14620 2009-10-28 13:40:30Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with SystemErrors,
     LexTokenStacks;

separate (Dictionary)
procedure GetAnyPrefixNeeded (Sym       : in     Symbol;
                              Scope     : in     Scopes;
                              Separator : in     String;
                              Prefix    :    out EStrings.T)
is
   DeclaredScope,
   CurrentScope  : Scopes;
   SymLocal      : Symbol;

   function IsAbstractProofFunctionInLocalScope (Sym : Symbol) return Boolean
   --# global in Dict;
   is
   begin
      return IsImplicitProofFunction (Sym) and then
         (GetImplicitProofFunction (IsAbstract, GetAdaFunction (Sym)) = Sym) and then
         (GetImplicitProofFunction (IsRefined, GetAdaFunction (Sym)) /= NullSymbol);
   end IsAbstractProofFunctionInLocalScope;

begin --GetAnyPrefixNeeded
   if (Sym = GetUnknownTypeMark or
       Sym = NullSymbol or
       Sym = GetNullVariable) then
      Prefix := EStrings.Empty_String;

   elsif Sym = Dict.Subprograms.UncheckedConversion then
      -- special handling for Unchecked_Conversion which is a strange beast -
      -- child subprogram in 95 and 2005, library-level subprogram in 83
      case CommandLineData.Content.LanguageProfile is
         when CommandLineData.SPARK95 |
           CommandLineData.SPARK2005 =>
            Prefix := EStrings.Copy_String (Str => "Ada");
         when CommandLineData.SPARK83 =>
            Prefix := EStrings.Empty_String;
      end case;

   else
      SymLocal := Sym;
      -- if the symbol is of an access type we dereference it first (access will get put back
      -- in GenerateSimpleName
      if IsType (SymLocal) and then TypeIsAccess (SymLocal) then
         SymLocal := DeReference (SymLocal);
      end if;

      -- if the symbol is an implicit in stream associated with a protected own variable
      -- then the prefix is that applicable to the associated own variable
      if RawDict.GetSymbolDiscriminant (SymLocal) = ProtectedImplicitInStreamSymbol then
         SymLocal := GetOwnVariableOfProtectedImplicitInStream (SymLocal);
      end if;

      -- if the symbol is a special on loop entry variable we use the original variable
      -- to determine whether a prefix is needed
      if RawDict.GetSymbolDiscriminant (SymLocal) = LoopEntryVariableSymbol then
         SymLocal := RawDict.GetLoopEntryVariableOriginalVar (SymLocal);
      end if;

      -- call to getmost enclosing object added so that when looking for
      -- prefix of X.F we do so in terms of X (the X.F part of the full name
      -- will be added by GenerateSimpleName
      DeclaredScope := GetScope (GetMostEnclosingObject (SymLocal));
      if IsPredefinedScope (DeclaredScope) or else
        DeclaredScope = VisibleScope (GetPredefinedPackageStandard) or else
        IsQuantifiedVariable (SymLocal)
      then --no prefix needed
         Prefix := EStrings.Empty_String;
      else --prefix may be needed so do search
         CurrentScope := Scope;
         loop
            exit when CurrentScope = DeclaredScope;
            exit when IsGlobalScope (CurrentScope);
            exit when IsPredefinedScope (CurrentScope);

            if IsPackage (GetRegion (CurrentScope)) then
               CurrentScope := PrivateScope (GetRegion (CurrentScope));
            end if;

            exit when CurrentScope = DeclaredScope;

            if IsPackage (GetRegion (CurrentScope)) then
               CurrentScope := VisibleScope (GetRegion (CurrentScope));
               exit;
            end if;
            CurrentScope := GetEnclosingScope (CurrentScope);
         end loop;

         -- The new error handling scheme intrduced by SEPR 1883 defers generation of error strings until after
         -- an entire source file has been processed.  This causes ones spceial-case problem: a package spec and
         -- its body are in a single file, the package declares an own variable, there is an error in the
         -- package spec involvignthe own variable, the own variable is declared in the package body.  This
         -- combination mens that we  might have CurrentScope equal to VISIBLE part of P and DeclaredScope
         -- being LOCAL scope of P.  This then triggers unwanted generation of a prefix.
         -- In all other situations, this "inversion" can't happen.  We don't want a prefix if both declaration
         -- and viewpoint are in the same REGION.  The test that the SCOPES be identical is too strict.  So we
         -- replace "if CurrentScope = DeclaredScope" with:
         if GetRegion (CurrentScope) = GetRegion (DeclaredScope) and then
           not IsAbstractProofFunctionInLocalScope (SymLocal)
         then
            Prefix := EStrings.Empty_String;
         else
            GenerateSimpleName (GetRegion (GetScope (GetMostEnclosingObject (SymLocal))),
                                Separator,
                                Prefix);
         end if;
      end if;
   end if;
end GetAnyPrefixNeeded;
