-- $Id: dictionary-generatesimplename.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with LexTokenStacks;

separate (Dictionary)
procedure GenerateSimpleName (Item      : in     Symbol;
                              Separator : in     String;
                              Name      :    out EStrings.T)
is

   --------------------------------------------------------------------------------

   function FetchSimpleName (Item : Symbol) return EStrings.T
      --# global in Dict;
      --#        in LexTokenManager.State;
   is
      ItemLocal : Symbol;
   begin
      ItemLocal := Item;
      if IsType (ItemLocal) and then TypeIsAccess (ItemLocal) then
         ItemLocal := DeReference (ItemLocal);
      end if;
      return LexTokenManager.Lex_String_To_String (Lex_Str => GetSimpleName (ItemLocal));
   end FetchSimpleName;

   --------------------------------------------------------------------------------

   -- Each protected own variable has an associated implicit in stream which is used for
   -- volatile flow analysis of shared protected state.  The names of these should never
   -- apepar in Examienr output; however, if they are needed for diagnostic reasons they
   -- can be constructed by this function.  For a stream associated with P we return P__in.
   function GetImplicitProtectedInStreamName (Item : Symbol) return EStrings.T
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      Name : EStrings.T;
   begin
      Name := FetchSimpleName (GetOwnVariableOfProtectedImplicitInStream (Item));
      EStrings.Append_String (E_Str => Name,
                              Str   => "__in");
      return Name;
   end GetImplicitProtectedInStreamName;

   --------------------------------------------------------------------------------

   function GetLoopName (TheLoop : Symbol) return EStrings.T
      --# global in Dict;
      --#        in LexTokenManager.State;
   is
      --# hide GetLoopName;

      Name : EStrings.T;

      --------------------------------------------------------------------------------

      function GetLoopNumber (TheLoop : Symbol) return Positive
         --# global in Dict;
      is
         Loops  : Iterator;
         Number : Positive;
      begin

         Loops := FirstLoop (GetEnclosingCompilationUnit (LocalScope (TheLoop)));
         Number := 1;

         loop
            exit when CurrentSymbol (Loops) = TheLoop;
            Loops := NextSymbol (Loops);
            Number := Number + 1;
         end loop;

         return Number;

      end GetLoopNumber;

      --------------------------------------------------------------------------------

      function Image (Number : Positive) return String
      is
         SignedImage : constant String := Positive'Image (Number);
      begin
         return SignedImage (2 .. SignedImage'Length);
      end Image;

      --------------------------------------------------------------------------------

   begin

      if LoopHasName (TheLoop) then
         Name := FetchSimpleName (TheLoop);
      else
         Name := EStrings.Copy_String (Str => "LOOP__");
         EStrings.Append_String (E_Str => Name,
                                 Str   => Image (GetLoopNumber (TheLoop)));
      end if;

      return Name;

   end GetLoopName; -- Hidden body expected

   --------------------------------------------------------------------------------

   procedure GetPackageName (ThePackage : in     Symbol;
                             Separator  : in     String;
                             Name       :    out EStrings.T)
      --# global in Dict;
      --#        in LexTokenManager.State;
      --# derives Name from Dict,
      --#                   LexTokenManager.State,
      --#                   Separator,
      --#                   ThePackage;
   is
      PackageLocal : Symbol;
      CurrentToken : LexTokenManager.Lex_String;
      Stack        : LexTokenStacks.Stacks;
      LocalName    : EStrings.T;
   begin --GetPackageName
      LocalName := EStrings.Empty_String;
      PackageLocal := ThePackage;

      LexTokenStacks.Clear (Stack);
      loop
         LexTokenStacks.Push (Stack, GetSimpleName (PackageLocal));
         PackageLocal := RawDict.GetPackageParent (PackageLocal);
         exit when PackageLocal = NullSymbol;
      end loop;

      loop
         LexTokenStacks.Pop (Stack, CurrentToken);
         EStrings.Append_Examiner_String
           (E_Str1 => LocalName,
            E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentToken));
         exit when LexTokenStacks.IsEmpty (Stack);
         EStrings.Append_String (E_Str => LocalName,
                                 Str   => Separator);
      end loop;
      Name := LocalName;
   end GetPackageName;

   --------------------------------------------------------------------------------

   procedure GetRecordVariableName (TheRecord  : in     Symbol;
                                    Separator  : in     String;
                                    Name       :    out EStrings.T)
      --# global in Dict;
      --#        in LexTokenManager.State;
      --# derives Name from Dict,
      --#                   LexTokenManager.State,
      --#                   Separator,
      --#                   TheRecord;
   is
      RecordLocal  : Symbol;
      CurrentToken : LexTokenManager.Lex_String;
      Stack        : LexTokenStacks.Stacks;
      LocalName    : EStrings.T;
   begin --GetRecordVariableName
      LocalName := EStrings.Empty_String;
      RecordLocal := TheRecord;

      LexTokenStacks.Clear (Stack);
      loop
         -- we want to ignore any inherited fields for name generation purposes
         if not (IsRecordSubcomponent (RecordLocal) and then
                 RecordComponentIsInherited (RawDict.GetSubcomponentComponent (RecordLocal))) then
            LexTokenStacks.Push (Stack, GetSimpleName (RecordLocal));
         end if;
         exit when not IsRecordSubcomponent (RecordLocal); --entire record var

         RecordLocal := GetEnclosingObject (RecordLocal);
      end loop;

      loop
         LexTokenStacks.Pop (Stack, CurrentToken);
         EStrings.Append_Examiner_String
           (E_Str1 => LocalName,
            E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => CurrentToken));
         exit when LexTokenStacks.IsEmpty (Stack);
         EStrings.Append_String (E_Str => LocalName,
                                 Str   => Separator);
      end loop;
      Name := LocalName;
   end GetRecordVariableName;

   --------------------------------------------------------------------------------

   function GetLoopEntryVariableName (TheLoopEntryVar : Symbol) return EStrings.T
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      Result   : EStrings.T;
      Loopname : EStrings.T;
   begin
      -- Loop on entry variable names are constructed from the original variable name
      -- and the associated loop name
      Result := FetchSimpleName (TheLoopEntryVar);
      Loopname := GetLoopName (RawDict.GetLoopEntryVariableTheLoop (TheLoopEntryVar));
      EStrings.Append_String (E_Str => Result,
                              Str   => "__entry__");
      EStrings.Append_Examiner_String (E_Str1 => Result,
                                       E_Str2 => Loopname);
      return Result;
   end GetLoopEntryVariableName;

   --------------------------------------------------------------------------------

   procedure GetParameterConstraintName (Item : in     Symbol;
                                         Name :    out EStrings.T)
   --# global in Dict;
   --#        in LexTokenManager.State;
   --# derives Name from Dict,
   --#                   Item,
   --#                   LexTokenManager.State;
   is
      -- because of temp use of 'image
      LocalName     : EStrings.T;

   begin
      LocalName := FetchSimpleName (Item);
      -- above line will return the name of the formal parameter associated with the constraint
      EStrings.Append_String (E_Str => LocalName,
                              Str   => "__index__subtype__");
      EStrings.Append_Examiner_String
        (E_Str1 => LocalName,
         E_Str2 => ELStrings.ToExaminerString
           (Maths.ValueToString
              (Maths.IntegerToValue
                 (GetSubprogramParameterConstraintDimension (Item)))));

      Name := LocalName;
   end GetParameterConstraintName;

   --------------------------------------------------------------------------------

begin --GenerateSimpleName

   if IsLoop (Item) then
      Name := GetLoopName (Item);
   elsif IsPackage (Item) then
      GetPackageName (Item,
                      Separator,
                        -- to get
                      Name);
   elsif IsRecordSubcomponent (Item) then
      GetRecordVariableName (Item,
                             Separator,
                              -- to get
                             Name);
   elsif RawDict.GetSymbolDiscriminant (Item) = ProtectedImplicitInStreamSymbol then
      Name := GetImplicitProtectedInStreamName (Item);

   elsif RawDict.GetSymbolDiscriminant (Item) = LoopEntryVariableSymbol then
      Name := GetLoopEntryVariableName (Item);

   elsif RawDict.GetSymbolDiscriminant (Item) = ParameterConstraintSymbol then
      GetParameterConstraintName (Item,
                                    -- to get
                                  Name);

   else
      Name := FetchSimpleName (Item);
   end if;

   -- Temp debug stuff.  Uncomment to append raw symbol value to name.  e.g.   State(123)
   --EStrings.AppendString (Name, "(");
   --EStrings.AppendString (Name, ExaminerConstants.RefType'Image (ExaminerConstants.RefType (Item)));
   --EStrings.AppendString (Name, ")");

end GenerateSimpleName;
