-- $Id: dictionary-binaryoperatorisdefined.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Dictionary)
function BinaryOperatorIsDefined (Name  : SPSymbols.SPSymbol;
                                  Left  : Symbol;
                                  Right : Symbol) return Boolean
is

   Result : Boolean;

   --------------------------------------------------------------------------------
   -- Logical operators (and, or, xor) are _defined_ (but not necessarily
   -- visible) according to the following table.  Abbreviations for type classes
   -- of the left and right operands are as follows:
   --   B  = Boolean
   --   BA = Boolean Array
   --   M  = Modular
   --   UI = Universal Integer (e.g. an integer literal)
   --   U  = Unknown
   --   O  = Other
   -- Result codes are
   --   Y  = Always defined
   --   N  = Never defined
   --   M  = (="Match") Defined if Left = Right
   -----
   --        Right  B   BA   M   UI   U   O
   -- Left
   --  B            M    N   N    N   Y   N
   --  BA           N    M   N    N   Y   N
   --  M            N    N   M    Y   Y   N
   --  UI           N    N   Y    N   Y   N
   --  U            Y    Y   Y    Y   M   N
   --  O            N    N   N    N   N   N
   --------------------------------------------------------------------------------
   function LogicalOperatorsAreDefined (Left, Right : Symbol) return Boolean
   --# global in Dict;
   is
      Result : Boolean;
   begin

      if TypeIsBoolean (Left) or else
         TypeIsBooleanArray (Left) then

         Result :=
            (Left = Right) or else
            TypeIsUnknown (Right);

      elsif TypeIsModular (Left) then

         Result :=
            (Left = Right) or else
            IsUniversalIntegerType (Right) or else
            TypeIsUnknown (Right);

      elsif IsUniversalIntegerType (Left) then

         Result :=
            TypeIsModular (Right) or else
            TypeIsUnknown (Right);

      elsif TypeIsUnknown (Left) then

         Result :=
            (Left = Right) or else
            TypeIsBoolean (Right) or else
            TypeIsBooleanArray (Right) or else
            TypeIsModular (Right) or else
            IsUniversalIntegerType (Right);

      else
         Result := False;
      end if;

      return Result;

   end LogicalOperatorsAreDefined;

   --------------------------------------------------------------------------------
   -- Covers = and /=
   --
   -- The EqualityDefined attribute is set when the type is added to the
   -- dictionary.
   --------------------------------------------------------------------------------
   function EqualityOperatorsAreDefined (Left, Right : Symbol) return Boolean
   --# global in Dict;
   is
      Result : Boolean;
   begin

      if Left = Right or else TypeIsUnknown (Right) then
         Result := EqualityDefined (Left);
      elsif TypeIsUnknown (Left) then
         Result := EqualityDefined (Right);
      else
         Result := False;
      end if;

      return Result;

   end EqualityOperatorsAreDefined;

   --------------------------------------------------------------------------------
   -- Relational ordering operators (<= >= < >) are _defined_ (but not necessarily
   -- visible) according to the following table.  Abbreviations for type classes
   -- of the left and right operands are as follows:
   --   S  = Scalar - Integer, Modular, Enumeration, Real (but not Boolean)
   --   B  = Boolean
   --   PS = Predefined String
   --   CS = Constrained String Subtype
   --   U  = Unknown
   --   O  = Other
   -- Result codes are
   --   Y  = Always defined
   --   N  = Never defined
   --   M  = (="Match") Defined if Left = Right
   -----
   --        Right  S   B  PS  CS   U   O
   -- Left
   --  S            M   N   N   N   Y   N
   --  B            N   N   N   N   N   N
   --  PS           N   N   M   N   Y   N
   --  CS           N   N   N   M   Y   N
   --  U            Y   N   Y   Y   Y   N
   --  O            N   N   N   N   N   N
   --------------------------------------------------------------------------------
   function RelationalOperatorsAreDefined (Left, Right : Symbol) return Boolean
      --# global in CommandLineData.Content;
      --#        in Dict;
   is
      Result : Boolean;

      function TypeIsOrdered (S : Symbol) return Boolean
      --# global in Dict;
      is
      begin
         return not (TypeIsBoolean (S) or else TypeIsGenericUnOrderedDiscrete (S));
      end TypeIsOrdered;

   begin

      if Left = Right or else TypeIsUnknown (Right) then
         if TypeIsScalar (Left) then
            Result := TypeIsOrdered (Left); --not TypeIsBoolean (Left);
         elsif TypeIsArray (Left) then
            if IsType (Left) then
               Result := IsPredefinedStringType (Left);
            else
               Result := IsPredefinedStringType (GetRootType (Left)) and then
                  ArrayTypeIsConstrained (Left);
            end if;
         elsif CommandLineData.RavenscarSelected
           and then IsPredefinedTimeType (Left) then
            Result := True;
         else
            Result := TypeIsUnknown (Left);
         end if;
      elsif TypeIsUnknown (Left) then
         if TypeIsScalar (Right) then
            Result := TypeIsOrdered (Right); -- not TypeIsBoolean (Right);
         elsif TypeIsArray (Right) then
            if IsType (Right) then
               Result := IsPredefinedStringType (Right);
            else
               Result := IsPredefinedStringType (GetRootType (Right)) and then
                  ArrayTypeIsConstrained (Right);
            end if;
         elsif CommandLineData.RavenscarSelected
           and then IsPredefinedTimeType (Right) then
            Result := True;
         else
            Result := False;
         end if;
      else
         Result := False;
      end if;

      return Result;

   end RelationalOperatorsAreDefined;

   --------------------------------------------------------------------------------

   --------------------------------------------------------------------------------
   -- Binary Adding operators + and - (but not &) are _defined_
   -- (but not necessarily visible) according
   -- to the following table.  Abbreviations for type classes of the left and
   -- right operands are as follows:
   --   I  = Signed Integer
   --   M  = Modular
   --   R  = Real
   --   UI = Universal Integer (e.g. an integer literal)
   --   UR = Universal Real (e.g. a real literal)
   --   U  = Unknown
   --   O  = Other
   -- Result codes are
   --   Y  = Always defined
   --   N  = Never defined
   --   M  = (="Match") Defined if Left = Right
   -----
   --        Right  I   M   R  UI  UR   U   O
   -- Left
   --  I            M   N   N   Y   N   Y   N
   --  M            N   M   N   Y   N   Y   N
   --  R            N   N   M   N   Y   Y   N
   --  UI           Y   Y   N   M   N   Y   N
   --  UR           N   N   Y   N   M   Y   N
   --  U            Y   Y   Y   Y   Y   M   N
   --  O            N   N   N   N   N   N   N
   --
   -- For the types Time and Time_Span in package Ada.Real_Time, the adding
   -- operators are defined as follows, with T = Time and TS = Time_Span:
   --
   -- Op "+" Right  T   TS  U   O         Op "-" Right  T   TS  U   O
   -- Left                                Left
   --  T            N   Y   Y   N          T            Y   Y   Y   N
   --  TS           Y   Y   Y   N          TS           N   Y   Y   N
   --  U            Y   Y   M   N          U            Y   Y   M   N
   --  O            N   N   N   N          O            N   N   N   N
   --------------------------------------------------------------------------------
   function AddingOperatorsAreDefined (Left, Right : Symbol) return Boolean
   --# global in CommandLineData.Content;
   --#        in Dict;
   --#        in Name;
   is
      Result : Boolean;
   begin
      if IsUniversalIntegerType (Left) then

         Result :=
            TypeIsInteger (Right) or else
            TypeIsModular (Right) or else
            (Left = Right) or else
            TypeIsUnknown (Right);

      elsif IsUniversalRealType (Left) then

         Result :=
            TypeIsReal (Right) or else
            (Left = Right) or else
            TypeIsUnknown (Right);

      elsif TypeIsInteger (Left) or else
            TypeIsModular (Left) then

         Result :=
            (Left = Right) or else
            IsUniversalIntegerType (Right) or else
            TypeIsUnknown (Right);

      elsif TypeIsReal (Left) then

         Result :=
            (Left = Right) or else
            IsUniversalRealType (Right) or else
            TypeIsUnknown (Right);

      elsif CommandLineData.RavenscarSelected and then IsPredefinedTimeType (Left) then

         if IsPredefinedTimeType (Right) then
            -- Do the cases that differ between "+" and "-" first
            if Right = GetPredefinedTimeType then
               -- Time op Time is not defined for "+" but is defined for "-".
               -- Time_Span op Time is defined for "+" but is not defined for "-".
               Result := (Left = Right and then Name = SPSymbols.minus) or else
                         (Left = GetPredefinedTimeSpanType and then Name = SPSymbols.plus);
            else
               Result := True; -- All other time type combinations are defined
            end if;
         else
            Result := TypeIsUnknown (Right);
         end if;

      elsif TypeIsUnknown (Left) then

         Result :=
            (Left = Right) or else
            TypeIsNumeric (Right) or else
            IsUniversalIntegerType (Right) or else
            (CommandLineData.RavenscarSelected and then IsPredefinedTimeType (Right));

      else

         Result := False;
      end if;

      return Result;
   end AddingOperatorsAreDefined;

   --------------------------------------------------------------------------------
   -- Modular types are Numeric types, so no change here to accomodate modular
   -- types.
   -- For the Time_Span type, multiplication is defined between TS and Integer.
   --------------------------------------------------------------------------------
   function MultiplicationOperatorIsDefined (Left, Right : Symbol) return Boolean
   --# global in CommandLineData.Content;
   --#        in Dict;
   is
      Result : Boolean;
   begin

      if Left = Right then
         Result := TypeIsNumeric (Left) or else TypeIsUnknown (Left);
      elsif TypeIsFixedPoint (Left) then
         Result := IsPredefinedIntegerType (Right) or else
            TypeIsFixedPoint (Right) or else
            TypeIsUnknown (Right);
      elsif TypeIsFixedPoint (Right) then
         Result := IsPredefinedIntegerType (Left) or else TypeIsUnknown (Left);
      elsif CommandLineData.RavenscarSelected and then Left = GetPredefinedTimeSpanType then
         Result := IsPredefinedIntegerType (Right) or else TypeIsUnknown (Right);
      elsif CommandLineData.RavenscarSelected and then Right = GetPredefinedTimeSpanType then
         Result := IsPredefinedIntegerType (Left) or else TypeIsUnknown (Left);
      elsif IsUniversalRealType (Left) then
         Result := IsUniversalIntegerType (Right) or else TypeIsUnknown (Right);
      elsif IsUniversalIntegerType (Left) then
         Result := IsUniversalRealType (Right) or else TypeIsUnknown (Right);
      elsif TypeIsNumeric (Left) then
         Result := TypeIsUnknown (Right);
      elsif TypeIsUnknown (Left) then
         Result := TypeIsNumeric (Right);
      else
         Result := False;
      end if;

      return Result;

   end MultiplicationOperatorIsDefined;

   --------------------------------------------------------------------------------
   -- / for signed integer, real, and modular types are defined for
   --    1) Any matching pair of numeric types
   --    2) Any pair of fixed point types
   --    3) Any fixed point type on the left with Standard.Integer on the right
   --    4) Universal real on the left, and Universal integer on the right
   --    5) Any type with an unknown type (to prevent needless propagation of errors)
   -- For type Time_Span, "/" is defined for TS/TS and TS/Integer.
   --------------------------------------------------------------------------------
   function DivisionOperatorIsDefined (Left, Right : Symbol) return Boolean
   --# global in CommandLineData.Content;
   --#        in Dict;
   is
      Result : Boolean;
   begin

      if Left = Right then
         Result := TypeIsNumeric (Left) or else
         (CommandLineData.RavenscarSelected and then Left = GetPredefinedTimeSpanType) or else
            TypeIsUnknown (Left);
      elsif TypeIsFixedPoint (Left) then
         Result := IsPredefinedIntegerType (Right) or else
            TypeIsFixedPoint (Right);
      elsif CommandLineData.RavenscarSelected and then Left = GetPredefinedTimeSpanType then
         Result := IsPredefinedIntegerType (Right) or else TypeIsUnknown (Right);
      elsif IsUniversalRealType (Left) then
         Result := IsUniversalIntegerType (Right);
      elsif TypeIsNumeric (Left) then
         Result := TypeIsUnknown (Right);
      elsif TypeIsUnknown (Left) then
         Result := TypeIsNumeric (Right) or else
           (CommandLineData.RavenscarSelected and then Right = GetPredefinedTimeSpanType);
      else
         Result := False;
      end if;

      return Result;

   end DivisionOperatorIsDefined;

   --------------------------------------------------------------------------------
   -- mod and rem for signed integer and modular types is defined for
   --  1) Matching pairs of integer or modular types
   --  2) Any integer or modular type and an unknown type
   --------------------------------------------------------------------------------
   function IntegerDivisionOperatorsAreDefined (Left, Right : Symbol) return Boolean
   --# global in Dict;
   is
      Result : Boolean;
   begin
      if Left = Right or else TypeIsUnknown (Right) then
         Result := TypeIsInteger (Left) or else
                   TypeIsModular (Left);
      elsif TypeIsUnknown (Left) then
         Result := TypeIsInteger (Right) or else
                   TypeIsModular (Right);
      else
         Result := False;
      end if;

      return Result;
   end IntegerDivisionOperatorsAreDefined;

   --------------------------------------------------------------------------------
   -- ** is defined (but not necessairily visible for)
   --   1) Any Integer type ** Standard.Integer
   --   2) Any Modular type ** Standard.Integer
   --   3) Any Floating point type ** Standard.Integer
   --   4) Any Unknown type ** Standard.Integer
   --   5) as 1) thru 4) with Unknown on the RHS
   --------------------------------------------------------------------------------
   function ExponentiationOperatorIsDefined (Left, Right : Symbol) return Boolean
   --# global in Dict;
   is
      Result : Boolean;
   begin

      if IsPredefinedIntegerType (Right) or else TypeIsUnknown (Right) then
         Result := TypeIsInteger (Left) or else
                   TypeIsFloatingPoint (Left) or else
                   TypeIsModular (Left) or else
                   TypeIsUnknown (Left);
      else
         Result := False;
      end if;

      return Result;

   end ExponentiationOperatorIsDefined;

   --------------------------------------------------------------------------------

begin
   SystemErrors.RTAssert ((IsType (Left) or else IsUnknownTypeMark (Left)) and then
                            (IsType (Right) or else IsUnknownTypeMark (Right)),
                          SystemErrors.PreconditionFailure,
                          "In call to BinaryOperatorIsDefined");

   case Name is
      when SPSymbols.RWand | SPSymbols.RWor | SPSymbols.RWxor =>
         Result := LogicalOperatorsAreDefined (Left, Right);
      when SPSymbols.equals | SPSymbols.not_equal =>
         Result := EqualityOperatorsAreDefined (Left, Right);
      when SPSymbols.less_than    | SPSymbols.less_or_equal |
         SPSymbols.greater_than | SPSymbols.greater_or_equal =>
         Result := RelationalOperatorsAreDefined (Left, Right);
      when SPSymbols.plus | SPSymbols.minus =>
         Result := AddingOperatorsAreDefined (Left, Right);
      when SPSymbols.multiply =>
         Result := MultiplicationOperatorIsDefined (Left, Right);
      when SPSymbols.divide =>
         Result := DivisionOperatorIsDefined (Left, Right);
      when SPSymbols.RWmod | SPSymbols.RWrem =>
         Result := IntegerDivisionOperatorsAreDefined (Left, Right);
      when SPSymbols.double_star =>
         Result := ExponentiationOperatorIsDefined (Left, Right);
      when others =>
         Result := False;
   end case;

   return Result;

end BinaryOperatorIsDefined;
