-- $Id: dag_io.adb 16015 2010-02-09 22:00:43Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with CStacks,
     EStrings,
     ELStrings,
     LexTokenManager,
     Clists,
     SPSymbols,
     SystemErrors,
     AdjustFDL_RWs,
     CommandLineData,
     Maths;

use type SPSymbols.SPSymbol;
use type LexTokenManager.Str_Comp_Result;
use type Maths.ErrorCode;

package body DAG_IO
is

   type ExpnType is (Conclusion, Condition, Hypothesis);

   ----------------------------------------------------------------------------

   procedure Put_String (File        : in SPARK_IO.File_Type;
                         Item        : in String)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Item;
   is
   begin
      SPARK_IO.Put_String (File, Item, 0);
   end Put_String;

   ----------------------------------------------------------------------------

   procedure PutExaminerString (File : in SPARK_IO.File_Type;
                                Item : in EStrings.T)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Item;
   is
   begin
      EStrings.Put_String (File  => File,
                           E_Str => EStrings.Lower_Case (E_Str => Item));
   end PutExaminerString;

   ----------------------------------------------------------------------------

   procedure Put_Char (File        : in SPARK_IO.File_Type;
                       Item        : in Character)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Item;
   is
   begin
      SPARK_IO.Put_Char (File, Item);
   end Put_Char;

   ----------------------------------------------------------------------------

   procedure PrintCellContents (Heap         : in     Cells.Heap_Record;
                                OutputFile   : in     SPARK_IO.File_Type;
                                CellName     : in     Cells.Cell;
                                SuppressWrap : in out Boolean;
                                Scope        : in     Dictionary.Scopes;
                                Wrap_Limit   : in     Positive;
                                Escape_DOT   : in     Boolean)
   is
      ThisCellSymValue    : Dictionary.Symbol;
      ThisCellKind        : Cells.Cell_Kind;
      ThisCellStringValue : LexTokenManager.Lex_String;
      ThisCellOp          : SPSymbols.SPSymbol;
      ExString            : EStrings.T;

      -------------------------------------------------------

      procedure CheckWrap (Width : in Integer)
      --# global in     OutputFile;
      --#        in     Wrap_Limit;
      --#        in out SPARK_IO.File_Sys;
      --#        in out SuppressWrap;
      --# derives SPARK_IO.File_Sys from *,
      --#                                OutputFile,
      --#                                SuppressWrap,
      --#                                Width,
      --#                                Wrap_Limit &
      --#         SuppressWrap      from *;
      is
      begin
         if SuppressWrap then
            SuppressWrap := False;
         else
            if SPARK_IO.Col (OutputFile) + Width > Wrap_Limit then
               SPARK_IO.New_Line (OutputFile, 1);
               SPARK_IO.Put_String (OutputFile, "           ", 0);
            end if;
         end if;
      end CheckWrap;

      -------------------------------------------------------

      procedure VCG_PutString (Str  : in EStrings.T)
      --# global in     OutputFile;
      --#        in     Wrap_Limit;
      --#        in out SPARK_IO.File_Sys;
      --#        in out SuppressWrap;
      --# derives SPARK_IO.File_Sys from *,
      --#                                OutputFile,
      --#                                Str,
      --#                                SuppressWrap,
      --#                                Wrap_Limit &
      --#         SuppressWrap      from *;
      is
      begin
         CheckWrap (EStrings.Get_Length (E_Str => Str));
         PutExaminerString (File => OutputFile,
                            Item => Str);
      end VCG_PutString;

      procedure VCG_PutLongString (Str  : in ELStrings.T)
      --# global in     OutputFile;
      --#        in     Wrap_Limit;
      --#        in out SPARK_IO.File_Sys;
      --#        in out SuppressWrap;
      --# derives SPARK_IO.File_Sys from *,
      --#                                OutputFile,
      --#                                Str,
      --#                                SuppressWrap,
      --#                                Wrap_Limit &
      --#         SuppressWrap      from *;
      is
      begin
         CheckWrap (ELStrings.Get_Length (E_Str => Str));
         ELStrings.Put_String (File  => OutputFile,
                               E_Str => ELStrings.Lower_Case (E_Str => Str));
      end VCG_PutLongString;

      -------------------------------------------------------

      procedure VCG_PutStringWithPrefix (Prefix : in String;
                                         Str    : in EStrings.T)
      --# global in     OutputFile;
      --#        in     Wrap_Limit;
      --#        in out SPARK_IO.File_Sys;
      --#        in out SuppressWrap;
      --# derives SPARK_IO.File_Sys from *,
      --#                                OutputFile,
      --#                                Prefix,
      --#                                Str,
      --#                                SuppressWrap,
      --#                                Wrap_Limit &
      --#         SuppressWrap      from *;
      is
         StrToPut : EStrings.T;
      begin
         StrToPut := EStrings.Copy_String (Str => Prefix);
         EStrings.Append_Examiner_String (E_Str1 => StrToPut,
                                          E_Str2 => Str);
         VCG_PutString (StrToPut);
      end VCG_PutStringWithPrefix;

      -------------------------------------------------------

      procedure VCG_PutInteger (Value  : in Integer)
      --# global in     OutputFile;
      --#        in     Wrap_Limit;
      --#        in out SPARK_IO.File_Sys;
      --#        in out SuppressWrap;
      --# derives SPARK_IO.File_Sys from *,
      --#                                OutputFile,
      --#                                SuppressWrap,
      --#                                Value,
      --#                                Wrap_Limit &
      --#         SuppressWrap      from *;
      is
         function Width (N : Natural) return Natural
         is
            Num,
            Wid : Natural;
         begin
            Num := N;
            Wid := 0;
            loop
               Num := Num / 10;
               Wid := Wid + 1;
               exit when Num = 0;
            end loop;
            return Wid;
         end Width;

      begin --VCG_PutInteger
         CheckWrap (Width (Value));
         SPARK_IO.Put_Integer (OutputFile, Value, 0, 10);
      end VCG_PutInteger;

      -------------------------------------------------------

      procedure PrintSymbol (File   : in SPARK_IO.File_Type;
                             Scope  : in Dictionary.Scopes;
                             Sym    : in Dictionary.Symbol;
                             Kind   : in Cells.Cell_Kind)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Wrap_Limit;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                File,
      --#                                Kind,
      --#                                LexTokenManager.State,
      --#                                Scope,
      --#                                Sym,
      --#                                Wrap_Limit;
      is
         PrefixFnLength : Natural;

         PackageExStr   : EStrings.T;

         ExStr  : EStrings.T;

         procedure PossiblyPrintUnderbar (S : in EStrings.T)
         --# global in     File;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                File,
         --#                                S;
         is
            --This horrible kludge is to add a trailing underbar to user-declared
            --quantifiers to prevent any capture by local variables of the same name.
            --The kludge is that we don't want to add one to quantifiers that have been
            --added by the Examiner for RTC purposes. These are characterized by
            --having names of the form xxxxx__nn where x are any chars and n are digits.
            --We add a trailing underbar if the terminal characters are not digits or
            --if the are not preceded by a double underbar

            subtype Numerals is Character range '0' .. '9';
            I : EStrings.Lengths;
            UnderBarWanted : Boolean := True;

         begin --PossiblyPrintUnderbar
               --can't be the kind we are interested in unless last char is a numeral
            if EStrings.Get_Element (E_Str => S,
                                     Pos   => EStrings.Get_Length (E_Str => S)) in Numerals then
               --now consume any other numerals that might be there
               I := EStrings.Get_Length (E_Str => S) - 1; --penultimate char
               while I > 2 and then --2 is the lowest index that could be ok e.g. "x__"
                 EStrings.Get_Element (E_Str => S,
                                       Pos   => I) in Numerals
               loop
                  I := I - 1;
               end loop;
               --I now points at the first non-numeral from the back of the string
               --we don't need an underbar if Ith and (I-1)th characters are underbars
               if EStrings.Get_Element (E_Str => S,
                                        Pos   => I) = '_' and then
                 EStrings.Get_Element (E_Str => S,
                                       Pos   => I - 1) = '_' then
                  UnderBarWanted := False;
               end if;
            end if;

            --print an underbar unless the above search has shown we don't want one
            if UnderBarWanted then
               Put_Char (File, '_');
            end if;
         end PossiblyPrintUnderbar;

      begin --PrintSymbol
         if Sym = Dictionary.NullSymbol then
            Put_String (File, "unexpected_null_symbol");
         elsif Dictionary.IsUnknownTypeMark (Sym) then
            Put_String (File, "unknown_type");

         else
            case Kind is
               when Cells.Mk_Aggregate   => PrefixFnLength := 4;
               when others              => PrefixFnLength := 0;
            end case;

            if Dictionary.IsRecordComponent (Sym) then
               PackageExStr := EStrings.Empty_String;
            else
               Dictionary.GetAnyPrefixNeeded (Sym, Scope, "__", PackageExStr);
            end if;

            Dictionary.GenerateSimpleName (Sym, "__", ExStr);

            if ((((SPARK_IO.Col (File) +
                     PrefixFnLength) +
                    EStrings.Get_Length (E_Str => PackageExStr)) +
                   EStrings.Get_Length (E_Str => ExStr)) + 12)  > Wrap_Limit then

               SPARK_IO.New_Line (File, 1);
               Put_String (File, "           ");
            end if;

            case Kind is
               when Cells.Mk_Aggregate => Put_String (File, "mk__");
               when others            => null;
            end case;

            if EStrings.Get_Length (E_Str => PackageExStr) > 0 then
               PutExaminerString (File => File,
                                  Item => PackageExStr);
               Put_String (File, "__");
            elsif not EStrings.Is_Empty (E_Str => CommandLineData.Content.FDLmangle) then
               AdjustFDL_RWs.PossiblyAdjust (ExStr, CommandLineData.Content.FDLmangle);
            end if;

            PutExaminerString (File => File,
                               Item => ExStr);
            if Dictionary.IsQuantifiedVariable (Sym) then
               PossiblyPrintUnderbar (ExStr);
            end if;
         end if;
      end PrintSymbol;

      -------------------------------------------------------

      procedure PrintProcedureExport (CellName : in Cells.Cell;
                                      Scope    : in Dictionary.Scopes)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     Heap;
      --#        in     LexTokenManager.State;
      --#        in     OutputFile;
      --#        in     Wrap_Limit;
      --#        in out SPARK_IO.File_Sys;
      --#        in out SuppressWrap;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CellName,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                Heap,
      --#                                LexTokenManager.State,
      --#                                OutputFile,
      --#                                Scope,
      --#                                SuppressWrap,
      --#                                Wrap_Limit &
      --#         SuppressWrap      from *;
      is
         PackExStr,
         SymExStr,
         ExportString : EStrings.T;
         VarSym       : Dictionary.Symbol;

      begin
         ExportString := EStrings.Empty_String;

         -- Now print export name part
         VarSym := Cells.Get_Symbol_Value (Heap, CellName);
         Dictionary.GetAnyPrefixNeeded (VarSym, Scope, "__", PackExStr);
         if EStrings.Get_Length (E_Str => PackExStr) > 0 then
            EStrings.Append_Examiner_String (E_Str1 => ExportString,
                                             E_Str2 => PackExStr);
            EStrings.Append_String (E_Str => ExportString,
                                    Str   => "__");
         end if;
         Dictionary.GenerateSimpleName (VarSym, "_", SymExStr);
         EStrings.Append_Examiner_String (E_Str1 => ExportString,
                                          E_Str2 => SymExStr);
         EStrings.Append_String (E_Str => ExportString,
                                 Str   => "__");

         -- now add counter
         EStrings.Append_Examiner_String
           (E_Str1 => ExportString,
            E_Str2 => LexTokenManager.Lex_String_To_String
              (Lex_Str => Cells.Get_Lex_Str (Heap, CellName)));
         VCG_PutString (ExportString);

      end PrintProcedureExport;

      procedure PrintManifestConstantCell (CellName : in Cells.Cell)
      --# global in     Heap;
      --#        in     LexTokenManager.State;
      --#        in     OutputFile;
      --#        in     Wrap_Limit;
      --#        in out SPARK_IO.File_Sys;
      --#        in out SuppressWrap;
      --# derives SPARK_IO.File_Sys from *,
      --#                                CellName,
      --#                                Heap,
      --#                                LexTokenManager.State,
      --#                                OutputFile,
      --#                                SuppressWrap,
      --#                                Wrap_Limit &
      --#         SuppressWrap      from *,
      --#                                CellName,
      --#                                Heap,
      --#                                LexTokenManager.State;
      is
         ExString : EStrings.T;
         ELString : ELStrings.T;
         Value    : Maths.Value;
         Err      : Maths.ErrorCode;
         LStr     : LexTokenManager.Lex_String;

         procedure PrintStringLiteral
         --# global in     ExString;
         --#        in     OutputFile;
         --#        in     Wrap_Limit;
         --#        in out SPARK_IO.File_Sys;
         --#        in out SuppressWrap;
         --# derives SPARK_IO.File_Sys from *,
         --#                                ExString,
         --#                                OutputFile,
         --#                                SuppressWrap,
         --#                                Wrap_Limit &
         --#         SuppressWrap      from *,
         --#                                ExString;
         is
            Separator : Character;
            Position  : Positive;

            procedure PrintOneElement (CharCode : in Integer)
            --# global in     OutputFile;
            --#        in     Position;
            --#        in     Separator;
            --#        in     Wrap_Limit;
            --#        in out SPARK_IO.File_Sys;
            --#        in out SuppressWrap;
            --# derives SPARK_IO.File_Sys from *,
            --#                                CharCode,
            --#                                OutputFile,
            --#                                Position,
            --#                                Separator,
            --#                                SuppressWrap,
            --#                                Wrap_Limit &
            --#         SuppressWrap      from *;
            is
            begin  -- PrintOneElement
               Put_Char (OutputFile, Separator);
               Put_Char (OutputFile, ' ');
               CheckWrap (5);
               Put_Char (OutputFile, '[');
               VCG_PutInteger (Position);
               Put_Char (OutputFile, ']');
               CheckWrap (4);
               Put_String (OutputFile, " := ");
               VCG_PutInteger (CharCode);
            end PrintOneElement;

         begin -- PrintStringLiteral
            case EStrings.Get_Length (E_Str => ExString) is
               when 0 | 1 =>
                  SystemErrors.FatalError (SystemErrors.PreconditionFailure,
                                           "Mal-formed string literal in VCG PrintStringLiteral");
               when 2 =>
                  -- Must be "" - the null string literal
                  SystemErrors.RTAssert (EStrings.Get_Element (E_Str => ExString,
                                                               Pos   => 1) = '"' and then
                                           EStrings.Get_Element (E_Str => ExString,
                                                                 Pos   => 2) = '"',
                                         SystemErrors.PreconditionFailure,
                                         "Mal-formed NULL string literal in VCG PrintStringLiteral");
                  CheckWrap (9);
                  Put_String (OutputFile, Null_String_Literal_Name);
               when others =>
                  CheckWrap (9);
                  Put_String (OutputFile, "mk__string");
                  Separator := '(';
                  Position := 1;
                  for I in EStrings.Positions range 2 .. EStrings.Get_Length (E_Str => ExString) - 1 loop
                     PrintOneElement (Character'Pos (EStrings.Get_Element (E_Str => ExString,
                                                                           Pos   => I)));
                     Separator := ',';
                     Position := Position + 1;
                  end loop;
                  SPARK_IO.Put_Char (OutputFile, ')');
            end case;
         end PrintStringLiteral;

      begin --PrintManifestConstantCell
         LStr := Cells.Get_Lex_Str (Heap, CellName);
         ExString := LexTokenManager.Lex_String_To_String (Lex_Str => LStr);
         if EStrings.Get_Element (E_Str => ExString,
                                  Pos   => 1) = ''' then --character literal
            VCG_PutInteger (Character'Pos (EStrings.Get_Element (E_Str => ExString,
                                                                 Pos   => 2)));

         elsif EStrings.Get_Element (E_Str => ExString,
                                     Pos   => 1) = '"' then --string literal
            PrintStringLiteral;
         else -- should be a numeric
            Maths.LiteralToValue (LStr,
                                    -- to get
                                  Value,
                                  Err);
            if Err = Maths.NoError then
               ELString := Maths.ValueToString (Value);
            else
               ELString := ELStrings.ToExaminerLongString (ExString);
            end if;
            VCG_PutLongString (ELString);
         end if;
      end PrintManifestConstantCell;

      -------------------------------------------------------

      procedure PutS (Item : in String)
      --# global in     OutputFile;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Item,
      --#                                OutputFile;
      is
      begin
         SPARK_IO.Put_String (OutputFile, Item, 0);
      end PutS;

   begin -- PrintCellContents
      case Cells.Get_Kind (Heap, CellName) is
         when Cells.Manifest_Const =>
            PrintManifestConstantCell (CellName);
         when Cells.Attrib_Value |
           Cells.Attrib_Function =>
            PutExaminerString
              (File => OutputFile,
               Item => LexTokenManager.Lex_String_To_String (Lex_Str => Cells.Get_Lex_Str (Heap, CellName)));
         when Cells.Field_Access_Function =>
            ExString := LexTokenManager.Lex_String_To_String (Lex_Str => Cells.Get_Lex_Str (Heap, CellName));
            if not EStrings.Is_Empty (E_Str => CommandLineData.Content.FDLmangle) then
               AdjustFDL_RWs.PossiblyAdjust (ExString, CommandLineData.Content.FDLmangle);
            end if;
            VCG_PutStringWithPrefix ("fld_", ExString);
         when Cells.Field_Update_Function =>
            ExString := LexTokenManager.Lex_String_To_String (Lex_Str => Cells.Get_Lex_Str (Heap, CellName));
            if not EStrings.Is_Empty (E_Str => CommandLineData.Content.FDLmangle) then
               AdjustFDL_RWs.PossiblyAdjust (ExString, CommandLineData.Content.FDLmangle);
            end if;
            VCG_PutStringWithPrefix ("upf_", ExString);
         when Cells.Element_Function =>
            PutS ("element");
         when Cells.Update_Function =>
            PutS ("update");
         when Cells.Pred_Function =>
            PutS ("pred");
         when Cells.Succ_Function =>
            PutS ("succ");
         when Cells.Abs_Function =>
            PutS ("abs");
         when Cells.Trunc_Function =>
            PutS ("round__");
         when Cells.List_Function =>
            null;
         when Cells.FDL_Div_Op =>
            PutS (" div ");
         when Cells.Op =>

            if SPARK_IO.Col (OutputFile) > (Wrap_Limit + 3) and then
              Cells.Get_Op_Symbol (Heap, CellName) /= SPSymbols.apostrophe then

               SPARK_IO.New_Line (OutputFile, 1);
               SPARK_IO.Put_String (OutputFile, "          ", 0);
            end if;

            case Cells.Get_Op_Symbol (Heap, CellName) is
               when SPSymbols.colon              =>
                  PutS (": ");
               when SPSymbols.comma              =>
                  PutS (", ");
               when SPSymbols.RWand              |
                 SPSymbols.RWandthen             =>
                  PutS (" and ");
               when SPSymbols.RWor               |
                 SPSymbols.RWorelse              =>
                  PutS (" or ");
               when SPSymbols.equals             =>
                  PutS (" = ");
               when SPSymbols.not_equal          =>
                  if Escape_DOT then
                     PutS (" \<\> ");
                  else
                     PutS (" <> ");
                  end if;
               when SPSymbols.less_than          =>
                  if Escape_DOT then
                     PutS (" \< ");
                  else
                     PutS (" < ");
                  end if;
               when SPSymbols.less_or_equal      =>
                  if Escape_DOT then
                     PutS (" \<= ");
                  else
                     PutS (" <= ");
                  end if;
               when SPSymbols.greater_than       =>
                  if Escape_DOT then
                     PutS (" \> ");
                  else
                     PutS (" > ");
                  end if;
               when SPSymbols.greater_or_equal   =>
                  if Escape_DOT then
                     PutS (" \>= ");
                  else
                     PutS (" >= ");
                  end if;
               when SPSymbols.implies            =>
                  if Escape_DOT then
                     PutS (" -\> ");
                  else
                     PutS (" -> ");
                  end if;
               when SPSymbols.is_equivalent_to   =>
                  if Escape_DOT then
                     PutS (" \<-\> ");
                  else
                     PutS (" <-> ");
                  end if;
               when SPSymbols.plus               =>
                  if Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, CellName)) then
                     -- Unary + is semantically meaningless, so no point
                     -- printing it.
                     SuppressWrap := True;
                  else
                     PutS (" + "); -- Binary
                  end if;

               when SPSymbols.minus              =>
                  if Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, CellName)) then
                     PutS (" -");  -- Unary
                     SuppressWrap := True;
                  else
                     PutS (" - "); -- Binary
                  end if;

               when SPSymbols.multiply           => PutS (" * ");
               when SPSymbols.divide             => PutS (" / ");
               when SPSymbols.RWmod              => PutS (" mod ");
               when SPSymbols.double_star        => PutS (" ** ");
               when SPSymbols.RWnot              => PutS ("not ");

               when SPSymbols.apostrophe         =>
                  ---------------------------------
                  -- apostrophe becomes "__" in FDL
                  ---------------------------------
                  PutS ("__");

               when SPSymbols.ampersand          => PutS (" & ");
               when SPSymbols.becomes            => PutS (" := ");
               when SPSymbols.double_dot         => PutS (" .. ");
               when SPSymbols.RWforall           => PutS ("for_all");
               when SPSymbols.RWforsome          => PutS ("for_some");
                  --------------------------------------------------------------------
               when SPSymbols.right_paren        => PutS (" Parenthesis_Requested");
               when others                       => PutS (" undef_op_value ");
            end case;

         when Cells.Named_Const       |
           Cells.Declared_Function    |
           Cells.Proof_Function       |
           Cells.Modified_Op          |
           Cells.Reference            |
           Cells.Constraining_Index   |
           Cells.Fixed_Var            |
           Cells.Unconstrained_Attribute_Prefix |
           Cells.Mk_Aggregate =>

            ThisCellSymValue    := Cells.Get_Symbol_Value (Heap, CellName);
            ThisCellKind        := Cells.Get_Kind (Heap, CellName);
            ThisCellStringValue := Cells.Get_Lex_Str (Heap, CellName);
            ThisCellOp          := Cells.Get_Op_Symbol (Heap, CellName);

            if ThisCellKind = Cells.Mk_Aggregate and then
               Dictionary.IsSubtype (ThisCellSymValue) then
               ThisCellSymValue := Dictionary.GetRootType (ThisCellSymValue);
            end if;

            PrintSymbol (OutputFile,
                         Scope,
                         ThisCellSymValue,
                         ThisCellKind);

            -- An unconstrained attribute prefix _might_ have a tilde,
            -- such as O~'First.  This is allowed by the grammar,
            -- but removed here, since the attributes of an unconstrained
            -- array parameter cannot change during the lifetime of the
            -- object.  Essentially, we know that O~'First = O'First,
            -- so we just print the latter.
            if ThisCellOp = SPSymbols.tilde and
               ThisCellKind /= Cells.Unconstrained_Attribute_Prefix then
               PutS ("~");
            end if;

            -- Only print numeric suffix if number present in string field
            if ThisCellKind = Cells.Declared_Function and then
              LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ThisCellStringValue,
                                                                   Lex_Str2 => LexTokenManager.Null_String) /= LexTokenManager.Str_Eq then
               PutS ("__");
               PutExaminerString
                 (File => OutputFile,
                  Item => LexTokenManager.Lex_String_To_String (Lex_Str => ThisCellStringValue));
            end if;

         when Cells.Procedure_Export =>
            PrintProcedureExport (CellName, Scope);

         when Cells.Procedure_Name |
            Cells.Call_Counter     => null;

         when Cells.Pending_Function =>
            PutS (" Incomplete_Function");
         when Cells.Aggregate_Counter |
           Cells.Incomplete_Aggregate =>
            PutS (" Incomplete_Aggregate");

         when Cells.Return_Var =>
            PutS ("return");

         when Cells.Root_Integer =>
            PutS ("system__");
            -- put out min or max
            PutExaminerString
              (File => OutputFile,
               Item => LexTokenManager.Lex_String_To_String (Lex_Str => Cells.Get_Lex_Str (Heap, CellName)));
            -- and complete with _int
            PutS ("_int");

         when Cells.Bitwise_Op =>
            if Dictionary.TypeIsArray (Cells.Get_Symbol_Value (Heap, CellName)) then

               PrintSymbol (OutputFile,
                            Scope,
                            Cells.Get_Symbol_Value (Heap, CellName),
                            Cells.Get_Kind (Heap, CellName));
               PutS ("__");
               case Cells.Get_Op_Symbol (Heap, CellName) is
                  when SPSymbols.RWand => PutS ("and");
                  when SPSymbols.RWor  => PutS ("or");
                  when SPSymbols.RWxor => PutS ("xor");
                  when SPSymbols.RWnot => PutS ("not");
                  when others          => PutS ("undef_op_value");
               end case;
            elsif Dictionary.TypeIsModular (Cells.Get_Symbol_Value (Heap, CellName)) then

               case Cells.Get_Op_Symbol (Heap, CellName) is
                  -- Note bitwise "not" for modular types is expanded in the
                  -- VCG.ProduceVCs.BuildGraph, and so should never appear here.
                  when SPSymbols.RWand => PutS ("bit__and");
                  when SPSymbols.RWor  => PutS ("bit__or");
                  when SPSymbols.RWxor => PutS ("bit__xor");
                  when others          => PutS ("undef_op_value");
               end case;

            end if;


         when Cells.Unknown_Kind =>
            PutS (" unknown_cell_kind ");
      end case;
   end PrintCellContents;

   -------------------------------------------------------------------------

   function IsLeaf (Node : Cells.Cell;
                    Heap : Cells.Heap_Record) return Boolean
   is
   begin
      return Cells.Is_Null_Cell (Cells.Get_B_Ptr (Heap, Node));
   end IsLeaf;

   --------------------------------------------------------------

   procedure PrintDagLocal (Heap        : in out Cells.Heap_Record;
                            OutputFile  : in     SPARK_IO.File_Type;
                            Root        : in     Cells.Cell;
                            Scope       : in     Dictionary.Scopes;
                            Wrap_Limit  : in     Positive)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives Heap,
   --#         Statistics.TableUsage from *,
   --#                                    Heap,
   --#                                    Root &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    LexTokenManager.State,
   --#                                    OutputFile,
   --#                                    Root,
   --#                                    Scope,
   --#                                    Wrap_Limit;
   is
      P,
      ParenthesisCell,
      SqBracketCell   : Cells.Cell;
      ParenthesisForm : SPSymbols.SPSymbol;
      ParReqd         : Boolean;
      S               : CStacks.Stack;
      SuppressWrap    : Boolean;

      --------------------------------------------------------------

      procedure Parenthesise (V               : in     Cells.Cell;
                              LeftTree        : in     Boolean;
                              ParReqd         :    out Boolean;
                              ParenthesisForm :    out SPSymbols.SPSymbol)
      --# global in Heap;
      --# derives ParenthesisForm from Heap,
      --#                              V &
      --#         ParReqd         from Heap,
      --#                              LeftTree,
      --#                              V;
      is

         VPrecedence,
         WPrecedence : Natural;
         Operand,
         W           : Cells.Cell;
         V_Kind      : Cells.Cell_Kind;

         --------------------------------------------

         function PrecedenceValue (C : Cells.Cell) return Natural
         --# global in Heap;
         is
            PrecVal : Natural;
         begin
            if Cells.Get_Kind (Heap, C) = Cells.FDL_Div_Op then
               PrecVal := 5;
            else
               case Cells.Get_Op_Symbol (Heap, C) is
                  when SPSymbols.RWand           |
                     SPSymbols.RWor            |
                     SPSymbols.RWandthen       |
                     SPSymbols.RWorelse        |
                     SPSymbols.implies         |
                     SPSymbols.RWnot           |
                     SPSymbols.is_equivalent_to  => PrecVal := 1;
                  when SPSymbols.equals          |
                     SPSymbols.not_equal       |
                     SPSymbols.less_than       |
                     SPSymbols.less_or_equal   |
                     SPSymbols.greater_than    |
                     SPSymbols.greater_or_equal  => PrecVal := 2;
                  when SPSymbols.plus            |
                     SPSymbols.minus           |
                     SPSymbols.ampersand         => PrecVal := 3;

                     -- arity is taken into account by examining node degrees of operator nodes
                     -- (see body of procedure Parenthesise).
                     --        when SPSymbols.unary_plus      |
                     --             SPSymbols.unary_minus       => PrecVal := 4;

                  when SPSymbols.multiply        |
                     SPSymbols.divide          |
                     SPSymbols.RWmod             => PrecVal := 5;
                  when SPSymbols.double_star       => PrecVal := 6;
                  when others                      => PrecVal := 7;
               end case;
            end if;
            return PrecVal;
         end PrecedenceValue;

         --------------------------------------------------------------------

      begin -- Parenthesise;
         ParReqd := False;
         ParenthesisForm := SPSymbols.left_paren;
         V_Kind := Cells.Get_Kind (Heap, V);
         if (V_Kind = Cells.Declared_Function) or
            (V_Kind = Cells.Proof_Function)    or
            (V_Kind = Cells.Attrib_Function)   or
            (V_Kind = Cells.Field_Access_Function) or
            (V_Kind = Cells.Mk_Aggregate)      or
            (V_Kind = Cells.List_Function)     or
            (V_Kind = Cells.Element_Function)  or
            (V_Kind = Cells.Update_Function)   or
            (V_Kind = Cells.Pred_Function)     or
            (V_Kind = Cells.Succ_Function)     or
            (V_Kind = Cells.Abs_Function)      or
            (V_Kind = Cells.Trunc_Function)    or
            (V_Kind = Cells.Field_Update_Function) or
            (V_Kind = Cells.Bitwise_Op)        then
            ParReqd := True;
            if (V_Kind = Cells.List_Function) then
               ParenthesisForm := SPSymbols.square_open;
            end if;
         elsif ((V_Kind = Cells.Op)
                  -- TEMPORARY FIX until right_paren given its own kind
                 and then ((Cells.Get_Op_Symbol (Heap, V) /= SPSymbols.right_paren)
                              -- END OF TEMPORARY FIX
                            and
                            (Cells.Get_Op_Symbol (Heap, V) /= SPSymbols.comma)))
            or else (V_Kind = Cells.FDL_Div_Op) then

            if Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, V)) then
               -- V is a monadic operator;
               Operand := Cells.Get_B_Ptr (Heap, V);
               if not IsLeaf (Operand, Heap) then
                  ParReqd := True;
               end if;
            else
               if LeftTree then
                  W := Cells.Get_A_Ptr (Heap, V);
               else
                  W := Cells.Get_B_Ptr (Heap, V);
               end if;
               if not Cells.Is_Null_Cell (W) then
                  if (Cells.Get_Kind (Heap, W) = Cells.Op) or else
                     (Cells.Get_Kind (Heap, W) = Cells.FDL_Div_Op) then

                     VPrecedence := PrecedenceValue (V);
                     WPrecedence := PrecedenceValue (W);

                     -- general rule for constructing unambiguous expressions:
                     ParReqd := (VPrecedence > WPrecedence) or
                        ((VPrecedence = WPrecedence) and not LeftTree);

                     -- supplementary rules, to improve clarity:
                     if (VPrecedence = 1) or        -- v is a logical operation;
                        (WPrecedence = 2) then      -- subtree is a relation;
                        ParReqd := True;
                     end if;
                  end if;
               end if;
            end if;
         end if;
      end Parenthesise;

      -----------------------------------------------------------------------

   begin -- PrintDagLocal
         -- Algorithm of D.E. Knuth, Fundamental Algorithms, p.317;
      SuppressWrap := False;
      CStacks.CreateStack (S);
      Cells.Create_Cell (Heap, ParenthesisCell);
      Cells.Set_Kind (Heap, ParenthesisCell, Cells.Op);
      Cells.Set_Op_Symbol (Heap, ParenthesisCell, SPSymbols.left_paren);
      Cells.Create_Cell (Heap, SqBracketCell);
      Cells.Set_Kind (Heap, SqBracketCell, Cells.Op);
      Cells.Set_Op_Symbol (Heap, SqBracketCell, SPSymbols.square_open);
      P := Root;
      loop
         loop
            exit when Cells.Is_Null_Cell (P);
            CStacks.Push (Heap, P, S);
            if IsLeaf (P, Heap) then
               P := Cells.Null_Cell;
            else
               if (not Cells.Is_Null_Cell (Cells.Get_A_Ptr (Heap, P))) then
                  Parenthesise (P, True, ParReqd, ParenthesisForm);
                  if ParReqd then
                     if ParenthesisForm = SPSymbols.left_paren then
                        Put_String (OutputFile, "(");
                        CStacks.Push (Heap, ParenthesisCell, S);
                     else
                        CStacks.Push (Heap, SqBracketCell, S);
                     end if;
                  end if;
               end if;
               P := Cells.Get_A_Ptr (Heap, P);
            end if;
         end loop;
         exit when CStacks.IsEmpty (S);
         P := CStacks.Top (Heap, S);
         CStacks.Pop (Heap, S);
         PrintCellContents (Heap, OutputFile, P, SuppressWrap, Scope, Wrap_Limit, False);
         if IsLeaf (P, Heap) then
            P := Cells.Null_Cell;
            loop
               exit when
                  not ((Cells.Are_Identical (CStacks.Top (Heap, S),
                                            ParenthesisCell))     or
                       (Cells.Are_Identical (CStacks.Top (Heap, S),
                                            SqBracketCell)));
               if (Cells.Are_Identical (CStacks.Top (Heap, S),
                                       ParenthesisCell)) then
                  Put_String (OutputFile, ")");
               else
                  Put_String (OutputFile, "]");
               end if;
               CStacks.Pop (Heap, S);
            end loop;
         else
            Parenthesise (P, False, ParReqd, ParenthesisForm);
            if ParReqd then
               if ParenthesisForm = SPSymbols.left_paren then
                  Put_String (OutputFile, "(");
                  CStacks.Push (Heap, ParenthesisCell, S);
               else
                  Put_String (OutputFile, "[");
                  CStacks.Push (Heap, SqBracketCell, S);
               end if;
            end if;
            P := Cells.Get_B_Ptr (Heap, P);
         end if;
      end loop;
   end PrintDagLocal;

   -------------------------------------------------------------------------

   procedure PrintDag (Heap        : in out Cells.Heap_Record;
                       OutputFile  : in     SPARK_IO.File_Type;
                       Root        : in     Cells.Cell;
                       Scope       : in     Dictionary.Scopes;
                       Wrap_Limit  : in     Positive)
   is
   begin
      PrintDagLocal (Heap,
                     OutputFile,
                     Root,
                     Scope,
                     Wrap_Limit);
   end PrintDag;

   -------------------------------------------------------------------------

   procedure Partition (Root        : in     Cells.Cell;
                        SubExpnList : in     Cells.Cell;
                        Heap        : in out Cells.Heap_Record)
   is
      P,
      SubExpn : Cells.Cell;
      S       : CStacks.Stack;
   begin
      CStacks.CreateStack (S);
      P := Root;
      loop
         loop
            exit when Cells.Is_Null_Cell (P);
            CStacks.Push (Heap, P, S);
            if (Cells.Get_Kind (Heap, P) = Cells.Op) and then
              ((Cells.Get_Op_Symbol (Heap, P) = SPSymbols.RWand) or
                 (Cells.Get_Op_Symbol (Heap, P) = SPSymbols.RWandthen)) then
               P := Cells.Get_A_Ptr (Heap, P);
            else
               Cells.Create_Cell (Heap, SubExpn);
               Cells.Set_B_Ptr (Heap, SubExpn, P);
               Clists.AppendCell (Heap, SubExpn, SubExpnList);
               P := Cells.Null_Cell;
            end if;
         end loop;
         exit when CStacks.IsEmpty (S);
         P := CStacks.Top (Heap, S);
         CStacks.Pop (Heap, S);
         if (Cells.Get_Kind (Heap, P) = Cells.Op) and then
           ((Cells.Get_Op_Symbol (Heap, P) = SPSymbols.RWand) or
              (Cells.Get_Op_Symbol (Heap, P) = SPSymbols.RWandthen)) then
            P := Cells.Get_B_Ptr (Heap, P);
         else
            P := Cells.Null_Cell;
         end if;
      end loop;
   end Partition;

   -------------------------------------------------------------------------

   procedure PrintLogicalExpn (Heap       : in out Cells.Heap_Record;
                               OutputFile : in     SPARK_IO.File_Type;
                               Scope      : in     Dictionary.Scopes;
                               Root       : in     Cells.Cell;
                               TypeOfExpn : in     ExpnType;
                               Wrap_Limit : in     Positive)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --# derives Heap,
   --#         Statistics.TableUsage from *,
   --#                                    Heap,
   --#                                    Root &
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    Dictionary.Dict,
   --#                                    Heap,
   --#                                    LexTokenManager.State,
   --#                                    OutputFile,
   --#                                    Root,
   --#                                    Scope,
   --#                                    TypeOfExpn,
   --#                                    Wrap_Limit;
   is
      SubExpnList : Cells.Cell;

      procedure PrintListOfExpns
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     OutputFile;
      --#        in     Scope;
      --#        in     SubExpnList;
      --#        in     TypeOfExpn;
      --#        in     Wrap_Limit;
      --#        in out Heap;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --# derives Heap,
      --#         Statistics.TableUsage from *,
      --#                                    Heap,
      --#                                    SubExpnList &
      --#         SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    Dictionary.Dict,
      --#                                    Heap,
      --#                                    LexTokenManager.State,
      --#                                    OutputFile,
      --#                                    Scope,
      --#                                    SubExpnList,
      --#                                    TypeOfExpn,
      --#                                    Wrap_Limit;
      is
         ClauseNmbr : Natural;
         ListMember : Cells.Cell;

         procedure PrintTypeOfExpn
         --# global in     OutputFile;
         --#        in     TypeOfExpn;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                OutputFile,
         --#                                TypeOfExpn;
         is
         begin
            case TypeOfExpn is
               when Conclusion => SPARK_IO.Put_String (OutputFile, "C", 0);
               when Condition  => SPARK_IO.Put_String (OutputFile, " ", 0);
               when Hypothesis => SPARK_IO.Put_String (OutputFile, "H", 0);
            end case;
         end PrintTypeOfExpn;

      begin

         --this looks like the place to suppress multiple Trues in hypotheses
         --and do something with trues in conclusions

         ClauseNmbr := 0;
         ListMember := Clists.FirstCell (Heap, SubExpnList);
         loop
            PrintTypeOfExpn;
            ClauseNmbr := ClauseNmbr + 1;
            SPARK_IO.Put_Integer (OutputFile, ClauseNmbr, 0, 10);
            SPARK_IO.Put_String (OutputFile, ":", 0);
            SPARK_IO.Set_Col (OutputFile, 8);
            PrintDag (Heap,
                      OutputFile,
                      Cells.Get_B_Ptr (Heap, ListMember),
                      Scope,
                      Wrap_Limit);
            SPARK_IO.Put_Line (OutputFile, " .", 0);
            ListMember := Clists.NextCell (Heap, ListMember);
            exit when Cells.Is_Null_Cell (ListMember);
         end loop;
      end PrintListOfExpns;

   begin -- PrintLogicalExpn
      Clists.CreateList (Heap, SubExpnList);
      Partition (Root, SubExpnList, Heap);
      PrintListOfExpns;
      Clists.DisposeOfList (Heap, SubExpnList);
   end PrintLogicalExpn;

   -------------------------------------------------------------------------

   procedure PrintLabel (Heap       : in out Cells.Heap_Record;
                         OutputFile : in     SPARK_IO.File_Type;
                         LabelName  : in     Labels.Label;
                         Scope      : in     Dictionary.Scopes;
                         Wrap_Limit : in     Positive)
   is
      PairCount   : Natural;
      CurrentPair : Pairs.Pair;

      -------------------------------------------------------

      procedure PrintPair (OutputFile : in    SPARK_IO.File_Type;
                           PairName   : in    Pairs.Pair)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Scope;
      --#        in     Wrap_Limit;
      --#        in out Heap;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --# derives Heap,
      --#         Statistics.TableUsage from *,
      --#                                    Heap,
      --#                                    PairName &
      --#         SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    Dictionary.Dict,
      --#                                    Heap,
      --#                                    LexTokenManager.State,
      --#                                    OutputFile,
      --#                                    PairName,
      --#                                    Scope,
      --#                                    Wrap_Limit;
      -- prints a predicate-action pair;
      is
         Action,
         ModCell,
         Predicate    : Cells.Cell;
         SuppressWrap : Boolean;
      begin -- PrintPair
         SuppressWrap := False;
         SPARK_IO.Put_Line (OutputFile,
                            "      Traversal condition:", 0);
         if Pairs.IsTrue (Heap, PairName) then
            SPARK_IO.Put_Line (OutputFile,
                               " 1:     true .", 0);
         else
            Predicate := Cells.Get_B_Ptr (Heap, Pairs.PairHead (PairName));
            PrintLogicalExpn (Heap,
                              OutputFile,
                              Scope,
                              Predicate,
                              Condition,
                              Wrap_Limit);
         end if;
         SPARK_IO.Put_Line (OutputFile,
                            "      Action:", 0);
         if Pairs.IsUnitAction (Heap, PairName) then
            SPARK_IO.Put_Line (OutputFile,
                               "        null .", 0);
         else
            -- print action;
            Action := Cells.Get_C_Ptr (Heap, Pairs.PairHead (PairName));
            ModCell := Clists.FirstCell (Heap, Action);

            SPARK_IO.Put_String (OutputFile, "        ", 0);
            PrintCellContents (Heap,
                               OutputFile,
                               ModCell,
                               SuppressWrap,
                               Scope,
                               Wrap_Limit,
                               False);
            SPARK_IO.Put_String (OutputFile,
                                 " := ", 0);
            PrintDag (Heap,
                      OutputFile,
                      Cells.Get_B_Ptr (Heap, ModCell),
                      Scope,
                      Wrap_Limit);
            ModCell := Clists.NextCell (Heap, ModCell);

            loop
               exit when Cells.Is_Null_Cell (ModCell);

               SPARK_IO.Put_Line (OutputFile, " &", 0);
               SPARK_IO.Put_String (OutputFile, "        ", 0);
               PrintCellContents (Heap,
                                  OutputFile,
                                  ModCell,
                                  SuppressWrap,
                                  Scope,
                                  Wrap_Limit,
                                  False);
               SPARK_IO.Put_String (OutputFile,
                                    " := ", 0);
               PrintDag (Heap,
                         OutputFile,
                         Cells.Get_B_Ptr (Heap, ModCell),
                         Scope,
                         Wrap_Limit);
               ModCell := Clists.NextCell (Heap, ModCell);
            end loop;
            SPARK_IO.Put_Line (OutputFile, " .", 0);
         end if;
      end PrintPair;

   begin -- PrintLabel
      PairCount := 1;
      CurrentPair := Labels.FirstPair (Heap, LabelName);
      loop
         exit when Pairs.IsNullPair (CurrentPair);
         SPARK_IO.Put_String (OutputFile, "    Path ", 0);
         SPARK_IO.Put_Integer (OutputFile, PairCount, 2, 10);
         SPARK_IO.New_Line   (OutputFile, 1);
         PrintPair (OutputFile, CurrentPair);
         PairCount := PairCount + 1;
         CurrentPair := Labels.NextPair (Heap, CurrentPair);
      end loop;
   end PrintLabel;

   -------------------------------------------------------------------------

   procedure PrintVCFormula (Heap          : in out Cells.Heap_Record;
                             OutputFile    : in     SPARK_IO.File_Type;
                             PredicatePair : in     Pairs.Pair;
                             Scope         : in     Dictionary.Scopes;
                             Wrap_Limit    : in     Positive)
   is
      ConclusionRoot,
      HypothesisRoot : Cells.Cell;

      function IsTriviallyTrue (DAG : Cells.Cell) return Boolean
      --# global in Dictionary.Dict;
      --#        in Heap;
      is
         CurrentCell : Cells.Cell;
         Result      : Boolean := True;

         function IsTrueCell (TheCell : Cells.Cell) return Boolean
         --# global in Dictionary.Dict;
         --#        in Heap;
         is
         begin
            return Cells.Get_Kind (Heap, TheCell) = Cells.Named_Const and then
               Cells.Get_Symbol_Value (Heap, TheCell) = Dictionary.GetTrue;
         end IsTrueCell;

         function AppropriateBinaryOperator (OpSym : SPSymbols.SPSymbol) return Boolean
         is
         begin
            return OpSym = SPSymbols.RWand or else
               OpSym = SPSymbols.RWandthen or else
               OpSym = SPSymbols.RWor or else
               OpSym = SPSymbols.RWorelse or else
               OpSym = SPSymbols.equals or else
               OpSym = SPSymbols.implies or else
               OpSym = SPSymbols.is_equivalent_to;
         end AppropriateBinaryOperator;

      begin --IsTriviallyTrue
         CurrentCell := DAG;
         loop
            exit when IsTrueCell (CurrentCell); --success condition

            --some expression other than an operator - fail
            if Cells.Get_Kind (Heap, CurrentCell) /= Cells.Op then
               Result := False;
               exit;
            end if;

            --inappropriate operator - fail
            if not AppropriateBinaryOperator (Cells.Get_Op_Symbol (Heap, CurrentCell)) then
               Result := False;
               exit;
            end if;

            --thing on left of operator is not true - fail
            if not IsTrueCell (Cells.Get_A_Ptr (Heap, CurrentCell)) then
               Result := False;
               exit;
            end if;

            --move down right hand chain of tree to get next sub-expression
            CurrentCell := Cells.Get_B_Ptr (Heap, CurrentCell);

            --fallen off the end - fail - (I think this check is redundant but safe)
            if Cells.Is_Null_Cell (CurrentCell) then
               Result := False;
               exit;
            end if;

         end loop;
         return Result;
      end IsTriviallyTrue;

   begin --PrintVCFormula
      HypothesisRoot := Cells.Get_B_Ptr (Heap, Pairs.PairHead (PredicatePair));
      ConclusionRoot := Cells.Get_C_Ptr (Heap, Pairs.PairHead (PredicatePair));

      if IsTriviallyTrue (ConclusionRoot) then
         SPARK_IO.Put_Line (OutputFile,
                            "*** true .          /* trivially true VC removed by Examiner */", 0);
         SPARK_IO.New_Line (OutputFile, 1);

      else
         PrintLogicalExpn (Heap,
                           OutputFile,
                           Scope,
                           HypothesisRoot,
                           Hypothesis,
                           Wrap_Limit);
         SPARK_IO.Put_Line (OutputFile, "        ->", 0);
         PrintLogicalExpn (Heap,
                           OutputFile,
                           Scope,
                           ConclusionRoot,
                           Conclusion,
                           Wrap_Limit);
         SPARK_IO.Put_Line (OutputFile, " ", 0);
      end if;
   end PrintVCFormula;

   -------------------------------------------------------------------------

   procedure PrintDPC (Heap          : in out Cells.Heap_Record;
                       OutputFile    : in     SPARK_IO.File_Type;
                       PredicatePair : in     Pairs.Pair;
                       Scope         : in     Dictionary.Scopes;
                       Wrap_Limit    : in     Positive)
   is
      HypothesisRoot : Cells.Cell;
   begin -- PrintDPC
      HypothesisRoot := Cells.Get_B_Ptr (Heap, Pairs.PairHead (PredicatePair));

      PrintLogicalExpn (Heap,
                        OutputFile,
                        Scope,
                        HypothesisRoot,
                        Hypothesis,
                        Wrap_Limit);
      SPARK_IO.Put_Line (OutputFile, "        ->", 0);
      SPARK_IO.Put_Line (OutputFile, "C1:    false .", 0);
      SPARK_IO.Put_Line (OutputFile, " ", 0);
   end PrintDPC;


   -------------------------------------------------------------------------

   procedure Print_DAG_Dot (Heap        : in out Cells.Heap_Record;
                            OutputFile  : in     SPARK_IO.File_Type;
                            Root        : in     Cells.Cell;
                            Scope       : in     Dictionary.Scopes;
                            Wrap_Limit  : in     Positive)
   is
      -- This procedure uses a recursive DAG traversal, so is not
      -- SPARK.

      --# hide Print_DAG_Dot;

      type Edge_Class is (A, B);

      function Cell_Kind_Image (K : in Cells.Cell_Kind) return String
      is
      begin
         case K is
            when Cells.Manifest_Const =>
               return "Manifest_Const";
            when Cells.Attrib_Value =>
               return "Attrib_Value";
            when Cells.Attrib_Function =>
               return "Attrib_Function";
            when Cells.Field_Access_Function =>
               return "Field_Access_Function";
            when Cells.Field_Update_Function =>
               return "Field_Update_Function";
            when Cells.Element_Function   =>
               return "Element_Function";
            when Cells.Update_Function    =>
               return "Update_Function";
            when Cells.Pred_Function      =>
               return "Pred_Function";
            when Cells.Succ_Function      =>
               return "Succ_Function";
            when Cells.Abs_Function       =>
               return "Abs_Function";
            when Cells.Trunc_Function     =>
               return "Trunc_Function";
            when Cells.List_Function      =>
               return "List_Function";
            when Cells.FDL_Div_Op         =>
               return "FDL_Div_Op";
            when Cells.Op            =>
               return "Op";
            when Cells.Named_Const   =>
               return "Named_Const";
            when Cells.Declared_Function  =>
               return "Declared_Function";
            when Cells.Proof_Function     =>
               return "Proof_Function";
            when Cells.Modified_Op        =>
               return "Modified";
            when Cells.Reference          =>
               return "Reference";
            when Cells.Constraining_Index =>
               return "Constraining_Index";
            when Cells.Fixed_Var          =>
               return "Fixed_Var";
            when Cells.Unconstrained_Attribute_Prefix =>
               return "Uncon_Attribute_Prefix";
            when Cells.Mk_Aggregate       =>
               return "Mk_Aggregate";
            when Cells.Procedure_Export   =>
               return "Procedure_Export";
            when Cells.Procedure_Name     =>
               return "Procedure_Name";
            when Cells.Pending_Function   =>
               return "Pending_Function";
            when Cells.Aggregate_Counter  =>
               return "Aggregate_Counter";
            when Cells.Call_Counter       =>
               return "Call_Counter";
            when Cells.Incomplete_Aggregate =>
               return "Incomplete_Aggregate";
            when Cells.Return_Var         =>
               return "Return_Var";
            when Cells.Root_Integer       =>
               return "Root_Integer";
            when Cells.Bitwise_Op         =>
               return "Bitwise_Op";
            when Cells.Unknown_Kind       =>
               return "Unknown";
         end case;
      end Cell_Kind_Image;


      procedure PutS (S : in String)
      is
      begin
         SPARK_IO.Put_String (OutputFile, S, 0);
      end PutS;

      procedure PutL (S : in String)
      is
      begin
         SPARK_IO.Put_Line (OutputFile, S, 0);
      end PutL;

      procedure Print_Node (Root : in Cells.Cell;
                            Rank : in Natural)
      is
         pragma Unreferenced (Rank);
         SuppressWrap : Boolean := False;
      begin
         PutS (Natural'Image (Cells.Cell_Ref (Root)) & " [shape=record,label=""{{");
         PutS (Cell_Kind_Image
                 (Cells.Get_Kind (Heap, Root)) &
                 "\l|" & Natural'Image (Cells.Cell_Ref (Root)) & "\r}|");

         PrintCellContents
           (Heap, OutputFile, Root, SuppressWrap, Scope, Wrap_Limit, True);
         PutL ("}""];");
      end Print_Node;

      procedure Print_Edge (Head, Tail : in Cells.Cell;
                            Class      : in Edge_Class)
      is
      begin
         if not Cells.Is_Null_Cell (Tail) then
            PutS (Natural'Image (Cells.Cell_Ref (Head)) & " ->" &
                    Natural'Image (Cells.Cell_Ref (Tail)) & "[style=solid,label=");
            PutS (Edge_Class'Image (Class));
            PutL ("];");
         end if;
      end Print_Edge;

      procedure Traverse_DAG (Root : in Cells.Cell;
                              Rank : in Natural)
      is
         A_Child : Cells.Cell;
         B_Child : Cells.Cell;
      begin
         if Cells.Is_Null_Cell (Root) then
            null;
         else
            Print_Node (Root, Rank);

            A_Child := Cells.Get_A_Ptr (Heap, Root);
            B_Child := Cells.Get_B_Ptr (Heap, Root);

            -- if the A_Child is not null and not pointing
            -- at itself, then print the edge and that sub-dag.
            if (not Cells.Is_Null_Cell (A_Child)) and then
               (A_Child /= Root) then
               Print_Edge (Root, A_Child, A);
               Traverse_DAG (A_Child, Rank + 1);
            end if;

            -- if the B_Child is not null and not pointing
            -- at itself, then print the edge and that sub-dag.
            if (not Cells.Is_Null_Cell (B_Child)) and then
               (B_Child /= Root) then
               Print_Edge (Root, B_Child, B);
               Traverse_DAG (B_Child, Rank + 1);
            end if;
         end if;


      end Traverse_DAG;
   begin
      PutL ("digraph DAG {");
      PutL ("ranksep=""1.0 equally"";");
      PutL ("nodesep=1.0;");
      PutL ("node [shape=box,fontname=helvetica];");
      PutL ("edge [labelfontname=helvetica,labelfontsize=10];");

      Traverse_DAG (Root, 0);

      PutL ("}");
   end Print_DAG_Dot;



end DAG_IO;
