-- $Id: dag-createstructconstraint.adb 15520 2010-01-07 12:53:45Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with EStrings;
with ContextManager;

separate (DAG)
procedure CreateStructConstraint (OutputFile       : in     SPARK_IO.File_Type;
                                  StartTypeSym     : in     Dictionary.Symbol;
                                  VarSym           : in     Cells.Cell;
                                  Scope            : in     Dictionary.Scopes;
                                  AssocVar         : in     Dictionary.Symbol;
                                  VCGHeap          : in out Cells.Heap_Record;
                                  ContainsReals    : in out Boolean;
                                  VCGFailure       : in out Boolean;
                                  StructConstraint :    out Cells.Cell)
is
   QuantIdentStack,
   ConstraintStack : CStacks.Stack;

   RecCompType,
   QuantIdSym,
   InitialVariable,
   IVarSym,                      -- symbol for variable associated with current type
   TypeSym : Dictionary.Symbol;

   TrueCell,
   OpCell,
   FuncCell,
   QuantIdent,
   Result,
   TOS,
   TOS2,
   ArrayElem,
   RecordElem : Cells.Cell;

   exitUp : Boolean;

   QuantNum  : Natural := 0;

   LoopCount, Dimmax : Positive;


   -- Front end for access to ConstraintStack
   procedure PopOffConstraintStack (Result : out Cells.Cell)
   --# global in out ConstraintStack;
   --#        in out VCGHeap;
   --# derives ConstraintStack,
   --#         Result,
   --#         VCGHeap         from ConstraintStack,
   --#                              VCGHeap;
   is
   begin
      CStacks.PopOff (VCGHeap, ConstraintStack, Result);
   end PopOffConstraintStack;


   -- Front end for access to ConstraintStack
   procedure PushOnConstraintStack (Input : in Cells.Cell)
   --# global in out ConstraintStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ConstraintStack       from VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    ConstraintStack,
   --#                                    Input;
   is
   begin
      CStacks.Push (VCGHeap, Input, ConstraintStack);
   end PushOnConstraintStack;


   -- Front end for access to QuantIdentStack
   procedure PopOffQuantIdentStack (Result : out Cells.Cell)
   --# global in out QuantIdentStack;
   --#        in out VCGHeap;
   --# derives QuantIdentStack,
   --#         Result,
   --#         VCGHeap         from QuantIdentStack,
   --#                              VCGHeap;
   is
   begin
      CStacks.PopOff (VCGHeap, QuantIdentStack, Result);
   end PopOffQuantIdentStack;


   -- Front end for access to QuantIdentStack
   procedure PushOnQuantIdentStack (Input : in Cells.Cell)
   --# global in out QuantIdentStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives QuantIdentStack       from VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    Input,
   --#                                    QuantIdentStack;
   is
   begin
      CStacks.Push (VCGHeap, Input, QuantIdentStack);
   end PushOnQuantIdentStack;


   procedure CreateIdentStr (IdentStr :    out LexTokenManager.Lex_String;
                             Suffix   : in     Natural)
   --# global in out LexTokenManager.State;
   --# derives IdentStr,
   --#         LexTokenManager.State from LexTokenManager.State,
   --#                                    suffix;
   is
      subtype ExStr_T is String (EStrings.Positions);
      ExStr   : ExStr_T;
      ExLin   : EStrings.T;
      Wid     : EStrings.Lengths;
      Success : Boolean;

      function Width (N : Natural) return Natural
      is
         Num,
         Wid : Natural;
      begin
         Num := N;
         Wid := 0;
         loop
            Num := Num / 10;
            Wid := Wid + 1;
            exit when Num = 0;
         end loop;
         return Wid;
      end Width;

   begin -- CreateIdentStr
         -- Add I___ to start of ExLin
         -- Changed to aggregate to avoid flow error
      ExLin := EStrings.Copy_String (Str => "I___");
      Wid := Width (Suffix);
      --676 ExStr := EStrings.Contents'(others => ' '); -- avoid flow error
      SPARK_IO.Put_Int_To_String (ExStr, Suffix, 1, 10);

      for I in EStrings.Lengths range 1 .. Wid loop
         --# accept F, 10, Success, "Ineffective assignment here OK";
         EStrings.Append_Char (E_Str   => ExLin,
                               Ch      => ExStr ((EStrings.Max_String_Length - Wid) + I),
                               Success => Success);
         --# end accept;
      end loop;
      LexTokenManager.Insert_Examiner_String (Str     => ExLin,
                                              Lex_Str => IdentStr);
      --# accept F, 33, Success, "Expect Success unused";
   end CreateIdentStr;


   procedure CreateQuantIdent (QuantIdent  :    out Dictionary.Symbol;
                               ThisTypeSym : in     Dictionary.Symbol)
   --# global in     Scope;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out QuantNum;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.FILE_SYS     from *,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    QuantNum,
   --#                                    Scope,
   --#                                    ThisTypeSym &
   --#         LexTokenManager.State,
   --#         QuantNum              from *,
   --#                                    QuantNum &
   --#         QuantIdent            from Dictionary.Dict;
   is
      IdentStr : LexTokenManager.Lex_String;
   begin
      -- increment counter for uniqueness
      QuantNum := QuantNum + 1;

      CreateIdentStr (IdentStr, QuantNum);
      Dictionary.AddQuantifiedVariable (Name        => IdentStr,
                                        Comp_Unit   => ContextManager.NullUnit,
                                        Declaration => Dictionary.Null_Location,
                                        TypeMark    => ThisTypeSym,
                                        Region      => Dictionary.GetRegion (Scope),
                                        Variable    => QuantIdent);
   end CreateQuantIdent;


   -- Creates a 1-d array access
   procedure CreateArrayAccess (ArrayElem              :    out Cells.Cell;
                                ArrayIdent, QuantIdent : in     Cells.Cell)
   --# global in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ArrayElem             from ArrayIdent,
   --#                                    QuantIdent,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    ArrayIdent,
   --#                                    QuantIdent,
   --#                                    VCGHeap;
   is
      Param,
      DAGCell,
      AccessList : Cells.Cell;
   begin
      CreateCellKind (AccessList, VCGHeap, Cells.List_Function);
      SetRightArgument (AccessList, QuantIdent, VCGHeap);
      SetLeftArgument (QuantIdent, QuantIdent, VCGHeap);

      CreateOpCell (Param, VCGHeap, SPSymbols.comma);
      SetLeftArgument (Param, ArrayIdent, VCGHeap);
      SetRightArgument (Param, AccessList, VCGHeap);

      CreateCellKind (DAGCell, VCGHeap, Cells.Element_Function);
      SetRightArgument (DAGCell, Param, VCGHeap);

      ArrayElem := DAGCell;
   end CreateArrayAccess;

   -- Add an extra array quantifier to an
   -- "element (r,[i1,...,iM]" expression
   procedure ConcatArrayQuant (HeadDAG, IdentCell : in Cells.Cell)
   --# global in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    HeadDAG,
   --#                                    IdentCell,
   --#                                    VCGHeap;
   is
      HeadBracket,
      Param,
      HeadVar : Cells.Cell;
   begin
      -- Starts with "element (r,[i1,i2,..,iM]" in HeadDAG
      -- and "i_N" in IdentCell
      HeadBracket := RightPtr (VCGHeap, HeadDAG); -- points to ,
      HeadBracket := RightPtr (VCGHeap, HeadBracket); -- points to [
      HeadVar := RightPtr (VCGHeap, HeadBracket); -- points to i1 or ,
      while Cells.Get_Kind (VCGHeap, HeadVar) = Cells.Op
      loop
         HeadBracket := HeadVar;
         HeadVar := RightPtr (VCGHeap, HeadVar);
      end loop;
      -- HeadBracket now points to [ or , and B-links to HeadVar
      -- HeadVar points to iM and B-links to ]

      -- Set up the pointers in the Ident cell 'iN'
      SetLeftArgument (IdentCell, IdentCell, VCGHeap);
      SetRightArgument (IdentCell, RightPtr (VCGHeap, HeadVar), VCGHeap);

      -- Get a new comma cell and set up pointers
      -- to iM (left) and iN (right)
      CreateOpCell (Param, VCGHeap, SPSymbols.comma);
      SetLeftArgument (Param, HeadVar, VCGHeap);
      SetRightArgument (Param, IdentCell, VCGHeap);

      -- Now insert the comma into the expression
      SetRightArgument (HeadBracket, Param, VCGHeap);
      -- Should result in "element (r,[i1,i2,..,iM,iN]"
   end ConcatArrayQuant;

   -- Create a 'forall' without an explicit predicate
   procedure CreateShortQuantExpr (QuantExpr   :    out Cells.Cell;
                                   QuantIdent  : in     Cells.Cell;
                                   ThisTypeSym : in     Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives QuantExpr             from Dictionary.Dict,
   --#                                    QuantIdent,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    QuantIdent,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap;
   is
      BaseTypeSym : Dictionary.Symbol;
      TypeRange,
      VarDecl,
      Predicate,
      QuantBody,
      QuantDag,
      TypeIdent : Cells.Cell;
   begin
      -- create cell for base type sym
      BaseTypeSym := Dictionary.GetRootType (ThisTypeSym);
      CreateFixedVarCell (TypeIdent, VCGHeap, BaseTypeSym);

      -- create VarDecl as QuantIdent : TypeIdent
      CreateOpCell (VarDecl, VCGHeap, SPSymbols.colon);
      SetLeftArgument (VarDecl, QuantIdent, VCGHeap);
      SetRightArgument (VarDecl, TypeIdent, VCGHeap);

      -- prefix True expression with implication of type range,
      -- if not Boolean
      CreateTrueCell (VCGHeap, Predicate);
      if not Dictionary.TypeIsBoolean (ThisTypeSym) then
         CreateRangeConstraint (QuantIdent, ThisTypeSym, VCGHeap, TypeRange);
         Imply (TypeRange, VCGHeap, Predicate);
      end if;

      -- create QuantBody as (VarDecl, --)
      CreateOpCell (QuantBody, VCGHeap, SPSymbols.comma);
      SetLeftArgument (QuantBody, VarDecl, VCGHeap);
      SetRightArgument (QuantBody, Predicate, VCGHeap);

      CreateOpCell (QuantDag, VCGHeap, SPSymbols.RWforall);
      SetRightArgument (QuantDag, QuantBody, VCGHeap);

      QuantExpr := QuantDag;
      -- Should return
      --   "forall (QI:TI,Range->True)" or
      --   "forall (QI:TI,True)"
   end CreateShortQuantExpr;


   procedure CreateQuantExpr (QuantExpr   :    out Cells.Cell;
                              QuantIdent  : in     Cells.Cell;
                              ThisTypeSym : in     Dictionary.Symbol;
                              BoolExpr    : in     Cells.Cell)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives QuantExpr             from BoolExpr,
   --#                                    Dictionary.Dict,
   --#                                    QuantIdent,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    BoolExpr,
   --#                                    Dictionary.Dict,
   --#                                    QuantIdent,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap;
   is
      BaseTypeSym : Dictionary.Symbol;
      TypeRange,
      VarDecl,
      Predicate,
      QuantBody,
      QuantDag,
      TypeIdent : Cells.Cell;
   begin
      -- create cell for base type sym
      BaseTypeSym := Dictionary.GetRootType (ThisTypeSym);
      CreateFixedVarCell (TypeIdent, VCGHeap, BaseTypeSym);

      -- create VarDecl as QuantIdent : TypeIdent
      CreateOpCell (VarDecl, VCGHeap, SPSymbols.colon);
      SetLeftArgument (VarDecl, QuantIdent, VCGHeap);
      SetRightArgument (VarDecl, TypeIdent, VCGHeap);

      -- prefix BoolExpr with implication of [sub]type range, but
      -- suppress range for Boolean since Boolean subtypes must be full range
      Predicate := BoolExpr;
      if not Dictionary.TypeIsBoolean (ThisTypeSym) then
         CreateRangeConstraint (QuantIdent, ThisTypeSym, VCGHeap, TypeRange);
         Imply (TypeRange, VCGHeap, Predicate);
      end if;

      -- create QuantBody as (VarDecl, Predicate)
      CreateOpCell (QuantBody, VCGHeap, SPSymbols.comma);
      SetLeftArgument (QuantBody, VarDecl, VCGHeap);
      SetRightArgument (QuantBody, Predicate, VCGHeap);

      CreateOpCell (QuantDag, VCGHeap, SPSymbols.RWforall);
      SetRightArgument (QuantDag, QuantBody, VCGHeap);

      QuantExpr := QuantDag;
      --Should return "forall (QI:TI,Range->Bool)"
   end CreateQuantExpr;

   -- Compose two 'forall' quantifiers into a composite
   procedure ConcatQuantExpr (HeadExpr, TailExpr : in Cells.Cell)
   --# global in     Dictionary.Dict;
   --#        in     OutputFile;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                Dictionary.Dict,
   --#                                OutputFile,
   --#                                TailExpr,
   --#                                VCGHeap &
   --#         VCGFailure        from *,
   --#                                Dictionary.Dict,
   --#                                TailExpr,
   --#                                VCGHeap &
   --#         VCGHeap           from *,
   --#                                Dictionary.Dict,
   --#                                HeadExpr,
   --#                                TailExpr;
   is
      TailComma,
      TailImp : Cells.Cell;
   begin
      -- HeadExpr is (forall y:T, inrange-> (forall x:T, ...)
      -- TailExpr is (forall z:T, inrange->P)
      TailComma := RightPtr (VCGHeap, TailExpr);  -- point to ,
      TailImp := RightPtr (VCGHeap, TailComma);  -- point to -> or True

      if (Cells.Get_Symbol_Value (VCGHeap, TailImp) = Dictionary.GetTrue) then
         -- It's a True Cell, so dispose of it
         Cells.Dispose_Of_Cell (VCGHeap, TailImp);
         -- Now tag the predicate after ,
         SetRightArgument (TailComma, HeadExpr, VCGHeap);
      elsif (Cells.Get_Kind (VCGHeap, TailImp) = Cells.Op and
             Cells.Get_Op_Symbol (VCGHeap, TailImp) = SPSymbols.implies)
      then
         -- Need to garbage-collect what's on the
         -- B arrow of TailImp already; should be a True cell
         Cells.Dispose_Of_Cell (VCGHeap, RightPtr (VCGHeap, TailImp));
         -- Tag the predicate after ->
         SetRightArgument (TailImp, HeadExpr, VCGHeap);
      else
         -- We are in error here: generate a
         -- a warning that the VC may be corrupted
         SPARK_IO.New_Line (OutputFile, 1);
         SPARK_IO.Put_Line (OutputFile,
                            "!!!        'forall' structure in VC is corrupt.",
                            0);
         VCGFailure := True;
      end if;

      -- Result is now in TailExpr as
      -- "forall z:T, inrange-> (forall y:T, ...)"
   end ConcatQuantExpr;


   function GetNextRecordComponent (RecComp : Dictionary.Symbol)
                                   return Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      NextComp,
      RecType : Dictionary.Symbol;
      Num : Positive;
   begin
      RecType := Dictionary.GetRecordType (RecComp);
      Num := 1;
      loop
         exit when RecComp = Dictionary.GetNonExtendedRecordComponent (RecType, Num);
         Num := Num + 1;
      end loop;
      if Num < Dictionary.GetNumberOfNonExtendedComponents (RecType) then
         NextComp := Dictionary.GetNonExtendedRecordComponent (RecType, Num + 1);
      else
         NextComp := Dictionary.NullSymbol;
      end if;
      return NextComp;
   end GetNextRecordComponent;


   procedure CreateRecordAccess (RecordElem  :    out Cells.Cell;
                                 RecCompType : in     Dictionary.Symbol;
                                 RecIdent    : in     Cells.Cell)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives RecordElem            from VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    RecCompType,
   --#                                    RecIdent;
   is
      DAGCell : Cells.Cell;
   begin

      CreateCellKind (DAGCell, VCGHeap, Cells.Field_Access_Function);
      Cells.Set_Symbol_Value (VCGHeap, DAGCell, RecCompType);
      Cells.Set_Lex_Str (VCGHeap, DAGCell,
                          Dictionary.GetSimpleName (RecCompType));
      SetRightArgument (DAGCell, RecIdent, VCGHeap);

      RecordElem := DAGCell;
   end CreateRecordAccess;


   -- Changed to use a stack to do a depth-first search,
   -- and altered from a function to a procedure to allow
   -- debugging side effects
   procedure ContainsQuantIdent (DataElem, QuantIdent : in     Cells.Cell;
                                 Result               :    out Boolean)
   --# global in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives Result,
   --#         VCGHeap               from DataElem,
   --#                                    QuantIdent,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    DataElem,
   --#                                    QuantIdent,
   --#                                    VCGHeap;
   is
      CurrElem : Cells.Cell;
      FoundQuantIdent : Boolean;
      MyTempStack : CStacks.Stack;
   begin
      CurrElem := DataElem;
      FoundQuantIdent := False;
      CStacks.CreateStack (MyTempStack);
      CStacks.Push (VCGHeap, CurrElem, MyTempStack);
      while not (CStacks.IsEmpty (MyTempStack) or FoundQuantIdent)
      loop
         CStacks.PopOff (VCGHeap, MyTempStack, CurrElem);
         if Cells.Get_Kind (VCGHeap, CurrElem) =
            Cells.Field_Access_Function then
            -- a record field
            CStacks.Push (VCGHeap, RightPtr (VCGHeap, CurrElem), MyTempStack);
         elsif Cells.Get_Kind (VCGHeap, CurrElem) =
            Cells.Fixed_Var then
            -- a fixed identifier
            if Cells.Get_Symbol_Value (VCGHeap, CurrElem) =
               Cells.Get_Symbol_Value (VCGHeap, QuantIdent)
            then
               FoundQuantIdent := True;
            end if;
         elsif Cells.Get_Kind (VCGHeap, CurrElem) =
            Cells.Element_Function then
            -- An array element
            -- Look in the array name itself
            CStacks.Push (VCGHeap,
                          LeftPtr (VCGHeap, RightPtr (VCGHeap, CurrElem)),
                          MyTempStack);
            -- Look at the comma or ident
            CStacks.Push (VCGHeap,
                          Cells.Get_B_Ptr (VCGHeap,
                                         RightPtr (VCGHeap, RightPtr (VCGHeap, CurrElem))),
                          MyTempStack);
         elsif Cells.Get_Kind (VCGHeap, CurrElem) =
            Cells.Op then
            -- Explore both sides
            CStacks.Push (VCGHeap, LeftPtr (VCGHeap, CurrElem), MyTempStack);
            CStacks.Push (VCGHeap, RightPtr (VCGHeap, CurrElem), MyTempStack);
         end if;

      end loop;
      Result := FoundQuantIdent;
   end ContainsQuantIdent;


   -- Adopted to concatenate forall structures
   -- if multi-dim arrays are present
   procedure QuantifyConstraint (ThisTypeSym : Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in     OutputFile;
   --#        in out ConstraintStack;
   --#        in out QuantIdentStack;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives ConstraintStack,
   --#         QuantIdentStack,
   --#         Statistics.TableUsage,
   --#         VCGFailure,
   --#         VCGHeap               from *,
   --#                                    ConstraintStack,
   --#                                    Dictionary.Dict,
   --#                                    QuantIdentStack,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap &
   --#         SPARK_IO.FILE_SYS     from *,
   --#                                    ConstraintStack,
   --#                                    Dictionary.Dict,
   --#                                    OutputFile,
   --#                                    QuantIdentStack,
   --#                                    ThisTypeSym,
   --#                                    VCGHeap;
   is
      NonArrayStack,
      MyTempStack : CStacks.Stack;
      QuantIdent, QuantExpr,
         LastElement, DataElem,
         MyRangeConstraint : Cells.Cell;
      LoopCount : Positive;
      Emptied, QuantFound, ElemFound, FirstLoop : Boolean;
   begin
      CStacks.CreateStack (MyTempStack);
      CStacks.CreateStack (NonArrayStack);

      -- Create a range constraint for the top element
      PopOffConstraintStack (DataElem);
      CreateRangeConstraint (DataElem, ThisTypeSym, VCGHeap, MyRangeConstraint);

      -- Find the first 'element (', if it exists,
      -- and set "LastElement" to point to it
      ElemFound := False;
      LastElement := DataElem; -- to avoid flow error
      FirstLoop := True;
      loop  -- findelt
         if Cells.Get_Kind (VCGHeap, DataElem) = Cells.Element_Function then
            -- Found.  If this is the first one, mark it as such
            if not ElemFound then
               LastElement := DataElem;
            end if;
            ElemFound := True;
         end if;
         if FirstLoop then
            FirstLoop := False;
         else
            CStacks.Push (VCGHeap, DataElem, NonArrayStack);
         end if;
         Emptied := CStacks.IsEmpty (ConstraintStack);
         if not Emptied then
            PopOffConstraintStack (DataElem);
         end if;
         exit when Emptied;
      end loop;  -- findelt

      -- Should now have all of initial ConstraintStack on NAS
      -- in reverse order

      if ElemFound then
         -- There must be identifiers around since
         -- we've found an 'element ()' reference
         -- which is in 'LastElement'
         -- Push identifiers onto MyTempStack to ensure they
         -- are available in same order as array elems
         while not CStacks.IsEmpty (QuantIdentStack)
         loop
            PopOffQuantIdentStack (QuantIdent);
            CStacks.Push (VCGHeap, QuantIdent, MyTempStack);
         end loop;

         -- Now construct and compose the forall structures
         LoopCount := 1;
         while not CStacks.IsEmpty (MyTempStack)
         loop  -- compose
               -- Shuffle round the identifier onto QIS
            CStacks.PopOff (VCGHeap, MyTempStack, QuantIdent);
            PushOnQuantIdentStack (QuantIdent);

            -- Look to see if LastElement contains the given identifier
            ContainsQuantIdent (LastElement, QuantIdent, QuantFound);
            -- Create the forall structure, if appropriate
            if QuantFound then
               -- All things considered, it *should* find an ident!
               if (LoopCount = 1) then
                  -- Quantify over the range constraint
                  CreateQuantExpr (QuantExpr,
                                   QuantIdent,
                                   Dictionary.GetType (Cells.Get_Symbol_Value (VCGHeap, QuantIdent)),
                                   MyRangeConstraint);
               else
                  CreateShortQuantExpr (QuantExpr,
                                        QuantIdent,
                                        Dictionary.GetType (Cells.Get_Symbol_Value (VCGHeap, QuantIdent)));
                  ConcatQuantExpr (LastElement, QuantExpr);
               end if;
               LastElement := QuantExpr;
               -- And handle the looping
               LoopCount := LoopCount + 1;
            end if;
         end loop; -- compose
      else
         -- ElemFound not True
         LastElement := MyRangeConstraint;
      end if;

      -- Return the nonarray cells to the stack
      while not CStacks.IsEmpty (NonArrayStack)
      loop
         CStacks.PopOff (VCGHeap, NonArrayStack, DataElem);
         PushOnConstraintStack (DataElem);
      end loop;

      if not ElemFound then
         -- Put range constraint on head of CS
         PushOnConstraintStack (MyRangeConstraint);
      else
         -- Put the 'forall' on top
         PushOnConstraintStack (LastElement);
      end if;

   end QuantifyConstraint;

   -----------------------

   function SafeAdvanceVariable (NewRec : Boolean) return Dictionary.Symbol
   --# global in ConstraintStack;
   --#        in Dictionary.Dict;
   --#        in VCGHeap;
   is
      Sym : Dictionary.Symbol;
   begin
      Sym := Cells.Get_Assoc_Var (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack));
      if Sym /= Dictionary.NullSymbol then
         if NewRec then
            Sym := Dictionary.GetFirstRecordSubcomponent (Sym);
         else
            Sym := Dictionary.GetNextRecordSubcomponent (Sym);
         end if;
      end if;
      return Sym;
   end SafeAdvanceVariable;

   -----------------------

   function GetArrayIndex (TypeSym   : Dictionary.Symbol;
                           Dimension : Positive) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   --#        in InitialVariable;
   is
      Result : Dictionary.Symbol;
   begin
      -- If TypeSym is the symbol of a constrained array (sub)type then return the Dimension'th
      -- index of that type.
      -- If TypeSym is unconstrained (and this can only be true at the outermost level the quantifier we
      -- are constructing) then return the Dimension'th ParameterConstraintSymbol associated with the
      -- object InitialVariable.
      if Dictionary.IsUnconstrainedArrayType (TypeSym) then
         Result := Dictionary.GetSubprogramParameterConstraint (InitialVariable, Dimension);
      else
         Result := Dictionary.GetArrayIndex (TypeSym, Dimension);
      end if;
      return Result;
   end GetArrayIndex;

   function IsSuitableForAlways_Valid (Sym  : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      Result : Boolean := False;
      Entire : Dictionary.Symbol;
      Enc    : Dictionary.Symbol;
   begin
      if Dictionary.IsVariable (Sym) then
         Result := Dictionary.GetOwnVariableOrConstituentMode (Sym) = Dictionary.InMode;
      elsif Dictionary.IsVariableOrSubcomponent (Sym) then
         Entire := Sym;
         loop
            Enc := Dictionary.GetEnclosingObject (Entire);
            exit when Enc = Dictionary.NullSymbol;
            Entire := Enc;
            exit when not Dictionary.IsVariableOrSubcomponent (Entire);
            exit when Dictionary.IsVariable (Entire);
         end loop;
         Result := Dictionary.GetOwnVariableOrConstituentMode (Entire) = Dictionary.InMode;
      end if;
      return Result;
   end IsSuitableForAlways_Valid;

begin -- CreateStructConstraint;

   CStacks.CreateStack (QuantIdentStack);
   CStacks.CreateStack (ConstraintStack);

   InitialVariable := Cells.Get_Symbol_Value (VCGHeap, VarSym);

   TypeSym := StartTypeSym;
   -- The TOS also keeps a reference to the variable associated with
   -- the type being processed.
   Cells.Set_Assoc_Var (VCGHeap, VarSym, AssocVar);
   PushOnConstraintStack (VarSym);

   -- ConstraintStack = VarSym(I)
   loop -- down
      if DiscreteTypeWithCheck (TypeSym, Scope) then
         -- ConstraintStack = VarSym(I) -> ...
         IVarSym := Cells.Get_Assoc_Var (VCGHeap,
                                       CStacks.Top (VCGHeap, ConstraintStack));
         if IVarSym /= Dictionary.NullSymbol and then
             IsSuitableForAlways_Valid (IVarSym) then
            if Dictionary.VariableOrSubcomponentIsMarkedValid (IVarSym) then
               -- The variable associated with this type is marked
               -- always valid. Hence generate a type_always_valid(X) hypothesis

               -- ConstraintStack = VarSym(I) -> ...

               PopOffConstraintStack (TOS);                     -- TOS=variable being accessed

               -- Create type'always_valid(variable)
               CreateCellKind (TOS2, VCGHeap, Cells.Reference);   -- TOS2=type/variable
               Cells.Set_Symbol_Value (VCGHeap, TOS2, TypeSym);   -- TOS2.Symvalue = type
               Cells.Set_Assoc_Var (VCGHeap, TOS2, IVarSym);   -- TOS2.AssocVar = variable
               CStacks.Push (VCGHeap, TOS2, ConstraintStack);   -- TOS2 is 2nd arg to PushOperator
               CreateCellKind (FuncCell, VCGHeap, Cells.Attrib_Function); -- Cell for Always_Valid token
               Cells.Set_Lex_Str (VCGHeap, FuncCell, LexTokenManager.Always_Valid_Token);
               Cells.Set_Assoc_Var (VCGHeap, FuncCell, IVarSym);
               SetRightArgument (FuncCell, TOS, VCGHeap);        -- attach the target variable
               CStacks.Push (VCGHeap, FuncCell, ConstraintStack);
               PushOperator (Binary,
                             SPSymbols.apostrophe,
                             VCGHeap,
                             ConstraintStack);

            else
               -- The variable associated with this type is in fact not marked
               -- always valid. At this stage the easiest thing to do is to generate a
               -- true hypothesis instead of the in-type hypothesis
               CStacks.Pop (VCGHeap, ConstraintStack);
               CreateTrueCell (VCGHeap, TOS);
               PushOnConstraintStack (TOS);
               -- ConstraintStack = TrueCell -> ...
            end if;
         else
            -- Get the quantifiers into the dictionary
            -- and replaced types with constrainted on stack
            QuantifyConstraint (TypeSym);
            -- ConstraintStack = Constraint -> ...
         end if;
         TypeSym := Dictionary.NullSymbol;

      elsif ArrayTypeWithCheck (TypeSym, Scope) then
         -- ConstraintStack = VarSym(I) -> ...
         -- Get the number of dimensions
         Dimmax := Dictionary.GetNumberOfDimensions (TypeSym);
         PopOffConstraintStack (TOS);

         -- Create the initial "element (TOS,[i1" in ArrayElem
         CreateQuantIdent (QuantIdSym,
                           GetArrayIndex (TypeSym, 1)); -- replaces: Dictionary.GetArrayIndex (TypeSym, 1));
         CreateFixedVarCell (QuantIdent, VCGHeap, QuantIdSym);
         PushOnQuantIdentStack (QuantIdent);
         CreateArrayAccess (ArrayElem, TOS, QuantIdent);

         -- Cope with multi-dimensions
         LoopCount := 2;
         while (LoopCount <= Dimmax)
         loop -- arr
            -- Generate a new identifier and put it onto the QI stack
            CreateQuantIdent (QuantIdSym,
                              GetArrayIndex (TypeSym, LoopCount)); -- replaces: Dictionary.GetArrayIndex ...)
            CreateFixedVarCell (QuantIdent, VCGHeap, QuantIdSym);
            PushOnQuantIdentStack (QuantIdent);
            -- join onto reference
            ConcatArrayQuant (ArrayElem, QuantIdent);
            LoopCount := LoopCount + 1;
         end loop; -- arr

         -- Push the array structure back onto ConstraintStack
         PushOnConstraintStack (ArrayElem);
         TypeSym := Dictionary.GetArrayComponent (TypeSym);

      elsif RecordTypeWithCheck (TypeSym, Scope) then
         -- Come here when a record is seen for the first time
         -- ConstraintStack = VarSym(I) -> ...

         -- Changed following:
         -- RecCompType := Dictionary.GetRecordComponent (TypeSym, 1);
         -- to:
         TypeSym := Dictionary.GetRootType (TypeSym);
         RecCompType := Dictionary.CurrentSymbol (Dictionary.FirstRecordComponent (TypeSym));
         -- because latter returns the "inherit" field of an extended record and former does not.
         -- Thus, RecCompType = The first record component (NB in the type, not value, realm)

         -- Get the current variable reference, its type should be TypeSym
         IVarSym := SafeAdvanceVariable (True);
         -- Thus, IVarSym = The first record component in the value realm

         TypeSym := Dictionary.GetType (RecCompType);

         -- If the first record component denotes a tagged null record,
         -- then skip it and move onto the next component.
         if Dictionary.TypeIsRecord (TypeSym) and then
           not Dictionary.RecordHasSomeFields (TypeSym) then

            RecCompType := GetNextRecordComponent (RecCompType);
            IVarSym := SafeAdvanceVariable (False);
            TypeSym := Dictionary.GetType (RecCompType);
         end if;

         -- put in a record cell marker for this RecordComponent (type)
         -- including a ref to its subcomponent (value)
         CreateOpCell (OpCell, VCGHeap, SPSymbols.RWrecord);
         Cells.Set_Symbol_Value (VCGHeap, OpCell, RecCompType);
         Cells.Set_Assoc_Var (VCGHeap, OpCell, IVarSym);

         -- just read TOS and don't pop it off as it will be
         -- needed by the other fields of the record
         TOS := CStacks.Top (VCGHeap, ConstraintStack);
         CreateRecordAccess (RecordElem, RecCompType, TOS);

         -- GOK
         Cells.Set_Assoc_Var (VCGHeap, RecordElem, IVarSym);

         PushOnConstraintStack (OpCell);
         PushOnConstraintStack (RecordElem);
         -- ConstraintStack = RecordElem(I') -> OpCell(I') -> VarSym(I) -> ...

      else
         -- encountered a type with no check
         -- push TRUE to be able to deal with other fields of any records
         -- ConstraintStack = VarSym(I) -> ...
         CStacks.Pop (VCGHeap, ConstraintStack);
         CreateTrueCell (VCGHeap, TrueCell);
         PushOnConstraintStack (TrueCell);
         TypeSym := Dictionary.NullSymbol;
         -- ConstraintStack = TrueCell -> ...
      end if;

      --447--check for reals moved here in support of optional real RTCs
      if IsRealType (TypeSym) then
         ContainsReals := True;
      end if;

      if TypeSym = Dictionary.NullSymbol then
         exitUp := False;
         loop -- up
            -- ConstraintStack should contain at least one element
            -- at this point
            -- ConstraintStack = RES(I') -> OpCell(I') -> VarSym(I) -> ...
            --             or  = RES -> <empty>
            PopOffConstraintStack (TOS);
            -- TOS = RES
            -- ConstraintStack = OpCell -> VarSym -> ...
            --             or  = <empty>

            if CStacks.IsEmpty (ConstraintStack) then
               -- ConstraintStack had only one element left
               -- set this to Result and exit up & down loops
               Result := TOS;
               exitUp := True;

            elsif Cells.Get_Kind (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack))
               = Cells.Op and then
               Cells.Get_Op_Symbol (VCGHeap, CStacks.Top (VCGHeap, ConstraintStack))
               = SPSymbols.RWrecord
            then
               -- found a record marker cell
               -- TOS = RES
               -- ConstraintStack = OpCell(Rec) -> VarSym -> ...

               RecCompType := GetNextRecordComponent (
                                 Cells.Get_Symbol_Value (VCGHeap,
                                                 CStacks.Top (VCGHeap, ConstraintStack)));
               if RecCompType = Dictionary.NullSymbol then
                  -- all the fields of the record have been processed
                  -- remove record marker from stack
                  -- ConstraintStack = OpCell(Rec) -> VarSym -> ...
                  CStacks.Pop (VCGHeap, ConstraintStack);
                  -- remove True top-of-stack which is no longer needed
                  -- ConstraintStack = VarSym -> ...
                  CStacks.Pop (VCGHeap, ConstraintStack);

                  -- ConstraintStack = ...
                  PushOnConstraintStack (TOS);
                  -- ConstraintStack = RES -> ...
               else
                  -- process next field of record
                  -- ConstraintStack = OpCell -> VarSym -> ...
                  -- Thus, advance the ivar to the next field
                  IVarSym := SafeAdvanceVariable (False);
                  PopOffConstraintStack (OpCell);
                  -- ConstraintStack = VarSym -> ...
                  Cells.Set_Symbol_Value (VCGHeap, OpCell, RecCompType);
                  Cells.Set_Assoc_Var (VCGHeap, OpCell, IVarSym);

                  -- just read TOS and don't pop it off as it will be
                  -- needed by the other fields of the record
                  CreateRecordAccess (RecordElem, RecCompType,
                                   CStacks.Top (VCGHeap, ConstraintStack));
                  TypeSym := Dictionary.GetType (RecCompType);
                  exitUp := True; -- exit up loop and go back to down loop

                  -- Push back record marker cell
                  PushOnConstraintStack (OpCell);
                  -- ConstraintStack = OpCell -> VarSym -> ...

                  -- Push back TOS which contains constraints from
                  -- earlier fields of the record
                  PushOnConstraintStack (TOS);
                  -- ConstraintStack = RES -> OpCell -> VarSym -> ...

                  -- GOK
                  Cells.Set_Assoc_Var (VCGHeap, RecordElem, IVarSym);

                  PushOnConstraintStack (RecordElem);
                  -- ConstraintStack = RecordElem -> RES -> OpCell -> VarSym -> ...
                  --                        I                  I

               end if; -- RecCompType = Dictionary.NullSymbol
            else
               -- must have just finished dealing with a non-first
               -- field of a record
               -- ConstraintStack = OpCell(~Rec) -> VarSym -> ...
               PopOffConstraintStack (TOS2);
               -- ConstraintStack = VarSym -> ...
               Conjoin (TOS2, VCGHeap, TOS);
               PushOnConstraintStack (TOS);
               -- ConstraintStack = RES&OpCell(~Rec) -> VarSym -> ...
            end if; -- CStacks.IsEmpty (ConstraintStack)

            exit when exitUp;
         end loop; -- up

      end if; -- TypeSym = Dictionary.NullSymbol

      exit when CStacks.IsEmpty (ConstraintStack);
   end loop; -- down

   -- At this point the ConstraintStack is empty
   -- (the up loop has been exited through its first exit)
   -- and Result contains the last TOS.

   --# accept F, 501,                   Result, "Result always defined here" &
   --#        F, 602, StructConstraint, Result, "Result always defined here";
   StructConstraint := Result;
   --Debug.PrintDAG ("Result = ", Result, VCGHeap, Scope);
end CreateStructConstraint;
