-- $Id: dag-buildgraph-incorporateconstraints.adb 16041 2010-02-11 18:45:59Z Trevor Jennings $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with EStrings;

separate (DAG.BuildGraph)
procedure IncorporateConstraints (SemanticErrorInSubprogram : in Boolean)
is
   type Equalities is (UseEquals, UseImplication);

   PreConNode,
   PostConNode  : STree.SyntaxNode;

   --------------------------------

   -- returns IsRefined if we are dealing with a subprogram that has a second,
   -- refined annotation, otherwise returns IsAbstract.
   -- Note that we are interested here in whether it has a second proof annotation
   --       or a second flow annotation, either is enough to trigger refinement VCs
   -- Note: When we do package initialization constraits we may need to extend this
   --       function to handle that case as well
   function WhichAbstraction return Dictionary.Abstractions
   --# global in Dictionary.Dict;
   --#        in Scope;
   is
      Result : Dictionary.Abstractions := Dictionary.IsAbstract;
      Sym    : Dictionary.Symbol;
   begin
      Sym := Dictionary.GetRegion (Scope);
      if Dictionary.IsSubprogram (Sym) or else Dictionary.IsTaskType (Sym) then
         Result := Dictionary.GetConstraintAbstraction (Sym,
                                                        Dictionary.LocalScope (Sym));
      end if;
      return Result;
   end WhichAbstraction;

   --------------------------------

   procedure PlantPrecondition (Node                      : in STree.SyntaxNode;
                                SemanticErrorInSubprogram : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ImportConstraints;
   --#        in     LoopStack;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         Graph.Table,
   --#         Statistics.TableUsage,
   --#         VCGHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ImportConstraints,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         LexTokenManager.State     from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LoopStack,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        STree.Table,
   --#                                        VCGHeap;
   is
      DAGCell : Cells.Cell;

   begin
      Graph.SetFirstProofContext (Graph.Precondition);
      -- if there is no precondition, or the subprogram contains semantic errors, then
      -- we force the precondition to TRUE
      if SemanticErrorInSubprogram or else
        Node = STree.NullNode then
         CreateTrueCell (VCGHeap, DAGCell);
         -- move this out of if-else
      else
         -- otehrwise we traverse the predicate and build a DAG for it.
         BuildAnnotationExpnDAG (Node,
                                 Scope,
                                 False,
                                 LoopStack,
                                 VCGHeap,
                                 -- to get
                                 DAGCell);
      end if;
      if not Cells.Is_Null_Cell (ImportConstraints) then
         Conjoin (ImportConstraints, VCGHeap, DAGCell);
      end if;
      Graph.SetFirstAssertionLocn (DAGCell);
   end PlantPrecondition;

   --------------------------------

   -- SubstituteImplicit replaces each occurrence of the implicit
   -- variable (ImplicitVar) in a function return annotation (Return_Anno_DAG).
   -- by the expression of the return statement (referenced by RetExpDAG).
   -- The return annotation is updated in place.
   procedure SubstituteImplicit (Return_Anno_DAG : in out Cells.Cell;        -- the Return Annotation
                                 ImplicitVar     : in     Dictionary.Symbol; -- thing to be substituted
                                 RetExpDAG       : in     Cells.Cell)        -- the thing to substitute
   --# global in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives Statistics.TableUsage,
   --#         Return_Anno_DAG,
   --#         VCGHeap               from *,
   --#                                    ImplicitVar,
   --#                                    RetExpDAG,
   --#                                    Return_Anno_DAG,
   --#                                    VCGHeap;
   is
      P              : Cells.Cell;
      S              : CStacks.Stack;

      function IsLeaf (Node : Cells.Cell) return Boolean
      --# global in VCGHeap;
      is
      begin
         return Cells.Is_Null_Cell (RightPtr (VCGHeap, Node));
      end IsLeaf;

   begin -- SubstituteImplicit
         -- DAG traversal algorithm of D.E. Knuth, Fundamental Algorithms, p.317;
      CStacks.CreateStack (S);
      P := Return_Anno_DAG;
      loop
         loop
            exit when Cells.Is_Null_Cell (P);
            CStacks.Push (VCGHeap, P, S);
            if IsLeaf (P) then
               P := Cells.Null_Cell;
            else
               P := LeftPtr (VCGHeap, P);
            end if;
         end loop;
         exit when CStacks.IsEmpty (S);
         P := CStacks.Top (VCGHeap, S);
         CStacks.Pop (VCGHeap, S);
         if IsLeaf (P) then
            if Cells.Get_Symbol_Value (VCGHeap, P) = ImplicitVar then
               Cells.Copy_Contents (VCGHeap,
                                   RetExpDAG,
                                   P);
            end if;
            P := Cells.Null_Cell;
         else
            P := RightPtr (VCGHeap, P);
         end if;
      end loop;
      --# accept F, 31, Return_Anno_DAG,
      --#   "Return_Anno_DAG is indirectly updated via the local pointer P" &
      --#        F, 50, Return_Anno_DAG, ImplicitVar,
      --#   "Return_Anno_DAG indirectly references ImplicitVar via local pointer P" &
      --#        F, 50, Return_Anno_DAG, RetExpDAG,
      --#   "Return_Anno_DAG indirectly references RetExpDAG via local pointer P" &
      --#        F, 50, Return_Anno_DAG, VCGHeap,
      --#   "Return_Anno_DAG indirectly references VCGHeap via local pointer P" &
      --#        W, 3, "Suppress warnings on ReturnAnnoDAG";
      -- Return_Anno_DAG appears to be only an input to Ada but it is
      -- effectively a pointer to a data structure which is updated.
      pragma Warnings (Off, Return_Anno_DAG);
   end SubstituteImplicit;

   -----------------------------------------------------------------------------------
   -- In general we don't need to RT check procedure exports because any expression
   -- used to assign to them will have been checked; however, where stream variables
   -- are involved we may have rogue values returned from the environment.  To avoid
   -- a large number of unnecessary checks we need to detect the special cases where
   -- streams are involved.
   --
   -- NOTE - this might mean that VCs generated when -flow=data is active
   -- can be different from those generated when -flow=info since, in the former
   -- case, we get additional flow relations that say export E _might_ depend on a
   -- moded import I which wouldn't be the case with the (more accurate) analysis
   -- provided by -flow=info.
   -----------------------------------------------------------------------------------

   procedure CheckTypeOfExports (Abstraction : in     Dictionary.Abstractions;
                                 SubProg     : in     Dictionary.Symbol;
                                 Conjuncts   : in out Cells.Cell)
   --# global in     CommandLineData.Content;
   --#        in     OutputFile;
   --#        in     Scope;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives Conjuncts,
   --#         ContainsReals,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         VCGFailure,
   --#         VCGHeap               from *,
   --#                                    Abstraction,
   --#                                    CommandLineData.Content,
   --#                                    Conjuncts,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Scope,
   --#                                    SubProg,
   --#                                    VCGHeap &
   --#         SPARK_IO.FILE_SYS     from *,
   --#                                    Abstraction,
   --#                                    CommandLineData.Content,
   --#                                    Conjuncts,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    OutputFile,
   --#                                    Scope,
   --#                                    SubProg,
   --#                                    VCGHeap;
   is
      It        : Dictionary.Iterator;
      ExportVar : Dictionary.Symbol;
      Type_Sym  : Dictionary.Symbol;

      function ExportDependsOnStream (Export : Dictionary.Symbol) return Boolean
      --# global in Abstraction;
      --#        in Dictionary.Dict;
      --#        in SubProg;
      is
         ImportIt : Dictionary.Iterator;
         Result : Boolean := False;
      begin -- ExportDependsOnStream
         ImportIt := Dictionary.FirstDependency (Abstraction,
                                                 SubProg,
                                                 Export);
         while not Dictionary.IsNullIterator (ImportIt) loop
            if Dictionary.GetOwnVariableOrConstituentMode
               (Dictionary.CurrentSymbol (ImportIt)) =
               Dictionary.InMode
            then
               Result := True;
               exit;
            end if;

            ImportIt := Dictionary.NextSymbol (ImportIt);
         end loop;
         return Result;
      end ExportDependsOnStream;

   begin -- CheckTypeOfExports
      It := Dictionary.FirstExport (Abstraction, SubProg);
      while not Dictionary.IsNullIterator (It) loop
         ExportVar := Dictionary.CurrentSymbol (It);
         if ExportDependsOnStream (ExportVar) and then
           -- RTC only for exports which are dependent on streams
           IsDirectlyVisible (ExportVar, Scope) and then
           -- and which are visible in Ada context
           (Dictionary.GetOwnVariableOrConstituentMode (ExportVar) = Dictionary.DefaultMode) then
            -- no RTC unless stream has been assigned to some non-moded export --903

            Type_Sym := Dictionary.GetType (ExportVar);
            -- cannot plant the type constraint check directly because it
            -- refers to the postcond.  Need to conjoin it with
            -- the rest of the postcond instead
            if not Dictionary.IsPrivateType (Type_Sym, Scope) or else
              Dictionary.IsPredefinedTimeType (Type_Sym) then
               ConjoinParamConstraint (Type_Sym,
                                       ExportVar,
                                       Dictionary.NullSymbol,
                                       Conjuncts);
            end if;
         end if;

         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckTypeOfExports;

   ----------------------------------------------------------------------------

   procedure PlantPostcondition (Node                      : in STree.SyntaxNode;
                                 SubProg                   : in Dictionary.Symbol;
                                 Abstraction               : in Dictionary.Abstractions;
                                 SemanticErrorInSubprogram : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     LoopStack;
   --#        in     OutputFile;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives ContainsReals,
   --#         Dictionary.Dict,
   --#         Graph.Table,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         VCGFailure,
   --#         VCGHeap                   from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        STree.Table,
   --#                                        SubProg,
   --#                                        VCGHeap &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Node,
   --#                                        OutputFile,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        STree.Table,
   --#                                        SubProg,
   --#                                        VCGHeap;
   is
      DAGCell : Cells.Cell;
   begin
      Graph.SetProofContext (Graph.Postcondition);
      if SemanticErrorInSubprogram then
         -- If subprgoram has semantic errors then force postcondition to FALSE to help user notice problem
         CreateFalseCell (VCGHeap, DAGCell);
         Graph.SetAssertionLocn (DAGCell);
      else
         -- Otherwise build TRUE or actual postcondition
         if Node = STree.NullNode then
            CreateTrueCell (VCGHeap, DAGCell);
            CheckTypeOfExports (Abstraction, SubProg, DAGCell);
            Graph.SetAssertionLocn (DAGCell);
         else
            BuildAnnotationExpnDAG (Node,
                                    Scope,
                                    False,
                                    LoopStack,
                                    VCGHeap,
                                    -- to get
                                    DAGCell);
            CheckTypeOfExports (Abstraction, SubProg, DAGCell);
            Graph.SetAssertionLocn (DAGCell);
         end if;
      end if;
   end PlantPostcondition;

   --------------------------------

   -- PlantReturnExpression is called to insert constraints determined from
   -- the return annotation and the return type of a function which must be
   -- proven to be satisfied the body of the function.
   -- PlantReturnExpression traverses the syntax tree for a sequnce of
   -- statements of the function subprogram to locate the
   -- (only) return statement which will be the last statement of the sequence.
   -- Once the return statement has been located the return expression subtree
   -- is obtained and is used as the basis of a new DAG node which is a
   -- translation of the return annotation.  The DAG node represents:
   --   for an explicit return annotation the expression
   --     return_expression <-> return_annotation -- for a Boolean return type
   --     return_expression  =  return_annotation -- for other return types
   --  for an implicit return annotation
   --     Each instance of the implicit_variable in the return-annotation is
   --     substituted with the return_expression.
   --  when there is no return annotation present the DAG node represents
   --  the expression "true".
   -- Once the DAG node has been associated with the appropriate expression
   -- a further constraint to check that the value of the return expression is
   -- in-type is conjoined to the DAG node.
   procedure PlantReturnExpression (ReturnType                : in Dictionary.Symbol;
                                    Node                      : in STree.SyntaxNode;
                                    SemanticErrorInSubprogram : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     DoAssumeLocalRvalues;
   --#        in     LineNmbr;
   --#        in     LoopStack;
   --#        in     OutputFile;
   --#        in     Scope;
   --#        in     StartNode;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out FlowHeap;
   --#        in out Graph.Table;
   --#        in out KindOfStackedCheck;
   --#        in out LexTokenManager.State;
   --#        in out ShortCircuitStack;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ContainsReals,
   --#         Dictionary.Dict,
   --#         FlowHeap,
   --#         KindOfStackedCheck,
   --#         ShortCircuitStack,
   --#         StmtStack.S,
   --#         VCGFailure                from *,
   --#                                        CheckStack,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        DoAssumeLocalRvalues,
   --#                                        FlowHeap,
   --#                                        Graph.Table,
   --#                                        KindOfStackedCheck,
   --#                                        LexTokenManager.State,
   --#                                        LineNmbr,
   --#                                        LoopStack,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        ShortCircuitStack,
   --#                                        StartNode,
   --#                                        StmtStack.S,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CheckStack,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        DoAssumeLocalRvalues,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        FlowHeap,
   --#                                        Graph.Table,
   --#                                        KindOfStackedCheck,
   --#                                        LexTokenManager.State,
   --#                                        LineNmbr,
   --#                                        LoopStack,
   --#                                        Node,
   --#                                        OutputFile,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        ShortCircuitStack,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        StartNode,
   --#                                        StmtStack.S,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         Graph.Table,
   --#         Statistics.TableUsage,
   --#         VCGHeap                   from *,
   --#                                        CheckStack,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        DoAssumeLocalRvalues,
   --#                                        FlowHeap,
   --#                                        Graph.Table,
   --#                                        KindOfStackedCheck,
   --#                                        LexTokenManager.State,
   --#                                        LineNmbr,
   --#                                        LoopStack,
   --#                                        Node,
   --#                                        ReturnType,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        ShortCircuitStack,
   --#                                        StartNode,
   --#                                        StmtStack.S,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         LexTokenManager.State     from *,
   --#                                        CheckStack,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        DoAssumeLocalRvalues,
   --#                                        FlowHeap,
   --#                                        Graph.Table,
   --#                                        KindOfStackedCheck,
   --#                                        LineNmbr,
   --#                                        LoopStack,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SemanticErrorInSubprogram,
   --#                                        ShortCircuitStack,
   --#                                        StartNode,
   --#                                        StmtStack.S,
   --#                                        STree.Table,
   --#                                        VCGHeap;
   is
      RetExpNode    : STree.SyntaxNode;
      RetExpDAG,
      AnnoRetExpDAG,
      PostCondDAG   : Cells.Cell;
      ImplicitVar   : Dictionary.Symbol;
      TheAnnoDAG,
      CopyRetExpDAG : Cells.Cell;

      procedure ConjoinToReturnExpr (Type_Sym : in     Dictionary.Symbol;
                                     Expr     : in     Cells.Cell;
                                     DAGCell  : in out Cells.Cell)
      --# global in     Dictionary.Dict;
      --#        in     Scope;
      --#        in out ContainsReals;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ContainsReals         from *,
      --#                                    Dictionary.Dict,
      --#                                    Type_Sym &
      --#         DAGCell,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    DAGCell,
      --#                                    Dictionary.Dict,
      --#                                    Expr,
      --#                                    Scope,
      --#                                    Type_Sym,
      --#                                    VCGHeap;
      is
         ConjunctCell : Cells.Cell;
      begin
         if DiscreteTypeWithCheck (Type_Sym, Scope) then
            CreateRangeConstraint (Expr,
                                   Type_Sym,
                                   VCGHeap,
                                   ConjunctCell);
            Conjoin (ConjunctCell, VCGHeap, DAGCell);
         end if;
         if IsRealType (Type_Sym) then
            ContainsReals := True;
         end if;
      end ConjoinToReturnExpr;

      --------------------------------------------

   begin -- PlantReturnExpression
      if SemanticErrorInSubprogram then
         -- If subprogram body has semantic errors then force postcondition to FALSE
         Graph.SetProofContext (Graph.Postcondition);
         CreateFalseCell (VCGHeap, TheAnnoDAG);
         Graph.SetAssertionLocn (TheAnnoDAG);
      else
         -- otherwise traverse return expression and construct predicate DAG
         RetExpNode := STree.Child_Node (StartNode);

         -- Now we have a sequence_of_statements which can be reduced to:
         -- sequence_of_statements statement | statement ;
         -- (See SPARK.LLA)
         -- If the sequence_of_statements is a sequence_of_statements followed by
         -- a statement then skip to the statement (which will be the final statement
         -- in the subprogram).
         if STree.SyntaxNodeType (RetExpNode) =
           SPSymbols.sequence_of_statements
         then
            RetExpNode := STree.Next_Sibling (RetExpNode);
         end if;

         -- The final statement in the subprogram should be a return statement, but we
         -- need to cater for labels because a statement can be reduced to:
         -- simple_statement | sequence_of_labels simple_statement ...
         -- (and a simple_statement can be reduced to a return_statement).

         -- The child node will either be a simple_statement or a sequence_of_labels
         RetExpNode := STree.Child_Node (RetExpNode);

         -- Skip the label(s) if present.
         if STree.SyntaxNodeType (RetExpNode) = SPSymbols.sequence_of_labels then
            RetExpNode := STree.Next_Sibling (RetExpNode);
         end if;

         -- Now we have reached the final statement in the subprogram. This should be
         -- a return statement. Locate the return expression...
         while STree.SyntaxNodeType (RetExpNode) /=
           SPSymbols.expression
         loop
            -- SEQUENCE_OF_LABELS
            RetExpNode := STree.Child_Node (RetExpNode);
         end loop;

         -- create DAG for SPARK return expression
         BuildExpnDAG (OutputFile,
                       RetExpNode,
                       Scope,
                       Scope,
                       LineNmbr,
                       True,
                       DoAssumeLocalRvalues,
                       LoopStack,
                       FlowHeap,
                       VCGHeap,
                       ContainsReals,
                       VCGFailure,
                       ShortCircuitStack,
                       CheckStack,
                       KindOfStackedCheck,
                        -- to get
                       RetExpDAG);
         UnStackRtcs (LineNmbr, VCGHeap, CheckStack, KindOfStackedCheck);
         Structures.CopyStructure (VCGHeap, RetExpDAG, CopyRetExpDAG);

         -- now deal with return annotation if there is one
         if Node /= STree.NullNode then
            if STree.SyntaxNodeType (Node) =
              SPSymbols.annotation_expression
            then -- build post condition for simple return expression
               BuildAnnotationExpnDAG (Node,
                                       Scope,
                                       False,
                                       LoopStack,
                                       VCGHeap,
                                       -- to get
                                       AnnoRetExpDAG);

               -- set two DAGs to be equal
               -- use <-> for boolean functions otherwise use =
               if Dictionary.TypeIsBoolean (ReturnType) then
                  CreateOpCell (PostCondDAG, VCGHeap, SPSymbols.is_equivalent_to);
               else
                  CreateOpCell (PostCondDAG, VCGHeap, SPSymbols.equals);
               end if;
               SetLeftArgument (PostCondDAG, RetExpDAG, VCGHeap);
               SetRightArgument (PostCondDAG, AnnoRetExpDAG, VCGHeap);
               TheAnnoDAG := PostCondDAG;

               -- move these to the end of PlantReturnExpression
               -- plant post condition
            else -- build post condition for implicit return expression
               -- get symbol of implicit return var
               ImplicitVar := Dictionary.GetImplicitReturnVariable (WhichAbstraction,
                                                                    Dictionary.GetRegion (Scope));

               -- walk predicate part of implicit return statement
               BuildAnnotationExpnDAG (STree.Next_Sibling (Node),
                                       Dictionary.LocalScope (ImplicitVar),
                                       False,
                                       LoopStack,
                                       VCGHeap,
                                       -- to get
                                       AnnoRetExpDAG);
               -- now subsitute actual return expression DAG for occurrences of
               -- implicit return variable
               SubstituteImplicit (AnnoRetExpDAG,
                                   ImplicitVar,
                                   RetExpDAG);

               TheAnnoDAG := AnnoRetExpDAG;

               -- move these to the end of PlantReturnExpression
               -- finally plant new DAG as post condition
            end if;
         else -- no return expression given so plant true
            CreateTrueCell (VCGHeap, TheAnnoDAG);
            -- do this at the end of PlantReturnExpression
         end if;

         -- We still want to do the check that the return expression is in type,
         -- in contrast to CheckTypeofExports in ModelPostCondition.
         ConjoinToReturnExpr (Dictionary.GetType (Dictionary.GetRegion (Scope)),
                              CopyRetExpDAG,
                              TheAnnoDAG);

         Graph.SetProofContext (Graph.Postcondition);
         Graph.SetAssertionLocn (TheAnnoDAG);
      end if;
   end PlantReturnExpression;

   --------------------------------

   -- Procedure, used for generation of refinement and subclass checks,  for creating
   -- required anno DAGs; created so we can deal with absence of explicit pre/post con
   -- and create a "true" cell instead
   procedure CreateAnnotationDAG (StartNode     : in     STree.SyntaxNode;
                                  Scope         : in     Dictionary.Scopes;
                                  ForceAbstract : in     Boolean;
                                  DAGRoot       :    out Cells.Cell)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LoopStack;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives DAGRoot                   from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ForceAbstract,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Scope,
   --#                                        StartNode,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ForceAbstract,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        StartNode,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         VCGHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ForceAbstract,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Scope,
   --#                                        StartNode,
   --#                                        STree.Table,
   --#                                        VCGHeap;
   is
      DAGRootLocal : Cells.Cell;
   begin
      if StartNode = STree.NullNode then
         CreateTrueCell (VCGHeap, DAGRootLocal);
      else
         BuildAnnotationExpnDAG (StartNode,
                                 Scope,
                                 ForceAbstract,
                                 LoopStack,
                                 VCGHeap,
                                 -- to get
                                 DAGRootLocal);
      end if;
      DAGRoot := DAGRootLocal;
   end CreateAnnotationDAG;

   ----------------------------------------------------------------------

   -- Performs similar function to CreateAnnotationDAG but deals with function
   -- return annotations (both explicit and implicit).  This procedure used to be
   -- local to PlantRefinementChecks but has been moved out so that it can also
   -- be used by PlantSubClassChecks.
   procedure CreateFunctionAnnotationDAG (SubProg          : in     Dictionary.Symbol;
                                          StartNode        : in     STree.SyntaxNode;
                                          Scope            : in     Dictionary.Scopes;
                                          ForceAbstract    : in     Boolean;
                                          ReturnCell       : in     Cells.Cell;
                                          EqualityOperator : in     Equalities;
                                          Abstraction      : in     Dictionary.Abstractions;
                                          DAGRoot          :    out Cells.Cell)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LoopStack;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives DAGRoot                     from Abstraction,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          EqualityOperator,
   --#                                          ForceAbstract,
   --#                                          LexTokenManager.State,
   --#                                          LoopStack,
   --#                                          ReturnCell,
   --#                                          Scope,
   --#                                          StartNode,
   --#                                          STree.Table,
   --#                                          SubProg,
   --#                                          VCGHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ForceAbstract,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        StartNode,
   --#                                        STree.Table,
   --#                                        SubProg,
   --#                                        VCGHeap &
   --#         LexTokenManager.State     from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ForceAbstract,
   --#                                        LoopStack,
   --#                                        Scope,
   --#                                        StartNode,
   --#                                        STree.Table,
   --#                                        SubProg,
   --#                                        VCGHeap &
   --#         Statistics.TableUsage,
   --#         VCGHeap                   from *,
   --#                                        Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EqualityOperator,
   --#                                        ForceAbstract,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        ReturnCell,
   --#                                        Scope,
   --#                                        StartNode,
   --#                                        STree.Table,
   --#                                        SubProg,
   --#                                        VCGHeap;
   is
      DAGRootLocal,
      ExpnDAGRoot       : Cells.Cell;
      ImplicitReturnVar : Dictionary.Symbol;
   begin -- CreateFunctionAnnotationDAG
      if StartNode = STree.NullNode then
         CreateTrueCell (VCGHeap, DAGRootLocal);
      elsif STree.SyntaxNodeType (StartNode) =
        SPSymbols.annotation_expression
      then -- build post condition for simple return expression
         CreateAnnotationDAG (StartNode,
                              Scope,
                              ForceAbstract,
                              ExpnDAGRoot);
         -- now we need to equate the ReturnCell with the ExpnDAG
         -- first cet up the desired operator
         if EqualityOperator = UseEquals then
            CreateOpCell (DAGRootLocal, VCGHeap, SPSymbols.equals);
         else
            CreateOpCell (DAGRootLocal, VCGHeap, SPSymbols.is_equivalent_to);
         end if;
         -- and build expression
         SetLeftArgument (DAGRootLocal, ReturnCell, VCGHeap);
         SetRightArgument (DAGRootLocal, ExpnDAGRoot, VCGHeap);

      else -- build post condition for implicit return expression
         ImplicitReturnVar := Dictionary.GetImplicitReturnVariable (Abstraction,
                                                                    SubProg);
         CreateAnnotationDAG (STree.Next_Sibling (StartNode),
                              Dictionary.LocalScope (ImplicitReturnVar),
                              ForceAbstract,
                              DAGRootLocal);
         -- substitute ReturnCell for all location in the above expression where
         -- the original annotation had an implciit return variable
         SubstituteImplicit (DAGRootLocal,
                             ImplicitReturnVar,
                             ReturnCell);
      end if;
      DAGRoot := DAGRootLocal;
   end CreateFunctionAnnotationDAG;

   ----------------------------------

   procedure PlantRefinementChecks
   --# global in     CommandLineData.Content;
   --#        in     ImportConstraints;
   --#        in     LoopStack;
   --#        in     OutputFile;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives ContainsReals,
   --#         Dictionary.Dict,
   --#         Graph.Table,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         VCGFailure,
   --#         VCGHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ImportConstraints,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ImportConstraints,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         SPARK_IO.FILE_SYS         from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ImportConstraints,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        OutputFile,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        VCGHeap;
   is
      SubProg            : Dictionary.Symbol;

      PreCheck,
      AbstractPre,
      RefinedPre,
      PreCheckHyp        : Cells.Cell;
      CommonHypotheses   : Cells.Cell;
      DataRefinement     : Cells.Cell;

      Data_View          : Dictionary.Abstractions;

      ----------------------------------------------------------------------

      procedure CreateDataRefinement (SubProg   : in     Dictionary.Symbol;
                                      InPostCon : in     Boolean;
                                      TheDAG    :    out Cells.Cell)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out LexTokenManager.State;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives LexTokenManager.State from *,
      --#                                    CommandLineData.Content,
      --#                                    Dictionary.Dict,
      --#                                    SubProg &
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    CommandLineData.Content,
      --#                                    Dictionary.Dict,
      --#                                    InPostCon,
      --#                                    LexTokenManager.State,
      --#                                    SubProg,
      --#                                    VCGHeap &
      --#         TheDAG                from CommandLineData.Content,
      --#                                    Dictionary.Dict,
      --#                                    InPostCon,
      --#                                    LexTokenManager.State,
      --#                                    SubProg,
      --#                                    VCGHeap;
      is
         type Passes is (ProcessGlobals, ProcessParameters);
         pragma Unreferenced (ProcessParameters);
         It         : Dictionary.Iterator;
         AbstractIt : Dictionary.Iterator;
         AbstractSym,
         CurrentSym : Dictionary.Symbol;
         FirstItem  : Boolean := True;
         DAGroot,
         DAGlocal   : Cells.Cell;

         ---------------------------

         procedure BuildOneRefinement (RefinedSym : in     Dictionary.Symbol;
                                       InPostCon,
                                       IsImport,
                                       IsExport   : in     Boolean;
                                       SubProg    : in     Dictionary.Symbol;
                                       TheDAG     :    out Cells.Cell)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in out LexTokenManager.State;
         --#        in out Statistics.TableUsage;
         --#        in out VCGHeap;
         --# derives LexTokenManager.State from *,
         --#                                    CommandLineData.Content,
         --#                                    Dictionary.Dict,
         --#                                    RefinedSym,
         --#                                    SubProg &
         --#         Statistics.TableUsage,
         --#         VCGHeap               from *,
         --#                                    CommandLineData.Content,
         --#                                    Dictionary.Dict,
         --#                                    InPostCon,
         --#                                    IsExport,
         --#                                    IsImport,
         --#                                    LexTokenManager.State,
         --#                                    RefinedSym,
         --#                                    SubProg,
         --#                                    VCGHeap &
         --#         TheDAG                from CommandLineData.Content,
         --#                                    Dictionary.Dict,
         --#                                    InPostCon,
         --#                                    IsExport,
         --#                                    IsImport,
         --#                                    LexTokenManager.State,
         --#                                    RefinedSym,
         --#                                    SubProg,
         --#                                    VCGHeap;

        is
            TildeVersion,
            OperatorCell,
            FldCell,
            ConcreteCell,
            AbstractCell : Cells.Cell;
            FieldName    : LexTokenManager.Lex_String;

            procedure BuildFieldName
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     RefinedSym;
            --#        in     SubProg;
            --#        in out LexTokenManager.State;
            --#           out FieldName;
            --# derives FieldName,
            --#         LexTokenManager.State from CommandLineData.Content,
            --#                                    Dictionary.Dict,
            --#                                    LexTokenManager.State,
            --#                                    RefinedSym,
            --#                                    SubProg;
            is
               Prefix : EStrings.T;
            begin -- BuildFieldName
               Dictionary.GetAnyPrefixNeeded (
                  RefinedSym,
                  Dictionary.LocalScope (
                     Dictionary.GetEnclosingPackage (
                        Dictionary.GetScope (SubProg))),
                  "__",
                  Prefix);
               if EStrings.Get_Length (E_Str => Prefix) > 0 then
                  EStrings.Append_String (E_Str => Prefix,
                                          Str   => "__");
               end if;
               EStrings.Append_Examiner_String
                 (E_Str1 => Prefix,
                  E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => Dictionary.GetSimpleName (RefinedSym)));
               LexTokenManager.Insert_Examiner_String (Str     => Prefix,
                                                       Lex_Str => FieldName);
            end BuildFieldName;

            ------------------------

         begin  -- BuildOneRefinement
            -- first define the operator to use
            CreateOpCell (OperatorCell, VCGHeap, SPSymbols.equals);
            if Dictionary.TypeIsBoolean (Dictionary.GetType (RefinedSym)) then
               Cells.Set_Op_Symbol (VCGHeap, OperatorCell, SPSymbols.is_equivalent_to);
            end if;

            -- now create lhs
            CreateReferenceCell (ConcreteCell, VCGHeap, RefinedSym);

            -- now rhs
            CreateCellKind (FldCell, VCGHeap, Cells.Field_Access_Function);
            Cells.Set_Symbol_Value (VCGHeap, FldCell, RefinedSym);
            BuildFieldName;
            Cells.Set_Lex_Str (VCGHeap, FldCell, FieldName);

            CreateReferenceCell (AbstractCell, VCGHeap, Dictionary.GetSubject (RefinedSym));

            SetRightArgument (FldCell, AbstractCell, VCGHeap);

            -- assemble
            SetLeftArgument (OperatorCell, ConcreteCell, VCGHeap);
            SetRightArgument (OperatorCell, FldCell, VCGHeap);

            -- mark as initial values if needed
            if InPostCon then
               -- might need some tildes else definitely don't
               if IsImport then
                  -- must be either a pure import or and import/export either of which
                  -- need a tilded version of some sort.  Whatever form this takes, the
                  -- abstract side will be tilded
                  Structures.CopyStructure (VCGHeap, OperatorCell, TildeVersion);
                  SetTilde (AbstractCell, VCGHeap);
                  if IsExport then
                     -- item is both imported and exported so we need to tilde both sides
                     SetTilde (ConcreteCell, VCGHeap);
                  end if; -- IsExport
                  Conjoin (TildeVersion, VCGHeap, OperatorCell);
               end if; -- IsImport
            end if; -- InPostCondition

            -- return
            TheDAG := OperatorCell;
         end BuildOneRefinement;

         -------------

         function RequiresRefinementRelation (Sym : Dictionary.Symbol) return Boolean
         --# global in Dictionary.Dict;
         --#        in SubProg;
         is
         begin
            return Dictionary.IsConstituent (Sym) and then
               Dictionary.GetOwner (Dictionary.GetSubject (Sym)) =
               Dictionary.GetEnclosingPackage (Dictionary.GetScope (SubProg));
         end RequiresRefinementRelation;

         -------------

         function IsAbstractOwnVariableOfThisPackage (Sym : Dictionary.Symbol) return Boolean
         --# global in Dictionary.Dict;
         --#        in SubProg;
         is
         begin
            return Dictionary.IsAbstractOwnVariable (Sym) and then
              (Dictionary.GetOwner (Sym) =
                 Dictionary.GetEnclosingPackage (Dictionary.GetScope (SubProg)));
         end IsAbstractOwnVariableOfThisPackage;

      begin -- CreateDataRefinement
         DAGroot := Cells.Null_Cell;
         for I in Passes
         loop
            if I = ProcessGlobals then
               It := Dictionary.FirstGlobalVariable (Dictionary.IsRefined,
                                                     SubProg);
            else
               It := Dictionary.FirstSubprogramParameter (SubProg);
            end if;

            while not Dictionary.IsNullIterator (It)
            loop
               CurrentSym := Dictionary.CurrentSymbol (It);

               -- we are interested if CurrentSym is a refinement constituent of a
               -- locally-declared abstract own variable
               if RequiresRefinementRelation (CurrentSym) then

                  BuildOneRefinement (CurrentSym,
                                      InPostCon,
                                      Dictionary.IsImport (Dictionary.IsRefined,
                                                           SubProg,
                                                           CurrentSym),
                                      Dictionary.IsExport (Dictionary.IsRefined,
                                                           SubProg,
                                                           CurrentSym),
                                      SubProg,
                                       -- to get
                                      DAGlocal);
                  if FirstItem then
                     DAGroot := DAGlocal;
                     FirstItem := False;
                  else
                     Conjoin (DAGlocal, VCGHeap, DAGroot);
                  end if;
               end if;
               It := Dictionary.NextSymbol (It);
            end loop;
         end loop;

         -- now loop through all constituents of the abstract own variables involved and
         -- treat their constituents as imports (i.e. read but unchanged) if they are not
         -- referenced at all by the subprogram.  Clearly if they are not referenced at all
         -- then they must be unchanged.  We need hypotheses to this effect to complete the
         -- data refinement

         AbstractIt := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract,
                                                       SubProg);
         while not Dictionary.IsNullIterator (AbstractIt)
         loop
            AbstractSym := Dictionary.CurrentSymbol (AbstractIt);

            if IsAbstractOwnVariableOfThisPackage (AbstractSym) then

               It := Dictionary.FirstConstituent (AbstractSym);
               while not Dictionary.IsNullIterator (It)
               loop
                  CurrentSym := Dictionary.CurrentSymbol (It);

                  -- we only need to do something if this constituent is neither an import or an
                  -- export in the refined view, otherwise it will already have been handled by
                  -- code above
                  if not (Dictionary.IsImport (Dictionary.IsRefined,
                                               SubProg,
                                               CurrentSym) or
                            Dictionary.IsExport (Dictionary.IsRefined,
                                                 SubProg,
                                                 CurrentSym)) then

                     BuildOneRefinement (CurrentSym,
                                         InPostCon,
                                         True,  -- treat as import
                                         False, -- but not export
                                         SubProg,
                                          -- to get
                                         DAGlocal);
                     if FirstItem then
                        DAGroot := DAGlocal;
                        FirstItem := False;
                     else
                        Conjoin (DAGlocal, VCGHeap, DAGroot);
                     end if;
                  end if;
                  It := Dictionary.NextSymbol (It);
               end loop;
            end if;

            AbstractIt := Dictionary.NextSymbol (AbstractIt);
         end loop;

         TheDAG := DAGroot;
      end CreateDataRefinement;

      ----------------------------------------------------------------------

      -- procedure for use in creating both procedure and function
      -- refinement post condition checks; this procedure creates the
      -- things which are common to all cases such as twiddled pre cons
      procedure CreateCommonHypotheses
        (Data_View        : in     Dictionary.Abstractions;
         AbstractPre      : in     Cells.Cell;
         RefinedPre       : in     Cells.Cell;
         CommonHypotheses :    out Cells.Cell)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ImportConstraints;
      --#        in     SubProg;
      --#        in out LexTokenManager.State;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives CommonHypotheses      from AbstractPre,
      --#                                    CommandLineData.Content,
      --#                                    RefinedPre,
      --#                                    Data_View,
      --#                                    Dictionary.Dict,
      --#                                    ImportConstraints,
      --#                                    LexTokenManager.State,
      --#                                    SubProg,
      --#                                    VCGHeap &
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    AbstractPre,
      --#                                    CommandLineData.Content,
      --#                                    RefinedPre,
      --#                                    Data_View,
      --#                                    Dictionary.Dict,
      --#                                    ImportConstraints,
      --#                                    LexTokenManager.State,
      --#                                    SubProg,
      --#                                    VCGHeap &
      --#         LexTokenManager.State from *,
      --#                                    CommandLineData.Content,
      --#                                    Data_View,
      --#                                    Dictionary.Dict,
      --#                                    SubProg;
      is
         CopyOfImportConstraints,
         CopyOfAbstractPre,
         CopyOfRefinedPre,
         DataRefinement            : Cells.Cell;
      begin
         Structures.CopyStructure (VCGHeap, AbstractPre, CopyOfAbstractPre);
         Structures.CopyStructure (VCGHeap, RefinedPre, CopyOfRefinedPre);

         -- The two precondition components of the hypothesis need any
         -- import/exports to be twiddled.
         -- Note that the refined precondition may have an abstract or refined
         -- data view depending on whether Data Refinement is present.
         SubstituteTwiddled (Dictionary.IsAbstract, SubProg, CopyOfAbstractPre);
         SubstituteTwiddled (Data_View, SubProg, CopyOfRefinedPre);

         -- Now assemble common parts of hypotheses
         Conjoin (CopyOfRefinedPre, VCGHeap, CopyOfAbstractPre);

         if Data_View = Dictionary.IsRefined then
            -- Data refinement is present and so add it to the common hypotheses.
            CreateDataRefinement (SubProg,
                                  True,       -- in post condition
                                  -- to get
                                  DataRefinement);
            Conjoin (DataRefinement, VCGHeap, CopyOfAbstractPre);
         end if;

         -- Obtain a copy of the import constraints for use in the refinement VCs.
         Structures.CopyStructure (VCGHeap,
                                   ImportConstraints,
                                   CopyOfImportConstraints);

         -- Add in import type information if it exists
         -- Note that import types may have an abstract or refined
         -- data view depending on whether Data Refinement is present.
         if not Cells.Is_Null_Cell (CopyOfImportConstraints) then
            SubstituteTwiddled (Data_View, SubProg, CopyOfImportConstraints);
            Conjoin (CopyOfImportConstraints, VCGHeap, CopyOfAbstractPre);
         end if;

         -- and return result
         CommonHypotheses := CopyOfAbstractPre;
      end CreateCommonHypotheses;

      ----------------------------------------------------------------------

      procedure ProcedureRefinementPostCheck
        (Data_View        : in Dictionary.Abstractions;
         CommonHypotheses : in Cells.Cell)
      --# global in     CommandLineData.Content;
      --#        in     LoopStack;
      --#        in     OutputFile;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in     SubProg;
      --#        in out ContainsReals;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Graph.Table;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out VCGFailure;
      --#        in out VCGHeap;
      --# derives ContainsReals,
      --#         Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         VCGFailure                from *,
      --#                                        CommandLineData.Content,
      --#                                        Data_View,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        LoopStack,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        SubProg,
      --#                                        VCGHeap &
      --#         ErrorHandler.ErrorContext from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        LoopStack,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        SubProg,
      --#                                        VCGHeap &
      --#         Graph.Table,
      --#         Statistics.TableUsage,
      --#         VCGHeap                   from *,
      --#                                        CommandLineData.Content,
      --#                                        CommonHypotheses,
      --#                                        Data_View,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        LoopStack,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        SubProg,
      --#                                        VCGHeap &
      --#         SPARK_IO.FILE_SYS         from *,
      --#                                        CommandLineData.Content,
      --#                                        Data_View,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        LoopStack,
      --#                                        OutputFile,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        SubProg,
      --#                                        VCGHeap;
      is
         PostCheck,
         PostHypotheses,
         AbstractPost,
         RefinedPost,
         ExportConstraints : Cells.Cell;

         procedure GetExportConstraints (Data_View : in Dictionary.Abstractions)
         --# global in     CommandLineData.Content;
         --#        in     OutputFile;
         --#        in     Scope;
         --#        in     SubProg;
         --#        in out ContainsReals;
         --#        in out Dictionary.Dict;
         --#        in out LexTokenManager.State;
         --#        in out SPARK_IO.FILE_SYS;
         --#        in out Statistics.TableUsage;
         --#        in out VCGFailure;
         --#        in out VCGHeap;
         --#           out ExportConstraints;
         --# derives ContainsReals,
         --#         Dictionary.Dict,
         --#         LexTokenManager.State,
         --#         Statistics.TableUsage,
         --#         VCGFailure,
         --#         VCGHeap               from *,
         --#                                    CommandLineData.Content,
         --#                                    Data_View,
         --#                                    Dictionary.Dict,
         --#                                    LexTokenManager.State,
         --#                                    Scope,
         --#                                    SubProg,
         --#                                    VCGHeap &
         --#         ExportConstraints     from CommandLineData.Content,
         --#                                    Data_View,
         --#                                    Dictionary.Dict,
         --#                                    LexTokenManager.State,
         --#                                    Scope,
         --#                                    SubProg,
         --#                                    VCGHeap &
         --#         SPARK_IO.FILE_SYS     from *,
         --#                                    CommandLineData.Content,
         --#                                    Data_View,
         --#                                    Dictionary.Dict,
         --#                                    LexTokenManager.State,
         --#                                    OutputFile,
         --#                                    Scope,
         --#                                    SubProg,
         --#                                    VCGHeap;
         is
            ExportIt : Dictionary.Iterator;
            VarSym,
            TypeSym  : Dictionary.Symbol;
         begin -- GetExportConstraints
            ExportConstraints := Cells.Null_Cell;
            -- The data view of the Exports may be Abstract or Refined
            -- dependent on the use of Data Refinement.
            ExportIt := Dictionary.FirstExport (Data_View,
                                                SubProg);
            while not Dictionary.IsNullIterator (ExportIt)
            loop
               VarSym := Dictionary.CurrentSymbol (ExportIt);
               if IsDirectlyVisible (VarSym, Scope) then
                  TypeSym := Dictionary.GetType (VarSym);

                  if not Dictionary.IsPrivateType (TypeSym, Scope) or else
                         Dictionary.IsPredefinedTimeType (TypeSym) then

                     ConjoinParamConstraint (TypeSym,
                                             VarSym,
                                             Dictionary.NullSymbol,
                                             ExportConstraints);
                  end if;
               end if;
               ExportIt := Dictionary.NextSymbol (ExportIt);
            end loop;
         end GetExportConstraints;

      begin -- ProcedureRefinementPostCheck
         CreateAnnotationDAG (STree.RefToNode
                              (Dictionary.GetPostcondition (Dictionary.IsAbstract, SubProg)),
                              Dictionary.LocalScope (SubProg),
                              True,
                              -- to get
                              AbstractPost);

         CreateAnnotationDAG (STree.RefToNode
                              (Dictionary.GetPostcondition (Dictionary.IsRefined, SubProg)),
                              Dictionary.LocalScope (SubProg),
                              False,
                              -- to get
                              RefinedPost);

         PostHypotheses := CommonHypotheses;

         -- build in info that exports must be in type
         GetExportConstraints (Data_View);
         if not Cells.Is_Null_Cell (ExportConstraints) then
            Conjoin (ExportConstraints, VCGHeap, PostHypotheses);
         end if;
         Conjoin (RefinedPost, VCGHeap, PostHypotheses);

         Cells.Create_Cell (VCGHeap, PostCheck);

         SetRightArgument (PostCheck, PostHypotheses, VCGHeap);     -- set up hypothesis
         SetAuxPtr (PostCheck, AbstractPost, VCGHeap);       -- set up conclusion
         Graph.SetRefinementPostCheck (PostCheck);           -- pass it to graph
      end ProcedureRefinementPostCheck;

      ----------------------------------------------------------------------

      procedure FunctionRefinementPostCheck (CommonHypotheses : in Cells.Cell)
      --# global in     CommandLineData.Content;
      --#        in     LoopStack;
      --#        in     OutputFile;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in     SubProg;
      --#        in out ContainsReals;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out Graph.Table;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out VCGFailure;
      --#        in out VCGHeap;
      --# derives ContainsReals,
      --#         Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         VCGFailure                from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        LoopStack,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        SubProg,
      --#                                        VCGHeap &
      --#         ErrorHandler.ErrorContext from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        LoopStack,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        SubProg,
      --#                                        VCGHeap &
      --#         Graph.Table,
      --#         Statistics.TableUsage,
      --#         VCGHeap                   from *,
      --#                                        CommandLineData.Content,
      --#                                        CommonHypotheses,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.State,
      --#                                        LoopStack,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        SubProg,
      --#                                        VCGHeap &
      --#         SPARK_IO.FILE_SYS         from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.State,
      --#                                        LoopStack,
      --#                                        OutputFile,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        SubProg,
      --#                                        VCGHeap;
      is

         ReturnType         : Dictionary.Symbol;
         ReturnCell         : Cells.Cell;
         EqualityOperator   : Equalities;

         ExportConstraint,
         PostCheck,
         PostHypotheses,
         AbstractPost,
         RefinedPost        : Cells.Cell;

      begin -- FunctionRefinementPostCheck
            -- identify function return type
         ReturnType := Dictionary.GetType (SubProg);

         -- create a ReturnVar cell containing the function return type
         CreateCellKind (ReturnCell, VCGHeap, Cells.Return_Var);
         Cells.Set_Symbol_Value (VCGHeap,
                               ReturnCell,
                               ReturnType);

         -- for explicit returns we need to equate the ReturnVar to the returned expression
         -- so we need to choose the appropriate operator
         if Dictionary.TypeIsBoolean (ReturnType) then
            EqualityOperator := UseImplication;
         else
            EqualityOperator := UseEquals;
         end if;

         -- now build abstract and refined "post conditions"
         CreateFunctionAnnotationDAG
           (SubProg,
            STree.RefToNode
              (Dictionary.GetPostcondition (Dictionary.IsAbstract, SubProg)),
            Dictionary.LocalScope (SubProg),
            True,
            ReturnCell,
            EqualityOperator,
            Dictionary.IsAbstract,
            -- to get
            AbstractPost);

         CreateFunctionAnnotationDAG
           (SubProg,
            STree.RefToNode
              (Dictionary.GetPostcondition (Dictionary.IsRefined, SubProg)),
            Dictionary.LocalScope (SubProg),
            False,
            ReturnCell,
            EqualityOperator,
            Dictionary.IsRefined,
            -- to get
            RefinedPost);

         PostHypotheses := CommonHypotheses;

         -- add in constraint that returned valued must be in type
         CreateStructConstraint (OutputFile,
                                 ReturnType,
                                 ReturnCell,
                                 Scope,
                                 Dictionary.NullSymbol,
                                 VCGHeap,
                                 ContainsReals,
                                 VCGFailure,
                                 -- to get
                                 ExportConstraint);
         Conjoin (ExportConstraint, VCGHeap, PostHypotheses);
         Conjoin (RefinedPost, VCGHeap, PostHypotheses);

         Cells.Create_Cell (VCGHeap, PostCheck);

         SetRightArgument (PostCheck, PostHypotheses, VCGHeap);     -- set up hypothesis
         SetAuxPtr (PostCheck, AbstractPost, VCGHeap);       -- set up conclusion
         Graph.SetRefinementPostCheck (PostCheck);           -- pass it to graph

      end FunctionRefinementPostCheck;

      ----------------------------------------------------------------------

   begin -- PlantRefinementChecks
      SubProg := Dictionary.GetRegion (Scope);

      -- first create VC: abstract pre --> refined pre
      CreateAnnotationDAG (STree.RefToNode
                           (Dictionary.GetPrecondition (Dictionary.IsAbstract, SubProg)),
                           Dictionary.LocalScope (SubProg),
                           True,
                           -- to get
                           PreCheckHyp);
      Structures.CopyStructure (VCGHeap, PreCheckHyp, AbstractPre);


      -- Determine whether the view of the data should be
      -- Abstract (Proof Refinement only) or
      -- Refined  (Data Refinement is present).
      CreateDataRefinement (SubProg,
                            False,       -- not in post condition
                              -- to get
                            DataRefinement);
      if Cells.Is_Null_Cell (DataRefinement) then

         -- No Data Refinement.
         Data_View := Dictionary.IsAbstract;
      else
         -- Data Refinement is present.
         Data_View := Dictionary.IsRefined;
      end if;

      if Data_View = Dictionary.IsRefined then
         -- Data Refinement is present and so add it in to the hypotheses
         Conjoin (DataRefinement, VCGHeap, PreCheckHyp);
      end if;

      if not Cells.Is_Null_Cell (ImportConstraints) then
         -- add in import type information to hypotheses if these exist
         Conjoin (ImportConstraints, VCGHeap, PreCheckHyp);
      end if;

      CreateAnnotationDAG (STree.RefToNode
                           (Dictionary.GetPrecondition (Dictionary.IsRefined, SubProg)),
                           Dictionary.LocalScope (SubProg),
                           False,
                           -- to get
                           RefinedPre);

      Cells.Create_Cell (VCGHeap, PreCheck);
      SetRightArgument (PreCheck, PreCheckHyp, VCGHeap);    -- set up hypothesis
      SetAuxPtr (PreCheck, RefinedPre, VCGHeap);     -- set up conclusion
      Graph.SetRefinementPreCheck (PreCheck);        -- pass to graph

      -- now create VC: refined post /\ refined pre /\ abstract pre --> abstract post

      CreateCommonHypotheses (Data_View, AbstractPre, RefinedPre, CommonHypotheses);

      if Dictionary.IsFunction (Dictionary.GetRegion (Scope)) then
         FunctionRefinementPostCheck (CommonHypotheses);
      else
         ProcedureRefinementPostCheck (Data_View, CommonHypotheses);
      end if;
   end PlantRefinementChecks;

   --------------------------------

   procedure PlantSubClassChecks
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LoopStack;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        VCGHeap &
   --#         Graph.Table,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         VCGHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.State,
   --#                                        LoopStack,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        VCGHeap;
   is
      SubProg,
      RootSubProg   : Dictionary.Symbol;
      PreCell,
      PostCell,
      RootPre,
      RootPost,
      ExtPre,
      ExtPost : Cells.Cell;
      ReturnType         : Dictionary.Symbol;
      ReturnCell         : Cells.Cell;
      EqualityOperator   : Equalities;

      procedure SubstituteParameters (Subprog,
                                      RootSubprog        : in Dictionary.Symbol;
                                      RootConstraintRoot : in Cells.Cell)
      --# global in     Dictionary.Dict;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    Dictionary.Dict,
      --#                                    RootConstraintRoot,
      --#                                    RootSubprog,
      --#                                    SubProg,
      --#                                    VCGHeap;
      is
         P,
         NewP           : Cells.Cell;
         S              : CStacks.Stack;
         VarSym         : Dictionary.Symbol;

         function IsLeaf (Node : Cells.Cell) return Boolean
         --# global in VCGHeap;
         is
         begin
            return Cells.Is_Null_Cell (RightPtr (VCGHeap, Node));
         end IsLeaf;

--       procedure InheritDereference (ParamSym, Subprog : in     Symbol;
--                                     NewCell           :    out Cells.Cell)
--       --# global VCGHeap,
--       --# Dictionary.Dict,
--       --# Statistics.TableUsage;
--       --# derives NewCell               from ParamSym,
--       --#                                    SubProg,
--       --#                                    -
--       --#         VCGHeap,
--       --#         Statistics.TableUsage from
--         from
--         is
--

      begin -- SubstituteParameters
            -- DAG traversal algorithm of D.E. Knuth, Fundamental
            -- Algorithms, p.317;

         CStacks.CreateStack (S);
         P := RootConstraintRoot;
         loop
            loop
               exit when Cells.Is_Null_Cell (P);
               CStacks.Push (VCGHeap, P, S);
               if IsLeaf (P) then
                  P := Cells.Null_Cell;
               else
                  P := LeftPtr (VCGHeap, P);
               end if;
            end loop;
            exit when CStacks.IsEmpty (S);
            P := CStacks.Top (VCGHeap, S);
            CStacks.Pop (VCGHeap, S);
            if IsLeaf (P) then
               if Cells.Get_Kind (VCGHeap, P) = Cells.Reference then
                  VarSym := Cells.Get_Symbol_Value (VCGHeap, P);
                  if Dictionary.IsSubprogramParameter (VarSym) then
                     -- Substitute name.
                     Cells.Set_Symbol_Value (VCGHeap,
                                           P,
                                           Dictionary.GetSubprogramParameter
                                             (Subprog,
                                              Dictionary.GetSubprogramParameterNumber (VarSym)));
                     if Dictionary.GetType (VarSym) = Dictionary.GetSubprogramControllingType (RootSubprog) then
                        -- Substitute an inherit dereference to the parameter used in Subprog (because we are
                        -- constructing VC in terms of SubProg not RootSubProg so references to the controlling
                        -- parameter of RootSubProg must be replaced by fld_inherit ("controlling param of SubProg")
                        Cells.Create_Cell (VCGHeap, NewP);
                        Cells.Copy_Contents (VCGHeap, P, NewP);
                        InsertInheritDeReference (Cells.Get_Symbol_Value (VCGHeap, P),
                                                  VCGHeap,
                                                  NewP);
                        Cells.Copy_Contents (VCGHeap, NewP, P);
                     end if;
                  end if;
               end if;
               P := Cells.Null_Cell;
            else
               P := RightPtr (VCGHeap, P);
            end if;
         end loop;
      end SubstituteParameters;


   begin  -- PlantSubClassChecks
      SubProg := Dictionary.GetRegion (Scope);
      if Dictionary.IsSubprogram (SubProg) then
         RootSubProg := Dictionary.GetOverriddenSubprogram (SubProg);
         if RootSubProg /= Dictionary.NullSymbol then
            -- subclass integrity checks required

            -- first do preconditon check: abstract root pre -> abstract extended pre
            -- create abstract root pre
            CreateAnnotationDAG (STree.RefToNode
                                   (Dictionary.GetPrecondition (Dictionary.IsAbstract, RootSubProg)),
                                 Dictionary.LocalScope (RootSubProg),
                                 True,
                                 -- to get
                                 RootPre);
            -- change parameter names in root op to match those in extended op
            SubstituteParameters (SubProg, RootSubProg, RootPre);

            -- create abstract extended pre
            CreateAnnotationDAG (STree.RefToNode
                                   (Dictionary.GetPrecondition (Dictionary.IsAbstract, SubProg)),
                                 Dictionary.LocalScope (SubProg),
                                 True,
                                 -- to get
                                 ExtPre);
            -- assemble check
            Cells.Create_Cell (VCGHeap, PreCell);
            SetRightArgument (PreCell, RootPre, VCGHeap);   -- set up hypothesis
            SetAuxPtr (PreCell, ExtPre, VCGHeap);           -- set up conclusion
            Graph.SetSubclassPreCheck (PreCell);   -- pass to graph

            -- go on to do post check here
            if Dictionary.IsFunction (SubProg) then
               ReturnType := Dictionary.GetType (SubProg);
               -- create a ReturnVar cell containing the function return type
               CreateCellKind (ReturnCell, VCGHeap, Cells.Return_Var);
               Cells.Set_Symbol_Value (VCGHeap,
                                     ReturnCell,
                                     ReturnType);

               -- for explicit returns we need to equate the ReturnVar to the returned expression
               -- so we need to choose the appropriate operator
               if Dictionary.TypeIsBoolean (ReturnType) then
                  EqualityOperator := UseImplication;
               else
                  EqualityOperator := UseEquals;
               end if;
               -- now create "postcondition" DAGs
               CreateFunctionAnnotationDAG
                 (SubProg,
                  STree.RefToNode
                    (Dictionary.GetPostcondition (Dictionary.IsAbstract, SubProg)),
                  Dictionary.LocalScope (SubProg),
                  True,
                  ReturnCell,
                  EqualityOperator,
                  Dictionary.IsAbstract,
                  -- to get
                  ExtPost);

               CreateFunctionAnnotationDAG
                 (RootSubProg,
                  STree.RefToNode
                    (Dictionary.GetPostcondition (Dictionary.IsAbstract, RootSubProg)),
                  Dictionary.LocalScope (RootSubProg),
                  True,
                  ReturnCell,
                  EqualityOperator,
                  Dictionary.IsAbstract,
                  -- to get
                  RootPost);

            else -- procedure
               -- create abstract extended post
               CreateAnnotationDAG (STree.RefToNode
                                      (Dictionary.GetPostcondition (Dictionary.IsAbstract, SubProg)),
                                    Dictionary.LocalScope (SubProg),
                                    True,
                                    -- to get
                                    ExtPost);

               -- create abstract root post
               CreateAnnotationDAG (STree.RefToNode
                                      (Dictionary.GetPostcondition (Dictionary.IsAbstract, RootSubProg)),
                                    Dictionary.LocalScope (RootSubProg),
                                    True,
                                 -- to get
                                    RootPost);
            end if;
            SubstituteParameters (SubProg, RootSubProg, RootPost);
            -- assemble check
            Cells.Create_Cell (VCGHeap, PostCell);
            SetRightArgument (PostCell, ExtPost, VCGHeap);   -- set up hypothesis
            SetAuxPtr (PostCell, RootPost, VCGHeap);         -- set up conclusion
            Graph.SetSubclassPostCheck (PostCell);  -- pass to graph
         end if;
      end if;
   end PlantSubClassChecks;

   --------------------------------

begin -- IncorporateConstraints

   -- first do Pre Condition if Subprogram
   if Dictionary.IsSubprogram (Dictionary.GetRegion (Scope)) or else
     Dictionary.IsTaskType (Dictionary.GetRegion (Scope)) then
      PreConNode := STree.RefToNode
         (Dictionary.GetPrecondition (WhichAbstraction,
                                      Dictionary.GetRegion (Scope)));
      PlantPrecondition (PreConNode,
                         SemanticErrorInSubprogram);

      if not SemanticErrorInSubprogram then
         -- Only create these VCs in the absence of semantic errors in the subprgoram body
         if WhichAbstraction = Dictionary.IsRefined then
            -- then a second annotation has been found so refinement proof is needed
            PlantRefinementChecks;
         end if;
         PlantSubClassChecks;
      end if;
   end if;

   -- then Post Condition for subprogram or package init
   PostConNode := STree.RefToNode
      (Dictionary.GetPostcondition (WhichAbstraction,
                                    Dictionary.GetRegion (Scope)));

   if Dictionary.IsFunction (Dictionary.GetRegion (Scope)) then
      PlantReturnExpression (Dictionary.GetType (Dictionary.GetRegion (Scope)),
                             PostConNode,
                             SemanticErrorInSubprogram);
   else
      PlantPostcondition (PostConNode,
                          Dictionary.GetRegion (Scope),
                          WhichAbstraction,
                          SemanticErrorInSubprogram);
   end if;

end IncorporateConstraints;
