-- $Id: contextmanager-ops.adb 16256 2010-03-04 14:17:05Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with CommandLineData;
with SystemErrors;
with SPSymbols;
with IndexManager;

use type SPSymbols.SPSymbol;

package body ContextManager.Ops
is

   type FileEntries is record
      Name            : LexTokenManager.Lex_String;
      Status          : ContextManager.FileStatus;
      File            : SPARK_IO.File_Type;
      ListingReq      : Boolean;
      ListingFileName : EStrings.T;
      LineContext     : SparkLex.LineContext;
      ErrorContext    : ErrorHandler.ErrorContexts;
      UnitCount       : Natural;
      ErrsReported    : Boolean;
   end record;

   NullFileEntry : constant FileEntries :=
     FileEntries'(Name            => LexTokenManager.Null_String,
                  Status          => ContextManager.NoFileEntry,
                  File            => SPARK_IO.Null_File,
                  ListingReq      => False,
                  ListingFileName => EStrings.Empty_String,
                  LineContext     => SparkLex.NullLineContext,
                  ErrorContext    => ErrorHandler.NullErrorContext,
                  UnitCount       => Natural'First,
                  ErrsReported    => False);

   type UnitEntries is record
      Name           : LexTokenLists.Lists;
      UnitType       : ContextManager.UnitTypes;
      Status         : ContextManager.UnitStatus;
      FileDescriptor : ContextManager.FileDescriptors;
      ParseTree      : STree.SyntaxNode;
      VCG            : Boolean;
      UnitNumber     : Natural;
      CycleDetected  : Boolean;
      Comp_Unit_Flag : Natural;
      --  Should use a record with discriminant :
      --  type Comp_Unit_T (Flag : Boolean) is record
      --     case Flag is
      --        when False => null;
      --        when True  => Decl_Line_Number : Positive;
      --     end case;
      --  end record;
      Inherit_Clause : STree.SyntaxNode;
   end record;

   NullUnitEntry : constant UnitEntries :=
      UnitEntries'(Name           => LexTokenLists.Null_List,
                   UnitType       => ContextManager.PackageSpecification,
                   Status         => ContextManager.NoUnitEntry,
                   FileDescriptor => ContextManager.NullFile,
                   ParseTree      => STree.NullNode,
                   VCG            => False,
                   UnitNumber     => 0,
                   CycleDetected  => False,
                   Comp_Unit_Flag => 0,
                   Inherit_Clause => STree.NullNode);

   subtype FilePointers is ContextManager.FileDescriptors range 1 .. ContextManager.FileDescriptors'Last;

   type FileHeapContents is array (FilePointers) of FileEntries;

   type FileHeaps is record
      Content  : FileHeapContents;
      LastUsed : ContextManager.FileDescriptors;
   end record;

   subtype UnitPointers is ContextManager.UnitDescriptors range 1 .. ContextManager.UnitDescriptors'Last;

   type UnitHeapContents is array (UnitPointers) of UnitEntries;

   type UnitHeaps is record
      Content  : UnitHeapContents;
      LastUsed : ContextManager.UnitDescriptors;
   end record;

   subtype StackHeights is Integer range
      0 .. ExaminerConstants.ContextManagerMaxUnits;

   subtype StackPointers is Integer range
      1 .. ExaminerConstants.ContextManagerMaxUnits;

   type StackContents is array (StackPointers) of ContextManager.UnitDescriptors;

   type UnitStacks is record
      Content : StackContents;
      Height  : StackHeights;
   end record;

   FileHeap  : FileHeaps;
   UnitHeap  : UnitHeaps;
   UnitStack : UnitStacks;

   function CurrentUnit return ContextManager.UnitDescriptors
   is
      Result : ContextManager.UnitDescriptors;
   begin
      if UnitStack.Height > 0 then
         Result := UnitStack.Content (UnitStack.Height);
      else
         Result := ContextManager.NullUnit;
      end if;
      return Result;
   end CurrentUnit;

   function GetUnitStatus (Descriptor : ContextManager.UnitDescriptors) return ContextManager.UnitStatus
   is
   begin
      return UnitHeap.Content (Descriptor).Status;
   end GetUnitStatus;

   procedure CreateUnitDescriptor (Descriptor : out ContextManager.UnitDescriptors)
   is
   begin
      if UnitHeap.LastUsed = UnitPointers'Last then
         SystemErrors.FatalError (SystemErrors.ContextUnitHeapOverflow, "");
      end if;
      UnitHeap.LastUsed := UnitHeap.LastUsed + 1;
      Descriptor := UnitHeap.LastUsed;
      UnitHeap.Content (UnitHeap.LastUsed) := NullUnitEntry;
   end CreateUnitDescriptor;

   procedure SetFileDescriptor (UnitDescriptor : in ContextManager.UnitDescriptors;
                                FileDescriptor : in ContextManager.FileDescriptors)
   is
   begin
      UnitHeap.Content (UnitDescriptor).FileDescriptor := FileDescriptor;
   end SetFileDescriptor;

   procedure GetFileDescriptor (UnitDescriptor : in     ContextManager.UnitDescriptors;
                                FileDescriptor :    out ContextManager.FileDescriptors)
   is
   begin
      FileDescriptor := UnitHeap.Content (UnitDescriptor).FileDescriptor;
   end GetFileDescriptor;

   procedure SetUnitStatus (Descriptor : in ContextManager.UnitDescriptors;
                            Status     : in ContextManager.UnitStatus)
   is
   begin
      UnitHeap.Content (Descriptor).Status := Status;
   end SetUnitStatus;

   procedure IncUnitCount (Descriptor : in ContextManager.FileDescriptors)
   is
   begin
      FileHeap.Content (Descriptor).UnitCount :=
         FileHeap.Content (Descriptor).UnitCount + 1;
   end IncUnitCount;

   procedure GetUnitCount (Descriptor : in     ContextManager.FileDescriptors;
                           Count      :    out Natural)
   is
   begin
      Count := FileHeap.Content (Descriptor).UnitCount;
   end GetUnitCount;

   procedure SetUnitNumber (Descriptor : in ContextManager.UnitDescriptors;
                            Count      : in Natural)
   is
   begin
      UnitHeap.Content (Descriptor).UnitNumber := Count;
   end SetUnitNumber;

   function GetUnitNumber (Descriptor : ContextManager.UnitDescriptors) return Natural
   is
   begin
      return UnitHeap.Content (Descriptor).UnitNumber;
   end GetUnitNumber;

   procedure MarkUnitInCycle (Descriptor : in ContextManager.UnitDescriptors)
   is
   begin
      UnitHeap.Content (Descriptor).CycleDetected := True;
   end MarkUnitInCycle;

   function UnitInCycle (Descriptor : ContextManager.UnitDescriptors) return Boolean
   is
   begin
      return UnitHeap.Content (Descriptor).CycleDetected;
   end UnitInCycle;

   function GetFileStatus (Descriptor : ContextManager.FileDescriptors) return ContextManager.FileStatus
   is
   begin
      return FileHeap.Content (Descriptor).Status;
   end GetFileStatus;

   procedure SetVCG (Descriptor : in ContextManager.UnitDescriptors;
                     VCG        : in Boolean)
   is
   begin
      UnitHeap.Content (Descriptor).VCG := VCG;
   end SetVCG;

   procedure GetVCG (Descriptor : in     ContextManager.UnitDescriptors;
                     VCG        :    out Boolean)
   is
   begin
      VCG := UnitHeap.Content (Descriptor).VCG;
   end GetVCG;

   procedure GetUnitByName (UnitName    : in     LexTokenLists.Lists;
                            UnitTypeSet : in     ContextManager.UnitTypeSets;
                            Descriptor  :    out ContextManager.UnitDescriptors)
   is
   begin
      Descriptor := ContextManager.NullUnit;
      for I in UnitPointers range 1 .. UnitHeap.LastUsed loop
         if LexTokenLists.Eq_Unit (First_Item => UnitName,
                                   Second     => UnitHeap.Content (I).Name) and
            UnitTypeSet (UnitHeap.Content (I).UnitType)
         then
            Descriptor := I;
            exit;
         end if;
      end loop;
   end GetUnitByName;

   procedure SetUnitName (Descriptor : in ContextManager.UnitDescriptors;
                          UnitName   : in LexTokenLists.Lists;
                          UnitType   : in ContextManager.UnitTypes)
   is
   begin
      UnitHeap.Content (Descriptor).Name := UnitName;
      UnitHeap.Content (Descriptor).UnitType := UnitType;
   end SetUnitName;

   procedure GetUnitName (Descriptor : in     ContextManager.UnitDescriptors;
                          UnitName   :    out LexTokenLists.Lists;
                          UnitType   :    out ContextManager.UnitTypes)
   is
   begin
      UnitName := UnitHeap.Content (Descriptor).Name;
      UnitType := UnitHeap.Content (Descriptor).UnitType;
   end GetUnitName;

   procedure SetParseTree (Descriptor : in ContextManager.UnitDescriptors;
                           ParseTree  : in STree.SyntaxNode)
   is
   begin
      UnitHeap.Content (Descriptor).ParseTree      := ParseTree;
      UnitHeap.Content (Descriptor).Inherit_Clause := STree.GetNode (It => STree.FindFirstNode (NodeKind    => SPSymbols.inherit_clause,
                                                                                                FromRoot    => ParseTree,
                                                                                                InDirection => STree.Down));
   end SetParseTree;

   procedure GetParseTree (Descriptor : in     ContextManager.UnitDescriptors;
                           ParseTree  :    out STree.SyntaxNode)
   is
   begin
      ParseTree := UnitHeap.Content (Descriptor).ParseTree;
   end GetParseTree;

   function FirstUnitDescriptor return ContextManager.UnitDescriptors
   is
      Result : ContextManager.UnitDescriptors;
   begin
      if UnitHeap.LastUsed = ContextManager.NullUnit then
         Result := ContextManager.NullUnit;
      else
         Result := UnitPointers'First;
      end if;
      return Result;
   end FirstUnitDescriptor;

   function NextUnitDescriptor (Descriptor : ContextManager.UnitDescriptors)
                               return ContextManager.UnitDescriptors
   is
      Result : ContextManager.UnitDescriptors;
   begin
      if Descriptor = UnitHeap.LastUsed then
         Result :=  ContextManager.NullUnit;
      else
         Result :=  Descriptor + 1;
      end if;
      return Result;
   end NextUnitDescriptor;

   procedure PushUnit (Descriptor : in ContextManager.UnitDescriptors)
   is
   begin
      if UnitStack.Height = StackHeights'Last then
         SystemErrors.FatalError (SystemErrors.ContextUnitStackOverflow, "");
      end if;

      UnitStack.Height := UnitStack.Height + 1;
      UnitStack.Content (UnitStack.Height) := Descriptor;
   end PushUnit;

   procedure PopUnit (Descriptor : out ContextManager.UnitDescriptors)
   is
   begin
      if UnitStack.Height = 0 then
         SystemErrors.FatalError (SystemErrors.ContextUnitStackUnderflow, "");
      end if;

      Descriptor := UnitStack.Content (UnitStack.Height);
      UnitStack.Height := UnitStack.Height - 1;
   end PopUnit;

   procedure CreateFileDescriptor (Descriptor : out ContextManager.FileDescriptors)
   is
   begin
      if FileHeap.LastUsed = FilePointers'Last then
         SystemErrors.FatalError (SystemErrors.ContextFileHeapOverflow, "");
      end if;
      FileHeap.LastUsed := FileHeap.LastUsed + 1;
      Descriptor := FileHeap.LastUsed;
      FileHeap.Content (FileHeap.LastUsed).Status := ContextManager.NoFileEntry;
      FileHeap.Content (FileHeap.LastUsed).UnitCount := 0;
      FileHeap.Content (FileHeap.LastUsed).ErrsReported := False;
   end CreateFileDescriptor;

   procedure SetSourceFileName (Descriptor     : in ContextManager.FileDescriptors;
                                SourceFileName : in LexTokenManager.Lex_String)
   is
   begin
      FileHeap.Content (Descriptor).Name := SourceFileName;
   end SetSourceFileName;

   function GetSourceFileName (Descriptor : in ContextManager.FileDescriptors) return LexTokenManager.Lex_String
   is
   begin
      return FileHeap.Content (Descriptor).Name;
   end GetSourceFileName;

   procedure SetSourceFile (Descriptor : in ContextManager.FileDescriptors;
                            SourceFile : in SPARK_IO.File_Type)
   is
   begin
      FileHeap.Content (Descriptor).File := SourceFile;
   end SetSourceFile;

   procedure GetSourceFile (Descriptor : in     ContextManager.FileDescriptors;
                            SourceFile :    out SPARK_IO.File_Type)
   is
   begin
      SourceFile := FileHeap.Content (Descriptor).File;
   end GetSourceFile;

   function ListingReqt (Descriptor : ContextManager.FileDescriptors) return Boolean
   is
   begin
      return FileHeap.Content (Descriptor).ListingReq;
   end ListingReqt;

   function FirstFileDescriptor return ContextManager.FileDescriptors
   is
      Result : ContextManager.FileDescriptors;
   begin
      if FileHeap.LastUsed = ContextManager.NullFile then
         Result := ContextManager.NullFile;
      else
         Result := FilePointers'First;
      end if;
      return Result;
   end FirstFileDescriptor;

   function NextFileDescriptor (Descriptor : ContextManager.FileDescriptors)
                               return ContextManager.FileDescriptors
   is
      Result : ContextManager.FileDescriptors;
   begin
      if Descriptor = FileHeap.LastUsed then
         Result := ContextManager.NullFile;
      else
         Result := Descriptor + 1;
      end if;
      return Result;
   end NextFileDescriptor;

   function GetFileByName (FileName : in LexTokenManager.Lex_String) return ContextManager.FileDescriptors
   is
      Descriptor : ContextManager.FileDescriptors;
   begin
      Descriptor := ContextManager.NullFile;
      for I in FilePointers range 1 .. FileHeap.LastUsed loop
         if LexTokenManager.Lex_String_Case_Sensitive_Compare
           (Lex_Str1         => FileName,
            Lex_Str2         => FileHeap.Content (I).Name) = LexTokenManager.Str_Eq then
            Descriptor := I;
            exit;
         end if;
      end loop;
      return Descriptor;
   end GetFileByName;

   procedure SetFileStatus (Descriptor : in ContextManager.FileDescriptors;
                            Status     : in ContextManager.FileStatus)
   is
   begin
      FileHeap.Content (Descriptor).Status := Status;
   end SetFileStatus;

   procedure SetListingReq (Descriptor : in ContextManager.FileDescriptors;
                            Req        : in Boolean)
   is
   begin
      FileHeap.Content (Descriptor).ListingReq := Req;
   end SetListingReq;

   procedure SetLineContext (Descriptor  : in ContextManager.FileDescriptors;
                             FileContext : in SparkLex.LineContext)
   is
   begin
      FileHeap.Content (Descriptor).LineContext := FileContext;
   end SetLineContext;

   procedure GetLineContext (Descriptor  : in     ContextManager.FileDescriptors;
                             FileContext :    out SparkLex.LineContext)
   is
   begin
      FileContext := FileHeap.Content (Descriptor).LineContext;
   end GetLineContext;

   procedure SetErrorContext (Descriptor : in ContextManager.FileDescriptors;
                              Context    : in ErrorHandler.ErrorContexts)
   is
   begin
      FileHeap.Content (Descriptor).ErrorContext := Context;
   end SetErrorContext;

   procedure GetErrorContext (Descriptor : in     ContextManager.FileDescriptors;
                              Context    :    out ErrorHandler.ErrorContexts)
   is
   begin
      Context := FileHeap.Content (Descriptor).ErrorContext;
   end GetErrorContext;

   procedure SetListingFileName (Descriptor      : in ContextManager.FileDescriptors;
                                 ListingFileName : in EStrings.T)
   is
   begin
      FileHeap.Content (Descriptor).ListingFileName := ListingFileName;
   end SetListingFileName;

   procedure GetListingFileName (Descriptor      : in     ContextManager.FileDescriptors;
                                 ListingFileName :    out EStrings.T)
   is
   begin
      ListingFileName := FileHeap.Content (Descriptor).ListingFileName;
   end GetListingFileName;

   procedure SetErrorsReported (Descriptor : in ContextManager.FileDescriptors)
   is
   begin
      FileHeap.Content (Descriptor).ErrsReported := True;
   end SetErrorsReported;

   function ErrorsReported (Descriptor : ContextManager.FileDescriptors) return Boolean
   is
   begin
      return FileHeap.Content (Descriptor).ErrsReported;
   end ErrorsReported;

   procedure Get_Unit (Descriptor      : in     ContextManager.FileDescriptors;
                       Unit_Descriptor :    out ContextManager.UnitDescriptors) is
      Id_Str : LexTokenManager.Lex_String;
   begin
      Unit_Descriptor := ContextManager.NullUnit;
      for I in UnitPointers range 1 .. UnitHeap.LastUsed loop
         if UnitHeap.Content (I).UnitType /= ContextManager.InterUnitPragma and then
           UnitHeap.Content (I).FileDescriptor = Descriptor then
            if Unit_Descriptor = ContextManager.NullUnit then
               Unit_Descriptor := I;
            else
               if LexTokenLists.Get_Length (List => UnitHeap.Content (I).Name) = 0 then
                  Id_Str := LexTokenManager.Null_String;
               else
                  Id_Str := LexTokenLists.Get_Element (List => UnitHeap.Content (I).Name,
                                                       Pos  => LexTokenLists.Get_Length (List => UnitHeap.Content (I).Name));
               end if;
               ErrorHandler.SLI_Generation_Warning
                 (Position => STree.NodePosition
                    (Node => STree.GetNode (It => STree.FindFirstNode (NodeKind    => SPSymbols.identifier,
                                                                       FromRoot    => UnitHeap.Content (I).ParseTree,
                                                                       InDirection => STree.Down))),
                  Id_Str   => Id_Str);
            end if;
         end if;
      end loop;
   end Get_Unit;

   procedure Get_Parent (Unit_Descriptor : in out ContextManager.UnitDescriptors)
   is
      Unit_Name  : LexTokenLists.Lists;
      Dummy_Item : LexTokenManager.Lex_String;
   begin
      Unit_Name := UnitHeap.Content (Unit_Descriptor).Name;
      --# accept F, 10, Dummy_Item, "Ineffective assignment here OK";
      LexTokenLists.Pop (List => Unit_Name,
                         Item => Dummy_Item);
      --# end accept;
      GetUnitByName (UnitName    => Unit_Name,
                     UnitTypeSet => ContextManager.UnitTypeSets'
                       (ContextManager.SubUnit | ContextManager.PackageBody | ContextManager.MainProgram => True,
                        others                                                                           => False),
                     Descriptor  => Unit_Descriptor);
      --# accept F, 33, Dummy_Item, "Expect Dummy_Item unused";
   end Get_Parent;

   procedure Dependency_Closure (Descriptor : in ContextManager.FileDescriptors) is
      It              : STree.Iterator;
      Lex_Str         : LexTokenLists.Lists;
      Unit_Descriptor : ContextManager.UnitDescriptors;
      Spec_Found      : Boolean;
      Components      : IndexManager.ComponentLists;

      Queue_Size : constant := ExaminerConstants.ContextManagerMaxUnits;

      subtype Queue_0 is Integer range 0 .. Queue_Size;
      subtype Queue_1 is Queue_0 range 1 .. Queue_0'Last;

      type Queue_Item is record
         Unit_Descriptor : ContextManager.UnitDescriptors;
         Done            : Boolean;
      end record;

      type Queue_Array is array (Queue_1) of Queue_Item;
      type Queue_T is record
         The_Array : Queue_Array;
         Top       : Queue_0;
      end record;

      Queue : Queue_T;

      --  Build a string list from a dotted name identifier (Node).
      function Build_List (Node : in STree.SyntaxNode) return LexTokenLists.Lists
      --# global in STree.Table;
      is
         It         : STree.Iterator;
         Return_Val : LexTokenLists.Lists;
      begin
         SystemErrors.RTAssert (C      => STree.SyntaxNodeType (Node) = SPSymbols.dotted_simple_name,
                                SysErr => SystemErrors.OtherInternalError,
                                Msg    => "CONTEXTMANAGER.BUILD_LIST : Node should be a SPSymbols.dotted_simple_name");
         Return_Val := LexTokenLists.Null_List;
         It := STree.FindFirstNode (NodeKind    => SPSymbols.identifier,
                                    FromRoot    => Node,
                                    InDirection => STree.Down);
         while It /= STree.NullIterator loop
            LexTokenLists.Append (List => Return_Val,
                                  Item => STree.NodeLexString (Node => STree.GetNode (It => It)));
            It := STree.NextNode (It => It);
         end loop;
         return Return_Val;
      end Build_List;

      --  Add the compilation unit descriptor (Descriptor) in the
      --  queue.
      procedure Add_Queue (Descriptor : in ContextManager.UnitDescriptors)
      --# global in out Queue;
      --#        in out UnitHeap;
      --# derives Queue,
      --#         UnitHeap from *,
      --#                       Descriptor,
      --#                       Queue;
      is
         Found : Boolean;
      begin
         --  Find if the compilation unit descriptor (Descriptor) has
         --  already been in the queue or is already in the queue.
         Found := False;
         for I in Queue_1 range 1 .. Queue.Top loop
            if Queue.The_Array (I).Unit_Descriptor = Descriptor then
               Found := True;
               exit;
            end if;
         end loop;
         if not Found then
            --  Never seen the compilation unit descriptor
            --  (Descriptor) in the queue => add the compilation unit
            --  descriptor (Descriptor) in the queue.
            if Queue.Top < Queue_Size then
               Queue.Top := Queue.Top + 1;
               Queue.The_Array (Queue.Top) := Queue_Item'(Unit_Descriptor => Descriptor,
                                                          Done            => False);
               --  Set the closure flag.
               UnitHeap.Content (Descriptor).Comp_Unit_Flag := 1;
            else
               SystemErrors.FatalError (SysErr => SystemErrors.QueueOverflow,
                                        Msg    => "CONTEXTMANAGER.ADD_QUEUE : Queue full");
            end if;
         end if;
      end Add_Queue;

      --  Get and remove the next compilation unit descriptor
      --  (Unit_Descriptor) from the queue.
      procedure Get_Next (Unit_Descriptor : out ContextManager.UnitDescriptors)
      --# global in out Queue;
      --# derives Queue,
      --#         Unit_Descriptor from Queue;
      is
      begin
         Unit_Descriptor := ContextManager.NullUnit;
         for I in Queue_1 range 1 .. Queue.Top loop
            if not Queue.The_Array (I).Done then
               Queue.The_Array (I).Done := True;
               Unit_Descriptor := Queue.The_Array (I).Unit_Descriptor;
               exit;
            end if;
         end loop;
      end Get_Next;

   begin
      --  Reset the closure flag.
      for I in UnitPointers range 1 .. UnitHeap.LastUsed loop
         UnitHeap.Content (I).Comp_Unit_Flag := 0;
      end loop;
      --  Initiate the closure calculation.
      Get_Unit (Descriptor      => Descriptor,
                Unit_Descriptor => Unit_Descriptor);
      Queue := Queue_T'(The_Array => Queue_Array'
                          (others => Queue_Item'(Unit_Descriptor => ContextManager.NullUnit,
                                                 Done            => True)),
                        Top       => 0);
      Spec_Found := False;
      while not Spec_Found loop
         if UnitHeap.Content (Unit_Descriptor).UnitType = ContextManager.PackageBody then
            --  Set the closure flag.
            UnitHeap.Content (Unit_Descriptor).Comp_Unit_Flag := 1;
            --  It is an Ada package body.
            --  Find the specification of the Unit_Pointer_Body.
            for I in UnitPointers range 1 .. UnitHeap.LastUsed loop
               if LexTokenLists.Eq_Unit (First_Item => UnitHeap.Content (I).Name,
                                         Second     => UnitHeap.Content (Unit_Descriptor).Name) and then
                 UnitHeap.Content (I).UnitType = ContextManager.PackageSpecification then
                  Add_Queue (Descriptor => I);
                  exit;
               end if;
            end loop;
            Spec_Found := True;
         elsif UnitHeap.Content (Unit_Descriptor).UnitType = ContextManager.PackageSpecification or
           UnitHeap.Content (Unit_Descriptor).UnitType = ContextManager.MainProgram then
            --  It is an Ada package specification or an Ada main
            --  program.
            Add_Queue (Descriptor => Unit_Descriptor);
            Spec_Found := True;
         elsif UnitHeap.Content (Unit_Descriptor).UnitType = ContextManager.SubUnit then
            --  Set the closure flag.
            UnitHeap.Content (Unit_Descriptor).Comp_Unit_Flag := 1;
            --  It is an Ada separate unit
            Get_Parent (Unit_Descriptor => Unit_Descriptor);
            Spec_Found := False;
         else
            Spec_Found := False;
            SystemErrors.FatalError (SysErr => SystemErrors.OtherInternalError,
                                     Msg    => "CONTEXTMANAGER.DEPENDENCY_CLOSURE PROGRAM ERROR");
         end if;
      end loop;
      --  Add the private childs units to the closure (if any).
      IndexManager.LookUpComponents (RequiredUnit => UnitHeap.Content (Unit_Descriptor).Name,
                                     Components   => Components);
      for I in IndexManager.ComponentIndex loop
         exit when Components (I) = LexTokenLists.Null_List;
         for J in UnitPointers range 1 .. UnitHeap.LastUsed loop
            if LexTokenLists.Eq_Unit (First_Item => UnitHeap.Content (J).Name,
                                      Second     => Components (I)) and then
              UnitHeap.Content (J).UnitType = ContextManager.PackageSpecification then
               Add_Queue (Descriptor => J);
               exit;
            end if;
         end loop;
      end loop;
      --  Calculate the closure.
      Get_Next (Unit_Descriptor => Unit_Descriptor);
      while Unit_Descriptor /= ContextManager.NullUnit loop
         It := STree.FindFirstNode (NodeKind    => SPSymbols.dotted_simple_name,
                                    FromRoot    => UnitHeap.Content (Unit_Descriptor).Inherit_Clause,
                                    InDirection => STree.Down);
         while It /= STree.NullIterator loop
            Lex_Str := Build_List (Node => STree.GetNode (It => It));
            GetUnitByName (Lex_Str, ContextManager.PackageSpecificationSet, Unit_Descriptor);
            if Unit_Descriptor /= ContextManager.NullUnit then
               Add_Queue (Descriptor => Unit_Descriptor);
            end if;
            It := STree.NextNode (It => It);
         end loop;
         Get_Next (Unit_Descriptor => Unit_Descriptor);
      end loop;
      if CommandLineData.Content.Debug.SLI then
         --  Debug
         SPARK_IO.Put_Line (File => SPARK_IO.Standard_Output,
                            Item => "DEBUG DEPENDENCY CLOSURE",
                            Stop => 0);
         for I in UnitPointers range 1 .. UnitHeap.LastUsed loop
            if UnitHeap.Content (I).Comp_Unit_Flag /= 0 then
               SPARK_IO.Put_String (File => SPARK_IO.Standard_Output,
                                    Item => "COMPILATION UNIT = ",
                                    Stop => 0);
               LexTokenLists.Print_List (File => SPARK_IO.Standard_Output,
                                         List => UnitHeap.Content (I).Name);
               if UnitHeap.Content (I).UnitType = ContextManager.PackageSpecification or
                 UnitHeap.Content (I).UnitType = ContextManager.GenericDeclaration then
                  SPARK_IO.Put_Line (File => SPARK_IO.Standard_Output,
                                     Item => " SPEC",
                                     Stop => 0);
               elsif UnitHeap.Content (I).UnitType = ContextManager.PackageBody or
                 UnitHeap.Content (I).UnitType = ContextManager.SubUnit or
                 UnitHeap.Content (I).UnitType = ContextManager.MainProgram then
                  SPARK_IO.Put_Line (File => SPARK_IO.Standard_Output,
                                     Item => " BODY",
                                     Stop => 0);
               else
                  SPARK_IO.Put_Line (File => SPARK_IO.Standard_Output,
                                     Item => " UNKNOWN",
                                     Stop => 0);
               end if;
            end if;
         end loop;
      end if;
   end Dependency_Closure;

   function In_Closure (Descriptor : in ContextManager.UnitDescriptors) return Boolean is
   begin
      return UnitHeap.Content (Descriptor).Comp_Unit_Flag /= 0;
   end In_Closure;

   procedure Set_Line_Number (Descriptor  : in ContextManager.UnitDescriptors;
                              Line_Number : in Positive) is
   begin
      UnitHeap.Content (Descriptor).Comp_Unit_Flag := Line_Number;
   end Set_Line_Number;

   function Get_Line_Number (Descriptor  : in ContextManager.UnitDescriptors) return Natural is
   begin
      return UnitHeap.Content (Descriptor).Comp_Unit_Flag;
   end Get_Line_Number;

begin
   UnitHeap := UnitHeaps'(Content  => UnitHeapContents'(others => NullUnitEntry),
                          LastUsed => 0);

   UnitStack := UnitStacks'(Content => StackContents'(others => ContextManager.NullUnit),
                            Height  => StackHeights'First);

   --# accept F, 23, FileHeap.Content, "Partial initialization OK here";
   FileHeap.LastUsed := 0;
   FileHeap.Content (FilePointers'First) := NullFileEntry;

   --# accept F, 602, FileHeap, FileHeap.Content, "Partial initialization OK here";
end ContextManager.Ops;
