-- $Id: commandlinedata.ads 15826 2010-01-28 18:09:51Z spark $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with EStrings;
with ELStrings;
with ExaminerConstants;

--# inherit EStrings,
--#         ELStrings,
--#         ExaminerConstants,
--#         FileSystem,
--#         SPARK_IO,
--#         XMLReport;

package CommandLineData
--# own Content : CommandLineContents;
is
   Meta_File_Extension       : constant String := "smf";
   Default_Index_Extension   : constant String := "idx";
   Default_Warning_Extension : constant String := "wrn";
   Default_Data_Extension    : constant String := "dat";
   Default_Config_Extension  : constant String := "cfg";

   type SourceFileEntry is record
      SourceFileName  : EStrings.T;
      Listing         : Boolean;
      ListingFileName : EStrings.T;
   end record;

   subtype SourceFileCounts is Integer range
      0 .. ExaminerConstants.MaxFilesOnCommandLine;
   subtype SourceFilePositions is Integer range
      0 .. ExaminerConstants.MaxFilesOnCommandLine;
   type SourceFileLists is array (SourceFilePositions) of SourceFileEntry;

   ---------------------------------------------------------------------
   -- To Add a new debug flag:
   --
   -- 1. Add a Boolean field to this record, and supply an initial value
   -- for it in the body of CommandLineData.Initialize
   --
   -- 2. Update CommandLineHandler.ParseCommandOptions to recognize
   -- and set the new flag.
   --
   -- 3. Document the new flag in the Examiner's help output in
   -- sparkmain.adb
   --
   -- 4. Document the new flag in the Examiner User Manual
   --
   -- 5. Test and use the flag to produce the output required
   ---------------------------------------------------------------------
   type DebugState is record
      Enabled       : Boolean;   -- any -debug flag is enabled.
      Expressions   : Boolean;   -- -debug=e - expression walk debugging
      HTML          : Boolean;   -- -debug=h - HTML generation debugging
      LookupTrace   : Boolean;   -- -debug=l - Dictionary look up call tracing
      FileNames     : Boolean;   -- -debug=f - Trace filename storage/open/create
      Units         : Boolean;   -- -debug=u - Trace required units and index lookup
      Invariants    : Boolean;   -- -debug=i - Print default loop invariants
      Components    : Boolean;   -- -debug=c - print state of component manager when
                                 --            adding subcomponents
      Rho           : Boolean;   -- -debug=r - Print computed Rho relation for subprograms
      Parser        : Boolean;   -- -debug=p - Print parser state on detection of syntax error
      FDL_Ranking   : Boolean;   -- -debug=k - trace ranking and printing of FDL declarations
      VCG           : Boolean;   -- -debug=v - print VCG state after DAG.BuildGraph
      VCG_All       : Boolean;   -- -debug=V - as VCG, but also print VCG state during each iteration
                                 --            of Graph.GenVCs
      DAG           : Boolean;   -- -debug=d - print FDL DAG following BuildExpnDAG
      SLI           : Boolean;   -- -debug=x - Print cross-reference debug
   end record;

   type ConcurrencyProfiles is (Sequential, Ravenscar);

   type LanguageProfiles    is (SPARK83, SPARK95, SPARK2005);

   type InfoFlowPolicies is (None, Safety, Security);

   -- A subtype of the above excluding "None"
   subtype DefinedInfoFlowPolicies is InfoFlowPolicies range Safety .. InfoFlowPolicies'Last;

   type JustificationOptions is (Ignore, Full, Brief);

   type BriefOptions is (NoPath, FullPath);

   -------------------------------------------------------------------------------------
   -- Proof rule generation policies for composite constants.  If this type
   -- changes, then the error message in CommandLineHandler.OutputError must be updated.
   --
   -- Whether a proof rule gets generated or not for a particular constant depends on
   -- the seeting of this switch, and the presence (or absence) of an object_assertion
   -- annotation for that constant.
   --
   -- The policies are as follows:
   --   NoRules  - NEVER  generate composite proof rules, regardless of annotations
   --   AllRules - ALWAYS generate composite proof rules, regardless of annotations
   --   Lazy     - Generate rules for constants where an annotation requests it.
   --              If no annotation is given then no rule is generated.
   --   Keen     - Generate rules for constants where an annotation requests it.
   --              If no annotation is given then a rule IS generated.
   -------------------------------------------------------------------------------------
   type RuleGenerationPolicies is (NoRules, Lazy, Keen, AllRules);

   type ErrorExplanations is (Off, FirstOccurrence, EveryOccurrence);

   type CommandLineContents is record
      Valid               : Boolean;

      Index               : Boolean;
      IndexFileName       : EStrings.T;

      Warning             : Boolean;
      WarningFileName     : EStrings.T;

      TargetData          : Boolean;
      TargetDataFile      : EStrings.T;

      TargetConfig        : Boolean;
      TargetConfigFile    : EStrings.T;

      SourceExtension     : EStrings.T;
      NumberSource        : SourceFileCounts;
      SourceFileList      : SourceFileLists;

      ListingExtension    : EStrings.T;
      NoListings          : Boolean;

      Report              : Boolean;
      ReportFileName      : EStrings.T;

      WriteDict           : Boolean;
      DictFileName        : EStrings.T;

      FDLreserved         : Boolean;
      FDLmangle           : EStrings.T;

      HTML                : Boolean;
      HTMLDirectory       : EStrings.T;

      OutputDirectory     : Boolean;
      OutputDirectoryName : EStrings.T;

      VCG                 : Boolean; -- Generate Verification Conditions
      DPC                 : Boolean; -- Generate Dead-Path Conjectures

      AnnoChar            : Character;

      ConcurrencyProfile  : ConcurrencyProfiles;
      LanguageProfile     : LanguageProfiles;

      InfoFlowPolicy      : InfoFlowPolicies;
      ConstantRules       : RuleGenerationPolicies;
      ErrorExplanation    : ErrorExplanations;
      JustificationOption : JustificationOptions;
      Debug               : DebugState;

      Echo                : Boolean;
      SyntaxOnly          : Boolean;
      WriteStatistics     : Boolean;
      DoInformationFlow   : Boolean;
      DefaultSwitchFile   : Boolean;
      PlainOutput         : Boolean;
      VersionRequested    : Boolean;
      HelpRequested       : Boolean;
      VCfingerprints      : Boolean;
      NoDuration          : Boolean;
      Brief               : Boolean;
      BriefOption         : BriefOptions;
      XML                 : Boolean;
      LegacyErrors        : Boolean;
      Generate_SLI        : Boolean;

      Casing_Standard     : Boolean;
      Casing_Identifier   : Boolean;
   end record;

   Option_Annotation_Character                : constant String    := "annotation_character";
   Option_Brief                               : constant String    := "brief";
   Option_Brief_No_Path                       : constant String    := "nopath";
   Option_Brief_Full_Path                     : constant String    := "fullpath";
   Option_Casing                              : constant String    := "casing";
   Option_Casing_Standard                     : constant Character := 's';
   Option_Casing_Identifier                   : constant Character := 'i';
   Option_Config_File                         : constant String    := "config_file";
   Option_Dictionary_File                     : constant String    := "dictionary_file";
   Option_Debug                               : constant String    := "debug";
   Option_Debug_C                             : constant Character := 'c';
   Option_Debug_D                             : constant Character := 'd';
   Option_Debug_E                             : constant Character := 'e';
   Option_Debug_F                             : constant Character := 'f';
   Option_Debug_H                             : constant Character := 'h';
   Option_Debug_I                             : constant Character := 'i';
   Option_Debug_K                             : constant Character := 'k';
   Option_Debug_L                             : constant Character := 'l';
   Option_Debug_P                             : constant Character := 'p';
   Option_Debug_R                             : constant Character := 'r';
   Option_Debug_U                             : constant Character := 'u';
   Option_Debug_V                             : constant Character := 'v';
   Option_Debug_V_Upper                       : constant Character := 'V';
   Option_Debug_X                             : constant Character := 'x';
   Option_Dpc                                 : constant String    := "dpc";
   Option_Error_Explanations                  : constant String    := "error_explanations";
   Option_Error_Explanations_Off              : constant String    := "off";
   Option_Error_Explanations_First_Occurrence : constant String    := "first_occurrence";
   Option_Error_Explanations_Every_Occurrence : constant String    := "every_occurrence";
   Option_Fdl_Identifiers                     : constant String    := "fdl_identifiers";
   Option_Fdl_Identifiers_Accept              : constant String    := "accept";
   Option_Fdl_Identifiers_Reject              : constant String    := "reject";
   Option_Flow_Analysis                       : constant String    := "flow_analysis";
   Option_Flow_Analysis_Information           : constant String    := "information";
   Option_Flow_Analysis_Data                  : constant String    := "data";
   Option_Html                                : constant String    := "html";
   Option_Help                                : constant String    := "help";
   Option_Index_File                          : constant String    := "index_file";
   Option_Justification_Option                : constant String    := "justification_option";
   Option_Justification_Option_Ignore         : constant String    := "ignore";
   Option_Justification_Option_Full           : constant String    := "full";
   Option_Justification_Option_Brief          : constant String    := "brief";
   Option_Listing_Extension                   : constant String    := "listing_extension";
   Option_Language                            : constant String    := "language";
   Option_Language_83                         : constant String    := "83";
   Option_Language_95                         : constant String    := "95";
   Option_Language_2005                       : constant String    := "2005";
   Option_No_Echo                             : constant String    := "noecho";
   Option_No_Dictionary                       : constant String    := "nodictionary";
   Option_No_Duration                         : constant String    := "noduration";
   Option_No_Switch                           : constant String    := "noswitch";
   Option_No_Sli                              : constant String    := "nosli";
   Option_No_Listings                         : constant String    := "nolistings";
   Option_Original_Flow_Errors                : constant String    := "original_flow_errors";
   Option_Output_Directory                    : constant String    := "output_directory";
   Option_Plain_Output                        : constant String    := "plain_output";
   Option_Profile                             : constant String    := "profile";
   Option_Profile_Ravenscar                   : constant String    := "ravenscar";
   Option_Profile_Sequential                  : constant String    := "sequential";
   Option_Policy                              : constant String    := "policy";
   Option_Policy_Security                     : constant String    := "security";
   Option_Policy_Safety                       : constant String    := "safety";
   Option_Report_File                         : constant String    := "report_file";
   Option_Rules                               : constant String    := "rules";
   Option_Rules_None                          : constant String    := "none";
   Option_Rules_Lazy                          : constant String    := "lazy";
   Option_Rules_Keen                          : constant String    := "keen";
   Option_Rules_All                           : constant String    := "all";
   Option_Source_Extension                    : constant String    := "source_extension";
   Option_Syntax_Check                        : constant String    := "syntax_check";
   Option_Statistics                          : constant String    := "statistics";
   Option_Target_Compiler_Data                : constant String    := "target_compiler_data";
   Option_Vcg                                 : constant String    := "vcg";
   Option_Version                             : constant String    := "version";
   Option_Warning_File                        : constant String    := "warning_file";
   Option_Xml                                 : constant String    := "xml";

   Content : CommandLineContents;

   procedure Initialize;
   --# global out Content;
   --# derives Content from ;

   -- If Content.OutputDirectory is set, then interpret and return F relative
   -- to the specified output directory
   procedure Normalize_FileName_To_Output_Directory (F : in out EStrings.T);
   --# global in Content;
   --# derives F from *,
   --#                Content;

   -- Echoes the various filenames in Content to the terminal,
   -- iff Content.Debug.FileNames has been selected.
   procedure Dump_File_Names;
   --# global in Content;
   --# derives null from Content;

   procedure Output_Command_Line (Prefix     : in     String;
                                  XML        : in     Boolean;
                                  Option_Str :    out ELStrings.T);
   --# global in     Content;
   --#        in     SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives Option_Str      from Content,
   --#                              Prefix,
   --#                              SPARK_IO.File_Sys,
   --#                              XML,
   --#                              XMLReport.State &
   --#         XMLReport.State from *,
   --#                              Content,
   --#                              XML;

   function RavenscarSelected return Boolean;
   --# global in Content;

end CommandLineData;
