-- $Id: commandlinedata.adb 16026 2010-02-10 18:17:13Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

---------------------------------------------------------------------------------------
-- WARNING: if you add or modify a command line switch be sure to update      --
--          PrintHelp in spark.apb                                            --
--------------------------------------------------------------------------------

with FileSystem;
with SPARK_IO;
with XMLReport;

package body CommandLineData
is
   Ada_Extension          : constant String := "ada";
   Lst_Extension          : constant String := "lst";
   Default_Report_File    : constant String := "spark.rep";
   Default_Dict_File      : constant String := "spark.dic";
   Default_HTML_Directory : constant String := "HTML";

   procedure Initialize is
   begin
      Content := CommandLineContents'
         (Valid               => True,
          Index               => False,
          Warning             => False,
          TargetData          => False,
          TargetConfig        => False,
          SourceExtension     => EStrings.Copy_String (Str => Ada_Extension),
          ListingExtension    => EStrings.Copy_String (Str => Lst_Extension),
          NumberSource        => 0,
          Echo                => True,
          Report              => True,
          WriteDict           => False,
          ReportFileName      => EStrings.Copy_String (Str => Default_Report_File),
          DictFileName        => EStrings.Copy_String (Str => Default_Dict_File),
          VCG                 => False,
          DPC                 => False,
          WriteStatistics     => False,
          DoInformationFlow   => True,
          DefaultSwitchFile   => False,
          HTML                => False,
          HTMLDirectory       => EStrings.Copy_String (Str => Default_HTML_Directory),
          OutputDirectory     => False,
          OutputDirectoryName => EStrings.Empty_String,
          IndexFileName       => EStrings.Empty_String,
          WarningFileName     => EStrings.Empty_String,
          TargetDataFile      => EStrings.Empty_String,
          TargetConfigFile    => EStrings.Empty_String,
          SourceFileList      =>
            SourceFileLists'(others => SourceFileEntry'
                               (SourceFileName  => EStrings.Empty_String,
                                Listing         => False,
                                ListingFileName => EStrings.Empty_String)),
          AnnoChar            => '#',
          SyntaxOnly          => False,
          FDLreserved         => True,
          FDLmangle           => EStrings.Empty_String,
          PlainOutput         => False,
          VersionRequested    => False,
          HelpRequested       => False,
          VCfingerprints      => False,
          ConcurrencyProfile  => Sequential,
          LanguageProfile     => SPARK95,
          NoDuration          => False,
          Brief               => False,
          BriefOption         => NoPath,
          XML                 => False,
          InfoFlowPolicy      => None,
          ConstantRules       => NoRules,
          LegacyErrors        => False,
          ErrorExplanation    => Off,
          JustificationOption => Full,
          NoListings          => False,
          Generate_SLI        => True,
          Casing_Standard     => False,
          Casing_Identifier   => False,
          Debug               => DebugState'
            (Enabled     => False,
             Expressions => False,
             HTML        => False,
             LookupTrace => False,
             FileNames   => False,
             Units       => False,
             Invariants  => False,
             Components  => False,
             Rho         => False,
             Parser      => False,
             FDL_Ranking => False,
             VCG         => False,
             SLI         => False,
             VCG_All     => False,
             DAG         => False));
   end Initialize;

   procedure Normalize_FileName_To_Output_Directory (F : in out EStrings.T)
   is
      Original : EStrings.T;

      procedure Debug
      --# derives ;
      is
         --# hide Debug;
      begin
         if Content.Debug.FileNames then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Normalizing ", 0);
            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => Original);
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, " with respect to ", 0);
            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => Content.OutputDirectoryName);
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, " results in ", 0);
            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => F);
         end if;
      end Debug;

   begin
      Original := F;
      if Content.OutputDirectory then
         F := FileSystem.InterpretRelative
           (FileName             => Original,
            RelativeToDirectory  => Content.OutputDirectoryName);
         Debug;
      end if;
   end Normalize_FileName_To_Output_Directory;

   procedure Dump_File_Names
   is
      --# hide Dump_File_Names;
   begin
      if Content.Debug.FileNames then
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "------------Dump of CommandLineData.Content File Names------------", 0);
         if Content.Index then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Index file is:", 0);
            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => Content.IndexFileName);
         else
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "No Index File", 0);
         end if;

         if Content.Warning then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Warning file is:", 0);
            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => Content.WarningFileName);
         else
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "No Warning File", 0);
         end if;

         if Content.TargetData then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Target Data File is:", 0);
            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => Content.TargetDataFile);
         else
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "No Target Data File", 0);
         end if;

         if Content.TargetConfig then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Target Config File is:", 0);
            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => Content.TargetConfigFile);
         else
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "No Target Config File", 0);
         end if;

         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Source extension is:", 0);
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => Content.SourceExtension);

         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Listing extension is:", 0);
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => Content.ListingExtension);

         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Report File Name is:", 0);
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => Content.ReportFileName);

         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Dictionary File Name is:", 0);
         EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                            E_Str => Content.DictFileName);

         if Content.OutputDirectory then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Output Directory is:", 0);
            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => Content.OutputDirectoryName);
         else
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "Output directory is CWD", 0);
         end if;

         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Number of Source Files is:", 0);
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Integer'Image (Content.NumberSource), 0);

         for I in SourceFilePositions range 1 .. Content.NumberSource loop
            EStrings.Put_String (File  => SPARK_IO.Standard_Output,
                                 E_Str => Content.SourceFileList (I).SourceFileName);

            if Content.NoListings then
               SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                                  ", All listing files suppressed", 0);
            elsif Content.SourceFileList (I).Listing then
               SPARK_IO.Put_String (SPARK_IO.Standard_Output, ", ", 0);
               EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                                  E_Str => Content.SourceFileList (I).ListingFileName);
            else
               SPARK_IO.Put_Line (SPARK_IO.Standard_Output, ", No listing file", 0);
            end if;
         end loop;

         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Examiner Lib Directory is: ", 0);

         if Content.PlainOutput and then
           FileSystem.UseWindowsCommandLine then
            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => EStrings.Lower_Case (E_Str => FileSystem.ExaminerLibDirectory));
         else
            EStrings.Put_Line (File  => SPARK_IO.Standard_Output,
                               E_Str => FileSystem.ExaminerLibDirectory);
         end if;

         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "------------End of CommandLineData.Content File Names-------------", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      end if;

   end Dump_File_Names;

   procedure Output_Command_Line (Prefix     : in     String;
                                  XML        : in     Boolean;
                                  Option_Str :    out ELStrings.T)
   is
      Dir_Actual_C : constant String := "output_directory (actual)=";

      Tmp_String  : EStrings.T;
      Tmp_Success : Boolean;
      Out_Dir     : EStrings.T;
      CWD         : EStrings.T;
      Is_First    : Boolean;

      procedure Output_Option (Prefix  : in     String;
                               Option  : in     EStrings.T;
                               Acc     : in out ELStrings.T;
                               XML     : in     Boolean;
                               For_XML : in     Boolean)
      --# global in out XMLReport.State;
      --# derives Acc             from *,
      --#                              For_XML,
      --#                              Option,
      --#                              Prefix,
      --#                              XML,
      --#                              XMLReport.State &
      --#         XMLReport.State from *,
      --#                              For_XML,
      --#                              XML;
      is
         Acc_Option : EStrings.T;
      begin
         if XML then
            if For_XML then
               Acc_Option := Option;
               XMLReport.Option (Opt => Acc_Option);
            else
               Acc_Option := EStrings.Empty_String;
            end if;
         else
            Acc_Option := EStrings.Copy_String (Str => Prefix);
            EStrings.Append_Examiner_String (E_Str1 => Acc_Option,
                                             E_Str2 => Option);
            EStrings.Append_Examiner_String (E_Str1 => Acc_Option,
                                             E_Str2 => FileSystem.End_Of_Line);
         end if;
         ELStrings.Append_Examiner_String (E_Str1 => Acc,
                                           E_Str2 => Acc_Option);
      end Output_Option;

      function Plain_Output (E_Str : EStrings.T;
                             XML   : Boolean)
                            return EStrings.T
      --# global in Content;
      is
         Result : EStrings.T;
      begin
         if Content.PlainOutput and then not XML then
            Result := EStrings.Lower_Case (E_Str => E_Str);
         else
            Result := E_Str;
         end if;
         return Result;
      end Plain_Output;

   begin
      Option_Str := ELStrings.Empty_String;
      if not Content.DefaultSwitchFile then
         Tmp_String := EStrings.Copy_String (Str => Option_No_Switch);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      --# assert True;

      if Content.Index then
         Tmp_String := EStrings.Copy_String (Str => Option_Index_File & "=");
         EStrings.Append_Examiner_String
           (E_Str1 => Tmp_String,
            E_Str2 => Plain_Output (E_Str => Content.IndexFileName,
                                    XML   => XML));
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      else
         Tmp_String := EStrings.Copy_String (Str => "no" & Option_Index_File);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => False);
      end if;

      if Content.Warning then
         Tmp_String := EStrings.Copy_String (Str => Option_Warning_File & "=");
         EStrings.Append_Examiner_String
           (E_Str1 => Tmp_String,
            E_Str2 => Plain_Output (E_Str => Content.WarningFileName,
                                    XML   => XML));
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      else
         Tmp_String := EStrings.Copy_String (Str => "no" & Option_Warning_File);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => False);
      end if;

      --# assert True;

      if Content.TargetData then
         Tmp_String := EStrings.Copy_String (Str => Option_Target_Compiler_Data & "=");
         EStrings.Append_Examiner_String
           (E_Str1 => Tmp_String,
            E_Str2 => Plain_Output (E_Str => Content.TargetDataFile,
                                    XML   => XML));
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      else
         Tmp_String := EStrings.Copy_String (Str => "no" & Option_Target_Compiler_Data);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => False);
      end if;

      if Content.TargetConfig then
         Tmp_String := EStrings.Copy_String (Str => Option_Config_File & "=");
         EStrings.Append_Examiner_String
           (E_Str1 => Tmp_String,
            E_Str2 => Plain_Output (E_Str => Content.TargetConfigFile,
                                    XML   => XML));
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      else
         Tmp_String := EStrings.Copy_String (Str => "no" & Option_Config_File);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => False);
      end if;

      --# assert True;

      Tmp_String := EStrings.Copy_String (Str => Option_Source_Extension & "=");
      EStrings.Append_Examiner_String
        (E_Str1 => Tmp_String,
         E_Str2 => Plain_Output (E_Str => Content.SourceExtension,
                                 XML   => XML));
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);

      Tmp_String := EStrings.Copy_String (Str => Option_Listing_Extension & "=");
      EStrings.Append_Examiner_String
        (E_Str1 => Tmp_String,
         E_Str2 => Plain_Output (E_Str => Content.ListingExtension,
                                 XML   => XML));
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);
      if Content.NoListings then
         Tmp_String := EStrings.Copy_String (Str => Option_No_Listings);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => False);
      end if;

      if not Content.Echo then
         Tmp_String := EStrings.Copy_String (Str => Option_No_Echo);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      if Content.WriteDict then
         Tmp_String := EStrings.Copy_String (Str => Option_Dictionary_File & "=");
         EStrings.Append_Examiner_String
           (E_Str1 => Tmp_String,
            E_Str2 => Plain_Output (E_Str => Content.DictFileName,
                                    XML   => XML));
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      else
         Tmp_String := EStrings.Copy_String (Str => "no" & Option_Dictionary_File);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => False);
      end if;

      --# assert True;

      if Content.Report then
         Tmp_String := EStrings.Copy_String (Str => Option_Report_File & "=");
         EStrings.Append_Examiner_String
           (E_Str1 => Tmp_String,
            E_Str2 => Plain_Output (E_Str => Content.ReportFileName,
                                    XML   => XML));
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      else
         Tmp_String := EStrings.Copy_String (Str => "no" & Option_Report_File);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => False);
      end if;

      if Content.HTML then
         Tmp_String := EStrings.Copy_String (Str => Option_Html & "=");
         EStrings.Append_Examiner_String
           (E_Str1 => Tmp_String,
            E_Str2 => Plain_Output (E_Str => Content.HTMLDirectory,
                                    XML   => XML));
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      else
         Tmp_String := EStrings.Copy_String (Str => "no" & Option_Html);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => False);
      end if;

      --# assert True;

      if Content.VCG then
         Tmp_String := EStrings.Copy_String (Str => Option_Vcg);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      if Content.DPC then
         Tmp_String := EStrings.Copy_String (Str => Option_Dpc);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      --# assert True;

      if Content.SyntaxOnly then
         Tmp_String := EStrings.Copy_String (Str => Option_Syntax_Check);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      --# assert True;

      if Content.PlainOutput then
         Tmp_String := EStrings.Copy_String (Str => Option_Plain_Output);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      if Content.NoDuration then
         Tmp_String := EStrings.Copy_String (Str => Option_No_Duration);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      if Content.Brief then
         Tmp_String := EStrings.Copy_String (Str => Option_Brief & "=");
         case Content.BriefOption is
            when NoPath   => EStrings.Append_String (E_Str => Tmp_String,
                                                     Str   => Option_Brief_No_Path);
            when FullPath => EStrings.Append_String (E_Str => Tmp_String,
                                                     Str   => Option_Brief_Full_Path);
         end case;
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      if Content.WriteStatistics then
         Tmp_String := EStrings.Copy_String (Str => Option_Statistics);
      else
         Tmp_String := EStrings.Copy_String (Str => "no" & Option_Statistics);
      end if;
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);

      --# assert True;

      Tmp_String := EStrings.Copy_String (Str => Option_Fdl_Identifiers & "=");
      if Content.FDLreserved then
         EStrings.Append_String (E_Str => Tmp_String,
                                 Str   => Option_Fdl_Identifiers_Accept);
      else
         if EStrings.Is_Empty (E_Str => Content.FDLmangle) then
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Fdl_Identifiers_Reject);
         else
            EStrings.Append_Examiner_String (E_Str1 => Tmp_String,
                                             E_Str2 => Content.FDLmangle);
         end if;
      end if;
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);

      Tmp_String := EStrings.Copy_String (Str => Option_Flow_Analysis & "=");
      if Content.DoInformationFlow then
         EStrings.Append_String (E_Str => Tmp_String,
                                 Str   => Option_Flow_Analysis_Information);
      else
         EStrings.Append_String (E_Str => Tmp_String,
                                 Str   => Option_Flow_Analysis_Data);
      end if;
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);

      --# assert True;

      Tmp_String := EStrings.Copy_String (Str => Option_Policy & "=");
      case Content.InfoFlowPolicy is
         when None =>
            null;
         when Safety =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Policy_Safety);
            Output_Option (Prefix  => Prefix,
                           Option  => Tmp_String,
                           Acc     => Option_Str,
                           XML     => XML,
                           For_XML => True);
         when Security =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Policy_Security);
            Output_Option (Prefix  => Prefix,
                           Option  => Tmp_String,
                           Acc     => Option_Str,
                           XML     => XML,
                           For_XML => True);
      end case;

      --# assert True;

      Tmp_String := EStrings.Copy_String (Str => Option_Language & "=");
      case Content.LanguageProfile is
         when SPARK83 =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Language_83);
         when SPARK95 =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Language_95);
         when SPARK2005 =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Language_2005);
      end case;
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);

      Tmp_String := EStrings.Copy_String (Str => Option_Profile & "=");
      case Content.ConcurrencyProfile is
         when Ravenscar =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Profile_Ravenscar);
         when Sequential =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Profile_Sequential);
      end case;
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);

      Tmp_String := EStrings.Copy_String (Str => Option_Annotation_Character & "=");
      --# accept F, 10, Tmp_Success, "Not required here";
      EStrings.Append_Char (E_Str   => Tmp_String,
                            Ch      => Content.AnnoChar,
                            Success => Tmp_Success);
      --# end accept;
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);

      --# assert True;


      Tmp_String := EStrings.Copy_String (Str => Option_Rules & "=");
      case Content.ConstantRules is
         when NoRules =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Rules_None);
         when Lazy =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Rules_Lazy);
         when Keen =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Rules_Keen);
         when AllRules =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Rules_All);
      end case;
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);

      --# assert True;

      if Content.LegacyErrors then
         Tmp_String := EStrings.Copy_String (Str => Option_Original_Flow_Errors);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      if not Content.Generate_SLI then
         Tmp_String := EStrings.Copy_String (Str => Option_No_Sli);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      --# assert True;

      Tmp_String := EStrings.Copy_String (Str => Option_Error_Explanations & "=");
      case Content.ErrorExplanation is
         when Off =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Error_Explanations_Off);
         when FirstOccurrence =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Error_Explanations_First_Occurrence);
         when EveryOccurrence =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Error_Explanations_Every_Occurrence);
      end case;
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);

      Tmp_String := EStrings.Copy_String (Str => Option_Justification_Option & "=");
      case Content.JustificationOption is
         when Ignore =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Justification_Option_Ignore);
         when Full =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Justification_Option_Full);
         when Brief =>
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => Option_Justification_Option_Brief);
      end case;
      Output_Option (Prefix  => Prefix,
                     Option  => Tmp_String,
                     Acc     => Option_Str,
                     XML     => XML,
                     For_XML => True);

      --# assert True;

      if Content.Casing_Standard or else Content.Casing_Identifier then
         Tmp_String := EStrings.Copy_String (Str => Option_Casing & "=");
         if Content.Casing_Standard then
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Casing_Standard,
                                  Success => Tmp_Success);
            --# end accept;
         end if;
         if Content.Casing_Identifier then
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Casing_Identifier,
                                  Success => Tmp_Success);
            --# end accept;
         end if;
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
      end if;

      if Content.Debug.Enabled then
         Tmp_String := EStrings.Copy_String (Str => Option_Debug);
         Is_First := True;

         if Content.Debug.Expressions then
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => '=',
                                  Success => Tmp_Success);
            --# end accept;
            Is_First := False;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_E,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.HTML then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_H,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.LookupTrace then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_L,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.FileNames then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_F,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.Units then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_U,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.Invariants then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_I,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.Components then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_C,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.Rho then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_R,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.Parser then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_P,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         if Content.Debug.FDL_Ranking then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_K,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.VCG then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_V,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.VCG_All then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_V_Upper,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.DAG then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
               Is_First := False;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_D,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         --# assert True;

         if Content.Debug.SLI then
            if Is_First then
               --# accept F, 10, Tmp_Success, "Not required here";
               EStrings.Append_Char (E_Str   => Tmp_String,
                                     Ch      => '=',
                                     Success => Tmp_Success);
               --# end accept;
            end if;
            --# accept F, 10, Tmp_Success, "Not required here";
            EStrings.Append_Char (E_Str   => Tmp_String,
                                  Ch      => Option_Debug_X,
                                  Success => Tmp_Success);
            --# end accept;
         end if;

         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => False);
      end if;

      --# assert True;

      if Content.OutputDirectory then

         -- output_directory _has_ been specified, so firstly
         -- report exactly what the user gave on the command-line
         -- as the requested output directory
         Tmp_String := EStrings.Copy_String (Str => Option_Output_Directory & "=");

         Out_Dir := Content.OutputDirectoryName;

         -- Out_Dir might have an EndOfPath (normally '/' or '\') on
         -- the end, so remove it here.
         FileSystem.RemoveEndOfPathIfPresent (Out_Dir);
         EStrings.Append_Examiner_String (E_Str1 => Tmp_String,
                                          E_Str2 => Out_Dir);
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);

         -- Now report the actual output directory.
         if Content.PlainOutput then
            -- Plain output mode, so just repeat the requested dir.
            Tmp_String := EStrings.Copy_String (Str => Dir_Actual_C);
            EStrings.Append_Examiner_String (E_Str1 => Tmp_String,
                                             E_Str2 => Out_Dir);
            Output_Option (Prefix  => Prefix,
                           Option  => Tmp_String,
                           Acc     => Option_Str,
                           XML     => XML,
                           For_XML => True);
         else
            -- Verbose output mode...imagine we're creating a dummy
            -- file called "a" in the requested output directory.
            -- Normalize this filename for the output dir, then
            -- find its full pathname relative to CWD, then report
            -- that, minus the "a" on the end...
            Tmp_String := EStrings.Copy_String (Str => Dir_Actual_C);
            Out_Dir := EStrings.Copy_String (Str => "a");
            Normalize_FileName_To_Output_Directory (Out_Dir);

            CWD := FileSystem.WorkingDirectory;
            FileSystem.AppendEndOfPathIfNeeded (CWD);

            Out_Dir := FileSystem.InterpretRelative
              (FileName            => Out_Dir,
               RelativeToDirectory => CWD);

            -- Drop the "a" off the end...
            Out_Dir := EStrings.Section (E_Str     => Out_Dir,
                                         Start_Pos => 1,
                                         Length    => EStrings.Get_Length (E_Str => Out_Dir) - 1);

            EStrings.Append_Examiner_String (E_Str1 => Tmp_String,
                                             E_Str2 => Out_Dir);
            Output_Option (Prefix  => Prefix,
                           Option  => Tmp_String,
                           Acc     => Option_Str,
                           XML     => XML,
                           For_XML => True);
         end if;
      else
         -- output_directory NOT specified, so report "."
         Tmp_String := EStrings.Copy_String (Str => Option_Output_Directory & "=");
         EStrings.Append_String (E_Str => Tmp_String,
                                 Str   => ".");
         Output_Option (Prefix  => Prefix,
                        Option  => Tmp_String,
                        Acc     => Option_Str,
                        XML     => XML,
                        For_XML => True);
         if Content.PlainOutput then
            -- Plain output, so report "." as actual output directory
            -- to avoid diff between runs and platforms
            Tmp_String := EStrings.Copy_String (Str => Dir_Actual_C);
            EStrings.Append_String (E_Str => Tmp_String,
                                    Str   => ".");
            Output_Option (Prefix  => Prefix,
                           Option  => Tmp_String,
                           Acc     => Option_Str,
                           XML     => XML,
                           For_XML => True);
         else
            -- Verbose output, so report full pathname of
            -- current working directory
            Tmp_String := EStrings.Copy_String (Str => Dir_Actual_C);
            CWD := FileSystem.WorkingDirectory;
            FileSystem.AppendEndOfPathIfNeeded (CWD);
            EStrings.Append_Examiner_String (E_Str1 => Tmp_String,
                                             E_Str2 => CWD);
            Output_Option (Prefix  => Prefix,
                           Option  => Tmp_String,
                           Acc     => Option_Str,
                           XML     => XML,
                           For_XML => True);
         end if;

      end if;

      --# accept F, 33, Tmp_Success, "Not required here";
   end Output_Command_Line;

   function RavenscarSelected return Boolean
   is
   begin
      return Content.ConcurrencyProfile = Ravenscar;
   end RavenscarSelected;

end CommandLineData;
