-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

-- Overview
-- Checks sub-program declarations from node subprogram_declaration.  These
-- nodes occur only in package declarations therefore well formation of
-- function_ and procedure_specifications are handled here as a special case
-- rather than using the more complex and general-purpose
-- wf_procedure_specification and wf_function_specification.
-- NOTE 11/6/02
-- Declarations also occur in protected types but this procedure can
-- deal with those as well
--------------------------------------------------------------------------------

separate (Sem.CompUnit)
procedure Wf_Subprogram_Declaration
  (Node            : in     STree.SyntaxNode;
   Current_Scope   : in     Dictionary.Scopes;
   The_Subprog_Sym :    out Dictionary.Symbol) is
   Spec_Node, Anno_Node, Constraint_Node, Formal_Part_Node : STree.SyntaxNode;
   Subprog_Sym                                             : Dictionary.Symbol;
   Is_Overriding                                           : Boolean := False;

   ------------------------------------------------------------------------

   -- If we are declaring a subprogram in a package spec and the spec contains
   -- protected types we search each of these to detect re-use of the subprogram
   -- name.  If we don't trap such re-use at this point then we end up with a
   -- legal package spec for which no legal body could be written (since its
   -- implementation would inevitably involve overload resolution of calls made from
   -- within the protected body.  e.g. type PT in package P declares operation K.  Package
   -- P also declares an operation K.  From inside the body of PT, a call to K could refer
   -- to either of the two Ks since both are directly visible.
   function Is_Defined_In_Visible_Protected_Type (Name  : LexTokenManager.Lex_String;
                                                  Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   is
      Result : Boolean := False;
      It     : Dictionary.Iterator;
   begin
      if Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
         It := Dictionary.FirstVisibleProtectedType (Dictionary.GetRegion (Scope));
         while not Dictionary.IsNullIterator (It) loop
            Result :=
              Dictionary.IsDirectlyDefined
              (Name,
               Dictionary.VisibleScope (Dictionary.CurrentSymbol (It)),
               Dictionary.ProofContext);
            exit when Result;
            It := Dictionary.NextSymbol (It);
         end loop;
      end if;
      return Result;
   end Is_Defined_In_Visible_Protected_Type;

   ------------------------------------------------------------------------

   procedure Check_Procedure_Specification
     (Node          : in     STree.SyntaxNode;
      Current_Scope : in     Dictionary.Scopes;
      Subprog_Sym   :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict,
   --#         Subprog_Sym                from ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   is
      Ident_Str : LexTokenManager.Lex_String;

      -- ditto for protected types in package private scope
      function Is_Defined_In_Private_Protected_Type
        (Name  : LexTokenManager.Lex_String;
         Scope : Dictionary.Scopes)
        return  Boolean
      --# global in Dictionary.Dict;
      --#        in LexTokenManager.State;
      is
         Result : Boolean := False;
         It     : Dictionary.Iterator;
      begin
         if Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
            It := Dictionary.FirstPrivateProtectedType (Dictionary.GetRegion (Scope));
            while not Dictionary.IsNullIterator (It) loop
               Result :=
                 Dictionary.IsDirectlyDefined
                 (Name,
                  Dictionary.VisibleScope (Dictionary.CurrentSymbol (It)),
                  Dictionary.ProofContext);
               exit when Result;
               It := Dictionary.NextSymbol (It);
            end loop;
         end if;
         return Result;
      end Is_Defined_In_Private_Protected_Type;

   begin -- Check_Procedure_Specification

      -- ASSUME Node = procedure_specification OR entry_specification
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.procedure_specification
           or else Syntax_Node_Type (Node => Node) = SPSymbols.entry_specification,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = procedure_specification OR entry_specification in Check_Procedure_Specification");
      Ident_Str := Node_Lex_String (Node => Child_Node (Current_Node => Child_Node (Current_Node => Node)));
      if Dictionary.IsDefined (Name              => Ident_Str,
                               Scope             => Current_Scope,
                               Context           => Dictionary.ProofContext,
                               Full_Package_Name => False) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 10,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => Ident_Str);
         Subprog_Sym := Dictionary.NullSymbol;
      elsif Is_Defined_In_Visible_Protected_Type (Name  => Ident_Str,
                                                  Scope => Current_Scope)
        or else Is_Defined_In_Private_Protected_Type (Name  => Ident_Str,
                                                      Scope => Current_Scope) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 988,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => Ident_Str);
         Subprog_Sym := Dictionary.NullSymbol;
      else
         Dictionary.AddSubprogram
           (Name          => Ident_Str,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                  End_Position   => Node_Position (Node => Node)),
            Scope         => Current_Scope,
            Context       => Dictionary.ProgramContext,
            Subprogram    => Subprog_Sym);
         Subprog_Sym := Subprog_Sym;
         if Syntax_Node_Type (Node => Node) = SPSymbols.entry_specification then
            Dictionary.SetSubprogramIsEntry (Subprog_Sym);
         end if;
      end if;
   end Check_Procedure_Specification;

   ------------------------------------------------------------------------

   procedure Check_Function_Specification
     (Node          : in     STree.SyntaxNode;
      Current_Scope : in     Dictionary.Scopes;
      Subprog_Sym   :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Subprog_Sym                from ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   is
      Return_Type_Node : STree.SyntaxNode;
      Ident_Str        : LexTokenManager.Lex_String;
      Type_Sym         : Dictionary.Symbol;
   begin
      -- ASSUME Node = function_specification
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.function_specification,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = function_specification in Check_Function_Specification");
      Ident_Str := Node_Lex_String (Node => Child_Node (Current_Node => Child_Node (Current_Node => Node)));
      if Dictionary.IsDefined (Name              => Ident_Str,
                               Scope             => Current_Scope,
                               Context           => Dictionary.ProofContext,
                               Full_Package_Name => False) then
         Subprog_Sym := Dictionary.NullSymbol;
         ErrorHandler.Semantic_Error
           (Err_Num   => 10,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => Ident_Str);
      elsif Is_Defined_In_Visible_Protected_Type (Name  => Ident_Str,
                                                  Scope => Current_Scope) then
         Subprog_Sym := Dictionary.NullSymbol;
         ErrorHandler.Semantic_Error
           (Err_Num   => 988,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => Ident_Str);
      else
         Dictionary.AddSubprogram
           (Name          => Ident_Str,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                  End_Position   => Node_Position (Node => Node)),
            Scope         => Current_Scope,
            Context       => Dictionary.ProgramContext,
            Subprogram    => Subprog_Sym);
      end if;
      Return_Type_Node := Last_Sibling_Of (Start_Node => Child_Node (Current_Node => Node));
      -- ASSUME Return_Type_Node = type_mark
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Return_Type_Node) = SPSymbols.type_mark,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Return_Type_Node = type_mark in Check_Function_Specification");
      Wf_Type_Mark
        (Node          => Return_Type_Node,
         Current_Scope => Current_Scope,
         Context       => Dictionary.ProgramContext,
         Type_Sym      => Type_Sym);
      if Dictionary.IsUnconstrainedArrayType (Type_Sym) then
         Type_Sym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.Semantic_Error
           (Err_Num   => 39,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Return_Type_Node),
            Id_Str    => Node_Lex_String (Node => Return_Type_Node));
      elsif Dictionary.IsPredefinedSuspensionObjectType (Type_Sym) or else Dictionary.TypeIsProtected (Type_Sym) then
         Type_Sym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.Semantic_Error
           (Err_Num   => 905,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Return_Type_Node),
            Id_Str    => LexTokenManager.Null_String);
      elsif Dictionary.TypeIsTagged (Type_Sym) and then (Dictionary.GetScope (Type_Sym) = Current_Scope) then
         -- attempt to declare primitive function with controlling return result
         Type_Sym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.Semantic_Error
           (Err_Num   => 840,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Return_Type_Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;

      if Subprog_Sym /= Dictionary.NullSymbol then
         Dictionary.AddReturnType
           (TheFunction   => Subprog_Sym,
            TypeMark      => Type_Sym,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            TypeReference => Dictionary.Location'(Start_Position => Node_Position (Node => Return_Type_Node),
                                                  End_Position   => Node_Position (Node => Return_Type_Node)));

         -- mark signature as not wellformed if wf_type_mark has returned the unknown type
         if Type_Sym = Dictionary.GetUnknownTypeMark then
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Subprog_Sym);
         end if;
      end if;
   end Check_Function_Specification;

   ------------------------------------------------------------------------

   procedure Get_Anno_And_Con_Nodes
     (Node            : in     STree.SyntaxNode;
      Anno_Node       :    out STree.SyntaxNode;
      Constraint_Node :    out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives Anno_Node,
   --#         Constraint_Node from Node,
   --#                              STree.Table;
   is
      Node_Local : STree.SyntaxNode;
   begin
      -- ASSUME Node = procedure_specification OR function_specification OR proof_function_declaration OR entry_specification
      if Syntax_Node_Type (Node => Node) = SPSymbols.procedure_specification
        or else Syntax_Node_Type (Node => Node) = SPSymbols.function_specification
        or else Syntax_Node_Type (Node => Node) = SPSymbols.entry_specification then
         -- ASSUME Node = procedure_specification OR function_specification OR entry_specification
         Node_Local := Next_Sibling (Current_Node => Node);
         -- ASSUME Node_Local = procedure_annotation OR function_annotation
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node_Local) = SPSymbols.procedure_annotation
              or else Syntax_Node_Type (Node => Node_Local) = SPSymbols.function_annotation,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node_Local = procedure_annotation OR function_annotation in Get_Anno_And_Con_Nodes");
         Constraint_Node := Child_Node (Current_Node => Node_Local);
         -- ASSUME Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation OR
         --                          procedure_constraint OR function_constraint
         if Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint
           or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint then
            -- ASSUME Constraint_Node = function_constraint OR procedure_constraint
            Anno_Node := STree.NullNode; -- only a constraint found
         elsif Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.moded_global_definition
           or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.dependency_relation
           or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.declare_annotation then
            -- ASSUME Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation
            Anno_Node       := Node_Local;
            Constraint_Node := Last_Sibling_Of (Start_Node => Constraint_Node);
         else
            Anno_Node       := STree.NullNode;
            Constraint_Node := STree.NullNode;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Constraint_Node = moded_global_definition OR dependency_relation OR declare_annotation OR procedure_constraint OR function_constraint in Get_Anno_And_Con_Nodes");
         end if;
      elsif Syntax_Node_Type (Node => Node) = SPSymbols.proof_function_declaration then
         -- ASSUME Node = proof_function_declaration
         Anno_Node       := STree.NullNode;
         Constraint_Node := STree.NullNode;
      else
         Anno_Node       := STree.NullNode;
         Constraint_Node := STree.NullNode;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = procedure_specification OR function_specification OR proof_function_declaration OR entry_specification in Get_Anno_And_Con_Nodes");
      end if;
      -- ASSUME Anno_Node = procedure_annotation OR function_annotation OR NULL
      SystemErrors.RT_Assert
        (C       => Anno_Node = STree.NullNode
           or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation
           or else Syntax_Node_Type (Node => Anno_Node) = SPSymbols.function_annotation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Anno_Node = procedure_annotation OR function_annotation OR NULL in Get_Anno_And_Con_Nodes");
      -- ASSUME Constraint_Node = function_constraint OR procedure_constraint OR NULL
      SystemErrors.RT_Assert
        (C       => Constraint_Node = STree.NullNode
           or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint
           or else Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Constraint_Node = function_constraint OR procedure_constraint OR NULL in Get_Anno_And_Con_Nodes");
   end Get_Anno_And_Con_Nodes;

begin -- Wf_Subprogram_Declaration

   -- ASSUME Node = subprogram_declaration OR entry_declaration
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.subprogram_declaration
        or else Syntax_Node_Type (Node => Node) = SPSymbols.entry_declaration,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = subprogram_declaration OR entry_declaration in Wf_Subprogram_Declaration");

   -- Determine and record in the variable Overriding_Indicator
   -- if the procedure overrides a parent.
   -- In SPARK 2005 "not overriding Procedure ..." is equivalent
   -- to "Procedure ...".
   -- This differs from Ada 2005 where a procedure may override
   -- a parent procedure when no overriding_indicator is present.

   Spec_Node := Child_Node (Current_Node => Node);
   -- ASSUME Spec_Node = overriding_indicator OR procedure_specification OR function_specification OR
   --                    proof_function_declaration OR entry_specification
   if Syntax_Node_Type (Node => Spec_Node) = SPSymbols.overriding_indicator then
      -- ASSUME Child_Node (Current_Node => Spec_Node) = RWoverriding OR RWnot
      if Syntax_Node_Type (Node => Child_Node (Current_Node => Spec_Node)) = SPSymbols.RWoverriding then
         -- ASSUME Child_Node (Current_Node => Spec_Node) = RWoverriding
         Is_Overriding := True;
      elsif Syntax_Node_Type (Node => Child_Node (Current_Node => Spec_Node)) /= SPSymbols.RWnot then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Child_Node (Current_Node => Spec_Node) = RWoverriding OR RWnot in Wf_Subprogram_Declaration");
      end if;
      Spec_Node := Next_Sibling (Current_Node => Spec_Node);
   elsif Syntax_Node_Type (Node => Spec_Node) /= SPSymbols.procedure_specification
     and then Syntax_Node_Type (Node => Spec_Node) /= SPSymbols.function_specification
     and then Syntax_Node_Type (Node => Spec_Node) /= SPSymbols.proof_function_declaration
     and then Syntax_Node_Type (Node => Spec_Node) /= SPSymbols.entry_specification then
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Spec_Node = overriding_indicator OR procedure_specification OR function_specification OR proof_function_declaration OR entry_specification in Wf_Subprogram_Declaration");
   end if;
   -- ASSUME Spec_Node = procedure_specification OR function_specification OR proof_function_declaration OR entry_specification
   Get_Anno_And_Con_Nodes (Node            => Spec_Node,
                           Anno_Node       => Anno_Node,
                           Constraint_Node => Constraint_Node);
   if Syntax_Node_Type (Node => Spec_Node) = SPSymbols.procedure_specification
     or else Syntax_Node_Type (Node => Spec_Node) = SPSymbols.entry_specification then
      -- ASSUME Spec_Node = procedure_specification OR entry_specification
      Check_Procedure_Specification (Node          => Spec_Node,
                                     Current_Scope => Current_Scope,
                                     Subprog_Sym   => Subprog_Sym);
      The_Subprog_Sym := Subprog_Sym; -- pass back to caller
      if Subprog_Sym /= Dictionary.NullSymbol then
         Formal_Part_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Spec_Node));
         -- ASSUME Formal_Part_Node = formal_part OR NULL
         if Syntax_Node_Type (Node => Formal_Part_Node) = SPSymbols.formal_part then
            -- ASSUME Formal_Part = formal_part
            Wf_Formal_Part
              (Node             => Formal_Part_Node,
               Current_Scope    => Current_Scope,
               Subprog_Sym      => Subprog_Sym,
               First_Occurrence => True,
               Context          => Dictionary.ProgramContext);
         elsif Formal_Part_Node /= STree.NullNode then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Formal_Part_Node = formal_part OR NULL in Wf_Subprogram_Declaration");
         end if;
         -- ASSUME Anno_Node = procedure_annotation OR NULL
         if Syntax_Node_Type (Node => Anno_Node) = SPSymbols.procedure_annotation then
            -- ASSUME Anno_Node = procedure_annotation
            Wf_Procedure_Annotation
              (Node          => Anno_Node,
               Current_Scope => Current_Scope,
               Subprog_Sym   => Subprog_Sym,
               First_Seen    => True);
         elsif Anno_Node = STree.NullNode then
            -- ASSUME Anno_Node = NULL
            if CommandLineData.Content.Language_Profile = CommandLineData.SPARK83
              or else CommandLineData.Content.Do_Information_Flow then
               -- no anno is always an error of 83 or if info flow is turned on
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Subprog_Sym);
               ErrorHandler.Semantic_Error
                 (Err_Num   => 154,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str    => Dictionary.GetSimpleName (Subprog_Sym));
            end if;
         else
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Anno_Node = procedure_annotation OR NULL in Wf_Subprogram_Declaration");
         end if;

         -- If we're in 95 or 2005 mode, and -flow=data, then synthesize
         -- a dependency relation for this subprogram.

         if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
           and then not CommandLineData.Content.Do_Information_Flow then
            CreateFullSubProgDependency (Node, Subprog_Sym, Dictionary.IsAbstract);
         end if;

         -- ASSUME Constraint_Node = procedure_constraint OR NULL
         if Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.procedure_constraint then
            -- ASSUME Constraint_Node = procedure_constraint
            wf_procedure_constraint (Constraint_Node, Dictionary.LocalScope (Subprog_Sym), True);
         elsif Constraint_Node /= STree.NullNode then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Constraint_Node = procedure_annotation OR NULL in Wf_Subprogram_Declaration");
         end if;
      end if;
      CheckNoOverloadingFromTaggedOps (Spec_Node, Subprog_Sym, Current_Scope, Dictionary.IsAbstract, Is_Overriding);
   elsif Syntax_Node_Type (Node => Spec_Node) = SPSymbols.function_specification then
      -- ASSUME Spec_Node = function_specification
      Check_Function_Specification (Node          => Spec_Node,
                                    Current_Scope => Current_Scope,
                                    Subprog_Sym   => Subprog_Sym);
      The_Subprog_Sym := Subprog_Sym; -- pass back to caller
      if Subprog_Sym /= Dictionary.NullSymbol then
         Formal_Part_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Spec_Node));
         -- ASSUME Formal_Part_Node = formal_part OR type_mark
         if Syntax_Node_Type (Node => Formal_Part_Node) = SPSymbols.formal_part then
            -- ASSUME Formal_Part_Node = formal_part
            Wf_Formal_Part
              (Node             => Formal_Part_Node,
               Current_Scope    => Current_Scope,
               Subprog_Sym      => Subprog_Sym,
               First_Occurrence => True,
               Context          => Dictionary.ProgramContext);
         elsif Syntax_Node_Type (Node => Formal_Part_Node) /= SPSymbols.type_mark then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Formal_Part_Node = formal_part OR type_mark in Wf_Subprogram_Declaration");
         end if;
         -- ASSUME Anno_Node = function_annotation OR NULL
         if Syntax_Node_Type (Node => Anno_Node) = SPSymbols.function_annotation then
            -- ASSUME Anno_Node = function_annotation
            Wf_Function_Annotation
              (Node          => Anno_Node,
               Current_Scope => Current_Scope,
               Subprog_Sym   => Subprog_Sym,
               First_Seen    => True);
         elsif Anno_Node /= STree.NullNode then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Anno_Node = function_annotation OR NULL in Wf_Subprogram_Declaration");
         end if;
         -- ASSUME Constraint_Node = function_constraint OR NULL
         if Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint then
            -- ASSUME Constraint_Node = function_constraint
            wf_function_constraint (Constraint_Node, Dictionary.LocalScope (Subprog_Sym), True);
         elsif Constraint_Node /= STree.NullNode then
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Constraint_Node = function_annotation OR NULL in Wf_Subprogram_Declaration");
         end if;
      end if;
      CheckNoOverloadingFromTaggedOps (Spec_Node, Subprog_Sym, Current_Scope, Dictionary.IsAbstract, Is_Overriding);
   elsif Syntax_Node_Type (Node => Spec_Node) = SPSymbols.proof_function_declaration then
      -- ASSUME Spec_Node = proof_function_declaration
      Wf_Proof_Function_Declaration (Node     => Spec_Node,
                                     Scope    => Current_Scope,
                                     Func_Sym => The_Subprog_Sym);
   else
      The_Subprog_Sym := Dictionary.NullSymbol;
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Spec_Node = procedure_specification OR function_specification OR proof_function_declaration OR entry_specification in Wf_Subprogram_Declaration");
   end if;
end Wf_Subprogram_Declaration;
