-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.wf_pragma)
procedure Wf_Attach_Handler (Pragma_Node : in STree.SyntaxNode) is
   -- Grammar: (wellformed case shown)
   --
   --          Pragma_Node
   --              |
   --          identifier --- argument_association_rep
   --                                   |
   --                         argument_association_rep --- argument_association
   --                                   |                          |
   --                          argument_association           ADA_expression
   --                                   |
   --                             ADA_expression
   -- Checks:
   --       (1) Positional association
   --       (2) Exactly 2 arguments
   --       (3) First is the procedure name
   --       (4) 2nd is ignore for now
   --       (5) pragma immediately follows procedure
   --       -- Rule 6 removed after design rethink (6) only one attach_handler per PT
   --       (7) procedure must be parameterless
   --       (8) must be in PT

   TheRegion      : Dictionary.Symbol;
   Proc_Spec_Node : STree.SyntaxNode;
   ProcIdent      : LexTokenManager.Lex_String;
   ProcSym        : Dictionary.Symbol;
   ErrorFound     : Boolean := False;

   procedure Find_Proc_Spec (Pragma_Node    : in     STree.SyntaxNode;
                             Proc_Spec_Node :    out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives Proc_Spec_Node from Pragma_Node,
   --#                             STree.Table;
   -- locates the place where a procedure_specification should be if the pragma is
   -- correctly placed.
   is
      Current_Node : STree.SyntaxNode;
   begin
      -- ASSUME Pragma_Node = apragma
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.apragma,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Pragma_Node = apragma in Find_Proc_Spec");

      -- There are two cases to consider: the attach_handler follows the first subprogram in the PT; or
      -- it follows some later subprogram declaration.
      -- Note that the protected_operation_declaration_rep grammar means that the sequence of declarations
      -- is "upside down" with the first declaration at the bottom.
      Current_Node := Child_Node (Current_Node => Parent_Node (Current_Node => Pragma_Node));
      -- protected_operation_declaration_rep to left of pragma

      if Child_Node (Current_Node) = STree.NullNode then
         -- The pragma is at the bottom of the sequence of protected_operation_declaration_reps and
         -- so we are dealing with FIRST subprogram in the PT (immediately after the priority pragma)

         -- Go to the top of the list of protected_operation_declaration_reps
         loop
            Current_Node := Parent_Node (Current_Node => Current_Node);
            exit when Syntax_Node_Type (Node => Current_Node) = SPSymbols.protected_operation_declaration;
         end loop;
         -- Move to procedure spec
         Proc_Spec_Node := Child_Node (Child_Node (Next_Sibling (Child_Node (Current_Node))));
      else
         -- we are dealing with a potential subprogram in the
         -- sequence of declarations in the PT declarative part
         Proc_Spec_Node := Child_Node (Next_Sibling (Child_Node (Current_Node)));
      end if;
   end Find_Proc_Spec;

   --------------------------------------------------------------------------------------

   procedure CheckArguments (Pragma_Node : in STree.SyntaxNode;
                             EntityName  : in LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorFound;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorFound                 from *,
   --#                                         EntityName,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         EntityName,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      ArgAssocRepNode : STree.SyntaxNode;
      SubprogNameNode : STree.SyntaxNode;

      procedure CheckRepresentSameName (ExpNode : in STree.SyntaxNode;
                                        Name    : in LexTokenManager.Lex_String)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     STree.Table;
      --#        in out ErrorFound;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorFound                 from *,
      --#                                         ExpNode,
      --#                                         LexTokenManager.State,
      --#                                         Name,
      --#                                         STree.Table &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ExpNode,
      --#                                         LexTokenManager.State,
      --#                                         Name,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table;
      is
         IsChain          : Boolean;
         IdNode, NextNode : STree.SyntaxNode;
      begin
         IdNode := ExpNode;
         loop
            IsChain  := Next_Sibling (IdNode) = STree.NullNode;
            NextNode := Child_Node (IdNode);
            exit when not IsChain or NextNode = STree.NullNode;

            IdNode := NextNode;
         end loop;

         if IsChain
           and then Syntax_Node_Type (Node => IdNode) = SPSymbols.identifier
           and then LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => Node_Lex_String (Node => IdNode),
            Lex_Str2 => Name) =
           LexTokenManager.Str_Eq then
            null;
         else -- Rule 3 failure
            ErrorFound := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 71,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => ExpNode),
               Id_Str    => Name);
         end if;
      end CheckRepresentSameName;

      --------------------------

   begin --CheckArguments
      ArgAssocRepNode := Child_Node (Next_Sibling (Child_Node (Pragma_Node)));
      if Syntax_Node_Type (Node => Child_Node (ArgAssocRepNode)) = SPSymbols.argument_association then --pragma has two arguments
         SubprogNameNode := Child_Node (Child_Node (ArgAssocRepNode));

         if Syntax_Node_Type (Node => SubprogNameNode) /= SPSymbols.ADA_expression then --form of expression wrong - Rule 1 failure
            ErrorFound := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 71,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => SubprogNameNode),
               Id_Str    => EntityName);
         else --form of expression ok so check name actually matches
            CheckRepresentSameName (SubprogNameNode, EntityName);
         end if;

      else --pragma does nor have exactly 2 arguments -- Rule 2 failure
         ErrorFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 69,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Pragma_Node),
            Id_Str    => LexTokenManager.Attach_Handler_Token);
      end if;
   end CheckArguments;

begin -- Wf_Attach_Handler

   -- ASSUME Pragma_Node = apragma
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Pragma_Node) = SPSymbols.apragma,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Pragma_Node = apragma in Wf_Attach_Handler");

   TheRegion := Dictionary.GetRegion (Scope);
   -- attach_Handler can only appear in the spec of a protected type
   if Dictionary.IsType (TheRegion) and then Dictionary.TypeIsProtected (TheRegion) then
      Find_Proc_Spec (Pragma_Node    => Pragma_Node,
                      Proc_Spec_Node => Proc_Spec_Node);
      -- ASSUME Proc_Spec_Node = procedure_specification
      if Syntax_Node_Type (Node => Proc_Spec_Node) = SPSymbols.procedure_specification then
         ProcIdent := Node_Lex_String (Node => Child_Node (Current_Node => Child_Node (Current_Node => Proc_Spec_Node)));
         CheckArguments (Pragma_Node, ProcIdent);
         if not ErrorFound then
            ProcSym := Dictionary.LookupItem (Name              => ProcIdent,
                                              Scope             => Scope,
                                              Context           => Dictionary.ProgramContext,
                                              Full_Package_Name => False);
            if Dictionary.GetNumberOfSubprogramParameters (ProcSym) = 0 then
               STree.Set_Node_Lex_String
                 (Sym  => ProcSym,
                  Node => Child_Node (Current_Node => Child_Node (Current_Node => Proc_Spec_Node)));
               Dictionary.SetIsInterruptHandler (ProcSym);
               Dictionary.SetTypeHasPragma (TheRegion, Dictionary.AttachHandler);
            else -- rule 7 failure
               ErrorHandler.Semantic_Error
                 (Err_Num   => 885,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Pragma_Node),
                  Id_Str    => LexTokenManager.Null_String);

            end if;
         end if; -- ErrorFound
      else -- rule 5 failure
         ErrorHandler.Semantic_Error
           (Err_Num   => 884,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Pragma_Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   else -- not in PT (Rule 8)
      ErrorHandler.Semantic_Error
        (Err_Num   => 884,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Pragma_Node),
         Id_Str    => LexTokenManager.Null_String);
   end if;
end Wf_Attach_Handler;
