-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure wf_named_argument_association
  (Node         : in     STree.SyntaxNode;
   Scope        : in     Dictionary.Scopes;
   EStack       : in out ExpStack.ExpStackType;
   HeapParam    : in out Lists.List_Heap;
   IsAnnotation : in     Boolean) is
   SIMPLE_LOOKUP : constant Annotation_Symbol_Table :=
     Annotation_Symbol_Table'(False => SPSymbols.simple_name,
                              True  => SPSymbols.annotation_simple_name);

   ExpResult, FunInfo                  : Exp_Record;
   FunSym, ParamSym                    : Dictionary.Symbol;
   IdentNode                           : STree.SyntaxNode;
   IdentStr                            : LexTokenManager.Lex_String;
   AlreadyPresent, NameIsParameterName : Boolean;
   UnusedValue                         : Maths.Value;
   ErrorFound                          : Boolean := False;

   --------------------------------------------------------------

   function ExpressionLocation (Node : STree.SyntaxNode) return STree.SyntaxNode
   --# global in IsAnnotation;
   --#        in STree.Table;
   is
      ExpLoc : STree.SyntaxNode;

   begin
      if Syntax_Node_Type (Node => Child_Node (Node)) = SIMPLE_LOOKUP (IsAnnotation) then
         ExpLoc := Next_Sibling (Child_Node (Node));
      else
         ExpLoc := Next_Sibling (Next_Sibling (Child_Node (Node)));
      end if;
      return ExpLoc;
   end ExpressionLocation;

   --------------------------------------------------------------

   function FindIdentifier (Node : STree.SyntaxNode) return STree.SyntaxNode
   --# global in IsAnnotation;
   --#        in STree.Table;
   is
      IdentNode : STree.SyntaxNode;
   begin
      if Syntax_Node_Type (Node => Child_Node (Node)) = SIMPLE_LOOKUP (IsAnnotation) then
         IdentNode := Child_Node (Child_Node (Node));
      else
         IdentNode := Child_Node (Next_Sibling (Child_Node (Node)));
      end if;
      return IdentNode;
   end FindIdentifier;

   --------------------------------------------------------------

   procedure RangeCheck (ARange : in Boolean;
                         Node   : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorFound;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorFound                 from *,
   --#                                         ARange &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from ARange,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
   begin
      if ARange then
         ErrorFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 341,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ExpressionLocation (Node)),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end RangeCheck;

   --------------------------------------------------------------

begin --wf_named_argument_association
   ExpStack.Pop (ExpResult, EStack);
   ExpStack.Pop (FunInfo, EStack);
   FunSym := FunInfo.Other_Symbol;

   FindNamedArgumentAssociationParameter
     (Node                => Node,
      SubprogSym          => FunSym,
      IsAnnotation        => IsAnnotation,
      NameIsParameterName => NameIsParameterName,
      ParamSym            => ParamSym);

   IdentNode := FindIdentifier (Node);
   IdentStr  := Node_Lex_String (Node => IdentNode);

   if NameIsParameterName then
      -- Seed syntax tree with expected type for run-time check;
      -- but, don't do this for instantiation of unchecked_conversion
      -- because we don't want any RTCs for association of those parameters
      -- (provided the function parameter subtype and actual subtype match)
      if not (Dictionary.IsAnUncheckedConversion (FunSym) and then ExpResult.Type_Symbol = Dictionary.GetType (ParamSym)) then
         STree.AddNodeSymbol (Node, Dictionary.GetType (ParamSym));
      end if;

      -- There is a special case involving functions an stream variables.  We allow a stream
      -- variable to be a parameter to an Unchecked_Conversion but need to ensure that
      -- the function inherits the restrictions associated with referencing a stream
      -- (e.g. cannot be used in gernal expression).  We can do this here by checking
      -- the StreamSymbol of the parameter expression (there will only be one if we are
      -- talking about an unchecked conversion) and if it is non-null then setting the
      -- stream symbol of the function result record (now an object) to the function symbol.
      -- Note that this clause will only be executed for an unchecked conversion because
      -- a parameter which is a stream would hav ebeen rejected at wf_primary in all other
      -- cases
      if ExpResult.Stream_Symbol /= Dictionary.NullSymbol then
         FunInfo.Stream_Symbol := FunSym;
      end if;

      AddName (IdentStr, FunInfo.Param_List, HeapParam, AlreadyPresent);
      if AlreadyPresent then
         ErrorFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 4,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => IdentNode),
            Id_Str    => IdentStr);
      else --not already present so do further checks
         RangeCheck (ExpResult.Is_ARange, Node);

         -- function is deemed constant if it is predefined and all its parameters
         -- are constant.
         FunInfo.Is_Constant := FunInfo.Is_Constant and ExpResult.Is_Constant;

         if (FunInfo.Tagged_Parameter_Symbol = ExpResult.Type_Symbol
               or else (FunInfo.Tagged_Parameter_Symbol = Dictionary.NullSymbol
                          and then Dictionary.CompatibleTypes (Scope, Dictionary.GetType (ParamSym), ExpResult.Type_Symbol))
               or else (not Dictionary.IsAnExtensionOf (ExpResult.Type_Symbol, FunInfo.Tagged_Parameter_Symbol)
                          and then Dictionary.CompatibleTypes (Scope, Dictionary.GetType (ParamSym), ExpResult.Type_Symbol))) then
            TaggedActualMustBeObjectCheck
              (NodePos         => Node_Position (Node => ExpressionLocation (Node)),
               FormalType      => Dictionary.GetType (ParamSym),
               ActualType      => ExpResult.Type_Symbol,
               ControllingType => Dictionary.GetSubprogramControllingType (FunSym),
               IsAVariable     => ExpResult.Is_AVariable,
               IsAConstant     => ExpResult.Is_Constant,
               ErrorFound      => ErrorFound);
            -- Following call will deal with scalar value constraint checking
            --# accept Flow, 10, UnusedValue, "Expected ineffective assignment";
            ConstraintCheck
              (ExpResult.Value,
               UnusedValue,
               IsAnnotation,
               Dictionary.GetType (ParamSym),
               Node_Position (Node => ExpressionLocation (Node)));
            --# end accept;
            -- Check array bounds etc.
            if Dictionary.TypeIsArray (Dictionary.GetType (ParamSym))
              and then not Dictionary.IsUnconstrainedArrayType (Dictionary.GetType (ParamSym)) then
               -- Formal is a constrained subtype of an unconstrained array
               if Dictionary.IsUnconstrainedArrayType (ExpResult.Type_Symbol) then

                  -- Actual is unconstrained.  In SPARK95 or 2005, this is OK if
                  -- the actual is a static String expression, but illegal
                  -- otherwise.
                  if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
                    and then Dictionary.IsPredefinedStringType (ExpResult.Type_Symbol) then
                     -- Formal must be a constrained String subtype, so we need
                     -- to check the upper bound of the actual against the expected
                     -- upper bound of the formal.
                     if ExpResult.Range_RHS = Maths.NoValue then
                        -- Actual is not static, so must be illegal
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 39,
                           Reference => ErrorHandler.No_Reference,
                           Position  => Node_Position (Node => ExpressionLocation (Node)),
                           Id_Str    => LexTokenManager.Null_String);
                     else
                        -- Actual is static, so check upper-bound against that expected
                        if ExpResult.Range_RHS /=
                          Maths.ValueRep
                          (Dictionary.GetScalarAttributeValue
                             (False,
                              LexTokenManager.Last_Token,
                              Dictionary.GetType (ParamSym))) then
                           ErrorHandler.Semantic_Error
                             (Err_Num   => 418,
                              Reference => ErrorHandler.No_Reference,
                              Position  => Node_Position (Node => ExpressionLocation (Node)),
                              Id_Str    => LexTokenManager.Null_String);

                        end if;
                     end if;
                  else
                     -- SPARK83 or not a String type, so illegal
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 39,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => ExpressionLocation (Node)),
                        Id_Str    => LexTokenManager.Null_String);
                  end if;

               elsif Illegal_Unconstrained (ExpResult.Type_Symbol, Dictionary.GetType (ParamSym)) then
                  -- Although both formal and actual are constrained their bounds don't match
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 418,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => ExpressionLocation (Node)),
                     Id_Str    => LexTokenManager.Null_String);
               end if;
            end if;

            -- To help the VCG with generating checks involving unconstrained formal parameters, we
            -- seed the syntax tree with a constraining type mark.  The positional_argument_association
            -- node is already used for RTC purposes, so we seed the expression node instead.
            if not IsAnnotation then
               PlantConstrainingType
                 (ExpResult.Type_Symbol,
                  ExpResult.Range_RHS,
                  STree.ExpressionFromNamedArgumentAssociation (Node));
            end if;
         else
            ErrorFound := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 38,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => ExpressionLocation (Node)),
               Id_Str    => LexTokenManager.Null_String);
         end if;
      end if;
   else
      ErrorFound := True;
      ErrorHandler.Semantic_Error_Lex1_Sym1
        (Err_Num   => 2,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => IdentNode),
         Id_Str    => IdentStr,
         Sym       => FunSym,
         Scope     => Scope);

   end if;
   FunInfo.Errors_In_Expression := ErrorFound or FunInfo.Errors_In_Expression or ExpResult.Errors_In_Expression;
   ExpStack.Push (FunInfo, EStack);

   --# accept Flow, 33, UnusedValue, "Expected to be neither referenced nor exported";
end wf_named_argument_association;
