-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

-- Grammar:
-- Node passed in is the identifier node of the generic unit being instantiated
-- The node to the right of this is the generic_actual_part if it exists.
-- Thus:
--   ... ... ... identifier [generic_actual_part]
--                                    |
--                             name_argument_list
--
--
--  name_argument_list :
--        named_argument_association
--      | positional_argument_association ;
--
--  named_argument_association :
--        named_argument_association simple_name expression
--      | simple_name expression ;
--
--  positional_argument_association :
--        positional_argument_association expression
--      | expression ;
--
-- Wffs:
-- 1. If generic has formal parameters then actual part must exist.
-- 2. If generic has not formals then actual part must not exist.
-- 3. There must be one actual for each formal.
-- 4. Each actual must be of the right type and kind (i.e. object or type).
--
-- Actions:
-- Wellformed formal/actual associations are written to the Dictionary
-- and associated with the InstantiationSym.

separate (Sem.CompUnit)
procedure wf_generic_actual_part
  (GenericNode      : in     STree.SyntaxNode;
   GenericSym       : in     Dictionary.Symbol;
   InstantiationSym : in     Dictionary.Symbol;
   Scope            : in     Dictionary.Scopes;
   ErrorFound       :    out Boolean) is
   ActualPartNode  : STree.SyntaxNode;
   ErrorFoundLocal : Boolean := False;

   procedure CheckMissingActualPart
     (GenericNode    : in     STree.SyntaxNode;
      ActualPartNode : in     STree.SyntaxNode;
      GenericSym     : in     Dictionary.Symbol;
      ErrorFound     : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorFound                 from *,
   --#                                         ActualPartNode,
   --#                                         Dictionary.Dict,
   --#                                         GenericSym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from ActualPartNode,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GenericNode,
   --#                                         GenericSym,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
   begin
      if Dictionary.GetNumberOfGenericFormalParameters (GenericSym) > 0 and then ActualPartNode = STree.NullNode then
         ErrorHandler.Semantic_Error
           (Err_Num   => 633,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => GenericNode),
            Id_Str    => Node_Lex_String (Node => GenericNode));
         ErrorFound := True;
      end if;
   end CheckMissingActualPart;

   ---------------------------

   procedure CheckSuperfluousActualPart
     (GenericNode    : in     STree.SyntaxNode;
      ActualPartNode : in     STree.SyntaxNode;
      GenericSym     : in     Dictionary.Symbol;
      ErrorFound     : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorFound                 from *,
   --#                                         ActualPartNode,
   --#                                         Dictionary.Dict,
   --#                                         GenericSym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from ActualPartNode,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GenericNode,
   --#                                         GenericSym,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
   begin
      if Dictionary.GetNumberOfGenericFormalParameters (GenericSym) = 0 and then ActualPartNode /= STree.NullNode then
         ErrorHandler.Semantic_Error
           (Err_Num   => 634,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ActualPartNode),
            Id_Str    => Node_Lex_String (Node => GenericNode));
         ErrorFound := True;
      end if;
   end CheckSuperfluousActualPart;

   ---------------------------

   procedure ProcessGenericTypeParameter
     (FormalSym        : in     Dictionary.Symbol;
      ActualExpression : in     STree.SyntaxNode;
      InstantiationSym : in     Dictionary.Symbol;
      Scope            : in     Dictionary.Scopes;
      ErrorFound       : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         ActualExpression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict,
   --#         ErrorFound                 from *,
   --#                                         ActualExpression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         FormalSym,
   --#                                         GlobalComponentData,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from ActualExpression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FormalSym,
   --#                                         GlobalComponentData,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State                  from *,
   --#                                         ActualExpression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      ExpressionResult : Exp_Record;
      IsAName          : Boolean;
      UnusedRefVars    : SeqAlgebra.Seq;
      ActualType       : Dictionary.Symbol;

      procedure CheckTypeCompatibility
        (ErrorPosition    : in     LexTokenManager.Token_Position;
         FormalType       : in     Dictionary.Symbol;
         InstantiationSym : in     Dictionary.Symbol;
         Scope            : in     Dictionary.Scopes;
         ActualType       : in out Dictionary.Symbol;
         ErrorFound       : in out Boolean)

      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives ActualType,
      --#         ErrorFound                 from *,
      --#                                         ActualType,
      --#                                         Dictionary.Dict,
      --#                                         FormalType,
      --#                                         InstantiationSym,
      --#                                         Scope &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from ActualType,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ErrorPosition,
      --#                                         FormalType,
      --#                                         InstantiationSym,
      --#                                         LexTokenManager.State,
      --#                                         Scope,
      --#                                         SPARK_IO.File_Sys;
      is

         FormalElementOrIndex : Dictionary.Symbol;
         ActualElementOrIndex : Dictionary.Symbol;
         FormalIndexIterator  : Dictionary.Iterator;
         ActualIndexIterator  : Dictionary.Iterator;
         LocalErrorFound      : Boolean := False;

         function IsValidAssociation (FormalType : Dictionary.Symbol;
                                      ActualType : Dictionary.Symbol) return Boolean
         --# global in Dictionary.Dict;
         --#        in InstantiationSym;
         --#        in Scope;
         is
            Result                         : Boolean;
            PreviouslyAssociatedActualType : Dictionary.Symbol;
            FormalTypeLocal                : Dictionary.Symbol;
         begin
            -- If formal type has already been associated (i.e. earlier in the parameter list), then
            -- get the previously given actual before doing the rest of the checks
            PreviouslyAssociatedActualType := Dictionary.ActualOfGenericFormal (FormalType, InstantiationSym);
            if PreviouslyAssociatedActualType /= Dictionary.NullSymbol then
               -- there is an earlier association for us to use
               FormalTypeLocal := PreviouslyAssociatedActualType;
            else
               -- no previous association so just run check on the formal type itself
               FormalTypeLocal := FormalType;
            end if;

            --PNA--14/12/05 as a result of above addition, case 2 below may be no longer needed

            -- Now go on to check the element and indexes thus:
            --   1. If the formal is a generic type then the actual must be a valid association
            --   2. If the formal is a generic formal type parameter then the actual must be the
            --      actual associated with that type earlier in the checking of this actual part
            --   3. If the formal is a predefined Ada type then the actual must identical.
            if Dictionary.TypeIsGeneric (FormalTypeLocal) then
               --PNA-- and then not Dictionary.TypeIsGenericArray (FormalTypeLocal) then
               -- Case 1.  Primitive generic types
               Result := Dictionary.IsValidGenericTypeAssociation (FormalTypeLocal, ActualType, Scope);
               --PNA-- elsif Dictionary.TypeIsGenericArray (FormalTypeLocal) then
               --PNA--    -- Case 2.  We are processing an array of arrays.
               --PNA--    Result := ActualType = Dictionary.ActualOfGenericFormal (FormalTypeLocal,
               --PNA--                                                             InstantiationSym);
            else -- straight Ada type
               Result := ActualType = FormalTypeLocal;

            end if;
            return Result;
         end IsValidAssociation;

      begin -- CheckTypeCompatibility
         if Dictionary.TypeIsGenericArray (FormalType) then
            if Dictionary.TypeIsArray (ActualType) then
               -- Both are arrays so check elements and indexes for compatibility
               -- First check element type
               FormalElementOrIndex := Dictionary.GetArrayComponent (FormalType);
               ActualElementOrIndex := Dictionary.GetArrayComponent (ActualType);

               if not IsValidAssociation (FormalElementOrIndex, ActualElementOrIndex) then
                  ErrorHandler.Semantic_Error2
                    (Err_Num   => 645,
                     Reference => ErrorHandler.No_Reference,
                     Position  => ErrorPosition,
                     Id_Str1   => Dictionary.GetSimpleName (ActualElementOrIndex),
                     Id_Str2   => Dictionary.GetSimpleName (FormalElementOrIndex));
                  LocalErrorFound := True;
               end if;
               -- now loop through indexes
               FormalIndexIterator := Dictionary.FirstArrayIndex (FormalType);
               ActualIndexIterator := Dictionary.FirstArrayIndex (ActualType);
               loop
                  exit when (FormalIndexIterator = Dictionary.NullIterator)
                    and then (ActualIndexIterator = Dictionary.NullIterator); -- normal exit

                  if FormalIndexIterator = Dictionary.NullIterator then
                     -- Formal has finished first
                     LocalErrorFound := True;
                     ErrorHandler.Semantic_Error2
                       (Err_Num   => 647,
                        Reference => ErrorHandler.No_Reference,
                        Position  => ErrorPosition,
                        Id_Str1   => Dictionary.GetSimpleName (ActualType),
                        Id_Str2   => Dictionary.GetSimpleName (FormalType));
                     exit;
                  end if;

                  if ActualIndexIterator = Dictionary.NullIterator then
                     -- Actual has finished first
                     LocalErrorFound := True;
                     ErrorHandler.Semantic_Error2
                       (Err_Num   => 648,
                        Reference => ErrorHandler.No_Reference,
                        Position  => ErrorPosition,
                        Id_Str1   => Dictionary.GetSimpleName (ActualType),
                        Id_Str2   => Dictionary.GetSimpleName (FormalType));
                     exit;
                  end if;

                  -- neither finished, so check they are compatible
                  FormalElementOrIndex := Dictionary.CurrentSymbol (FormalIndexIterator);
                  ActualElementOrIndex := Dictionary.CurrentSymbol (ActualIndexIterator);
                  if not IsValidAssociation (FormalElementOrIndex, ActualElementOrIndex) then
                     ErrorHandler.Semantic_Error2
                       (Err_Num   => 646,
                        Reference => ErrorHandler.No_Reference,
                        Position  => ErrorPosition,
                        Id_Str1   => Dictionary.GetSimpleName (ActualElementOrIndex),
                        Id_Str2   => Dictionary.GetSimpleName (FormalElementOrIndex));
                     LocalErrorFound := True;
                  end if;

                  FormalIndexIterator := Dictionary.NextSymbol (FormalIndexIterator);
                  ActualIndexIterator := Dictionary.NextSymbol (ActualIndexIterator);
               end loop;

               -- Both formal and actual must be same constrained/unconstrained
               if Dictionary.IsUnconstrainedArrayType (FormalType)
                 and then not Dictionary.IsUnconstrainedArrayType (ActualType) then
                  LocalErrorFound := True;
                  ErrorHandler.Semantic_Error2
                    (Err_Num   => 649,
                     Reference => ErrorHandler.No_Reference,
                     Position  => ErrorPosition,
                     Id_Str1   => Dictionary.GetSimpleName (ActualType),
                     Id_Str2   => Dictionary.GetSimpleName (FormalType));
               end if;
               -- reverse check
               if Dictionary.IsUnconstrainedArrayType (ActualType)
                 and then not Dictionary.IsUnconstrainedArrayType (FormalType) then
                  LocalErrorFound := True;
                  ErrorHandler.Semantic_Error2
                    (Err_Num   => 650,
                     Reference => ErrorHandler.No_Reference,
                     Position  => ErrorPosition,
                     Id_Str1   => Dictionary.GetSimpleName (ActualType),
                     Id_Str2   => Dictionary.GetSimpleName (FormalType));
               end if;
            else
               -- Formal is an array but actual isn't, so definitely wrong
               ErrorHandler.Semantic_Error2
                 (Err_Num   => 636,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPosition,
                  Id_Str1   => Dictionary.GetSimpleName (ActualType),
                  Id_Str2   => Dictionary.GetSimpleName (FormalType));
               LocalErrorFound := True;
            end if;
         else -- Formal is not an array so just check actual against formal
            if not Dictionary.IsValidGenericTypeAssociation (FormalType, ActualType, Scope) then
               ErrorHandler.Semantic_Error2
                 (Err_Num   => 636,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPosition,
                  Id_Str1   => Dictionary.GetSimpleName (ActualType),
                  Id_Str2   => Dictionary.GetSimpleName (FormalType));
               LocalErrorFound := True;
            end if;
         end if;

         if LocalErrorFound then
            ErrorFound := True;
            ActualType := Dictionary.GetUnknownTypeMark;
         end if;
      end CheckTypeCompatibility;

   begin -- ProcessGenericTypeParameter

      -- The actual parameter takes the form of an expression but must actually be a typemark.
      -- We can use WalkName to recover the symbol, then check that it is a type and that it
      -- compatible with generic type.  Matching pairs are added to the dictionary and associated
      -- with the symbol of the instantiated unit.
      SeqAlgebra.CreateSeq (TheHeap, UnusedRefVars);
      Walk_Name
        (Exp_Node       => ActualExpression,
         Scope          => Scope,
         Component_Data => GlobalComponentData,
         Result         => ExpressionResult,
         Is_A_Name      => IsAName,
         Ref_Var_Param  => UnusedRefVars);
      SeqAlgebra.DisposeOfSeq (TheHeap, UnusedRefVars);
      if not IsAName then
         -- we have a general expression which cannot possibly be a type mark
         ErrorFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 95,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ActualExpression),
            Id_Str    => LexTokenManager.Null_String);
      elsif ExpressionResult.Sort /= Is_Type_Mark then
         -- we have a name but it's not a type mark
         ErrorFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 95,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ActualExpression),
            Id_Str    => LexTokenManager.Null_String);
      else
         -- we have a type mark
         ActualType := ExpressionResult.Type_Symbol;
         CheckTypeCompatibility
           (Node_Position (Node => ActualExpression),
            Dictionary.GetType (FormalSym),
            InstantiationSym,
            Scope,
            ActualType,
            ErrorFound);
         -- above call will change ActualType to the UnknownType if there is an error
         Dictionary.AddGenericAssociation
           (SubprogramOrPackage => InstantiationSym,
            Comp_Unit           => ContextManager.Ops.Current_Unit,
            Declaration         => Dictionary.Location'(Start_Position => Node_Position (Node => ActualExpression),
                                                        End_Position   => Node_Position (Node => ActualExpression)),
            FormalSym           => Dictionary.GetType (FormalSym),
            ActualSym           => ActualType);

      end if;
   end ProcessGenericTypeParameter;

   ---------------------------

   procedure ProcessGenericObjectParameter
     (FormalSym        : in     Dictionary.Symbol;
      ActualExpression : in     STree.SyntaxNode;
      InstantiationSym : in     Dictionary.Symbol;
      Scope            : in     Dictionary.Scopes;
      ErrorFound       : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from ActualExpression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         FormalSym,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorFound,
   --#         Statistics.TableUsage      from *,
   --#                                         ActualExpression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         FormalSym,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from ActualExpression,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FormalSym,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      ActualType          : Dictionary.Symbol;
      UnusedComponentData : ComponentManager.ComponentData;
      UnwantedSeq         : SeqAlgebra.Seq;
      ExpResult           : Exp_Record;
      ScalarStoreVal      : LexTokenManager.Lex_String;
      ConstantName        : LexTokenManager.Lex_String;
      ConstantSym         : Dictionary.Symbol;
      ConstantLocation    : Dictionary.Location;
   begin

      -- genericTBD -- do we allow formal objects that are arrays?

      ActualType := Dictionary.GetType (FormalSym);
      -- The ActualType we have got here might be a predefined Ada type such as Integer.
      -- If so, then that is the type we expect the actual expression to be; however,
      -- if the type is a generic type then we need to look up the actual associated with it:
      if Dictionary.TypeIsGeneric (ActualType) then
         ActualType := Dictionary.ActualOfGenericFormal (ActualType, InstantiationSym);
      end if;

      -- walk expression
      SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
      ComponentManager.Initialise (UnusedComponentData);
      --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
      WalkExpression
        (Exp_Node                => ActualExpression,
         Scope                   => Scope,
         Type_Context            => ActualType,
         Context_Requires_Static => False,
         Result                  => ExpResult,
         Ref_Var                 => UnwantedSeq,
         Component_Data          => UnusedComponentData);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
      Maths.StorageRep (ExpResult.Value, ScalarStoreVal);        -- scalar value if needed later

      -- check constant
      if ExpResult.Is_Constant then
         AssignmentCheck (Node_Position (Node => ActualExpression), Scope, ActualType, ExpResult);
         if ExpResult = UnknownTypeRecord then
            ErrorFound := True;
         else
            -- Fundamentally ok to add
            -- mark any errors (both to avoid use of invalid instantiation and to supress rule generation)
            if ExpResult.Errors_In_Expression then
               ErrorFound := True;
            end if;

            -- first add a constant declaration to local scope of InstantiationSym
            -- synthetic constant can have same name as formal it is associated with
            ConstantName := Dictionary.GetSimpleName (FormalSym);
            -- and can be "located" at expression node
            ConstantLocation :=
              Dictionary.Location'
              (Start_Position => Node_Position (Node => ActualExpression),
               End_Position   => Node_Position (Node => ActualExpression));

            if Dictionary.IsScalarTypeMark (ActualType, Scope) then
               Dictionary.AddScalarConstant
                 (Name            => ConstantName,
                  TypeMark        => ActualType,
                  TypeReference   => ConstantLocation,
                  Value           => ScalarStoreVal,
                  ExpIsWellFormed => not ExpResult.Errors_In_Expression,
                  ExpNode         => STree.NodeToRef (ActualExpression),
                  Static          => ExpResult.Is_Static,
                  Comp_Unit       => ContextManager.Ops.Current_Unit,
                  Declaration     => ConstantLocation,
                  Scope           => Dictionary.LocalScope (InstantiationSym),
                  Context         => Dictionary.ProofContext);
            elsif Dictionary.IsArrayTypeMark (ActualType, Scope) then
               Dictionary.AddArrayConstant
                 (Name            => ConstantName,
                  TypeMark        => ActualType,
                  TypeReference   => ConstantLocation,
                  Value           => LexTokenManager.Null_String,
                  ExpIsWellFormed => not ExpResult.Errors_In_Expression,
                  ExpNode         => STree.NodeToRef (ActualExpression),
                  Static          => ExpResult.Is_Static,
                  Comp_Unit       => ContextManager.Ops.Current_Unit,
                  Declaration     => ConstantLocation,
                  Scope           => Dictionary.LocalScope (InstantiationSym),
                  Context         => Dictionary.ProofContext);
            else -- must be record
               Dictionary.AddRecordConstant
                 (Name            => ConstantName,
                  TheType         => ActualType,
                  TypeReference   => ConstantLocation,
                  ExpIsWellFormed => not ExpResult.Errors_In_Expression,
                  ExpNode         => STree.NodeToRef (ActualExpression),
                  Comp_Unit       => ContextManager.Ops.Current_Unit,
                  Declaration     => ConstantLocation,
                  Scope           => Dictionary.LocalScope (InstantiationSym),
                  Context         => Dictionary.ProofContext);
            end if;

            -- have to get constant symbol back via lookup
            ConstantSym :=
              Dictionary.LookupItem
              (Name              => ConstantName,
               Scope             => Dictionary.LocalScope (InstantiationSym),
               Context           => Dictionary.ProofContext,
               Full_Package_Name => False);

            -- it must be there!
            SystemErrors.RT_Assert
              (C       => ConstantSym /= Dictionary.NullSymbol,
               Sys_Err => SystemErrors.Assertion_Failure,
               Msg     => "constant added at subprogram instantiation not found in wf_generic_actual_part");

            Dictionary.AddGenericAssociation
              (SubprogramOrPackage => InstantiationSym,
               Comp_Unit           => ContextManager.Ops.Current_Unit,
               Declaration         => Dictionary.Location'(Start_Position => Node_Position (Node => ActualExpression),
                                                           End_Position   => Node_Position (Node => ActualExpression)),
               FormalSym           => FormalSym,
               ActualSym           => ConstantSym);
         end if;
      else
         ErrorFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 640,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ActualExpression),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end ProcessGenericObjectParameter;

   ---------------------------

   procedure ProcessPositionalArgumentAssociation
     (PosArgAssNode    : in     STree.SyntaxNode;
      GenericSym       : in     Dictionary.Symbol;
      InstantiationSym : in     Dictionary.Symbol;
      Scope            : in     Dictionary.Scopes;
      ErrorFound       : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         ErrorFound,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GenericSym,
   --#                                         GlobalComponentData,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         PosArgAssNode,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GenericSym,
   --#                                         GlobalComponentData,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         PosArgAssNode,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      FormalIt       : Dictionary.Iterator;
      ActualIt       : STree.Iterator;
      CurrentFormal  : Dictionary.Symbol;
      FormalKind     : Dictionary.GenericParameterKind;
      ExpressionNode : STree.SyntaxNode;
   begin
      FormalIt       := Dictionary.FirstGenericFormalParameter (GenericSym);
      ActualIt       :=
        Find_First_Node (Node_Kind    => SPSymbols.expression,
                         From_Root    => PosArgAssNode,
                         In_Direction => STree.Down);
      ExpressionNode := PosArgAssNode; -- default value in case loop below is skipped

      while not Dictionary.IsNullIterator (FormalIt) and then not STree.IsNull (ActualIt) loop

         -- gather up formal data
         CurrentFormal := Dictionary.CurrentSymbol (FormalIt);
         FormalKind    := Dictionary.GetGenericFormalParameterKind (CurrentFormal);

         --gather up actual data
         ExpressionNode := Get_Node (It => ActualIt);

         --wff them together here
         case FormalKind is
            when Dictionary.GenericTypeParameter =>
               ProcessGenericTypeParameter
                 (FormalSym        => CurrentFormal,
                  ActualExpression => ExpressionNode,
                  InstantiationSym => InstantiationSym,
                  Scope            => Scope,
                  ErrorFound       => ErrorFound);

            when Dictionary.GenericObjectParameter =>
               ProcessGenericObjectParameter
                 (FormalSym        => CurrentFormal,
                  ActualExpression => ExpressionNode,
                  InstantiationSym => InstantiationSym,
                  Scope            => Scope,
                  ErrorFound       => ErrorFound);
         end case;

         FormalIt := Dictionary.NextSymbol (FormalIt);
         ActualIt := STree.NextNode (ActualIt);
      end loop;

      -- completeness check, both loops should run out at same time
      if not Dictionary.IsNullIterator (FormalIt) or not STree.IsNull (ActualIt) then
         ErrorFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 635,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ExpressionNode),
            Id_Str    => Dictionary.GetSimpleName (GenericSym));
      end if;
   end ProcessPositionalArgumentAssociation;

   ---------------------------

   procedure ProcessNamedArgumentAssociation
     (NamArgAssNode    : in     STree.SyntaxNode;
      GenericSym       : in     Dictionary.Symbol;
      InstantiationSym : in     Dictionary.Symbol;
      Scope            : in     Dictionary.Scopes;
      ErrorFound       : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         ErrorFound,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GenericSym,
   --#                                         GlobalComponentData,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         NamArgAssNode,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GenericSym,
   --#                                         GlobalComponentData,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         NamArgAssNode,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      FormalIt       : Dictionary.Iterator;
      CurrentFormal  : Dictionary.Symbol;
      FormalKind     : Dictionary.GenericParameterKind;
      ExpressionNode : STree.SyntaxNode;
   begin
      -- Check that each named assoication appears and appears once only
      CheckNamedAssociation (TheFormals             => GenericSym,
                             Scope                  => Scope,
                             NamedArgumentAssocNode => NamArgAssNode);

      -- now process each formal in turn
      FormalIt := Dictionary.FirstGenericFormalParameter (GenericSym);
      while not Dictionary.IsNullIterator (FormalIt) loop
         -- gather up formal data
         CurrentFormal := Dictionary.CurrentSymbol (FormalIt);
         FormalKind    := Dictionary.GetGenericFormalParameterKind (CurrentFormal);

         -- gather up actual data
         ExpressionNode := FindActualNode (For_Formal                => CurrentFormal,
                                           Named_Argument_Assoc_Node => NamArgAssNode);

         if ExpressionNode /= STree.NullNode then
            -- wff them together here
            case FormalKind is
               when Dictionary.GenericTypeParameter =>
                  ProcessGenericTypeParameter
                    (FormalSym        => CurrentFormal,
                     ActualExpression => ExpressionNode,
                     InstantiationSym => InstantiationSym,
                     Scope            => Scope,
                     ErrorFound       => ErrorFound);

               when Dictionary.GenericObjectParameter =>
                  ProcessGenericObjectParameter
                    (FormalSym        => CurrentFormal,
                     ActualExpression => ExpressionNode,
                     InstantiationSym => InstantiationSym,
                     Scope            => Scope,
                     ErrorFound       => ErrorFound);
            end case;
         end if;

         FormalIt := Dictionary.NextSymbol (FormalIt);
      end loop;

   end ProcessNamedArgumentAssociation;

   ---------------------------

begin -- wf_generic_actual_part
   ActualPartNode := Next_Sibling (GenericNode);
   CheckMissingActualPart (GenericNode, ActualPartNode, GenericSym, ErrorFoundLocal);

   CheckSuperfluousActualPart (GenericNode, ActualPartNode, GenericSym, ErrorFoundLocal);

   if not ErrorFoundLocal and then Dictionary.GetNumberOfGenericFormalParameters (GenericSym) /= 0 then
      -- a generic actual part is both present and needed
      ActualPartNode := Child_Node (Child_Node (ActualPartNode));
      if Syntax_Node_Type (Node => ActualPartNode) = SPSymbols.named_argument_association then
         ProcessNamedArgumentAssociation (ActualPartNode, GenericSym, InstantiationSym, Scope, ErrorFoundLocal);
      else -- must be positional
         ProcessPositionalArgumentAssociation (ActualPartNode, GenericSym, InstantiationSym, Scope, ErrorFoundLocal);
      end if;
   end if;
   ErrorFound := ErrorFoundLocal;
end wf_generic_actual_part;
