-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.WalkStatements)
procedure wf_case_choice (Node  : in STree.SyntaxNode;
                          Scope : in Dictionary.Scopes) is
   type CaseChoiceSorts is (SingleExpression, ExplicitRange, RangeConstraint);

   CaseChoiceSort            : CaseChoiceSorts;
   First_Node, Second_Node   : STree.SyntaxNode;
   FirstResult, SecondResult : Exp_Record;
   RefVar                    : SeqAlgebra.Seq;

   CaseFlags          : Typ_Case_Flags;
   CompleteADT        : CompleteCheck.T;
   CaseTypeSymbol     : Dictionary.Symbol;
   CaseTypeLowerBound : Typ_Type_Bound;
   CaseTypeUpperBound : Typ_Type_Bound;

   IsARange                           : Boolean;     -- these refer to the value/range
   ChoiceLowerMathsValue              : Maths.Value; -- specified in the case choice
   ChoiceUpperMathsValue              : Maths.Value    := Maths.NoValue;  -- init to remove spurious flowerrs
   ChoiceLowerBound, ChoiceUpperBound : Typ_Type_Bound := Unknown_Type_Bound;  -- init to remove spurious flowerrs

   LowerBoundUnknown    : Boolean;
   UpperBoundUnknown    : Boolean := False;
   LowerBoundOutOfRange : Boolean;
   UpperBoundOutOfRange : Boolean := False;

   SemanticErrorsFound : Boolean := False;

   OutOfRangeSeen            : Boolean;
   OverlapSeen               : CompleteCheck.TypOverlapState;
   BothChoiceBoundsKnown     : Boolean := False;
   RangeConstraintLowerBound : Typ_Type_Bound;
   RangeConstraintUpperBound : Typ_Type_Bound;

   ------------------------------------------------------------------------

   procedure ConvertChoiceBound
     (MathsValue      : in     Maths.Value;
      Bound           :    out Typ_Type_Bound;
      UnknownBound    :    out Boolean;
      BoundOutOfRange :    out Boolean)
   --# derives Bound,
   --#         BoundOutOfRange,
   --#         UnknownBound    from MathsValue;
   is
      Int        : Integer;
      MathsError : Maths.ErrorCode;
   begin
      if Maths.HasNoValue (MathsValue) then
         Bound           := Typ_Type_Bound'(Value      => 0,
                                            Is_Defined => False);
         UnknownBound    := True;
         BoundOutOfRange := False;
      else
         Maths.ValueToInteger (MathsValue, Int, MathsError);
         if MathsError = Maths.NoError then
            Bound           := Typ_Type_Bound'(Value      => Int,
                                               Is_Defined => True);
            UnknownBound    := False;
            BoundOutOfRange := False;
         else
            Bound           := Typ_Type_Bound'(Value      => 0,
                                               Is_Defined => False);
            UnknownBound    := False;
            BoundOutOfRange := True;
         end if;
      end if;
   end ConvertChoiceBound;

   ------------------------------------------------------------------------
   -- note: returns True if any of the bounds is undefined, unless the
   -- choice is not a range, in which case, ChoiceUpper is unused
   function IsChoiceInRange
     (ChoiceLower   : Typ_Type_Bound;
      ChoiceUpper   : Typ_Type_Bound;
      ChoiceIsRange : Boolean;
      RangeLower    : Typ_Type_Bound;
      RangeUpper    : Typ_Type_Bound)
     return          Boolean
   is
      Result : Boolean;
   begin
      if (ChoiceLower.Is_Defined and RangeLower.Is_Defined and ChoiceLower.Value < RangeLower.Value) or
        (ChoiceLower.Is_Defined and RangeUpper.Is_Defined and ChoiceLower.Value > RangeUpper.Value) or
        (ChoiceIsRange and ChoiceUpper.Is_Defined and RangeUpper.Is_Defined and ChoiceUpper.Value > RangeUpper.Value) then
         Result := False;
      else
         Result := True;
      end if;

      return Result;
   end IsChoiceInRange;

   ------------------------------------------------------------------------

   function IsRangeEmpty (RangeLower : Typ_Type_Bound;
                          RangeUpper : Typ_Type_Bound) return Boolean is
      Result : Boolean;
   begin
      if RangeLower.Is_Defined and RangeUpper.Is_Defined and RangeLower.Value > RangeUpper.Value then
         Result := True;
      else
         Result := False;
      end if;

      return Result;
   end IsRangeEmpty;

   -----------------------------------------------------------------------

   procedure ConvertBooleanMathsValue (Value : in out Maths.Value)
   --# derives Value from *;
   is
   begin
      if Value = Maths.FalseValue then
         Value := Maths.ZeroInteger;
      elsif Value = Maths.TrueValue then
         Value := Maths.OneInteger;
      end if;
   end ConvertBooleanMathsValue;

begin -- wf_case_choice

   -- ASSUME Node = case_choice
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.case_choice,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = case_choice in Wf_Case_Choice");

   SecondResult := UnknownTypeRecord;
   Case_Stack.Pop (CaseFlags, CompleteADT, CaseTypeSymbol, CaseTypeLowerBound, CaseTypeUpperBound);

   First_Node := Child_Node (Current_Node => Node);
   -- ASSUME First_Node = simple_expression
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => First_Node) = SPSymbols.simple_expression,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect First_Node = simple_expression in Wf_Case_Choice");

   SeqAlgebra.CreateSeq (TheHeap, RefVar);
   WalkExpression
     (Exp_Node                => First_Node,
      Scope                   => Scope,
      Type_Context            => CaseTypeSymbol,
      Context_Requires_Static => False,
      Result                  => FirstResult,
      Ref_Var                 => RefVar,
      Component_Data          => GlobalComponentData);

   --# assert True;

   Second_Node := Next_Sibling (Current_Node => First_Node);
   -- ASSUME Second_Node = range_constraint OR simple_expression OR NULL
   if Second_Node = STree.NullNode then
      -- ASSUME Second_Node = NULL
      CaseChoiceSort := SingleExpression;
   else
      WalkExpression
        (Exp_Node                => Second_Node,
         Scope                   => Scope,
         Type_Context            => CaseTypeSymbol,
         Context_Requires_Static => False,
         Result                  => SecondResult,
         Ref_Var                 => RefVar,
         Component_Data          => GlobalComponentData);
      -- ASSUME Second_Node = range_constraint OR simple_expression
      if Syntax_Node_Type (Node => Second_Node) = SPSymbols.simple_expression then
         -- ASSUME Second_Node = simple_expression
         CaseChoiceSort := ExplicitRange;
      elsif Syntax_Node_Type (Node => Second_Node) = SPSymbols.range_constraint then
         -- ASSUME Second_Node = range_constraint
         CaseChoiceSort := RangeConstraint;
      else
         CaseChoiceSort := SingleExpression;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Second_Node = range_constraint OR simple_expression in Wf_Case_Choice");
      end if;
   end if;

   --# assert True;

   case CaseChoiceSort is
      when SingleExpression =>
         if not FirstResult.Is_Static then
            ErrorHandler.Semantic_Error
              (Err_Num   => 36,
               Reference => 1,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         if not Dictionary.CompatibleTypes (Scope, FirstResult.Type_Symbol, CaseTypeSymbol) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 38,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;

         -- code to work out whether we have a single choice or a
         -- range and to collect the appropriate values
         -- note that these will be nonsense if semantic errors have been found
         ChoiceLowerMathsValue := FirstResult.Value;
         if FirstResult.Is_ARange then
            IsARange              := True;
            ChoiceUpperMathsValue := FirstResult.Range_RHS;
         else
            IsARange := False;
         end if;

      when ExplicitRange =>
         if not Dictionary.CompatibleTypes (Scope, FirstResult.Type_Symbol, SecondResult.Type_Symbol) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 42,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Second_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         elsif not Dictionary.CompatibleTypes (Scope, FirstResult.Type_Symbol, CaseTypeSymbol) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 106,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         if not (FirstResult.Is_Static and SecondResult.Is_Static) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 45,
               Reference => 1,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;

         -- code to collect the appropriate values for the extent of the range
         -- note that these will be nonsense if semantic errors have been found
         ChoiceLowerMathsValue := FirstResult.Value;
         ChoiceUpperMathsValue := SecondResult.Value;
         IsARange              := True;

      when RangeConstraint =>
         if not Dictionary.CompatibleTypes (Scope, FirstResult.Type_Symbol, SecondResult.Type_Symbol) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 42,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Second_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         elsif not Dictionary.CompatibleTypes (Scope, FirstResult.Type_Symbol, CaseTypeSymbol) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 106,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         if not (FirstResult.Is_Constant and FirstResult.Is_ARange) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 95,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => First_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;
         if not SecondResult.Is_Static then
            ErrorHandler.Semantic_Error
              (Err_Num   => 45,
               Reference => 1,
               Position  => Node_Position (Node => Second_Node),
               Id_Str    => LexTokenManager.Null_String);
            SemanticErrorsFound := True;
         end if;

         -- code to collect the appropriate values for the extent of the range
         -- note that these will be nonsense if semantic errors have been found
         ChoiceLowerMathsValue := SecondResult.Value;
         ChoiceUpperMathsValue := SecondResult.Range_RHS;
         IsARange              := True;

         -- somewhere need to check that SecondResult range is within the type
         -- given by FirstResult
   end case;

   --# assert True;

   if not SemanticErrorsFound then
      ConvertBooleanMathsValue (ChoiceLowerMathsValue);
      ConvertChoiceBound (ChoiceLowerMathsValue, ChoiceLowerBound, LowerBoundUnknown, LowerBoundOutOfRange);

      if IsARange then
         ConvertBooleanMathsValue (ChoiceUpperMathsValue);
         ConvertChoiceBound (ChoiceUpperMathsValue, ChoiceUpperBound, UpperBoundUnknown, UpperBoundOutOfRange);
      else
         ChoiceUpperBound := Unknown_Type_Bound;
      end if;

      if LowerBoundOutOfRange or (IsARange and then UpperBoundOutOfRange) then
         BothChoiceBoundsKnown := False;
         ErrorHandler.Semantic_Warning
           (Err_Num  => 305,
            Position => Node_Position (Node => First_Node),
            Id_Str   => LexTokenManager.Null_String);
      elsif LowerBoundUnknown or (IsARange and UpperBoundUnknown) then
         BothChoiceBoundsKnown      := False;
         CompleteADT.Undeterminable := True;
         ErrorHandler.Semantic_Warning
           (Err_Num  => 200,
            Position => Node_Position (Node => First_Node),
            Id_Str   => LexTokenManager.Null_String);
      else
         BothChoiceBoundsKnown := True;
      end if;

      --# assert True;

      if BothChoiceBoundsKnown then
         -- check the case choice lies within controlling type
         if not IsChoiceInRange (ChoiceLowerBound, ChoiceUpperBound, IsARange, CaseTypeLowerBound, CaseTypeUpperBound) then
            if CaseChoiceSort = RangeConstraint then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 410,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Second_Node),
                  Id_Str    => LexTokenManager.Null_String);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 410,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => First_Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
            SemanticErrorsFound := True;
         elsif IsARange and IsRangeEmpty (ChoiceLowerBound, ChoiceUpperBound) then
            if CaseChoiceSort = RangeConstraint then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 409,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Second_Node),
                  Id_Str    => LexTokenManager.Null_String);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 409,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => First_Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
            SemanticErrorsFound := True;
         end if;

         --# assert True;

         -- check the case choice lies within RangeConstraint type
         if CaseChoiceSort = RangeConstraint then
            Get_Type_Bounds
              (Type_Symbol => FirstResult.Type_Symbol,
               Lower_Bound => RangeConstraintLowerBound,
               Upper_Bound => RangeConstraintUpperBound);

            if not IsChoiceInRange
              (ChoiceLowerBound,
               ChoiceUpperBound,
               IsARange,
               RangeConstraintLowerBound,
               RangeConstraintUpperBound) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 413,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Second_Node),
                  Id_Str    => LexTokenManager.Null_String);
               SemanticErrorsFound := True;
            end if;
         end if;
      end if;
   end if;

   --# assert True;

   if (not SemanticErrorsFound)
     and then BothChoiceBoundsKnown
     and then (CaseFlags.Check_Completeness or CaseFlags.Check_Overlap) then
      if IsARange then
         CompleteCheck.SeenRange (CompleteADT, ChoiceLowerBound.Value, ChoiceUpperBound.Value, OutOfRangeSeen, OverlapSeen);
      else
         CompleteCheck.SeenElement (CompleteADT, ChoiceLowerBound.Value, OutOfRangeSeen, OverlapSeen);
      end if;
      if OutOfRangeSeen then
         CaseFlags.Out_Of_Range_Seen := True;
      end if;
      if CaseFlags.Check_Overlap and OverlapSeen = CompleteCheck.Overlap then
         ErrorHandler.Semantic_Error
           (Err_Num   => 407,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => First_Node),
            Id_Str    => LexTokenManager.Null_String);
         SemanticErrorsFound := True;
      end if;
   end if;

   -- add reference variable list to RefList hash table
   RefList.AddRelation (Table, TheHeap, Node, Dictionary.NullSymbol, RefVar);

   --# assert True;

   if SemanticErrorsFound then
      CaseFlags.Check_Completeness := False;
   end if;

   Case_Stack.Push (CaseFlags, CompleteADT, CaseTypeSymbol, CaseTypeLowerBound, CaseTypeUpperBound);
end wf_case_choice;
