-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================
separate (Sem.CompUnit)
procedure CalcAttribute
  (Node         : in     STree.SyntaxNode;
   AttribName   : in     LexTokenManager.Lex_String;
   PrefixKind   : in     Dictionary.PrefixSort;
   Prefix       : in     Dictionary.Symbol;
   BaseFound    : in     Boolean;
   IsAnnotation : in     Boolean;
   Argument     : in out Maths.Value;
   RHSofRange   :    out Maths.Value) is
   pragma Unreferenced (PrefixKind);

   type ErrLookUp is array (Boolean) of Integer;
   WhichErr : constant ErrLookUp := ErrLookUp'(False => 402,
                                               True  => 399);

   Err           : Maths.ErrorCode;
   LocalBaseType : Dictionary.Symbol;
   ArgumentLocal : Maths.Value;

   procedure CalcArrayAttribute
   --# global in     AttribName;
   --#        in     BaseFound;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     Prefix;
   --#        in out Argument;
   --#        in out Err;
   --#        in out RHSofRange;
   --# derives Argument,
   --#         Err,
   --#         RHSofRange from *,
   --#                         Argument,
   --#                         AttribName,
   --#                         BaseFound,
   --#                         Dictionary.Dict,
   --#                         LexTokenManager.State,
   --#                         Prefix;
   is
      Dimension   : Positive;
      Vlow, Vhigh : Maths.Value;
      FCS         : Dictionary.Symbol; -- First Constrained Subtype

      procedure GetDimension (Argument : Maths.Value)
      --# global out Dimension;
      --# derives Dimension from Argument;
      is
         UnusedErr : Maths.ErrorCode;
      begin
         if Maths.HasNoValue (Argument) then
            Dimension := 1;
         else
            --# accept Flow, 10, UnusedErr, "Expected ineffective assignment";
            Maths.ValueToInteger (Argument, -- expect ineffective assign to UnusedErr
                                  -- to get
                                  Dimension, UnusedErr);
            --# end accept;
         end if;
         --# accept Flow, 33, UnusedErr, "Expected to be neither referenced nor exported";
      end GetDimension;

   begin -- CalcArrayAttribute
      if BaseFound then
         -- the only valid attribute would be size and we never know what
         -- the size of things is so we can only return the null value
         Argument := Maths.NoValue;
      elsif Dictionary.IsUnconstrainedArrayType (Prefix) then
         Argument := Maths.NoValue;
      else -- a constrained array type or subtype
         if Dictionary.IsSubtype (Prefix) then
            FCS := Dictionary.GetFirstConstrainedSubtype (Prefix);
         else
            FCS := Prefix;
         end if;

         GetDimension (Argument);
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttribName,
                                                                 Lex_Str2 => LexTokenManager.Range_Token) =
           LexTokenManager.Str_Eq then
            Argument   := Maths.ValueRep (Dictionary.GetArrayAttributeValue (LexTokenManager.First_Token, FCS, Dimension));
            RHSofRange := Maths.ValueRep (Dictionary.GetArrayAttributeValue (LexTokenManager.Last_Token, FCS, Dimension));

         elsif LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => AttribName,
            Lex_Str2 => LexTokenManager.Length_Token) =
           LexTokenManager.Str_Eq then

            Vlow  := Maths.ValueRep (Dictionary.GetArrayAttributeValue (LexTokenManager.First_Token, FCS, Dimension));
            Vhigh := Maths.ValueRep (Dictionary.GetArrayAttributeValue (LexTokenManager.Last_Token, FCS, Dimension));
            --# accept Flow, 10, Err, "Expected ineffective assignment";
            Maths.Subtract (Vhigh, -- flow error expected
                            Vlow, Argument, Err);
            --# end accept;
            Vhigh := Argument;
            Maths.Add (Vhigh, Maths.OneInteger, Argument, Err);
         else -- first/last
            Argument := Maths.ValueRep (Dictionary.GetArrayAttributeValue (AttribName, FCS, Dimension));
         end if;
      end if;
   end CalcArrayAttribute;

begin -- CalcAttribute
      -- this procedure is only called wf_attribute_designator if the attribute
      -- is well-formed.

   RHSofRange := Maths.NoValue;  -- default value unless 'RANGE processed

   if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttribName,
                                                           Lex_Str2 => LexTokenManager.Succ_Token) =
     LexTokenManager.Str_Eq then
      Maths.SuccOp (Argument, Err);

      LocalBaseType := Dictionary.GetRootType (Prefix);

      if Dictionary.TypeIsModular (LocalBaseType) then

         Maths.Modulus
           (FirstNum  => Argument,
            SecondNum => Maths.ValueRep
              (Dictionary.GetScalarAttributeValue
                 (Base     => False,
                  Name     => LexTokenManager.Modulus_Token,
                  TypeMark => LocalBaseType)),
            Result    => ArgumentLocal,
            Ok        => Err);

      else
         ConstraintCheck (Argument, ArgumentLocal, IsAnnotation, LocalBaseType, Node_Position (Node => Node));
      end if;
      Argument := ArgumentLocal;

   elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttribName,
                                                              Lex_Str2 => LexTokenManager.Pred_Token) =
     LexTokenManager.Str_Eq then
      Maths.PredOp (Argument, Err);

      LocalBaseType := Dictionary.GetRootType (Prefix);

      if Dictionary.TypeIsModular (LocalBaseType) then

         Maths.Modulus
           (FirstNum  => Argument,
            SecondNum => Maths.ValueRep
              (Dictionary.GetScalarAttributeValue
                 (Base     => False,
                  Name     => LexTokenManager.Modulus_Token,
                  TypeMark => LocalBaseType)),
            Result    => ArgumentLocal,
            Ok        => Err);

      else
         ConstraintCheck (Argument, ArgumentLocal, IsAnnotation, LocalBaseType, Node_Position (Node => Node));
      end if;
      Argument := ArgumentLocal;

   elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttribName,
                                                              Lex_Str2 => LexTokenManager.Floor_Token) =
     LexTokenManager.Str_Eq then
      Maths.Floor (Argument, ArgumentLocal, Err);
      Argument      := ArgumentLocal;
      LocalBaseType := Dictionary.GetRootType (Prefix);
      ConstraintCheck (Argument, ArgumentLocal, IsAnnotation, LocalBaseType, Node_Position (Node => Node));
      Argument := ArgumentLocal;

   elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttribName,
                                                              Lex_Str2 => LexTokenManager.Ceiling_Token) =
     LexTokenManager.Str_Eq then
      Maths.Ceiling (Argument, ArgumentLocal, Err);
      Argument      := ArgumentLocal;
      LocalBaseType := Dictionary.GetRootType (Prefix);
      ConstraintCheck (Argument, ArgumentLocal, IsAnnotation, LocalBaseType, Node_Position (Node => Node));
      Argument := ArgumentLocal;

   elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttribName,
                                                              Lex_Str2 => LexTokenManager.Mod_Token) =
     LexTokenManager.Str_Eq then
      LocalBaseType := Dictionary.GetRootType (Prefix);
      Maths.Modulus
        (FirstNum  => Argument,
         SecondNum => Maths.ValueRep
           (Dictionary.GetScalarAttributeValue
              (Base     => False,
               Name     => LexTokenManager.Modulus_Token,
               TypeMark => LocalBaseType)),
         Result    => ArgumentLocal,
         Ok        => Err);
      Argument := ArgumentLocal;

   elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttribName,
                                                              Lex_Str2 => LexTokenManager.Val_Token) =
     LexTokenManager.Str_Eq then
      Err := Maths.NoError;
      -- upper and lower bounds check required

   elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttribName,
                                                              Lex_Str2 => LexTokenManager.Pos_Token) =
     LexTokenManager.Str_Eq then
      Err := Maths.NoError; -- no action required, no error can occur

   elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttribName,
                                                              Lex_Str2 => LexTokenManager.Valid_Token) =
     LexTokenManager.Str_Eq then
      Err := Maths.NoError; -- no action required, no error can occur

   elsif LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => AttribName,
                                                              Lex_Str2 => LexTokenManager.Size_Token) =
     LexTokenManager.Str_Eq then
      Err      := Maths.NoError;
      Argument := Maths.ValueRep (Dictionary.TypeSizeAttribute (Prefix));

   elsif Dictionary.TypeIsScalar (Prefix) then
      Err      := Maths.NoError;
      Argument := Maths.ValueRep (Dictionary.GetScalarAttributeValue (BaseFound, AttribName, Prefix));

   elsif Dictionary.TypeIsArray (Prefix) then
      Err := Maths.NoError;
      CalcArrayAttribute;

   else -- non-implemented attribute - should never occur
      Argument := Maths.NoValue;
      Err      := Maths.NoError;
   end if;

   case Err is
      when Maths.NoError =>
         null;
      when Maths.DivideByZero =>
         Argument := Maths.NoValue;
         ErrorHandler.Semantic_Error
           (Err_Num   => 400,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      when Maths.ConstraintError =>
         Argument := Maths.NoValue;
         ErrorHandler.Semantic_Error
           (Err_Num   => WhichErr (IsAnnotation),
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      when Maths.OverFlow =>
         Argument := Maths.NoValue;
         ErrorHandler.Semantic_Warning
           (Err_Num  => 200,
            Position => Node_Position (Node => Node),
            Id_Str   => LexTokenManager.Null_String);
      when others => -- indicates internal error in maths package
         Argument := Maths.NoValue;
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Math_Error,
                                   Msg     => "in CalcAttribute");
   end case;
   --# accept Flow, 30, PrefixKind, "Expected to be unused";
end CalcAttribute; -- expect PrefixKind unused
