-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
package body AggregateStack
--# own State is S, TopPtr;
is
   type TypEntry is record
      TypeSym     : Dictionary.Symbol;
      LowerBound  : Sem.Typ_Type_Bound;
      UpperBound  : Sem.Typ_Type_Bound;
      AggFlags    : Sem.Typ_Agg_Flags;
      Counter     : Natural;
      CompleteRec : CompleteCheck.T;
   end record;

   NullEntry : constant TypEntry :=
     TypEntry'
     (TypeSym     => Dictionary.NullSymbol,
      LowerBound  => Sem.Unknown_Type_Bound,
      UpperBound  => Sem.Unknown_Type_Bound,
      AggFlags    => Sem.Null_Typ_Agg_Flags,
      Counter     => 0,
      CompleteRec => CompleteCheck.NullT);

   subtype IndexRange is Integer range 1 .. ExaminerConstants.AggregateStackMax;
   type StackArray is array (IndexRange) of TypEntry;
   subtype TopRange is Integer range 0 .. ExaminerConstants.AggregateStackMax;

   S      : StackArray;
   TopPtr : TopRange;

   procedure Init
   --# global out S;
   --#        out TopPtr;
   --# derives S,
   --#         TopPtr from ;
   is
   begin
      TopPtr := 0;
      S      := StackArray'(others => NullEntry);
   end Init;

   procedure Push
     (TypeSym     : in Dictionary.Symbol;
      LowerBound  : in Sem.Typ_Type_Bound;
      UpperBound  : in Sem.Typ_Type_Bound;
      AggFlags    : in Sem.Typ_Agg_Flags;
      Counter     : in Natural;
      CompleteRec : in CompleteCheck.T)
   --# global in out S;
   --#        in out TopPtr;
   --# derives S      from *,
   --#                     AggFlags,
   --#                     CompleteRec,
   --#                     Counter,
   --#                     LowerBound,
   --#                     TopPtr,
   --#                     TypeSym,
   --#                     UpperBound &
   --#         TopPtr from *;
   is
   begin
      if TopPtr < ExaminerConstants.AggregateStackMax then
         TopPtr     := TopPtr + 1;
         S (TopPtr) :=
           TypEntry'
           (TypeSym     => TypeSym,
            LowerBound  => LowerBound,
            UpperBound  => UpperBound,
            AggFlags    => AggFlags,
            Counter     => Counter,
            CompleteRec => CompleteRec);
      else
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Aggregate_Stack_Over_Flow,
                                   Msg     => "");
      end if;
   end Push;

   procedure Pop
     (TypeSym     : out Dictionary.Symbol;
      LowerBound  : out Sem.Typ_Type_Bound;
      UpperBound  : out Sem.Typ_Type_Bound;
      AggFlags    : out Sem.Typ_Agg_Flags;
      Counter     : out Natural;
      CompleteRec : out CompleteCheck.T)
   --# global in     S;
   --#        in out TopPtr;
   --# derives AggFlags,
   --#         CompleteRec,
   --#         Counter,
   --#         LowerBound,
   --#         TypeSym,
   --#         UpperBound  from S,
   --#                          TopPtr &
   --#         TopPtr      from *;
   is
   begin
      if TopPtr = 0 then
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Aggregate_Stack_Under_Flow,
                                   Msg     => "");
      end if;
      TypeSym     := S (TopPtr).TypeSym;
      LowerBound  := S (TopPtr).LowerBound;
      UpperBound  := S (TopPtr).UpperBound;
      AggFlags    := S (TopPtr).AggFlags;
      Counter     := S (TopPtr).Counter;
      CompleteRec := S (TopPtr).CompleteRec;
      TopPtr      := TopPtr - 1;
   end Pop;

   function TopTypeSym return  Dictionary.Symbol
   --# global in S;
   --#        in TopPtr;
   is
   begin
      return S (TopPtr).TypeSym;
   end TopTypeSym;

end AggregateStack;
