-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SystemErrors;

separate (FlowAnalyser)
package body IFA_Stack is

   function IsEmpty (S : Stack) return Boolean is
   begin
      return S.StackPointer = 0;
   end IsEmpty;

   function Top (S : Stack) return StackMember is
      Result : StackMember;
   begin
      if S.StackPointer > 0 then
         Result := S.StackVector (S.StackPointer);
      else
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Relation_Stack_Underflow,
                                   Msg     => "in StackManager.Top");
         Result := NullMember;  -- unreachable, assignment just to avoid DF error
      end if;
      return Result;
   end Top;

   procedure ClearStack (S : out Stack) is
   begin
      S.StackPointer := 0;
      --# accept Flow, 32, S.StackVector, "Init. is partial but effective" &
      --#        Flow, 31, S.StackVector, "Init. is partial but effective" &
      --#        Flow, 602, S, S.StackVector, "Init. is partial but effective";
   end ClearStack;

   procedure Push (S : in out Stack;
                   M : in     StackMember) is
   begin
      if S.StackPointer < ExaminerConstants.StackManagerStackSize then
         S.StackPointer                 := S.StackPointer + 1;
         S.StackVector (S.StackPointer) := M;
      else
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Relation_Stack_Overflow,
                                   Msg     => "in StackManager.Push");
      end if;
   end Push;

   procedure Pop (S : in out Stack;
                  M :    out StackMember) is
   begin
      if S.StackPointer > 0 then
         M              := S.StackVector (S.StackPointer);
         S.StackPointer := S.StackPointer - 1;
      else
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Relation_Stack_Underflow,
                                   Msg     => "in StackManager.Pop");
         M := NullMember; -- unreachable, just to avoid DF
      end if;
   end Pop;

   procedure EstablishMember (TheHeap : in out Heap.HeapRecord;
                              Kind    : in     MemberType;
                              M       :    out StackMember) is
      DefinedVars, UnPreservedVars, AllVars, SeqOfExpns : SeqAlgebra.Seq;
      Lambda, Mu, Rho, Theta, ThetaTilde, RhoProd       : RelationAlgebra.Relation;
   begin
      SeqAlgebra.CreateSeq (TheHeap, DefinedVars);
      SeqAlgebra.CreateSeq (TheHeap, UnPreservedVars);
      SeqAlgebra.CreateSeq (TheHeap, AllVars);
      SeqAlgebra.CreateSeq (TheHeap, SeqOfExpns);
      RelationAlgebra.CreateRelation (TheHeap, Lambda);
      RelationAlgebra.CreateRelation (TheHeap, Mu);
      RelationAlgebra.CreateRelation (TheHeap, Rho);
      RelationAlgebra.CreateRelation (TheHeap, Theta);
      RelationAlgebra.CreateRelation (TheHeap, ThetaTilde);
      RelationAlgebra.CreateRelation (TheHeap, RhoProd);
      M :=
        StackMember'
        (MemberKind      => Kind,
         DefinedVars     => DefinedVars,
         UnPreservedVars => UnPreservedVars,
         AllVars         => AllVars,
         SeqOfExpns      => SeqOfExpns,
         Lambda          => Lambda,
         Mu              => Mu,
         Rho             => Rho,
         Theta           => Theta,
         ThetaTilde      => ThetaTilde,
         RhoProd         => RhoProd);
   end EstablishMember;

   procedure DisposeOfMember (TheHeap : in out Heap.HeapRecord;
                              M       : in     StackMember) is
   begin
      SeqAlgebra.DisposeOfSeq (TheHeap, M.DefinedVars);
      SeqAlgebra.DisposeOfSeq (TheHeap, M.UnPreservedVars);
      SeqAlgebra.DisposeOfSeq (TheHeap, M.AllVars);
      SeqAlgebra.DisposeOfSeq (TheHeap, M.SeqOfExpns);
      RelationAlgebra.DisposeOfRelation (TheHeap, M.Lambda);
      RelationAlgebra.DisposeOfRelation (TheHeap, M.Mu);
      RelationAlgebra.DisposeOfRelation (TheHeap, M.Rho);
      RelationAlgebra.DisposeOfRelation (TheHeap, M.Theta);
      RelationAlgebra.DisposeOfRelation (TheHeap, M.ThetaTilde);
      RelationAlgebra.DisposeOfRelation (TheHeap, M.RhoProd);
   end DisposeOfMember;

end IFA_Stack;
