------------------------------------------------------------------------------
--                                                                          --
--                           GNAT RAVENSCAR for NXT                         --
--                                                                          --
--                       Copyright (C) 2010, AdaCore                        --
--                                                                          --
-- This is free software; you can  redistribute it  and/or modify it under  --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. This is distributed in the hope that it will be useful, but WITH-  --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
------------------------------------------------------------------------------

with Memory_Copy;
with Interfaces;
with System;

package body Nxt.Display is
   Font_Width : constant := 6;

   Max_X : constant Natural := 100 / Font_Width;

   Cur_X : Char_Column_Type;
   Cur_Y : Char_Row_Type;

   Screen : array (Char_Column_Type, Char_Row_Type) of Character;

   procedure Raw_Putc (C : Character) is
      use Interfaces;
      UART0_DR : Unsigned_32;
      for UART0_DR'Address use System'To_Address (16#101f_1000#);
   begin
      UART0_DR := Character'Pos (C);
   end Raw_Putc;

   procedure Screen_Update is
   begin
      for I in Char_Row_type loop
         Raw_Putc ('|');
         for J in Screen'Range (1) loop
            Raw_Putc (Screen (J, I));
         end loop;
         Raw_Putc ('|');
         Raw_Putc (ASCII.CR);
         Raw_Putc (ASCII.LF);
      end loop;
      for I in 1 .. 18 loop
         Raw_Putc ('-');
      end loop;
      Raw_Putc (ASCII.CR);
      Raw_Putc (ASCII.LF);
   end Screen_Update;

   procedure Clear_Screen_Noupdate is
   begin
      Screen := (others => (others => ' '));
      Cur_X := 0;
      Cur_Y := 0;
   end Clear_Screen_Noupdate;

   procedure Clear_Screen is
   begin
      Clear_Screen_Noupdate;
      Screen_Update;
   end Clear_Screen;

   procedure Set_Xy (X : Char_Column_Type; Y : Char_Row_Type) is
   begin
      Cur_X := X;
      Cur_Y := Y;
   end Set_Xy;

   procedure Newline_Noupdate is
   begin
      Cur_X := 0;
      if Cur_Y = Char_Row_Type'Last then
         for I in 0 .. Char_Row_Type'Last - 1 loop
            for J in Char_Column_Type loop
               Screen (J, I) := Screen (J, I + 1);
            end loop;
         end loop;
         for J in Char_Column_Type loop
            Screen (J, Char_Row_Type'Last) := ' ';
         end loop;
      else
         Cur_Y := Cur_Y + 1;
      end if;
   end Newline_Noupdate;

   procedure Put_Noupdate (C : Character) is
   begin
      if C in ' ' .. Character'Val (126) then
         Screen (Cur_X, Cur_Y) := C;
         if Cur_X = Char_Column_Type'Last then
            Newline_Noupdate;
         else
            Cur_X := Cur_X + 1;
         end if;
      else
         case C is
            when ASCII.CR =>
               Cur_X := 0;
            when ASCII.LF =>
               Newline_Noupdate;
            when others =>
               null;
         end case;
      end if;
   end Put_Noupdate;

   procedure Put (C : Character) is
   begin
      Put_Noupdate (C);
      Screen_Update;
   end Put;

   procedure Put_Noupdate (S : String) is
   begin
      for I in S'Range loop
         Put_Noupdate (S (I));
      end loop;
   end Put_Noupdate;

   procedure Put (S : String) is
   begin
      Put_Noupdate (S);
      Screen_Update;
   end Put;

   procedure Put_Line (S : String) is
   begin
      Put_Noupdate (S);
      Newline_Noupdate;
      Screen_Update;
   end Put_Line;

   procedure Newline is
   begin
      Newline_Noupdate;
      Screen_Update;
   end Newline;

   Hexdigits : constant array (0 .. 15) of Character := "0123456789ABCDEF";

   procedure Put_Hex (Val : Unsigned_32) is
   begin
      for I in reverse 0 .. 7 loop
         Put_Noupdate (Hexdigits (Natural (Shift_Right (Val, 4 * I) and 15)));
      end loop;
   end Put_Hex;

   procedure Put_Hex (Val : Unsigned_16) is
   begin
      for I in reverse 0 .. 3 loop
         Put_Noupdate (Hexdigits (Natural (Shift_Right (Val, 4 * I) and 15)));
      end loop;
   end Put_Hex;

   procedure Put_Hex (Val : Unsigned_8) is
   begin
      for I in reverse Integer range 0 .. 1 loop
         Put_Noupdate (Hexdigits (Natural (Shift_Right (Val, 4 * I) and 15)));
      end loop;
   end Put_Hex;

   procedure Put_Noupdate (V : Integer) is
      Val : Integer := V;
      Res : String (1 .. 9);
      Pos : Natural := Res'Last;
   begin
      if Val < 0 then
         Put_Noupdate ('-');
      else
         Val := -Val;
      end if;
      loop
         Res (Pos) := Character'Val (Character'Pos ('0') - (Val mod (-10)));
         Val := Val / 10;
         exit when Val = 0;
         Pos := Pos - 1;
      end loop;
      for I in Pos .. Res'Last loop
         Put_Noupdate (Res (I));
      end loop;
   end Put_Noupdate;

   procedure Put_Exception (Addr : Unsigned_32) is
   begin
      Set_Xy (0, 0);
      Put_Noupdate ("ERR@");
      Put_Hex (Addr);
   end Put_Exception;
begin
   Clear;
end Nxt.Display;
