/**
 * NXT bootstrap interface; low-level USB functions.
 *
 * Copyright 2006 David Anderson <david.anderson@calixo.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#ifndef _WIN32
#include <termios.h>
#endif
#include <usb.h>

#include "lowlevel.h"

int verbose;

enum nxt_usb_ids {
  VENDOR_LEGO   = 0x0694,
  VENDOR_ATMEL  = 0x03EB,
  PRODUCT_NXT   = 0x0002,
  PRODUCT_SAMBA = 0x6124
};

struct nxt_t {
  int fd;
  struct usb_device *dev;
  struct usb_dev_handle *hdl;
  int is_in_reset_mode;
};


nxt_error_t nxt_init (nxt_t **nxt, const char *device)
{
  *nxt = calloc(1, sizeof(**nxt));

  if (device == NULL || strcmp (device, "usb") == 0)
    {
      usb_init();
      (*nxt)->fd = -1;
      return NXT_OK;
    }
#ifndef _WIN32
  (*nxt)->fd = open (device, O_RDWR | O_NOCTTY);
  if ((*nxt)->fd >= 0)
    return NXT_OK;
#endif

  fprintf (stderr, "can't open %s: %s\n", device, strerror (errno));
  return NXT_NOT_PRESENT;
}


nxt_error_t nxt_find(nxt_t *nxt)
{
  struct usb_bus *busses, *bus;

  if (nxt->fd != -1)
    {
      nxt->is_in_reset_mode = 1;
      return NXT_OK;
    }

  usb_find_busses();
  usb_find_devices();

  busses = usb_get_busses();

  for (bus = busses; bus != NULL; bus = bus->next)
    {
      struct usb_device *dev;

      for (dev = bus->devices; dev != NULL; dev = dev->next)
        {
          if (dev->descriptor.idVendor == VENDOR_ATMEL &&
              dev->descriptor.idProduct == PRODUCT_SAMBA)
            {
              nxt->dev = dev;
              nxt->is_in_reset_mode = 1;
              return NXT_OK;
            }
          else if (dev->descriptor.idVendor == VENDOR_LEGO &&
                   dev->descriptor.idProduct == PRODUCT_NXT)
            {
              nxt->dev = dev;
              return NXT_OK;
            }
        }
    }

  return NXT_NOT_PRESENT;
}


nxt_error_t
nxt_open(nxt_t *nxt)
{
  char buf[4];
  int ret;
  int len;

#ifndef _WIN32
  if (nxt->fd != -1)
    {
      struct termios tios;

      if (tcgetattr (nxt->fd, &tios) < 0)
        return NXT_CONFIGURATION_ERROR;
      cfmakeraw (&tios);
      tios.c_cflag &= ~CLOCAL;
      if (tcsetattr (nxt->fd, TCSADRAIN, &tios) < 0)
        return NXT_CONFIGURATION_ERROR;
    }
  else
#endif
    {
      nxt->hdl = usb_open(nxt->dev);

      ret = usb_set_configuration(nxt->hdl, 1);

      if (ret < 0)
        {
          fprintf (stderr, "usb: usb_set_configuration failure: %d (%s)\n",
                   errno, strerror (errno));
          usb_close(nxt->hdl);
          return NXT_CONFIGURATION_ERROR;
        }

      ret = usb_claim_interface(nxt->hdl, 1);

      if (ret < 0)
        {
          usb_close(nxt->hdl);
          return NXT_IN_USE;
        }
    }

  /* NXT handshake */
  nxt_send_str(nxt, "#");

  /* Read the LF+CR.  */
  nxt_recv_buf(nxt, buf, 2);
  if (memcmp(buf, "\n\r", 2) != 0)
    goto err;
  /* Read the prompt.  */
  nxt_recv_buf(nxt, buf, 1);
  if (buf[0] != '>')
    goto err;

  return NXT_OK;

err:
  nxt_close (nxt);
  return NXT_HANDSHAKE_FAILED;
}


nxt_error_t
nxt_close(nxt_t *nxt)
{
  if (nxt->fd != -1)
    {
      close (nxt->fd);
    }
  else
    {
      usb_release_interface(nxt->hdl, 1);
      usb_close(nxt->hdl);
    }

  free(nxt);

  return NXT_OK;
}


int
nxt_in_reset_mode(nxt_t *nxt)
{
  return nxt->is_in_reset_mode;
}

static void
dump_data (const char *msg, const char *buf, int len)
{
  int i, j;

  printf ("%s (%d bytes):\n", msg, len);
  for (i = 0; i < len; i += 16)
    {
      printf ("%03x: ", i);
      for (j = 0; j < 16; j++)
        {
          if (i + j < len)
            printf (" %02x", (unsigned char)buf[i + j]);
          else
            printf ("   ");
        }
      printf ("  ");
      for (j = 0; j < 16; j++)
        {
          if (i + j < len)
            putchar (buf[i + j] > 32 ? buf[i + j] : '.');
          else
            putchar (' ');
        }
      printf ("\n");
    }
  fflush (stdout);
}

nxt_error_t
nxt_send_buf(nxt_t *nxt, char *buf, int len)
{
  int ret;

  if (verbose > 1)
    dump_data ("Send", buf, len);

#ifndef _WIN32
  if (nxt->fd != -1)
    {
      while (len > 0)
        {
          ret = write (nxt->fd, buf, len);
          if (ret < 0)
            {
              if (errno == EAGAIN)
                continue;
              return NXT_USB_WRITE_ERROR;
            }
          buf += ret;
          len -= ret;
        }
    }
  else
#endif
    {
      ret = usb_bulk_write(nxt->hdl, 0x1, buf, len, 1000);
      if (ret < 0)
        return NXT_USB_WRITE_ERROR;
    }

  return NXT_OK;
}


nxt_error_t
nxt_send_str(nxt_t *nxt, char *str)
{
  return nxt_send_buf(nxt, str, strlen(str));
}


nxt_error_t
nxt_recv_buf(nxt_t *nxt, char *buf, int len)
{
  int ret;
  int orig_len = len;
  char *orig_buf = buf;

#ifndef _WIN32
  if (nxt->fd != -1)
    {
      while (len > 0)
        {
          ret = read (nxt->fd, buf, len);
          if (ret < 0)
            {
              if (errno == EAGAIN)
                continue;
              return NXT_USB_READ_ERROR;
            }
          buf += ret;
          len -= ret;
        }
    }
  else
#endif
    {
      ret = usb_bulk_read(nxt->hdl, 0x82, buf, len, 1000);
      if (ret < 0)
        return NXT_USB_READ_ERROR;
    }

  if (verbose > 1)
    dump_data ("Receive", orig_buf, orig_len);
  return NXT_OK;
}
