------------------------------------------------------------------------------
--                                                                          --
--                           GNAT RAVENSCAR for NXT                         --
--                                                                          --
--                    Copyright (C) 2010-2011, AdaCore                      --
--                                                                          --
-- This is free software; you can  redistribute it  and/or modify it under  --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. This is distributed in the hope that it will be useful, but WITH-  --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
------------------------------------------------------------------------------

with NXT.Registers;  use NXT.Registers;

with System;  use System;
with Ada.Unchecked_Conversion;

pragma Warnings (Off);
with System.BB.Interrupts; use System.BB.Interrupts;
pragma Warnings (On);

package body NXT.Audio is

   Tone_Generation_Overhead : constant := 1;
   --  The approximate time in milliseconds it takes to generate the tone
   --  representation.  Hand-tune as necessary.

   PDM_Buffer_Length : constant := 64;

   subtype Buffer_Index is Integer range 0 .. PDM_Buffer_Length - 1;

   type Sample_Buffer is array (Buffer_Index) of Unsigned_32;

   subtype Buffer_Selector is Unsigned_8 range 0 .. 1;

   type Double_Buffer is array (Buffer_Selector) of Sample_Buffer;

   type Amplification is array (Integer range 0 .. 256) of Unsigned_8;

   type Sound_Modes is (None, Tone, PCM);
   pragma Discard_Names (Sound_Modes);

   type Waveform is array (Natural range <>) of Unsigned_8;

   Sine_Wave : constant Waveform :=
     (16#C0#, 16#C8#, 16#D0#, 16#D8#, 16#E0#, 16#EA#, 16#F4#, 16#FF#,
      16#FF#, 16#F0#, 16#E5#, 16#DC#, 16#D4#, 16#CC#, 16#C4#, 16#BC#);
   --  The waveform for signals generated by Play_Tone

   function As_Register_32 is new Ada.Unchecked_Conversion
     (Source => Address, Target => Unsigned_32);

   procedure Disable_Sound_Interrupt;
   --  Disables the three kinds of interrupt

   procedure Enable_Sound_Interrupt (Kind : Unsigned_32);
   --  Enables Kind and disables the others

   procedure Disable_Sound;
   --  prevent output to the speaker

   procedure Enable_Sound;
   --  allow output to the speaker

   procedure Set_Current_Volume
     (Requested_Volume : Allowed_Volume;
      Amplifier        : in out Amplification;
      Actual_Volume    : in out Allowed_Volume);
   --  Sets Actual_Volume to Requested_Volume, but also generates a new table
   --  if the current setting is being changed. We use a logarithmic system
   --  mapped into a range of 0 to 120, in which 0 is muted, 100 is full
   --  volume, and 120 is driving into overload.

   procedure Create_Tone
     (Shape  : Waveform;
      Buffer : in out Sample_Buffer;
      Length : Unsigned_32;
      Amp    : Amplification);
   --  Generates an amplified tone of the requested frequency and requested
   --  duration into Buffer. Shape is the waveform for the newly created tone.

   type Sound_Sample is array (Natural) of Unsigned_8;
   for Sound_Sample'Component_Size use Unsigned_8'Size;
   --  Represents a sound sample, such as that within a wav file, as an
   --  arbitrary length of contiguous bytes. We never actually declare a value
   --  of this type so size is not an issue. Hence in practice any input file
   --  size can be accommodated.

   type Sound_Sample_Reference is access all Sound_Sample;
   for Sound_Sample_Reference'Storage_Size use 0;
   pragma No_Strict_Aliasing (Sound_Sample_Reference);
   --  we are converting from an address...

   procedure Fill
     (Buffer       : in out Sample_Buffer;
      Input        : Sound_Sample_Reference;
      Amplifier    : Amplification;
      Sample_Count : in out Integer;
      Next_In      : in out Natural);
   --  Moves a slice of Input into Buffer for transmission. Each 8-bit sample
   --  is expanded into eight 32-bit numbers, i.e., 256 bits altogether,
   --  corresponding to the value used in Play_Sample. The expanded bytes are
   --  amplified as they are placed into Buffer.

   procedure Initialize_Hardware;
   --  set up the SSC and the PIO controller for the speaker

   ---------------
   -- Generator --
   ---------------

   protected Generator is

      procedure Play_Tone
        (Frequency        : Unsigned_32;
         Interval         : Unsigned_32;
         Requested_Volume : Allowed_Volume);

      procedure Play_Sample
        (Input            : Sound_Sample_Reference;
         Input_Length     : Unsigned_32;
         Requested_Volume : Allowed_Volume;
         Rate             : Sampling_Rates := Default_Rate);

      function Time_Remaining return Integer;

   private

      Sample_Count       : Integer;
      Sample_Buffers     : Double_Buffer;
      Buffer_Id          : Buffer_Selector := 0;
      Input_Sample       : Sound_Sample_Reference;
      Next_Sample        : Natural;
      Amplifier          : Amplification;
      Current_Volume     : Allowed_Volume := 0;
      Clock_Divisor      : Unsigned_32;
      Sample_Length      : Unsigned_32;
      Current_Sound_Mode : Sound_Modes := None;

      pragma Interrupt_Priority (Max_Interrupt_Priority);

      procedure Sound_IRQ_Handler;
      pragma Attach_Handler (Sound_IRQ_Handler, AT91C_ID_SSC);
      --  sets AIC_SVR (AT91C_ID_SSC) to Sound_IRQ_Handler'Address

   end Generator;

   protected body Generator is

      ---------------
      -- Play_Tone --
      ---------------

      procedure Play_Tone
        (Frequency        : Unsigned_32;
         Interval         : Unsigned_32;
         Requested_Volume : Allowed_Volume)
      is
         Actual_Interval : constant Unsigned_32 := Interval +
           Tone_Generation_Overhead;
         --  to prevent gaps when playing a series of tones we extend the
         --  requested interval to cover the time required to generate it

         Index            : Buffer_Selector;
         Length           : Unsigned_32;
         Actual_Frequency : Unsigned_32;
      begin
         if Frequency = 0 then
            return;
         elsif Frequency > 1000 then
            Length := 16;
         elsif Frequency < 500 then
            Length := 64;
         else
            Length := 32;
         end if;
         Current_Sound_Mode := Tone;
         Set_Current_Volume (Requested_Volume, Amplifier, Current_Volume);
         Index := Buffer_Id xor 1;
         Create_Tone (Sine_Wave, Sample_Buffers (Index), Length, Amplifier);
         Disable_Sound_Interrupt;
         Clock_Divisor := (Clock_Frequency / (Length * 32 * 2) + Frequency / 2)
           / Frequency;
         Actual_Frequency := (Clock_Frequency / (2 * Clock_Divisor))
           / (Length * 32);
         if Actual_Interval <= Tone_Generation_Overhead then
            Sample_Count := 0;
         else
            Sample_Count :=
              Integer ((Actual_Frequency * Actual_Interval + 1000 - 1) / 1000);
         end if;
         Sample_Length := Length;
         Next_Sample := Sample_Buffers (Index)'First;
         Buffer_Id := Index;
         SSC_PTCR := PDC_TXTEN;
         Enable_Sound_Interrupt (SSC_TXBUFE);
      end Play_Tone;

      -----------------
      -- Play_Sample --
      -----------------

      procedure Play_Sample
        (Input            : Sound_Sample_Reference;
         Input_Length     : Unsigned_32;
         Requested_Volume : Allowed_Volume;
         Rate             : Sampling_Rates := Default_Rate)
      is
         Expanded_Sample_Bits : constant := 256;
         --  Each 8-bit sample is expanded into this many bits by Fill. This
         --  must match the actual behavior of Fill!

         Divisor : Unsigned_32;
      begin
         Divisor := (Clock_Frequency / (2 * Expanded_Sample_Bits) + Rate / 2)
           / Rate;
         Set_Current_Volume (Requested_Volume, Amplifier, Current_Volume);
         Disable_Sound_Interrupt;
         Current_Sound_Mode := PCM;
         Sample_Count := Integer (Input_Length);
         Input_Sample := Input;
         Next_Sample := Input'First;
         Sample_Length := PDM_Buffer_Length;
         Clock_Divisor := Divisor;
         Enable_Sound_Interrupt (SSC_TXBUFE);
         SSC_PTCR := PDC_TXTEN;
      end Play_Sample;

      -----------------------
      -- Sound_IRQ_Handler --
      -----------------------

      procedure Sound_IRQ_Handler is
      begin
         if Sample_Count > 0 then
            SSC_CMR := Clock_Divisor;
            Enable_Sound;
            if SSC_TCR = 0 then
               if Current_Sound_Mode = PCM then
                  Buffer_Id := Buffer_Id xor 1;  -- use the other buffer
                  Fill (Sample_Buffers (Buffer_Id),
                        Input_Sample,
                        Amplifier,
                        Sample_Count,
                        Next_Sample);
                  SSC_TPR := As_Register_32
                     (Sample_Buffers (Buffer_Id)'Address);
               --  Transmit Pointer Register
               else
                  SSC_TPR := As_Register_32
                     (Sample_Buffers (Buffer_Id) (Next_Sample)'Address);
               end if;
               SSC_TCR := Sample_Length; -- Transmit Counter Register
               Sample_Count := Sample_Count - 1;
            end if;
            if Current_Sound_Mode = PCM then
               Buffer_Id := Buffer_Id xor 1;  -- use the other buffer
               Fill (Sample_Buffers (Buffer_Id),
                     Input_Sample,
                     Amplifier,
                     Sample_Count,
                     Next_Sample);
               SSC_TNPR := As_Register_32 (Sample_Buffers (Buffer_Id)'Address);
               --  Transmit Next Pointer Register
            else
               SSC_TNPR := As_Register_32 (Sample_Buffers
                                           (Buffer_Id) (Next_Sample)'Address);
            end if;
            SSC_TNCR := Sample_Length; -- Transmit Next Counter Register
            Sample_Count := Sample_Count - 1;
            Enable_Sound_Interrupt (SSC_ENDTX);
         else
            Current_Sound_Mode := None;
            Disable_Sound;
            Disable_Sound_Interrupt;
         end if;
      end Sound_IRQ_Handler;

      --------------------
      -- Time_Remaining --
      --------------------

      function Time_Remaining return Integer is
         Result : Unsigned_32;
      begin
         if Current_Sound_Mode = None then
            return 0;
         else
            Result := (Unsigned_32 (Sample_Count) * 1000 * Sample_Length * 32)
              / (Clock_Frequency / (2 * Clock_Divisor));
            if Current_Sound_Mode = Tone and Result > 0 then
               Result := Result - Tone_Generation_Overhead;
            end if;
         end if;
         return Integer (Result);
      end Time_Remaining;

   end Generator;

   ---------------
   -- Play_Tone --
   ---------------

   procedure Play_Tone
     (Frequency : Unsigned_32;
      Interval  : Unsigned_32;
      Volume    : Allowed_Volume)
   is
   begin
      Generator.Play_Tone (Frequency, Interval, Volume);
   end Play_Tone;

   -----------------
   -- Play_Sample --
   -----------------

   procedure Play_Sample
     (Input        : System.Address;
      Input_Length : Unsigned_32;
      Volume       : Allowed_Volume;
      Rate         : Sampling_Rates := Default_Rate)
   is
      function As_Sample is
        new Ada.Unchecked_Conversion (Address, Sound_Sample_Reference);
   begin
      if Input /= Null_Address and Input_Length /= 0 then
         Generator.Play_Sample (As_Sample (Input), Input_Length, Volume, Rate);
      end if;
   end Play_Sample;

   --------------------
   -- Time_Remaining --
   --------------------

   function Time_Remaining return Integer is
   begin
      return Generator.Time_Remaining;
   end Time_Remaining;

   -----------------
   -- Create_Tone --
   -----------------

   procedure Create_Tone
     (Shape  : Waveform;
      Buffer : in out Sample_Buffer;
      Length : Unsigned_32;
      Amp    : Amplification)
   is
      Num_Samples  : constant Integer := Integer (Length * 32) / 2;
      Step         : constant Integer := Num_Samples / Shape'Length;
      Initial_Bit  : constant Unsigned_32 := 16#8000_0000#;
      Initial_Bit2 : constant Unsigned_32 := 16#0000_0001#;

      Word      : Integer := 0;
      Bit       : Unsigned_32 := Initial_Bit;
      Bit2      : Unsigned_32 := Initial_Bit2;
      Error     : Integer := 0;
      Error2    : Integer := 0;
      Error_Out : Integer := 0;
      Bits      : Unsigned_32 := 0;
      Bits2     : Unsigned_32 := 0;
      Index     : Natural := 0;
      Input     : Integer;
   begin
      for K in 1 .. Num_Samples / Step loop
         Input := Integer (Shape (Index));
         Index := Index + 1;
         Input := Integer (Amp (Input) - 128);

         for J in 1 .. Step loop
            Error := Input - Error_Out + Error;
            Error2 := Error - Error_Out + Error2;
            if Error2 > 0 then
               Error_Out := 127;
               Bits := Bits or Bit;
            else
               Error_Out := -127;
               Bits2 := Bits2 or Bit2;
            end if;
            Bit2 := Shift_Left (Bit2, 1);
            Bit := Shift_Right (Bit, 1);
            if Bit = 0 then
               Bit := Initial_Bit;
               Bit2 := Initial_Bit2;
               Buffer (Word) := Bits;
               Word := Word + 1;
               Buffer (Integer (Length) - Word) := Bits2;
               Bits := 0;
               Bits2 := 0;
            end if;
         end loop;
      end loop;
   end Create_Tone;

   -----------------------------
   -- Disable_Sound_Interrupt --
   -----------------------------

   procedure Disable_Sound_Interrupt is
   begin
      SSC_IDR := SSC_TXBUFE;
      SSC_IDR := SSC_ENDTX;
      SSC_IDR := SSC_TXEMPTY;
   end Disable_Sound_Interrupt;

   ----------------------------
   -- Enable_Sound_Interrupt --
   ----------------------------

   procedure Enable_Sound_Interrupt (Kind : Unsigned_32) is
   begin
      SSC_IDR := SSC_TXBUFE;
      SSC_IDR := SSC_ENDTX;
      SSC_IDR := SSC_TXEMPTY;
      SSC_IER := Kind;
   end Enable_Sound_Interrupt;

   -------------------
   -- Disable_Sound --
   -------------------

   procedure Disable_Sound is
   begin
      PIOA_PER := PA17_TD;
      --  speaker pin is controlled by the PIO controller
   end Disable_Sound;

   ------------------
   -- Enable_Sound --
   ------------------

   procedure Enable_Sound is
   begin
      PIOA_PDR := PA17_TD;
      --  speaker pin is not controlled by the PIO controller
   end Enable_Sound;

   Volume_Table : constant array (1 .. 12) of Integer :=
        (0, 8, 24, 40, 56, 80, 104, 128, 162, 196, 255, 255);

   ------------------------
   -- Set_Current_Volume --
   ------------------------

   procedure Set_Current_Volume
     (Requested_Volume : Allowed_Volume;
      Amplifier        : in out Amplification;
      Actual_Volume    : in out Allowed_Volume)
   is
      Output : Integer;
   begin
      if Actual_Volume /= Requested_Volume then

         --  generate the amplifier table for this requested volume
         Output := Volume_Table (Requested_Volume / 10);
         Output := Output +
            ((Volume_Table (Requested_Volume / 10 + 1) - Output)
                 * (Requested_Volume mod 10)) / 10;
         for K in 0 .. 128 loop
            declare
               A : Integer := (K * Output) / 128;
               B : Integer := A;
            begin
               if A > 127 then
                  A := 127;
                  B := 128;
               end if;
               Amplifier (128 - K) := Unsigned_8 (128 - B);
               Amplifier (K + 128) := Unsigned_8 (A + 128);
            end;
         end loop;

         Actual_Volume := Requested_Volume;
      end if;
   end Set_Current_Volume;

   --  Values with 0-32 evenly spaced bits set, used when expanding the input
   --  samples into 32-bit quantities
   Sample_Pattern : constant array (Unsigned_8 range 0 .. 32) of Unsigned_32 :=
     (16#0000_0000#, 16#8000_0000#, 16#8000_8000#, 16#8020_0400#,
      16#8080_8080#, 16#8208_1040#, 16#8420_8420#, 16#8844_2210#,
      16#8888_8888#, 16#9122_4488#, 16#9248_9248#, 16#A492_4924#,
      16#A4A4_A4A4#, 16#A94A_5294#, 16#AA54_AA54#, 16#AAAA_5554#,
      16#AAAA_AAAA#, 16#D555_AAAA#, 16#D5AA_D5AA#, 16#D6B5_AD6A#,
      16#DADA_DADA#, 16#DB6D_B6DA#, 16#EDB6_EDB6#, 16#EEDD_BB76#,
      16#EEEE_EEEE#, 16#F7BB_DDEE#, 16#FBDE_FBDE#, 16#FDF7_EFBE#,
      16#FEFE_FEFE#, 16#FFDF_FBFE#, 16#FFFE_FFFE#, 16#FFFF_FFFE#,
      16#FFFF_FFFF#);

   Pattern_Masks : constant array (Unsigned_8 range 0 .. 7) of Unsigned_8 :=
     (16#00#, 16#10#, 16#22#, 16#4A#, 16#55#, 16#6D#, 16#77#, 16#7F#);

   Samples_Per_Fill    : constant := 8;
   Iterations_Per_Fill : constant := PDM_Buffer_Length / Samples_Per_Fill;

   ----------
   -- Fill --
   ----------

   procedure Fill
     (Buffer       : in out Sample_Buffer;
      Input        : Sound_Sample_Reference;
      Amplifier    : Amplification;
      Sample_Count : in out Integer;
      Next_In      : in out Natural)
   is
      Sample         : Unsigned_8;
      Mask           : Unsigned_8;
      Shifted_Sample : Unsigned_8;
      Pattern_Index  : Unsigned_8;
      Next_Out       : Integer := Buffer'First;
      --  Next_Out is logiclly a Buffer_Index value but the code is cleaner if
      --  we allow Next_Out to range from Buffer'First .. Buffer'Last + 1. Note
      --  we never use that last value (exceeding the index range) so no
      --  problems should occur.

      --  Expand the next eight 8-bit samples in Input into eight 32-bit
      --  numbers, ie 256 bits, and place them into Buffer. This number of bits
      --  must match the sample size used in Play_Sample!
   begin
      for K in 1 .. Iterations_Per_Fill loop
         if Sample_Count > 0 then
            Sample := Amplifier (Integer (Input (Next_In)));
         else
            Sample := 128;
         end if;
         Mask := Pattern_Masks (Sample and 7);
         Shifted_Sample := Shift_Right (Sample, 3);

         --  the logical loop expanding the first 6 expanded bytes is unrolled
         --  for efficiency

         Pattern_Index := Shifted_Sample + (Mask and 1);
         Buffer (Next_Out) := Sample_Pattern (Pattern_Index);
         Mask := Shift_Right (Mask, 1);
         Next_Out := Next_Out + 1;

         Pattern_Index := Shifted_Sample + (Mask and 1);
         Buffer (Next_Out) := Sample_Pattern (Pattern_Index);
         Mask := Shift_Right (Mask, 1);
         Next_Out := Next_Out + 1;

         Pattern_Index := Shifted_Sample + (Mask and 1);
         Buffer (Next_Out) := Sample_Pattern (Pattern_Index);
         Mask := Shift_Right (Mask, 1);
         Next_Out := Next_Out + 1;

         Pattern_Index := Shifted_Sample + (Mask and 1);
         Buffer (Next_Out) := Sample_Pattern (Pattern_Index);
         Mask := Shift_Right (Mask, 1);
         Next_Out := Next_Out + 1;

         Pattern_Index := Shifted_Sample + (Mask and 1);
         Buffer (Next_Out) := Sample_Pattern (Pattern_Index);
         Mask := Shift_Right (Mask, 1);
         Next_Out := Next_Out + 1;

         Pattern_Index := Shifted_Sample + (Mask and 1);
         Buffer (Next_Out) := Sample_Pattern (Pattern_Index);
         Mask := Shift_Right (Mask, 1);
         Next_Out := Next_Out + 1;

         --  last two bytes of this iteration...

         Pattern_Index := Shifted_Sample + (Mask and 1);
         Buffer (Next_Out) := Sample_Pattern (Pattern_Index);
         --  don't update Mask
         Next_Out := Next_Out + 1;

         --  don't use the mask at all
         Buffer (Next_Out) := Sample_Pattern (Shifted_Sample);
         Next_Out := Next_Out + 1;

         Sample_Count := Sample_Count - 1;
         Next_In := Next_In + 1;
      end loop;
   end Fill;

   -------------------------
   -- Initialize_Hardware --
   -------------------------

   procedure Initialize_Hardware is
      Interrupt_Mode : constant Unsigned_32 := AIC_Lowest_Priority_Level or
        AIC_Internal_Src_Edge_Triggered;

      Clock_Mode : constant := SSC_Divided_Clock
        + SSC_Continuous_Clock_RK
        + SSC_Start_Continuous;

      Frame_Mode : constant Unsigned_32 := 31 + Shift_Left (7, 8) + SSC_MSBF;
   begin
      Disable_Sound_Interrupt;
      Disable_Sound;

      PMC_PCER := 2 ** AT91C_ID_SSC;  -- Peripheral Clock Enable Register

      --  Port A, pin 17 is connected to the speaker, per the NXT schematic
      --  (as SOUND_ARMA in the schematic)
      PIOA_ODR  := PA17_TD;  -- Output Disable Register
      PIOA_OWDR := PA17_TD;  -- Output Write Disable Register
      PIOA_MDDR := PA17_TD;  -- Multi-driver Disable Register
      PIOA_PUDR := PA17_TD;  -- Pull-up Disable Register
      PIOA_IFDR := PA17_TD;  -- Input Filter Disable Register
      PIOA_CODR := PA17_TD;  -- Clear Output Data Register
      PIOA_IDR  := PA17_TD;  -- Interrupt Disable Register

      SSC_CR   := SSC_SWRST;   -- SSC_Software_Reset
      SSC_TCMR := Clock_Mode;
      SSC_TFMR := Frame_Mode;
      SSC_CR   := SSC_TXEN;  -- SSC_Transmit_Enable

      AIC_IECR := 2 ** AT91C_ID_SSC;  -- enable SSC peripheral interrupts
      AIC_ICCR := 2 ** AT91C_ID_SSC;  -- clear SSC interrupt
      AIC_SMR (AT91C_ID_SSC) := Interrupt_Mode;
   end Initialize_Hardware;

   procedure Mute renames Disable_Sound;

   procedure Unmute renames Enable_Sound;

begin
   Disable_Interrupt (AT91C_ID_SSC);
   Initialize_Hardware;
   Enable_Interrupt (AT91C_ID_SSC);
end NXT.Audio;
