------------------------------------------------------------------------------
--                                                                          --
--                           GNAT RAVENSCAR for NXT                         --
--                                                                          --
--                       Copyright (C) 2011, AdaCore                        --
--                                                                          --
-- This is free software; you can  redistribute it  and/or modify it under  --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. This is distributed in the hope that it will be useful, but WITH-  --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
------------------------------------------------------------------------------

with NXT.AVR;
with Interfaces; use Interfaces;

package body Discrete_IO is

   procedure Configure_Pin
     (Port      : Sensor_Id;
      Wire      : Wire_Colors;
      Direction : Sensor_Ports.Modes);
   --  set the given pin on the specified port to the required I/O direction

   ------------------------
   -- Configure_Hardware --
   ------------------------

   procedure Configure_Hardware (Pin_Map : IO_Mapping) is
   begin
      for Port in Pin_Map'Range loop
         Reset (Port);
         for Wire in IO_Pins'Range loop
            Configure_Pin (Port, Wire, Direction => Pin_Map (Port) (Wire));
         end loop;
      end loop;
   end Configure_Hardware;

   -----------
   -- Value --
   -----------

   function Value (Port : Sensor_Id;  Wire : Wire_Colors) return Pin_States is
      Threshold : constant Unsigned_16 := 700;
   begin
      case Wire is
         when White =>
            --  we are presenting the analog input as a discrete input, so if a
            --  "reasonable" voltage is applied we consider it High
            if NXT.AVR.Raw_Input (Port) <= Threshold then
               return High;
            else
               return Low;
            end if;
         when Yellow =>
            return Current_State (Port, Digital_0);
         when Blue =>
            return Current_State (Port, Digital_1);
      end case;
   end Value;

   -------------------
   -- Configure_Pin --
   -------------------

   procedure Configure_Pin
     (Port      : Sensor_Id;
      Wire      : Wire_Colors;
      Direction : Sensor_Ports.Modes)
   is
   begin
      case Wire is
         when White =>
            --  in this app the ADC wire has to be an input
            if Direction = Output then
               raise Constraint_Error;
            end if;
            --  no other configuration is required, the ADC line comes up ready
            --  to be read
         when Yellow =>
            Set_Pin_Mode (Port, Digital_0, Direction);
         when Blue =>
            Set_Pin_Mode (Port, Digital_1, Direction);
      end case;
   end Configure_Pin;

end Discrete_IO;
