------------------------------------------------------------------------------
--                                                                          --
--                           GNAT RAVENSCAR for NXT                         --
--                                                                          --
--                       Copyright (C) 2011, AdaCore                        --
--                                                                          --
-- This is free software; you can  redistribute it  and/or modify it under  --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. This is distributed in the hope that it will be useful, but WITH-  --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
------------------------------------------------------------------------------

with NXT.Display;       use NXT.Display;
with NXT.AVR;
with Ada.Real_Time;     use Ada.Real_Time;
with Discrete_IO.Display;

package body Discrete_IO.HMI is

   --  We allow the user to select among the various output pins, if any, so
   --  that they can change the values produced (High or Low). Only outputs can
   --  be selected because setting an input would be nonsensical.

   procedure Go_To_Next_Output;
   --  move the 'cursor' to the next pin configured as an output, relative to
   --  the currently selected pin

   procedure Go_To_Previous_Output;
   --  move the 'cursor' to the previous pin configured as an output, relative
   --  to the currently selected pin

   procedure Await_Any_Button_Cycle;
   --  Awaits any new button press and release

   --------------------
   -- Initialize_HMI --
   --------------------

   procedure Initialize_HMI (Pin_Map : IO_Mapping) is
   begin
      for Port in Pin_Map'Range loop
         for Wire in Wire_Colors loop
            if Pin_Map (Port) (Wire) = Output then
               Last_Output_Index := Last_Output_Index + 1;
               Known_Outputs (Last_Output_Index) := (Port, Wire);
            end if;
         end loop;
      end loop;

      if Last_Output_Index /= 0 then
         Discrete_IO.Display.Mark_Current
           (Known_Outputs (Current_Selection).Port,
            Known_Outputs (Current_Selection).Wire);
      end if;
   end Initialize_HMI;

   ------------------------
   -- Interact_With_User --
   ------------------------

   procedure Interact_With_User (Pin_Map : IO_Mapping) is
   begin
      if Last_Output_Index /= 0 then
         if NXT.AVR.Button /= No_Button then
            case NXT.AVR.Button is
               when Left_Button =>
                  Go_To_Previous_Output;
               when Right_Button =>
                  Go_To_Next_Output;
               when Middle_Button =>
                  Toggle_Output (Known_Outputs (Current_Selection));
               when others =>
                  null;
            end case;
            --  a button was pressed so now wait for the corresponding release
            Await_Release : loop
               exit Await_Release when NXT.AVR.Button = No_Button;
               delay until Clock + Milliseconds (10);
            end loop Await_Release;
         end if;
      end if;
   end Interact_With_User;

   -------------------
   -- Toggle_Output --
   -------------------

   procedure Toggle_Output (Output : Output_Pin) is
      Pin : Pin_Id;
   begin
      case Output.Wire is
         when Yellow =>
            Pin := Digital_0;
         when Blue =>
            Pin := Digital_1;
         when White =>
            --  this is the ADC, it cannot be used as an output in this app
            raise Constraint_Error;
      end case;
      if Current_State (Output.Port, Pin) = High then
         Set_Pin_State (Output.Port, Pin, Low);
      else
         Set_Pin_State (Output.Port, Pin, High);
      end if;
   end Toggle_Output;

   -----------------------
   -- Go_To_Next_Output --
   -----------------------

   procedure Go_To_Next_Output is
   begin
      if Last_Output_Index /= 0 then
         if Current_Selection < Last_Output_Index then
            Current_Selection := Current_Selection + 1;
         else
            Current_Selection := 1;
         end if;

         Discrete_IO.Display.Mark_Current
           (Known_Outputs (Current_Selection).Port,
            Known_Outputs (Current_Selection).Wire);
      end if;
   end Go_To_Next_Output;

   ---------------------------
   -- Go_To_Previous_Output --
   ---------------------------

   procedure Go_To_Previous_Output is
   begin
      if Last_Output_Index /= 0 then
         if Current_Selection > 1 then
            Current_Selection := Current_Selection - 1;
         else
            Current_Selection := Last_Output_Index;
         end if;

         Discrete_IO.Display.Mark_Current
           (Known_Outputs (Current_Selection).Port,
            Known_Outputs (Current_Selection).Wire);
      end if;
   end Go_To_Previous_Output;

   ----------------------------
   -- Display_Welcome_Screen --
   ----------------------------

   procedure Display_Welcome_Screen is
   begin
      Put_Line ("    Discrete");
      Put_Line ("   I/O Monitor");
      New_Line;
      Put_Line ("L/R moves *");
      Put_Line ("Middle toggles");
      Put_Line ("Power to exit");
      New_Line;
      Put ("Press any now");
      Await_Any_Button_Cycle;
   end Display_Welcome_Screen;

   ----------------------------
   -- Await_Any_Button_Cycle --
   ----------------------------

   procedure Await_Any_Button_Cycle is
      use NXT;
   begin
      loop
         exit when NXT.AVR.Button = No_Button;
         delay until Clock + Milliseconds (10);
      end loop;
      loop
         exit when NXT.AVR.Button /= No_Button;
         delay until Clock + Milliseconds (10);
      end loop;
      loop
         exit when NXT.AVR.Button = No_Button;
         delay until Clock + Milliseconds (10);
      end loop;
   end Await_Any_Button_Cycle;

end Discrete_IO.HMI;
